/* -*- mode:C; c-file-style:"gnu"; indent-tabs-mode:nil; -*- */
/* 
 * Copyright (C) 2006 OpenedHand Ltd.
 *
 * Author: Tomas Frydrych <tf@o-hand.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 *
 *
 * Exmap server
 */

#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <netinet/in.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <glib.h>

#include "exmap.h"

static int sckt_in;

static void
cmd_end ()
{
  write (sckt_in, "EXMP_DONE\n", 10);
}

static void
write_line (int s, const char * buf, size_t len)
{
  write (s, buf, len);
  write (s, "\n", 1);
}

static void
server_spawn (const char * cmd)
{
  GError * error = NULL;
  gchar  * out = NULL;
  gchar  * err = NULL;
  gint     status;

  if (g_spawn_command_line_sync (cmd, &out, &err, &status, &error))
    {
      if (out)
        {
          write (sckt_in, out, strlen (out));
          g_free (out);
        }

      if (WEXITSTATUS (status))
        {
          if (err)
            {
              write (sckt_in, err, strlen (err));
              g_free (err);
            }
        }
    }
  else
    {
      write (sckt_in, error->message, strlen (error->message));
      g_error_free (error);
    }

  cmd_end ();
}

static void
server_load_pids ()
{
  const gchar * n;
  GDir        * dir = g_dir_open ("/proc", 0, NULL);

  if (!dir)
    return;

  while ((n = g_dir_read_name (dir)) != NULL)
    {
      if (isdigit(*n))
        {
          write_line (sckt_in, n, strlen (n));
        }
    }

  cmd_end ();
  g_dir_close (dir);

  return;
}

static void
server_get_pid_maps (pid_t pid)
{
  char buf [256];
  gchar    * cmd_f;
  FILE     * fcmd = NULL;

  if (!pid)
    {
      g_snprintf (buf, sizeof (buf),
                  "EXMP_ERROR Invalid pid %d\n",
                  pid);
      write (sckt_in, buf, strlen (buf));
      return;
    }
  
  cmd_f = g_strdup_printf ("/proc/%d/maps",pid);
  fcmd = fopen (cmd_f, "r");
  g_free (cmd_f);

  if (!fcmd)
    {
      g_snprintf (buf, sizeof (buf),
                  "EXMP_ERROR Could not read maps for %d\n",
                  pid);
      write (sckt_in, buf, strlen (buf));
      return;
    }

  while (fgets(buf, sizeof (buf), fcmd))
    {
      write (sckt_in, buf, strlen (buf));
    }

  cmd_end ();
}

static void
server_get_pid_cmdline (pid_t pid)
{
  char buf [256];
  gchar    * cmd_f;
  FILE     * fcmd = NULL;

  if (!pid)
    {
      g_snprintf (buf, sizeof (buf),
                  "EXMP_ERROR Invalid pid %d\n",
                  pid);
      write (sckt_in, buf, strlen (buf));
      return;
    }
  
  cmd_f = g_strdup_printf ("/proc/%d/cmdline",pid);
  fcmd = fopen (cmd_f, "r");
  g_free (cmd_f);
  
  if (fcmd)
    {
      /* cmd line can be empty, in which case we send a single newline */
      if (fgets(buf, sizeof (buf), fcmd))
        write_line (sckt_in, buf, strlen (buf));
      else
        write (sckt_in, "\n", 1);
      cmd_end ();
    }
  else
    {
      g_snprintf (buf, sizeof (buf),
                  "EXMP_ERROR Could not read cmdline for %d\n",
                  pid);
      write (sckt_in, buf, strlen (buf));
    }
}

static void
server_get_pid_data (pid_t pid)
{
  char buf [256];
  FILE * fexmp = NULL;

  if (!pid)
    {
      g_snprintf (buf, sizeof (buf),
                  "EXMP_ERROR Invalid pid %d\n",
                  pid);
      write (sckt_in, buf, strlen (buf));
      return;
    }
  
  fexmp = fopen (EXMFILE, "w+");

  if (fexmp)
    {
      if (EOF == fprintf (fexmp, "%d", pid))
        {
          g_snprintf (buf, sizeof (buf),
                      "EXMP_ERROR Could not read exmap data for %d\n",
                      pid);
          write (sckt_in, buf, strlen (buf));
        }
      else
        {
          while (fgets (buf, sizeof(buf), fexmp))
            {
              write (sckt_in, buf, strlen (buf));
            }

          cmd_end ();
        }
    }
  else
    {
      g_snprintf (buf, sizeof (buf),
                  "EXMP_ERROR Could not open /proc/exmap\n");
      write (sckt_in, buf, strlen (buf));
    }
}

/* This the work horse -- it parses incomming commands and acts
 * correspondingly.
 *
 * The cmd set is very limitted -- the server only providies access to
 * to /proc/pid/cmdline and /proc/exmap. Everything else is done by the
 * client.
 *
 * Each command line looks like this
 *
 * EXMP_CMD cmd [param1 param2 ...]
 *
 */
static void
server_parse_line (const char * line)
{
  if (line && !strncmp (line, "EXMP_CMD", 8))
    {
      const char * p = line + 8;

      while (isspace(*p))
        ++p;

      if (!strncmp (p, "SYSPIDS", 7))
        {
          server_load_pids ();
          return;
        }
      
      if (!strncmp (p, "PIDCMDLINE", 10))
        {
          p += 10;
          while (isspace(*p))
            ++p;

          pid_t pid = atoi (p);
          server_get_pid_cmdline (pid);
          return;
        }

      if (!strncmp (p, "PIDMAPS", 7))
        {
          p += 7;
          while (isspace(*p))
            ++p;

          pid_t pid = atoi (p);
          server_get_pid_maps (pid);
          return;
        }

      if (!strncmp (p, "PIDDATA", 7))
        {
          p += 7;
          while (isspace(*p))
            ++p;

          pid_t pid = atoi (p);
          server_get_pid_data (pid);
          return;
        }

      if (!strncmp (p, "SPAWN", 5))
        {
          p += 5;
          while (isspace(*p))
            ++p;

          server_spawn (p);
          return;
        }
    }
  else
    {
      char buf [256];

      g_snprintf (buf, sizeof (buf),
                  "EXMP_ERROR Invalid input '%s'\n",
                  line);
      write (sckt_in, buf, strlen (buf));
    }
}

static void
server_loop (int port)
{
  int sckt;
  struct sockaddr addr_in;

  union 
  {
    struct sockaddr s;
    struct sockaddr_in in;
  } addr_lstn;

  addr_lstn.in.sin_family = AF_INET;
  addr_lstn.in.sin_port = port;
  addr_lstn.in.sin_addr.s_addr = INADDR_ANY;


  if ((sckt = socket(AF_INET, SOCK_STREAM, 0)) < 0)
    {
      g_warning ("Unable to open socket");
      return;
    }
    
  if (bind (sckt, &addr_lstn.s, sizeof (addr_lstn)) < 0)
    {
      g_warning ("Unable to bind socket");
      return;
    }

  while (1)
    {
      listen (sckt, 1);
      char buf[100];
      gint len;
        
      do
	{
          socklen_t size = sizeof (addr_in);
          sckt_in = accept (sckt, &addr_in, &size);

          if (sckt_in >= 0)
            {
              gboolean version_check = FALSE;
              gboolean prot_check    = FALSE;

              /* Handshake -- check that it is a exmap client on the
               * other hand and that its version matches ours.
               */
              if (0 < (len = read (sckt_in, buf, sizeof (buf))))
                {
                  char * p;
                  if ((p = g_strstr_len (buf, len, "EXMP_HANDSHAKE")))
                    {
                      prot_check = TRUE;
                      p = p + 14;

                      while (*p && isspace (*p))
                        ++p;

                      if (0 == strncmp (p, EXMAP_PROTOCOL_VERSION,
                                        strlen (EXMAP_PROTOCOL_VERSION)))
                        {
                          version_check = TRUE;
                        }
                    }
                }

              if (prot_check)
                {
                  if (!version_check)
                    {
                      g_snprintf (buf, sizeof (buf),
                                  "EXMP_ERROR Version mismatch -- "
                                  "server %s\n",
                                  VERSION "-" EXMAP_VERSION);

                      write (sckt_in, buf, strlen (buf));
                      close (sckt_in);
                      sckt_in = -1;
                    }
                  else
                    {
                      cmd_end();
                    }
                }
                    
              if (version_check)
                {
                  /* All preliminaries successfully completed -- wait for
                   * instructions
                   */
                  while (0 < (len = read (sckt_in, buf, sizeof (buf))))
                    {
                      char * line = buf;
                      char * p = g_strstr_len (buf, len, "\n");

                      while (p)
                        {
                          len -= (p - line + 1);
                          *p = 0;
                    
                          server_parse_line (line);
                    
                          line = p + 1;
                          p = g_strstr_len (line, len, "\n");
                        }
                    }

                  close (sckt_in);
                  sckt_in = -1;
                }
            }
	}
      while (1);
    }
    
  close (sckt);
}

static void
signal_handler (int sig)
{
  if (sig == SIGPIPE)
    {
      sckt_in = -1;
      g_debug ("Caught SIGPIPE\n");
    }
}

static void
usage (const char * name)
{
  g_print ("\nThis is Exmap-server for remote exmap sessions.\n");
  g_print ("\nUsage: %s [--no-fork] port\n\n", name);
  g_print ("Options:\n");
  g_print ("  --no-fork             : Do not fork daemon on startup\n\n");
}

int
main (int argc, char ** argv)
{
  int        i;
  gboolean   no_fork = FALSE;
  pid_t      pid;
  char     * modp_args [3] = {"modprobe", "exmap", NULL};
  int        modp_status = 0;
  GError   * error = NULL;
  
  if (argc < 2)
    {
      usage (argv[0]);
      return -1;
    }

  for (i = 1; i < argc; ++i)
    {
      if (strncmp ("--", argv[i], 2))
        break;
      
      if (!strcmp (argv[i], "--no-fork"))
        {
          no_fork = TRUE;
          continue;
        }

      if (!strcmp (argv[i], "--help"))
        {
          usage (argv[0]);
          return 0;
        }

      if (!strcmp (argv[i], "--version"))
        {
          g_print (VERSION "\n");
          return 0;
        }

      if (!strcmp (argv[i], "--module-version"))
        {
          g_print (EXMAP_VERSION "\n");
          return 0;
        }
    }

  if (i >= argc)
    {
      usage (argv[0]);
      return -1;
    }
  
  int port = atoi (argv[i]);

  if (!port)
    {
      usage (argv[0]);
      return -1;
    }

  /* The daemon fork */
  if (!no_fork)
    {
      g_print("Exmap server: forking; run with --no-fork to prevent fork\n");
      
      pid = fork();
      
      switch (pid) 
	{
	case -1:
	  g_error("Fork failed.\n");
	  break;
          
	case 0:
	  /* Child - live and let live */
	  break;
          
	default:
          /* Original process -- time to die and let the younsters to
	   * to take over the world.
	   */
	  exit(0);
	}
    }

  if (getuid() == 0)
    {
      /* if running a root, we can make sure that the exmap module is loaded */
      if (g_spawn_sync (NULL, (char**)&modp_args, NULL, G_SPAWN_SEARCH_PATH,
                        NULL, NULL, NULL, NULL, &modp_status, &error))
        {
          if (modp_status)
            g_print ("Error loading exmap module; modprobe returned %d\n",
                     modp_status);
        }
      else
        {
          g_print ("Could not spawn modprobe: %s\n",
                   error ? error->message : "");
        }
    }
  
  /* handle SIGPIPE */
  struct sigaction sa;
  sigfillset (&sa.sa_mask);
  sa.sa_handler = signal_handler;
  sigaction (SIGPIPE, &sa, NULL);
  
  server_loop (port);

  return 0;
}

