from __future__ import annotations

from abc import ABC, abstractmethod
from typing import TYPE_CHECKING

from pydantic import BaseModel, ConfigDict, Field

from npe2.manifest.utils import Executable
from npe2.types import LayerData

if TYPE_CHECKING:
    from npe2._command_registry import CommandRegistry


class _SampleDataContribution(BaseModel, ABC):
    """Contribute sample data for use in napari.

    Sample data can take the form of a **command** that returns layer data, or a simple
    path or **uri** to a local or remote resource (assuming there is a reader plugin
    capable of reading that path/URI).
    """

    key: str = Field(..., description="A unique key to identify this sample.")
    display_name: str = Field(
        ..., description="String to show in the UI when referring to this sample"
    )

    @abstractmethod
    def open(
        self, *args, _registry: CommandRegistry | None = None, **kwargs
    ) -> list[LayerData]: ...


class SampleDataGenerator(_SampleDataContribution, Executable[list[LayerData]]):
    """Contribute a callable command that creates data on demand."""

    command: str = Field(
        ...,
        description="Identifier of a command that returns layer data tuple. "
        "Note that this command cannot return `[(None,)]`.",
    )

    def open(
        self, *args, _registry: CommandRegistry | None = None, **kwargs
    ) -> list[LayerData]:
        return self.exec(args, kwargs, _registry=_registry)

    model_config = ConfigDict(title="Sample Data Function")


class SampleDataURI(_SampleDataContribution):
    """Contribute a URI to static local or remote data. This can be data included in
    the plugin package, or a URL to remote data.  The URI must be readable by either
    napari's builtin reader, or by a plugin that is included/required."""

    uri: str = Field(
        ...,
        description="Path or URL to a data resource. "
        "This URI should be a valid input to `io_utils.read`",
    )
    reader_plugin: str | None = Field(
        None,
        description="Name of plugin to use to open URI",
    )

    def open(self, *args, **kwargs) -> list[LayerData]:
        from npe2.io_utils import read

        return read([self.uri], plugin_name=self.reader_plugin, stack=False)

    model_config = ConfigDict(title="Sample Data URI")


SampleDataContribution = SampleDataGenerator | SampleDataURI
