/*
 *  @(#)AbstractWriterServer.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Part of the GroboUtils package at:
 *  http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */
package net.sourceforge.groboutils.autodoc.v1.testserver;

import java.io.IOException;
import java.io.Writer;

import org.apache.log4j.Logger;


/**
 * An interface which corresponds to a part of the framework that knows how
 * to deal with the framework's <tt>TestData</tt>.  It may directly deal with
 * the data, or pass it off to a remote server.
 *
 * @author     Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version    $Date: 2003/02/10 22:52:13 $
 * @since      March 30, 2002
 */
public abstract class AbstractWriterServer implements Server
{
    private static final Logger LOG = Logger.getLogger( AbstractWriterServer.class );
    
    
    /**
     * Receives <tt>td</tt> from a Monitor and handles it in an implementation
     * specific manner.
     *
     * @param td the data to deal with.
     */
    public void addTestData( TestData td )
    {
        if (td == null)
        {
            // allow null aruments: fail gracefully.
            LOG.warn("addTestData() received null TestData.");
            return;
        }
        
        // Write the result with all encountered logs.
        Writer w;
        try
        {
            w = openOutput( td );
        }
        catch (IOException e)
        {
            LOG.error( "Problem opening output.", e );
            
            // don't do any writing since the writer could not be opened.
            return;
        }
        
        try
        {
            writeTestData( td, w );
        }
        catch (IOException e)
        {
            LOG.error( "Problem writing to output.", e );
        }
        finally
        {
            // no matter what the write result may be, close the
            // output at the end.
            try
            {
                closeOutput( w );
            }
            catch (IOException e)
            {
                LOG.error( "Problem closing output.", e );
            }
        }
    }
    
    
    /**
     * Write the BugTestResult with all the result's encountered BugTestLog
     * events.
     */
    protected abstract void writeTestData( TestData td, Writer w )
            throws IOException;
    
    
    /**
     * Open a writer stream.  This will be done once per result, so log-like
     * actions may need to append to the previous results.
     */
    protected abstract Writer openOutput( TestData td )
            throws IOException;
    
    
    /**
     * Close the given Writer, which was opened in the <tt>openOutput()</tt>
     * method.  Since nearly all implementations will be the same, the
     * default implementation simply executes <code>w.close()</code>.
     */
    protected void closeOutput( Writer w )
            throws IOException
    {
        w.close();
    }
}

