
/*
  CoreLinux++ 
  Copyright (C) 1999,2000 CoreLinux Consortium
  
   The CoreLinux++ Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The CoreLinux++ Library Library is distributed in the hope that it will 
   be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the GNU C Library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  
*/

#if   !defined(__COMMON_HPP)
#include <Common.hpp>
#endif

#if   !defined(__CORELINUXGUARDGROUP_HPP)
#include <CoreLinuxGuardGroup.hpp>
#endif

#if   !defined(__GUARDSEMAPHORE_HPP)
#include <GuardSemaphore.hpp>
#endif

namespace corelinux
{
   //
   // Basic group constructor
   //

   CoreLinuxGuardGroup::CoreLinuxGuardGroup( Short aSemCount )
      throw(Assertion,SemaphoreException)
      :
      SemaphoreGroup( aSemCount ),
      theUsedMap( ::new AbstractSemaphorePtr[aSemCount] )
   {
      REQUIRE( theUsedMap != NULLPTR );
      memset( theUsedMap,0,aSemCount * sizeof(VoidPtr) );
   }

   //
   // Destructor
   //

   CoreLinuxGuardGroup::~CoreLinuxGuardGroup( void )
   {
      if( theUsedMap != NULLPTR )
      {
         delete [] theUsedMap;
         theUsedMap = NULLPTR;
      }
      else
      {
         ;  // do nothing
      }
   }

   //
   //  Creates a default semaphore
   //

   AbstractSemaphorePtr CoreLinuxGuardGroup::createSemaphore( void )
                           throw( SemaphoreException ) 
   {
      AbstractSemaphorePtr aSem(NULLPTR);
      Short maxSems( getSemaphoreCount() );

      for( Short x = 0; x < maxSems && aSem == NULLPTR ; ++x )
      {
         if( theUsedMap[x] == NULLPTR )
         {
            SemaphoreIdentifier  aSemId(x);
            theUsedMap[x] = aSem = 
               new GuardSemaphore
                  ( 
                     this,
                     aSemId
                  );
         }
         else
         {
            ;  // try next
         }
      }

      if( aSem == NULLPTR )
      {
         throw SemaphoreException
            (
               "Can't create semaphore",
               LOCATION,
               Exception::CONTINUABLE
            );
      }

      return aSem;
   }

   //
   // Creates, or opens, a specific semaphore in the
   // group
   //

   AbstractSemaphorePtr CoreLinuxGuardGroup::createSemaphore
   ( 
      SemaphoreIdentifierRef aIdentifier,
      CreateDisposition       disp,
      bool                    Recursive, 
      bool                    Balking 
   ) throw( SemaphoreException ) 
   {
      AbstractSemaphorePtr aSem( NULLPTR );
      SemaphoreIdentifier  maxSems( getSemaphoreCount() );

      ENSURE( aIdentifier <= maxSems );
      ENSURE( aIdentifier >= SemaphoreIdentifier(0) );

      if( theUsedMap[aIdentifier.getScalar()] != NULLPTR )
      {
         if( disp != FAIL_IF_EXISTS )
         {
            aSem = theUsedMap[aIdentifier.getScalar()];
         }
         else
         {
            throw SemaphoreException
               (
                  "Semaphore exists exception",
                  LOCATION,
                  Exception::CONTINUABLE
               );
         }
      }
      else
      {
         if( disp != FAIL_IF_NOTEXISTS )
         {
            theUsedMap[aIdentifier.getScalar()] = aSem = 
               new GuardSemaphore
                     ( 
                        this,
                        aIdentifier
                     );
         }
         else
         {
            throw SemaphoreException
               (
                  "Semaphore does not exist",
                  LOCATION
               );
         }
      }

      return aSem;
   }

   //
   // Creates or opens a specific named semaphore in the
   // group (not implemented)
   //

   AbstractSemaphorePtr CoreLinuxGuardGroup::createSemaphore
   ( 
      std::string          aName,
      CreateDisposition    disp,
      bool                 Recursive, 
      bool                 Balking 
   ) throw( SemaphoreException ) 
   {
      AbstractSemaphorePtr aSem(NULLPTR);
      throw SemaphoreException
         (
            "Semaphore does not exist",
            LOCATION
         );
      return aSem;
   }

   //
   // Destroy the semapore
   //

   void  CoreLinuxGuardGroup::destroySemaphore( AbstractSemaphorePtr aPtr )
                     throw( SemaphoreException ) 
   {
      REQUIRE( aPtr != NULLPTR );

      if( aPtr->getGroupIdentifier() == getIdentifier() )
      {
         if( theUsedMap[aPtr->getIdentifier().getScalar()] != NULLPTR )
         {
            theUsedMap[aPtr->getIdentifier().getScalar()] = NULLPTR;
            delete aPtr;
         }
         else
         {
            ;  // double delete exception MAYBE!!!
         }

      }
      else
      {
         throw SemaphoreException
            (
               "Semaphore not part of group",
               LOCATION
            );
      }

   }
}

/*
   Common rcs information do not modify
   $Author: frankc $
   $Revision: 1.3 $
   $Date: 2000/06/02 11:51:52 $
   $Locker:  $
*/



