/* Unit specific windows and dialogs for the Mac interface to Xconq.
   Copyright (C) 1998-2000 Hans Ronne and
  1992-1999  Stanley T. Shebs (closeup code).

Xconq is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.  See the file COPYING.  */

/* This file now contains all windows (closeups) and dialogs that refer to individual units.
Much of the stuff is from the old macadv.c file, some from macwins.c. global_advance_dialog 
has been renamed to side_research_dialog and is now in macwins.c. */ 

#include "conq.h"
#include "macconq.h"

/* Base numbers of resource cicns. */
#define LARGE_CICNS 2000
#define SMALL_CICNS 2100

/* Max number of facilities and occs displayed in city dialog. Should always agree with the DITL!  */
#define MAX_DISPLAYED_FACS 12
#define MAX_DISPLAYED_OCCS 24
#define CLEAR_AGENDA 99

int default_drawsizes = TRUE;		/* Draw unit size in maps, closeups and lists. */

extern void mac_init_cicn(Image *img);
extern void unit_research_dialog(Unit *unit);			
extern void unit_plan_dialog(Unit *unit);
extern int unit_build_dialog(Unit *unit);

static void draw_simple_closeup(UnitCloseup *closeup);
static void draw_advanced_closeup(UnitCloseup *closeup);
static void copy_rect_from_gworld(WindowPtr win, Map *map, Rect cityrect, Rect itemrect);
static void draw_closeup_content(DialogPtr win, Unit *unit);
static void plot_resource_cicns(Map *map, Rect plotrect, int x, int y);
static void draw_landuse_near_unit(Map *map, WindowPtr win, Rect itemrect, Unit *unit, int near);
static void toggle_landuse_one_cell(Map *map, WindowPtr win, Rect itemrect, Unit *unit, int x, int y);

DialogPtr unit_advance_win = nil;
DialogPtr unit_build_win = nil;
DialogPtr unit_plan_win = nil;

/* Globals for unit closeup windows. */

int lastunitcloseuph = -1;
int lastunitcloseupv = -1;

static int m_per_row = 2;
static char *nummrows;

int closeup_spacing = 16;
int closeupwinwid = 240;

/* Unit closeups. */

/* This is the top-level access to bring up a unit closeup. */

void
show_unit_closeup(Unit *unit)
{
	UnitCloseup *unitcloseup;

	/* Find the closeup if it exists. */
	unitcloseup = find_unit_closeup(unit);
	/* Or create it. */
	if (!unitcloseup)
	     unitcloseup = create_unit_closeup(unit);
	/* Check that we have one before proceeding. */
	if (unitcloseup && unitcloseup->window) {
		SelectTheWindow(unitcloseup->window);
		update_window(unitcloseup->window);	
	}
}

UnitCloseup *
find_unit_closeup(Unit *unit)
{
	UnitCloseup *unitcloseup;

	for_all_unit_closeups(unitcloseup) {
		if (unitcloseup->unit == unit && unitcloseup->window)
		    return unitcloseup;
	}
	return NULL;
}

UnitCloseup *
unit_closeup_from_window(WindowPtr win)
{
	UnitCloseup *unitcloseup;
	
	for_all_unit_closeups(unitcloseup) {
		if (unitcloseup->window == win)
		  return unitcloseup;
	}
	return NULL;
}

UnitCloseup *
create_unit_closeup(Unit *unit)
{
	UnitCloseup *unitcloseup;
	DialogPtr win;
	int u, w, h;
		
	/* Stop if we lack a display. */
	if (!active_display(dside) || unit == NULL)
	    return NULL;
	/* Stop if we may not examine this unit. */
	if (!side_sees_unit(dside, unit)
	    && (!frontmap || !frontmap->see_all))
		return NULL;
	DGprintf("Creating a closeup of %s\n", unit_desig(unit));
	unitcloseup = (UnitCloseup *) xmalloc(sizeof(UnitCloseup));
	unitcloseup->unit = unit;
	u = unit->type;

	if (u_advanced(unit->type))
		win = GetNewDialog(dCity, NULL, NULL);
	else	win = GetNewDialog(dCloseup, NULL, NULL);
	unitcloseup->window = win;
	
	/* Size and position code for non-advanced closeups. */
	if (u_advanced(unit->type) != TRUE) {
		/* Put it adjacent to the clicked or list-selected unit. */
		if (frontmap) {
			GrafPtr oldport;
			Point upoint;
			int sx, sy;
							
			GetPort(&oldport);
			SetPort(frontmap->window);
			xform(frontmap, unit->x + 2, unit->y - 2, &sx, &sy);
			upoint.h = sx; 
			upoint.v = sy;
			LocalToGlobal(&upoint);
			MoveWindow(win, upoint.h, upoint.v, true);
			SetPort(oldport);
		} else {
			/* This only happens if no maps are open and a closeup 
			is chosen from a list. */
			stagger_window(unitcloseup->window, 
						&lastunitcloseuph, &lastunitcloseupv);
		}
		preferred_closeup_size(u, &w, &h);
		SizeWindow(win, w, h, 1);
	}
	/* Add window and menu titles. */
	sprintf(spbuf, "Closeup on %s", short_unit_handle(unit));
	add_window_menu_item(spbuf, win);
	/* Add it to the closeup list. */
	unitcloseup->next = unitcloseuplist;
	unitcloseuplist = unitcloseup;
	MakeDialogFloat(win);
	return unitcloseup;
}

/* Compute the right size for a unit's closeup. More complicated units
   with more state get bigger closeups. */

void
preferred_closeup_size(int u, int *widp, int *hgtp)
{
	int wid = closeupwinwid, hgt, m, u2, count;

	closeup_spacing = small_line_spacing + 1;
	/* Add 4 pixels at bottom to make room for 3rd task. */
	hgt = 4 + 32 + 4 + 3 * closeup_spacing + 4;
	if (nummrows == NULL) {
		/* Compute and cache the space needed to display each unit's supply. */
		nummrows = xmalloc(numutypes);
		for_all_unit_types(u2) {
			nummrows[u2] = count = 0;
			for_all_material_types(m) {
				if (um_storage_x(u2, m) > 0)
		  		  ++count;
		  	}
		  	if (count > 0) {
		  		nummrows[u2] = max(1, (count + (m_per_row - 1)) / m_per_row);
		  	}
		}
	}
	hgt += nummrows[u] * closeup_spacing;
	if (u_acp(u) > 0)
	  hgt += 5 * closeup_spacing;
	/* (should make some room for tooling state) */
	*widp = wid;  *hgtp = hgt;
}

/* Draw all the fields and displays in a unit closeup. */

void
draw_unit_closeup(UnitCloseup *closeup)
{
	DialogPtr 	win = closeup->window;
	Unit 		*unit = closeup->unit;
	GrafPtr 	oldport;

	/* Skip if we lack a display. */
	if (!active_display(dside))
		return;
	/* Blast the closeup if the unit is dead or has defected. */
	if (!in_play(unit)
	      || (!side_sees_unit(dside, unit)
	    	&& (!frontmap || !frontmap->see_all))) { 
		remove_window_menu_item(win);
		destroy_unit_closeup(closeup);
		HideTheWindow(win);
		return;
	}
	/* Make sure the unit's cell is updated before proceeding. */
	update_cell_display(dside, unit->x, unit->y, UPDATE_ALWAYS);

	GetPort(&oldport);
	SetPort(win);

	/* Draw the appropriate type of closeup. */
	if (u_advanced(unit->type))
		draw_advanced_closeup(closeup);
	else	draw_simple_closeup(closeup);

	SetPort(oldport);
}	

void
draw_simple_closeup(UnitCloseup *closeup)
{
	int u, m, sx = 4, sy, count, mrow, sx2, sy2;
	char infobuf[BUFSIZE];
	Rect tmprect;
	DialogPtr win = closeup->window;
	Unit *unit = closeup->unit;

	/* Use the small font. */
	TextFont(small_font_id);
	TextSize(small_font_size);
	EraseRect(&win->portRect);
	u = unit->type;
	/* Draw the unit's image. */
	SetRect(&tmprect, sx, sx, sx + 32, sx + 32); 
	EraseRect(&tmprect);
	draw_unit_image(win, tmprect.left, tmprect.top, 
				tmprect.right - tmprect.left, tmprect.bottom - tmprect.top,
				unit->type, side_number(unit->side), 
				(frontmap ? frontmap->sidecolors : default_sidecolors), 
				0, !completed(unit),
				(frontmap ? frontmap->draw_emblems : default_draw_emblems));
	/* Draw unit size on top of advanced unit. */
	if (u_advanced(unit->type) && default_drawsizes) 
		draw_unit_size(unit, win, tmprect.left, tmprect.top,
						tmprect.right - tmprect.left, tmprect.bottom - tmprect.top);

	/* Draw the unit's name. */
	name_or_number(unit, infobuf);
	MoveTo(44, 4 + closeup_spacing);
	DrawText(infobuf, 0, strlen(infobuf));
	if (Debug || DebugG || DebugM) {
		sprintf(infobuf, " %d ", unit->id);
		MoveTo(closeupwinwid - 30, 15);
		DrawText(infobuf, 0, strlen(infobuf));
	}
	/* Draw the unit's side and type. */
	side_and_type_name(infobuf, dside, u, unit->side);
	MoveTo(44, 4 + closeup_spacing * 2);
	DrawText(infobuf, 0, strlen(infobuf));
	/* Draw the unit's location. */
	location_desc(infobuf, dside, unit, unit->type, unit->x, unit->y);
	sy = 4 + 32 + 4 + closeup_spacing;
	MoveTo(sx, sy);
	DrawText(infobuf, 0, strlen(infobuf));
	/* Very briefly list the numbers and types of the occupants. */
	occupants_desc(infobuf, unit);
	sy += closeup_spacing;
	MoveTo(sx, sy);
	DrawText(infobuf, 0, strlen(infobuf));
	/* Draw the unit's hit points. */
	hp_desc(infobuf, unit, TRUE);
	strcat(infobuf, "   ");
	acp_desc(infobuf + strlen(infobuf), unit, TRUE);
	sy += closeup_spacing;
	MoveTo(sx, sy);
	DrawText(infobuf, 0, strlen(infobuf));
	/* Draw the unit's supplies. */
	mrow = 0;
	while (supply_desc(infobuf, unit, mrow)) {
		sy += closeup_spacing;
		MoveTo(sx, sy);
		DrawText(infobuf, 0, strlen(infobuf));
		++mrow;
	}
	/* Draw the unit's plan, if it has one. */
	if (unit->plan) {

		Task *task;
		Plan *plan = unit->plan;
		
		/* plan_desc now also includes goal_desc on the same line. To avoid 
		repetition only the first half of plan_desc is therefore included below. */ 
		strcpy(infobuf, "Plan: ");
		strcat(infobuf, plantypenames[plan->type]);
		if (plan->waitingfortasks)
		  strcat(infobuf, " Waiting");
		if (plan->asleep)
		  strcat(infobuf, " Asleep");
		if (plan->reserve)
		  strcat(infobuf, " Reserve");
		if (plan->delayed)
		  strcat(infobuf, " Delay");
		if (!plan->aicontrol)
		  strcat(infobuf, " NoAI");
		if (plan->supply_is_low)
		  strcat(infobuf, " SupplyLow");
		if (plan->tasks) {
			int i = 0;
			for_all_tasks(plan, task)
				++i;
			tprintf(infobuf, " (%d task%s)", i, (i == 1 ? "" : "s"));
		} 
		sy += closeup_spacing;
		MoveTo(sx, sy);
		DrawText(infobuf, 0, strlen(infobuf));
    	if (plan->maingoal) {
			strcpy(infobuf, "Goal: ");
	    		goal_desc(tmpbuf, plan->maingoal);
			strncat(infobuf, tmpbuf, strlen(tmpbuf));
			sy += closeup_spacing;			
			MoveTo(sx, sy);
			DrawText(infobuf, 0, strlen(infobuf));
    	}
    	if (plan->formation) {
			strcpy(infobuf, "Formation: ");
    			goal_desc(infobuf+strlen(infobuf), plan->formation);
			sy += closeup_spacing;			
			MoveTo(sx, sy);
			DrawText(infobuf, 0, strlen(infobuf));
    	}
		if (plan->tasks) {
			int i = 0;
	    		for_all_tasks(plan, task) {
				++i;
				sprintf(infobuf, "Task %d: ", i);
				task_desc(tmpbuf, unit->side, unit, task);			
				strcat(infobuf, tmpbuf);
				sy += closeup_spacing;			
				MoveTo(sx, sy);
				DrawText(infobuf, 0, strlen(infobuf));
			}
		}
	}
	SetCursor(&QD(arrow));
	draw_default_button(win, CancelButton);
}

/* First half of the old modal city_dialog. */

void
draw_advanced_closeup(UnitCloseup *closeup)
{
	int			run =0, width, height, downsx, downsy, downx, downy, sx, sy, m, d;
	short 		ditem, mitem, done = FALSE, disabled = TRUE, i, u, u2, a, x, y;
	char			buf[32], cname[32], uname[32], sname[32];
	MenuHandle	buildMenu, advanceMenu, planMenu;
	Rect 			itemrect, cityrect;
	Str255 		pname, pnumber;
	Handle 		itemhandle;  
	RGBColor		oldBack;
	Point			mouse;
	Unit 			*unit;
	Side			*side;
 	DialogPtr		win;
 	Map			*map;	

	win = closeup->window;
	unit = closeup->unit;
	side = unit->side;
	map = frontmap;
	
	/* Give controlling side (including debuggers 
	and designers) full control. */
	if (side_controls_unit(dside, unit))
		disabled = FALSE;

	/* Make sure visibility is up to date. */
	if (side) {
		for_all_cells_within_reach(unit, x, y) {
			if (terrain_visible(side, x, y)) {
				see_exact(side, x, y);
			}
		}
	}

	planMenu = GetMenu(mPlanTypes);
	buildMenu = build_construction_menu(unit);
	advanceMenu = build_research_menu(side);
	
	/* Get the type name of unit that is being constructed, if any. */
	if (unit->plan->tasks 
	      && unit->plan->tasks->type == TASK_BUILD
	      && is_unit_type(unit->plan->tasks->args[0])) {
	   	 strcpy(uname, u_type_name(unit->plan->tasks->args[0]));
	} else strcpy(uname, "Idle");

	/* Set build popup to uname. */
	m = CountMItems(buildMenu);
	GetDItem(win, diCityBuildPopup, NULL, &itemhandle, NULL);
	SetCtlMax((ControlHandle) itemhandle, m);
	for (i = 1; i <= m; i++) {
		GetItem(buildMenu, i, pname);
		p2c(pname, cname);
		if (strcmp(uname, cname) != 0)
			continue;
		SetCtlValue((ControlHandle) itemhandle, i);
	}

	/* Get the advance that is being researched by this unit, if any. */
	if (unit->curadvance != NOADVANCE) { 
	   	 strcpy(sname, a_type_name(unit->curadvance));
	} else strcpy(sname, "Idle"); 

	/* Set advance popup to sname. */
	m = CountMItems(advanceMenu);
	GetDItem(win, diCityAdvancePopup, NULL, &itemhandle, NULL);
	SetCtlMax((ControlHandle) itemhandle, m);
	for (i = 1; i <= m; i++) {
		GetItem(advanceMenu, i, pname);
		p2c(pname, cname);
		if (strcmp(sname, cname) != 0)
			continue;
		SetCtlValue((ControlHandle) itemhandle, i);
	}

	GetDItem(win, diCityMap, NULL, NULL, &itemrect);
	width = itemrect.right - itemrect.left;
	height = itemrect.bottom - itemrect.top;

	/* Save the Dialog background color. */
	oldBack = ((CGrafPtr) win)->rgbBkColor;
	BackColor(whiteColor);

	/* Draw background and frame. */
	InsetRect(&itemrect, -2, -2);
	RGBForeColor(&backcolor);
	PaintRect(&itemrect);
	PenSize(2, 2);
	ForeColor(blackColor);
	FrameRect(&itemrect);
	PenNormal();
	InsetRect(&itemrect, 2, 2);

	if (map) {
		/* Find the city position. */
		xform(map, unit->x, unit->y, &sx, &sy);

		/* Center cityrect around the city. */
		cityrect.left 	= sx + map->vp->hw/2 - width/2;
		cityrect.right 	= sx + map->vp->hw/2 + width/2;
		cityrect.top 	= sy + map->vp->hh/2 - height/2;
		cityrect.bottom 	= sy + map->vp->hh/2 + height/2;
		
		/* Find cityrect position in the offscreen gworld. */
		OffsetRect(&cityrect, map->offsetx + map->bufx - map->conw, 
					        map->offsety + map->bufy - map->toph);

		/* Copy cityrect from the offscreen gworld. */
		copy_rect_from_gworld(win, map, cityrect, itemrect);

		/* Handle cityrect part that extends beyond left end of gworld. */
		if (area.xwrap && cityrect.left < map->gworldPortPtr->portRect.left) {
			OffsetRect(&cityrect, map->vp->sxmax, 0);
			copy_rect_from_gworld(win, map, cityrect, itemrect);
		}
		/* Handle cityrect part that extends beyond right end of gworld. */
		if (area.xwrap && cityrect.right > map->gworldPortPtr->portRect.right) {
			OffsetRect(&cityrect, -map->vp->sxmax, 0);
			copy_rect_from_gworld(win, map, cityrect, itemrect);
		}
				
		/* Draw landuse within the cityrect. */
		d = max(width/map->vp->hw, height/map->vp->hh);	/* d could be smaller? */
		draw_landuse_near_unit(map, win, itemrect, unit, d);
	}

	/* Restore dialog background. */
	RGBBackColor(&oldBack);

	/*Hide "click map text from disabled sides. */
	if (disabled)
		HideDItem(win, diCityClickText);

	/* Set the AI control checkbox. */
	GetDItem(win, diCityAICheck, NULL, &itemhandle, NULL);
	SetCtlValue((ControlHandle) itemhandle, unit->plan->aicontrol);
	/* Don't allow disabled sides to use the AI box. */
	if (disabled)
		HiliteControl((ControlHandle) itemhandle, 255); 

	/* Set the plan checkbox. */
	GetDItem(win, diCityPlanCheck, NULL, &itemhandle, NULL);
	SetCtlValue((ControlHandle) itemhandle, unit->autoplan);

	/* Set the advance checkbox. */
	GetDItem(win, diCityAdvanceCheck, NULL, &itemhandle, NULL);
	SetCtlValue((ControlHandle) itemhandle, unit->autoresearch);

	/* Set the autobuild radio button. */
	GetDItem(win, diCityBuildCheck, NULL, &itemhandle, NULL);
	SetCtlValue((ControlHandle) itemhandle, unit->autobuild);

	/* Set the manual build radio button. */
	GetDItem(win, diCityBuildRepeat, NULL, &itemhandle, NULL);
	SetCtlValue((ControlHandle) itemhandle, !unit->autobuild);

	/* Draw the run in its textfield. */
	GetDItem(win, diCityBuildEdit, NULL, &itemhandle, NULL);
	if (unit->plan->tasks->args[3] > 0)
		NumToString(unit->plan->tasks->args[3], pname);
	else	NumToString(g_default_runlength(), pname);
	SetIText(itemhandle, pname);
	SelIText(win, diCityBuildEdit, 0, 32767);

	/* Set plan type popup menu to current plan. */
	GetDItem(win, diCityPlanPopup, NULL, &itemhandle, NULL);
	SetCtlValue((ControlHandle) itemhandle, unit->plan->type + 1);

	/* Also update the checkmarks, since this menu is used elsewhere. */
	CheckItem(planMenu, miPlanTypeNone, (unit->plan->type == PLAN_NONE));
	CheckItem(planMenu, miPlanTypePassive, (unit->plan->type == PLAN_PASSIVE));
	CheckItem(planMenu, miPlanTypeDefensive, (unit->plan->type == PLAN_DEFENSIVE));
	CheckItem(planMenu, miPlanTypeExploratory, (unit->plan->type == PLAN_EXPLORATORY));
	CheckItem(planMenu, miPlanTypeOffensive, (unit->plan->type == PLAN_OFFENSIVE));
	CheckItem(planMenu, miPlanTypeColonizing, (unit->plan->type == PLAN_COLONIZING));
	CheckItem(planMenu, miPlanTypeImproving, (unit->plan->type == PLAN_IMPROVING));
	CheckItem(planMenu, miPlanTypeRandom, (unit->plan->type == PLAN_RANDOM));

	/* Disable controls for disabled sides and when under active AI control. */
	if (disabled || (side_has_ai(side) && unit->plan->aicontrol)) {
		GetDItem(win, diCityPlanCheck, NULL, &itemhandle, NULL);
		HiliteControl((ControlHandle) itemhandle, 255); 
		GetDItem(win, diCityAdvanceCheck, NULL, &itemhandle, NULL);
		HiliteControl((ControlHandle) itemhandle, 255); 
		GetDItem(win, diCityBuildCheck, NULL, &itemhandle, NULL);
		HiliteControl((ControlHandle) itemhandle, 255); 
		GetDItem(win, diCityBuildRepeat, NULL, &itemhandle, NULL);
		HiliteControl((ControlHandle) itemhandle, 255); 
		GetDItem(win, diCityPlanPopup, NULL, &itemhandle, NULL);
		HiliteControl((ControlHandle) itemhandle, 255); 
		GetDItem(win, diCityAdvancePopup, NULL, &itemhandle, NULL);
		HiliteControl((ControlHandle) itemhandle, 255); 
		GetDItem(win, diCityBuildPopup, NULL, &itemhandle, NULL);
		HiliteControl((ControlHandle) itemhandle, 255); 
		/* Hide the runlength Edit field. */
		HideDItem(win, diCityBuildEdit);
		HideDItem(win, diCityBuildTimes);
	}
	
	/* Draw all the other stuff. */
	draw_closeup_content(win, unit);
	DrawDialog(win);

	SetCursor(&QD(arrow));
	draw_default_button(win, OkButton);
}

static void
copy_rect_from_gworld(WindowPtr win, Map *map, Rect cityrect, Rect itemrect)
{
	Rect	portrect = map->gworldPortPtr->portRect;
	Rect	destrect = itemrect;
	Rect	sourcerect;
	
	/* Clip sourcerect to portrect. */
	SectRect(&cityrect, &portrect, &sourcerect);

	/* Clip destrect so that it will match sourcerect. */ 
	destrect.top 	+= max(0, portrect.top - cityrect.top);
	destrect.bottom 	-= max(0, cityrect.bottom - portrect.bottom);
	destrect.left 	+= max(0, portrect.left - cityrect.left);
	destrect.right 	-= max(0, cityrect.right - portrect.right);

	/* Copy from the map gworld. */
	LockPixels(GetGWorldPixMap(map->gworldPortPtr));
	CopyBits(  &((GrafPtr) map->gworldPortPtr)->portBits,
			&((GrafPtr) win)->portBits,
			&sourcerect, &destrect, srcCopy, NULL  );
	UnlockPixels(GetGWorldPixMap(map->gworldPortPtr));
}		

void
draw_closeup_content(DialogPtr win, Unit *unit)
{
	int			occupants[MAX_DISPLAYED_OCCS] = {0};
	char			cname[32], sname[32], buf[BUFSIZE];
	RGBColor 		tmpcolor, oldBack;
	Str255 		pname, pnumber;
	Handle 		itemhandle;  
	CIconHandle 	cicnhandle;
	Rect			itemrect, tmprect;
	Unit			*unit2;
	int			i, m; 

	/* Draw the unit icon. */
	GetDItem(win, diCityIcon, NULL, NULL, &itemrect);
	draw_unit_image(win, itemrect.left, itemrect.top, 
				itemrect.right - itemrect.left, itemrect.bottom - itemrect.top,
				unit->type, side_number(unit->side), 
				(frontmap ? frontmap->sidecolors : default_sidecolors), 
				0, !completed(unit),
				(frontmap ? frontmap->draw_emblems : default_draw_emblems));
	/* Draw unit size on top of the icon. */
	if (default_drawsizes) 
		draw_unit_size(unit, win, itemrect.left, itemrect.top,
				itemrect.right - itemrect.left, itemrect.bottom - itemrect.top);

	/* Draw the unit's name, type and side. */
	GetDItem(win, diCityName, NULL, &itemhandle, NULL);
	if (indep(unit))
		strcpy(buf, "Independent");
	else	strcpy(buf, side_adjective(unit->side));
	strcat(buf, " ");
	strcat(buf, u_type_name(unit->type));
	strcat(buf, " ");
	strcat(buf, unit->name);
	c2p(buf, pname);
	SetIText(itemhandle, pname);

	/* Draw the unit's size. */
	GetDItem(win, diCitySize, NULL, &itemhandle, NULL);
	strcpy(buf, "Size : ");
	NumToString(unit->size, pnumber);
	p2c(pnumber, cname);
	strcat(buf, cname);
	c2p(buf, pname);
	SetIText(itemhandle, pname);

	/* Draw AI control info. */
	GetDItem(win, diCityStats, NULL, &itemhandle, NULL);
	strcpy(buf, "Under ");
	if (indep(unit) && unit->plan->aicontrol)
		strcat(buf, "brainless");
	else if (unit->side && side_has_ai(unit->side) && unit->plan->aicontrol)
		strcat(buf, unit->side->player->aitypename);
	else 	strcat(buf, "manual"); 
	strcat(buf, " control");
	c2p(buf, pname);
	SetIText(itemhandle, pname);

	/* Draw cyan background for materials panel. */ 
	GetDItem(win, diCityMatPanel, NULL, NULL, &itemrect);
	tmpcolor.red = 0xBFFF;
	tmpcolor.green = 0xFFFF; 
	tmpcolor.blue = 0xFFFF;
	RGBForeColor(&tmpcolor);
	FillRect(&itemrect, QDPat(black));
	ForeColor(blackColor);
	FrameRect(&itemrect);
	
	/* Draw green background for facilities panel. */ 
	GetDItem(win, diCityFacPanel, NULL, NULL, &itemrect);
	tmpcolor.red = 0xBFFF;
	tmpcolor.green = 0xFFFF; 
	tmpcolor.blue = 0xBFFF;
	RGBForeColor(&tmpcolor);
	FillRect(&itemrect, QDPat(black));
	ForeColor(blackColor);
	FrameRect(&itemrect);

	/* Draw pink background for garrison panel. */ 
	GetDItem(win, diCityOccPanel, NULL, NULL, &itemrect);
	tmpcolor.red = 0xFFFF;
	tmpcolor.green = 0xBFFF; 
	tmpcolor.blue = 0xFFFF; 
	RGBForeColor(&tmpcolor);
	FillRect(&itemrect, QDPat(black));
	ForeColor(blackColor);
	FrameRect(&itemrect);

	/* Draw statistics for material types 1 to 4. */
	for (m = 1; m < min(5, nummtypes); m++) {
		if (m_resource_icon(m)) {
			/* Plot material icon. */
			GetDItem(win, diCityMatBase + (5 * m), NULL, NULL, &itemrect);
			cicnhandle = GetCIcon(LARGE_CICNS + 10 * m_resource_icon(m) + 6);
			PlotCIcon(&itemrect, cicnhandle);
		}
		/* Plot material name. */
		GetDItem(win, diCityMatBase + (5 * m) + 1, NULL, &itemhandle, NULL);
		c2p(mtypes[m].name, pname);
		SetIText(itemhandle, pname);
		/* Plot material production. */
		GetDItem(win, diCityMatBase + (5 * m) + 2, NULL, &itemhandle, NULL);
		NumToString(unit->production[m], pnumber);
		SetIText(itemhandle, pnumber);
		/* Plot material supply. */
		GetDItem(win, diCityMatBase + (5 * m) + 3, NULL, &itemhandle, NULL);
		NumToString(unit->supply[m], pnumber);
		SetIText(itemhandle, pnumber);
		/* Plot global treasury supply. */
		GetDItem(win, diCityMatBase + (5 * m) + 4, NULL, &itemhandle, NULL);
		if (unit->side && m_treasury(m))
			NumToString(unit->side->treasury[m], pnumber);
		else	NumToString(0, pnumber);
		SetIText(itemhandle, pnumber);
	}

	/* Draw construction status. */
	if (unit->plan && unit->plan->tasks && unit->plan->tasks->type == TASK_BUILD) {
		GetDItem(win, diCityBuildStatus, NULL, &itemhandle, NULL);

		/* Print cps of current unit if it exists, else print "0". */
	    	if (find_unit(unit->plan->tasks->args[1]) != NULL) {
			 NumToString((find_unit(unit->plan->tasks->args[1]))->cp, pnumber);
			 p2c(pnumber, cname);
		} else	 strcpy(cname, "0");

		strcat(cname, " / ");
		NumToString(u_cp(unit->plan->tasks->args[0]), pnumber);
		p2c(pnumber, sname);
		strcat(cname, sname);
		
		/* Also print run progress. */
		if (unit->plan->tasks->args[3] > 1) {
			strcat(cname, "   ( ");
			NumToString(unit->plan->tasks->args[2] + 1, pnumber);
			p2c(pnumber, sname);
			strcat(cname, sname);
			strcat(cname, " of ");
			NumToString(unit->plan->tasks->args[3], pnumber);
			p2c(pnumber, sname);
			strcat(cname, sname);
			strcat(cname, " )");			
		}			
		c2p(cname, pnumber);
		SetIText(itemhandle, pnumber);
	}

	/* Draw research status. */
	if (unit->curadvance > 0 && unit->side) { 
		GetDItem(win, diCityResearchStatus, NULL, &itemhandle, NULL);
		NumToString(unit->side->advance[unit->curadvance], pnumber);
		p2c(pnumber, cname);
		strcat(cname, " / ");
		NumToString(a_rp(unit->curadvance), pnumber);
		p2c(pnumber, sname);
		strcat(cname, sname);
		c2p(cname, pnumber);
		SetIText(itemhandle, pnumber);
	}
	/* Sort the occupants. First load facilities. */
	i = 0;
	for_all_occupants(unit, unit2)
		if (alive(unit2) && u_facility(unit2->type))
			occupants[i++] = unit2->id;

	/* Move to the end of the facilities list. */
	i = MAX_DISPLAYED_FACS;

	/* Then load other units. */
	for_all_occupants(unit, unit2)
		if (alive(unit2) &! u_facility(unit2->type))
			occupants[i++] = unit2->id;

	/* Save the old Dialog background color. */
	oldBack = ((CGrafPtr) win)->rgbBkColor;
	BackColor(whiteColor);

	/* Draw all facilities and other occupants. */
	m = 0;
	for (i = 0; i < MAX_DISPLAYED_OCCS; i++) {
		unit2 = find_unit(occupants[i]);
		if (unit2) {
			/* First plot each unit icon. */
			GetDItem(win, diCityOccBase + 2 * m, 0, 0, &itemrect);
			draw_unit_image(win, itemrect.left, itemrect.top,
						itemrect.right - itemrect.left, itemrect.bottom - itemrect.top,
						unit2->type, side_number(unit2->side), 
						(frontmap ? frontmap->sidecolors : default_sidecolors), 
						0, !completed(unit2),
						(frontmap ? frontmap->draw_emblems : default_draw_emblems));
			/* Then plot each unit name. */
			GetDItem(win, diCityOccBase + 2 * m + 1, NULL, &itemhandle, NULL);
			if (unit2->name) {
			   	strcpy(buf, u_type_name(unit2->type));
			    	strcat(buf, " ");
			    	strcat(buf, unit2->name);
			/* Skip the unit number for facilities. */
			} else if (unit2->number > 0 &! u_facility(unit2->type)) {
			   	NumToString(unit2->number, pnumber);
				p2c(pnumber, cname);
		 	   	strcpy(buf, cname);
		 	    	strcat(buf, ordinal_suffix(unit2->number));
			    	strcat(buf, " ");
		 	    	strcat(buf, u_type_name(unit2->type));
			} else	strcpy(buf, u_type_name(unit2->type));

			/* Write out cps for each unit if debugging. */
			if (Debug || DebugG || DebugM) {
				strcat(buf, " ");
				NumToString(unit2->cp, pnumber);
				p2c(pnumber, cname);
				strcat(buf, cname);
				strcat(buf, "/");
				NumToString(u_cp(unit2->type), pnumber);
				p2c(pnumber, cname);
				strcat(buf, cname);
			}
			c2p(buf, pname);
			SetIText(itemhandle, pname);
		} else {
			GetDItem(win, diCityOccBase + 2 * m + 1, NULL, &itemhandle, NULL);
			c2p("", pname);
			SetIText(itemhandle, pname);
		}
		m++;
	}
	/* Restore dialog background. */
	RGBBackColor(&oldBack);
}

/* 
	Draw the size number of a city on top of its unit icon. 
*/

void
draw_unit_size(Unit *unit, WindowPtr win, int sx, int sy, int sw, int sh)
{
	UnitCloseup	*closeup;
	Map 			*map;
	List			*list;
	Str255  		pnumber;
	short 		curfont, cursize, curstyle;
	int			e;

	/* Filter out very small images. */
	if (sw < 16)
		return; 
	/* NULL pointer check. */
	if (!unit || !unit->size)
		return;
		 
	/* Find current closeup, map or list. */
	closeup = unit_closeup_from_window(win);
	map = map_from_window(win);
	list = list_from_window(win);
	e = side_number(unit->side);
		
	/* Save the current font. */
	curfont = QD(thePort)->txFont;
	cursize = QD(thePort)->txSize;
	curstyle = QD(thePort)->txFace;

	/* Default colors used in the absence of defined side colors. */
	RGBForeColor(&forecolor);
	RGBBackColor(&maskcolor);

		/* Load city size into pascal string. */
		NumToString(unit->size, pnumber);

		/* Use Chicago. */
		TextFont(0);

		/* Use fixed optimized fonts if asked to do so. */
	if ((map && map->optimize_fonts)
	    || (list && default_optimize_fonts)
	    || (closeup && default_optimize_fonts)) {

			switch (sh) {
				case 16:	TextSize(9); break;
				case 32:	TextSize(12); break;
				case 64:	TextSize(24); break;
				case 128:	TextSize(48); break;
				default:	TextSize(9); break;
			}

		/* Else scale text sizes according to formula. */
		} else TextSize(min(max(9, sh/2), 48));

		/* First use the mask color to plot the core. */
	if (((map && map->sidecolors) 
	    || (list && list->sidecolors)
	    || (closeup && default_sidecolors)) 
		    && icon_mask_color[e]) {
		  	RGBForeColor(&(get_sideColor(e, icon_mask_color[e])));
		} else	RGBForeColor(&maskcolor);
		TextFace(bold + condense);

		/* Handtuned setting for 16 x 16. */
		if (sh < 32) MoveTo(sx + 1, sy + 10);
		else MoveTo(sx + sw/5, sy + 3 * sh/5);
		DrawString(pnumber);

		/* Then use the main color to plot the outline. */
	if (((map && map->sidecolors) 
	    || (list && list->sidecolors) 
	    || (closeup && default_sidecolors)) 
		    && main_icon_color[e]) {
		  	RGBForeColor(&(get_sideColor(e, main_icon_color[e])));
		} else	RGBForeColor(&forecolor);
		TextFace(shadow + condense);

		/* Handtuned setting for 16 x 16. */
		if (sh < 32) MoveTo(sx + 1, sy + 10);
		else MoveTo(sx + sw/5, sy + 3 * sh/5);
		DrawString(pnumber);

	/* Restore the current font. */
	TextFont(curfont);
	TextSize(cursize);
	TextFace(curstyle);

	/* Restore colors. */
	ForeColor(blackColor);
	BackColor(whiteColor);
}

/* 
	Draw landuse in all cells near unit. Assumes the port is already set. 
*/

static void
draw_landuse_near_unit(Map *map, WindowPtr win, Rect itemrect, Unit *unit, int near)
{
	int 			mapsx, mapsy, winsx, winsy, sx, sy, x, y;
	Rect 			imagerect;
	Str255		pnumber;
	RgnHandle 		tmprgn;

	/* Return if landuse is not defined. */
	if (!user_defined())
		return;

	/* Clip to itemrect. */
	tmprgn = NewRgn();
	GetClip(tmprgn);
	ClipRect(&itemrect);

	/* Find win coordinates of unit (city) cell. */ 
	winsx = (itemrect.right + itemrect.left) / 2 - map->vp->hw/2;
	winsy = (itemrect.bottom + itemrect.top) / 2 - map->vp->hh/2;
	
	/* Find map coordinates of unit (city) cell. */
	xform(map, unit->x, unit->y, &mapsx, &mapsy);
	
	/* Zero the number of used cells. */
	unit->usedcells = 0;
	/* Go through all cells within "near" steps from unit. */
	for_all_cells_within_range(unit->x, unit->y, near, x, y) {
		/* Skip if cell is unused. */
		if (user_at(x, y) == NOUSER)
			continue;
		/* Skip if cell is outside area. */
		if (!inside_area(x, y))
			continue;
		/* Skip if the cell is not visible to the unit's side. */
		if (!terrain_visible(unit->side, x, y))
			continue;
		/* Recalculate the number of used cells. */
		if (user_at(x, y) == unit->id)
			unit->usedcells += 1;
		/* Find map coordinates of cell. */
		xform(map, x, y, &sx, &sy);
		/* Add win-map offset. */
		sx += winsx - mapsx;
		sy += winsy - mapsy;
		/* Adjust to unit part of cell. */
		sx += (map->vp->hw - map->vp->uw) / 2;  
		sy += (map->vp->hh - map->vp->uh) / 2;
		/* Set imagerect. */
		SetRect(&imagerect, sx, sy, sx + map->vp->uw, sy + map->vp->uh);
		/* Draw a colored square if cell is used by another unit. */
		if (find_unit(user_at(x, y)) && user_at(x, y) != unit->id) {
			int e = side_number(find_unit(user_at(x, y))->side);
			PenSize(2, 2);
			RGBForeColor(&(get_sideColor(e, main_icon_color[e])));
			FrameRect(&imagerect);
			PenNormal();
			ForeColor(blackColor);
		/* Else plot the resource cicns for that cell. */
		} else plot_resource_cicns(map, imagerect, x, y);
	}
	/* Restore old clip. */
	SetClip(tmprgn);
	DisposeRgn(tmprgn);
}

/* 
	Plot resource cicns in single cell. Assumes that the port already is set.
	Max 4 material types can be displayed at the same time. The icon used for
	each material type is determined by the mtype property m_resource_icon. 
*/

void
plot_resource_cicns(Map *map, Rect plotrect, int x, int y)
{
	CIconHandle cicnhandle;
	int m, n, i;

	n = (map->vp->uh > 16 ? LARGE_CICNS : SMALL_CICNS);
	for_all_material_types(m) {
		i = m_resource_icon(m);
		if (i) {
			cicnhandle = GetCIcon(n + 10 * i + tm_production(terrain_at(x, y), m));
			PlotCIcon(&plotrect, cicnhandle);
		}
	}
}

int
do_mouse_down_unit_closeup(UnitCloseup *unitcloseup, Point mouse, int mods)
{
	ControlHandle control;
	short part;
	WindowPtr window = unitcloseup->window;

	part = FindControl(mouse, window, &control);
	if (0 /* some control */) {
	} else {
	
		Unit	*unit;
		Map	*map;

		unit = unitcloseup->unit;
		for_all_maps(map) {
			unselect_all(map);
			select_unit_on_map(map, unit);
		}
		update_cell_display(dside, unit->x, unit->y, UPDATE_ALWAYS);
		return TRUE;
	}
}

/* This function examines keystrokes destined for closeup dialogs. 
It returns DONE (-1) for keys that got handled, TRUE for keys that
were not handled now but should be handled by the dialog manager,
and FALSE for all other keys. */ 

int 
do_key_down_closeup(UnitCloseup *closeup, char key)	/* Note: keyCode, not charCode! */
{
	Point mouse = {0, 0};		/* Dummy variable. */
	int	advanced;
	
	advanced = u_advanced(closeup->unit->type);

	/* We hit the Enter or Return key. */
	if (key == 0x4C || key == 0x24) {
		if (advanced)
			hit_closeup_dialog(closeup->window, OkButton, mouse);
		else	close_window(closeup->window);
		return DONE;
	/* We hit the Escape key. */
	} else if (key == 0x35) {
		close_window(closeup->window);
		return DONE;
	/* 0-9 should be forwarded to the dialog's Edit text field instead of 
	being interpreted as command prefixes. Also forward backspace. */
	} else if (advanced 
		    && (between(0x12, key, 0x17)	/* Keyboard 1-6 */ 
		    	|| key == 0x19				/* Keyboard 9 */ 
		    	|| key == 0x1A				/* Keyboard 7 */ 
		    	|| key == 0x1C				/* Keyboard 8 */ 
		    	|| key == 0x1D				/* Keyboard 0 */ 
		    	|| key == 0x33)) {			/* Backspace   */ 
		return TRUE;
	}
	return FALSE;
}

/* Second half of the old modal city_dialog. */

void
hit_closeup_dialog(WindowPtr win, int ditem, Point mouse)
{
	int			run =0, width, height, downsx, downsy, downx, downy, sx, sy, m;
	short			aicontrol, autoplan, autoresearch, autobuild, plantype, advance, build;
	short 		mitem, done = FALSE, disabled = TRUE, i, u, u2, a;
	char			buf[32], cname[32], uname[32], sname[32];
	MenuHandle	buildMenu, advanceMenu;
	Str255 		pname, pnumber;
	UnitCloseup	*unitcloseup;
	HelpNode 		*helpnode;
	Handle 		itemhandle;  
	Rect 			itemrect;
	GrafPtr 		oldport;
	Map			*map;
	Unit 			*unit;
	Side			*side;

	unitcloseup = unit_closeup_from_window(win);
	unit = unitcloseup->unit;
	side = unit->side;
	map = frontmap;
	buildMenu = build_construction_menu(unit);
	advanceMenu = build_research_menu(side);

	/* Give controlling side (including debuggers 
	and designers) full control. */
	if (side_controls_unit(dside, unit))
		disabled = FALSE;

	GetPort(&oldport);	
	SetPort(win);
	DrawDialog(win);

	switch (ditem) {

	case 	diCityMap:		/* Find the cell that was clicked. */
		if (disabled)	/* Don't allow other sides to change things. */	
			break;
		GetDItem(win, diCityMap, NULL, NULL, &itemrect);
		width = itemrect.right - itemrect.left;
		height = itemrect.bottom - itemrect.top;
		/* Find the city position. */
		xform(map, unit->x, unit->y, &sx, &sy);
		GlobalToLocal(&mouse);
		downsx = mouse.h - itemrect.left + sx + map->vp->hw/2 - width/2;
		downsy = mouse.v - itemrect.top + sy + map->vp->hh/2 - height/2;
		m_nearest_cell(map, downsx, downsy, &downx, &downy);
		if (!inside_area(downx, downy)) 
			break;
		if (!cell_is_within_reach(unit, downx, downy))
			break;
		/* Toggle landuse on or off for the cell. */
		toggle_landuse_one_cell(map, win, itemrect, unit, downx, downy);
	break;

	case diCityAICheck: 	/* Toggle AI control checkbox. */
		if (disabled)	/* Don't allow other sides to change things. */	
			break;
		GetDItem(win, ditem, NULL, &itemhandle, NULL);
		SetCtlValue((ControlHandle) itemhandle, !GetCtlValue((ControlHandle) itemhandle));
		if (side_has_ai(side) && GetCtlValue((ControlHandle) itemhandle)) {
			/* Disable controls. */
			GetDItem(win, diCityPlanCheck, NULL, &itemhandle, NULL);
			HiliteControl((ControlHandle) itemhandle, 255); 
			GetDItem(win, diCityAdvanceCheck, NULL, &itemhandle, NULL);
			HiliteControl((ControlHandle) itemhandle, 255); 
			GetDItem(win, diCityBuildCheck, NULL, &itemhandle, NULL);
			HiliteControl((ControlHandle) itemhandle, 255); 
			GetDItem(win, diCityBuildRepeat, NULL, &itemhandle, NULL);
			HiliteControl((ControlHandle) itemhandle, 255); 
			GetDItem(win, diCityPlanPopup, NULL, &itemhandle, NULL);
			HiliteControl((ControlHandle) itemhandle, 255); 
			GetDItem(win, diCityAdvancePopup, NULL, &itemhandle, NULL);
			HiliteControl((ControlHandle) itemhandle, 255); 
			GetDItem(win, diCityBuildPopup, NULL, &itemhandle, NULL);
			HiliteControl((ControlHandle) itemhandle, 255); 
			/* Hide the runlength Edit field. */
			HideDItem(win, diCityBuildEdit);
			HideDItem(win, diCityBuildTimes);
			/* Update AI control info. */
			GetDItem(win, diCityStats, NULL, &itemhandle, NULL);
			strcpy(buf, "Under ");
			strcat(buf, side->player->aitypename);
			strcat(buf, " control");
			c2p(buf, pname);
			SetIText(itemhandle, pname);
		} else {
			/* Enable controls. */
			GetDItem(win, diCityPlanCheck, NULL, &itemhandle, NULL);
			HiliteControl((ControlHandle) itemhandle, 0); 
			GetDItem(win, diCityAdvanceCheck, NULL, &itemhandle, NULL);
			HiliteControl((ControlHandle) itemhandle, 0); 
			GetDItem(win, diCityBuildCheck, NULL, &itemhandle, NULL);
			HiliteControl((ControlHandle) itemhandle, 0); 
			GetDItem(win, diCityBuildRepeat, NULL, &itemhandle, NULL);
			HiliteControl((ControlHandle) itemhandle, 0); 
			GetDItem(win, diCityPlanPopup, NULL, &itemhandle, NULL);
			HiliteControl((ControlHandle) itemhandle, 0); 
			GetDItem(win, diCityAdvancePopup, NULL, &itemhandle, NULL);
			HiliteControl((ControlHandle) itemhandle, 0); 
			GetDItem(win, diCityBuildPopup, NULL, &itemhandle, NULL);
			HiliteControl((ControlHandle) itemhandle, 0); 
			/* Show the runlength Edit field. */
			ShowDItem(win, diCityBuildEdit);
			ShowDItem(win, diCityBuildTimes);
			/* Update AI control info. */
			GetDItem(win, diCityStats, NULL, &itemhandle, NULL);
			strcpy(buf, "Under manual control");
			c2p(buf, pname);
			SetIText(itemhandle, pname);
		}
	break;

	case diCityPlanCheck: 		/* Toggle autoplan checkbox. */
		if (disabled)	/* Don't allow other sides to change things. */	
			break;
		GetDItem(win, ditem, NULL, &itemhandle, NULL);
		SetCtlValue((ControlHandle) itemhandle, !GetCtlValue((ControlHandle) itemhandle));
	break;

	case diCityAdvanceCheck: 	/* Toggle autoresearch checkbox. */
		if (disabled)	/* Don't allow other sides to change things. */	
			break;
		GetDItem(win, ditem, NULL, &itemhandle, NULL);
		SetCtlValue((ControlHandle) itemhandle, !GetCtlValue((ControlHandle) itemhandle));
	break;

	case diCityBuildCheck: 	/* Switch autobuild button. */
		if (disabled)	/* Don't allow other sides to change things. */	
			break;
		GetDItem(win, ditem, NULL, &itemhandle, NULL);
		SetCtlValue((ControlHandle) itemhandle, !GetCtlValue((ControlHandle) itemhandle));
		GetDItem(win, diCityBuildRepeat, NULL, &itemhandle, NULL);
		SetCtlValue((ControlHandle) itemhandle, !GetCtlValue((ControlHandle) itemhandle));
	break;

	case diCityBuildRepeat: 	/* Switch manual build button. */
		if (disabled)	/* Don't allow other sides to change things. */	
			break;
		GetDItem(win, ditem, NULL, &itemhandle, NULL);
		SetCtlValue((ControlHandle) itemhandle, !GetCtlValue((ControlHandle) itemhandle));
		GetDItem(win, diCityBuildCheck, NULL, &itemhandle, NULL);
		SetCtlValue((ControlHandle) itemhandle, !GetCtlValue((ControlHandle) itemhandle));
	break;

	case 	OkButton:
		/* Don't allow others to save any changes. */
		if (disabled) {
			done = TRUE;
			break;
		}
		/* Get the AI control state. */
		GetDItem(win, diCityAICheck, NULL, &itemhandle, NULL);
		aicontrol = GetCtlValue((ControlHandle) itemhandle);

		/* Get the autoplan state. */
		GetDItem(win, diCityPlanCheck, NULL, &itemhandle, NULL);
		autoplan = GetCtlValue((ControlHandle) itemhandle);

		/* Get the autoresearch state. */
		GetDItem(win, diCityAdvanceCheck, NULL, &itemhandle, NULL);
		autoresearch = GetCtlValue((ControlHandle) itemhandle);

		/* Get the autobuild state. */
		GetDItem(win, diCityBuildCheck, NULL, &itemhandle, NULL);
		autobuild = GetCtlValue((ControlHandle) itemhandle);

		/* Get the selected plan type. */
		GetDItem(win, diCityPlanPopup, NULL, &itemhandle, NULL);
		plantype = GetCtlValue((ControlHandle) itemhandle) - 1;

		/* Get the selected research task. */
		GetDItem(win, diCityAdvancePopup, NULL, &itemhandle, NULL);
		mitem = GetCtlValue((ControlHandle) itemhandle);
		GetItem(advanceMenu, mitem, pname);
		p2c(pname, sname);
		advance = NOADVANCE;
		for_all_advance_types(a) {
			if (strcmp(a_type_name(a), sname) == NULL) {
				advance = a;
				break;
			}
		}

		/* Always set runlength to default if on autobuild. */
		if (unit->autobuild 
		    || (side && side_has_ai(side) && unit->plan->aicontrol))
			run = g_default_runlength();
		else {					
			/* Else get the runlength from its text field. */
			GetDItem(win, diCityBuildEdit, NULL, &itemhandle, NULL);
        			GetIText(itemhandle, pname);
			StringToNum(pname, (long *) &run);
			run = min(max(run, 0), CLEAR_AGENDA);
		}

		/* Get the selected build task. */
		GetDItem(win, diCityBuildPopup, NULL, &itemhandle, NULL);
		mitem = GetCtlValue((ControlHandle) itemhandle);
		GetItem(buildMenu, mitem, pname);
		p2c(pname, uname);
		build = NONUTYPE;
		for_all_unit_types(u) {
			/* Find the utype that was selected in the menu. */
			if (strcmp(u_type_name(u), uname) == NULL) {
				build = u;
				break;				
			}
		}

		/* We need to do this last of all since the net_set functions call
		update_unit_display which redraws the closeup itself. The first
		call to a net_set function therefore resets all popup menus to the
		initial values, thus destroying the settings we want to save. */

		net_set_unit_ai_control(side, unit, aicontrol, FALSE);
		net_set_unit_autoplan(side, unit, autoplan);
		net_set_unit_autoresearch(side, unit, autoresearch);
		net_set_unit_autobuild(side, unit, autobuild);
		net_set_unit_plan_type(side, unit, plantype);
		net_set_unit_curadvance(side, unit, advance);
		if (build == NONUTYPE) {
			/* No matching utype means "Idle" was selected. */
			net_clear_task_agenda(side, unit);
			net_set_unit_asleep(side, unit, TRUE, FALSE);
		} else	net_set_build_task(unit, build, run, 0, 0);

		done = TRUE;
	break;
	
	case CancelButton:
		done = TRUE;
	break;

	case HelpButton:		
		helpnode = find_help_node(first_help_node, u_type_name(unit->type));
		if (helpnode)
			show_help_window(helpnode);
		else 	beep();
	break;
		
	default:
	break;
	}

	if (done)
	    close_window(win);

	/* Restore old port. */
	SetPort(oldport);
}

/* 
	Toggle landuse for a single cell. Assumes the port already is set. 
*/

void
toggle_landuse_one_cell(Map *map, WindowPtr win, Rect itemrect, Unit *unit, int x, int y)
{
	int 			mapsx, mapsy, winsx, winsy, sx, sy, nearx, neary;
	Rect 			maprect, winrect;
	RgnHandle		tmprgn;

	/* Return if landuse is undefined. */
	if (!user_defined())
		return;
	/* Return if the cell is not visible to this side. */
	if (!terrain_visible(unit->side, x, y))
		return;
	/* Return if the cell is used by another unit. */
	if (user_at(x, y) != NOUSER && user_at(x, y) != unit->id)
		return;
	/* Return if using maxcells and we are trying to add one more. */
	if (unit->usedcells >= unit->maxcells && user_at(x, y) != unit->id)
		return;
	/* Return if independents or untrusted side has a unit in the cell. */
	if (unit_at(x, y) != NULL) {
		Unit *unit2;
		for_all_stack(x, y, unit2)
			if (!trusted_side(unit->side, unit2->side))
				return;
	}
	/* Toggle landuse by unit either on or off for the cell. */
	if (user_at(x, y) == NOUSER) {
		set_user_at(x, y, unit->id);
		unit->usedcells += 1;
	} else if (user_at(x, y) == unit->id) {
		set_user_at(x, y, NOUSER);
		unit->usedcells -= 1;
	} else return;
	
	/* Clip to itemrect. */
	tmprgn = NewRgn();
	GetClip(tmprgn);
	ClipRect(&itemrect);

	/* Find the cell position. */
	xform(map, x, y, &sx, &sy);

	/* Adjust to unit part of cell. */
	sx += (map->vp->hw - map->vp->uw) / 2;  
	sy += (map->vp->hh - map->vp->uh) / 2;

	/* Find cell rect position in the map. */
	SetRect(&maprect, sx, sy, sx + map->vp->uw, sy + map->vp->uh);

	/* Find cell rect position in the offscreen gworld. */
	OffsetRect(&maprect, map->offsetx + map->bufx - map->conw, 
				        map->offsety + map->bufy - map->toph);

	/* Find win coordinates of unit (city) cell. */ 
	winsx = (itemrect.right + itemrect.left) / 2 - map->vp->hw/2;
	winsy = (itemrect.bottom + itemrect.top) / 2 - map->vp->hh/2;
	
	/* Find map coordinates of unit (city) cell. */
	xform(map, unit->x, unit->y, &mapsx, &mapsy);
	
	/* Add win-map offset. */
	sx += winsx - mapsx;
	sy += winsy - mapsy;

	/* Find cell rect position in the dialog window. */
	SetRect(&winrect, sx, sy, sx + map->vp->uw, sy + map->vp->uh);

	/* Copy cellrect from gworld to dialog window. */
	LockPixels(GetGWorldPixMap(map->gworldPortPtr));
	CopyBits(  &((GrafPtr) map->gworldPortPtr)->portBits,
			&((GrafPtr) win)->portBits,
			&maprect, &winrect, srcCopy, NULL  );
	UnlockPixels(GetGWorldPixMap(map->gworldPortPtr));

	/* Plot resource cicns on top if cell is used by city. */
	if (user_at(x, y) == unit->id) 
		plot_resource_cicns(map, winrect, x, y);

	/* Restore old clip. */
	SetClip(tmprgn);
	DisposeRgn(tmprgn);
}

void
destroy_unit_closeup(UnitCloseup *unitcloseup)
{
	UnitCloseup *unitcloseup2;
	Unit	*unit;
	Map	*map;

	unit = unitcloseup->unit;
	update_cell_display(dside, unit->x, unit->y, UPDATE_ALWAYS);
	if (unitcloseuplist == unitcloseup) {
		unitcloseuplist = unitcloseup->next;
	} else {
		for_all_unit_closeups(unitcloseup2) {
			if (unitcloseup2->next == unitcloseup) {
				unitcloseup2->next = unitcloseup->next;
			}
		}
	}
	/* (should destroy substructs) */
	free(unitcloseup);
}

/*
	Research dialog for a single advanced unit. 
*/

void
unit_research_dialog(Unit *unit)
{
	short 		ditem, mitem, done = FALSE, i, m, d, s;
	char			cname[32], sname[32], buf[BUFSIZE];
	Str255 		pname, pnumber;
	Handle 		itemhandle; 
	MenuHandle	popMenu;
	GrafPtr 		oldport;
 	DialogPtr		win;
	
	/* Open the dialog. */
	win = GetNewDialog(dAdvance, NULL, (DialogPtr) -1);
	GetPort(&oldport);	
	SetPort(win);

	/* Hide all scientists. */
	HideDItem(win, diAdvanceMadScientist);
	HideDItem(win, diAdvanceBusyScientist);
	HideDItem(win, diAdvanceIdleScientist);
	popMenu = GetMenu(mResearchPopup);

	s = unit->curadvance;
	/* Research is going on. */
	if (s >= 0) { 
	   	strcpy(sname, a_type_name(s));
		/* Check if current advance has been completed. */
		if (has_advance(unit->side, s)) {
			/* Do bells and whistles. */
			strcpy(buf, "Scientists in ");
			strcat(buf, unit->name);
			strcat(buf, " discover ");
			strcat(buf, sname);
			strcat(buf, "!\r\r");
			strcat(buf, "Pick new research:");
			/* Show mad scientist. */
			ShowDItem(win, diAdvanceMadScientist);
			/* Only do it once. */
				unit->curadvance = NOADVANCE;
		} else {
			/* Report ongoing research. */
			strcpy(buf, "The scientists in ");
			strcat( buf, unit->name);
			strcat( buf, " are researching ");
			strcat(buf, sname);
			strcat(buf, ".");			
			/* Show busy scientist. */
			ShowDItem(win, diAdvanceBusyScientist);
		}
	/* No current research. */
	} else {
		strcpy(buf, "The scientists in ");
		strcat( buf, unit->name);
		strcat( buf, " are idle.\r");
		strcat( buf, "You should put them to work!\r\r");
		strcat( buf, "Pick new research:");
		/* Show idle scientist. */
		ShowDItem(win, diAdvanceIdleScientist);
	}
	c2p(buf, pname);
	GetDItem(win, diAdvanceMainText, NULL, &itemhandle, NULL);
	SetIText(itemhandle, pname);
	/* Set the advance auto checkbox. */
	GetDItem(win, diAdvanceCheck, NULL, &itemhandle, NULL);
	SetCtlValue((ControlHandle) itemhandle, unit->autoresearch);
	/* Load researchable advance types into popup menu. */
	for_all_advance_types(s) {
		if (has_advance_to_research(unit->side, s)
		    /* Skip advances already done. */
		    && unit->side->advance[s] != DONE) {
   			c2p(a_type_name(s), pname);
			AppendMenu(popMenu, pname);
		}
	}

	ShowWindow(win);
	while (!done) {
		SetCursor(&QD(arrow));
		draw_default_button(win, OkButton);

		/* Set popup to sname. */
		m = CountMItems(popMenu);
		GetDItem(win, diAdvancePopup, NULL, &itemhandle, NULL);
		SetCtlMax((ControlHandle) itemhandle, m);	/* Important! */
		for (i = 1; i <= m; i++) {
			GetItem(popMenu, i, pname);
			p2c(pname, cname);
			if (strcmp(sname, cname) != 0)
				continue;
			SetCtlValue((ControlHandle) itemhandle, i);
		}

		ModalDialog(NULL, &ditem);
		switch (ditem) {
			case diAdvanceCheck: /* Toggle advance auto checkbox. */
				GetDItem(win, ditem, NULL, &itemhandle, NULL);
				SetCtlValue((ControlHandle) itemhandle, !GetCtlValue((ControlHandle) itemhandle));
				break;

			case 	diAdvancePopup: /* Current research popup menu. */
				GetDItem(win, ditem, NULL, &itemhandle, NULL);
				mitem = GetCtlValue((ControlHandle) itemhandle);
				GetItem(popMenu, mitem, pname);
				p2c(pname, sname);
		    		break;

			case 	OkButton:
				/* Get the advance auto checkbox state. */
				GetDItem(win, diAdvanceCheck, NULL, &itemhandle, NULL);
				unit->autoresearch = GetCtlValue((ControlHandle) itemhandle);
				/* Stop any research if "Idle" was selected. */
				if (strcmp("Idle", sname) == 0) {
					unit->curadvance = NOADVANCE;
				/* Else select new advance to research. */
				} else for_all_advance_types(s) {
					/* Skip until we find atype that is selected in popMenu. */
					if (strcmp(a_type_name(s), sname) != 0)
						continue;
					unit->curadvance = s;
					}
				done = TRUE;
				break;

			case 	CancelButton:
				done = TRUE;
				break;

			default:
				break;
		}
	}

	/* Close the dialog. */
	DisposeDialog(win);
	/* Restore old port. */
	SetPort(oldport);
}


/*
	Build dialog for a single advanced unit. 
*/

int
unit_build_dialog(Unit *unit)
{
	short 		ditem, mitem, done = FALSE, i, m, d, u, u2;
	char			cname[32], uname[32], buf[BUFSIZE];
	Task 			*lasttask = &(unit->plan->last_task);
	Str255 		pname, pnumber;
	Handle 		itemhandle; 
	MenuHandle	popMenu;
	GrafPtr 		oldport;
 	DialogPtr		win;
	int			run;
	
	/* Open the dialog. */
	win = GetNewDialog(dBuild, NULL, (DialogPtr) -1);
	GetPort(&oldport);	
	SetPort(win);

	/* Hide all scientists. */
	HideDItem(win, diBuildMadScientist);
	HideDItem(win, diBuildBusyScientist);
	HideDItem(win, diBuildIdleScientist);

	/* Check if a real building task just was completed. */
	if (lasttask->type == TASK_BUILD
	     && unit->plan->last_task_outcome == TASK_IS_COMPLETE
	     && lasttask->args[3] > 0) {
	   	NumToString(lasttask->args[3], pnumber);
		p2c(pnumber, cname);
		if (lasttask->args[3] > 1) {
 	   		 strcpy(uname, cname);
	    		 strcat(uname, " ");
			 strcat(uname, plural_form(u_type_name(lasttask->args[0])));
		} else strcpy(uname, u_type_name(lasttask->args[0]));
		strcpy(buf, "Workers in ");
		strcat(buf, unit->name);
		strcat(buf, " complete ");
		strcat(buf, uname);
		strcat(buf, "!\r\r");
		strcat(buf, "Pick new building task:");
		/* Show mad scientist. */
		ShowDItem(win, diBuildMadScientist);
		/* Set menu task to Idle. */
		strcpy(uname, "Idle");
	/* We are not building anything at all. */
	} else {
		strcpy(uname, "Idle");
		strcpy(buf, "The workers in ");
		strcat( buf, unit->name);
		strcat( buf, " are idle.\r");
		strcat( buf, "You should put them to work!\r\r");
		strcat( buf, "Pick new building task:");
		/* Show idle scientist. */
		ShowDItem(win, diBuildIdleScientist);
	}
	c2p(buf, pname);
	GetDItem(win, diBuildMainText, NULL, &itemhandle, NULL);
	SetIText(itemhandle, pname);

	/* Set the autobuild radio button. */
	GetDItem(win, diBuildCheck, NULL, &itemhandle, NULL);
	SetCtlValue((ControlHandle) itemhandle, unit->autobuild);

	/* Set the manual build radio button. */
	GetDItem(win, diBuildStatus, NULL, &itemhandle, NULL);
	SetCtlValue((ControlHandle) itemhandle, !unit->autobuild);

	/* Draw the run in its textfield. */
	GetDItem(win, diBuildEdit, NULL, &itemhandle, NULL);
	if (lasttask->args[3] > 0)
		NumToString(lasttask->args[3], pname);
	else	NumToString(g_default_runlength(), pname);
	SetIText(itemhandle, pname);
	SelIText(win, diBuildEdit, 0, 32767);

	/* Load constructable unit type names into popup menu. */
	popMenu = GetMenu(mBuildPopup);
	for_all_unit_types(u) {
		if (type_allowed_on_side(u, unit->side)
		    && has_advance_to_build(unit->side, u)
		    && could_create(unit->type, u)
		    && (type_can_occupy(u, unit)
		    	  || u == unit->plan->tasks->args[0])) {
	   		c2p(u_type_name(u), pname);
			AppendMenu(popMenu, pname);
		}
	}
	ShowWindow(win);
	while (!done) {
		SetCursor(&QD(arrow));
		draw_default_button(win, OkButton);

		/* Set build popup to uname. */
		m = CountMItems(popMenu);
		GetDItem(win, diBuildPopup, NULL, &itemhandle, NULL);
		SetCtlMax((ControlHandle) itemhandle, m);
		for (i = 1; i <= m; i++) {
			GetItem(popMenu, i, pname);
			p2c(pname, cname);
			if (strcmp(uname, cname) != 0)
				continue;
			SetCtlValue((ControlHandle) itemhandle, i);
		}

		ModalDialog(NULL, &ditem);
		switch (ditem) {

			case diBuildCheck: 	/* Toggle autobuild and manual build buttons. */
				GetDItem(win, ditem, NULL, &itemhandle, NULL);
				SetCtlValue((ControlHandle) itemhandle, !GetCtlValue((ControlHandle) itemhandle));
				GetDItem(win, diBuildStatus, NULL, &itemhandle, NULL);
				SetCtlValue((ControlHandle) itemhandle, !GetCtlValue((ControlHandle) itemhandle));
				break;

			case diBuildStatus: 	/* Toggle autobuild and manual build buttons. */
				GetDItem(win, ditem, NULL, &itemhandle, NULL);
				SetCtlValue((ControlHandle) itemhandle, !GetCtlValue((ControlHandle) itemhandle));
				GetDItem(win, diBuildCheck, NULL, &itemhandle, NULL);
				SetCtlValue((ControlHandle) itemhandle, !GetCtlValue((ControlHandle) itemhandle));
				break;


			case 	diBuildPopup: 	/* Find selected unit type. */
				GetDItem(win, ditem, NULL, &itemhandle, NULL);
				mitem = GetCtlValue((ControlHandle) itemhandle);
				GetItem(popMenu, mitem, pname);
				p2c(pname, uname);
		    		break;

			case 	OkButton:
				/* Save the autobuild state. */
				GetDItem(win, diBuildCheck, NULL, &itemhandle, NULL);
				unit->autobuild = GetCtlValue((ControlHandle) itemhandle);

				/* Get the runlength from its text field. */
				GetDItem(win, diBuildEdit, NULL, &itemhandle, NULL);
	        			GetIText(itemhandle, pname);
				StringToNum(pname, (long *) &run);
				run = min(max(run, 0), CLEAR_AGENDA);

				/* Save the selected build task. */
				u = NONUTYPE;
				for_all_unit_types(u2) {
					/* Find the utype that was selected in the menu. */
					if (strcmp(u_type_name(u2), uname) == NULL) {
						net_set_build_task(unit, u2, run, 0, 0);
						u = u2;
						break;
					}
				}
				if (u == NONUTYPE) {
					/* No matching utype means "Idle" was selected. */
					net_clear_task_agenda(unit->side, unit);
					net_set_unit_asleep(unit->side, unit, TRUE, FALSE);
				}
				done = TRUE;
				break;

			case 	CancelButton:
				done = TRUE;
				break;

			default:
				break;
		}
	}
	/* Close the dialog. */
	DisposeDialog(win);
	/* Restore old port. */
	SetPort(oldport);
	/* Return the selected unit type. */
	return u;
}

/*
	Planning dialog for a single advanced unit. 
*/

void
unit_plan_dialog(Unit *unit)
{
	short 		ditem, mitem, done = FALSE, i, m, d, s;
	char			cname[32], sname[32], buf[BUFSIZE];
	Str255 		pname, pnumber;
	Handle 		itemhandle; 
	MenuHandle	planMenu;
	GrafPtr 		oldport;
 	DialogPtr		win;
	
	/* Open the dialog. */
	win = GetNewDialog(dPlan, NULL, (DialogPtr) -1);
	GetPort(&oldport);	
	SetPort(win);

	strcpy( buf, unit->name);
	strcat( buf, " needs a plan.\r\rPick one:\r");
	c2p(buf, pname);
	GetDItem(win, diPlanMainText, NULL, &itemhandle, NULL);
	SetIText(itemhandle, pname);

	/* Set plan type popup menu to current plan. */
	GetDItem(win, diPlanPopup, NULL, &itemhandle, NULL);
	if (unit->plan->type == PLAN_NONE)
		SetCtlValue((ControlHandle) itemhandle, miPlanTypeNone);
	else if (unit->plan->type == PLAN_PASSIVE)
		SetCtlValue((ControlHandle) itemhandle, miPlanTypePassive);
	else if (unit->plan->type == PLAN_DEFENSIVE)
		SetCtlValue((ControlHandle) itemhandle, miPlanTypeDefensive);
	else if (unit->plan->type == PLAN_EXPLORATORY)
		SetCtlValue((ControlHandle) itemhandle, miPlanTypeExploratory);
	else if (unit->plan->type == PLAN_OFFENSIVE)
		SetCtlValue((ControlHandle) itemhandle, miPlanTypeOffensive);
	else if (unit->plan->type == PLAN_COLONIZING)
		SetCtlValue((ControlHandle) itemhandle, miPlanTypeColonizing);
	else if (unit->plan->type == PLAN_IMPROVING)
		SetCtlValue((ControlHandle) itemhandle, miPlanTypeImproving);
	else if (unit->plan->type == PLAN_RANDOM)
		SetCtlValue((ControlHandle) itemhandle, miPlanTypeRandom);
	/* Also update the checkmarks, since this menu is used elsewhere. */
	planMenu = GetMenu(mPlanTypes);
	CheckItem(planMenu, miPlanTypeNone, (unit->plan->type == PLAN_NONE));
	CheckItem(planMenu, miPlanTypePassive, (unit->plan->type == PLAN_PASSIVE));
	CheckItem(planMenu, miPlanTypeDefensive, (unit->plan->type == PLAN_DEFENSIVE));
	CheckItem(planMenu, miPlanTypeExploratory, (unit->plan->type == PLAN_EXPLORATORY));
	CheckItem(planMenu, miPlanTypeOffensive, (unit->plan->type == PLAN_OFFENSIVE));
	CheckItem(planMenu, miPlanTypeColonizing, (unit->plan->type == PLAN_COLONIZING));
	CheckItem(planMenu, miPlanTypeImproving, (unit->plan->type == PLAN_IMPROVING));
	CheckItem(planMenu, miPlanTypeRandom, (unit->plan->type == PLAN_RANDOM));

	ShowWindow(win);
	while (!done) {
		SetCursor(&QD(arrow));
		draw_default_button(win, OkButton);

		ModalDialog(NULL, &ditem);
		switch (ditem) {
			case diPlanCheck: /* Toggle plan auto checkbox. */
				GetDItem(win, ditem, NULL, &itemhandle, NULL);
				SetCtlValue((ControlHandle) itemhandle, !GetCtlValue((ControlHandle) itemhandle));
				break;

			case 	OkButton:
				/* Get the advance auto checkbox state. */
				GetDItem(win, diPlanCheck, NULL, &itemhandle, NULL);
				unit->autoplan = GetCtlValue((ControlHandle) itemhandle);
				/* Get selected plan type. */
				GetDItem(win, diPlanPopup, NULL, &itemhandle, NULL);
				mitem = GetCtlValue((ControlHandle) itemhandle);
				switch (mitem)  {
					case miPlanTypeNone:
						net_set_unit_plan_type(unit->side, unit, PLAN_NONE);
						break;
					case miPlanTypePassive:
						net_set_unit_plan_type(unit->side, unit, PLAN_PASSIVE);
						break;
					case miPlanTypeDefensive:
						net_set_unit_plan_type(unit->side, unit, PLAN_DEFENSIVE);
						break;
					case miPlanTypeExploratory:
						net_set_unit_plan_type(unit->side, unit, PLAN_EXPLORATORY);
						break;
					case miPlanTypeOffensive:
						net_set_unit_plan_type(unit->side, unit, PLAN_OFFENSIVE);
						break;
					case miPlanTypeColonizing:
						net_set_unit_plan_type(unit->side, unit, PLAN_COLONIZING);
						break;
					case miPlanTypeImproving:
						net_set_unit_plan_type(unit->side, unit, PLAN_IMPROVING);
						break;
					case miPlanTypeRandom:
						net_set_unit_plan_type(unit->side, unit, PLAN_RANDOM);
						break;
				}
				done = TRUE;
				break;

			case 	CancelButton:
				done = TRUE;
				break;

			default:
				break;
		}
	}

	/* Close the dialog. */
	DisposeDialog(win);
	/* Restore old port. */
	SetPort(oldport);
}

MenuHandle
build_construction_menu(Unit *unit)
{
	MenuHandle menu;
	Str255 pname;
	int i, u, items;

	menu = GetMenu(mBuildPopup);
	/* First clean out all old items except the first ("Idle"). */
	items = CountMenuItems(menu);
	for (i = 1; i < items; i++)
		DelMenuItem(menu, 2);
	/* Load constructable unit type names into popup menu. */
	for_all_unit_types(u) {
		if (type_allowed_on_side(u, unit->side)
 		      && has_advance_to_build(unit->side, u)
		      && could_create(unit->type, u) 
		      && (type_can_occupy(u, unit)
		    	   || u == unit->plan->tasks->args[0])) {
	   		c2p(u_type_name(u), pname);
			AppendMenu(menu, pname);
		}
	}
 	return menu;
 }

MenuHandle
build_research_menu(Side *side)
{
	MenuHandle menu;
	Str255 pname;
	int i, a, items;

	menu = GetMenu(mResearchPopup);
	/* First clean out all old items except the first ("Idle"). */
	items = CountMenuItems(menu);
	for (i = 1; i < items; i++)
		DelMenuItem(menu, 2);
	/* Load scientific advances that can be researched by side. */
	for_all_advance_types(a) {
		if (has_advance_to_research(side, a)
		     &! has_advance(side, a)) {
   			c2p(a_type_name(a), pname);
			AppendMenu(menu, pname);
		}
	}
 	return menu;
 }

