
/*--------------------------------------------------------------------*/
/*--- A header file containing architecture-specific definitions   ---*/
/*--- for Valgrind on x86-family processors.			   ---*/
/*---                                               vg_archdefs.h  ---*/
/*--------------------------------------------------------------------*/

/*
   This file is part of Valgrind, an extensible x86 protected-mode
   emulator for monitoring program execution on x86-Unixes.

   Copyright (C) 2000-2004 Julian Seward 
      jseward@acm.org

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.

   The GNU General Public License is contained in the file COPYING.
*/

#ifndef __VG_ARCHDEFS_H
#define __VG_ARCHDEFS_H

/* ---------------------------------------------------------------------
   Constants pertaining to the simulated CPU state, VG_(baseBlock),
   which need to go here to avoid ugly circularities.
   ------------------------------------------------------------------ */

/* How big is the saved SSE/SSE2 state?  Note that this subsumes the
   FPU state.  On machines without SSE, we just save/restore the FPU
   state into the first part of this area. */
/* A general comment about SSE save/restore: It appears that the 7th
   word (which is the MXCSR) has to be &ed with 0x0000FFBF in order
   that restoring from it later does not cause a GP fault (which is
   delivered as a segfault).  I guess this will have to be done
   any time we do fxsave :-(  7th word means word offset 6 or byte
   offset 24 from the start address of the save area.
 */
#define VG_SIZE_OF_SSESTATE 512
/* ... and in words ... */
#define VG_SIZE_OF_SSESTATE_W ((VG_SIZE_OF_SSESTATE+3)/4)


/* This is the hardware-format for a segment descriptor, ie what the
   x86 actually deals with.  It is 8 bytes long.  It's ugly.  */

typedef struct _LDT_ENTRY {
    union {
       struct {
          UShort      LimitLow;
          UShort      BaseLow;
          unsigned    BaseMid         : 8;
          unsigned    Type            : 5;
          unsigned    Dpl             : 2;
          unsigned    Pres            : 1;
          unsigned    LimitHi         : 4;
          unsigned    Sys             : 1;
          unsigned    Reserved_0      : 1;
          unsigned    Default_Big     : 1;
          unsigned    Granularity     : 1;
          unsigned    BaseHi          : 8;
       } Bits;
       struct {
          UInt word1;
          UInt word2;
       } Words;
    } 
    LdtEnt;
} VgLdtEntry;

/* Maximum number of LDT entries supported (by the x86). */
#define VG_M_LDT_ENTRIES     8192
/* The size of each LDT entry == sizeof(VgLdtEntry) */
#define VG_LDT_ENTRY_SIZE  8

/* ---------------------------------------------------------------------
   Exports of vg_ldt.c
   ------------------------------------------------------------------ */

/* Alloc & copy, and dealloc. */
extern VgLdtEntry* 
         VG_(allocate_LDT_for_thread) ( VgLdtEntry* parent_ldt );
extern void       
         VG_(deallocate_LDT_for_thread) ( VgLdtEntry* ldt );
extern void       
         VG_(clear_TLS_for_thread) ( VgLdtEntry* tls );

/* Simulate the modify_ldt syscall. */
extern Int VG_(sys_modify_ldt) ( ThreadId tid,
                                 Int func, void* ptr, UInt bytecount );

/* Simulate the {get,set}_thread_area syscalls. */
extern Int VG_(sys_set_thread_area) ( ThreadId tid,
                                      struct vki_modify_ldt_ldt_s* info );
extern Int VG_(sys_get_thread_area) ( ThreadId tid,
                                      struct vki_modify_ldt_ldt_s* info );

/* Called from generated code.  Given a segment selector and a virtual
   address, return a linear address, and do limit checks too. */
extern Addr VG_(do_useseg) ( UInt seg_selector, Addr virtual_addr );


/* Total number of spill slots available for allocation, if a TempReg
   doesn't make it into a RealReg.  Just bomb the entire system if
   this value is too small; we don't expect it will ever get
   particularly high. */
#define VG_MAX_SPILLSLOTS 24

/* Valgrind's signal stack size, in words */
#define VG_SIGSTACK_SIZE_W    10000

/* Architecture-specific part of a ThreadState */
typedef struct arch_thread {
   /* Pointer to this thread's Local (Segment) Descriptor Table.
      Starts out as NULL, indicating there is no table, and we hope to
      keep it that way.  If the thread does __NR_modify_ldt to create
      entries, we allocate a 8192-entry table at that point.  This is
      a straight copy of the Linux kernel's scheme.  Don't forget to
      deallocate this at thread exit. */
   VgLdtEntry* ldt;

   /* TLS table. This consists of a small number (currently 3) of
      entries from the Global Descriptor Table. */
   VgLdtEntry tls[VKI_GDT_TLS_ENTRIES];

   /* Saved machine context.  Note the FPU state, %EIP and segment
      registers are not shadowed.

      Although the segment registers are 16 bits long, storage
      management here and in VG_(baseBlock) is
      simplified if we pretend they are 32 bits. */
   UInt m_cs;
   UInt m_ss;
   UInt m_ds;
   UInt m_es;
   UInt m_fs;
   UInt m_gs;

   UInt m_eax;
   UInt m_ebx;
   UInt m_ecx;
   UInt m_edx;
   UInt m_esi;
   UInt m_edi;
   UInt m_ebp;
   UInt m_esp;
   UInt m_eflags;
   UInt m_eip;

   /* The SSE/FPU state.  This array does not (necessarily) have the
      required 16-byte alignment required to get stuff in/out by
      fxsave/fxrestore.  So we have to do it "by hand".
   */
   UInt m_sse[VG_SIZE_OF_SSESTATE_W];

   UInt sh_eax;
   UInt sh_ebx;
   UInt sh_ecx;
   UInt sh_edx;
   UInt sh_esi;
   UInt sh_edi;
   UInt sh_ebp;
   UInt sh_esp;
   UInt sh_eflags;
} arch_thread_t;

/* Accessors for the arch_thread_t */
#define ARCH_INSTR_POINTER(regs)	((regs).m_eip)
#define ARCH_STACK_POINTER(regs)	((regs).m_esp)
#define ARCH_FRAME_POINTER(regs)	((regs).m_ebp)

/* are these linux-specific? */
#define ARCH_SYSCALL_NUM(regs)		((regs).m_eax)
#define ARCH_SYSCALL_RET(regs)		((regs).m_eax)
#define ARCH_SYSCALL_ARG1(regs)		((regs).m_ebx)
#define ARCH_SYSCALL_ARG2(regs)		((regs).m_ecx)
#define ARCH_SYSCALL_ARG3(regs)		((regs).m_edx)
#define ARCH_SYSCALL_ARG4(regs)		((regs).m_esi)
#define ARCH_SYSCALL_ARG5(regs)		((regs).m_edi)
#define ARCH_SYSCALL_ARG6(regs)		((regs).m_ebp)

#define ARCH_PRE_SYSCALL_RESULT(regs, val)	((regs).m_eax = (val))

#define ARCH_CLREQ_ARGS(regs)		((regs).m_eax)

/* Interesting register numbers */
#define R_SYSCALL_NUM	R_EAX
#define R_SYSCALL_ARG1	R_EBX
#define R_SYSCALL_RET	R_EAX
#define R_STACK_PTR	R_ESP
#define R_FRAME_PTR	R_EBP

/* Stack frame layout and linkage */
#define FIRST_STACK_FRAME(ebp)	(ebp)
#define STACK_FRAME_RET(ebp)	(((UInt*)ebp)[1])
#define STACK_FRAME_NEXT(ebp)	(((UInt*)ebp)[0])

#define GET_STACK_ADDRS(ebp, esp)	do {				\
   asm("movl %%ebp, %0; movl %%esp, %1" : "=r" (ebp), "=r" (esp));	\
} while (0)

/* Write a value to a client's thread register, and shadow (if necessary) */
#define SET_THREAD_REG( zztid, zzval, zzreg, zzREG, zzevent, zzargs... ) \
   do { VG_(threads)[zztid].arch.m_##zzreg = (zzval);			 \
        VG_TRACK( zzevent, zztid, R_##zzREG, ##zzargs );		 \
   } while (0)

#define SET_SYSCALL_RETVAL(zztid, zzval) \
   SET_THREAD_REG(zztid, zzval, eax, EAX, post_reg_write_syscall_return)

#define SET_SIGNAL_EDX(zztid, zzval) \
   SET_THREAD_REG(zztid, zzval, edx, EDX, post_reg_write_deliver_signal)

#define SET_SIGNAL_ESP(zztid, zzval) \
   SET_THREAD_REG(zztid, zzval, esp, ESP, post_reg_write_deliver_signal)

#define SET_CLREQ_RETVAL(zztid, zzval) \
   SET_THREAD_REG(zztid, zzval, edx, EDX, post_reg_write_clientreq_return)

#define SET_CLCALL_RETVAL(zztid, zzval, f) \
   SET_THREAD_REG(zztid, zzval, edx, EDX, post_reg_write_clientcall_return, f)

#define SET_PTHREQ_ESP(zztid, zzval) \
   SET_THREAD_REG(zztid, zzval, esp, ESP, post_reg_write_pthread_return)

#define SET_PTHREQ_RETVAL(zztid, zzval) \
   SET_THREAD_REG(zztid, zzval, edx, EDX, post_reg_write_pthread_return)

/* ---------------------------------------------------------------------
   Exports of vg_helpers.S
   ------------------------------------------------------------------ */

/* Mul, div, etc, -- we don't codegen these directly. */
extern void VG_(helper_idiv_64_32);
extern void VG_(helper_div_64_32);
extern void VG_(helper_idiv_32_16);
extern void VG_(helper_div_32_16);
extern void VG_(helper_idiv_16_8);
extern void VG_(helper_div_16_8);

extern void VG_(helper_imul_32_64);
extern void VG_(helper_mul_32_64);
extern void VG_(helper_imul_16_32);
extern void VG_(helper_mul_16_32);
extern void VG_(helper_imul_8_16);
extern void VG_(helper_mul_8_16);

extern void VG_(helper_CLD);
extern void VG_(helper_STD);
extern void VG_(helper_get_dirflag);

extern void VG_(helper_CLC);
extern void VG_(helper_STC);
extern void VG_(helper_CMC);

extern void VG_(helper_shldl);
extern void VG_(helper_shldw);
extern void VG_(helper_shrdl);
extern void VG_(helper_shrdw);

extern void VG_(helper_IN);
extern void VG_(helper_OUT);

extern void VG_(helper_RDTSC);
extern void VG_(helper_CPUID);

extern void VG_(helper_bsfw);
extern void VG_(helper_bsfl);
extern void VG_(helper_bsrw);
extern void VG_(helper_bsrl);

extern void VG_(helper_fstsw_AX);
extern void VG_(helper_SAHF);
extern void VG_(helper_LAHF);
extern void VG_(helper_DAS);
extern void VG_(helper_DAA);
extern void VG_(helper_AAS);
extern void VG_(helper_AAA);
extern void VG_(helper_AAD);
extern void VG_(helper_AAM);

extern void VG_(helper_cmpxchg8b);

/* -----------------------------------------------------
   Exports of arch/x86-common/vg_x86_state.c
   -------------------------------------------------- */

/* Insert and extract the D flag from eflags */
extern UInt VG_(insertDflag)(UInt eflags, Int d);
extern Int VG_(extractDflag)(UInt eflags);

/* For setting up baseBlock */
extern void arch_alloc_low_baseBlock ( Addr client_eip, Addr esp_at_startup );
extern void arch_alloc_baseBlock( Addr client_eip, Addr esp_at_startup );

extern void VG_(arch_process_options)(void);

extern void VG_(arch_load_state) ( arch_thread_t *, ThreadId );
extern void VG_(arch_save_state) ( arch_thread_t *, ThreadId );
extern void VG_(arch_clear_thread) ( arch_thread_t * );
extern void VG_(arch_thread_init) ( arch_thread_t * );
extern void VG_(arch_thread_cleanup) ( arch_thread_t * );
extern void VG_(arch_child_setup) ( arch_thread_t *, arch_thread_t * );
extern Int VG_(arch_set_child_regs_from_BB)( Int );
extern Int VG_(arch_set_child_regs)( Int pid, arch_thread_t * );
extern UInt *VG_(arch_reg_addr_from_BB)( Int reg );
extern UInt *VG_(arch_reg_addr)( Int reg, arch_thread_t * );
extern void arch_set_arg_and_ret( ThreadId tid, UInt arg, Addr ret );
extern void arch_thread_initial_stack( ThreadId tid, UInt arg, Addr ret );

/* -----------------------------------------------------
   Exports of arch/x86-common/vg_x86_signal.c
   -------------------------------------------------- */

extern void VG_(arch_deliver_signal)(ThreadId tid, Addr esp_top_of_frame,
				     const vki_ksiginfo_t *siginfo,
				     void *handler, UInt flags,
				     const vki_ksigset_t *mask);
extern Int VG_(arch_signal_return)(ThreadId tid, Bool has_siginfo);
extern void VG_(arch_fill_elfregs_from_BB)(struct user_regs_struct *);
extern void VG_(arch_fill_elfregs)(struct user_regs_struct *,
				   const arch_thread_t *);
extern void VG_(arch_fill_elffpregs_from_BB)(elf_fpregset_t *);
extern void VG_(arch_fill_elffpregs)(elf_fpregset_t *, const arch_thread_t *);
extern void VG_(arch_fill_elffpxregs_from_BB)(elf_fpxregset_t *);
extern void VG_(arch_fill_elffpxregs)(elf_fpxregset_t *,
				      const arch_thread_t *);
extern void VG_(arch_flush_state)( struct vki_ucontext * );

/* -----------------------------------------------------
   Exports of arch/x86-common/vg_x86_syscall.c
   -------------------------------------------------- */

extern void VG_(arch_restart_syscall)(arch_thread_t *);

/* -----------------------------------------------------
   Read-write parts of baseBlock.
   -------------------------------------------------- */

/* State of the simulated CPU. */
extern Int VGOFF_(m_eax);
extern Int VGOFF_(m_ecx);
extern Int VGOFF_(m_edx);
extern Int VGOFF_(m_ebx);
extern Int VGOFF_(m_esp);
extern Int VGOFF_(m_ebp);
extern Int VGOFF_(m_esi);
extern Int VGOFF_(m_edi);
extern Int VGOFF_(m_eflags);
extern Int VGOFF_(m_ssestate);
extern Int VGOFF_(m_eip);

extern Int VGOFF_(m_dflag);	/* D flag is handled specially */

extern Int VGOFF_(m_cs);
extern Int VGOFF_(m_ss);
extern Int VGOFF_(m_ds);
extern Int VGOFF_(m_es);
extern Int VGOFF_(m_fs);
extern Int VGOFF_(m_gs);

/* Reg-alloc spill area (VG_MAX_SPILLSLOTS words long). */
extern Int VGOFF_(spillslots);

/* Records the valid bits for the 8 integer regs & flags reg. */
extern Int VGOFF_(sh_eax);
extern Int VGOFF_(sh_ecx);
extern Int VGOFF_(sh_edx);
extern Int VGOFF_(sh_ebx);
extern Int VGOFF_(sh_esp);
extern Int VGOFF_(sh_ebp);
extern Int VGOFF_(sh_esi);
extern Int VGOFF_(sh_edi);
extern Int VGOFF_(sh_eflags);

/*
 * Generic definitions
 */
#define VGOFF_INSTR_POINTER	VGOFF_(m_eip)
#define VGOFF_STACK_POINTER	VGOFF_(m_esp)
#define VGOFF_FRAME_POINTER	VGOFF_(m_ebp)

/* -----------------------------------------------------
   Read-only parts of baseBlock.
   -------------------------------------------------- */

/* This thread's LDT pointer. */
extern Int VGOFF_(ldt);

/* This thread's TLS pointer. */
extern Int VGOFF_(tls_ptr);

/* Nb: Most helper offsets are in include/vg_skin.h, for use by skins */
extern Int VGOFF_(helper_undefined_instruction);

/* Struct used to convey architecture-specific info from pthread_create
   to thread_wrapper in vg_libpthread.c.  */
typedef
   struct arch_thread_info {
      void*         tls_data;
      int           tls_segment;
      unsigned long sysinfo;
   } arch_thread_info_t;

/* Exports of vg_x86_libpthread.c */
void __vg_arch_thread_exit(void);
void __vg_arch_thread_create(arch_thread_info_t *info);
void __vg_arch_thread_wrapper(arch_thread_info_t *info);
int  __vg_arch_has_tls(void);

/* How vg_tt and VG_(tt_fast) are indexed. */
#define VG_TT_HASH(addr)        (((UInt)(addr)) % VG_TT_SIZE)
#define VG_TT_FAST_INDEX(addr)  ((UInt)(addr) & VG_TT_FAST_MASK)

/* Redirect _dl_sysinfo_int80, which is glibc's default system call
   routine, to the routine in our trampoline page so that the
   special sysinfo unwind hack in vg_execontext.c will kick in.
*/
#define DO_ARCH_REDIRECTS	\
   VG_(add_redirect_addr)("soname:ld-linux.so.2", "_dl_sysinfo_int80",	\
                          VG_(client_trampoline_code)+VG_(tramp_syscall_offset));

#endif /* __VG_ARCHDEFS_H */
