/*
 * tools/vgscan.c
 *
 * Copyright (C) 1997 - 1999  Heinz Mauelshagen, Germany
 *
 * Oktober  1997
 * January  1998
 * May-July,September 1998
 *
 * LVM is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * LVM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU CC; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA. 
 *
 */

/*
 * Changelog
 *
 *    11/01/1998 - added creation of volume group directory and special files
 *    03/05/1998 - enhanced error checking with lvm_tab_vg_insert()
 *                 and lvm_tab_vg_remove()
 *    27/06/1998 - changed lvm_tab_* calling convention
 *    10/07/1998 - implemented -A option
 *    02/09/1998 - corrected some messages
 *
 */

#include <lvm_user.h>

char *cmd = NULL;

#ifdef DEBUG
int opt_d = 0;
#endif


int main ( int argc, char **argv)
{
   int c = 0;
   int i = 0;
   int opt_A = 1;
   int opt_v = 0;
   int np = 0;
   int p = 0;
   int ret = 0;
#ifdef DEBUG
   char *options = "A:dh?v";
#else
   char *options = "A:h?v";
#endif
   char command[NAME_LEN] = { 0, };
   char **vg_name_ptr = NULL;
   pv_t **pv = NULL;
   vg_t *vg = NULL;

   cmd = basename ( argv[0]);

   SUSER_CHECK;

   while ( ( c = getopt ( argc, argv, options)) != EOF) {
      switch ( c) {
         case 'A':
            if ( opt_A > 1) {
               fprintf ( stderr, "%s -- A option already given\n\n", cmd);
               return LVM_EINVALID_CMD_LINE;
            }
            if ( strcmp ( optarg, "y") == 0);
            else if ( strcmp ( optarg, "n") == 0) opt_A = 0;
            else {
               fprintf ( stderr, "%s -- invalid option argument \"%s\"\n\n",
                                 cmd, optarg);
               return LVM_EINVALID_CMD_LINE;
            }
            break;

#ifdef DEBUG
         case 'd':
            if ( opt_d > 0) {
               fprintf ( stderr, "%s -- d option already given\n\n", cmd);
               return LVM_EINVALID_CMD_LINE;
            }
            opt_d++;
            break;
#endif

         case 'h':
         case '?':
            printf ( "\n%s\n\n%s -- Volume Group Scan\n\n"
                     "Synopsis:\n"
                     "---------\n\n"
                     "%s\n"
                     "\t[-A y/n]\n"
#ifdef DEBUG
                     "\t[-d]\n"
#endif
                     "\t[-h/?]\n"
                     "\t[-v]]\n\n",
                     lvm_version, cmd, cmd);
            return 0;
            break;

         case 'v':
            if ( opt_v > 0) {
               fprintf ( stderr, "%s -- v option already given\n\n", cmd);
               return LVM_EINVALID_CMD_LINE;
            }
            opt_v++;
            break;

         default:
            fprintf ( stderr, "%s -- invalid command line option \"%c\"\n",
                      cmd, c);
            return LVM_EINVALID_CMD_LINE;
      }
   }

   CMD_MINUS_CHK;
   LVM_CHECK_IOP;
   LVM_LOCK ( 1);

   if ( optind < argc) {
      fprintf ( stderr, "%s -- invalid command line\n\n", cmd);
      return LVM_EINVALID_CMD_LINE;
   }

   printf ( "%s -- reading all physical volumes (this may take a while...)\n",
            cmd);
   if ( ( ret = pv_read_all_pv ( &pv, TRUE)) < 0) {
      fprintf ( stderr, "%s -- ERROR %d reading all physical volumes\n",
                        cmd, ret);
      return LVM_EVGSCAN_PV_READ_ALL_PV;
   }

   lvm_dont_interrupt ( 0);

   if ( opt_v > 0) printf ( "%s -- removing \"%s\" and \"%s\"\n",
                            cmd, LVMTAB, LVMTAB_DIR);
   sprintf ( command, "rm -fr %s %s%c", LVMTAB, LVMTAB_DIR, 0);
   system ( command);

   if ( opt_v > 0) printf ( "%s -- inserting found volume group "
                            "names into \"%s\"\n", cmd, LVMTAB);
   np = 0;
   for ( p = 0; pv != NULL && pv[p] != NULL; p++) {
      if ( pv_check_consistency ( pv[p]) == 0) {
         if ( system_id_check_exported ( pv[p]->system_id) == FALSE) {
            if ( pv_check_new ( pv[p]) == FALSE) {
               if ( opt_v > 0) printf ( "%s -- inserting volume group \"%s\" "
                                        "on \"%s\"\n",
                                        cmd, pv[p]->vg_name,
                                        pv[p]->pv_name);
               if ( ( ret = lvm_tab_vg_insert ( pv[p]->vg_name)) < 0 &&
                    ret != -LVM_ELVM_TAB_VG_INSERT_VG_EXISTS) {
                  fprintf ( stderr, "%s -- ERROR %d inserting volume group "
                                    "\"%s\" into \"%s\"\n",
                                    cmd, ret, pv[p]->vg_name, LVMTAB);
                  return LVM_EVGSCAN_VG_INSERT;
               }
               np++;
            }
         } else {
            printf ( "%s -- exported physical volume \"%s\" found\n",
                      cmd, pv[p]->pv_name);
         }
      } else {
         fprintf ( stderr, "%s -- invalid device \"%s\" found\n",
                           cmd, pv[p]->pv_name);
      }
   }

   if ( ret == -LVM_ELVM_TAB_VG_INSERT_VG_EXISTS) ret = 0;

   if ( opt_A == 0) printf ( "%s -- you will have NO VGDA copies in \"%s\"\n",
                             cmd, LVMTAB_DIR);
   if ( np == 0) {
      ret = lvm_tab_create ();
   } else {
      if ( ( vg_name_ptr = lvm_tab_vg_check_exist_all_vg ()) != NULL) {
         for ( i = 0; vg_name_ptr[i] != NULL; i++) {
            printf ( "%s -- found volume group \"%s\"\n", cmd, vg_name_ptr[i]);
            if ( opt_A > 0) {
               if ( opt_v > 0) printf ( "%s -- backing up volume group "
                                        "\"%s\"\n",
                                        cmd, vg_name_ptr[i]);
               if ( ( ret = vg_cfgbackup ( vg_name_ptr[i], LVMTAB_DIR,
                                           opt_v, NULL)) < 0) {
                  fprintf ( stderr, "%s -- ERROR: unable to do a backup "
                                    "of volume group \"%s\"\n",
                                    cmd, vg_name_ptr[i]);
                  if ( ( ret = lvm_tab_vg_remove ( vg_name_ptr[i])) < 0) {
                     fprintf ( stderr, "%s -- ERROR %d removing volume group "
                                       "\"%s\" from \"%s\"\n",
                                       cmd, ret, vg_name_ptr[i], LVMTAB);
                  }
                  continue;
               }
               if ( opt_v > 0) printf ( "%s -- reading volume group data of "
                                        "\"%s\"from lvmtab\n",
                                        cmd, vg_name_ptr[i]);
               if ( ( ret = lvm_tab_vg_read_with_pv_and_lv ( vg_name_ptr[i],
                                                             &vg)) < 0) {
                  fprintf ( stderr, "%s -- ERROR %d reading volume group "
                                    "data for \"%s\"from lvmtab\n",
                                    cmd, ret, vg_name_ptr[i]);
                  continue;
               }
               if ( opt_v > 0) printf ( "%s -- creating volume group directory "
                                        "and special files for \"%s\"\n",
                                        cmd, vg_name_ptr[i]);
               if ( ( ret = vg_create_dir_and_group_and_nodes ( vg,
                                                                opt_v)) < 0) {
                  fprintf ( stderr, "%s -- ERROR %d creating volume group "
                                    "directory and special files\n", cmd, ret);
                  continue;
               }
            } else {
               if ( opt_v > 0) printf ( "%s -- removing \"%s\" and \"%s\"\n",
                                        cmd, LVMTAB, LVMTAB_DIR);
               sprintf ( command, "rm -fr %s %s%c", LVMTAB, LVMTAB_DIR, 0);
               system ( command);
            }
         }
      } else ret = LVM_EVGSCAN_NO_VG;
   }

   lvm_interrupt ();
   LVM_UNLOCK ( 1);

   if ( opt_A > 0) {
      if ( ret == 0) {
         printf ( "%s -- \"%s\" and \"%s\" successfully created\n",
                  cmd, LVMTAB, LVMTAB_DIR);
      } else {
         fprintf ( stderr, "%s -- ERROR %d creating \"%s\" and \"%s\"\n",
                           cmd, ret, LVMTAB, LVMTAB_DIR);
         ret = LVM_EVGSCAN_LVMTAB;
      }

      printf ( "%s -- WARNING: you may not have an actual backup of your "
               "volume group%s\n\n", cmd, i > 1 ? "s" : "");
   } else putchar ( '\n');

   return ret;
}
