/*
 * tools/lvmdiskscan.c
 *
 * Copyright (C) 1997 - 1999  Heinz Mauelshagen, Germany
 *
 * May,June,November 1998
 *
 * LVM is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * LVM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU CC; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA. 
 *
 */

/*
 * Changelog
 *
 *   11/16/1998 - avoided checking partition special files, if
 *                disk doesn't exist
 *
 */

#include <lvm_user.h>

char *cmd = NULL;

#ifdef DEBUG
int opt_d = 0;
#endif

int main ( int argc, char **argv)
{
   short st_rdev = -1;
   int c = 0;
   int in = 0;
   int dir_cache_size = 0;
   int len = 0;
   int maxlen = 0;
   int opt_l = 0;
   int opt_v = 0;
   int n_disks = 0;
   int n_whole_disks = 0;
   int n_lvm_pvs = 0;
   int n_multiple_devices = 0;
   int n_partitions = 0;
   int p = 0;
   int size = 0;
   int tst = -1;
   char *dummy = NULL;
   char fmt[32] = { 0, };
#ifdef DEBUG
   char *options = "dh?lv";
#else
   char *options = "h?lv";
#endif
   struct partition part;
   dir_cache_t *dir_cache = NULL;

   cmd = basename ( argv[0]);

   SUSER_CHECK;
   LVMTAB_CHECK;

   while ( ( c = getopt ( argc, argv, options)) != EOF) {
      switch ( c) {
#ifdef DEBUG
         case 'd':
            if ( opt_d > 0) {
               fprintf ( stderr, "%s -- d option already given\n\n", cmd);
               return LVM_EINVALID_CMD_LINE;
            }
            opt_d++;
            break;
#endif

         case 'h':
         case '?':
            printf ( "\n%s\n\n%s -- Physical Volume Scan\n\n"
                     "Synopsis:\n"
                     "---------\n\n"
                     "%s\n"
#ifdef DEBUG
                     "\t[-d]\n"
#endif
                     "\t[-h/?]\n"
                     "\t[-l]\n"
                     "\t[-v]\n\n",
                     lvm_version, cmd, cmd);
            return 0;
            break;

         case 'l':
            if ( opt_l > 0) {
               fprintf ( stderr, "%s -- l option already given\n\n", cmd);
               return LVM_EINVALID_CMD_LINE;
            }
            opt_l++;
            break;

         case 'v':
            if ( opt_v > 0) {
               fprintf ( stderr, "%s -- v option already given\n\n", cmd);
               return LVM_EINVALID_CMD_LINE;
            }
            opt_v++;
            break;

         default:
            fprintf ( stderr, "%s -- invalid command line option \"%c\"\n",
                      cmd, c);
            return LVM_EINVALID_CMD_LINE;
      }
   }

   CMD_MINUS_CHK;

   printf ( "%s -- reading all disks / partitions (this may take a while...)\n",
            cmd);
   if ( opt_v > 0) printf ( "%s -- filling directory cache...\n", cmd);
   if ( ( dir_cache_size = lvm_dir_cache ( &dir_cache)) <= 0) {
      fprintf ( stderr, "%s -- no disks / partitions found\n\n", cmd);
      return LVM_ELVMDISKSCAN_NO_FILES_FOUND;
   }

   if ( opt_v > 0) printf ( "%s -- walking through all found disks "
                            "/ partitions\n", cmd);
   for ( p = 0; p < dir_cache_size; p++) {
      len = strlen ( dir_cache[p].dev_name);
      if ( maxlen < len) maxlen = len;
   }
   sprintf ( fmt, "%s%d%s ", "%s -- %-", maxlen, "s [%14s]");

   if ( opt_l > 0) printf ( "%s -- only showing LVM partitions/disks...\n",
                            cmd);
   for ( p = 0; p < dir_cache_size; p++) {
      if ( ( tst = open ( dir_cache[p].dev_name, O_RDONLY)) == -1) {
         if ( MAJOR ( dir_cache[p].st_rdev) != MD_MAJOR &&
              MINOR ( dir_cache[p].st_rdev) % 16 == 0) {
            p += 15;
            continue;
         }
      } else close ( tst);

      if ( st_rdev > -1 && ( ( MINOR ( st_rdev) & 0xF0) ==
           ( MINOR ( dir_cache[p].st_rdev) & 0xF0))) continue;

      in = -1;
      if ( MAJOR ( dir_cache[p].st_rdev) != MD_MAJOR &&
           ( MINOR ( dir_cache[p].st_rdev) % 16 == 0) &&
           ( in = open ( dir_cache[p].dev_name, O_RDONLY)) != -1) n_disks++;
      if ( in != -1) close ( in);

      if ( ( size = pv_get_size ( dir_cache[p].dev_name,
                                  &part)) < 0) continue;
      n_partitions++;
      if ( opt_l > 0 && part.sys_ind != LVM_PARTITION) continue;

      printf ( fmt, cmd, dir_cache[p].dev_name,
               ( dummy = lvm_show_size ( size / 2, SHORT)));
      free ( dummy); dummy = NULL;
      if ( part.sys_ind == 0) {
         n_partitions--;
         if ( MAJOR ( dir_cache[p].st_rdev) == MD_MAJOR) {
            n_multiple_devices++;
            printf ( "multiple device");
         } else {
            n_whole_disks++;
            printf ( "whole disk");
            st_rdev = dir_cache[p].st_rdev;
         }
      } else {
         if ( opt_l == 0) {
            if ( MINOR ( dir_cache[p].st_rdev) % 16 > 4)
               printf ( "Extended ");
            else if ( part.sys_ind != DOS_EXTENDED_PARTITION)
               printf ( "Primary  ");
         }
         if ( part.sys_ind == 0x6)
            printf ( "DOS 16bit");
         else if ( part.sys_ind == 0x83)
            printf ( "LINUX native partition");
         else if ( part.sys_ind == 0x64 || part.sys_ind == 0x65)
            printf ( "Novell Netware");
#ifdef BSD_PARTITION
         else if ( part.sys_ind == BSD_PARTITION)
            printf ( "BSD");
#endif
         else if ( part.sys_ind == LVM_PARTITION) {
            n_lvm_pvs++;
            if ( opt_l == 0) printf ( "LVM partition");
         } else if ( part.sys_ind == DOS_EXTENDED_PARTITION)
            printf ( "DOS extended partition");
         else if ( part.sys_ind == LINUX_EXTENDED_PARTITION)
            printf ( "LINUX extended partition");
#if LINUX_VERSION_CODE >= KERNEL_VERSION ( 2, 1, 0)
         else if ( part.sys_ind == LINUX_SWAP_PARTITION)
            printf ( "LINUX swap partition");
         else if ( part.sys_ind == WIN98_EXTENDED_PARTITION)
            printf ( "Windows98 extended partition");
#endif
      }
      if ( opt_l == 0) printf ( " [0x%X]\n", part.sys_ind);
      else             putchar ( '\n');
   }

   if ( n_partitions == 0) {
      printf ( "%s -- no valid disks / partitions found\n"
               "%s -- please check your disk device special files!\n\n",
               cmd, cmd);
   } else {
      printf ( "%s -- %d disk%s\n",
               cmd, n_disks, n_disks == 0 || n_disks > 1 ? "s" : "");
      printf ( "%s -- %d whole disk%s\n",
               cmd, n_whole_disks,
               n_whole_disks == 0 || n_whole_disks > 1 ? "s" : "");
      printf ( "%s -- %d multiple device%s\n",
               cmd, n_multiple_devices,
               n_multiple_devices == 0 || n_multiple_devices > 1 ? "s" : "");
      printf ( "%s -- %d partition%s\n",
               cmd, n_partitions,
               n_partitions == 0 || n_partitions  > 1 ? "s" : "");
      printf ( "%s -- %d LVM physical volume partition%s\n",
               cmd, n_lvm_pvs,
               n_lvm_pvs == 0 || n_lvm_pvs  > 1 ? "s" : "");
      printf ( "\n\n");
   }

   if ( n_disks == 0 && n_whole_disks== 0 && n_lvm_pvs == 0 &&
        n_multiple_devices == 0 && n_partitions == 0)
      return LVM_ELVMDISKSCAN_NO_DISKS_FOUND;

   return 0;
}
