/*
 * Author: Heinz Mauelshagen, Germany
 *
 * February, October 1997
 * May-June 1998
 *
 * vgremove is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * vgremove is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU CC; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA. 
 *
 */

/*
 * Changelog
 *
 *    10/12/1997 - added messages
 *    05/03/1998 - enhanced error messages with lvm_tab_vg_remove()
 *    05/16/1998 - added lvmtab checking
 *    06/13/1998 - change to use lvm_tab_vg_read_with_pv_and_lv()
 *                 instead of vg_read() and pv_read_all_pv_of_vg()
 *    06/27/1998 - changed lvm_tab_* calling convention
 *
 */

#include <lvm_user.h>

char *cmd = NULL;

#ifdef DEBUG
int opt_d = 0;
#endif

int main ( int argc, char **argv)
{
   int c = 0;
   int opt_v = 0;
   int p = 0;
   int ret = 0;
#ifdef DEBUG
   char *options = "dh?v";
#else
   char *options = "dh?v";
#endif
   char *vg_name = NULL;
   vg_t *vg = NULL;
   pv_t **pv = NULL;

   cmd = basename ( argv[0]);

   SUSER_CHECK;
   LVMTAB_CHECK;

   while ( ( c = getopt ( argc, argv, options)) != EOF) {
      switch ( c) {
#ifdef DEBUG
         case 'd':
            if ( opt_d > 0) {
               fprintf ( stderr, "%s -- d option yet given\n\n", cmd);
               return 1;
            }
            opt_d++;
            break;
#endif

         case 'h':
         case '?':
            printf ( "\n%s\n\n%s -- Volume Group Remove\n\n"
                     "Synopsis:\n"
                     "---------\n\n"
                     "%s\n"
#ifdef DEBUG
                     "\t[-d]\n"
#endif
                     "\t[-h/?]\n"
                     "\t[-v]\n"
                     "\tVolumeGroupName [VolumeGroupName...]\n\n",
                     lvm_version, cmd, cmd);
            return 0;
            break;

         case 'v':
            if ( opt_v > 0) {
               fprintf ( stderr, "%s -- v option yet given\n\n", cmd);
               return 1;
            }
            opt_v++;
            break;

         default:
            fprintf ( stderr, "%s -- invalid command line option \"%c\"\n",
                      cmd, c);
            return 1;
      }
   }

   if ( optind == argc) {
      fprintf ( stderr, "%s -- please enter a volume group name\n\n", cmd);
      return 1;
   }

   LVM_LOCK ( 0);
   LVM_CHECK_IOP;

   for ( ; optind < argc; optind++) {
      vg_name = argv[optind];

      if ( opt_v > 0) printf ( "%s -- checking volume group name\n", cmd);
      if ( vg_check_name ( vg_name) < 0) {
         fprintf ( stderr, "%s -- no valid volume group name %s\n",
                           cmd, vg_name);
         continue;
      }
   
      /* does VG exist? */
      if ( opt_v > 0) printf ( "%s -- checking volume group %s existence\n",
                               cmd, vg_name);
      if ( ( ret = lvm_tab_vg_check_exist ( vg_name, NULL)) != TRUE) {
         if ( ret == -LVM_EVG_CHECK_EXIST_PV_COUNT) {
            fprintf ( stderr, "%s -- not all physical volumes of volume group "
                              "%s are online\n", cmd, vg_name);
         } else if ( ret == FALSE) {
            fprintf ( stderr, "%s -- volume group %s doesn't exist\n",
                              cmd, vg_name);
         } else {
            fprintf ( stderr, "%s -- ERROR %d: volume group %s doesn't exist\n",
                              cmd, ret, vg_name);
         }
         continue;
      }
   
      if ( opt_v > 0) printf ( "%s -- checking volume group %s activity\n",
                               cmd, vg_name);
      if ( vg_check_active ( vg_name) == TRUE) {
         fprintf ( stderr, "%s -- can't remove active volume group %s\n",
                           cmd, vg_name);
         continue;
      }
   
      if ( opt_v > 0) printf ( "%s -- reading data of volume group %s\n",
                               cmd, vg_name);
      if ( lvm_tab_vg_read_with_pv_and_lv ( vg_name, &vg) < 0) {
         fprintf ( stderr, "%s -- ERROR %d: can't read volume group descriptor "
                           " area of volume group %s from lvmtab\n",
                           cmd, ret, vg_name);
         continue;
      }
   
      if ( opt_v > 0) printf ( "%s -- checking for logical volumes in "
                               "volume group %s\n",
                               cmd, vg_name);
      if ( vg->lv_cur > 0) {
         fprintf ( stderr, "%s -- can't remove volume group %s with"
                           " %u logical volume", cmd, vg_name, vg->lv_cur);
         if ( vg->lv_cur > 1) fprintf ( stderr, "s\n");
         continue;
      }
   
      /* init physical volumes */
      if ( opt_v > 0) printf ( "%s -- giving free physical volumes of "
                               "volume group %s\n",
                               cmd, vg_name);
      for ( p = 0; vg->pv[p] != NULL; p++) {
         if ( opt_v > 0) printf ( "%s -- freeing physical volume %s\n",
                                  cmd, pv[p]->pv_name);
         pv_setup_for_create ( vg->pv[p]->pv_name, vg->pv[p],
                               vg->pv[p]->pv_size);
      }
   
      if ( opt_v > 0) printf ( "%s -- storing physical volume data of "
                               "volume group %s on disk(s)\n",
                               cmd, vg_name);
   
      lvm_dont_interrupt ( 0);
   
      if ( ( ret = pv_write_all_pv_of_vg ( vg)) != 0) {
         fprintf ( stderr, "%s -- can't remove all %u physical volumes "
                           "of volume group %s\n\n",
                           cmd, vg->pv_cur, vg_name);
         return 1;
      }
   
      if ( opt_v > 0) printf ( "%s -- removing special files of "
                               "volume group %s\n", cmd, vg_name);
      vg_remove_dir_and_group_and_nodes ( vg_name);
   
   
      if ( opt_v > 0) printf ( "%s -- removing lvmtab of volume group %s",
                               cmd, vg_name);
      if ( ( ret = lvm_tab_vg_remove ( vg_name)) < 0) {
         fprintf ( stderr, "%s -- ERROR %d removing %s from %s\n",
                           cmd, ret, vg_name, LVMTAB);
         continue;
      }
   
      printf ( "%s -- volume group %s successfully removed\n", cmd, vg_name);

      lvm_interrupt ();
   }

   LVM_UNLOCK ( 0);

   printf ( "\n");
   return 0;
}
