/*
 * Author: Heinz Mauelshagen, Germany
 *
 * June 1998
 *
 * LVM is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * LVM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU CC; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA. 
 *
 */

#include <lvm_user.h>

char *cmd = NULL;

#ifdef DEBUG
int opt_d = 0;
#endif

int main ( int argc, char **argv) {
   int c = 0;
   int l = 0;
   int np = 0;
   int np_sav = 0;
   int opt_A = 1;
   int opt_l = 0;
   int opt_t = 0;
   int opt_v = 0;
   int p = 0;
   int ret = 0;
   int v = 0;
#ifdef DEBUG
   char *options = "A:dh?ltv";
#else
   char *options = "A:h?ltv";
#endif
   char *vg_name_to = NULL;
   char *vg_name_from = NULL;
   vg_t *vg_to = NULL;
   vg_t *vg_from = NULL;
   vg_t vg_to_this;
   vg_t vg_from_this;

   cmd = basename ( argv[0]);

   SUSER_CHECK;
   LVMTAB_CHECK;

   while ( ( c = getopt ( argc, argv, options)) != EOF) {
      switch ( c) {
         case 'A':
            if ( opt_A > 1) {
               fprintf ( stderr, "%s -- A option yet given\n\n", cmd);
               return 1;
            }
            if ( strcmp ( optarg, "y") == 0);
            else if ( strcmp ( optarg, "n") == 0) opt_A = 0;
            else {
               fprintf ( stderr, "%s -- invalid option argument %s\n\n",
                                 cmd, optarg);
               return 1;
            }
            break;

#ifdef DEBUG
         case 'd':
            if ( opt_d > 0) {
               fprintf ( stderr, "%s -- d option yet given\n\n", cmd);
               return 1;
            }
            opt_d++;
            break;
#endif

         case 'h':
         case '?':
            printf ( "\n%s\n\n%s -- Volume Group Merge\n\n"
                     "Synopsis:\n"
                     "---------\n\n"
                     "%s\n"
                     "\t[-A y/n]\n"
#ifdef DEBUG
                     "\t[-d]\n"
#endif
                     "\t[-h/-?]\n"
                     "\t[-l]\n"
                     "\t[-t]\n"
                     "\t[-v]\n"
                     "\tDestinationVolumeGroupName SourceVolumeGroupName\n\n",
                     lvm_version, cmd, cmd);
            return 0;
            break;

         case 'l':
            if ( opt_l > 0) {
               fprintf ( stderr, "%s -- l option yet given\n\n", cmd);
               return 1;
            }
            opt_l++;
            break;

         case 't':
            if ( opt_t > 0) {
               fprintf ( stderr, "%s -- t option yet given\n\n", cmd);
               return 1;
            }
            opt_t++;
            break;

         case 'v':
            if ( opt_v > 0) {
               fprintf ( stderr, "%s -- v option yet given\n\n", cmd);
               return 1;
            }
            opt_v++;
            break;

         default:
            fprintf ( stderr, "%s -- invalid command line option \"%c\"\n\n",
                      cmd, c);
            return 1;
      }
   }
  
   if ( optind != argc - 2) {
      fprintf ( stderr, "%s -- please enter 2 volume group names\n\n", cmd);
      return 1;
   }

   for ( v = optind; v < argc; v++) {
      if ( vg_check_name ( argv[v]) < 0) {
         fprintf ( stderr, "%s -- invalid volume group name %s\n\n",
                           cmd, argv[v]);
         return 1;
      }
   }

   vg_name_to = argv[optind++];
   vg_name_from = argv[optind];

   LVM_LOCK ( 0);

   if ( opt_v > 0) printf ( "%s -- checking for volume group %s activity\n",
                            cmd, vg_name_from);
   if ( vg_check_active ( vg_name_from) == TRUE) {
      fprintf ( stderr, "%s -- ERROR: volume group %s must be inactive\n\n",
                        cmd, vg_name_from);
      return 1;
   }

   if ( opt_v > 0) printf ( "%s -- reading volume group data of %s\n",
                            cmd, vg_name_to);
   if ( ( ret = lvm_tab_vg_read_with_pv_and_lv ( vg_name_to, &vg_to)) != 0) {
      if ( ret != -LVM_EVG_CFGRESTORE_OPEN) {
         fprintf ( stderr, "%s -- ERROR %d: couldn't get volume "
                           "group data of %s\n\n",
                           cmd, ret, vg_name_to);
      } else fprintf ( stderr, "\n");
      return 1;
   }

   memcpy ( &vg_to_this, vg_to, sizeof ( vg_to_this));
   vg_to = &vg_to_this;

   if ( opt_v > 0) printf ( "%s -- reading volume group data of %s\n",
                            cmd, vg_name_from);
   if ( ( ret = lvm_tab_vg_read_with_pv_and_lv ( vg_name_from, &vg_from)) != 0) {
      fprintf ( stderr, "%s -- ERROR %d: couldn't get volume "
                        "group data of %s\n\n",
                        cmd, ret, vg_name_from);
      return 1;
   }

   memcpy ( &vg_from_this, vg_from, sizeof ( vg_from_this));
   vg_from = &vg_from_this;

   np = 0;
   while ( vg_to->pv[np] != NULL) np++;
   np_sav = np;


   if ( opt_v > 0) printf ( "%s -- merging volume group descriptor area of "
                            "%s into %s\n", cmd, vg_name_from, vg_name_to);
   if ( ( ret = vg_setup_for_merge ( vg_to, vg_from)) < 0) {
      if ( ret == -LVM_EVG_SETUP_FOR_MERGE_PE_SIZE) {
         fprintf ( stderr, "%s -- ERROR: physical extend sizes of %s "
                           "and %s differ\n\n",
                           cmd, vg_name_from, vg_name_to);
      } else if ( ret == -LVM_EVG_SETUP_FOR_MERGE_PV_MAX) {
         fprintf ( stderr, "%s -- ERROR: maximum physical volume count "
                           "exceeded for volume group %s\n\n",
                           cmd, vg_name_to);
      } else if ( ret == -LVM_EVG_SETUP_FOR_MERGE_LV_MAX) {
         fprintf ( stderr, "%s -- ERROR: maximum logical volume count "
                           "exceeded for volume group %s\n\n",
                           cmd, vg_name_to);
      } else {
         fprintf ( stderr, "%s -- ERROR &d merging volume group %s into %s\n\n",
                           cmd, vg_name_from, vg_name_to);
      }
      return 1;
   }

   if ( opt_l > 0) vg_show_with_pv_and_lv ( vg_to);

   lvm_dont_interrupt ( 0);

   if ( vg_check_active ( vg_name_to) == TRUE) {
      /* extend vg_to with new physical volumes */
      for ( ; vg_to->pv[np] != NULL; np++) {
         if ( opt_v > 0) printf ( "%s -- extending volume group %s with "
                                  "%s in kernel\n",
                                  cmd, vg_name_to, vg_to->pv[np]->pv_name);
         if ( opt_t == 0 &&
              ( ret = vg_extend ( vg_name_to, vg_to->pv[np], vg_to)) < 0) {
            fprintf ( stderr, "%s -- ERROR %d extending %s with %s in kernel\n",
                              cmd, ret, vg_name_to, vg_to->pv[np]->pv_name);
            for ( p = np_sav; p < np; p++) vg_reduce ( vg_name_to,
                                                       vg_to->pv[p],
                                                       vg_to);
            return 1;
         }
      }
   
      /* extend vg_to with new logical volumes */
      for ( l = 0; l < vg_to->lv_max; l++) {
         if ( vg_to->lv[l] != NULL) {
            if ( opt_t == 0 &&
                 ( ret = lv_create ( vg_to,
                                     vg_to->lv[l],
                                     vg_to->lv[l]->lv_name)) < 0) {
               if ( ret != -EEXIST) {
                  fprintf ( stderr, "%s -- ERROR %d creating logical volume "
                                    "%s in kernel\n",
                                    cmd, ret, vg_to->lv[l]->lv_name);
               }
            }
         }
      }
   }

   /* store vg on disk(s) */
   if ( opt_v > 0) printf ( "%s -- storing volume group data of "
                            "%s on disk(s)\n", cmd, vg_name_to);
   if ( opt_t == 0 &&
        ( ret = vg_write_with_pv_and_lv ( vg_to)) < 0) {
      fprintf ( stderr, "%s -- ERROR %d storing volume group data "
                        "of %s on disk(s)\n\n", cmd, ret, vg_name_to);
      return 1;
   }

   if ( opt_v > 0) printf ( "%s -- changing lvmtab\n", cmd);
   if ( opt_t == 0 &&
        vg_cfgbackup ( vg_name_to, LVMTAB_DIR, opt_v, vg_to) == 0 &&
        opt_A > 0) {
      printf ( "%s -- doing automatic backup of %s\n", cmd, vg_name_to);
      vg_cfgbackup ( vg_name_to, VG_BACKUP_DIR, opt_v, vg_to);
   }

   if ( opt_v > 0) printf ( "%s -- removing special files for volume "
                            "group %s\n", cmd, vg_name_from);
   if ( opt_t == 0) vg_remove_dir_and_group_and_nodes ( vg_name_from);

   if ( opt_v > 0) printf ( "%s -- removing special files for volume "
                            "group %s\n", cmd, vg_name_from);
   if ( opt_t == 0) vg_remove_dir_and_group_and_nodes ( vg_name_to);

   if ( opt_v > 0) printf ( "%s -- creating directory and nodes of %s\n",
                            cmd, vg_name_to);
   if ( opt_t == 0 &&
        ( ret = vg_create_dir_and_group_and_nodes ( vg_to, opt_v)) < 0) {
      fprintf ( stderr, "%s -- ERROR %d: creating volume group "
                        "nodes of %s\n\n",
                        cmd, ret, vg_name_to);
      return 1;
   }

   if ( opt_v > 0) printf ( "%s -- removing lvmtab of volume group %s\n",
                            cmd, vg_name_from);
   if ( opt_t == 0 &&
        ( ret = lvm_tab_vg_remove ( vg_name_from)) < 0) {
      fprintf ( stderr, "%s -- ERROR %d removing %s from %s\n",
                        cmd, ret, vg_name_from, LVMTAB);
      return 1;
   }

   printf ( "%s -- volume group %s successfully merged into %s\n",
             cmd, vg_name_from, vg_name_to);
   if ( opt_t > 0) printf ( "%s -- this only has been a test run\n", cmd);

   lvm_interrupt ();
   LVM_UNLOCK ( 0);

   printf ( "\n");

   if ( ret == 0) return 0;
   else           return 1;
}
