/*
 * Author: Heinz Mauelshagen, Germany
 *
 * March 1997
 * April-June 1998
 *
 * LVM is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * LVM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU CC; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA. 
 *
 */

/* Changelog
 *
 *    11/09/1997 - added lvmtab handling
 *    04/30/1998 - changed to lv_status_byname
 *    05/16/1998 - added lvmtab checking
 *    06/27/1998 - changed lvm_tab_* calling convention
 *
 */


#include <lvm_user.h>

char *cmd = NULL;

#ifdef DEBUG
int opt_d = 0;
#endif

int main ( int argc, char **argv)
{
   int c = 0;
   int c1 = 0;
   int opt_A = 1;
   int opt_f = 0;
   int opt_v = 0;
   int ret = 0;
#ifdef DEBUG
   char *options = "A:dfh?v";
#else
   char *options = "A:fh?v";
#endif
   char *vg_name = NULL;
   char *lv_name = NULL;
   vg_t *vg = NULL;
   lv_t *lv = NULL;

   cmd = basename ( argv[0]);

   SUSER_CHECK;
   LVMTAB_CHECK;

   while ( ( c = getopt ( argc, argv, options)) != EOF) {
      switch ( c) {
         case 'A':
            if ( opt_A > 1) {
               fprintf ( stderr, "%s -- A option yet given\n\n", cmd);
               return 1;
            }
            if ( strcmp ( optarg, "y") == 0);
            else if ( strcmp ( optarg, "n") == 0) opt_A = 0;
            else {
               fprintf ( stderr, "%s -- invalid option argument %s\n\n",
                                 cmd, optarg);
               return 1;
            }
            break;

#ifdef DEBUG
         case 'd':
            if ( opt_d > 0) {
               fprintf ( stderr, "%s -- d option yet given\n\n", cmd);
               return 1;
            }
            opt_d++;
            break;
#endif

         case 'h':
         case '?':
            printf ( "\n%s\n\n%s -- Logical Volume Remove\n\n"
                     "Synopsis:\n"
                     "---------\n\n"
                     "%s\n"
                     "\t[-A y/n]\n"
#ifdef DEBUG
                     "\t[-d]\n"
#endif
                     "\t[-f]\n"
                     "\t[-h/?]\n"
                     "\t[-v]\n"
                     "\tLogicalVolumePath [LogicalVolumePath...]\n\n",
                     lvm_version, cmd, cmd);
            return 0;
            break;

         case 'f':
            if ( opt_f > 0) {
               fprintf ( stderr, "%s -- f option yet given\n\n", cmd);
               return 1;
            }
            opt_f++;
            break;

         case 'v':
            if ( opt_v > 0) {
               fprintf ( stderr, "%s -- v option yet given\n\n", cmd);
               return 1;
            }
            opt_v++;
            break;

         default:
            fprintf ( stderr, "%s -- invalid command line option \"%c\"\n",
                      cmd, c);
            return 1;
      }
   }

   if ( optind == argc) {
      fprintf ( stderr, "%s -- please enter a logical volume path\n\n", cmd);
      return 1;
   }

   LVM_LOCK ( 0);
   LVM_CHECK_IOP;

   for ( ; optind < argc; optind++) {
      lv_name = argv[optind];
   
      if ( opt_v > 0) printf ( "%s -- checking logical volume path %s\n",
                               cmd, lv_name);
      if ( ( ret = lv_check_name ( lv_name)) < 0) {
         fprintf ( stderr, "%s -- ERROR: no valid logical volume "
                           "name %s\n", cmd, lv_name);
         continue;
      }
   
      /* does VG exist? */
      vg_name = vg_name_of_lv ( lv_name);

      if ( opt_v > 0) printf ( "%s -- checking existence of volume group %s\n",
                               cmd, vg_name);
      if ( lvm_tab_vg_check_exist ( vg_name, NULL) != TRUE) {
         fprintf ( stderr, "%s -- volume group %s doesn't exist\n",
                   cmd, vg_name);
         continue;
      }
   
      if ( opt_v > 0) printf ( "%s -- checking volume group %s activity\n",
                               cmd, vg_name);
      if ( vg_check_active ( vg_name) == FALSE) {
         fprintf ( stderr, "%s -- can't remove logical volume in "
                           "inactive volume group %s\n", cmd, vg_name);
         continue;
      }
   
      if ( opt_v > 0) printf ( "%s -- getting status of volume group %s "
                               "from VGDA in kernel\n", cmd, vg_name);
      if ( vg_status ( vg_name, &vg) < 0) {
         fprintf ( stderr, "%s -- can't remove logical volume in inactive"
                           " volume group %s\n", cmd, vg_name);
         continue;
      }
   
      if ( opt_v > 0) printf ( "%s -- getting status of logical volume "
                               "%s from VGDA in kernel\n", cmd, lv_name);
      if ( ( ret = lv_status_byname ( vg_name, lv_name, &lv)) < 0) {
         if ( ret == -ENXIO) {
            fprintf ( stderr, "%s -- logical volume %s doesn't exist\n",
                              cmd, lv_name);
         } else {
            fprintf ( stderr, "%s -- ERROR %d getting status"
                              " of logical volume %s\n",
                              cmd, ret, lv_name);
         }
         continue;
      }
   
      if ( lv->lv_open > 0) {
         fprintf ( stderr, "%s -- can't remove open logical volume %s\n",
                           cmd, lv_name);
         continue;
      }
   
      if ( opt_v > 0) printf ( "%s -- reading volume group data of %s "
                               "from disk(s)\n", cmd, lv_name);
      if ( ( ret = lvm_tab_vg_read_with_pv_and_lv ( vg_name, &vg)) != 0) {
         fprintf ( stderr, "%s -- ERROR %d reading volume group data of %s\n",
                           cmd, ret, vg_name);
         continue;
      }

      if ( opt_f == 0) {
         c = 0;
         while ( c != 'y' && c != 'n') {
            if ( c == '\n' || c == 0)
               printf ( "%s -- do you really want to remove %s? [y/n]: ",
                        cmd, lv_name);
            c = tolower ( getchar ());
         }
         c1 = c;
         while ( c != '\n') c = tolower ( getchar ());
         if ( c1 == 'n') {
            printf ( "%s -- %s not removed\n", cmd, lv_name);
            continue;
         }
      }

      /* release it in structures */
      if ( opt_v > 0) printf ( "%s -- releasing logical volume %s\n",
                               cmd, lv_name);
      if ( ( ret = lv_release ( vg, lv_name)) != 0) {
         fprintf ( stderr, "%s -- ERROR %d removing %s\n", 
                   cmd, ret, lv_name);
         continue;
      }
   
   
      lvm_dont_interrupt ( 0);
   
      /* remove it from kernel */
      if ( opt_v > 0) printf ( "%s -- removing logical volume from VGDA "
                               "in kernel\n", cmd);
      if ( ( ret = lv_remove ( vg, lv, lv_name)) != 0) {
         fprintf ( stderr, "%s -- ERROR %d removing %s\n", 
                   cmd, ret, lv_name);
         lvm_interrupt ();
         continue;
      }
   
      if ( opt_v > 0) printf ( "%s -- unlinking special file %s\n",
         cmd, lv_name);
      if ( unlink ( lv_name) == -1)
         fprintf ( stderr, "%s -- ERROR %d unlinking %s\n",
                           cmd, errno, lv_name);
   
      /* store it on disks */
      if ( opt_v > 0) printf ( "%s -- storing volume group data on disk(s)\n",
                               cmd);
      if ( ( ret = vg_write_with_pv_and_lv ( vg)) != 0) {
         fprintf ( stderr, "%s -- ERROR %d storing volume group data of %s"
                           " on disks\n\n", 
                   cmd, ret, vg_name);
         lvm_interrupt ();
         return 1;
      }


      if ( opt_v > 0) printf ( "%s -- changing lvmtab\n", cmd);
      if ( vg_cfgbackup ( vg_name, LVMTAB_DIR, opt_v, vg) == 0 &&
           opt_A > 0) {
         printf ( "%s -- doing automatic backup of %s\n", cmd, vg_name);
         vg_cfgbackup ( vg_name, VG_BACKUP_DIR, opt_v, vg);
      }

      lvm_interrupt ();

      printf ( "%s -- logical volume %s successfully removed\n",
               cmd, lv_name);
      if ( argc - optind > 1 && opt_v > 0) printf ( "\n");
   }

   LVM_UNLOCK ( 0);

   printf ( "\n");
   return 0;
}
