/*
 * Author: Heinz Mauelshagen, Germany
 *
 * May 1997
 * April-May 1998
 *
 * LVM is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * LVM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU CC; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA. 
 *
 */

/*
 * Changelog
 *
 *    11/09/1997 - added lvmtab handling
 *    02/12/1998 - added missing lvm_tab_vg_insert call
 *               - corrected VG number
 *    04/30/1998 - changed to new lv_create_kdev_t() calling convention
 *    05/03/1998 - enhanced error checking with lvm_tab_vg_insert()
 *               - avoided lv_create_name()
 *    05/16/1998 - added lvmtab checking
 *    05/17/1998 - obsoleted physical volume name correction (now in pv_read())
 *
 */

#include <lvm_user.h>

#ifdef DEBUG
int opt_d = 0;
#endif

int main ( int argc, char **argv) {
   int c = 0;
   int l = 0;
   int p = 0;
   int p1 = 0;
   int pv_number = 0;
   int pv_number_this = 0;
   int *pv_number_count = NULL;
   int ret = 0;
   char *pv_name = NULL;
   char *vg_name = NULL;
   char vg_name_this[NAME_LEN] = { 0, };
#ifdef DEBUG
   char *options = "dh?v";
#else
   char *options = "h?v";
#endif
   int opt_v = 0;
   char *cmd = NULL;
   pv_t *pv = NULL;
   pv_t **pv_this = NULL;
   vg_t *vg = NULL;

   cmd = basename ( argv[0]);

   SUSER_CHECK;
   LVMTAB_CHECK;

   while ( ( c = getopt ( argc, argv, options)) != EOF) {
      switch ( c) {
#ifdef DEBUG
         case 'd':
            if ( opt_d > 0) {
               fprintf ( stderr, "%s -- d option yet given\n\n", cmd);
               return 1;
            }
            opt_d++;
            break;
#endif

         case 'h':
         case '?':
            printf ( "\n%s\n\n%s -- Volume group import\n\n"
                     "Synopsis:\n"
                     "---------\n\n"
                     "%s\n"
#ifdef DEBUG
                     "\t[-d]\n"
#endif
                     "\t[-h/?]\n"
                     "\t[-v]\n"
                     "\tVolumeGroupName PhysicalVolumePath "
                     "[PhysicalVolumePath...]\n\n",
                     lvm_version, cmd, cmd);
            return 0;
            break;

         case 'v':
            if ( opt_v > 0) {
               fprintf ( stderr, "%s -- v option yet given\n\n", cmd);
               return 1;
            }
            opt_v++;
            break;

         default:
            fprintf ( stderr, "%s -- invalid command line option \"%c\"\n\n",
                      cmd, c);
            return 1;
      }
   }
  
   if ( optind == argc) {
      fprintf ( stderr, "%s -- please give a volume group name\n\n", cmd);
      return 1;
   }

   vg_name = argv[optind++];
   if ( opt_v > 0) printf ( "%s -- checking volume group name\n", cmd);
   if ( vg_check_name ( vg_name) < 0) {
      fprintf ( stderr, "%s -- invalid volume group name %s\n\n",
                        cmd, vg_name);
      return 1;
   }

   if ( optind == argc) {
      fprintf ( stderr, "%s -- please enter a physical volume path\n\n", cmd);
      return 1;
   }

   LVM_LOCK ( 0);
   LVM_CHECK_IOP;

   if ( opt_v > 0) printf ( "%s -- checking volume group existence\n", cmd);
   if ( ( ret = lvm_tab_vg_check_exist ( vg_name, NULL)) == TRUE) {
      fprintf ( stderr, "%s -- a volume group named %s allready exists\n\n",
                cmd, vg_name);
      return 1;
   }

   /* get the physical volume data */
   if ( opt_v > 0) printf ( "%s -- trying to read the physical volumes\n", cmd);
   p = 0;
   for ( ; optind < argc; optind++) {
      pv_name = argv[optind];

      if ( opt_v > 0) printf ( "%s -- checking for multiple "
                               "physical volumes\n", cmd);
      for ( p1 = 0; pv_this != NULL && pv_this[p1] != NULL; p1++) {
         if ( strcmp ( pv_name, pv_this[p1]->pv_name) == 0) {
            fprintf ( stderr,
                      "%s -- ERROR: physical volume %s occurs "
                      "multiple times\n\n",
                      cmd, pv_name);
            return 1;
         }
      }

      if ( opt_v > 0) printf ( "%s -- checking physical volume path %s\n",
                               cmd, pv_name);
      if ( pv_check_name ( pv_name) < 0) {
         fprintf ( stderr, "%s -- invalid physical volume path %s\n\n",
                   cmd, pv_name);
         return 1;
      }

      if ( opt_v > 0) printf ( "%s -- reading physical volume data of %s "
                               "from disk\n", cmd, pv_name);
      if ( ( ret = pv_read ( pv_name, &pv, NULL)) < 0 &&
           ret != -LVM_EPV_READ_PV_EXPORTED) {
         fprintf ( stderr, "%s -- ERROR %d reading physical volume %s\n\n",
                   cmd, ret, pv_name);
         return 1;
      }

      if ( opt_v > 0) printf ( "%s -- checking for exported "
                               "physical volume %s\n", cmd, pv_name);
      if ( system_id_check_exported ( pv->system_id) == FALSE) {
         fprintf ( stderr, "%s -- %s doesn't belong to an exported "
                           "volume group\n\n", cmd, pv_name);
         return 1;
      }

      if ( opt_v > 0) printf ( "%s -- (re)allocating memory\n", cmd);
      if ( pv_this == NULL) pv_this = malloc  ( 2 * sizeof ( pv_t*));
      if ( ( pv_this = realloc ( pv_this,
                                 ( p + 2) * sizeof ( pv_t*))) == NULL) {
         fprintf ( stderr, "%s -- alloc error in file %s [line %d]\n\n",
                           cmd, __FILE__, __LINE__);
         return 1;
      }

      if ( ( pv_this[p] = malloc ( sizeof ( pv_t))) == NULL) {
         fprintf ( stderr, "%s -- malloc error in file %s [line %d]\n\n",
                           cmd, __FILE__, __LINE__);
         return 1;
      }
      bcopy ( pv, pv_this[p], sizeof ( pv_t));

      if ( opt_v > 0) printf ( "%s -- checking consistency of physical "
                               "volume %s\n", cmd, pv_name);
      if ( ( ret = pv_check_consistency ( pv_this[p])) < 0) {
         fprintf ( stderr, "%s -- ERROR %d checking physical "
                           "volume consistency of %s\n",
                   cmd, ret, pv_name);
         continue;
      }

      p++;
      pv_this[p] = NULL;
   }
   pv_number_this = p;

   if ( p == 0) {
      fprintf ( stderr, "%s -- no valid physical volumes found for "
                        "import of %s\n\n", cmd, vg_name);
      return 1;
   }

   /* check volume group consistency of found physical volumes */
   pv_number = pv_this[0]->pv_number;
   for ( p = 1; pv_this[p] != NULL; p++) {
      if ( strcmp ( pv_this[0]->vg_name, pv_this[p]->vg_name) != 0 ||
           strcmp ( pv_this[0]->system_id, pv_this[p]->system_id) != 0) {
         fprintf ( stderr, "%s -- %s doesn't belong to the same exported "
                           "volume group as %s\n\n",
                   cmd, pv_this[p]->pv_name, pv_this[0]->pv_name);
         return 1;
      }
      /* get the highest pv_number */
      if ( pv_number < pv_this[p]->pv_number) pv_number = pv_this[p]->pv_number;
   }

   if ( pv_number != pv_number_this) {
      fprintf ( stderr, "%s -- ERROR: wrong number of physical volumes "
                        "given to import %s\n\n", cmd, vg_name);
      return 1;
   }

   /* get the physical volume numbers counter array */
   if ( ( pv_number_count = malloc ( pv_number * sizeof ( int))) == NULL) {
      fprintf ( stderr, "%s -- malloc error in file %s [line %d]\n\n",
                        cmd, __FILE__, __LINE__);
      return 1;
   }
   memset ( pv_number_count, 0, pv_number * sizeof ( int));

   /* fill the counter array to check physical volume numbers */
   for ( p = 0; pv_this[p] != NULL; p++)
      pv_number_count[pv_this[p]->pv_number-1]++;

   for ( p = 0; p < pv_number; p++) {
      if ( pv_number_count[p] != 1) {
         fprintf ( stderr, "%s -- physical volume number %lu "
                           "of %s occurs %d times\n\n",
                            cmd, pv_this[p]->pv_number,
                            pv_this[p]->pv_name, pv_number_count[p]);
         return 1;
      }
   }
   free ( pv_number_count);
   pv_number_count = NULL;

   /* read the exported volume group */
   memset ( vg_name_this, 0, NAME_LEN);
   strncpy ( vg_name_this,
             pv_this[0]->vg_name,
             strlen ( pv_this[0]->vg_name) - strlen ( EXPORTED));
   if ( opt_v > 0) printf ( "%s -- reading the exported volume group data "
                            " of %s from disk(s)\n", cmd, vg_name_this);
   if ( ( ret = vg_read_with_pv_and_lv ( pv_this[0]->vg_name, &vg)) < 0) {
      fprintf ( stderr, "%s -- ERROR %d reading volume group data of "
                        "exported volume group %s\n\n",
                cmd, ret, vg_name_this);
      return 1;
   }

   if ( pv_number_this != vg->pv_cur) {
      fprintf ( stderr, "%s -- ERROR: only %d physical volumes of %lu "
                        "given\n\n", cmd, pv_number_this, vg->pv_cur);
      return 1;
   }

   /* continue checking consistency in red exported volume group */
   if ( opt_v > 0) printf ( "%s -- checking for correct system ids on "
                            "physical volumes of %s\n", cmd, vg->vg_name);
   for ( p = 1; vg->pv[p] != NULL; p++) {
      if ( strcmp ( vg->pv[0]->vg_name, vg->pv[p]->vg_name) != 0 ||
           strcmp ( vg->pv[0]->system_id, vg->pv[p]->system_id) != 0) {
         fprintf ( stderr, "%s -- %s has a different system_id than %s\n\n",
                   cmd, vg->pv[p]->pv_name, vg->pv[0]->pv_name);
         return 1;
      }
   }

   if ( opt_v > 0) printf ( "%s -- checking exported volume group "
                            "%s consistency\n", cmd, vg->vg_name);
   if ( ( ret = vg_check_consistency_with_pv_and_lv ( vg)) < 0 &&
        ret != -LVM_EVG_CHECK_CONSISTENCY_VG_STATUS) {
      fprintf ( stderr, "%s -- ERROR %d: exported volume group %s "
                        "is inconsistent\n"
                        "%s -- it's system id is \"%s\"\n\n",
                        cmd, ret, vg_name_this, cmd, vg->pv[0]->system_id);
      return 1;
   }

   if ( vg->lv_cur > MAX_LV || vg->lv_max > MAX_LV) {
      fprintf ( stderr, "%s -- too many logical volumes to import in "
                        "exported volume group %s\n\n",
                        cmd, vg_name_this);
      return 1;
   }

   /* correct the volume group name */
   strcpy ( vg->vg_name, vg_name);

   if ( opt_v > 0) printf ( "%s -- importing physical volumes of %s\n",
                            cmd, vg->vg_name);
   for ( p = 0; vg->pv[p] != NULL; p++) {
      system_id_set_imported ( vg->pv[p]->system_id);
      strcpy ( vg->pv[p]->vg_name, vg_name);
   }

   lvm_dont_interrupt ( 0);

   if ( opt_v > 0) printf ( "%s -- getting free volume group number for %s\n\n",
                            cmd, vg_name);
   if ( ( vg->vg_number = lvm_tab_get_free_vg_number ()) < 0) {
      fprintf ( stderr, "%s -- ERROR getting free volume group "
                        "number for %s\n\n",
                cmd, vg_name);
      return 1;
   }

   if ( opt_v > 0) printf ( "%s -- importing logical volumes of %s\n",
                            cmd, vg->vg_name);
   for ( l = 0; l < vg->lv_max; l++) {
      if ( vg->lv[l] != NULL) {
         strcpy ( vg->lv[l]->vg_name, vg_name);
         strcpy ( vg->lv[l]->lv_name,
                  lv_change_vgname ( vg_name, vg->lv[l]->lv_name));
         vg->lv[l]->lv_dev = lv_create_kdev_t ( vg->vg_number, l + 1);
      }
   }

   vg->vg_status &= ~VG_EXPORTED;

/* TEST without path information ( lv_create_kdev_t() won't work...)
   for ( l = 0; l < vg->lv_max && vg->lv[l] != NULL; l++) {
      if ( ( lv_name_ptr = strrchr ( vg->lv[l]->lv_name, '/')) != NULL) {
         lv_name_ptr++;
         strcpy ( vg->lv[l]->lv_name, lv_name_ptr);
      }
   }
*/

   if ( opt_v > 0) printf ( "%s -- storing imported volume group data of %s "
                            "on disk(s)\n", cmd, vg_name);
   if ( ( ret = vg_write_with_pv_and_lv ( vg)) < 0) {
      fprintf ( stderr, "%s -- ERROR %d storing imported volume group data "
                        "of %s on disks\n\n",
                cmd, ret, vg_name);
      return 1;
   }

   if ( ( ret = vg_create_dir_and_group_and_nodes ( vg, cmd, opt_v)) < 0)
      fprintf ( stderr, "%s -- %d errors creating volume group "
                        "directory and special files\n", cmd, -ret);

   /* create VGDA in kernel */
   if ( opt_v > 0) printf ( "%s -- creating VGDA in kernel\n", cmd);
   if ( ( ret = vg_create ( vg_name, vg)) < 0) {
      fprintf ( stderr, "%s -- couldn't create VGDA for imported volume "
                        "group %s in kernel\n", cmd, vg_name);
      if ( ret == -LVM_EVG_CREATE_REMOVE_OPEN)
         fprintf ( stderr, "%s -- LVM not in kernel?\n", cmd);
      fprintf ( stderr, "\n");
      return 1;
   }

   if ( opt_v > 0) printf ( "%s -- inserting %s into %s\n",
                            cmd, vg_name, LVMTAB);
   if ( ( ret = lvm_tab_vg_insert ( vg_name)) < 0) {
      fprintf ( stderr, "%s -- ERROR %d inserting %s into %s\n\n",
                        cmd, ret, vg_name, LVMTAB);
      return 1;
   }

   if ( ( ret = vg_cfgbackup ( vg_name, LVMTAB_DIR, cmd, opt_v, vg)) == 0) {
      printf ( "%s -- doing automatic backup of %s\n", cmd, vg_name);
      if ( ( ret = vg_cfgbackup ( vg_name, VG_BACKUP_DIR, cmd,
                                  opt_v, vg)) < 0) {
         fprintf ( stderr, "%s -- ERROR %d writing VG backup\n\n", cmd, ret);
         return 1;
      }
   } else {
      fprintf ( stderr, "%s -- ERROR %d writing lvmtab\n\n", cmd, ret);
      return 1;
   }


   lvm_interrupt ();
   LVM_UNLOCK ( 0);

   printf ( "%s -- volume group %s successfully imported and activated\n\n",
            cmd, vg_name);

   return 0;
}
