/*
 * Author: Heinz Mauelshagen, Germany
 *
 * March-July,November 1997
 * January,May 1998
 *
 * LVM is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * LVM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU CC; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA. 
 *
 */

/*
 * Changelog
 *
 *    10/07/1997 - option s implemented
 *    11/09/1997 - added use /etc/lvmtab in addition direct disk access (opt_D)
 *               - added -a option
 *    05/16/1998 - added lvmtab check
 *    01/18/1998 - added output of free volume group size to option s output
 *
 */

#include <lvm_user.h>

#ifdef DEBUG
int opt_d = 0;
#endif

int main ( int argc, char **argv)
{
   int c = 0;
   int exit_code = 0;
   int opt_A = 0;
   int opt_D = 0;
   int opt_s = 0;
   int opt_v = 0;
   int ret = 0;
   char *cmd = NULL;
   char *dummy = NULL;
#ifdef DEBUG
   char *options = "AdDh?sv";
#else
   char *options = "ADh?sv";
#endif
   char *vg_name = NULL;
   vg_t *vg = NULL;

   cmd = basename ( argv[0]);

   SUSER_CHECK;
   LVMTAB_CHECK;

   while ( ( c = getopt ( argc, argv, options)) != EOF) {
      switch ( c) {
         case 'A':
            if ( opt_A > 0) {
               fprintf ( stderr, "%s -- A option yet given\n\n", cmd);
               return 1;
            }
            opt_A++;
            break;

#ifdef DEBUG
         case 'd':
            if ( opt_d > 0) {
               fprintf ( stderr, "%s -- d option yet given\n\n", cmd);
               return 1;
            }
            opt_d++;
            break;
#endif

         case 'D':
            if ( opt_D > 0) {
               fprintf ( stderr, "%s -- D option yet given\n\n", cmd);
               return 1;
            }
            opt_D++;
            break;

         case 'h':
         case '?':
            printf ( "\n%s\n\n%s -- Volume group display\n\n"
                     "Synopsis:\n"
                     "---------\n\n"
                     "%s\n"
                     "\t[-A]\n"
#ifdef DEBUG
                     "\t[-d]\n"
#endif
                     "\t[-D]\n"
                     "\t[-h/?]\n"
                     "\t[-s]\n"
                     "\t[-v[v]]\n"
                     "\t[VolumeGroupName...]\n\n",
                     lvm_version, cmd, cmd);
            return 0;
            break;

         case 's':
            if ( opt_s > 0) {
               fprintf ( stderr, "%s -- s option yet given\n\n", cmd);
               return 1;
            }
            opt_s++;
            break;

         case 'v':
            if ( opt_v > 1) {
               fprintf ( stderr, "%s -- v option yet given two times\n\n", cmd);
               return 1;
            }
            opt_v++;
            break;

         default:
            fprintf ( stderr, "%s -- invalid command line option \"%c\"\n",
                      cmd, c);
            return 1;
      }
   }

   if ( opt_s > 0 && opt_v > 0) {
      fprintf ( stderr, "%s -- don't give options s and v together\n\n", cmd);
      return 1;
   }

   if ( opt_A > 0 && opt_D > 0) {
      fprintf ( stderr, "%s -- option D is not allowed with option A\n\n", cmd);
      return 1;
   }

   if ( optind < argc) {
      if ( opt_v > 1) printf ( "%s -- using volume group(s) on command line\n",
                               cmd);
      if ( opt_A > 0) {
         fprintf ( stderr, "%s -- option A is not allowed with volume group "
                           "names\n\n", cmd);
         return 1;
      }
   } else {
      if ( opt_v > 1) {
         printf ( "%s -- finding all ", cmd);
         if ( opt_A > 0) printf ( "active ");
         printf ( "volume group(s)\n");
      }
      if ( opt_A > 0) argv = vg_check_active_all_vg ();
      else            argv = lvm_tab_vg_check_exist_all_vg ();
      argc = optind = 0;
      if ( argv != NULL) while ( argv[argc] != NULL) argc++;
   }

   if ( optind == argc) {
      printf ( "%s -- no ", cmd);
      if ( opt_A > 0) printf ( "active ");
      printf ( "volume groups found\n\n");
      return VGDISPLAY_NOVG;
   }

   LVM_LOCK ( 1);
   LVM_CHECK_IOP;

   for ( ; optind < argc; optind++) {
      vg_name = argv[optind];

      if ( opt_v > 1) printf ( "%s -- checking volume group name %s\n",
                               cmd, vg_name);
      if ( vg_check_name ( vg_name) < 0) {
         fprintf ( stderr, "%s -- %s is an invalid volume group name\n",
                   cmd, vg_name);
         continue;
      }

      if ( opt_v > 1) printf ( "%s -- checking volume group %s existence\n",
                               cmd, vg_name);
      if ( ( ret = lvm_tab_vg_check_exist ( vg_name, NULL)) == TRUE ||
           ret == -LVM_EVG_READ_VG_EXPORTED ||
           ret == -LVM_EVG_READ_LVM_STRUCT_VERSION) {
         if ( ret == -LVM_EVG_READ_VG_EXPORTED)
            printf ( "%s -- WARNING: volume group %s is exported\n",
                     cmd, vg_name);
         if ( opt_s > 0) {
            printf ( "%s -- %s (", cmd, vg_name);
            if ( vg_check_active ( vg_name) != TRUE) printf ( "inactive)\n");
            else  {
               printf ( "active");
               vg_status ( vg_name, &vg);
               if ( vg->lv_open > 0) printf ( "/in use");
               else                  printf ( "/idle");
               printf ( ") %s / ",
                        ( dummy = show_size ( vg->pe_total * vg->pe_size / 2,
                                              SHORT)));
               free ( dummy); dummy = NULL;
               printf ( "%s free\n",
                        ( dummy = show_size ( ( vg->pe_total - vg->pe_allocated)
                                              * vg->pe_size / 2, SHORT)));
               free ( dummy); dummy = NULL;
            }
            continue;
         }
         if ( opt_D == 0) {
            if ( opt_v > 1) printf ( "%s -- checking volume group "
                                     "%s activity\n", cmd, vg_name);
            if ( vg_check_active ( vg_name) == TRUE) {
               if ( opt_v > 1) printf ( "%s -- getting VGDA of %s from "
                                        "kernel\n", cmd, vg_name);
               if ( opt_v > 0) {
                  if ( ( ret = vg_status_with_pv_and_lv ( vg_name, &vg,
                                                          TRUE)) < 0) {
                     fprintf ( stderr,
                               "%s -- ERROR %d getting status of %s\n",
                                cmd, ret, vg_name);
                     continue;
                  }
                  vg_show ( vg);
                  printf ( "\n");
                  lv_show_all_lv_of_vg ( vg);
                  printf ( "\n");
                  pv_show_all_pv_of_vg_short ( vg);
                  printf ( "\n");
               } else {
                  vg_status ( vg_name, &vg);
                  vg_show ( vg);
                  printf ( "\n");
               }
            } else {
               printf ( "%s -- %s is NOT active; try -D\n",
                        cmd, vg_name);
            }
         } else { /* display from disks */
            if ( opt_v > 1) printf ( "%s -- reading volume group data of "
                                     "%s from disk(s)\n", cmd, vg_name);
            if ( opt_v > 0) {
               if ( ( ret = vg_read_with_pv_and_lv ( vg_name,
                                                     &vg)) < 0 &&
                      ret != -LVM_EVG_READ_VG_EXPORTED) {
                  exit_code = 1;
                  if ( ret == -LVM_EVG_READ_LVM_STRUCT_VERSION) {
                     fprintf ( stderr,
                               "%s -- can't show volume group %s with "
                               "invalid physical volume version\n",
                                cmd, vg_name);
                  } else {
                     fprintf ( stderr,
                               "%s -- ERROR %d getting data of %s form "
                               "kernel\n",
                               cmd, ret, vg_name);
                  }
                  continue;
               }
               vg_show ( vg);
               printf ( "\n");
               lv_show_all_lv_of_vg ( vg);
               printf ( "\n");
               pv_show_all_pv_of_vg_short ( vg);
            } else {
               if ( ( ret = lvm_tab_vg_read ( vg_name, &vg)) < 0) {
                  exit_code = 1;
                  if ( ret == -LVM_EVG_READ_LVM_STRUCT_VERSION) {
                     fprintf ( stderr,
                               "%s -- can't show volume group %s with "
                               "invalid physical volume version\n",
                                cmd, vg_name);
                  } else {
                     fprintf ( stderr,
                               "%s -- ERROR %d reading data of %s "
                               "from disk(s)\n",
                               cmd, ret, vg_name);
                  }
                  continue;
               }
               else {
                  vg_show ( vg);
                  printf ( "\n");
               }
            }
         }
      } else if ( ret == -LVM_EPV_READ_ALL_PV_OF_VG_NP) {
         fprintf ( stderr, "%s -- volume group %s doesn't exist\n",
                   cmd, vg_name);
         exit_code = 1;
      } else if ( ret == -LVM_EVG_CHECK_EXIST_PV_COUNT) {
         fprintf ( stderr, "%s -- ERROR: not all physical volumes of "
                           "volume group %s online\n", cmd, vg_name);
         exit_code = 1;
      } else {
         fprintf ( stderr, "%s -- volume group %s not found\n",
                           cmd, vg_name);
         exit_code = 1;
      }
   }

   LVM_UNLOCK ( 1);

   printf ( "\n");
   return exit_code;
}
