#!/usr/bin/env python
# -*- Mode: Python; py-indent-offset: 4 -*-

try:
    from cString import *
except:
    from string import *

import sys
import os
import re
import exceptions
import errno
import getopt
import message

try:
    import cPickle
    pickle = cPickle
except:
    import pickle

declaration_serial = 0

sgml_id_trans = maketrans ('<>_, ~=!:*()+/', '-----denlmposv')

global_symbol_table = {}
namespace_symbol_tables = {}

full_process_phase = 0

def fix_for_sgml (string):
    return lower (translate (string, sgml_id_trans))

class Declaration:
    def __init__ (self, type):
        global declaration_serial
        self.serial = declaration_serial
        declaration_serial = declaration_serial + 1
        self.type = type
        self.children = []
        self.parent = None
        self.qualifier_ = None
        self.in_private_namespace_ = None
        self.name = None # symbol name
        self.qualified_name = None
        self.xref_id = None
        self.is_private = 0
        self.is_type = 0

    def add_child (self, child, context):
        if child.parent:
            message.die ("adding child decl to two parents")
        
        self.children.append (child)
        child.parent = self

    def spew (self, indent):
        for child in self.children:
            child.spew (indent + "  ")

    def get_xref_id (self):
        return None

    def get_scopes (self):
        scopes = []
        p = self.parent
        while p and p.type != 'header':
            scopes.append (p.name)
            p = p.parent

        scopes.reverse ()
        return scopes

    def is_global (self):
        if not self.parent:
            return 1
        elif self.parent.type == 'header' or self.parent.type == 'namespace':
            return 1
        else:
            return 0

    def qualifier (self):
        if not self.qualifier_:
            scopes = self.get_scopes ()
            if (scopes != []):
                self.qualifier_ = join (scopes, '::') + '::'
            else:
                self.qualifier_ = ''
        return self.qualifier_

    def in_private_namespace (self):
        if self.in_private_namespace_ == None:
            self.in_private_namespace_ = 0
            if self.parent == None:
                return 0
            p = self.parent
            while p.type != 'header':
                if p.type == 'namespace' and p.name == 'Private':
                    self.in_private_namespace_ = 1
                    break
                p = p.parent

            if not self.in_private_namespace_:
                # we can still be considered private by association
                # if we require private types to work
                if self.type == 'method' or self.type == 'function' or \
                   self.type == 'typedef':
                    if find ('Private::', self.decl) >= 0:
                        self.in_private_namespace_ = 1

                
        return self.in_private_namespace_

    def lookup (self, symbol, required_types):
        if hasattr (self, 'get_symbol_table'):
            syms = self.get_symbol_table ()
            if syms.has_key (symbol):
                decl = syms[symbol]
                if decl.type in required_types:
                    return decl

        if self.parent:
            return self.parent.lookup (symbol, required_types)
        else:
            return None

    def get_child_types (self):
        return []

    def get_children_recursively (self):
        if not self.children:
            return []
        retval = []
        for c in self.children:
            retval.append (c)
            retval = retval + c.get_children_recursively ()

        return retval        

    def get_header (self):
            p = self
            while p and p.type != 'header':
                p = p.parent

            if p and p.type == 'header':
                return p
            else:
                return None

class ScopingDecl (Declaration):
    def __init__(self, type):
        Declaration.__init__(self, type)
        self.symbols = None

    def get_symbol_table (self):
        if not full_process_phase:
            message.die ("Called get_symbol_table without full symbols")
        
        if self.symbols:
            return self.symbols
        else:
            self.symbols = {}
            for c in self.children:
                # immediate children don't need
                # to be scoped
                if c.name:
                    self.symbols[c.name] = c

                # now merge the child's symbol table,
                # if appropriate, adding the child's
                # qualifier to each symbol name
                if hasattr (c, 'get_symbol_table'):
                    child_symbols = c.get_symbol_table ()
                    child_qual = c.name + '::'
                    for symname in child_symbols.keys ():
                        scoped = child_qual + symname
                        self.symbols[scoped] = child_symbols[symname]

            return self.symbols

    def get_child_types (self):
        retval = []
        for c in self.children:
            if c.is_type:
                retval.append (c)
            retval = retval + c.get_child_types ()

        return retval
        
        
class Header (ScopingDecl):
    def __init__ (self):
        ScopingDecl.__init__(self, 'header')
        self.name = None
        self.topic_docs = []

    def set_name (self, name):
        self.name = strip(name)

    def spew (self, indent):
        print indent + "header " + self.name + '\n'
        Declaration.spew (self, indent)

    def add_topic_doc (self, doc):
        self.topic_docs.append (doc)

    def set_doc (self, doc):
        Declaration.set_doc (self, doc)
        if doc.type != 'header':
            message.die ("tried to document a header with wrong doc type: " + doc.type)

    def get_xref_id (self):
        if not self.xref_id:
            id = 'header-' + replace (self.name[:-2], '/', '-')
            self.xref_id = fix_for_sgml (id)
        return self.xref_id

    def get_symbol_table (self):
        return global_symbol_table
        
class Namespace (ScopingDecl):
    def __init__ (self):
        ScopingDecl.__init__(self, 'namespace')
        self.name = None

    def set_name (self, name):
        self.name = strip(name)

    def spew (self, indent):
        print indent + "namespace " + self.name + '\n'
        Declaration.spew (self, indent)

    def set_doc (self, doc):
        Declaration.set_doc (self, doc)
        message.die ("tried to document a namespace; not supported")

    def get_symbol_table (self):
        if not full_process_phase:
            message.die ("Called get_symbol_table without full symbols")
        
        if self.symbols:
            return self.symbols
        else:
            global namespace_symbol_tables
            
            namespace_table = None
            if namespace_symbol_tables.has_key (self.qualified_name):
                namespace_table = namespace_symbol_tables[self.qualified_name]
            else:
                # we're the first decl of our namespace to try this
                namespace_table = {}
                namespace_symbol_tables[self.qualified_name] = namespace_table

            ScopingDecl.get_symbol_table (self) # set self.symbols
            # copy to namespace-wide table
            for k in self.symbols.keys ():
                namespace_table[k] = self.symbols[k]
            # our symbols are in the shared table, so
            # use that.
            self.symbols = namespace_table

            return self.symbols

    def get_xref_id (self):
        if not self.xref_id:
            id = 'namespace-'
            scopes = self.get_scopes ()
            id = id + join (scopes, '-')
            id = id +  '-' + self.name
            self.xref_id = fix_for_sgml (id)
            # uniquify this instance of the namespace
            # by adding the header ID
            self.xref_id = self.xref_id + self.get_header().get_xref_id ()
        return self.xref_id

class ClassOrStruct (ScopingDecl):
    def __init__ (self, type):
        ScopingDecl.__init__(self, type)
        self.name = None # 'Foo'
        self.decl = None # 'template <class T> class Foo : public Bar'
        self.super_names = []
        self.subs = []
        self.supers = []
        self.unlinkable_super_names = []
        self.linkable_super_names = []
        self.access = None
        self.is_type = 1        
        self.is_abstract = 0

    def set_access (self, access):
        self.access = access

    def set_name (self, name):
        self.name = strip(name)

    def set_decl (self, decl):
        self.decl = decl

    def add_parent_name (self, parent):
        self.super_names.append (parent)

    def get_xref_id (self):
        if not self.xref_id:
            id = 'class-'
            scopes = self.get_scopes ()
            id = id + join (scopes, '-')
            id = id +  '-' + self.name
            self.xref_id = fix_for_sgml (id)
        return self.xref_id

    def get_documented_supers (self):
        doc_parents = []
        for name in self.super_names:
            if find (name, 'Private') < 0:
                doc_parents.append (name)
            
        return doc_parents
        

class Class (ClassOrStruct):
    def __init__ (self):
        ClassOrStruct.__init__(self, 'class')

    def spew (self, indent):
        print indent + "class " + self.name + '\n'
        Declaration.spew (self, indent)

    def set_doc (self, doc):
        Declaration.set_doc (self, doc)
        if doc.type != 'class':
            message.die ("tried to document a class with wrong doc type: " + doc.type)

class Struct (ClassOrStruct):
    def __init__ (self):
        ClassOrStruct.__init__(self, 'struct')

    def spew (self, indent):
        print indent + "struct " + self.name + '\n'
        Declaration.spew (self, indent)

    def set_doc (self, doc):
        Declaration.set_doc (self, doc)
        if doc.type != 'class':
            message.die ("tried to document a struct with wrong doc type: " + doc.type)

class MethodOrFunction (Declaration):
    def __init__ (self, type):
        Declaration.__init__(self, type)
        self.decl = None
        self.access = None # 'protected', 'private', 'public'
        self.children = None # can't have child blocks
        self.args = None # should be a list of (type,name) tuples
        self.name = None
        self.returns = None
        self.template_goo = ''
        self.long_name = None

    def set_decl (self, decl):
        self.decl = decl

    def set_access (self, access):
        self.access = access

    def set_args (self, args):
        self.args = args

    def set_name (self, name):
        self.name = name

    def set_returns (self, type):
        self.returns = type

    def get_xref_id (self):
        if not self.xref_id:
            id = 'method-'
            scopes = self.get_scopes ()
            id = id + join (scopes, '-')
            id = id + '-' + self.name                
            self.xref_id = fix_for_sgml (id)
        return self.xref_id

class Method (MethodOrFunction):
    def __init__ (self):
        MethodOrFunction.__init__(self, 'method')
        self.is_constructor = 0
        self.is_destructor = 0
        self.is_virtual = 0
        self.is_const = 0
        self.is_pure_virtual = 0

    def spew (self, indent):
        self.spew_docs (indent)
        print indent + self.access + " method: " + self.decl + '\n'

    def set_doc (self, doc):
        Declaration.set_doc (self, doc)
        if doc.type != 'method':
            message.die ("tried to document a method with wrong doc type: " + doc.type)

    def set_is_constructor (self, setting):
        self.is_constructor = setting

    def set_is_destructor (self, setting):
        self.is_destructor = setting

    def set_is_virtual (self, setting):
        self.is_virtual = setting

class Function (MethodOrFunction):
    def __init__ (self):
        MethodOrFunction.__init__(self, 'function')

    def spew (self, indent):
        self.spew_docs (indent)
        print indent + self.access + " function: " + self.decl + '\n'

    def set_doc (self, doc):
        Declaration.set_doc (self, doc)
        if doc.type != 'method':
            message.die ("tried to document a function with wrong doc type: " + doc.type)

class Variable (Declaration):
    def __init__ (self):
        Declaration.__init__(self, 'variable')
        self.decl = None
        self.access = None # 'protected', 'private', 'public'
        self.children = None # can't have child blocks
        self.is_static = 0

    def set_decl (self, decl):
        self.decl = decl

    def set_access (self, access):
        self.access = access

    def spew (self, indent):
        print indent + self.access + " variable: " + self.decl + '\n'
        self.spew_docs (indent)

    def set_doc (self, doc):
        Declaration.set_doc (self, doc)
        if doc.type != 'variable':
            message.die ("tried to document a variable with wrong doc type: " + doc.type)

    def get_xref_id (self):
        if not self.xref_id:
            id = 'variable-'
            scopes = self.get_scopes ()
            id = id + join (scopes, '-')
            id = id +  '-' + self.name
            self.xref_id = fix_for_sgml (id)
        return self.xref_id

class TypedefOrEnum (Declaration):
    def __init__ (self, type):
        Declaration.__init__(self, type)
        self.decl = None
        self.access = None # 'protected', 'public'
        self.children = None # can't have child blocks
        self.name = None
        self.is_type = 1
        self.long_name = None
    
    def get_xref_id (self):
        if not self.name:
            self.name = 'FIXME'
        if not self.xref_id:
            id = 'type-'
            scopes = self.get_scopes ()
            id = id + join (scopes, '-')
            id = id +  '-' + self.name
            self.xref_id = fix_for_sgml (id)
        return self.xref_id

    def set_name (self, name):
        self.name = strip(name)

    def set_decl (self, decl):
        self.decl = decl

    def set_access (self, access):
        self.access = access


class Typedef (TypedefOrEnum):
    def __init__ (self):
        TypedefOrEnum.__init__(self, 'typedef')

    def spew (self, indent):
        print indent + self.access + " typedef: " + self.decl + '\n'
        self.spew_docs (indent)

    def set_doc (self, doc):
        Declaration.set_doc (self, doc)
        if doc.type != 'typedef':
            message.die ("tried to document a typedef with wrong doc type: " + doc.type)

class Enum (TypedefOrEnum):
    def __init__ (self):
        TypedefOrEnum.__init__(self, 'enum')
        self.name = None
        self.vals = None # list of strings

    def set_vals (self, vals):
        self.vals = vals

    def spew (self, indent):
        if self.name:
            print indent + self.access + " enum " + self.name + '\n'
        else:
            print indent + self.access + " enum (name never found!!!) \n"
        self.spew_docs (indent)

    def set_doc (self, doc):
        Declaration.set_doc (self, doc)
        if doc.type != 'enum':
            message.die ("tried to document an enum with wrong doc type: " + doc.type)

class NameStack:
    def __init__ (self):
        self.stack = []
        self.qualifier_ = ""

    def push (self, name):
        self.stack.append (name)
        self.qualifier_ = join (self.stack, '::') + '::'

    def pop (self):
        self.stack.pop ()
        self.qualifier_ = join (self.stack, '::') + '::'

    def qualifier (self):
        return self.qualifier_

def postprocess_children (decl, namestack):
    for c in decl.children:
        recursive_postprocess (c, namestack)

public_parent = re.compile ('public ([A-Za-z0-9:<> ]+)')
inheritance_keyword = re.compile ('public') # we don't document private/protected

def postprocess_class_or_struct (decl, namestack):
    #      if not decl.doc and not decl.in_private_namespace ():
    #          message.warn ("In scope " + namestack.qualifier ())
    #          message.warn ("No docs for class or struct: " + decl.decl)

    match = inheritance_keyword.search (decl.decl)
    if match:
        inheritance_decls = decl.decl[match.end():]
        supers = inheritance_keyword.split (inheritance_decls)

        for s in supers:
            s = strip (s)
            if s[-1] == ',':
                s = s[:-1]
            decl.add_parent_name (s)

    postprocess_children (decl, namestack)

enum_val = re.compile ('([A-Za-z0-9_]+)( *=[^,]*)?,?')

def postprocess_enum (decl, namestack):
    #     if not decl.doc and not decl.in_private_namespace ():
    #         message.warn ("In scope " + namestack.qualifier ())
    #         message.warn ("No docs for enum: " + decl.name)

    vals_and_assignments = enum_val.findall (decl.decl)

    vals = []
    for (v, blah) in vals_and_assignments:
        vals.append (v)

    decl.set_vals (vals)

default_arg_val = re.compile (' *= *([^ ]*)')

def postprocess_method_or_function (decl, namestack):
    ##     if not decl.doc and not decl.in_private_namespace ():
    ##         message.warn ("In scope " + namestack.qualifier ())
    ##         message.warn ("No docs for method or function: " + decl.decl)

##    if decl.doc and decl.doc.important:
##        decl.is_important = 1

    ## Parse the declaration
    right = rfind (decl.decl, ')')
    if right < 0:
        message.die ("function declaration has no right paren: " + decl.decl)

    if find (decl.decl[right:], 'const') >= 0:
        decl.is_const = 1

    if find (decl.decl[right:], '= 0') >= 0 or \
       find (decl.decl[right:], '=0') >= 0:
        decl.is_pure_virtual = 1
        decl.parent.is_abstract = 1
    
    nest_depth = 1
    i = right - 1
    while i >= 0:
        if decl.decl[i] == ')':
            nest_depth = nest_depth + 1
        elif decl.decl[i] == '(':
            nest_depth = nest_depth - 1
            if nest_depth == 0:
                break

        i = i - 1;

    if nest_depth > 0:
        message.die ("didn't find parens in: " + decl.decl)

    # the name is the contiguous word before the arglist.

    name_end = i

    i = i + 1
    arglist = decl.decl[i:right]

    arglist = strip (arglist)

    last_comma = -1
    template_nest = 0
    i = 0
    args = []
    while i < len (arglist):
        if arglist[i] == ',':
            if template_nest == 0:
                last_comma = last_comma + 1
                a = strip (arglist[last_comma:i])
                args.append (a)
                last_comma = i
        elif arglist[i] == '<':
            template_nest = template_nest + 1
        elif arglist[i] == '>':
            template_nest = template_nest - 1

        i = i + 1

    last_comma = last_comma + 1
    if last_comma < i:
        a = strip (arglist[last_comma:i])
        args.append (a)

    i = name_end - 1
    # skip whitespace just before parens
    while i >= 0:
        if not decl.decl[i] in whitespace:
            break
        i = i - 1
    name_end = i + 1
    # go until we see whitespace or * or & before function name
    end_chars = whitespace + '*&'
    if decl.decl[i] == '*': # hack for operator*
        i = i - 1
    while i >= 0:
        if decl.decl[i] in end_chars:
            break
        i = i - 1

    i = i + 1
    name_start = i
    funcname = strip (decl.decl[name_start:name_end])

    if funcname == '':
        message.die ("no function name parsed for " + decl.decl)
    elif funcname[0] == '*':
        message.die ("internal screwup, put a * in the function name")

    # before the function name, we have the return type,
    # and potentially also some template crap

    prelude = strip (decl.decl[:name_start])
    return_type = prelude
    if prelude[:8] == 'template':
        i = find (prelude, '<')
        i = i + 1
        angle_nest = 1
        while angle_nest > 0:
            if prelude[i] == '<':
                angle_nest = angle_nest + 1
            elif prelude[i] == '>':
                angle_nest = angle_nest - 1

            i = i + 1
        return_type = prelude[i:]
        decl.template_goo = strip (prelude[:i])

    if find (return_type, 'virtual') >= 0:
        decl.set_is_virtual (1)

    return_type = replace (return_type, 'static', '')
    return_type = replace (return_type, 'inline', '')
    return_type = replace (return_type, 'explicit', '')
    return_type = replace (return_type, 'virtual', '')
    return_type = strip (return_type)

    ## split args into type/name
    
    real_args = []
    for a in args:
        if find (a, '...') >= 0:
            real_args.append (('', '...', None))
        else:
            default = None
            match = default_arg_val.search (a)
            if match:
                default = match.group (1)
            a = default_arg_val.sub ('', a)

            i = len (a) - 1
            while i >= 0:
                if a[i] in '&* \t<>':
                    break
                i = i - 1
            i = i + 1
            argname = strip (a[i:])
            argtype = strip (a[:i])

            real_args.append ((argtype, argname, default))

    decl.set_args (real_args)
    decl.set_name (funcname)
    decl.set_returns (return_type)

    if decl.returns == '':
        if decl.name[0] == '~':
            decl.set_is_destructor (1)
        else:
            decl.set_is_constructor (1)

    long_name = decl.template_goo + ' '    
    if decl.type == 'method' and decl.is_virtual:
        long_name = long_name = 'virtual '
    long_name = long_name + decl.returns + ' '
    long_name = long_name + decl.name
    long_name = long_name + ' ('
    for (type, name, default) in decl.args:
        if long_name[-1] != '(':
            long_name = long_name + ', '
        long_name = long_name + type + ' ' + name
        if default:
            long_name = long_name + ' = ' + default
    long_name = long_name + ')'
    if decl.type == 'method' and decl.is_const:
        long_name = long_name + ' const'
    if decl.type == 'method' and decl.is_pure_virtual:
        long_name = long_name + ' = 0'
    decl.long_name = strip (long_name)

    #    print decl.long_name

function_var = re.compile ('\(.*\*(.*?)\) *\(.*?\);', re.DOTALL)
normal_var = re.compile ('.* +.*?([A-Za-z0-9_]+);', re.DOTALL)

def set_typedef_or_variable_name (decl):
    decl.decl = strip (decl.decl)
    str = decl.decl
    if str[-1] != ';':
        message.die ("typedef or variable decl supposed to end with semicolon")

    decl.long_name = str[:-1]

    if str[:len('static')] == 'static':
        decl.is_static = 1

    match = function_var.search (str)
    if match:
        decl.name = match.group (1)
        #        print 'typedef/variable function: ' + decl.name
        return

    match = normal_var.search (str)
    if match:
        decl.name = match.group (1)
        #        print 'typedef/variable name: ' + decl.name
        return

    message.die ("Failed to parse %s decl: %s" % (decl.type, decl.decl))
    

def postprocess_typedef (decl, namestack):
    set_typedef_or_variable_name (decl)

def postprocess_variable (decl, namestack):
    set_typedef_or_variable_name (decl)

xref_id_dict = {}

def recursive_postprocess (decl, namestack):
    named = 0
    if decl.type == 'namespace' or \
       decl.type == 'class' or \
       decl.type == 'struct':
        named = 1
        namestack.push (decl.name)

    if decl.type == 'method' or \
       decl.type == 'function':
        postprocess_method_or_function (decl, namestack)
    elif decl.type == 'class' or \
         decl.type == 'struct':
        postprocess_class_or_struct (decl, namestack)
    elif decl.type == 'enum':
        postprocess_enum (decl, namestack)
    elif decl.type == 'typedef':
        postprocess_typedef (decl, namestack)
    elif decl.type == 'variable':
        postprocess_variable (decl, namestack)
    elif decl.type == 'namespace' or \
         decl.type == 'header':
        postprocess_children (decl, namestack)
    else:
        message.die ("didn't handle decl type " + decl.type)

    if hasattr (decl, 'access') and decl.access == 'private':
        decl.is_private = 1

    # Assign unique xref id. To work, depends on the fact that the
    # children of an item are in an ordered list based on order of
    # appearance in the .h file
    if decl.name:
        decl.qualified_name = decl.qualifier () + decl.name
        
        xref = decl.get_xref_id ()
        if xref:
            while xref_id_dict.has_key (xref):
                xref = xref + '-dup'
            decl.xref_id = xref
            xref_id_dict[xref] = decl

    if named:
        namestack.pop ()
        
def postprocess_decls (decl):
    stack = NameStack ()
    recursive_postprocess (decl, stack)
    for v in xref_id_dict.values ():
        # has to be done after the other postprocessing
        if v.in_private_namespace ():
            v.is_private = 1

def dump_decls (filename):
    print filename
    p = pickle.Pickler (open (filename, 'w'), 1)
    p.dump (xref_id_dict)

def strip_template (klassname):
    i = find (klassname, '<')
    if i >= 0:
        return klassname[:i]
    else:
        return klassname

decl_typesplitter = re.compile ('([A-Za-z0-9_]+)(.*)')

def split_type (type):
    type = strip (type)
    match = decl_typesplitter.match (type)
    if match:
        name = match.group (1)
        other = match.group (2)
        return (name, other)
    else:
        return (type, '')

def full_postprocess ():
    global full_process_phase

    full_process_phase = 1
    
    # populate our symbol tables
    for decl in xref_id_dict.values ():
        if decl.type == 'namespace':
            syms = decl.get_symbol_table ()
            if syms != namespace_symbol_tables[decl.qualified_name]:
                message.die ("Oops, bug")
                
    for decl in xref_id_dict.values ():
        if decl.type == 'class' or decl.type == 'struct':
            decl.linkable_super_names = []
            # add ourselves as a subclass of our parent classes,
            # and add parent class to our list
            for sname in decl.super_names:
                # lookup in the parent scope (remember
                # that's where superclasses are declared)
                typepair = split_type (sname)
                super = decl.parent.lookup (typepair[0], ['class', 'struct'])
                
                if super:
                    super.subs.append (decl)
                    decl.supers.append (super)
                    decl.linkable_super_names.append (sname)
                else:
                    message.warn ("Failed to locate parent class %s of %s" % \
                                  (sname, decl.qualified_name))
                    decl.unlinkable_super_names.append (sname)
                    
                
def load_decls (filename):
    p = pickle.Unpickler (open (filename))
    new_dict = p.load ()

    # merge into xref_id_dict

    for k in new_dict.keys ():
        if xref_id_dict.has_key (k):
            message.die ("xref ID collision, %s" % k)
        else:
            xref_id_dict[k] = new_dict[k]

                    
