#!/usr/bin/env python
# -*- Mode: Python; py-indent-offset: 4 -*-

try:
    from cString import *
except:
    from string import *

import sys
import os
import re
import exceptions
import errno
import getopt

import parser
import docoutput
from docobjects import *
from linebuf import *
import message
from output import *
import codegen
import codefile

#####################################################################
### Subroutines

def write_header_copyright(filehandle):
  filehandle.write("""
/*
  Inti is a C++ development platform based on glib, GTK+, and the
  Standard Template Library. This header file is machine-generated
  by a script; do not edit it directly.

  This file is distributed under the GNU Library General Public License.
*/
""")

def write_implementation_copyright (filehandle):

  filehandle.write("""
/*
  Inti is a C++ development platform based on glib, GTK+, and the
  Standard Template Library. This source file is machine-generated
  by a script; do not edit it directly.

  This file is distributed under the GNU Library General Public License.
*/
""")
    

class Done (exceptions.Exception):
    def __init__(self):
        pass

class NamespaceStack:
    def __init__ (self):
        self.stack = []
        self.name_stack = []
        self.is_class_stack = []
        self.declaration_stack = [] # list of declaration strings

    def push (self, name, is_class):
        self.stack.append (name)
        self.name_stack.append (name)
        self.is_class_stack.append (is_class)
        if is_class:
            self.declaration_stack.append ("")

    def push_declaration (self, decl):
        i = 0
        while i < len (self.declaration_stack):
            self.declaration_stack[i] = self.declaration_stack[i] + decl
            i = i + 1

    def add_brace (self, brace):
        if brace == '{':
            self.stack.append (brace)
        elif brace == '}':
            top = self.stack[-1]
            if top != '{':
                all_lines.die ("Close brace but didn't see the open brace")
            self.stack.pop ()
            try:
                top = self.stack[-1]
            except IndexError:
                return None# not inside any classes

            if top != '{':
                # top is a class name, we just closed the class
                self.stack.pop ()
                self.name_stack.pop ()
                # if it's a class, pop its declaration
                decl = None
                if self.is_class_stack[-1]:
                    decl = self.declaration_stack[-1]
                    self.declaration_stack.pop ()
                self.is_class_stack.pop ()

                return decl
            

    def top (self):
        return self.name_stack[-1]

    def full (self):
        # return a copy of the stack
        return self.name_stack[:]

    def qualifier (self):
        return join (self.name_stack, '::')

    def namespaces (self):
        retval = []
        i = 0
        while i < len (self.name_stack):
            if self.is_class_stack[i]:
                break # assume no namespaces inside classes
            else:
                retval.append(self.name_stack[i])

            i = i + 1

        return retval

    def classes (self):
        retval = []
        i = 0
        while i < len (self.name_stack):
            if self.is_class_stack[i]:
                retval.append(self.name_stack[i])

            i = i + 1

        return retval

######################################################################
### Script

## Compile our regular expressions

# entered a namespace or class
namespace_forward_decl = re.compile (" *namespace +([a-zA-Z_]+) *;")
class_forward_decl = re.compile (" *class +([a-zA-Z_0-9<>,: ]+) *;")

namespace_decl = re.compile (" *namespace +([a-zA-Z_]+)")
class_decl = re.compile (" *class +([a-zA-Z_0-9<>, ]+)")

# /*$ doc whatever params
doc_start = re.compile (" */\*\$ doc +([a-z]+) *(.*)$")

# start and end of magic comment /*$ $*/
magic_start = re.compile ("/\*\$");
magic_end = re.compile ("\$\*/");

# keywords in the docs
doc_keyword = re.compile (" *\$([a-z]+)")
doc_param = re.compile (" *@([a-z_A-Z0-9]+)")

# regular comments
cpp_comment = re.compile (".*(//.*)$")
#oneline_comment = re.compile ("(.*)(/\*.*\*/)")
#comment_start = re.compile ("(.*)(/\*.*)")
#comment_end = re.compile ("(.*\*/)(.*)")

# "directives"
impl_directive = re.compile ("//#implementation")
code_directive = re.compile ("( +)/\*! +([a-z]+) *([^!]*)!\*/")

## Other globals

total_lines = None
all_lines = None

namespace_stack = NamespaceStack()
header_doc = HeaderDoc()

interface_target = None ## '/cvs/mycvs/inti/src/inti/main.h'
impl_target = None ## '/cvs/mycvs/inti/src/inti/main.cc'
template = None ## '/cvs/mycvs/inti/src/template/main.gen'
docs_dir = None ## '/cvs/mycvs/inti/src/doc/extracted'
script_dir = None ## '/cvs/mycvs/inti/src/scripts'
file_basename = None ## 'main'
submodule = None ## "base" or "gtk" or "gdk" etc.
code_dir = None
code_files = []
code_repository = None

opts, args = getopt.getopt(sys.argv[1:], '', ['target=', 'template=', 'submodule=', 'script-dir=', 'code-dir=', 'code-file='])

for (arg, val) in opts:
    if arg == '--template':
        template = val
        if template[-4:] != '.gen':
            die ("template must end in .gen")
        file_basename = os.path.basename (template)[:-4]
    elif arg == '--target':
        if file_basename == None:
            die ("specify --template before --target")
        interface_target = val + '/' + file_basename + '.h'
        impl_target = val + '/' + file_basename + '.cc'
    elif arg == '--script-dir':
        script_dir = val
    elif arg == '--submodule':
        submodule = val
    elif arg == '--code-dir':
        code_dir = val
    elif arg == '--code-file':
        code_files.append (val)

if template == None:
    message.die ("must specify .gen file with --template")

if interface_target == None:
    message.die ("must specify target directory with --target")

if submodule == None:
    message.die ("must specify sub-module")

if script_dir == None:
    message.die ("must specify scripts dir")

if code_dir:
    code_files.append (os.path.join (code_dir, file_basename + '.code'))
    code_files.append (os.path.join (code_dir, 'enums.code'))

    code_repository = codefile.CodeRepository (code_files)

outfile = OutputTarget ()
outfile.set ( open (interface_target, 'w'),
              open (impl_target, 'w') )
infile = open (template, 'r')


# Separate header file from .cc file
rawlines = infile.readlines ()
header_lines = []
impl_lines = []
in_implementation = 0
for line in rawlines:
    if in_implementation:
        impl_lines.append (line)
    else:
        match = impl_directive.match (line)
        if match:
            in_implementation = 1
        else:
            header_lines.append (line)
        
all_lines = LineBuf()
all_lines.set_lines (template, header_lines)
        
#### Main code

## Check the boilerplate at the top, and copy it to the implementation file
line = all_lines.pop()
if find (line, "-*- C++ -*-") < 0:
    all_lines.die ("first line must contain Emacs magic")
emacs_magic = line
outfile.write_to_header(emacs_magic);

line = all_lines.pop()
if find (line, "Copyright") < 0:
    all_lines.die ("second line must have copyright notice")
copyright = line
outfile.write_to_header(copyright);

## write include guards
include_guard = "_INTI_" + upper (submodule) + "_" +  upper(file_basename) + "_H_"
new_include_guard = ""
for c in include_guard:
    if c in letters:
        new_include_guard = new_include_guard + c
    else:
        new_include_guard = new_include_guard + '_'
include_guard = new_include_guard

outfile.write_to_header ("#ifndef " + include_guard + "\n")
outfile.write_to_header ("#define " + include_guard + "\n")

## write the boilerplate
write_header_copyright (outfile.header_handle())
outfile.write_to_impl (emacs_magic)
outfile.write_to_impl (copyright)
write_implementation_copyright (outfile.impl_handle())

# write the .cc file
for line in impl_lines:
    outfile.write_to_impl (line)
    impl_lines = None # GC the impl_lines

outfile.write_to_impl ("\n/////////// Everything below here is machine-generated\n")
    
## Now process the header lines
in_comment = 0
while all_lines:
    line = all_lines.pop ()

    # Look for code to generate
    match = code_directive.match (line)
    if match:
        codegen.process_code_directive (outfile,
                                        code_repository,
                                        match.group (1),
                                        match.group (2),
                                        match.group (3),
                                        all_lines,
                                        namespace_stack)
        continue


    #### All this mess is just to keep track of the current
    #### namespace

    # find the text on this line that's not commented out;
    # or if none, go forward until we find some.

    # first if we're still in a multiline comment, skim past it.
    if in_comment:
        comment_end = find (line, "*/")
        if comment_end >= 0:
            code = line[comment_end+2:]
            in_comment = 0
        else:
            outfile.write (line)
            continue
    else:
        code = line

    # chop off trailing c++-style comment
    match = cpp_comment.match (code)
    if match:
        code = code[:match.start (1)]

    # look at C-style comments
    while 1:
        comment_start = find (code, "/*")
        if comment_start >= 0:
            comment_end = find (code, "*/")
            if comment_end >= 0:
                code = code[:comment_start] + code[comment_end+2:]
            else:
                code = code[:comment_start]
                in_comment = 1
                break
        else:
            break

    ## Now we have the real code (no comments), and need to keep track
    ## of classes/namespaces
            
    # look for braces in here
    braces = code

    match = namespace_decl.match (code)
    if match:
        badmatch = namespace_forward_decl.match (code)
        if not badmatch:
            braces = code[match.end (1):]
            namespace_stack.push (match.group (1), 0)
            if find('{', braces) >= 0:
                all_lines.warn ("brace on same line as namespace declaration")

    match = class_decl.match (code)
    if match:
        badmatch = class_forward_decl.match (code)
        if not badmatch:
            braces = code[match.end (1):]
            namespace_stack.push (strip(match.group (1)), 1)
            if find('{', braces) >= 0:
                all_lines.warn ("brace on same line as class declaration")

    namespace_stack.push_declaration (code)

    for char in braces:
        if char == '{' or char == '}':
            namespace_stack.add_brace (char)
                
    # write line to output file. 
    outfile.write (line)

## close the include guard
outfile.write_to_header ("#endif // " + include_guard + "\n")

## Make sure we got a header file
#  if header_doc == None or header_doc.get_name() == None:
#      message.die ("didn't find header file documentation")

#  ## Write out docs based on the processed file.
#  try:
#      os.mkdir (docs_dir)
#  except OSError, e:
#      if e.errno != errno.EEXIST:
#          message.die (e)

#  docoutput.output_header_docs(header_doc, docs_dir + '/' + file_basename)







