// -*- C++ -*-
// Copyright (C) 2000 Red Hat, Inc.
// Typing game example
/*
  The following license applies to this example code, but not to
  the Inti library itself:
  
    Permission is hereby granted, free of charge, to any person obtaining
    a copy of this software and associated documentation files (the
    "Software"), to deal in the Software without restriction, including
    without limitation the rights to use, copy, modify, merge, publish,
    distribute, sublicense, and/or sell copies of the Software, and to
    permit persons to whom the Software is furnished to do so, subject to
    the following conditions:

    The above copyright notice and this permission notice shall be
    included in all copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
    EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
    MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
    NONINFRINGEMENT.  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
    DISTRIBUTOR OF THE SOFTWARE BE LIABLE FOR ANY CLAIM, DAMAGES OR
    OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
    OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
    OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#include "badguy.h"
#include "game.h"
#include <inti/gdk/drawable.h>
#include <inti/gdk/colormap.h>
#include <inti/gdk/color.h>
#include <inti/gtk/style.h>
#include <iostream>

Badguy::~Badguy ()
{
  

}
  
Badguy::Badguy (Game * game)
  : game_ (game)
{


}

void
Badguy::set_extents (const Gdk::Rectangle & rect)
{
  extents_ = rect;
}

class MovingBadguy : public Badguy
{
public:
  virtual void update (long msecs_elapsed);

protected:
  MovingBadguy (Game * game, double xvel, double yvel);
  void set_x_velocity (double velocity);
  void set_y_velocity (double velocity);
  // pixels per second per second
  void set_acceleration (double accel);
  
private:
  double unused_x_distance_;
  double unused_y_distance_;
  // pixels per second
  double x_velocity_;
  double y_velocity_;

  double acceleration_;
};

void
MovingBadguy::update (long msecs_elapsed)
{
  if (msecs_elapsed == 0)
    return;

  // Reverse velocity if we're off-screen.
  
  if (extents ().x () < 0 &&
      x_velocity_ < 0)
    x_velocity_ = - x_velocity_;
  else if ((extents ().x () + extents ().width ()) > game ()->width () &&
           x_velocity_ > 0)
    x_velocity_ = - x_velocity_;

  if (extents ().y () < 0 &&
      y_velocity_ < 0)
    y_velocity_ = - y_velocity_;
  else if ((extents ().y () + extents ().height ()) > game ()->height () &&
           y_velocity_ > 0)
    y_velocity_ = - y_velocity_; 

  //  cout << "x vel: " << x_velocity_ << " y vel: " << y_velocity_ << endl;
  
  double seconds = double (msecs_elapsed) / 1000000.0;
  double x_distance = x_velocity_ * seconds;
  double y_distance = y_velocity_ * seconds;
  int x_pixels = int (x_distance + unused_x_distance_);
  int y_pixels = int (y_distance + unused_y_distance_);
  
  Gdk::Rectangle translated = extents ();
  translated.set_position (translated.x () + x_pixels,
                           translated.y () + y_pixels);
  
  set_extents (translated);

  if (x_pixels != 0)
    unused_x_distance_ = 0.0;
  else
    unused_x_distance_ += x_distance;
  
  if (y_pixels != 0)
    unused_y_distance_ = 0.0;
  else
    unused_y_distance_ += y_distance;

  if (x_velocity_ > 0)
    x_velocity_ += acceleration_ * seconds;
  else
    x_velocity_ -= acceleration_ * seconds;

  if (y_velocity_ > 0)
    y_velocity_ += acceleration_ * seconds;
  else
    y_velocity_ -= acceleration_ * seconds;
}

MovingBadguy::MovingBadguy (Game * game, double xvel, double yvel)
  : Badguy (game), unused_x_distance_ (0.0), unused_y_distance_ (0.0),
    x_velocity_ (xvel), y_velocity_ (yvel), acceleration_ (0.0)
{

}

void
MovingBadguy::set_x_velocity (double velocity)
{
  x_velocity_ = velocity;
}

void
MovingBadguy::set_y_velocity (double velocity)
{
  y_velocity_ = velocity;
}

void
MovingBadguy::set_acceleration (double acceleration)
{
  acceleration_ = acceleration;
}

//////////

class EasyBadguy : public MovingBadguy
{
public:
  EasyBadguy (Game * game);

  virtual void damage (const string & str);
  virtual void render (Gtk::Widget * widget, Gdk::Drawable * drawable,
                       const Gdk::Rectangle & clip);

private:
  string word_;
  unsigned int damaged_;

};

EasyBadguy::EasyBadguy (Game * game)
  : MovingBadguy (game, 90.0, 60.0), word_ ("testword"), damaged_ (0)
{
  set_extents (Gdk::Rectangle (0, 0, 100, 100));
  set_acceleration (3.0);
}

void
EasyBadguy::damage (const string & str)
{
  if (damaged_ == word_.length ())
    return;
  // FIXME broken for UTF8
  else if (word_[damaged_] == str[0])
    ++damaged_;
}

void
EasyBadguy::render (Gtk::Widget * widget,
                    Gdk::Drawable * drawable,
                    const Gdk::Rectangle & clip)
{
  // this is super-slow.
  
  ptr<Gdk::GC> gc (drawable->create_gc ());

  Gdk::Color color ("green");
  drawable->colormap ()->alloc_color (&color);
  gc->set_foreground (color);

  gc->set_clip (clip);
  
  drawable->draw_rectangle (gc, true,
                            extents ());
  
  drawable->colormap ()->free_color (color);
}


//////////

class ScaryBadguy : public MovingBadguy
{
public:
  ScaryBadguy (Game * game);

  virtual void damage (const string & str);
  virtual void render (Gtk::Widget * widget, Gdk::Drawable * drawable,
                       const Gdk::Rectangle & clip);

private:
  string word_;
  unsigned int damaged_;

};

ScaryBadguy::ScaryBadguy (Game * game)
  : MovingBadguy (game, 170.0, 200.0), word_ ("testword"), damaged_ (0)
{
  set_extents (Gdk::Rectangle (20, 57, 50, 90));
  set_acceleration (4.0);
}

void
ScaryBadguy::damage (const string & str)
{
  if (damaged_ == word_.length ())
    return;
  // FIXME broken for UTF8
  else if (word_[damaged_] == str[0])
    ++damaged_;
}

void
ScaryBadguy::render (Gtk::Widget * widget, Gdk::Drawable * drawable,
                    const Gdk::Rectangle & clip)
{
  // this is super-slow.
  
  ptr<Gdk::GC> gc (drawable->create_gc ());

  Gdk::Color color ("red");
  drawable->colormap ()->alloc_color (&color);
  gc->set_foreground (color);

  gc->set_clip (clip);
  
  drawable->draw_rectangle (gc, true,
                            extents ());
  
  drawable->colormap ()->free_color (color);
}

/////////

Badguy*
create_badguy (Game * game, BadguyType type)
{
  switch (type)
    {
    case BADGUY_EASY:
      return new EasyBadguy (game);
      break;

    case BADGUY_SCARY:
      return new ScaryBadguy (game);
      break;
    }

  return 0;
}
