
// Copyright (C) 1997-1999 Cygnus Solutions
//
// This file is part of the GNU ISO C++ Library.  This library is free
// software; you can redistribute it and/or modify it under the
// terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option)
// any later version.

// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License along
// with this library; see the file COPYING.  If not, write to the Free
// Software Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307,
// USA.

// As a special exception, you may use this file as part of a free software
// library without restriction.  Specifically, if other files instantiate
// templates or use macros or inline functions from this file, or you compile
// this file and link it with other files to produce an executable, this
// file does not by itself cause the resulting executable to be covered by
// the GNU General Public License.  This exception does not however
// invalidate any other reasons why the executable file might be covered by
// the GNU General Public License.

#include <bits/std_clocale.h>
#include <bits/std_cstring.h>
#include <bits/std_cassert.h>
#include <bits/std_limits.h>
#include <bits/std_exception.h>
#include <bits/std_stdexcept.h>
#include <bits/std_locale.h>

#if _G_USE_NAMESPACE
namespace std {
#endif

// Implementation of locale::_Impl members
//////////////////////////////////////////


  // destructor
  locale::_Impl::~_Impl () throw ()
  {
    std::vector<facet*>::iterator it = _M_facets.begin ();
    for (; it != _M_facets.end (); ++it)
      (*it)->_M_remove_reference ();
  }

  // copy constructors
  locale::_Impl::_Impl (size_t numfacets, size_t refs)
  : _M_num_references (refs-1)
  , _M_facets (numfacets, (facet*)0)
  , _M_category_names (_S_num_categories, string ("*"))
  , _M_has_name (false)
  , _M_cached_name_ok (false)
  , _M_cached_name (string ("*"))
  { }
  
  locale::_Impl::_Impl (const _Impl& other,
			size_t refs)
  : _M_num_references (refs)
  , _M_facets (other._M_facets)
  , _M_category_names (other._M_category_names)
  , _M_has_name (other._M_has_name)
  , _M_cached_name_ok (other._M_cached_name_ok)
  , _M_cached_name (other._M_cached_name)
  {
    std::vector<facet*>::iterator it = _M_facets.begin ();
    for (; it != _M_facets.end (); ++it)
      (*it)->_M_add_reference ();
  }
  
  //////////
  void
  locale::_Impl::_M_replace_categories (const _Impl* other,
					category cats)
  {
    assert((cats & locale::all) && !(cats && ~locale::all));
    
    unsigned mask = (locale::all & -(unsigned)locale::all);
    for (unsigned ix = 0; (-mask & cats) != 0; ++ix, (mask <<= 1))
      {
	if (mask & cats)
	  {
	    _M_replace_category (other, _S_facet_categories[ix]);
	    _M_category_names[ix] = other->_M_category_names[ix];
	  }
      }
  }

  //////////
  void
  locale::_Impl::_M_replace_category (const _Impl* other,
				      const locale::id* const* idpp)
  {
    for (; *idpp; ++idpp)
      _M_replace_facet (other, *idpp);
  }
  
  //////////
  void
  locale::_Impl::_M_replace_facet (const _Impl* other,
				   const locale::id* idp)
  {
    size_t index = idp->_M_index;
    if (index == 0 ||
	other->_M_facets.size () <= index ||
	other->_M_facets[index] == 0)
      throw runtime_error ("no locale facet");
	
    _M_install_facet (idp, other->_M_facets[index]); 
  }

  //////////
  void
  locale::_Impl::_M_install_facet (const locale::id* idp,
				   facet* fp)
    {
    if (fp == 0)
      return;

    size_t& index = idp->_M_index;
    if (!index)
      index = ++locale::id::_S_highwater;  // XXX MT

    if (index >= _M_facets.size ())
      _M_facets.resize (index+1, 0);  // might throw
    facet*& fpr = _M_facets[index];
    // order matters, here:
    fp->_M_add_reference ();
    if (fpr) fpr->_M_remove_reference ();
    fpr = fp;
  }

  // locale facet category descriptions
  //////////
  const locale::id* const
  locale::_Impl::_S_id_collate[] =
    {
      &std::collate<char>::id,
      &std::collate<wchar_t>::id,
      0
    };
  
  const locale::id* const
  locale::_Impl::_S_id_ctype[] =
    {
      &std::ctype<char>::id, &std::ctype<wchar_t>::id,
      &std::codecvt<char,char,mbstate_t>::id,
      &std::codecvt<wchar_t,char,mbstate_t>::id,
      0
    };

  const locale::id* const
  locale::_Impl::_S_id_monetary[] =
    {
      &std::moneypunct<char,false>::id, &std::moneypunct<wchar_t,false>::id,
      &std::moneypunct<char,true >::id, &std::moneypunct<wchar_t,true >::id,
      &std::money_get<char>::id,        &std::money_get<wchar_t>::id,
      &std::money_put<char>::id,        &std::money_put<wchar_t>::id,
      0
    };

  const locale::id* const
  locale::_Impl::_S_id_numeric[] =
    {
      &std::numpunct<char>::id, &std::numpunct<wchar_t>::id,
      &std::num_get<char>::id,  &std::num_get<wchar_t>::id,
      &std::num_put<char>::id,  &std::num_put<wchar_t>::id,
      0
    };

  const locale::id* const
  locale::_Impl::_S_id_time[] =
    {
      &std::time_get<char>::id, &std::time_get<wchar_t>::id,
      &std::time_put<char>::id, &std::time_put<wchar_t>::id,
      0
    };

  const locale::id* const
  locale::_Impl::_S_id_messages[] =
    {
      &std::time_get<char>::id, &std::time_get<wchar_t>::id,
      &std::time_put<char>::id, &std::time_put<wchar_t>::id,
      0
    };
  
  const locale::id* const* const
  locale::_Impl::_S_facet_categories[] =
    {
      //  order must match the decl order in class locale.
      locale::_Impl::_S_id_collate,
      locale::_Impl::_S_id_ctype,
      locale::_Impl::_S_id_monetary,
      locale::_Impl::_S_id_numeric,
      locale::_Impl::_S_id_time,
      locale::_Impl::_S_id_messages,
      0
    };

  ///////////////////////////////////
  // Implementation of locale members
  ///////////////////////////////////

  // locale static data
  /////////////////////
  locale::_Impl* locale::_S_global;  // init'd to 0 before static ctors run
  locale::_Impl* locale::_S_classic; // init'd to 0 before static ctors run

  // locale constructors
  //////////////////////
  
  ////////
  locale::locale (_Impl* ip) throw ()
    : _M_impl (ip)
    { ip->_M_add_reference (); }

  //////////
  locale::locale (const locale& other,
		  const locale& one,
		  category cats)
  {
    cats = _S_normalize_category (cats);    // might throw
    _M_impl = new _Impl (*other._M_impl, 1);  // might throw

    try { _M_impl->_M_replace_categories (one._M_impl, cats); }
    catch (...) { _M_impl->_M_remove_reference (); throw; }

    _M_impl->_M_cached_name_ok = false;
    if (!other._M_impl->_M_has_name)
      _M_impl->_M_has_name = false;
  }


  // other locale members
  ///////////////////////

  ///////
  const locale&
  locale::operator= (const locale& other) throw ()
  {
    other._M_impl->_M_add_reference ();
    _M_impl->_M_remove_reference ();
    _M_impl = other._M_impl;
    return *this;
  }

  ///////
  locale
  locale::global (const locale& other)
  {
    // XXX MT
    _S_initialize ();
    locale keep (_S_global);
    other._M_impl->_M_add_reference ();
    _S_global->_M_remove_reference ();
    _S_global = other._M_impl; 
    if (_S_global->_M_has_name)
      std::setlocale (LC_ALL, other.name ().c_str ());
    return keep;
  }

  /////////
  string
  locale::name () const
  {
    // XXX not done
    return "*";
  }

  ///////
  locale const&
  locale::classic ()
  {
    static locale* the_classic_locale;
    // XXX MT
    if (!_S_classic)
      {
	try 
	  {
	    _S_classic = _S_global = new _Impl (26u, 2u);
	    // one reference for _M_classic, one for _M_global
	    // (constructor for (*the_classic_locale) adds a third)
	    
	    
	    // collate category
	    _S_classic->_M_init_facet (new std::collate<char>);
	    _S_classic->_M_init_facet (new std::collate<wchar_t>);
	    
	    // ctype category
	    _S_classic->_M_init_facet (new std::ctype<char>);
	    _S_classic->_M_init_facet (new std::ctype<wchar_t>);
	    _S_classic->_M_init_facet (new std::codecvt<char,char,mbstate_t>);
	    _S_classic->_M_init_facet
	      (new std::codecvt<wchar_t,char,mbstate_t>);
	    
	    // monetary category
	    _S_classic->_M_init_facet (new std::moneypunct<char,false>);
	    _S_classic->_M_init_facet (new std::moneypunct<wchar_t,false>);
	    _S_classic->_M_init_facet (new std::moneypunct<char,true >);
	    _S_classic->_M_init_facet (new std::moneypunct<wchar_t,true >);
	    _S_classic->_M_init_facet (new std::money_get<char>);
	    _S_classic->_M_init_facet (new std::money_get<wchar_t>);
	    _S_classic->_M_init_facet (new std::money_put<char>);
	    _S_classic->_M_init_facet (new std::money_put<wchar_t>);
	    
	    // numeric category
	    _S_classic->_M_init_facet (new std::numpunct<char>);
	    _S_classic->_M_init_facet (new std::numpunct<wchar_t>);
	    _S_classic->_M_init_facet (new std::num_get<char>);
	    _S_classic->_M_init_facet (new std::num_get<wchar_t>);
	    _S_classic->_M_init_facet (new std::num_put<char>);
	    _S_classic->_M_init_facet (new std::num_put<wchar_t>);
	    
	    // time category
	    _S_classic->_M_init_facet (new std::time_get<char>);
	    _S_classic->_M_init_facet (new std::time_get<wchar_t>);
	    _S_classic->_M_init_facet (new std::time_put<char>);
	    _S_classic->_M_init_facet (new std::time_put<wchar_t>);
	    
	    // messages category
	    _S_classic->_M_init_facet (new std::messages<char>);
	    _S_classic->_M_init_facet (new std::messages<wchar_t>);
	    
	    // finesse static init order hassles
	    the_classic_locale = new locale (_S_classic);
	  }
	catch(...)
	  {
	    delete the_classic_locale;
	    if (_S_classic)
	      {
		_S_classic->_M_remove_reference ();
		_S_global->_M_remove_reference ();
	      }
	    _S_classic = _S_global = 0;
	    // XXX MT
	    throw;
	  }
      }
    return *the_classic_locale;
  }

  ///////
  int
  locale::_S_normalize_category (int cats) 
  {
    if ((cats & all) && !(cats & ~all))
      return cats;

    // may be a C-style "LC_ALL" category; convert.
    switch (cats)
      {
      case LC_COLLATE:  return collate; 
      case LC_CTYPE:    return ctype;
      case LC_MONETARY: return monetary;
      case LC_NUMERIC:  return numeric;
      case LC_TIME:     return time; 
#ifndef _G_NO_CLOCALE_HAS_MESSAGES
      case LC_MESSAGES: return messages;
#endif	
      case LC_ALL:      return all;
      }
    
    // XXX should throw derived class here
    throw runtime_error ("bad locale category");
    /* NOTREACHED */
  }

//////////////////////////////////
// Implementation of locale::facet
//////////////////////////////////

  locale::facet::facet (size_t refs) throw ()
    : _M_num_references (refs-1) {}

  void  locale::facet::_M_add_reference () throw ()
    { if (this) ++_M_num_references; }                     // XXX MT

  void  locale::facet::_M_remove_reference () throw ()
  {
    if (this && _M_num_references-- == 0)
      {
        try { delete this; }  // XXX MT
	catch (...) { }
      }
  }

  char const* 
  _Bad_use_facet::what() const throw()
    { return "bad_cast thrown from use_facet"; }

  _Bad_use_facet::~_Bad_use_facet() throw() {}
  
///////////////////////////////
// Implementation of locale::id
///////////////////////////////

  size_t locale::id::_S_highwater;  // init'd to 0 by linker

  
//////////////////////////////////
// Implementation of ctype<char>.
//////////////////////////////////
  

  /////////
  locale::id ctype<char>::id;

  ctype<char>::~ctype ()
    { if (_M_del) delete[] table (); }

#if defined(__GLIBC__) && (__GLIBC__ >= 2)
  // XXX these are Glibc-specific.
  const int* const& ctype<char>::_S_toupper = __ctype_toupper;
  const int* const& ctype<char>::_S_tolower = __ctype_tolower;
  const ctype_base::mask* const& ctype<char>::_S_table = __ctype_b;
#else
  const int* const& ctype<char>::_S_toupper = (const int* const&)0;
  const int* const& ctype<char>::_S_tolower = (const int* const&)0;
  const ctype_base::mask* const& ctype<char>::_S_table = (const ctype_base::mask* const&)0;
#endif
  
  char
  ctype<char>::do_toupper (char c) const
  {
    return _S_toupper[(int) c];
  }


  const char*
  ctype<char>::do_toupper (char* low, const char* high) const
  {
    while (low < high)
      {
	*low = _S_toupper[(int) *low];
	++low;
      }
    return high;
  }


  char
  ctype<char>::do_tolower (char c) const
  {
    return _S_tolower[(int) c];
  }



  const char* 
  ctype<char>::do_tolower (char* low, const char* high) const
  {
    while (low < high)
      {
	*low = _S_tolower[(int) *low];
	++low;
      }
    return high;
  }


  char
  ctype<char>::do_widen (char c) const
  {
    return c;
  }
  
  const char* 
  ctype<char>::do_widen (const char* low, const char* high, char* dest) const
  {
    std::memcpy (dest, low, high - low);
    return high;
  }
  
  
  char
  ctype<char>::do_narrow (char c, char dfault) const
  {
    return c;
  }
  
  
  const char* 
  ctype<char>::do_narrow (const char* low, const char* high, char dfault,
			    char* dest) const
  {
    std::memcpy (dest, low, high - low);
    return high;
  }

  
  ////////////
  locale::id ctype<wchar_t>::id;
#ifdef __linux__ 
  // XXX these are Glibc-specific.
  const int*& ctype<wchar_t>::_S_toupper = __ctype_toupper;
  const int*& ctype<wchar_t>::_S_tolower = __ctype_tolower;
  const ctype_base::mask*& ctype<wchar_t>::_S_table = __ctype_b;
#else
  const int*& ctype<wchar_t>::_S_toupper = (const int*&)0;
  const int*& ctype<wchar_t>::_S_tolower = (const int*&)0;
  const ctype_base::mask*& ctype<wchar_t>::_S_table = (const ctype_base::mask*&)0;
#endif

  ctype<wchar_t>::ctype (size_t refs)
  : _Ctype<wchar_t> (refs)
  {
  }


  ctype<wchar_t>::~ctype () { }
  

  bool
  ctype<wchar_t>::do_is (mask m, char_type c) const
  {
    return (c < _S_table_size) && (_S_table[c] & m); 
  }
  

  const wchar_t* 
  ctype<wchar_t>::do_is (const wchar_t* low, const wchar_t* high,
			   mask* vec) const
  {
    for (;low < high; ++low, ++vec)
      *vec = (*low < _S_table_size) ? _S_table[*low] : mask(0);
    return high;
  }
  

  const wchar_t* 
  ctype<wchar_t>::do_scan_is (mask m, const wchar_t* low,
				const wchar_t* high) const
  {
    while (low < high && (_S_table_size < *low || !(_S_table[*low] & m)))
      ++low;
    return low;
  }
  

  const wchar_t*
  ctype<wchar_t>::do_scan_not (mask m, const char_type* low,
		   const char_type* high) const
  {
    while (low < high && *low < _S_table_size && (_S_table[*low] & m))
      ++low;
    return low;
  }
  


  wchar_t
  ctype<wchar_t>::do_toupper (char_type c) const
  {
    return (c < _S_table_size) ? _S_toupper[c] : c ;
  }
  

  const wchar_t*
  ctype<wchar_t>::do_toupper (char_type* low, const char_type* high) const
  {
    for (;low < high; ++low)
      if (*low < _S_table_size)
        *low = _S_toupper[*low];
    return high;
  }
  

  wchar_t
  ctype<wchar_t>::do_tolower (char_type c) const
  {
    return (c < _S_table_size) ? _S_tolower[c] : c ;
  }
  


  const wchar_t*
  ctype<wchar_t>::do_tolower (char_type* low, const char_type* high) const
  {
    for (;low < high; ++low)
      if (*low < _S_table_size)
        *low = _S_toupper[*low];
    return high;
  }
  

  wchar_t
  ctype<wchar_t>::do_widen (char c) const
  {
    return static_cast<wchar_t> ((unsigned char)c);
  }
  

  const char* 
  ctype<wchar_t>::do_widen (const char* low, const char* high,
			      wchar_t* dest) const
  {
    while (low < high)
      *dest++ = static_cast<wchar_t> ((unsigned char)*low++);
    return high;
  }


  char
  ctype<wchar_t>::do_narrow (wchar_t c, char dfault) const
  {
    return (c < _S_table_size) ? static_cast<char> (c) : dfault;
  }
  
  

  const wchar_t*
  ctype<wchar_t>::do_narrow (const wchar_t* low, const wchar_t* high,
			       char dfault, char* dest) const
  {
    for (; low < high; ++dest, ++low)
      *dest = (*low < _S_table_size) ? static_cast<char> (*low) : dfault;
    return high;
  }

  template<>
    ctype_byname<char>::ctype_byname (const char*, size_t refs)
  : ctype<char> (new mask[table_size], true, refs)
  {
  }
  

  template<>
    ctype_byname<wchar_t>::ctype_byname (const char*, size_t refs)
  : ctype<wchar_t> (refs)
  {
  }
  

  ///////////
  locale::id codecvt<char,char,mbstate_t>::id;

  codecvt<char,char,mbstate_t>::codecvt (size_t refs)
  : _Codecvt<char,char,mbstate_t> (refs)
  {
  }

  codecvt<char,char,mbstate_t>::~codecvt () { }
  
  codecvt_base::result
  codecvt<char,char,mbstate_t>::do_out (state_type& state,
	      const intern_type* from, const intern_type* from_end,
	      const intern_type*& from_next,
	      extern_type* to, extern_type* to_limit,
	      extern_type*& to_next) const
    { from_next = from; to_next = to; return noconv;  }
  
  codecvt_base::result
  codecvt<char,char,mbstate_t>::do_unshift (state_type& state,
		  extern_type* to, extern_type* to_limit,
		  extern_type*& to_next) const
    { to_next = to; return noconv; }
  
  codecvt_base::result
  codecvt<char,char,mbstate_t>::do_in (state_type& state,
	     const extern_type* from, const extern_type* from_end,
	     const extern_type*& from_next,
	     intern_type* to, intern_type* to_limit,
	     intern_type*& to_next) const
    { from_next = from; to_next = to; return noconv;  }
  

  int 
  codecvt<char,char,mbstate_t>::do_encoding () const throw ()
      { return 1; }
  
  bool 
  codecvt<char,char,mbstate_t>::do_always_noconv () const throw ()
      { return true; }
  
  int 
  codecvt<char,char,mbstate_t>::do_length (const state_type&, 
					     const extern_type* from,
					     const extern_type* end, 
					     size_t max) const
      { return (max < size_t(end-from)) ? max : end-from; }
  
  int codecvt<char,char,mbstate_t>::do_max_length () const throw ()
      { return 1; }
  

  //////////////////
  locale::id codecvt<wchar_t,char,mbstate_t>::id;

  codecvt<wchar_t,char,mbstate_t>::codecvt (size_t refs)
  : _Codecvt<wchar_t,char,mbstate_t> (refs)
  {
  }

  codecvt<wchar_t,char,mbstate_t>::~codecvt () { }
  
  codecvt_base::result
  codecvt<wchar_t,char,mbstate_t>::do_out (state_type&,
	      const intern_type* from, const intern_type* from_end,
	      const intern_type*& from_next,
	      extern_type* to, extern_type* to_limit,
	      extern_type*& to_next) const
  {
    for (;from < from_end && to < to_limit; ++from, ++to)
      *to = static_cast<char>(*from);
    from_next = from; to_next = to;
    return from == from_end ? ok : partial;
  }
  
  codecvt_base::result
  codecvt<wchar_t,char,mbstate_t>::do_unshift (state_type& state,
		  extern_type* to, extern_type* to_limit,
		  extern_type*& to_next) const
  {
    to_next = to;
    return noconv;
  }
  
  codecvt_base::result
  codecvt<wchar_t,char,mbstate_t>::do_in (state_type& state,
	     const extern_type* from, const extern_type* from_end,
	     const extern_type*& from_next,
	     intern_type* to, intern_type* to_limit,
	     intern_type*& to_next) const
  {
    for (;from < from_end && to < to_limit; ++from, ++to)
      *to = static_cast<wchar_t> (*from);
    from_next = from; to_next = to;
    return from == from_end ? ok : partial;
  }
  
  int codecvt<wchar_t,char,mbstate_t>::do_encoding () const throw ()
      { return 1; }
  
  bool 
  codecvt<wchar_t,char,mbstate_t>::do_always_noconv () const throw ()
     { return false; }
  
  int 
  codecvt<wchar_t,char,mbstate_t>::do_length (const state_type&,
						const extern_type* from,
						const extern_type* end,
						size_t max) const
      { return (max < size_t(end-from)) ? max : end-from; }
  
  int 
  codecvt<wchar_t,char,mbstate_t>::do_max_length () const throw ()
      { return 1; }
  

  codecvt_byname<char,char,mbstate_t>::codecvt_byname (const char*,
						       size_t refs)
  : codecvt<char,char,mbstate_t> (refs)
  {
  }

  codecvt_byname<char,char,mbstate_t>::~codecvt_byname () { }

  codecvt_byname<wchar_t,char,mbstate_t>::codecvt_byname (const char*,
							  size_t refs)
  : codecvt<wchar_t,char,mbstate_t> (refs)
  {
  }
  
  codecvt_byname<wchar_t,char,mbstate_t>::~codecvt_byname () {}

  ///////
  locale::id collate<char>::id;

  collate<char>::collate (size_t refs)
  : _Collate<char> (refs)
  {
  }
  

  collate<char>::~collate () { }
  
  int collate<char>::do_compare (const char* lo1, const char* hi1,
				 const char* lo2, const char* hi2) const
  {
    for (; lo1 < hi1 && lo2 < hi2; ++lo1, ++lo2) 
      if (*lo1 != *lo2) return (*lo1 < *lo2) ? -1 : 1;
    if (lo1 < hi1) return 1;
    else if (lo2 < hi2) return -1;
    else return 0;
  }
  
  string
  collate<char>::do_transform(const char* lo, const char* hi) const
  {
    return string(lo, hi-lo);
  }
  
  long
  collate<char>::do_hash (const char* lo, const char* hi) const
  {
    unsigned long val = 0xdeadbeef;
    for (; lo < hi; ++lo)
      val = *lo ^ ((val << 7) & 
		   (val >> (numeric_limits<unsigned long>::digits - 1)));
    return val;
  }
  
  //////////
  locale::id collate<wchar_t>::id;

  collate<wchar_t>::collate (size_t refs)
  : _Collate<wchar_t> (refs)
  {
  }
  
  collate<wchar_t>::~collate () { }

  int collate<wchar_t>::do_compare (const wchar_t* lo1, const wchar_t* hi1,
				      const wchar_t* lo2, 
				      const wchar_t* hi2) const
  {
    return 0; // XXX not done
  }
  
  wstring collate<wchar_t>::do_transform (const wchar_t* lo, 
					    const wchar_t* hi) const
  {
    return wstring(); // XXX not done
  }
  
  long collate<wchar_t>::do_hash (const wchar_t* lo, const wchar_t* hi) const
  {
    return 0; // XXX not done
  }
  

  template <>
    collate_byname<char>::collate_byname (const char*, size_t refs)
  : collate<char> (refs)
  {
  }
  

  template <>
    numpunct<char>::numpunct (size_t refs)
  : _Numpunct<char> (refs)
  {
    _M_init('.', ',', "");
    _M_init_boolnames("false", "true");
  }

  template <>
    numpunct<wchar_t>::numpunct (size_t refs)
  : _Numpunct<wchar_t> (refs)
  {
    _M_init(L'.', L',', "");
    _M_init_boolnames(L"false", L"true");
  }

  template <>
    numpunct_byname<char>::numpunct_byname (const char*, size_t refs)
  : numpunct<char> (refs)
  {
  }

  template <>
    numpunct_byname<wchar_t>::numpunct_byname (const char*, size_t refs)
  : numpunct<wchar_t> (refs)
  {
  }

  template <>
    collate_byname<wchar_t>::collate_byname (const char*, size_t refs)
  : collate<wchar_t> (refs)
  {
  }
  
  template <> 
    moneypunct<char,false>::moneypunct (size_t refs)
  : _Moneypunct<char> (refs)
  {
  }
  
  template <> 
    moneypunct<char,true>::moneypunct (size_t refs)
  : _Moneypunct<char> (refs)
  {
  }
  
  template <> 
    moneypunct<wchar_t,false>::moneypunct (size_t refs)
  : _Moneypunct<wchar_t> (refs)
  {
  }
  
  template <> 
    moneypunct<wchar_t,true>::moneypunct (size_t refs)
  : _Moneypunct<wchar_t> (refs)
  {
  }
  
  template <>
    moneypunct_byname<char,false>::moneypunct_byname(const char*, size_t refs)
  : moneypunct<char,false> (refs)
  {
  }
  
  template <>
    moneypunct_byname<char,true>::moneypunct_byname (const char*, size_t refs)
  : moneypunct<char,true> (refs)
  {
  }
  
  template <>
    moneypunct_byname<wchar_t,false>::moneypunct_byname (const char*,
							 size_t refs)
  : moneypunct<wchar_t,false> (refs)
  {
  }
  
  template <>
    moneypunct_byname<wchar_t,true>::moneypunct_byname (const char*,
							size_t refs)
  : moneypunct<wchar_t,true> (refs)
  {
  }
  
  template <> 
    messages<char>::messages (size_t refs)
  : _Messages<char> (refs)
  {
  }
  
  template <>
    messages<wchar_t>::messages (size_t refs)
  : _Messages<wchar_t> (refs)
  {
  }
  
  template <>
    messages_byname<char>::messages_byname (const char*, size_t refs)
  : messages<char> (refs)
  {
  }
  
  template <>
    messages_byname<wchar_t>::messages_byname (const char*, size_t refs)
  : messages<wchar_t> (refs)
  {
  }

#if _G_USE_NAMESPACE
} // namespace std
#endif
