#!/usr/bin/perl -s

########################################################################
#
# File:   tester.pl
# Author: Jeffrey Oldham
# Date:   02/27/2001
#
# Contents: Configure Pooma 2.3.0, compile it and examples, and run examples.
#
#
########################################################################

# This script configures the source code, compiles the source code and
# example programs, and runs the examples, using
# 
# 1. tester.pl [options] <g++-location-prefix>
#
# where <g++-location-prefix> specifies which gcc program to use.
# We assume gcc is at <g++-location-prefix>/bin/gcc and the
# associated C++ libraries are located at <g++-location-prefix>/lib.
#
# Command-Line Options:
#  -log=<log-location> specifies the file where the test output.  The default
#    value is "tester.log".
#  -make=<gnu-make-location> specifies the Gnu Make executable.  The
#    default value is "make".
#  -make_options=<make options> specifies options to pass to make.
#    For example, -make_options="-j 4".
#  -time=<time-command> specifies where a "time" command that is POSIX
#    1003.2 compatible.  For example, -time="/usr/local/gnu/bin/time -p".
#
# Assumptions:
# o Gnu make is available.
# o Perl is available.
# o Posix 1003.2 timing is available.
#
# Possible Difficulties:
# o The script compiles with "-static" and sets LD_LIBRARY_PATH (and
#   LD_LIBRARYN32_PATH) to avoid dynamic linking problems when running
#   executables.  If a dynamic linker error occurs during testing,
#   revise the script so it works.

######################################################################
# Process the command-line arguments.				     #
$log = "tester.log" if !defined (${log});
$make = "make" if !defined (${make});
$make_options = "" if !defined (${make_options});
$time = "/usr/bin/time -p" if !defined (${time});

######################################################################
# Record the test data.						     #
use FileHandle;
open (LOGFILE, ">${log}")
    || die ("Failed to open the log file ${log}.\n");
autoflush LOGFILE 1;	# Force all writes to buffer in case of
			# premature termination.
print (LOGFILE "Execute Pooma 2.3.0 gcc 3.0 test suite on ");
print (LOGFILE scalar localtime, "\n");

######################################################################
# Substitute configuration values in files.			     #

use File::Basename;
die "$0: Specify the location of g++ as the command-line argument."
    if ($#ARGV == -1);
$prefix = shift;
chop $prefix if $prefix =~ /\/$/;
$gxx_location = "${prefix}/bin/g++";
($gcc_location = $gxx_location) =~ s/g\+\+/gcc/;
$library_location = "${prefix}/lib/";
print (LOGFILE "g++ location: $gxx_location\n");
print (LOGFILE "gcc location: $gcc_location\n");
print (LOGFILE "library location: $library_location\n");

$uname = `uname`;
if ($uname =~ m/linux/i)	{ $arch = "LINUXgcc"; }
elsif ($uname =~ m/irix64/i)	{ $arch = "SGIN32gcc"; }
## FIX How do I know when to use "SGI64gcc"?
elsif ($uname =~ m/sunos/i)	{ $arch = "SUNgcc"; }
else
{
    die ("The ${uname} architecture is not supported.");
}

open (CONFIGIN, "./config/arch/${arch}-template.conf")
    || die ("Cannot find the configuration file.");
open (CONFIGOUT, ">./config/arch/${arch}.conf")
    || die ("Cannot write the configuration file.");
while (<CONFIGIN>)
{
    s/CXX-PLACEHOLDER/$gxx_location/g;
    s/C-PLACEHOLDER/$gcc_location/g;
    s/LIB-PLACEHOLDER/$library_location/g;
    print(CONFIGOUT);
}
close (CONFIGOUT);
die ("Failed to finish writing the configuration file.") if $?;
close (CONFIGIN);
die ("Failed to finish reading the configuration file.") if $?;

open (VARIABLETEMPLATE, "./config/Shared/variables-template.mk")
    || die ("Cannot find the shared variables template file.");
open (VARIABLES, ">./config/Shared/variables.mk")
    || die ("Cannot write the shared variables file.");
while (<VARIABLETEMPLATE>)
{
    s@TIME-PLACEHOLDER@${time}@ if /TIME-PLACEHOLDER/;
    print (VARIABLES);
}
close (VARIABLES);
die ("Failed to finish writing the shared variables file.") if $?;
close (VARIABLETEMPLATE);
die ("Failed to finish reading the shared variables template file.") if $?;

######################################################################
# Set the environment variables.				     #
$ENV{LD_LIBRARY_PATH} = "${library_location}:$ENV{LD_LIBRARY_PATH}";
if ($uname eq "SGIN32gcc")
{
    $ENV{LD_LIBRARYN32_PATH} = "${library_location}:$ENV{LD_LIBRARYN32_PATH}";
}
elsif ($uname eq "SGI64gcc")
{
    $ENV{LD_LIBRARY64_PATH} = "${library_location}:$ENV{LD_LIBRARY64_PATH}";
}

######################################################################
# Configure for compilation.					     #

open (COMMAND, "./configure --arch ${arch} --opt |")
    || die ("Failed to configure.\n");
while (<COMMAND>)
{
    print (LOGFILE);
    print;
}
close (COMMAND);
if ($?)
{
    print (LOGFILE "Configuring Pooma 2.3.0 failed.\n");
    close (LOGFILE);
    die ("Configuring Pooma 2.3.0 failed.");
}

######################################################################
# Compile Pooma.						     #

@files = ();
open (COMMAND, "${make} ${make_options} POOMASUITE=${arch} |")
    || die ("Failed to compile Pooma.\n");
while (<COMMAND>)
{
    if (/\.info/)
    {
	($ignored, $ignored, $file) = split;
	push (@files, $file);
    }
    print (LOGFILE);
    print;
}
foreach $file (@files)
{
    print (LOGFILE "==> ${file} <==\n");
    print (LOGFILE `cat ${file}`);
}
close (COMMAND);
if ($?)
{
    print (LOGFILE "Compiling Pooma 2.3.0 failed.\n");
    close (LOGFILE);
    die ("Compiling Pooma 2.3.0 failed.  Please submit ${log}.");
}

######################################################################
# Compile and run the tests.					     #

@files = ();
open (COMMAND, "cd examples && ${make} ${make_options} ARCH=${arch} |")
    || die ("Failed to compile the examples.\n");
while (<COMMAND>)
{
    if (/\.info/)
    {
	($ignored, $ignored, $file) = split;
	push (@files, $file);
    }
    print (LOGFILE);
    print;
}
close (COMMAND);
if ($?)
{
    print (LOGFILE "Compiling and executing the Pooma 2.3.0 examples failed.\n");
    close (LOGFILE);
    die ("Compiling and executing the Pooma 2.3.0 examples failed.  Please submit ${log}.");
}

######################################################################
# Print the compilation times.					     #

open (TIMES, "make showtimes |")
    || die ("Failed to summarize the compile times.\n");
while (<TIMES>)
{
    print (LOGFILE);
    print;
}
close (COMMAND);
if ($?)
{
    print (LOGFILE "Failed to summarize the compile times.\n");
    close (LOGFILE);
    die ("Failed to summarize the compile times.  Please submit ${log}.");
}

######################################################################
# Finish.							     #

print ("All testing succeeded.\n");
print (LOGFILE "All testing succeeded.\n");
close (LOGFILE);
die ("Failed to finish writing the log file ${log}.\n") if $?
