// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

#ifndef POOMA_UTILITIES_REFCOUNTED_H
#define POOMA_UTILITIES_REFCOUNTED_H

//-----------------------------------------------------------------------------
// Classes:
//   RefCounted
//   Shared<T>
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Overview:
//   RefCounted: Mix-in class that encapsulates the reference-counting 
//               of an object.
//   Shared<T>:  A template that simply inherits from RefCounted,
//               and has a member that returns the contained data.
//               The data member is protected, so this can be used
//               via inheritance.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Include Files
//-----------------------------------------------------------------------------

#include "Utilities/PAssert.h"
#include "Threads/PoomaMutex.h"

///////////////////////////////////////////////////////////////////////////////
// namespace Pooma {

//-----------------------------------------------------------------------------
//
// Full Description:
// RefCounted
//
// RefCounted is a mix-in class that supports reference counting
// of an object. It encapsulates the count and provides an interface
// for manipulating and checking the count.
//
// When running in a threaded environment, RefCounted protects the
// reference count with a mutex, so this class is thread safe.
//
//-----------------------------------------------------------------------------

class RefCounted 
{
public:
  
  //============================================================
  // Constructors, Destructor, Assignment...
  //============================================================

  // Default constructor.
  // Starts count at zero. The client that creates the RefCounted
  // object is responsible for calling addReference().

  RefCounted() 
    : count_m(0) 
    { }

  // Copy constructor.
  // This may appear a bit odd. If a RefCounted object is
  // copied, then this creates a NEW RefCounted object
  // that must be reference counted separately from the
  // old one. Thus its count must be initialized to zero.
  // Ordinarily RefCounted objects aren't copied. However,
  // clients may wish to implement a clone() operation
  // that does explicitly make a (deep) copy.

  RefCounted(const RefCounted &) 
    : count_m(0) 
    { }

  // Trivial destructor:

  ~RefCounted() { }

  //============================================================
  // Accessors
  //============================================================

  bool isShared() const;

  //============================================================
  // Mutators
  //============================================================

  // These simply increment and decrement the reference count.

  void addReference();
  void removeReference();

  // Remove reference and check if it leaves garbage.

  bool removeRefAndCheckGarbage();
  
  // Expose lock and unlock:

  void lock() const
  {
    mutex_m.lock();
  }

  void unlock() const
  {
    mutex_m.unlock();
  }

  // Return the current value of the reference count.

  int count() const;
  
  // Ditto, but without locking the mutex while copying it.
  
  int countUnlocked() const;

private:

  // Assignment is private and unimplemented.

  RefCounted & operator=(const RefCounted &); // UNIMPLEMENTED

  int count_m;

  // Mutex is declared mutable since we must be able to lock and
  // unlock the mutex in accessors that don't change the logical state
  // of the RefCounted object.

  mutable Pooma::Mutex_t mutex_m;

};


//-----------------------------------------------------------------------------
// Inline functions
//-----------------------------------------------------------------------------

inline bool 
RefCounted::isShared() const 
{ 
  mutex_m.lock();
  bool test = count_m > 1;
  mutex_m.unlock();
  return test; 
}


inline void 
RefCounted::addReference()
{
  mutex_m.lock();
  ++count_m;
  mutex_m.unlock();
}

inline void 
RefCounted::removeReference()
{
  mutex_m.lock();
  --count_m;
  PAssert(count_m >= 0);
  mutex_m.unlock();
}

inline bool
RefCounted::removeRefAndCheckGarbage() 
{ 
  mutex_m.lock();
  PAssert(count_m > 0); 
  bool test = --count_m == 0;
  mutex_m.unlock();
  return test;
}

inline int
RefCounted::count() const
{
  mutex_m.lock();
  int count = count_m;
  mutex_m.unlock();
  return count;
}

inline int
RefCounted::countUnlocked() const
{
  return count_m;
}


//-----------------------------------------------------------------------------
//
// Full Description:
// Shared<T>
//
//  Simple template class encapsulating a single data item and
//  inheriting from RefCounted. 
//
//-----------------------------------------------------------------------------

template <class T>
class Shared : public RefCounted
{
public:

  //============================================================
  // Constructors, Destructor, Assignment...
  //============================================================

  Shared(const T &d) : data_m(d) {};

  Shared(const Shared<T> & model) 
    : data_m(model.data_m) 
  { }

  Shared<T> & operator=(const Shared<T> &model)
  { 
    if (&model == this) return *this;
    data_m = model.data_m;
    return *this;
  }

  Shared<T> & operator=(const T & d) 
  { 
    data_m = d; 
    return *this; 
  }

  //============================================================
  // Accessors
  //============================================================

  inline
  T &data() { return data_m; }

  inline
  const T &data() const { return data_m; }

  bool operator==(const Shared<T> &rhs) const
    { return data_m == rhs.data_m; }

  bool operator!=(const Shared<T> &rhs) const
    { return data_m != rhs.data_m; }

protected:

  T data_m;
};



// } // namespace Pooma
///////////////////////////////////////////////////////////////////////////////

#endif // POOMA_UTILITIES_REFCOUNTED_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: RefCounted.h,v $   $Author: swhaney $
// $Revision: 1.17 $   $Date: 2000/03/07 13:18:27 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
