//-----------------------------------------------------------------------------
// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
#include <string>
#include <iostream>
#include <fstream>
#include "Utilities/Tester.h"
#include "Utilities/PAssert.h"
#include "Pooma/Pooma.h"
#include "Tiny/Vector.h"
#include "Tiny/TinyMatrix.h"
#include "IO/MatrixIO.h"

//-----------------------------------------------------------------------------
// Test of Pooma matrix element serializers
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------

void itemCheck(Pooma::Tester& tstr, bool cond, const std::string& m);
void itemCheck(Pooma::Tester& tstr, bool cond, const std::string& m){
  tstr.check(cond);
  if(cond){
    tstr.out() << "PASSED "<<m<<std::endl;
  }
  else{
    tstr.out() << "FAILED "<<m<<std::endl;
  }
}

template <class T>
void bufferCheck(const T& t, int nBytes, int buffsize);
template <class T>
void bufferCheck(const T& t, int nBytes, int buffsize){
  PInsist((nBytes+serialSizeof(t))<buffsize,
	  "Test buffer size exceeded");
}


//-----------------------------------------------------------------------------

int main(int argc, char *argv[])
{
  Pooma::initialize(argc, argv);
  Pooma::Tester tester(argc, argv);

  tester.out() <<"Test of matrix element serializers:"<< std::endl;

  // Open a file for the test.
  std::fstream datafile("matrixSerializerTest.dat", 
			std::ios::out | std::ios::binary);

  // Create a buffer for the test.
  const int buffsize= 10000;
  char databuffer[buffsize];
  char* databuf= (char*) &databuffer[0];
  int nBytes=0;
  int ksize;

  // Create double and int vector objects.
  Vector<10,int> ivec;
  Vector<10> dvec;

  // Assign values.
  int i;
  for(i=0;i<10;i++){
    ivec(i)= i*10;
    dvec(i)= i+0.5;
  }

  // Check type labels.
  std::string glabel= PoomaCTTI(ivec);
  tester.out()<<glabel<<std::endl;

  glabel= PoomaCTTI(dvec);
  tester.out()<<glabel<<std::endl;

  
  // Serialize the vectors to the file.
  ksize= serialize(datafile, ivec);
  itemCheck(tester,(ksize==serialSizeof(ivec)),
	    "Size computation on int vector");
  
  ksize= serialize(datafile, dvec);
  itemCheck(tester,(ksize==serialSizeof(dvec)),
	    "Size computation on double vector");

  // Serialize the vectors to the buffer.
  bufferCheck(ivec,nBytes,buffsize);
  nBytes+= serialize(databuf, ivec);
  bufferCheck(dvec,nBytes,buffsize);
  nBytes+= serialize(databuf, dvec);

  // Create a TinyMatrix object.
  TinyMatrix<3,4> dmat;

  // Assign some values.
  int j;
  for(j=0;j<4;j++){
    for(i=0;i<3;i++){
      dmat(i,j)= i*10 + 0.1*j;
    }
  }

  // Check type label.
  glabel= PoomaCTTI(dmat);
  tester.out()<<glabel
	   <<std::endl;

  // Serialize the matrix to the file.
  ksize= serialize(datafile, dmat);
  itemCheck(tester,(ksize==serialSizeof(dmat)),
      "Size computation for double TinyMatrix");

  // Serialize the matrix to the buffer.
  bufferCheck(dmat,nBytes,buffsize);
  nBytes+= serialize(databuf, dmat);

  // Create a full Tensor object.
  Tensor<3,double> ftens;

  // Assign values.
  for(j=0;j<3;j++){
    for(i=0;i<3;i++){
      ftens(i,j)= i*10 + 0.1*j;
    }
  }

  // Check type label.
  glabel= PoomaCTTI(ftens);
  tester.out()<<glabel
	   <<std::endl;

  // Serialize the Tensor to the file.
  ksize= serialize(datafile, ftens);
  itemCheck(tester,(ksize==serialSizeof(ftens)),
      "Size computation on full double tensor");

  // Serialize the Tensor to the buffer.
  bufferCheck(ftens,nBytes,buffsize);
  nBytes+= serialize(databuf, ftens);

  // Create a symmetric Tensor object.
  Tensor<3,double,Symmetric> stens;

  // Assign values.
  for(j=0;j<3;j++){
    for(i=0;i<(j+1);i++){
      stens(i,j)= i*10 + 0.1*j;
    }
  }

  // Check type label.
  glabel= PoomaCTTI(stens);
  tester.out()<<glabel
	      <<std::endl;

  // Serialize the Tensor to the file.
  ksize= serialize(datafile, stens);
  itemCheck(tester,(ksize==serialSizeof(stens)),
      "Size computation on double symmetric Tensor");

  // Serialize the Tensor to the buffer.
  bufferCheck(stens,nBytes,buffsize);
  nBytes+= serialize(databuf, stens);

  // Create an diagonal Tensor object.
  Tensor<3,double,Diagonal> dtens;

  // Assign values.
  for(i=0;i<3;i++){
    dtens(i,i)= i*10 + 0.1*i;
   }

  // Check type label.
  glabel= PoomaCTTI(dtens);
  tester.out()<<glabel
	      <<std::endl;

  // Serialize the Tensor to the file.
  ksize= serialize(datafile, dtens);
  itemCheck(tester,(ksize==serialSizeof(dtens)),
      "Size computation on diagonal double Tensor");

  // Serialize the Tensor to the buffer.
  bufferCheck(dtens,nBytes,buffsize);
  nBytes+= serialize(databuf, dtens);
  
  // Create an antisymmetric Tensor object.
  Tensor<3,double,Antisymmetric> atens;

  // Assign values.
  for(j=0;j<3;j++){
    for(i=0;i<j;i++){
      atens(i,j)= i*10 + 0.1*j;
    }
  }

  // Check type label.
  glabel= PoomaCTTI(atens);
  tester.out()<<glabel
	      <<std::endl;

  // Serialize the Tensor to the file.
  ksize= serialize(datafile, atens);
  itemCheck(tester,(ksize==serialSizeof(atens)),
      "Size computation on antisymmetric double Tensor");

  // Serialize the Tensor to the buffer.
  bufferCheck(atens,nBytes,buffsize);
  nBytes+= serialize(databuf, atens);

  // Close the file.
  datafile.close();

  // Reopen as another file object.
  std::fstream dfile("matrixSerializerTest.dat", 
                     std::ios::in | std::ios::binary);

  // Reset the buffer pointer.
  databuf= (char*) &databuffer[0];

  // Create new vector objects to retrieve data.
  Vector<10,int> ivec2, ivec3;
  Vector<10> dvec2, dvec3;

  // Deserialize the vectors from the file.
  deserialize(ivec2, dfile);
  deserialize(dvec2, dfile);

  // Deserialize the vectors from the buffer.
  deserialize(ivec3, databuf);
  deserialize(dvec3, databuf);

  // Print out results.

  tester.out()<<"int Vector values:"<<std::endl;
  for(i=0;i<10;i++){
    tester.out()<<ivec2(i)<<" ";
  }
  tester.out()<<std::endl;

  tester.out()<<"double Vector values:"<<std::endl;
  for(i=0;i<10;i++){
    tester.out()<<dvec2(i)<<" ";
  }
  tester.out()<<std::endl;

  // Verify results elementwise.
  bool compare=true;
  for(i=0;i<10;i++){
    compare= compare&& (ivec(i)==ivec2(i));
  }
  itemCheck(tester,compare,"Check on int vector from file");
  compare= true;
  for(i=0;i<10;i++){
    compare= compare&& (ivec(i)==ivec3(i));
  }
  itemCheck(tester,compare,"Check on int vector from buffer");
  compare= true;
  for(i=0;i<10;i++){
    compare= compare&& (dvec(i)==dvec2(i));
  }
  itemCheck(tester,compare,"Check on double vector from file");
  compare= true;
  for(i=0;i<10;i++){
    compare= compare&& (dvec(i)==dvec3(i));
  }
  itemCheck(tester,compare,"Check on double vector from buffer");

  // Create a new TinyMatrix objects.
  TinyMatrix<3,4> dmat2, dmat3;

  // Deserialize from file.
  deserialize(dmat2, dfile);
  // Deserialize from buffer.
  deserialize(dmat3, databuf);

  // Print out the result.
  tester.out()<<"double TinyMatrix values:"<<std::endl;

  for(j=0;j<4;j++){
    for(i=0;i<3;i++){
      tester.out()<<dmat2(i,j)<<" ";
    }
    tester.out()<<std::endl;
  }
  tester.out()<<std::endl;

  // Validate elementwise.
  compare= true;
  for(j=0;j<4;j++){
    for(i=0;i<3;i++){
      compare= compare&& (dmat(i,j)==dmat2(i,j));
    }
  }
  itemCheck(tester,compare,"Check on TinyMatrix from file");
  compare= true;
  for(j=0;j<4;j++){
    for(i=0;i<3;i++){
      compare= compare&& (dmat(i,j)==dmat3(i,j));
    }
  }
  itemCheck(tester,compare,"Check on TinyMatrix from buffer");

  // Create a new full Tensor object.
  Tensor<3,double> ftens2, ftens3;

  // Deserialize the Tensor from the file.
  deserialize(ftens2, dfile);
  // Deserialize the Tensor from the buffer.
  deserialize(ftens3, databuf);

  // Print out the values.
  tester.out()<<"Tensor<3,double,Full> values:"<<std::endl;

  for(j=0;j<3;j++){
    for(i=0;i<3;i++){
      tester.out()<<ftens2(i,j)<<" ";
    }
    tester.out()<<std::endl;
  }
  tester.out()<<std::endl;

  // Validate the result elementwise.
  compare= true;
  for(j=0;j<3;j++){
    for(i=0;i<3;i++){
      compare= compare&& (ftens(i,j)==ftens2(i,j));
    }
  }
  itemCheck(tester,compare,"Check on full tensor from file");
  compare= true;
  for(j=0;j<3;j++){
    for(i=0;i<3;i++){
      compare= compare&& (ftens(i,j)==ftens3(i,j));
    }
  }
  itemCheck(tester,compare,"Check on full tensor from buffer");

  // Create a new symmetric Tensor objects.
  Tensor<3,double,Symmetric> stens2, stens3;

  // Deserialize the Tensor from the file.
  deserialize(stens2, dfile);
  // Deserialize the Tensor from the buffer.
  deserialize(stens3, databuf);

  // Print out all the values.
  tester.out()<<"Tensor<3,double,Symmetric> values:"<<std::endl;

  for(j=0;j<3;j++){
    for(i=0;i<3;i++){
      tester.out()<<stens2(i,j)<<" ";
    }
    tester.out()<<std::endl;
  }
  tester.out()<<std::endl;

  // Validate elementwise.
  compare= true;
  for(j=0;j<3;j++){
    for(i=0;i<3;i++){
      compare= compare&& (stens(i,j)==stens2(i,j));
    }
  }
  itemCheck(tester,compare,"Check on symmetric tensor from file");
  compare= true;
  for(j=0;j<3;j++){
    for(i=0;i<3;i++){
      compare= compare&& (stens(i,j)==stens3(i,j));
    }
  }
  itemCheck(tester,compare,"Check on symmetric tensor from buffer");


  // Create a new diagonal Tensor objects.
  Tensor<3,double,Diagonal> dtens2, dtens3;

  // Deserialize the Tensor from the file.
  deserialize(dtens2, dfile);
  // Deserialize the Tensor from the buffer.
  deserialize(dtens3, databuf);

  // Print out all the values.
  tester.out()<<"Tensor<3,double,Diagonal> values:"<<std::endl;

  for(j=0;j<3;j++){
    for(i=0;i<3;i++){
      tester.out()<<dtens2(i,j)<<" ";
    }
    tester.out()<<std::endl;
  }
  tester.out()<<std::endl;

  // Validate elementwise.
  compare= true;
  for(j=0;j<3;j++){
    for(i=0;i<3;i++){
      compare= compare&& (dtens(i,j)==dtens2(i,j));
    }
  }
  itemCheck(tester,compare,"Check on diagonal tensor from file");
  compare= true;
  for(j=0;j<3;j++){
    for(i=0;i<3;i++){
      compare= compare&& (dtens(i,j)==dtens3(i,j));
    }
  }
  itemCheck(tester,compare,"Check on diagonal tensor from buffer");

  // Create a new antisymmetric Tensor objects.
  Tensor<3,double,Antisymmetric> atens2, atens3;

  // Deserialize the Tensor from the file.
  deserialize(atens2, dfile);
  // Deserialize the Tensor from the buffer.
  deserialize(atens3, databuf);

  // Print out all the values.
  tester.out()<<"Tensor<3,double,Antisymmetric> values:"<<std::endl;

  for(j=0;j<3;j++){
    for(i=0;i<3;i++){
      tester.out()<<atens2(i,j)<<" ";
    }
    tester.out()<<std::endl;
  }
  tester.out()<<std::endl;

  // Validate elementwise.
  compare= true;
  for(j=0;j<3;j++){
    for(i=0;i<3;i++){
      compare= compare&& (atens(i,j)==atens2(i,j));
    }
  }
  itemCheck(tester,compare,"Check on antisymmetric tensor from file");
  compare= true;
  for(j=0;j<3;j++){
    for(i=0;i<3;i++){
      compare= compare&& (atens(i,j)==atens3(i,j));
    }
  }
  itemCheck(tester,compare,"Check on antisymmetric tensor from buffer");
  dfile.close();

  int retval = tester.results("Test of matrix serializers");


  Pooma::finalize();
  return retval;
}

















