// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

// ----------------------------------------------------------------------------
// Field test 2: Try out non-Vert/non-Cell centerings, including tests with
// multicomponent Field element types
// ----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Pooma/Fields.h"
#include "Utilities/Tester.h"

//-----------------------------------------------------------------------------
// Main program:
//-----------------------------------------------------------------------------

int main(int argc, char *argv[])
{
  Pooma::initialize(argc,argv);
  Pooma::Tester tester(argc, argv);

  // Create the mesh

  const int D = 2; // dimensionality
  const int N = 6; // number of verts in each direction
  int d;
  
  Interval<D> allVerts;
  for (d = 0; d < D; d++) 
    allVerts[d] = Interval<1>(N);
  
  Vector<D,double> origin;
  Vector<D,double> spacings;
  for (d = 0; d < D; d++) 
    {
      origin(d) = d;
      spacings(d) = d + 1;
    }
  UniformRectilinearMesh<2> mesh(allVerts, origin, spacings);
  
  // Create the geometry.
  typedef RectilinearCentering<D,VectorFaceRCTag<D> > Centering_t;
  typedef DiscreteGeometry<Centering_t , UniformRectilinearMesh<D> > Geometry_t;
  
  Geometry_t geom(mesh, GuardLayers<D>(1));
  
  // Make the field.
  Field<Geometry_t, Vector<D,double> > 
    f(geom);
  f.addBoundaryConditions(AllZeroFaceBC());

  // Initialize some values:
  int i,j;
  for (i = 0; i <= f.physicalDomain()[0].last(); i++) {
    for (j = 0; j <= f.physicalDomain()[1].last(); j++) {
      f(i, j) = Vector<D,double>(i + j);
    }
  }
   
  // Apply boundary conditions:      
  f.applyBoundaryConditions();

  tester.check("f(4,4)", f(4,4), Vector<D>(8));
  tester.check("f.read(4,4)", f.read(4,4), Vector<D>(8));
  tester.check("f(0,3)", f(0,3), Vector<D>(3));
  tester.check("f(6,6)", f(6,6), Vector<D>(0));
  tester.check("f.x(0,0)(0)", f.x(0,0)(0), Vector<D>(0,2));
  tester.check("f.xComp(0).read(0,0)", f.xComp(0).read(0,0), Vector<D>(0,2));
  tester.check("f.x(0,0)(1)", f.x(0,0)(1), Vector<D>(0.5,1));

  tester.out() << "********** Field<...VectorFaceRCTag....> ***********" 
    << std::endl;
  tester.out() << "^^^^^^^^^^^^^^^Field values^^^^^^^^^^^^^^^" 
    << std::endl;
  tester.out() << f << std::endl;
  tester.out() << "^^^^^^^^^^^^^^^Field.x values^^^^^^^^^^^^^^^" 
    << std::endl;
  tester.out() << f.x() << std::endl;
  tester.out() << std::endl;

  // X-Face-Centered Field

  // Create the geometry.
  typedef RectilinearCentering<D,FaceRCTag<1> > Centering1_t;
  DiscreteGeometry<Centering1_t, UniformRectilinearMesh<D> > 
    geom1(mesh, GuardLayers<D>(1));

  // Make the field.
  Field<DiscreteGeometry<Centering1_t, UniformRectilinearMesh<D> >, 
    Tensor<D,double,Full> > 
    f1(geom1);
  f1.addBoundaryConditions(AllZeroFaceBC());

  // Initialize some values:
  for (i = 0; i <= f1.physicalDomain()[0].last(); i++) {
    for (j = 0; j <= f1.physicalDomain()[1].last(); j++) {
      //      f1(i, j) = Tensor<D,double>(i + j);
      f1(i, j) = i + j;
    }
  }
  
  f1.applyBoundaryConditions();

  tester.out() << "********** Field<...FaceRCTag<1>....> ***********" 
    << std::endl;
  tester.out() << "^^^^^^^^^^^^^^^Field values^^^^^^^^^^^^^^^" 
    << std::endl;
  tester.out() << f1 << std::endl;
  tester.out() << "^^^^^^^^^^^^^^^Field.x values^^^^^^^^^^^^^^^" 
    << std::endl;
  tester.out() << f1.x() << std::endl;
  tester.out() << "f1.x().read(1,1)" << std::endl;
  tester.out() << f1.x().read(1,1) << std::endl;
  tester.out() << std::endl;

  tester.check("f1(4,4)", f1(4,4), Tensor<D,double,Full>(8));
  tester.check("f1.read(4,4)", f1.read(4,4), Tensor<D,double,Full>(8));
  tester.check("f1(0,3)", f1(0,3), Tensor<D,double,Full>(3));
  tester.check("f1(5,6)", f1(5,6), Tensor<D,double,Full>(0));
  tester.check("f1.x(0,0)", f1.x(0,0), Vector<D>(0.5,1));
  tester.check("f1.x(1,1)", f1.x(1,1), Vector<D>(1.5,3));

  int ret = tester.results("field_test2");
  Pooma::finalize();
  return ret;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: field_test2.cpp,v $   $Author: julianc $
// $Revision: 1.14 $   $Date: 2000/07/20 22:21:29 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
