// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

// ----------------------------------------------------------------------------
// TestFieldNDIndexing.cpp , Tim Williams 6/23/1999
// 
// Tests the use of an NDIndex object in single brackets instead of N Index
// objects in their own brackets for indexing a Field. Includes testing of
// index arithmetic on the NDIndex objects.
// ----------------------------------------------------------------------------

// include files
#include "Pooma/Fields.h"
#include "Utilities/Tester.h"

int main(int argc, char* argv[])
{
  // Initialize POOMA and Tester class.
  Pooma::initialize(argc, argv);
  Pooma::Tester tester(argc, argv);

  tester.out() << argv[0] << ": Field indexing operations" << std::endl;
  tester.out() << "------------------------------------------------"
               << std::endl;

  const int Dim = 2; // Dimensionality
  int d;

  // Vertex and cell domains:
  int nVerts[3] = {5, 5, 5};
  Interval<Dim> vertDomain, cellDomain;
  for (d = 0; d < Dim; d++) {
    vertDomain[d] = Interval<1>(nVerts[d]);
    cellDomain[d] = Interval<1>(nVerts[d] - 1);
  }

  // Create the mesh; default origin and spacings:
  typedef UniformRectilinearMesh<Dim> Mesh_t;
  Mesh_t mesh(vertDomain);
  
  // Create the geometry:
  DiscreteGeometry<Cell, Mesh_t> geomc(mesh, GuardLayers<Dim>(2));
  
  // Make some fields:
  typedef Field<DiscreteGeometry<Cell, Mesh_t>, double> ScalarField_t;
  typedef Field<DiscreteGeometry<Cell, Mesh_t>, Vector<Dim> > VectorField_t;
  ScalarField_t s1(geomc);
  VectorField_t v1(geomc);

  Interval<Dim> interiorCells;
  for (d = 0; d < Dim; d++) {
    interiorCells[d] = Interval<1>(1, cellDomain[d].max() - 1);
  }

  // Test 1: indexing Fields with an Interval<Dim>
  s1(s1.totalDomain()) = 0.0;
  Vector<Dim> vector0(0.0), vector1(1.0), vector4(4.0), vectorD(2<<(Dim-1));
  v1(v1.totalDomain()) = vector0;
  s1(interiorCells) = 1.0;
  v1(interiorCells) = vector1;
  tester.out() << "Test 1" << std::endl;
  if (!tester.check(sum(s1) == double(2<<(Dim-1)))) {
    tester.out() << "...failed" << std::endl;
    tester.out() << "sum(s1) = " << sum(s1) << " ; should be = "
		 << double(2<<(Dim-1)) << std::endl;
  }
  tester.out() << "Test 1v" << std::endl;
  if (!tester.check(sum(v1) == vectorD)) {
    tester.out() << "...failed" << std::endl;
    tester.out() << "sum(v1) = " << sum(v1) << " ; should be = " << vectorD
		 << std::endl;
  }


  //------------------------------------------
  // Interval<Dim> arithmetic:

  // Test 2: (Interval<Dim> + Loc<Dim>) and (Interval<Dim> - Loc<Dim>):
  ScalarField_t s2(geomc);
  s1 = 1.0;
  s2 = 1.0;
  Loc<Dim> ones(1);
  s2(interiorCells) = s1(interiorCells + ones) + s1(interiorCells - ones);
  s1 = s2 - s1;
  double correctSum = interiorCells.size();
  tester.out() << "Test 2" << std::endl;
  if (!tester.check(sum(s1) == correctSum)) {
    tester.out() << "...failed" << std::endl;
    tester.out() << "sum(s1) = " << sum(s1) << " ; should be = " << correctSum
		 << std::endl;
  }    

  // Test 3: (Loc<Dim> + Interval<Dim>) (and (Interval<Dim> - Loc<Dim>)):
  s1 = 1.0;
  s2 = 1.0;
  s2(interiorCells) = s1(ones + interiorCells) + s1(interiorCells - ones);
  s1 = s2 - s1;
  correctSum = interiorCells.size();
  tester.out() << "Test 3" << std::endl;
  if (!tester.check(sum(s1) == correctSum)) {
    tester.out() << "...failed" << std::endl;
    tester.out() << "sum(s1) = " << sum(s1) << " ; should be = " << correctSum
		 << std::endl;
  }    

  // Test 4: (Interval<Dim> * Loc<Dim>):
  s1 = 1.0;
  s2 = s1;
  Loc<Dim> twos(2);
  Interval<Dim> zeroToOneByOne;
  Range<Dim> zeroToTwoByTwo;
  for (d = 0; d < Dim; d++) {
    const Interval<1> iii(2);
    zeroToOneByOne[d] = Interval<1>(2);
    zeroToTwoByTwo[d] = Range<1>(0,2,2);
  }
  s1(zeroToTwoByTwo) = 2.0;
  s1(zeroToTwoByTwo) /= s1(zeroToOneByOne*twos);
  tester.out() << "Test 4" << std::endl;
  if (!tester.check(sum(s1) == sum(s2))) {
    tester.out() << "...failed" << std::endl;
    tester.out() << "sum(s1) = " << sum(s1) << " ; should be = "
		 << "sum(s2) = " << sum(s2) << std::endl;
  }

  // Test 4: (Loc<Dim> * Interval<Dim>):
  s1 = 1.0;
  s2 = s1;
  s1(zeroToTwoByTwo) = 2.0;
  s1(zeroToTwoByTwo) /= s1(twos*zeroToOneByOne);
  tester.out() << "Test 5" << std::endl;
  if (!tester.check(sum(s1) == sum(s2))) {
    tester.out() << "...failed" << std::endl;
    tester.out() << "sum(s1) = " << sum(s1) << " ; should be = "
		 << "sum(s2) = " << sum(s2) << std::endl;
  }

  // (-Interval<Dim>) (changed much from r1, because r2 doesn't allow Fields
  // to be defined on non-zero-based domains):
  s1.all() = 0.0;
  for (d = 0; d < Dim; d++) {
    s1.all() += s1.xAll().comp(d);
  }
  s2.all() = s1.all();
  Interval<Dim> src, dest;
  for (d = 0; d < Dim; d++) {
    dest[d] = Interval<1>(-s1.geometry().guardLayers().lower(d), -1);
    src[d] = Interval<1>(0, s1.geometry().guardLayers().lower(d) - 1);
  }
  s1(-dest) = -s1(src);
  tester.out() << "Test 6" << std::endl;
  if (!tester.check((sum(s1.all() - s2.all()) == 
		     (sum(s1(-dest) - s2(-dest)))))) {
    tester.out() << "...failed" << std::endl;
    tester.out() << "dest = " << dest << " ; src = " << src 
		 << " ; -dest = " << -dest << std::endl;
    tester.out() << "s1.all(): " << std::endl << s1.all() << std::endl;
    tester.out() << "s2.all(): " << std::endl << s2.all() << std::endl;
    tester.out() << "s1(-dest): " << std::endl << s1(-dest) << std::endl;
    tester.out() << "s2(src): " << std::endl << s2(src) << std::endl;
    tester.out() << "sum(s1.all() - s2.all()) = " << sum(s1.all() - s2.all())
		 << "; should = sum(s2(-dest) - s1(-dest)) = " 
		 << sum(s1(-dest) - s2(-dest)) << std::endl;
  }
 
  // (Range<Dim> / Loc<Dim>):
  s1.all() = 0.0;
  for (d = 0; d < Dim; d++) {
    s1.all() += s1.xAll().comp(d);
  }
  s2.all() = s1.all();
  s1(zeroToTwoByTwo/twos) = 0.0;
  tester.out() << "Test 7" << std::endl;
  if (!tester.check((sum(s1.all() - s2.all())) == -sum(s2(zeroToOneByOne)))) {
    tester.out() << "...failed" << std::endl;
    tester.out() << "zeroToTwoByTwo = " << zeroToTwoByTwo 
		 << " ; zeroToOneByOne = " << zeroToOneByOne << " ; twos = " 
		 << twos << std::endl;
    tester.out() << "sum(s1.all() - s2.all()) = " << sum(s1.all() - s2.all())
		 << " ; should = -sum(s2(zeroToOneByOne)) = "
		 << -sum(s2(zeroToOneByOne)) << std::endl;
  }
 
  // (Loc<Dim> - Interval<Dim>):
  s1.all() = 1.0;
  s2.all() = s1.all();
  s1(zeroToOneByOne) = 2.0;
  const Loc<Dim> threes(3);
  s1(zeroToOneByOne) = s1(threes - zeroToOneByOne);
  tester.out() << "Test 8" << std::endl;
  if (!tester.check(sum(s1) == sum(s2))) {
    tester.out() << "...failed" << std::endl;
    tester.out() << "threes = " << threes << " ; zeroToOneByOne = "
		 << zeroToOneByOne << std::endl;
    tester.out() << "threes - zeroToOneByOne = " << threes - zeroToOneByOne
		 << std::endl;
    tester.out() << "sum(s1) = " << sum(s1) << " ; should be = "
		 << "sum(s2) = " << sum(s2) << std::endl;
  }

  tester.out() << "------------------------------------------------"
               << std::endl;
  int retval = tester.results("TestFieldIndexing");
  Pooma::finalize();
  return retval;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: TestFieldIndexing.cpp,v $   $Author: sa_smith $
// $Revision: 1.14 $   $Date: 2000/07/05 20:20:43 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
