// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Classes:
// 
// Average
//-----------------------------------------------------------------------------

#ifndef POOMA_FIELD_DIFFOPS_AVERAGE_LRM_H
#define POOMA_FIELD_DIFFOPS_AVERAGE_LRM_H

//////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// Overview: 
//
// Classes:
//
// Average  : Averaging operator on Fields. See
//            Average.h for details, and the average() function template
//            definition.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Field/DiffOps/Average.h"
#include "Geometry/DiscreteGeometry.h"

//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
//
// Full Description:
// 
// Classes:
//
// Average:
// 
// Partial specializations of the generic Average (averaging) functor. See
// Average.h for general comments.  These are for RectilinearMesh-based
// DiscreteGeometry.
// 
// Average is a functor class serving as the "Functor" template parameter for
// FieldStencil<Functor>. Its operator() functions implement a 
// discrete averaging operator.  
// The types of the input and output Field differ in that the input centering 
// is different than the output.
// Partial specializations implement various combinations of input and output
// centerings, for specific coordinate systems.
// 
// Exported typedefs:
//
// OutputCentering_t - Centering parameter of the output DiscreteGeometry class
// OutputElement_t   - Type of the elements in the output ConstField; 
//                     restricted to a scalar type (vector input) or vector
//                     (tensor input)
// 
// Accessors:
//
// lowerExtent(int d) - Returns the stencil width in direction d, at the "low"
//                      end of the (logically) rectilinear mesh. This is the
//                      maximim positive integer offset from the element 
//                      indexed by integer i in the input Field's index space
//                      along dimension d used in outputting the element
//                      indexed by integer i in the output Field's index space
//                      along dimension d
// upperExtent(int d) - Same as lowerExtent(), but for the "high" end of the 
//                      mesh. That is, the maximum (magnitude) *negative*
//                      offset from i in direction d.
// 
// Other methods:
// 
// operator()(...) - The actual implementation for the stencil. This acts on a
//              set of scalar-indexed values in the input Field's index space
//              making up the stencil, as offset from the fixed index point
//              specified by the function's input arguments (list of scalar
//              index values).  The stencil must be written so that the same
//              fixed index point specified by the input arguments where the
//              values are to be assigned in the index space of the output
//              Field. This means, for example, that if the operator is going
//              from one centering to a different output centering, the index
//              bookkeeping must be done correctly by this operator()()
//              function implementation.
//-----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
// Partial specializations of Average and WeightedAverage
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
// Average Vert -> Cell
// ----------------------------------------------------------------------------

template<class T, class Mesh>
class Average<Cell, DiscreteGeometry<Vert, Mesh>, T, true>
{
public:

  typedef Cell OutputCentering_t;
  typedef T OutputElement_t;

  //
  // Methods.
  //

  int lowerExtent(int d) const { return 0; }
  int upperExtent(int d) const { return 1; }
      
  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1) const
  {
    return 0.5*(f(i1 + 1) + f(i1));
  }

  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1, int i2) const
  {
    return 0.25*(f(i1 + 1, i2 + 1) + f(i1    , i2 + 1) + 
		 f(i1 + 1, i2    ) + f(i1    , i2    ));
  }

  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1, int i2, int i3) const
  {
    return 0.125*(f(i1 + 1, i2 + 1, i3 + 1) + f(i1    , i2 + 1, i3 + 1) + 
		  f(i1 + 1, i2    , i3 + 1) + f(i1    , i2    , i3 + 1) +
		  f(i1 + 1, i2 + 1, i3    ) + f(i1    , i2 + 1, i3    ) +
		  f(i1 + 1, i2    , i3    ) + f(i1    , i2    , i3    ));
  }
};

// ----------------------------------------------------------------------------
// Average Cell -> Vert
// ----------------------------------------------------------------------------

template<class T, class Mesh>
class Average<Vert, DiscreteGeometry<Cell, Mesh>, T, true>
{
public:

  typedef Vert OutputCentering_t;
  typedef T OutputElement_t;

  //
  // Methods.
  //

  int lowerExtent(int) const { return 1; }
  int upperExtent(int) const { return 0; }
      
  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1) const
  {
    return 0.5*(f(i1 - 1) + f(i1));
  }

  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1, int i2) const
  {
    return 0.25*(f(i1 - 1, i2 - 1) + f(i1    , i2 - 1) + 
		 f(i1 - 1, i2    ) + f(i1    , i2    ));
  }

  template<class F>
  inline OutputElement_t
  operator()(const F &f, int i1, int i2, int i3) const
  {
    return 0.125*(f(i1 - 1, i2 - 1, i3 - 1) + f(i1    , i2 - 1, i3 - 1) + 
		  f(i1 - 1, i2    , i3 - 1) + f(i1    , i2    , i3 - 1) +
		  f(i1 - 1, i2 - 1, i3    ) + f(i1    , i2 - 1, i3    ) +
		  f(i1 - 1, i2    , i3    ) + f(i1    , i2    , i3    ));
  }

};

// ----------------------------------------------------------------------------
// WeightedAverage Vert -> Cell
// ----------------------------------------------------------------------------

template<class T, class Mesh, class TW>
class WeightedAverage<Cell, DiscreteGeometry<Vert, Mesh>, T, TW, true>
{
public:

  typedef Cell OutputCentering_t;
  typedef typename BinaryReturn<T,TW,OpMultiply>::Type_t OutputElement_t;

  //
  // Methods.
  //

  int lowerExtent(int d) const { return 0; }
  int upperExtent(int d) const { return 1; }
      
  template<class F, class W>
  inline OutputElement_t
  operator()(const F &f, const W &w, int i1) const
  {
    return (f(i1 + 1)*w(i1 + 1) + f(i1)*w(i1))/(w(i1 + 1) + w(i1));
  }

  template<class F, class W>
  inline OutputElement_t
  operator()(const F &f, const W &w, int i1, int i2) const
  {
    return ((f(i1 + 1, i2 + 1)*w(i1 + 1, i2 + 1) + 
	     f(i1    , i2 + 1)*w(i1    , i2 + 1) + 
	     f(i1 + 1, i2    )*w(i1 + 1, i2    ) + 
	     f(i1    , i2    )*w(i1    , i2    )) / 
	    (w(i1 + 1, i2 + 1) + w(i1    , i2 + 1) + 
	     w(i1 + 1, i2    ) + w(i1    , i2    )));
  }

  template<class F, class W>
  inline OutputElement_t
  operator()(const F &f, const W &w, int i1, int i2, int i3) const
  {
    return ((f(i1 + 1, i2 + 1, i3 + 1)*w(i1 + 1, i2 + 1, i3 + 1) + 
	     f(i1    , i2 + 1, i3 + 1)*w(i1    , i2 + 1, i3 + 1) + 
	     f(i1 + 1, i2    , i3 + 1)*w(i1 + 1, i2    , i3 + 1) + 
	     f(i1    , i2    , i3 + 1)*w(i1    , i2    , i3 + 1) +
	     f(i1 + 1, i2 + 1, i3    )*w(i1 + 1, i2 + 1, i3    ) + 
	     f(i1    , i2 + 1, i3    )*w(i1    , i2 + 1, i3    ) +
	     f(i1 + 1, i2    , i3    )*w(i1 + 1, i2    , i3    ) + 
	     f(i1    , i2    , i3    )*w(i1    , i2    , i3    )) / 
	    (w(i1 + 1, i2 + 1, i3 + 1) + w(i1    , i2 + 1, i3 + 1) + 
	     w(i1 + 1, i2    , i3 + 1) + w(i1    , i2    , i3 + 1) +
	     w(i1 + 1, i2 + 1, i3    ) + w(i1    , i2 + 1, i3    ) +
	     w(i1 + 1, i2    , i3    ) + w(i1    , i2    , i3    )));
  }
};

// ----------------------------------------------------------------------------
// WeightedAverage Cell -> Vert
// ----------------------------------------------------------------------------

template<class T, class Mesh, class TW>
class WeightedAverage<Vert, DiscreteGeometry<Cell, Mesh>, T, TW, true> 
{
public:

  typedef Vert OutputCentering_t;
  typedef typename BinaryReturn<T,TW,OpMultiply>::Type_t OutputElement_t;

  //
  // Methods.
  //

  int lowerExtent(int d) const { return 1; }
  int upperExtent(int d) const { return 0; }
      
  template<class F, class W>
  inline OutputElement_t
  operator()(const F &f, const W &w, int i1) const
  {
    return 0.5*(f(i1 - 1) + f(i1));
  }

  template<class F, class W>
  inline OutputElement_t
  operator()(const F &f, const W &w, int i1, int i2) const
  {
    return 0.25*(f(i1 - 1, i2 - 1) + f(i1    , i2 - 1) + 
		 f(i1 - 1, i2    ) + f(i1    , i2    ));
  }

  template<class F, class W>
  inline OutputElement_t
  operator()(const F &f, const W &w, int i1, int i2, int i3) const
  {
    return 0.125*(f(i1 - 1, i2 - 1, i3 - 1) + f(i1    , i2 - 1, i3 - 1) + 
		  f(i1 - 1, i2    , i3 - 1) + f(i1    , i2    , i3 - 1) +
		  f(i1 - 1, i2 - 1, i3    ) + f(i1    , i2 - 1, i3    ) +
		  f(i1 - 1, i2    , i3    ) + f(i1    , i2    , i3    ));
  }

};

#endif     // POOMA_FIELD_DIFFOPS_AVERAGE_LRM_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Average.LRM.h,v $   $Author: swhaney $
// $Revision: 1.11 $   $Date: 2000/03/07 13:17:09 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
