// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Class:
//   CombineDomainOpt
//-----------------------------------------------------------------------------

#ifndef POOMA_DOMAIN_COMBINEDOMAINOPT_H
#define POOMA_DOMAIN_COMBINEDOMAINOPT_H

//////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// Overview:
//
// CombineDomainOpt is a class that can be used to optimize the operation
// NewDomainN<>::combineSlice(domain, s1, s2, ...).
//
// Typically NewDomain is used by arrays to construct a view domain that
// could be a slice, so typically you would call
//
// NewDomainN<>::combineSlice(a.totalDomain(), s1, s2);
//
// If the result is single-valued, the domain of a is not used, but the
// function call a.totalDomain() may be hard to optimize away.  To avoid this
// function call you can now say:
//
// typedef NewDomainN<...> NewDomain_t;
// typedef typename NewDomain_t::SliceType_t SliceDomain_t;
//
// SliceDomain_t s(
//     CombineDomainOpt<NewDomain_t, SliceDomain_t::singleValued>::
//     make(a, s1, s2, ...)
//                );
//
// If s is single-valued, the array a is never used.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

template<class NewDomain, bool sv>
struct CombineDomainOpt;

// Single valued specialization.
// Since Locs can construct themselves from other Locs and
// integers, we just use the constructors.  Another option is
// return Loc<dim>(s1, s2, ...).

template<class NewDomain>
struct CombineDomainOpt<NewDomain, true>
{
  typedef typename NewDomain::SliceType_t Type_t;

  template<class Array, class Sub1>
  inline static
  Type_t make(const Array &, const Sub1 &s1)
  {
    return Type_t(s1);
  }

  template<class Array, class Sub1, class Sub2>
  inline static
  Type_t make(const Array &, const Sub1 &s1, const Sub2 &s2)
  {
    return Type_t(s1, s2);
  }

  template<class Array, class Sub1, class Sub2, class Sub3>
  inline static
  Type_t make(const Array &,
	      const Sub1 &s1, const Sub2 &s2, const Sub3 &s3)
  {
    return Type_t(s1, s2, s3);
  }

  template<class Array, class Sub1, class Sub2, class Sub3,
    class Sub4>
  inline static
  Type_t make(const Array &,
	      const Sub1 &s1, const Sub2 &s2, const Sub3 &s3,
	      const Sub4 &s4)
  {
    return Type_t(s1, s2, s3, s4);
  }

  template<class Array, class Sub1, class Sub2, class Sub3,
    class Sub4, class Sub5>
  inline static
  Type_t make(const Array &,
	      const Sub1 &s1, const Sub2 &s2, const Sub3 &s3,
	      const Sub4 &s4, const Sub5 &s5)
  {
    return Type_t(s1, s2, s3, s4, s5);
  }

  template<class Array, class Sub1, class Sub2, class Sub3,
    class Sub4, class Sub5, class Sub6>
  inline static
  Type_t make(const Array &,
	      const Sub1 &s1, const Sub2 &s2, const Sub3 &s3,
	      const Sub4 &s4, const Sub5 &s5, const Sub6 &s6)
  {
    return Type_t(s1, s2, s3, s4, s5, s6);
  }

  template<class Array, class Sub1, class Sub2, class Sub3,
    class Sub4, class Sub5, class Sub6, class Sub7>
  inline static
  Type_t make(const Array &,
	      const Sub1 &s1, const Sub2 &s2, const Sub3 &s3,
	      const Sub4 &s4, const Sub5 &s5, const Sub6 &s6,
	      const Sub7 &s7)
  {
    return Type_t(s1, s2, s3, s4, s5, s6, s7);
  }
};

// Multi-valued version.  This one calls combineSlice to create
// the final domain.

template<class NewDomain>
struct CombineDomainOpt<NewDomain, false>
{
  typedef typename NewDomain::SliceType_t Type_t;

  template<class Array, class Sub1>
  inline static
  Type_t make(const Array &a, const Sub1 &s1)
  {
    return NewDomain::combineSlice(a.totalDomain(), s1);
  }

  template<class Array, class Sub1, class Sub2>
  inline static
  Type_t make(const Array &a, const Sub1 &s1, const Sub2 &s2)
  {
    return NewDomain::combineSlice(a.totalDomain(), s1, s2);
  }

  template<class Array, class Sub1, class Sub2, class Sub3>
  inline static
  Type_t make(const Array &a,
	      const Sub1 &s1, const Sub2 &s2, const Sub3 &s3)
  {
    return NewDomain::combineSlice(a.totalDomain(), s1, s2, s3);
  }

  template<class Array, class Sub1, class Sub2, class Sub3,
    class Sub4>
  inline static
  Type_t make(const Array &a,
	      const Sub1 &s1, const Sub2 &s2, const Sub3 &s3,
	      const Sub4 &s4)
  {
    return NewDomain::combineSlice(a.totalDomain(), s1, s2, s3, s4);
  }

  template<class Array, class Sub1, class Sub2, class Sub3,
    class Sub4, class Sub5>
  inline static
  Type_t make(const Array &a,
	      const Sub1 &s1, const Sub2 &s2, const Sub3 &s3,
	      const Sub4 &s4, const Sub5 &s5)
  {
    return NewDomain::combineSlice(a.totalDomain(), s1, s2, s3, s4, s5);
  }

  template<class Array, class Sub1, class Sub2, class Sub3,
    class Sub4, class Sub5, class Sub6>
  inline static
  Type_t make(const Array &a,
	      const Sub1 &s1, const Sub2 &s2, const Sub3 &s3,
	      const Sub4 &s4, const Sub5 &s5, const Sub6 &s6)
  {
    return NewDomain::combineSlice(a.totalDomain(), 
      s1, s2, s3, s4, s5, s6);
  }

  template<class Array, class Sub1, class Sub2, class Sub3,
    class Sub4, class Sub5, class Sub6, class Sub7>
  inline static
  Type_t make(const Array &a,
	      const Sub1 &s1, const Sub2 &s2, const Sub3 &s3,
	      const Sub4 &s4, const Sub5 &s5, const Sub6 &s6,
	      const Sub7 &s7)
  {
    return NewDomain::combineSlice(a.totalDomain(), 
      s1, s2, s3, s4, s5, s6, s7);
  }
};

#endif     // POOMA_DOMAIN_COMBINEDOMAINOPT_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: CombineDomainOpt.h,v $   $Author: swhaney $
// $Revision: 1.3 $   $Date: 2000/03/07 13:16:33 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
