// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

#ifndef POOMA_UTILITIES_ELEMENTPROPERTIES_H
#define POOMA_UTILITIES_ELEMENTPROPERTIES_H

//-----------------------------------------------------------------------------
// Classes:
//   ElementProperties<T> and specializations
//   TrivialElementProperties<T>
//-----------------------------------------------------------------------------

///////////////////////////////////////////////////////////////////////////////
// namespace POOMA {

//-----------------------------------------------------------------------------
// Overview:
//
// ElementProperties<T>
//   Traits class for determining, and possibly modifying, the
//   construction and destruction properties of elements of type T.
// TrivialElementProperties<T>
//  - ElementProperties for "trivial" data types.
// ElementProperties<bool>
// ElementProperties<char>
// ElementProperties<int>
// ElementProperties<float>
// ElementProperties<double>
//  - specializations for basic C++ data types.
//-----------------------------------------------------------------------------



//-----------------------------------------------------------------------------
//
// Full Description:
// ElementProperties<T>
//
// Traits class for determining, and possibly modifying, the
// construction and destruction properties of elements of type T.
//
// In detail, this class serves several purposes:
//
//   First it allows RefCountedBlockPtr to optimize away the
//   constructor and destructor calls for classes with "trivial"
//   default constructors and destructors (e.g. the native C data
//   types, but see below for details).
//
//   Second, it allows specializations to provide special construct,
//   clone, and destruct methods that override the default behavior.
//   The primary reason for this capability is to allow RefCountedPtr
//   and RefCountedBlockPtr to store deep copies of (and potentially
//   make further deep copies of) objects that have shallow copy
//   semantics. This can be done by specializing ElementProperties to
//   provide construct and clone methods that make deep copies of the
//   model object.
//
//   Finally, one might want RefCountedPtr<T> to point to an object
//   that inherits from T. In such situations, asking the
//   RefCountedPtr to make a deep copy of its pointee would, with the
//   default behavior, cause the object to be sliced (as T's copy
//   constructor would be used). If T has a "virtual constructor" (a
//   virtual clone method), then one can specialize ElementProperties'
//   clone() method to call the virtual constructor and make the
//   proper copy.
//
// The first capability is provided by defining the two bool fields:
//
//   static const bool hasTrivialDefaultConstructor;
//   static const bool hasTrivialDestructor;
//
// hasTrivialDefaultConstructor is true for data types whose default
// constructors have the same semantics as the native C data types;
// i.e. they do nothing: no allocations, no default values, etc.
// Normally RefCountedBlockPtr calls placement operator new to
// initizlize objects in the space that it allocates and manages.
// However, this is unnecessary overhead for types whose default
// constructor does nothing. Thus if hasTrivialDefaultConstructor is
// true, RefCountedBlockPtr will leave memory uninitialized in the
// default case.
//
// Versions of ElementProperties for the most common C data types are
// defined below. Similar specializations might also be useful for
// other statically sized data types, such as TinyArrays. (Note that
// one could optionally define the specializations for native data
// types to initialize memory with some obviously corrupt value in
// order to help track initialization problems during debugging,
// although purify is probably a better tool for such investigations.)
//
// Similarly, hasTrivialDestructor == true causes RefCountedBlockPtr
// to skip the explicit destructor calls that are normally necessary
// when destroying an object created with placement new.  
// This will almost always have the same value as
// hasTrivialDefaultConstructor, but the additional flexibility
// carries no additional cost so it was included.
//
// The class must also define the following static functions:
//
//   inline static void construct(T * addr, const T & model)
//
//   inline static void T * clone(const T & model);
//
//   inline static void construct(T * addr)
//
//   inline static void destruct(T *addr)
//
// If the "trivial" flags are true, then the last two functions will
// never be called, but they must be defined or the compiler will
// complain. In these cases it is best to define the functions to
// throw an exception (see below).
//    
// The non-specialized ElementProperties<T> class defines both flags
// to be false.  It defines the construct methods to use the default
// and copy constructors with placement new, respectively, under the
// assumption that these will make deep copies. Finally, it defines
// the destruct method to explicitly invoke the destructor on the
// object.
//
//-----------------------------------------------------------------------------

template <class T>
struct ElementProperties
{ 
  //---------------------------------------------------------------------------
  // Convenience typedef (not needed here, but useful for
  // specializations, which may want to copy parts of this definition)

  typedef T This_t;

  //---------------------------------------------------------------------------
  // By default, we assume that the default constructor and the
  // destructor do something.

  static const bool hasTrivialDefaultConstructor = false;

  static const bool hasTrivialDestructor = false;

  //---------------------------------------------------------------------------
  // We specialize this struct for concrete types. These are types that
  // have no pointers, etc., so that their data can be copied with routines
  // such as std::copy or std::memcpy.

  static const bool concrete = false;

  //---------------------------------------------------------------------------
  // Since the above are false, the code will use placement new for
  // initialization, and thus must explicitly call the destructor. The
  // following are the default methods for doing these things.

  static void construct(This_t * addr)
  {
    new (addr) This_t();
  }

  static void construct(This_t * addr, const This_t & model)
  {
    new (addr) This_t(model);
  }

  static This_t * clone(const This_t &model)
  {
    return new This_t(model);
  }

  static void destruct(This_t * addr)
  {
    addr->~This_t();
  }
};


//-----------------------------------------------------------------------------
// Concrete types that have trivial default construction and destruction
// semantics can just inherit from this:

template <class T>
struct TrivialElementProperties
{
  typedef T This_t;

  static const bool hasTrivialDefaultConstructor = true;

  static const bool hasTrivialDestructor = true;

  static const bool concrete = true;
  
  static void construct(This_t * addr, const This_t & model)
  {
    new (addr) This_t(model);
  }

  static This_t * clone(const This_t &model)
  {
    return new This_t(model);
  }

  static void construct(This_t *addr)
  {
    new (addr) This_t();
  }

  static void destruct(This_t *)
  {
    PInsist(0,"TrivialElementProperties<T>::destruct(addr) not allowed!");
  }
};


//-----------------------------------------------------------------------------
// Classes that have shallow copy semantics and "makeOwnCopy" methods
// can specialize ElementProperties<T> by simply inheriting from this.

template <class T>
struct MakeOwnCopyProperties
{
  typedef T This_t;

  static const bool hasTrivialDefaultConstructor = false;

  static const bool hasTrivialDestructor = false;

  static const bool concrete = false;

  static void construct(This_t * addr)
  {
    new (addr) This_t;
    addr->makeOwnCopy();
  }

  static void construct(This_t * addr, const This_t & model)
  {
    new (addr) This_t(model);
    addr->makeOwnCopy();
  }

  static This_t * clone(const This_t &model)
  {
    This_t * temp = new This_t(model);
    temp->makeOwnCopy();
    return temp;
  }

  static void destruct(This_t * addr)
  {
    addr->~This_t();
  }

};


//-----------------------------------------------------------------------------
// Specializations for standard C++ types, which do not do initialization
// in their "default constructors".

template <>
struct ElementProperties<bool>   : public TrivialElementProperties<bool>
{ };

template <>
struct ElementProperties<char>   : public TrivialElementProperties<char>
{ };

template <>
struct ElementProperties<unsigned char>   
  : public TrivialElementProperties<unsigned char>
{ };

template <>
struct ElementProperties<short>  : public TrivialElementProperties<short>
{ };

template <>
struct ElementProperties<unsigned short>
  : public TrivialElementProperties<unsigned short>
{ };

template <>
struct ElementProperties<int>    : public TrivialElementProperties<int>
{ };

template <>
struct ElementProperties<unsigned int>
  : public TrivialElementProperties<unsigned int>
{ };

template <>
struct ElementProperties<long>   : public TrivialElementProperties<long>
{ };

template <>
struct ElementProperties<unsigned long>
  : public TrivialElementProperties<unsigned long>
{ };

template <>
struct ElementProperties<float>  : public TrivialElementProperties<float>
{ };

template <>
struct ElementProperties<double> : public TrivialElementProperties<double>
{ };


// } // namespace POOMA
///////////////////////////////////////////////////////////////////////////////

#endif // POOMA_UTILITIES_ELEMENTPROPERTIES_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: ElementProperties.h,v $   $Author: julianc $
// $Revision: 1.13 $   $Date: 2000/04/27 18:30:38 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
