// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license
//-----------------------------------------------------------------------------
// Class: TinyMatrix, Full, TinyMatrixEngine<D,T,Full>
//-----------------------------------------------------------------------------

#ifndef POOMA_TINY_TINYMATRIX_H
#define POOMA_TINY_TINYMATRIX_H

//-----------------------------------------------------------------------------
// Overview:
// An interface class for an N-dimensional TinyMatrix of numeric objects,
// and an engine class for defining a general TinyMatrix.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

class Full;

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Utilities/PAssert.h"
#include "Utilities/ElementProperties.h"
#include "PETE/PETE.h"
#include "Pooma/PoomaOperatorTags.h"
#include "Domain/Loc.h"
#include "Tiny/TinyMatrixEngine.h"
#include "Tiny/TinyMatrixElements.h"
#include "Tiny/TinyMatrixOperators.h"
#include <iosfwd>

//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------

template<int D1, int D2, class T, class E> class TinyMatrix;

//-----------------------------------------------------------------------------
//
// Full Description:
//
// TinyMatrix is an interface class that takes three template parameters:
//
//   int D1, int D2: The number of components in each rank of the TinyMatrix.
//   class T: The type of the components.  
//   class EngineTag: A policy parameter for the storage type.
//
//-----------------------------------------------------------------------------

#if !POOMA_NO_DEPENDENT_TEMPLATE_ARGS
template<int D1, int D2=D1, class T=double, class EngineTag=Full>
#else
template<int D1, int D2, class T=double, class EngineTag=Full>
#endif
class TinyMatrix
{
public:

  //----------------------------------------------------------------------
  // Typedefs
  //----------------------------------------------------------------------

  // Export the input types.
  enum { dimensions=2 };
  enum { d1=D1, d2=D2 };
  typedef T Element_t;
  typedef EngineTag EngineTag_t;

  // Deduce the engine type from the tamplate parameters.
  typedef TinyMatrixEngine<D1,D2,T,EngineTag> Engine_t;

  // Return types for accessor functions.
  typedef typename Engine_t::ElementRef_t       ElementRef_t;
  typedef typename Engine_t::ConstElementRef_t  ConstElementRef_t;

  // Record the type of the current class.
  typedef TinyMatrix<D1,D2,T,EngineTag> This_t;


  //----------------------------------------------------------------------
  // Constructors and Destructor

  // Null ctor uses the engine's null ctor.
  TinyMatrix() {}

  // Copy ctor is deep.
  TinyMatrix(const This_t& x) : engine_m(x.engine_m) {}

  // Construct from an arbitrary single object.
  // The object must be indexable using TinyMatrixElem.
  template<class X>
    explicit TinyMatrix(const X& x) : engine_m(x) {}

  // Construct from two, three, ..., nine objects.
  template<class X1, class X2>
    TinyMatrix(const X1& x, const X2& y) : engine_m(x,y) {}
  template<class X1, class X2, class X3>
    TinyMatrix(const X1& x, const X2& y, const X3& z) : engine_m(x,y,z) {}
  template<class X1, class X2, class X3, class X4>
    TinyMatrix(const X1& x1, const X2& x2, const X3& x3, const X4& x4) 
    : engine_m(x1,x2,x3,x4) {}
  template<class X1, class X2, class X3, class X4, class X5>
    TinyMatrix(const X1& x1, const X2& x2, const X3& x3, const X4& x4,
           const X5& x5) 
    : engine_m(x1,x2,x3,x4,x5) {}
  template<class X1, class X2, class X3, class X4, class X5, class X6>
    TinyMatrix(const X1& x1, const X2& x2, const X3& x3, const X4& x4,
           const X5& x5, const X6& x6) 
    : engine_m(x1,x2,x3,x4,x5,x6) {}
  template<class X1, class X2, class X3, class X4, class X5, class X6,
           class X7>
    TinyMatrix(const X1& x1, const X2& x2, const X3& x3, const X4& x4,
           const X5& x5, const X6& x6, const X7& x7) 
    : engine_m(x1,x2,x3,x4,x5,x6,x7) {}
  template<class X1, class X2, class X3, class X4, class X5, class X6,
           class X7, class X8>
    TinyMatrix(const X1& x1, const X2& x2, const X3& x3, const X4& x4,
           const X5& x5, const X6& x6, const X7& x7, const X8& x8) 
    : engine_m(x1,x2,x3,x4,x5,x6,x7,x8) {}
  template<class X1, class X2, class X3, class X4, class X5, class X6,
           class X7, class X8, class X9>
    TinyMatrix(const X1& x1, const X2& x2, const X3& x3, const X4& x4,
           const X5& x5, const X6& x6, const X7& x7, const X8& x8, const X9& x9) 
    : engine_m(x1,x2,x3,x4,x5,x6,x7,x8,x9) {}

  // Let the engine destroy itself.
  ~TinyMatrix() {}


  //----------------------------------------------------------------------
  // Assignment

  // Assign from the same kind of TinyMatrix.
  This_t& operator=(const This_t& x) 
    { 
      if ( this != &x )
        engine() = x.engine();
      return *this;
    }

  // Assign from an arbitrary type.
  template<class V>
    This_t& operator=(const V& x)
        {
          engine() = x;
          return *this;
        }

  //----------------------------------------------------------------------
  // Element access

  ConstElementRef_t operator()(int i,int j) const 
    {
#if POOMA_BOUNDS_CHECK
      PInsist4((i >= 0) && (i < D1) && (j >= 0) && (j < D2), 
        "TinyMatrix Bounds Violation:"
        "indices = (%d,%d), TinyMatrix size = %d x %d.",
        i, j, D1, D2);
#endif
      return engine()(i,j); 
    }
  ElementRef_t operator()(int i,int j)
    {
#if POOMA_BOUNDS_CHECK
      PInsist4((i >= 0) && (i < D1) && (j >= 0) && (j < D2), 
        "TinyMatrix Bounds Violation:"
        "indices = (%d,%d), TinyMatrix size = %d x %d.",
        i, j, D1, D2);
#endif
      return engine()(i,j); 
    }

  ConstElementRef_t  operator()(int i) const { return engine()(i); }
  ElementRef_t       operator()(int i)       { return engine()(i); }


  // An accessor to get the engine.
  const Engine_t& engine() const { return engine_m; }
  Engine_t& engine() { return engine_m; }


  // Output to a stream.
  // The format is: ((t(0,0) t(1,0),... ) (t(0,1) t(1,1) ... ) ... ))

  template<class Out>
  void print(Out &out) const
  {
    // Maintain the input formatting state through the multiple output
    // statements following:
    std::ios::fmtflags incomingFormatFlags = out.flags();
    long width = out.width();
    long precision = out.precision();
    out.width(0);
    out << "(";
    for (int i = 0; i < D2; i++) {
      out << "(";
      out.flags(incomingFormatFlags);
      out.width(width);
      out.precision(precision);
      out << (*this)(0, i);
      for (int j = 1; j < D1; j++) {
        out << " ";
        out.flags(incomingFormatFlags);
        out.width(width);
        out.precision(precision);
        out << (*this)(j, i);
      }
      out << ")";
    }
    out << ")";
  }

private:

  // The only data is the engine itself.
  Engine_t engine_m;

};


//-----------------------------------------------------------------------------
// Output to a stream.
// The format is: ( ( t(0,0) t(1,0),... ) ( t(0,1) t(1,1) ... ) ... )
//-----------------------------------------------------------------------------

template<int D1, int D2, class T, class E>
std::ostream &operator<<(std::ostream &out, const TinyMatrix<D1,D2,T,E> &t)
{
  t.print(out);
  return out;
}


//-----------------------------------------------------------------------------
// Specialization of ElementProperties struct for TinyMatrix.
//-----------------------------------------------------------------------------

template <int D1, int D2, class T, class E>
struct ElementProperties< TinyMatrix<D1,D2,T,E> > 
  : public TrivialElementProperties< TinyMatrix<D1,D2,T,E> >
{ };


//-----------------------------------------------------------------------------
// Definitions for a Full TinyMatrix.
//-----------------------------------------------------------------------------

template<int D1, int D2, class T>
class TinyMatrixEngine<D1,D2,T,Full>
{
public:

  //----------------------------------------------------------------------
  // Typedefs
  //----------------------------------------------------------------------

  // Export the input types.
  enum { dimensions=2 };
  enum { d1=D1, d2=D2 };
  typedef T Element_t;
  typedef Full EngineTag_t;

  // Return types for accessor functions.
  typedef       T&  ElementRef_t;
  typedef const T&  ConstElementRef_t;

  // Record the type of the current class.
  typedef TinyMatrixEngine<D1,D2,T,Full> This_t;


  //----------------------------------------------------------------------
  // Constructors and Destructor

  // Null ctor takes no action.
  TinyMatrixEngine() {}

  // Copy ctor is deep.
  TinyMatrixEngine(const TinyMatrixEngine<D1,D2,T,Full>& x)
    {
      TinyMatrixAssign<This_t,This_t,OpAssign,0,D1,0,D2>
	::apply(*this,x,OpAssign());
    }

  // Construct from an argument of arbitrary type.
  // The arg must be indexable using TinyMatrixElem.
  template<class X>
    TinyMatrixEngine(const X& x)
      {
        TinyMatrixAssign<This_t,X,OpAssign,0,D1,0,D2>::apply(*this,x,OpAssign());
      }

  // Construct from two, three, ... nine objects.
  // Just fill in the 2D array using them.
  template<class X1, class X2>
    TinyMatrixEngine(const X1& x, const X2& y)
      {
        CTAssert( D1*D2 == 2 );
        (*this)(0) = x1;
        (*this)(1) = x2;
      }
  template<class X1, class X2, class X3>
    TinyMatrixEngine(const X1& x, const X2& y, const X3& z)
      {
        CTAssert( D1*D2 == 3 );
        (*this)(0) = x1;
        (*this)(1) = x2;
        (*this)(2) = x3;
      }

  template<class X1, class X2, class X3, class X4>
    TinyMatrixEngine(const X1& x1, const X2& x2, const X3& x3, const X4& x4) 
      {
        CTAssert( D1*D2 == 4 );
        (*this)(0) = x1;
        (*this)(1) = x2;
        (*this)(2) = x3;
        (*this)(3) = x4;
      }
    
  template<class X1, class X2, class X3, class X4, class X5>
    TinyMatrixEngine(const X1& x1, const X2& x2, const X3& x3, const X4& x4,
           const X5& x5) 
      {
        CTAssert( D1*D2 == 5 );
        (*this)(0) = x1;
        (*this)(1) = x2;
        (*this)(2) = x3;
        (*this)(3) = x4;
        (*this)(4) = x5;
      }
    
  template<class X1, class X2, class X3, class X4, class X5, class X6>
    TinyMatrixEngine(const X1& x1, const X2& x2, const X3& x3, const X4& x4,
           const X5& x5, const X6& x6) 
      {
        CTAssert( D1*D2 == 6 );
        (*this)(0) = x1;
        (*this)(1) = x2;
        (*this)(2) = x3;
        (*this)(3) = x4;
        (*this)(4) = x5;
        (*this)(5) = x6;
      }
    
  template<class X1, class X2, class X3, class X4, class X5, class X6,
           class X7>
    TinyMatrixEngine(const X1& x1, const X2& x2, const X3& x3, const X4& x4,
           const X5& x5, const X6& x6, const X7& x7) 
      {
        CTAssert( D1*D2 == 7 );
        (*this)(0) = x1;
        (*this)(1) = x2;
        (*this)(2) = x3;
        (*this)(3) = x4;
        (*this)(4) = x5;
        (*this)(5) = x6;
        (*this)(6) = x7;
      }
    
  template<class X1, class X2, class X3, class X4, class X5, class X6,
           class X7, class X8>
    TinyMatrixEngine(const X1& x1, const X2& x2, const X3& x3, const X4& x4,
           const X5& x5, const X6& x6, const X7& x7, const X8& x8) 
      {
        CTAssert( D1*D2 == 8 );
        (*this)(0) = x1;
        (*this)(1) = x2;
        (*this)(2) = x3;
        (*this)(3) = x4;
        (*this)(4) = x5;
        (*this)(5) = x6;
        (*this)(6) = x7;
        (*this)(7) = x8;
      }
    
  template<class X1, class X2, class X3, class X4, class X5, class X6,
           class X7, class X8, class X9>
    TinyMatrixEngine(const X1& x1, const X2& x2, const X3& x3, const X4& x4,
           const X5& x5, const X6& x6, const X7& x7, const X8& x8, const X9& x9) 
      {
        CTAssert( D1*D2 == 9 );
        (*this)(0) = x1;
        (*this)(1) = x2;
        (*this)(2) = x3;
        (*this)(3) = x4;
        (*this)(4) = x5;
        (*this)(5) = x6;
        (*this)(6) = x7;
        (*this)(7) = x8;
        (*this)(8) = x9;
      }
    
  // Let the engine destroy itself.
  ~TinyMatrixEngine() {}


  //----------------------------------------------------------------------
  // Assignment

  // Assign from the same kind of TinyMatrix.
  const This_t&
    operator=(const This_t& x)
      {
        if ( this != &x )
          TinyMatrixAssign<This_t,This_t,OpAssign,0,D1,0,D2>
	    ::apply(*this,x,OpAssign());
        return *this;
      }

  // Assign from an arbitrary type.
  template<class V>
    const This_t& 
      operator=(const V& x)
        {
          TinyMatrixAssign<This_t,V,OpAssign,0,D1,0,D2>::apply(*this,x,OpAssign());
          return *this;
        }

  //----------------------------------------------------------------------
  // Element access

  ConstElementRef_t  operator()(int i,int j) const
    {
      PAssert((i>=0)&&(i<D1));
      PAssert((j>=0)&&(j<D2));
      return x_m[i+D1*j]; 
    }
  ElementRef_t operator()(int i,int j)
    {
      PAssert((i>=0)&&(i<D1));
      PAssert((j>=0)&&(j<D2));
      return x_m[i+D1*j]; 
    }

  ConstElementRef_t  operator()(int i) const 
    {
      PAssert((i>=0)&&(i<D1*D2));
      return x_m[i]; 
    }
  ElementRef_t operator()(int i)
    {
      PAssert((i>=0)&&(i<D1*D2));
      return x_m[i]; 
    }

private:

  // The actual data is just an array of T's.
  // Store in fortran order.
  T x_m[D1*D2];
};


//-----------------------------------------------------------------------------
// ComponentAccess is an interface class that is used to provide an API for
// accessing components of a composite type. This version works with TinyMatrixs.
//-----------------------------------------------------------------------------

template<class T, class Components> struct ComponentAccess;

template<int D1, int D2, class T, class E, int N>
struct ComponentAccess< TinyMatrix<D1, D2, T, E>, Loc<N> >
{
  typedef TinyMatrix<D1, D2, T, E> V;
  typedef typename V::Element_t Element_t;
  typedef typename V::ElementRef_t ElementRef_t;
  
  static inline ElementRef_t indexRef(V &t, const Loc<N> &l)
    {
      CTAssert(N==2);
      return t(l[0].first(), l[1].first());
    }
  
  static inline Element_t index(const V &t, const Loc<N> &l)
    {
      CTAssert(N==2);
      return t(l[0].first(), l[1].first());
    }
};


//-----------------------------------------------------------------------------
// TinyMatrixElem specialization for Full TinyMatrix engines:
//-----------------------------------------------------------------------------

template<int D1, int D2, class T, int I, int J>
struct TinyMatrixElem< TinyMatrixEngine<D1,D2,T,Full> , I , J>
{
  typedef TinyMatrixEngine<D1,D2,T,Full> V;
  typedef TinyMatrixEngineElem<D1,D2,T,Full,I,J> TE;
  typedef typename TE::Element_t         Element_t;
  typedef typename TE::ConstElementRef_t ConstElementRef_t;
  typedef typename TE::ElementRef_t      ElementRef_t;
  static ConstElementRef_t get(const V& x) { return TE::get(x); }
  static      ElementRef_t get(V& x)       { return TE::get(x); }
};

#endif // POOMA_TINY_TINYMATRIX_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: TinyMatrix.h,v $   $Author: jac $
// $Revision: 1.10 $   $Date: 2000/04/28 05:48:54 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
