// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

#ifndef POOMA_CONTEXTMAPPER_H
#define POOMA_CONTEXTMAPPER_H

//-----------------------------------------------------------------------------
// Classes:
// ContextMapper
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
// Overview:
//
// ContextMapper is a base class used to map a node list to contexts
// and affinity. 
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Include Files
//-----------------------------------------------------------------------------

#include "Pooma/Pooma.h"
#include "Domain/Interval.h"
#include "Utilities/PAssert.h"
#include "Layout/Node.h"


///////////////////////////////////////////////////////////////////////////////
// namespace POOMA {

//-----------------------------------------------------------------------------
//
// Full Description:
//

//-----------------------------------------------------------------------------

template<int Dim>
class ContextMapper
{
public:

  //============================================================
  // Typedefs and enumerations
  //============================================================
  typedef Interval<Dim>                       Domain_t;
  typedef Node<Domain_t>                      Value_t;
  typedef std::vector<Value_t *>              List_t;

  //============================================================
  // Constructors
  //============================================================

  ContextMapper(){};

  virtual void map(const List_t & templist) const = 0;

  
 void setAffinity(const List_t & templist) const
  {
    int affinityMax = Smarts::concurrency();
    int idMax = 0;
	
    List_t::const_iterator start = templist.begin();
    List_t::const_iterator end = templist.end();

    for ( ; start != end ; ++start)
      if((*start)->context()==Pooma::context())
	{
	  (*start)->localID()=idMax;
	  ++idMax;
	}
	
    start = templist.begin();
    for ( ; start != end ; ++start)
      { 
	if((*start)->context()==Pooma::context())
	  (*start)->affinity() = static_cast<int>( affinityMax * 
                                 ( (*start)->localID() / 
                                 static_cast<double>(idMax) ) );
      }

    return;
  } 

};

template<int Dim>
class LocalMapper
  : public ContextMapper<Dim>
{ 
public:
  //============================================================
  // Typedefs and enumerations
  //============================================================
  typedef Interval<Dim>                       Domain_t;
  typedef Node<Domain_t>                      Value_t;
  typedef std::vector<Value_t *>              List_t;

  template<class Partitioner>
  LocalMapper(const Partitioner &)
  {}
  
  LocalMapper()
  {}
  
  void map(const List_t & templist) const
  {
    int idMax = templist.size();
    int naff = Smarts::concurrency();
    for (int i = 0; i< templist.size(); ++i)
      {
	templist[i]->context() = -1;
	templist[i]->localID() = i;
	templist[i]->affinity() = static_cast<int>( ( naff * ( i / 
                                  static_cast<double>(idMax) ) ) );
      }
  }



};


#endif     // POOMA_CONTEXTMAPPER_H
