// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Include files
//-----------------------------------------------------------------------------

#include "Utilities/PAssert.h"
#include <iostream>

//-----------------------------------------------------------------------------
// Forward declarations
//-----------------------------------------------------------------------------

template <class T, class E> class DynamicArray;


//-----------------------------------------------------------------------------
// Gather/Scatter global function definitions for 
// Particle-Field interpolation
//-----------------------------------------------------------------------------

// gather/scatter using particle position attribute

template <class PA, class FC, class PPos, class InterpolatorTag>
void gather(const PA& attrib, const FC& field, const PPos& pos,
	    const InterpolatorTag&)
{
  // check that dimensions of field and position attribute match
  typedef typename PPos::Element_t PPosElement_t;
  CTAssert(int(FC::dimensions) == int(PPosElement_t::d1));

  // ask Interpolator class to perform the gather
  typedef typename PPosElement_t::Element_t AxisType_t;
  typedef Interpolator<FC::dimensions,AxisType_t,InterpolatorTag> Interp_t;
  Interp_t::gather(attrib,field,pos);
}

template <class PA, class FC, class PPos, class InterpolatorTag>
void scatter(const PA& attrib, const FC& field, const PPos& pos,
	     const InterpolatorTag&)
{
  // check that dimensions of field and position attribute match
  typedef typename PPos::Element_t PPosElement_t;
  CTAssert(int(FC::dimensions) == int(PPosElement_t::d1));

  // ask Interpolator class to perform the scatter
  typedef typename PPosElement_t::Element_t AxisType_t;
  typedef Interpolator<FC::dimensions,AxisType_t,InterpolatorTag> Interp_t;
  Interp_t::scatter(attrib,field,pos);
}

template <class T, class FC, class PPos, class InterpolatorTag>
void scatterValue(const T& value, const FC& field, const PPos& pos,
		  const InterpolatorTag&)
{
  // check that dimensions of field and position attribute match
  typedef typename PPos::Element_t PPosElement_t;
  CTAssert(int(FC::dimensions) == int(PPosElement_t::d1));

  // ask Interpolator class to perform the scatter
  typedef typename PPosElement_t::Element_t AxisType_t;
  typedef Interpolator<FC::dimensions,AxisType_t,InterpolatorTag> Interp_t;
  Interp_t::scatterValue(value,field,pos);
}

// gather/scatter using particle position attribute and
// cache interpolation data

template <class PA, class FC, class PPos, class Cache, class InterpolatorTag>
void gatherCache(const PA& attrib, const FC& field, const PPos& pos,
                 const Cache& cache, const InterpolatorTag&)
{
  // check that dimensions of field and position attribute match
  typedef typename PPos::Element_t PPosElement_t;
  CTAssert((int)FC::dimensions == (int)PPosElement_t::d1);

  // ask Interpolator class to perform the gather and cache data
  typedef typename PPosElement_t::Element_t AxisType_t;
  typedef Interpolator<FC::dimensions,AxisType_t,InterpolatorTag> Interp_t;
  Interp_t::gatherCache(attrib,field,pos,cache);
}

template <class PA, class FC, class PPos, class Cache, class InterpolatorTag>
void scatterCache(const PA& attrib, const FC& field, const PPos& pos,
                  const Cache& cache, const InterpolatorTag&)
{
  // check that dimensions of field and position attribute match
  typedef typename PPos::Element_t PPosElement_t;
  CTAssert((int)FC::dimensions == (int)PPosElement_t::d1);

  // ask Interpolator class to perform the scatter and cache data
  typedef typename PPosElement_t::Element_t AxisType_t;
  typedef Interpolator<FC::dimensions,AxisType_t,InterpolatorTag> Interp_t;
  Interp_t::scatterCache(attrib,field,pos,cache);
}

template <class T, class FC, class PPos, class Cache, class InterpolatorTag>
void scatterValueCache(const T& value, const FC& field, const PPos& pos,
                       const Cache& cache, const InterpolatorTag&)
{
  // check that dimensions of field and position attribute match
  typedef typename PPos::Element_t PPosElement_t;
  CTAssert(int(FC::dimensions) == int(PPosElement_t::d1));

  // ask Interpolator class to perform the scatter and cache data
  typedef typename PPosElement_t::Element_t AxisType_t;
  typedef Interpolator<FC::dimensions,AxisType_t,InterpolatorTag> Interp_t;
  Interp_t::scatterValueCache(value,field,pos,cache);
}

// gather/scatter using cached interpolation data

template <class PA, class FC, class Cache, class InterpolatorTag>
void gatherCache(const PA& attrib, const FC& field, const Cache& cache,
                 const InterpolatorTag&)
{
  // check that dimensions of field and cache data match
  typedef typename Cache::Element_t CacheData_t;
  CTAssert((int)FC::dimensions == (int)CacheData_t::dimensions);

  // ask Interpolator class to perform the gather using cached data
  typedef typename CacheData_t::AxisType_t AxisType_t;
  typedef Interpolator<FC::dimensions,AxisType_t,InterpolatorTag> Interp_t;
  Interp_t::gatherCache(attrib,field,cache);
}

template <class PA, class FC, class Cache, class InterpolatorTag>
void scatterCache(const PA& attrib, const FC& field, const Cache& cache,
                  const InterpolatorTag&)
{
  // check that dimensions of field and cache data match
  typedef typename Cache::Element_t CacheData_t;
  CTAssert((int)FC::dimensions == (int)CacheData_t::dimensions);

  // ask Interpolator class to perform the scatter using cached data
  typedef typename CacheData_t::AxisType_t AxisType_t;
  typedef Interpolator<FC::dimensions,AxisType_t,InterpolatorTag> Interp_t;
  Interp_t::scatterCache(attrib,field,cache);
}

template <class T, class FC, class Cache, class InterpolatorTag>
void scatterValueCache(const T& value, const FC& field, const Cache& cache,
                       const InterpolatorTag&)
{
  // check that dimensions of field and cache data match
  typedef typename Cache::Element_t CacheData_t;
  CTAssert((int)FC::dimensions == (int)CacheData_t::dimensions);

  // ask Interpolator class to perform the scatter using cached data
  typedef typename CacheData_t::AxisType_t AxisType_t;
  typedef Interpolator<FC::dimensions,AxisType_t,InterpolatorTag> Interp_t;
  Interp_t::scatterValueCache(value,field,cache);
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Interpolation.cpp,v $   $Author: swhaney $
// $Revision: 1.9 $   $Date: 2000/03/07 13:17:47 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
