// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license
// ----------------------------------------------------------------------
// UpdaterRetargetBase & UpdaterBase: the base classes for all updaters.
//-----------------------------------------------------------------------------

#ifndef POOMA_NEWFIELD_UPDATER_UPDATERBASE_H
#define POOMA_NEWFIELD_UPDATER_UPDATERBASE_H

#include "NewField/Updater/UpdaterListItem.h"


//-----------------------------------------------------------------------------
// Pooma::removeUpdaters() removes the updaters associated with a given object
// if it makes sense. This default version does nothing.
//-----------------------------------------------------------------------------

namespace Pooma {

  template<class T>
  void removeUpdaters(T &) { }
  
}


//-----------------------------------------------------------------------------
// UpdaterRetargetBase is an abstract base class for all automatic updaters.
// It is sometimes necessary to be able to replace the subject of an
// updater with another object. UpdaterListItem is not templated so the
// idea is to do a dynamic_cast to an UpdaterRetargetBase and then call the
// virtual retarget function, which needs to be implemented in UpdaterBase,
// which actually knows the category.
//-----------------------------------------------------------------------------

template<class Subject>
class UpdaterRetargetBase : public UpdaterListItem {
public:

  //---------------------------------------------------------------------------
  // Convenient typedefs. 

  typedef UpdaterRetargetBase<Subject> This_t;  


  //---------------------------------------------------------------------------
  // Constructors. 

  // Initialize the subject.
  
  UpdaterRetargetBase(const Subject &s)
  : subject_m(s) 
    {
      // We need to remove any updaters associated with the subject to
      // prevent loops when running the update. 
      
      Pooma::removeUpdaters(s);
    }

  // Copy constructor. 

  UpdaterRetargetBase(const This_t &model)
  : UpdaterListItem(model),
    subject_m(model.subject_m) 
    { }
    
      
  //---------------------------------------------------------------------------
  // Trivial destructor, but virtual since we will be subclassing.

  virtual ~UpdaterRetargetBase() { }

  
  //---------------------------------------------------------------------------
  // Subject accessor.
  
  Subject &subject() { return subject_m; }
  const Subject &subject() const { return subject_m; }


  //---------------------------------------------------------------------------
  // Retarget function. Clones this updater for a new subject.
 
  virtual UpdaterListItem *retarget(const Subject &s) const = 0;

    
private:

  Subject subject_m;
    
};


//-----------------------------------------------------------------------------
// UpdaterRetargetBase is an abstract base class for all automatic updaters.
// It is sometimes necessary to be able to replace the subject of an
// updater with another object. UpdaterListItem is not templated so the
// idea is to do a dynamic_cast to an UpdaterRetargetBase and then call the
// virtual retarget function, which needs to be implemented in UpdaterBase,
// which actually knows the category.
//-----------------------------------------------------------------------------

template<class Subject, class Category>
class UpdaterBase : public UpdaterRetargetBase<Subject> {
public:

  //---------------------------------------------------------------------------
  // Convenient typedefs. 

  typedef UpdaterBase<Subject, Category> This_t;  


  //---------------------------------------------------------------------------
  // Constructors. 

  // Initialize the subject and category.
  
  UpdaterBase(const Subject &s, const Category &c)
  : UpdaterRetargetBase<Subject>(s),
    category_m(c)
    { }

  // Copy constructor. 

  UpdaterBase(const This_t &model)
  : UpdaterRetargetBase<Subject>(model),
    category_m(model.category_m) 
    { }
    
      
  //---------------------------------------------------------------------------
  // Trivial destructor, but virtual since we will be subclassing.

  virtual ~UpdaterBase() { }

  
  //---------------------------------------------------------------------------
  // Category accessor.
  
  Category &category() { return category_m; }
  const Category &category() const { return category_m; }


  //---------------------------------------------------------------------------
  // Retarget function. Clones this updater for a new subject.
 
  virtual UpdaterListItem *retarget(const Subject &s) const
    {
      return category_m.create(s);
    }

    
private:

  Category category_m;
    
};

#endif // POOMA_NEWFIELD_UPDATER_UPDATERBASE_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: UpdaterBase.h,v $   $Author: swhaney $
// $Revision: 1.2 $   $Date: 2000/07/20 21:03:51 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
