// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Class:
// RNGValue
// RNGSeed
// RNGAdvance
// RNGAdvanceProxy<RNG>
//-----------------------------------------------------------------------------

#ifndef POOMA_FUNCTIONS_RNG_COMPONENT_H
#define POOMA_FUNCTIONS_RNG_COMPONENT_H

//////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// Overview:
//
// These component accessors allow you to access values from arrays of random
// numbers and to advance them.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------

template<class T, class Components> struct ComponentAccess;

//-----------------------------------------------------------------------------
//
// RNGValue:
//
// RNGValue lets you access the value of random numbers stored in an array.
//
// Array<2, RNG> a;
//
// b = a.comp(RNGValue());
//
// While a.comp(RNGValue()) can be an Array (not just a Array), it is an
// error to attempt to assign to the values.
//-----------------------------------------------------------------------------

struct RNGValue { };

template<class RNG>
struct ComponentAccess< RNG, RNGValue >
{
  typedef typename RNG::Type_t Element_t;
  typedef Element_t ElementRef_t;

  static inline ElementRef_t indexRef(RNG &rng, const RNGValue &)
  {
    return rng.value();
  }
  
  static inline Element_t index(const RNG &rng, const RNGValue &)
  {
    return rng.value();
  }
};

//-----------------------------------------------------------------------------
//
// RNGAdvance:
//
// RNGAdvance lets you advance the value of random numbers stored in an array.
//
// Array<2, RNG> a;
//
// a.comp(RNGAdvance()) = 5;
//
// (advance the random numbers 5 times at every location in a)
//
//-----------------------------------------------------------------------------

struct RNGAdvance { };

template<class RNG>
struct RNGAdvanceProxy
{ 
  RNGAdvanceProxy(RNG &rng)
    : rng_m(rng)
  { }

  inline RNGAdvanceProxy &operator=(int i)
  {
    rng_m.advance(i);
    return *this;
  }

  inline const RNGAdvanceProxy &operator=(int i) const
  {
    rng_m.advance(i);
    return *this;
  }

  inline operator int() const
  {
    return 0;
  }

  mutable RNG &rng_m;
};

template<class RNG>
struct ComponentAccess< RNG, RNGAdvance >
{
  typedef int Element_t;
  typedef RNGAdvanceProxy<RNG> ElementRef_t;
  
  static inline ElementRef_t indexRef(RNG &rng, const RNGAdvance &)
  {
    return ElementRef_t(rng);
  }
  
  static inline Element_t index(const RNG &rng, const RNGAdvance &)
  {
    return 0;
  }
};

//-----------------------------------------------------------------------------
//
// RNGSeed:
//
// RNGSeed lets you set and get the seeds of random numbers stored in an array.
//
// Array<2, RNG> a;
//
// oldSeed = a.comp(RNGSeed());
// a.comp(RNGSeed()) = newSeed;
//
// (Some RNGs may mess with the seed value when you set it to make it more
// random.  In that case, the values you read can be different from the values
// you assign.)
//-----------------------------------------------------------------------------

struct RNGSeed { };

template<class RNG>
struct RNGSeedProxy
{
  RNGSeedProxy(RNG &rng)
    : rng_m(rng)
  { }

  inline RNGSeedProxy &operator=(long i)
  {
    rng_m.seed(i);
    return *this;
  }

  inline const RNGSeedProxy &operator=(long i) const
  {
    rng_m.seed(i);
    return *this;
  }

  inline operator long() const
  {
    return rng.seed();
  }

  RNG &rng_m;
};

template<class RNG>
struct ComponentAccess< RNG, RNGSeed >
{
  typedef int Element_t;
  typedef RNGSeedProxy<RNG> ElementRef_t;
  
  static inline ElementRef_t indexRef(RNG &rng, const RNGSeed &)
  {
    return ElementRef_t(rng);
  }
  
  static inline Element_t index(const RNG &rng, const RNGSeed &)
  {
    return rng.seed();
  }
};

//////////////////////////////////////////////////////////////////////

#endif     // POOMA_FUNCTIONS_RNG_COMPONENT_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: RNGComponent.h,v $   $Author: swhaney $
// $Revision: 1.7 $   $Date: 2000/07/20 15:39:27 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
