// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

// ---------------------------------------------------------------------------
// TestMultiPatchField.cpp , Tim Williams 7/2/1999
// 
// Tests non-uniform MultiPatch Field construction and basic assignment.
// ---------------------------------------------------------------------------

// include files
#include "Pooma/Fields.h"
#include "Utilities/Tester.h"

// Forward declarations:
template<int SliceDim, int Dim>
SliceInterval<Dim, SliceDim>
convertToSlice(const Interval<Dim> &i);


int main(int argc, char* argv[])
{
  // Initialize POOMA and Tester class.
  Pooma::initialize(argc, argv);
  Pooma::Tester tester(argc, argv);

  tester.out() << argv[0] << ": " << std::endl;
  tester.out() << "------------------------------------------------"
               << std::endl;

  // Dimensionalities:
  const int Dim1 = 1;
  const int Dim2 = 2;
  const int Dim3 = 3;
  int d;

  // Vertex domains:
  int nVerts[4] = {6, 4, 5};
  Interval<Dim1> vertDomain1;
  for (d = 0; d < Dim1; d++) {
    vertDomain1[d] = Interval<1>(nVerts[d]);
  }
  Interval<Dim2> vertDomain2;
  for (d = 0; d < Dim2; d++) {
    vertDomain2[d] = Interval<1>(nVerts[d]);
  }
  Interval<Dim3> vertDomain3;
  for (d = 0; d < Dim3; d++) {
    vertDomain3[d] = Interval<1>(nVerts[d]);
  }

  // Create the meshes; default origin and spacings:
  typedef UniformRectilinearMesh<Dim1> Mesh1_t;
  typedef UniformRectilinearMesh<Dim2> Mesh2_t;
  typedef UniformRectilinearMesh<Dim3> Mesh3_t;
  Mesh1_t mesh1(vertDomain1);
  Mesh2_t mesh2(vertDomain2);
  Mesh3_t mesh3(vertDomain3);
  
  // Create the geometries,
  DiscreteGeometry<Vert, Mesh1_t> geomV1(mesh1, GuardLayers<Dim1>(1));
  DiscreteGeometry<Vert, Mesh2_t> geomV2(mesh2, GuardLayers<Dim2>(1));
  DiscreteGeometry<Vert, Mesh3_t> geomV3(mesh3, GuardLayers<Dim3>(1));
  
  // Create the partitioners:
  Loc<Dim1> blocks1d(2);
  GridPartition<Dim1> partition1d(blocks1d, GuardLayers<Dim1>(1));
  Loc<Dim2> blocks2d(2,2);
  GridPartition<Dim2> partition2d(blocks2d, GuardLayers<Dim2>(1));
  Loc<3> blocks3d(2,2,1);
  GridPartition<3> partition3d(blocks3d, GuardLayers<Dim3>(1));

  // Create the layouts:
  GridLayout<Dim1> layout1d(vertDomain1, partition1d, ReplicatedTag());
  GridLayout<Dim2> layout2d(vertDomain2, partition2d, ReplicatedTag());
  GridLayout<Dim3> layout3d(vertDomain3, partition3d, ReplicatedTag());

  // Create the MultiPatch Fields:
  typedef MultiPatch<GridTag, CompressibleBrick> MP_t;
  typedef Field<DiscreteGeometry<Vert, Mesh1_t>, double, MP_t> MPScalarField1_t;
  typedef Field<DiscreteGeometry<Vert, Mesh2_t>, double, MP_t> MPScalarField2_t;
  typedef Field<DiscreteGeometry<Vert, Mesh3_t>, double, MP_t> MPScalarField3_t;
  MPScalarField1_t a1(geomV1,  layout1d);
  MPScalarField2_t a2(geomV2,  layout2d);
  MPScalarField3_t a3(geomV3,  layout3d);

  // Now try assigning them:
  a1 = 1.0;
  a2 = 1.0;
  a3 = 1.0;

  // Try some slice-to-lower-dimensional-Array assignments:
  //works  a1 = a2(vertDomain2[0], 1);
  Interval<Dim2> slice = vertDomain2;
  slice[1] = Interval<Dim1>(1,1);
  tester.out() << "Here" << std::endl;
  tester.out() << "slice" << slice << std::endl;
  tester.out() << "convertToSlice<Dim1>(slice)" << 
    convertToSlice<Dim1>(slice) << std::endl;
  a1 = a2(convertToSlice<Dim1>(slice));
  tester.out() << "...here" << std::endl;

  tester.out() << "------------------------------------------------"
               << std::endl;
  int retval = tester.results("TestMultiPatchField");
  Pooma::finalize();
  return retval;
}

// Function courtesy of Scott Haney; a tidy way for users to make slices in as
// dimensionality-independent a fashion as is meaningful:
template<int SliceDim, int Dim>
SliceInterval<Dim, SliceDim>
convertToSlice(const Interval<Dim> &i)
{
  int num, d;
  SliceInterval<Dim, SliceDim> s;
  s.totalDomain() = i;
  for (d = 0, num = 0; d < Dim; d++) {
    if (i[d].length() == 1) { num++; }
    s.ignorable(d) = (i[d].length() == 1);
  }
  PAssert(num == SliceDim);
  s.setSliceFromTotal();
  return s;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: TestMultiPatchField.cpp,v $   $Author: julianc $
// $Revision: 1.9 $   $Date: 2000/07/20 22:21:30 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
