// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license
//-----------------------------------------------------------------------------
// BrickEngine test code
//-----------------------------------------------------------------------------

#include "Pooma/Pooma.h"
#include "Utilities/PAssert.h"
#include "Utilities/Tester.h"
#include "Domain/Interval.h"
#include "Domain/Range.h"
#include "Engine/BrickEngine.h"

#include <iomanip>

typedef Engine<1,double,Brick>         Array_t;
typedef Engine<1,double,BrickView> View_t;

int main(int argc, char *argv[])
{
  Pooma::initialize(argc,argv);
  Pooma::Tester tester(argc,argv);
#if POOMA_EXCEPTIONS
  try {
#endif
    tester.out() << "\nTesting BrickEngine." << std::endl;

    Interval<1> I(10);
    Array_t A(I);

    for (int i = 0; i < 10; i++)
      A(Loc<1>(i)) = 2.0 + i - i*i;
    
    for (int i = 0; i < 10; i++)
      tester.out() <<  A(Loc<1>(i)) << " ";
    tester.out() << std::endl;

    Interval<1> J(2,5);
    View_t B(A,J);

    for (int i = 0; i < 4; i++)
      tester.out() <<  B(Loc<1>(i)) << " ";
    tester.out() << std::endl;

    Range<1> K(1,9,2);
    View_t C(A,K);
    
    for (int i = 0; i < 5; i++)
      tester.out() << C(Loc<1>(i)) << " ";
    tester.out() << std::endl;

    Array_t AC = A;

    AC(Loc<1>(2)) = -999;
    tester.out() << "AC(2) = " << AC(Loc<1>(2)) << std::endl;
    tester.out() << "A(2) = " << A(Loc<1>(2)) << std::endl;

    AC.makeOwnCopy();

    AC(Loc<1>(7)) = -111;
    tester.out() << "AC(2) = " << AC(Loc<1>(7)) << std::endl;
    tester.out() << "A(2) = " << A(Loc<1>(7)) << std::endl;

    Array_t E(I);
    for (int i = 0; i < 10; i++)
      E(i) = i;

    tester.out() << "E: ";
    for (int i = 0; i < 10; i++)
      tester.out() << E(i) << " ";
    tester.out() << std::endl;

    Array_t F = E;
    
    tester.out() << "F == E" << std::endl;
    tester.out() << "F: ";
    for (int i = 0; i < 10; i++)
      tester.out() << F(i) << " ";
    tester.out() << std::endl;

    Array_t G(I);
    for (int i = 0; i < 10; i++)
      G(i) = i*i;

    tester.out() << "G: ";
    for (int i = 0; i < 10; i++)
      tester.out() << G(i) << " ";
    tester.out() << std::endl;

    E = G;

    tester.out() << "E = G;" << std::endl;
    tester.out() << "E: ";
    for (int i = 0; i < 10; i++)
      tester.out() << E(i) << " ";
    tester.out() << std::endl;

    //    PAssert(!F.isShared());

    AC(Loc<1>(2)) = -999;
    tester.out() << "AC(2) = " << AC(Loc<1>(2)) << std::endl;
    tester.out() << "A(2) = " << A(Loc<1>(2)) << std::endl;

    tester.out() << "\nTesting BrickEngine<double,3>." << std::endl;

    Interval<3> III(I,I,I);
    Engine<3,double,Brick> AAA(III);

    int Imax = I.length();

    for (int i = 0; i < Imax; i++)
      for (int j = 0; j < Imax; j++)
	for (int k = 0; k < Imax; k++)
	  {
	    AAA(Loc<3>(i,j,k)) = i + j + k;
	  }

    for (int i = 0; i < Imax; i++)
      {
	tester.out() << "Slice i = " << i << std::endl;
	for (int j = 0; j < Imax; j++)
	  {
	    for (int k = 0; k < Imax; k++)
	      {
		tester.out() << std::setw(3) << AAA(Loc<3>(i,j,k)) << " ";
	      }
	    tester.out() << std::endl;
	  }
	tester.out() << std::endl;
      }

    Range<1> J2(2,8,2);

    Range<3> JJJ(J2,J2,J2);

    Engine<3,double,BrickView> AV(AAA,JJJ);

    int Jmax = J2.length();

    for (int i = 0; i < Jmax; i++)
      for (int j = 0; j < Jmax; j++)
	for (int k = 0; k < Jmax; k++)
	  {
	    AV(Loc<3>(i,j,k)) = -1;
	  }

    for (int i = 0; i < Imax; i++)
      {
	tester.out() << "Slice i = " << i << std::endl;
	for (int j = 0; j < Imax; j++)
	  {
	    for (int k = 0; k < Imax; k++)
	      {
		tester.out() << std::setw(3) << AAA(Loc<3>(i,j,k)) << " ";
	      }
	    tester.out() << std::endl;
	  }
	tester.out() << std::endl;
      }

    Range<1> J3(0,2,2);
    Range<1> J0(3);
    Range<3> JJJJ(J0,J3,J0);

    Engine<3,double,BrickView> AVV(AV,JJJJ);
     
    tester.out() << "Domain of AAA = " << std::endl << std::endl;
    tester.out() << AAA.domain() << std::endl << std::endl;

    Interval<3> avdom = AV.domain();

    tester.out() << "Domain of AV  = " << std::endl << std::endl;
    tester.out() << avdom << std::endl << std::endl;
    tester.out() << avdom[0].length() << std::endl << std::endl;

#if POOMA_EXCEPTIONS
  }
  catch(const char *err) 
    { 
      tester.exceptionHandler( err );
      tester.set( false );
    }
  catch(const Pooma::Assertion &err)
    { 
      tester.exceptionHandler( err );
      tester.set( false );
    }
#endif    
  int ret = tester.results("brick_test1");
  Pooma::finalize();
  return ret;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: brick_test1.cpp,v $   $Author: sa_smith $
// $Revision: 1.14 $   $Date: 2000/07/11 23:06:53 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
