// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license
// ----------------------------------------------------------------------
// BCond: intermediate templated base class for all boundary conditions.
//-----------------------------------------------------------------------------

#ifndef POOMA_BCONDS_BCOND_H
#define POOMA_BCONDS_BCOND_H

#include "BConds/BCondItem.h"
#include "Domain/Loc.h"

//-----------------------------------------------------------------------------
// BCond is a base template for all boundary conditions, which are constructed
// as partial specializations to this class. These partial specializations
// have the following interface requirements:
//
// (1) They must inherit publicly from BCondBase<Domain, Subject>.
// (2) They must define a constructor that takes a const reference to the
//     subject and a const reference to a Category object.
// (3) They must define an applyBoundaryCondition() function that actually
//     applies the boundary condition.
//-----------------------------------------------------------------------------

template<class Subject, class Category>
class BCond;


//-----------------------------------------------------------------------------
// BCondCategory is a base class for the boundary condition category tags.
// It only contains a factory method for creating a boundary condition given
// a subject, which is typically a Field. 
//
// Users wanting to make a particular sort of boundary condition (e.g., 
// ExtrapolateBC, EurekaBC) should inherit from this class using the 
// Glommable pattern; that is,
//
//   class ExtrapolateBC: public BCondCategory<ExtrapolateBC> { ... }
//
// They should also define data members in the sub-class for storing 
// information the boundary condition needs to evaluate itself.
//-----------------------------------------------------------------------------

template<class Category>
class BCondCategory
{
public:

  //---------------------------------------------------------------------------
  // Default and copy constructors. Trivial since we have no data. Supplied
  // to keep purify from griping.
  
  BCondCategory() { }
  BCondCategory(const BCondCategory<Category> &) { }

  //---------------------------------------------------------------------------
  // Assignment operator. Does deep assignment.
  
  BCondCategory<Category> &operator=(const BCondCategory<Category> &)
  {
    return *this;
  }
  
  //---------------------------------------------------------------------------
  // Factory method. Actually makes the boundary condition.
  
  template<class Subject>
  BCondItem *create(const Subject &s) const
  {
    return new BCond<Subject, Category>
      (s, static_cast<const Category&>(*this));
  }
};


//-----------------------------------------------------------------------------
// ComponentBC is used to construct boundary conditions that work on
// particular components of a multi-compoment Field or Array. The idea is to,
// for example, build an ExtrapolateBC object and then use this to create a 
// ComponentBC<1, ExtrapolateBC> object, which builds a boundary condition by
// taking a one-dimensional component view of specified subject. This means 
// that boundary condition objects need only to work on scalar fields.
//-----------------------------------------------------------------------------

template<int Dim, class Category>
class ComponentBC { };

//---------------------------------------------------------------------------
// Super-secret factory method. Actually makes the boundary condition. Called
// by create(). We use this two-step approach so we can specialize behavior
// for specific classes.
//-----------------------------------------------------------------------------
  
template<class Subject, int N, class Category>
inline
BCondItem *createComponentwiseBoundaryCondition(const Subject &s,
  const ComponentBC<N, Category> &c)
{
  return c.category().create(s.comp(c.comp()));
}  

template<class Category>
class ComponentBC<1, Category>
{
public:

  //---------------------------------------------------------------------------
  // Constructors. Need to store a copy of the category and which component
  // we're dealing with.
  
  ComponentBC(int c1, const Category &cat) 
  : c1_m(c1), category_m(cat) { }
  ComponentBC(const Loc<1> &c, const Category &cat) 
  : c1_m(c[0].first()), category_m(cat) { }
  ComponentBC(const ComponentBC<1, Category> &model)
  : category_m(model.category()), 
    c1_m(model.comp1()) { }
  
  //---------------------------------------------------------------------------
  // Short-circuit constructors: pass extra arguments directly to category.
  
  template<class T1>
  ComponentBC(int c1, const T1 &arg1)
  : c1_m(c1), category_m(arg1) { }

  template<class T1, class T2>
  ComponentBC(int c1, const T1 &arg1, const T2 &arg2)
  : c1_m(c1), category_m(arg1, arg2) { }
  
  //---------------------------------------------------------------------------
  // Assignment operator. Does deep assignment.
  
  ComponentBC<1, Category> &operator=(const ComponentBC<1, Category> &rhs)
  {
    c1_m = rhs.comp1();
    category_m = rhs.category();
    
    return *this;
  }
  
  //---------------------------------------------------------------------------
  // Accessors for data members.
  
  const Category &category() const { return category_m; }
  int comp1() const { return c1_m; }
  Loc<1> comp() const { return Loc<1>(c1_m); }
  
  //---------------------------------------------------------------------------
  // Factory method. Actually makes the boundary condition. Takes a component
  // view of the subject. 
  
  template<class Subject>
  BCondItem *create(const Subject &s) const
  {
    return createComponentwiseBoundaryCondition(s, *this);
  }
  
private:

  Category category_m;
  int c1_m;
};

template<class Category>
class ComponentBC<2, Category>
{
public:

  //---------------------------------------------------------------------------
  // Constructors. Need to store a copy of the category and which component
  // we're dealing with.
  
  ComponentBC(int c1, int c2, const Category &cat) 
  : category_m(cat), c1_m(c1), c2_m(c2) { }
  ComponentBC(const Loc<2> &c, const Category &cat) 
  : category_m(cat), c1_m(c[0].first()), c2_m(c[1].first()) { }
  ComponentBC(const ComponentBC<2, Category> &model)
  : category_m(model.category()), 
    c1_m(model.comp1()), c2_m(model.comp2()) { }
  
  //---------------------------------------------------------------------------
  // Short-circuit constructors: pass extra arguments directly to category.
  
  template<class T1>
  ComponentBC(int c1, int c2, const T1 &arg1)
  : c1_m(c1), c2_m(c2), category_m(arg1) { }

  template<class T1, class T2>
  ComponentBC(int c1, int c2, const T1 &arg1, const T2 &arg2)
  : c1_m(c1), c2_m(c2), category_m(arg1, arg2) { }
  
  //---------------------------------------------------------------------------
  // Assignment operator. Does deep assignment.
  
  ComponentBC<2, Category> &operator=(const ComponentBC<2, Category> &rhs)
  {
    c1_m = rhs.comp1();
    c2_m = rhs.comp2();
    category_m = rhs.category();
    
    return *this;
  }
  
  //---------------------------------------------------------------------------
  // Accessors for data members.
  
  const Category &category() const { return category_m; }
  int comp1() const { return c1_m; }
  int comp2() const { return c2_m; }
  Loc<2> comp() const { return Loc<2>(c1_m, c2_m); }
  
  //---------------------------------------------------------------------------
  // Factory method. Actually makes the boundary condition. Takes a component
  // view of the subject. 
  
  template<class Subject>
  BCondItem *create(const Subject &s) const
  {
    return createComponentwiseBoundaryCondition(s, *this);
  }
  
private:

  Category category_m;
  int c1_m, c2_m;
};

#endif // POOMA_BCONDS_BCOND_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: BCond.h,v $   $Author: swhaney $
// $Revision: 1.5 $   $Date: 2000/03/07 13:16:10 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
