// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license
//-----------------------------------------------------------------------------
// Array test 3: Vector array elements.
//-----------------------------------------------------------------------------

// Include files

#include "Pooma/Pooma.h"
#include "Utilities/Tester.h"
#include "Engine/BrickEngine.h"
#include "Array/Array.h"
#include "Tiny/Vector.h"

#include <iostream>

static int checkNum = 1;

template<class T>
inline void check(const T &ans, const T &correct, Pooma::Tester &tester)
{
  bool ok = (ans == correct);
  tester.out() << "Check #" << checkNum << std::endl;
  tester.out() << "Answer:  " << ans << std::endl;
  if (!ok)
    {
      tester.out() << "Correct: " << correct << std::endl;
    }

  checkNum++;      
  tester.check(ok);
}

int main(int argc, char *argv[])
{
  Pooma::initialize(argc, argv);
  Pooma::Tester tester(argc, argv);

  Array<2, int> a(2, 2, modelElement(7));
  Array<2, Vector<3> > b(2, 2, modelElement(Vector<3>(1.0, 2.0, 3.0))), 
    c(2, 2);
  Array<2> d(2,2);
  int i, j;

  for (j = 0; j < 2; j++)
    for (i = 0; i < 2; i++)
      {
        check(a(i,j), 7, tester);
        check(b(i,j), Vector<3>(1.0, 2.0, 3.0), tester);
      }
  
  b.comp(1) = 6.0;

  Pooma::blockAndEvaluate();
  for (j = 0; j < 2; j++)
    for (i = 0; i < 2; i++)
      {
        check(b(i,j), Vector<3>(1.0, 6.0, 3.0), tester);
      }

  b.comp(0) = a + b.comp(1) + b.comp(2);

  Pooma::blockAndEvaluate();
  for (j = 0; j < 2; j++)
    for (i = 0; i < 2; i++)
      {
        check(b(i,j), Vector<3>(16.0, 6.0, 3.0), tester);
      }

  c = a + b;

  Pooma::blockAndEvaluate();
  for (j = 0; j < 2; j++)
    for (i = 0; i < 2; i++)
      {
        check(c(i,j), Vector<3>(23.0, 13.0, 10.0), tester);
      }

  c = a + 2 * b;

  Pooma::blockAndEvaluate();
  for (j = 0; j < 2; j++)
    for (i = 0; i < 2; i++)
      {
        check(c(i,j), Vector<3>(39.0, 19.0, 13.0), tester);
      }

  Vector<3> x(-1,-2,-3);
  c = a + x * b;
  
  Pooma::blockAndEvaluate();
  for (j = 0; j < 2; j++)
    for (i = 0; i < 2; i++)
      {
        check(c(i,j), Vector<3>(-9.0, -5.0, -2.0), tester);
      }

  c = a + b * x;
  
  Pooma::blockAndEvaluate();
  for (j = 0; j < 2; j++)
    for (i = 0; i < 2; i++)
      {
        check(c(i,j), Vector<3>(-9.0, -5.0, -2.0), tester);
      }

  d = a + dot(x,b);
  
  Pooma::blockAndEvaluate();
  for (j = 0; j < 2; j++)
    for (i = 0; i < 2; i++)
      {
        check(d(i,j), -30.0, tester);
      }

  d = a - dot(c,b);
  
  Pooma::blockAndEvaluate();
  for (j = 0; j < 2; j++)
    for (i = 0; i < 2; i++)
      {
        check(d(i,j), 187.0, tester);
      }

  b.comp(0) = a + b.comp(1) + b.comp(2) - 1;

  Pooma::blockAndEvaluate();
  for (j = 0; j < 2; j++)
    for (i = 0; i < 2; i++)
      {
        check(b(i,j), Vector<3>(15.0, 6.0, 3.0), tester);
      }
         
  int ret = tester.results( "array_test3" );
  Pooma::finalize();
  return ret; 
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: array_test3.cpp,v $   $Author: swhaney $
// $Revision: 1.18 $   $Date: 2000/07/20 15:36:25 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
