// This file is part of the program FRYSK.
//
// Copyright 2005, Red Hat Inc.
//
// FRYSK is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; version 2 of the License.
//
// FRYSK is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
// type filter text
// You should have received a copy of the GNU General Public License
// along with FRYSK; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
// 
// In addition, as a special exception, Red Hat, Inc. gives You the
// additional right to link the code of FRYSK with code not covered
// under the GNU General Public License ("Non-GPL Code") and to
// distribute linked combinations including the two, subject to the
// limitations in this paragraph. Non-GPL Code permitted under this
// exception must only link to the code of FRYSK through those well
// defined interfaces identified in the file named EXCEPTION found in
// the source code files (the "Approved Interfaces"). The files of
// Non-GPL Code may instantiate templates or use macros or inline
// functions from the Approved Interfaces without causing the
// resulting work to be covered by the GNU General Public
// License. Only Red Hat, Inc. may make changes or additions to the
// list of Approved Interfaces. You must obey the GNU General Public
// License in all respects for all of the FRYSK code and other code
// used in conjunction with FRYSK except the Non-GPL Code covered by
// this exception. If you modify this file, you may extend this
// exception to your version of the file, but you are not obligated to
// do so. If you do not wish to provide this exception without
// modification, you must delete this exception statement from your
// version and license this file solely under the GPL without
// exception.


package frysk.gui.sessions;

import java.util.Date;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Observable;
import java.util.Observer;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.jdom.Element;

import frysk.gui.Gui;
import frysk.gui.common.IconManager;
import frysk.gui.monitor.GuiObject;
import frysk.gui.monitor.GuiProc;
import frysk.gui.monitor.ObservableLinkedList;
import frysk.gui.monitor.WindowManager;
import frysk.gui.monitor.observers.ObserverManager;
import frysk.gui.monitor.observers.ObserverRoot;
import frysk.gui.srcwin.SourceWindowFactory;
import frysk.proc.Proc;

/**
 * A Session object is used to hold and save user preferences with respect to a
 * debug session.
 */
public class Session
    extends GuiObject
{

  private ObservableLinkedList procs;
  private ObservableLinkedList observers;
  
  private boolean procsAdded = false;
  
  private Logger errorLog = Logger.getLogger(Gui.ERROR_LOG_ID);

  public SessionType sessionType;
  
  public static class SessionType{
    public static final SessionType DebugSession = new SessionType("Debugging Session");
    public static final SessionType MonitorSession = new SessionType("Monitoring Session");
    
    private String name;
    
    private SessionType(String name){
      this.name = name;
    }
    
    public String getName(){
      return this.name;
    }

    public static SessionType getSessionTypeByName (String name)
    {
      if(name.equals(DebugSession.getName())){
	return DebugSession;
      }
      
      if(name.equals(MonitorSession.getName())){
	return MonitorSession;
      }
      
      throw new IllegalArgumentException("the given name ["+name+"] is not a valid SessionType name");
    }
    
    public String toString ()
    {
      return super.toString() + ": " + this.name;
    }
  }
  
  /**
   * Creates a new empty session object, with an empty list processes. Debug
   * processes should be added to this session.
   */
  public Session ()
  {
    super();
    procs = new ObservableLinkedList();
    observers = new ObservableLinkedList();
    this.initListObservers();
    
    this.setProcsAdded(false);
    
    this.sessionType = SessionType.MonitorSession;
  }

  /**
   * Creates a new session, which is clone of the session that is passed in as a
   * paraemter
   * 
   * @param other - the session you want this session to clone.
   */
  public Session (final Session other)
  {
    super(other);

    procs = new ObservableLinkedList(other.procs, true);
    observers = new ObservableLinkedList(other.observers, false);
    this.initListObservers();
    
    this.setProcsAdded(other.areProcsAdded());
    
    this.sessionType = other.sessionType;
  }

  /**
   * Creates a new empty session object, with an empty list processes. Debug
   * processes should be added to this session.
   * 
   * @param name - the name of the session
   * @param toolTip - the tool-tip or additional associative information.
   */
  public Session (final String name, final String toolTip)
  {
    super(name, toolTip);
    procs = new ObservableLinkedList();
    observers = new ObservableLinkedList();
    this.initListObservers();
    
    this.setProcsAdded(false);
    
    this.sessionType = SessionType.MonitorSession;
  }

  public void addDefaultObservers(){
    Iterator iterator = ObserverManager.theManager.getDefaultObservers().iterator();
    while (iterator.hasNext())
      {
        ObserverRoot observer = (ObserverRoot) iterator.next();
        this.observers.add(observer);
      }
  }
  
  private void initListObservers(){
    //Every time a process is added add all existing observers to it
    this.procs.itemAdded.addObserver(new Observer()
    {
      public void update (Observable observable, Object object)
      {
	addAllObservers((DebugProcess) object);
      }
    });
    
    //Every time a process is removed remove all existing observers from it
    this.procs.itemRemoved.addObserver(new Observer()
    {
      public void update (Observable observable, Object object)
      {
	removeAllObservers((DebugProcess) object);
      }
    });
    
    //Every time an observer is added add it to all existing processes
    this.observers.itemAdded.addObserver(new Observer()
    {
      public void update (Observable observable, Object object)
      {
	addObserverToAllProcs( (ObserverRoot) object);
      }
    });
    
    //Every time an observer is removed remove it from all existing processes
    this.observers.itemRemoved.addObserver(new Observer()
    {
      public void update (Observable observable, Object object)
      {
	removeObserverFromAllProcs((ObserverRoot) object);
      }
    });
  }
  
  /*
   * (non-Javadoc)
   * 
   * @see frysk.gui.monitor.GuiObject#setName(java.lang.String)
   */
  public void setName (final String name)
  {
    if(name.length() == 0){
      throw new IllegalArgumentException("You cannot set the name of a session to an empty string");
    }
    super.setName(name);
  }

  public void addObserver(ObserverRoot observer){
    this.observers.add(observer);
  }
  
  public void removeObserver(ObserverRoot observer){
    this.observers.remove(observer);
  }
  
  private void addAllObservers(DebugProcess debugProcess){
    Iterator iterator = this.observers.iterator();
    while (iterator.hasNext())
      {
        ObserverRoot observer = (ObserverRoot) iterator.next();
        debugProcess.addObserver(observer);
      }
  }
  
  private void removeAllObservers(DebugProcess debugProcess){
    Iterator iterator = this.observers.iterator();
    while (iterator.hasNext())
      {
        ObserverRoot observer = (ObserverRoot) iterator.next();
        debugProcess.removeObserver(observer);
      }
  }

  private void addObserverToAllProcs(ObserverRoot observer){
    Iterator iterator = this.procs.iterator();
    while (iterator.hasNext())
      {
        DebugProcess debugProcess = (DebugProcess) iterator.next();
        debugProcess.addObserver(observer);
      }
  }
  
  private void removeObserverFromAllProcs(ObserverRoot observer){
    Iterator iterator = this.procs.iterator();
    while (iterator.hasNext())
      {
        DebugProcess debugProcess = (DebugProcess) iterator.next();
        debugProcess.removeObserver(observer);
      }
  }
  
  /**
   * Add a debug process to this session
   * 
   * @param process - The Debug Process that is to be added.
   */
  public void addDebugProcess (final DebugProcess process)
  {
    procs.add(process);
  }

  /**
   * Remove a debug process from this session.
   * 
   * @param process - a reference to the debug process that is to be removed.
   */
  public void removeDebugProcess (final DebugProcess process)
  {
    procs.remove(process);
  }

  /**
   * Searches to find if there is already a DebugProcess representing
   * the executable of the given GuiProc. If so the proc is added to
   * that DebugProcess. Otherwise a new DebugProcess is created
   * and added to the session.
   * @param guiProc
   */
  public void addGuiProc(GuiProc guiProc){
    Iterator iterator = this.procs.iterator();
    while (iterator.hasNext())
      {
	DebugProcess debugProcess = (DebugProcess) iterator.next();
	if(debugProcess.getExecutablePath().equals(guiProc.getNiceExecutablePath())){
	  debugProcess.addProc(guiProc);
	  return;
	}
      }
    
    //no DebugProcess was found create a new one
    DebugProcess debugProcess = new DebugProcess(guiProc.getExecutableName(), guiProc.getExecutableName(), guiProc.getNiceExecutablePath());
    this.addDebugProcess(debugProcess);
  }
  
  /**
   * 
   * @param proc
   */
  public void removeGuiProc(GuiProc proc){
    Iterator iterator = this.procs.iterator();
    while (iterator.hasNext())
      {
	DebugProcess debugProcess = (DebugProcess) iterator.next();
	if(debugProcess.getExecutablePath().equals(proc.getNiceExecutablePath())){
	  debugProcess.removeProc(proc);
	  return;
	}
      }
    
    //no DebugProcess was found
    throw new IllegalArgumentException("The given proc ["+proc+"] is not a member of this session");
  }
  
  /*
   * (non-Javadoc)
   * 
   * @see frysk.gui.monitor.GuiObject#getCopy()
   */
  public GuiObject getCopy ()
  {
    return new Session(this);
  }

  /**
   * Return a list of debug process that are contained within this session
   * object
   * 
   * @return ObservableLinkedList of Debug Processes.
   */
  public ObservableLinkedList getProcesses ()
  {
    return procs;
  }

  public void clearProcesses ()
  {
    this.procs.clear();
  }

  private void saveObservers(Element node){
    Iterator iterator = observers.iterator();
    while (iterator.hasNext())
      {
        GuiObject object = (GuiObject) iterator.next();
        Element elementXML = new Element("element");
        elementXML.setAttribute("name", object.getName());
        node.addContent(elementXML);
      }
  }
  
  private void loadObservers(Element node){
    List list = node.getChildren("element");
    Iterator i = list.iterator();

    while (i.hasNext())
      {
        Element elementXML = (Element) i.next();
//        ObserverRoot observer = ObserverManager.theManager.getObserverCopy(ObserverManager.theManager.getObserverByName(elementXML.getAttributeValue("name")));
        ObserverRoot observer = ObserverManager.theManager.getObserverByName(elementXML.getAttributeValue("name"));
        if (observer == null)
          {
            errorLog.log(Level.SEVERE,
                         new Date()
                             + " DebugProcess.load(Element node): observer "
                             + elementXML.getAttributeValue("name")
                             + " not found in configuration \n");
          }else{
          addObserver(observer);
        }
      }
  }

  public void startSession ()
  {
    if (this.sessionType == SessionType.MonitorSession)
      {
	LinkedList ll = new LinkedList();
	ll.add(WindowManager.theManager.mainWindow);
	IconManager.trayIcon.setPopupWindows(ll);
	WindowManager.theManager.mainWindow.hideTerminal();
	WindowManager.theManager.mainWindow.showAll();
	WindowManager.theManager.sessionManagerDialog.hideAll();
      }
    else
      {
	LinkedList list = new LinkedList();

	Iterator debugProcs = this.getProcesses().iterator();
	while (debugProcs.hasNext())
	  {
	    DebugProcess debugProcess = (DebugProcess) debugProcs.next();
	    list.addAll(debugProcess.getProcs());
	  }

	if (list.size() > 0)
	  {
	    Iterator i = list.iterator();
	    Proc[] procs = new Proc[list.size()];
	    int j = 0;

	    while (i.hasNext())
	      {
		procs[j++] = ((GuiProc) i.next()).getProc();
	      }
	    SourceWindowFactory.createSourceWindow(procs);
	  }
	// else
	// {
	// if (procWiseTreeView.getSelectedObjects() != null)
	// {
	// int j = 0;
	// LinkedList llist = procWiseTreeView.getSelectedObjects();
	// Proc[] procs = new Proc[llist.size()];
	// Iterator i = llist.iterator();
	// while (i.hasNext())
	// {
	// procs[j++] = ((GuiProc) i.next()).getProc();
	// }
	// SourceWindowFactory.createSourceWindow(procs);
	// hide();
	// }
	// }
      }
  }
  
  public ObservableLinkedList getObservers ()
  {
    return this.observers;
  }

  public void setProcsAdded (boolean procsAdded)
  {
    this.procsAdded = procsAdded;
  }

  public boolean areProcsAdded ()
  {
    return procsAdded;
  }

  public SessionType getSessoinType(){
    return this.sessionType;
  }
  
  public void setSessionType(SessionType sessionType){
    this.sessionType = sessionType; 
  }
 
  public void load (final Element node)
  {
    super.load(node);

    final Element procsXML = node.getChild("procs");
    procs.load(procsXML);
    
    final Element observersXML = node.getChild("observers");
    this.loadObservers(observersXML);
    
    sessionType = SessionType.getSessionTypeByName(node.getAttributeValue("SessionType"));
  }

  public void save (final Element node)
  {
    super.save(node);
    final Element procsXML = new Element("procs");
    procs.save(procsXML);
    node.addContent(procsXML);
    
    final Element observersXML = new Element("observers");
    this.saveObservers(observersXML);
    node.addContent(observersXML);
    
    node.setAttribute("SessionType", this.sessionType.getName());
  }
 
}
