/*
 * Copyright (c) 2019-2022 Free Software Foundation, Inc.
 *
 * This file is part of Wget.
 *
 * Wget is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Wget is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Wget.  If not, see <https://www.gnu.org/licenses/>.
 *
 *
 * Testing Wget2 cookie functionality
 *
 */

#include <config.h>

#include <stdlib.h> // exit()
#include "libtest.h"

#ifndef _WIN32
static const char cookie_jar_data[] =
	"# HTTP Cookie File\n"
	"#Generated by libwget "PACKAGE_VERSION". Edit at your own risk.\n"
	"\n"
	"localhost\tFALSE\t/\tFALSE\t0\tfoo\tbar\n";
static const char cookie_jar_data2[] =
	"# HTTP Cookie File\n"
	"#Generated by libwget "PACKAGE_VERSION". Edit at your own risk.\n"
	"\n"
	"localhost\tFALSE\t/\tFALSE\t253382024933\tfoo2\tbar2\n";
#else
static const char cookie_jar_data[] =
	"# HTTP Cookie File\r\n"
	"#Generated by libwget "PACKAGE_VERSION". Edit at your own risk.\r\n"
	"\r\n"
	"localhost\tFALSE\t/\tFALSE\t0\tfoo\tbar\r\n";
static const char cookie_jar_data2[] =
	"# HTTP Cookie File\r\n"
	"#Generated by libwget "PACKAGE_VERSION". Edit at your own risk.\r\n"
	"\r\n"
	"localhost\tFALSE\t/\tFALSE\t253382024933\tfoo2\tbar2\r\n";
#endif

int main(void)
{
	wget_test_url_t urls[]={
		{	.name = "/file1.txt",
			.code = "200 Dontcare",
			.body = "file1",
			.headers = {
				"Content-type: text/plain",
				"Set-Cookie: foo=bar",
			},
		},
		{	.name = "/file2.txt",
			.code = "200 Dontcare",
			.body = "file2",
			.headers = {
				"Content-type: text/plain",
			},
			.expected_req_headers = {
				"Cookie: foo=bar",
			},
		},
		{	.name = "/file3.txt",
			.code = "200 Dontcare",
			.body = "file1",
			.headers = {
				"Content-type: text/plain",
				"Set-Cookie: foo2=bar2; expires=Tue, 11-May-9999 07:48:53 GMT; path=/",
			},
		},
		{	.name = "/file4.txt",
			.code = "200 Dontcare",
			.body = "file2",
			.headers = {
				"Content-type: text/plain",
			},
			.expected_req_headers = {
				"Cookie: foo2=bar2",
			},
		},
	};

	// functions won't come back if an error occurs
	wget_test_start_server(
		WGET_TEST_RESPONSE_URLS, &urls, countof(urls),
		WGET_TEST_FEATURE_MHD,
		0);

#ifdef WITH_LIBPSL
	// save the cookies into a cookie jar
	// because of --no-local-db (in libtest.c) we need to enable cookies explicitly
	// foo=bar will not be saved since 'localhost' is a public suffix, so expect an empty cookie.jar
	wget_test(
		// WGET_TEST_KEEP_TMPFILES, 1,
		WGET_TEST_OPTIONS, "-nH --cookies --keep-session-cookies --save-cookies cookie.jar",
		WGET_TEST_REQUEST_URL, "file1.txt",
		WGET_TEST_EXPECTED_ERROR_CODE, 0,
		WGET_TEST_EXPECTED_FILES, &(wget_test_file_t []) {
			{ urls[0].name + 1, urls[0].body },
			{ "cookie.jar", "" },
			{	NULL } },
		0);
#endif

	// because of --no-local-db (in libtest.c) we need to enable cookies explicitly
	// foo=bar will be saved due to disabling PSL
	//
	// --no-cookie-suffixes does not load a PSL from file, but uses psl_latest() built-in from libpsl.
	// --cookie-suffixes="" disables PSL completely (or nay other filename that doesn't exist).

	// save the cookies into a cookie jar
	wget_test(
		WGET_TEST_KEEP_TMPFILES, 1, // keep the cookie.jar for the next test
		WGET_TEST_OPTIONS, "-nH --cookie-suffixes=\"\" --cookies --keep-session-cookies --save-cookies cookie.jar",
		WGET_TEST_REQUEST_URL, "file1.txt",
		WGET_TEST_EXPECTED_ERROR_CODE, 0,
		WGET_TEST_EXPECTED_FILES, &(wget_test_file_t []) {
			{ urls[0].name + 1, urls[0].body },
			{ "cookie.jar", cookie_jar_data },
			{	NULL } },
		0);

	// load the cookies from the cookie jar, checked by 'expected_req_headers' for file2.txt (see above)
	// -nc is needed for the HTTP/2 tests, else we get a file2.txt.1
	wget_test(
		// WGET_TEST_KEEP_TMPFILES, 1,
		WGET_TEST_CLEAN_DIRECTORY, 0, // do not remove the files from the previous test
		WGET_TEST_OPTIONS, "-nH -nc --cookie-suffixes=\"\" --cookies --keep-session-cookies --load-cookies cookie.jar",
		WGET_TEST_REQUEST_URL, "file2.txt",
		WGET_TEST_EXPECTED_ERROR_CODE, 0,
		WGET_TEST_EXPECTED_FILES, &(wget_test_file_t []) {
			{ urls[0].name + 1, urls[0].body },
			{ urls[1].name + 1, urls[1].body },
			{ "cookie.jar", cookie_jar_data },
			{	NULL } },
		0);

	// save the cookies into a cookie jar
	wget_test(
		WGET_TEST_KEEP_TMPFILES, 1, // keep the cookie.jar for the next test
		WGET_TEST_OPTIONS, "-nH --cookie-suffixes=\"\" --cookies --save-cookies cookie.jar",
		WGET_TEST_REQUEST_URL, "file3.txt",
		WGET_TEST_EXPECTED_ERROR_CODE, 0,
		WGET_TEST_EXPECTED_FILES, &(wget_test_file_t []) {
			{ urls[2].name + 1, urls[2].body },
			{ "cookie.jar", cookie_jar_data2 },
			{	NULL } },
		0);

	// load the cookies from the cookie jar, checked by 'expected_req_headers' for file4.txt (see above)
	// -nc is needed for the HTTP/2 tests, else we get a file4.txt.1
	wget_test(
		// WGET_TEST_KEEP_TMPFILES, 1,
		WGET_TEST_CLEAN_DIRECTORY, 0, // do not remove the files from the previous test
		WGET_TEST_OPTIONS, "-nH -nc --cookie-suffixes=\"\" --cookies --load-cookies cookie.jar",
		WGET_TEST_REQUEST_URL, "file4.txt",
		WGET_TEST_EXPECTED_ERROR_CODE, 0,
		WGET_TEST_EXPECTED_FILES, &(wget_test_file_t []) {
			{ urls[2].name + 1, urls[2].body },
			{ urls[3].name + 1, urls[3].body },
			{ "cookie.jar", cookie_jar_data2 },
			{	NULL } },
		0);

	exit(EXIT_SUCCESS);
}
