/* This file is part of GNU Pies.
   Copyright (C) 2015-2022 Sergey Poznyakoff

   GNU Pies is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3, or (at your option)
   any later version.

   GNU Pies is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with GNU Pies.  If not, see <http://www.gnu.org/licenses/>. */

#include <config.h>
#include <stdlib.h>
#include <grecs.h>
#include <sysexits.h>
#include <wordsplit.h>

static struct grecs_txtacc *pp_cmd;
static char const *pp_path;

void
pp_init (char const *path)
{
  pp_path = path;
}

void
pp_add_option (const char *opt, const char *arg)
{
  if (!pp_cmd)
    {
      if (!grecs_preprocessor)
	{
	  grecs_error (NULL, 0, _("no preprocessor configured"));
	  exit (EX_CONFIG);
	}
      pp_cmd = grecs_txtacc_create ();
      grecs_txtacc_grow_string (pp_cmd, grecs_preprocessor);
    }
  grecs_txtacc_grow_char (pp_cmd, ' ');
  grecs_txtacc_grow_string (pp_cmd, opt);
  
  if (arg)
    {
      grecs_txtacc_grow_char (pp_cmd, '\'');
      for (; *arg; ++arg)
	{
	  if (*arg == '\'')
	    grecs_txtacc_grow_string (pp_cmd, "'\\''");
	  else
	    grecs_txtacc_grow (pp_cmd, arg, 1);
	}
      grecs_txtacc_grow_char (pp_cmd, '\'');
    }
}

char *
pp_command_line (void)
{
  char *ret;
  
  if (!pp_cmd)
    {
      if (!grecs_preprocessor)
	return NULL;
      pp_cmd = grecs_txtacc_create ();
      grecs_txtacc_grow_string (pp_cmd, grecs_preprocessor);
    }

  if (pp_path)
    {
      struct wordsplit ws;
      ws.ws_delim = ":";
      if (wordsplit (pp_path, &ws,
		     WRDSF_DELIM |
		     WRDSF_ENOMEMABRT |
		     WRDSF_SHOWERR |
		     WRDSF_NOCMD |
		     WRDSF_NOVAR) == 0)
	{
	  size_t i;
	  for (i = 0; i < ws.ws_wordc; i++)
	    pp_add_option ("-I", ws.ws_wordv[i]);
	}
      wordsplit_free (&ws);
    }
  
  grecs_txtacc_grow_char (pp_cmd, 0);
  ret = grecs_txtacc_finish (pp_cmd, 1);
  grecs_txtacc_free (pp_cmd);
  pp_cmd = NULL;
  return ret;
}
