/* shake256-test.c

   Copyright (C) 2017 Daiki Ueno
   Copyright (C) 2017 Red Hat, Inc.

   This file is part of GNU Nettle.

   GNU Nettle is free software: you can redistribute it and/or
   modify it under the terms of either:

     * the GNU Lesser General Public License as published by the Free
       Software Foundation; either version 3 of the License, or (at your
       option) any later version.

   or

     * the GNU General Public License as published by the Free
       Software Foundation; either version 2 of the License, or (at your
       option) any later version.

   or both in parallel, as here.

   GNU Nettle is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received copies of the GNU General Public License and
   the GNU Lesser General Public License along with this program.  If
   not, see http://www.gnu.org/licenses/.
*/

#include "testutils.h"

#include "sha3.h"

#define MIN(a,b) (((a) < (b)) ? (a) : (b))

const struct nettle_hash nettle_shake256 =
  {
   "shake256",
   sizeof(struct sha3_256_ctx),
   0,
   SHA3_256_BLOCK_SIZE,
   (nettle_hash_init_func *) sha3_256_init,
   (nettle_hash_update_func *) sha3_256_update,
   (nettle_hash_digest_func *) sha3_256_shake,
  };

static void
test_incremental (const struct tstring *msg,
		  const struct tstring *digest,
		  size_t interval)
{
  struct sha3_256_ctx ctx;
  size_t offset = 0;
  uint8_t *buffer = xalloc (digest->length);

  sha3_256_init (&ctx);
  sha3_256_update (&ctx, msg->length, msg->data);

  while (offset < digest->length)
    {
      size_t to_read = MIN(digest->length - offset, interval);

      sha3_256_shake_output (&ctx, to_read, buffer + offset);

      offset += to_read;
    }

  ASSERT (MEMEQ (digest->length, digest->data, buffer));
  free (buffer);
}

void
test_main(void)
{
  const struct tstring *msg, *digest;

  /* Extracted from ShortMsgKAT_SHAKE256.txt. */
  test_hash (&nettle_shake256, /* 0 octets */
	     SHEX(""),
	     SHEX("46B9DD2B0BA88D13233B3FEB743EEB243FCD52EA62B81B82B50"
		  "C27646ED5762FD75DC4DDD8C0F200CB05019D67B592F6FC821C"
		  "49479AB48640292EACB3B7C4BE141E96616FB13957692CC7EDD"
		  "0B45AE3DC07223C8E92937BEF84BC0EAB862853349EC75546F5"
		  "8FB7C2775C38462C5010D846C185C15111E595522A6BCD16CF8"
		  "6F3D122109E3B1FDD943B6AEC468A2D621A7C06C6A957C62B54"
		  "DAFC3BE87567D677231395F6147293B68CEAB7A9E0C58D864E8"
		  "EFDE4E1B9A46CBE854713672F5CAAAE314ED9083DAB4B099F8E"
		  "300F01B8650F1F4B1D8FCF3F3CB53FB8E9EB2EA203BDC970F50"
		  "AE55428A91F7F53AC266B28419C3778A15FD248D339EDE785FB"
		  "7F5A1AAA96D313EACC890936C173CDCD0FAB882C45755FEB3AE"
		  "D96D477FF96390BF9A66D1368B208E21F7C10D04A3DBD4E3606"
		  "33E5DB4B602601C14CEA737DB3DCF722632CC77851CBDDE2AAF"
		  "0A33A07B373445DF490CC8FC1E4160FF118378F11F0477DE055"
		  "A81A9EDA57A4A2CFB0C83929D310912F729EC6CFA36C6AC6A75"
		  "837143045D791CC85EFF5B21932F23861BCF23A52B5DA67EAF7"
		  "BAAE0F5FB1369DB78F3AC45F8C4AC5671D85735CDDDB09D2B1E"
		  "34A1FC066FF4A162CB263D6541274AE2FCC865F618ABE27C124"
		  "CD8B074CCD516301B91875824D09958F341EF274BDAB0BAE316"
		  "339894304E35877B0C28A9B1FD166C796B9CC258A064A8F57E2"
		  "7F2A"));
  test_hash (&nettle_shake256, /* 1 octets */
	     SHEX("CC"),
	     SHEX("DDBF55DBF65977E3E2A3674D33E479F78163D592666BC576FEB"
		  "5E4C404EA5E5329C3A416BE758687DE1A55E23D9E48A7D3F3CE"
		  "6D8F0B2006A935800ECA9C9FC903D86F065367221067658B4D7"
		  "473ED54800D196FBE1089811DD9B47F21E3698B1573653ADAD2"
		  "31C39F145B586D6C0133378416138E4423F7AF7DACFFE965706"
		  "A3C49024002B53BA05871E3F066694B163630B0053BE41FA45D"
		  "4317EAA84214254038A37A9C83D62EB9EA6E6ACAFA6BBFE5AF9"
		  "F389690D5A9E27BF97C1B93D93ECF36DF6DA212E12B24481541"
		  "56E94687336B6DA91E368512B9F34C616166542923F3618640D"
		  "930F922A3DDDD2F87920378541446F2223F593931BD1BA02E29"
		  "07CE3404621F26B900D05A1119A9E4934A7CD818DD9237445BF"
		  "50475A011EA0078788801D21DFECB7A2BB294E4956DFA71D8CC"
		  "472405342BF80120FE18A551D88D6ABC24D83F077BFB25EBDE5"
		  "F4B03678D677EE646DCE3496D5138BE108782CA5A00AAFF3CB4"
		  "BB873EC0F2E932DD74847033EC5F07254E3027B0AC12DB7D6D3"
		  "F90B53D8A5BD63B99C3BF5CD38B453D7CB12D0AE2BF1CFD3EE8"
		  "8AF71BB6606B0B791DEFC2D762C8641BB8319FE17321EBA407E"
		  "B744699D92B35ABD79F5B9A85408C93D61233FECE638023875A"
		  "A02B9EDBACC84003A28CCA2D55A0742D635FDA888905986CA01"
		  "C1E6420D49EC25224D84E915DFD1638A492282F1FD053770168"
		  "953C"));
  test_hash (&nettle_shake256, /* 2 octets */
	     SHEX("41FB"),
	     SHEX("B64ECACD5F7499ACC085C908D35DCC1FC0131816F28D360592E"
		  "1265079F92A5F844C4BF6AA50D98D52720797E8C992F43C76A7"
		  "3FD95F9BC4CD272157842ADA2518190FCA342DC20D0C57CDDF0"
		  "1B3DDF77977EDED63445E40BE82DF8D26DB629A2D307EE9FE28"
		  "D2FE557E3971858C6D67C42BE2CF44DD7570521CE0647446742"
		  "5B7AAAE39DB90945BAD388009ED5715C684BB4E4981EEA324EC"
		  "F66584AD08D9F27C6A4DCF615591857BC7364E8A7C136661AE5"
		  "FFE828C734DD5EA5A071276E8477B8525E02B7B445D91CC6E37"
		  "D58740DC2B069BE6D92E7DF95C1AB52B76F7761AE34328962EA"
		  "C7156E460B3C04FFECAEC8722A56E7373285E42D4CAC5498F8D"
		  "7DD5ECDA9F9973A32F8D425171E1390BFC812C9EE4D4AB8FA9A"
		  "0D93AA90A4C258FC64D77BBCF49977E87C3810C80C458516899"
		  "6A31F446F9391A193B888CD321E22E9368F4F11495FE124141C"
		  "04015532345D7CB0A13A4DD9007D737B3A176A88E5FC153D4AC"
		  "2E8CD641C40C4261BBA70E1B87114030FF67CB22ACEC90AC288"
		  "D6B59D25B00038468B4780254FAC4EF158EC2CD52C0AB9217EE"
		  "D1ED0A5E7B4C4B3A64B1824E2B27AA53398765D5352BD1ED0E9"
		  "C7B3FB264D141741659F7D8FD0EEEC9F9163C42AFDB540D5F2C"
		  "87833880A0C942AE4CCEA7FFF2F4C798B8AAF24C33BE8054A09"
		  "459A3AF7200D555334241709A18ECF88CE93C99234D6AB02859"
		  "16AE"));
  test_hash (&nettle_shake256, /* 3 octets */
	     SHEX("1F877C"),
	     SHEX("F6BF0397DBFBB20E4AE30F0A47FE976CD109B3AA09B0E3F29F5"
		  "60E4ED333C0D083326B03F6EAEB57E277BBFE1CCE36C499434D"
		  "838CB4C8CD8B02A87790F4A6717B22D46F9220391C420A1A1BF"
		  "AA9ED5B85116BA1D9E17FF16F6BCE6704C80A49FD9AC42689DB"
		  "0996C6BD3266694077C6DE120043A827D44979CE8CCC6AA7E53"
		  "08EBA64ACF9FFFF51D36BC4401F8117D4B96340C62D106B0A64"
		  "45F01987F9C4C0A420E1A9BAEB594BCB1BDBFE59B6065EB91CB"
		  "EB252473C7858ECA475E1C81E8425C7E2C1706C4C4ABB3AEAE3"
		  "9332479ECDEFDFA93C60EC4007A51C5DD093B527264155F2202"
		  "E01D2083D27D71A6F6C92D839E6EA7D24AFDB5C43630F1BD06E"
		  "2B45B2C0D0AD70BDA111363298AB9754F26DB00A3FD93D72981"
		  "DD463DEFD286320A274F5422E9400DC6E7C78D75534C75AF6B0"
		  "398596EEE5048919B8FE33467C2B7C227BC88994B28819815D8"
		  "21DC27BAD0458D57F40B12A8A933729A0F6BAE74DBD150325D1"
		  "4C302835AB95DABDE87A2ACEE9347609A6175DBB538FDFB278B"
		  "E0B3D5F062768F9E0EB1AC9C36B4E05E3ADFCA73981CEBAF6E0"
		  "47A18196EA78846D9A782B6E1CEE2036866DBCA749644E74EF5"
		  "A4538D4AFB1706B360F734814313D20A1AC626BEE7AC0FE9F27"
		  "7D48BC50816AC9743A61E32F8B26CA6F4DC81FE7F3558AA2F18"
		  "586D89A8E3070868C76B4CA4727F541072DCDAB3D547D52ABB6"
		  "F356"));
  test_hash (&nettle_shake256, /* 4 octets */
	     SHEX("C1ECFDFC"),
	     SHEX("CE7FBC15503986E3B84530D84A16EF64332A6EA57E354E9F205"
		  "4BFC2AA8891F94B4FDD42FD227C94B33F9AD45CF3982239F81F"
		  "BC2CBAC7809F01E1C294073AD6D2821C1283C5C427F41FD4645"
		  "5EF7B4EA0D6D5E249AF95FAC80A8678C1A5318443E63122DDFE"
		  "D2591FC690527F97A09920941B02439AF3BA36B17FD588E187F"
		  "CBC1FF109AB4A9ECFC671926EF0CC9C82EE06EC6759E2758A88"
		  "59B8FA9DDF46D6C049621FF589F3FF56C9419D6F48A68B68FEF"
		  "D068ABEC24824D7FC150277C2439BF78D15D59DBAA2CB17E528"
		  "2E6E9ED744841F4A4BBB778CFEAB0471CE850B2A2F948DB3926"
		  "F733EF7B3AA9AB8EA84278DCA62B0691F5DD13EA11660A8E2FB"
		  "5CD8BA74A352C044E7219E750F29B62F94ED268A68029B94B64"
		  "F3D4F2193A7FC69ED34A59657471F04C4961084EBB581ABCC9D"
		  "50E3674A7DEBB285FC7215BF6606DB1392A7F1619B347A4D07D"
		  "230DA07811FDE2AEBE45A70178CF841C619F712EF26BEEECC41"
		  "83A1040076E73FCF6FABE5C25A4B71D564A97CF3C88F1856D9C"
		  "8B42E94F746CE4605D2AAEB56D1DA5527598E17E5680D309788"
		  "E09910BEB74DF7D3C3CD4EC680083F5845F4A1C15070E57979C"
		  "01B89DF7BE64435EA4817BC0AD23ACCA6CB878F7131F02F9215"
		  "E2D5C12CF3B4D24B29620C891A54AC8BE6E3DEC08397887DE0E"
		  "A86B81A5459B968FBAE18B4B0324DE7E7C1AEEFC7598644CE1F"
		  "F8F9"));
  test_hash (&nettle_shake256, /* 5 octets */
	     SHEX("21F134AC57"),
	     SHEX("BB8A8447517BA9CA7FA34EC99A80004F228AB282472841EB3D3"
		  "A76225C9DBE77F7E40A066776D32C74941202F9F4AA43D12C62"
		  "64AFA59639C44E11F5E14F1E5695208DB78F37CF3AC172469DE"
		  "67B80014D1A4B968A56F855BAF35A275B8EC77B9CA591257AA2"
		  "EF048663CD05D76B72CF3A9D2BB44B154B16E95EB8B61BD8415"
		  "A867367F0031BFF5D49237157C197473309FDAAFC8146AB9FCD"
		  "4254477B052AB306BB575ECA6872863B5F43F026C2BE3289E85"
		  "7D50441330EC02F6AB4C6329B18D3AF0A1ED9DB70F34016EA0C"
		  "AFF1F0EF08BA6D26566DF6CCE9A457E1900FBA3338A771E40CF"
		  "B39581F16019D521E4F398539C4A6F2CA22ECF9158B96064C0F"
		  "26B8A076E58B5F0BA3187342A5C9A5569E6D08A2AC27258CF04"
		  "26D995724579C723A0FA8269B4239026A08D88738156943F1DF"
		  "F6E0FFF5E4696149436CD97732424366626CE7B67A3BBE480E4"
		  "25B51872AAE7CE59E70995C194E00FEE82119A4320A4B91158D"
		  "B65B83195C9B799D3EAF3CBF8593D11DC67EE8C0CD7CA4FAE69"
		  "D9F46A7E70186A33DFAAD510D8DEF82AA5710B0ABB70DFC396E"
		  "CFD33F56F66957B82A220D1ECA09166F72358A71054A179C196"
		  "1B64F26A3D7AAFB030C28A73337862E8364D3F562B324CA2EBE"
		  "F58B7A8F089213178C8D0ADDB5A18022154CF010C722C279AE6"
		  "0AA984597EC6542EE1ABDCAAE8D5545FF45E781ABC7145AE2A3"
		  "1249"));
  test_hash (&nettle_shake256, /* 6 octets */
	     SHEX("C6F50BB74E29"),
	     SHEX("69401EBF1D4C7EA500BE60EF06783DCB009A13B4FCD8B341E5F"
		  "2E40F39E732237E86716F26873C1820BC36D20F6D5E64A9C5E6"
		  "36855305CBC3F4BE4A9905B5F79FB769A794299EAFA8E81FFA5"
		  "1BB4D602480B1C40F9EE9BEE017D5D7F6DE68C8A79EE92A20B9"
		  "DA5BD3D01BC565CDD3E5786FC4D48F719C01CD6D3CAFCD500F4"
		  "F768600E6891102FFC10F1394E3A40A1BD54381775BB400EE6E"
		  "20B95420FCB2C86BBFCF380BC95EEFCA33BC8DE823E13B4B04A"
		  "7AE1FDEB8279F3EF3E4FD50B2FDE530B0D4C8DABDA6F225A412"
		  "1A79FED79D8A013C9DF1103ABF081294FCBBC5ABF20BA38E007"
		  "5169A9C13CFFF94AEB271A46875710A60CD78915C33892D7FCD"
		  "6C97A1C74846ADDA847F58ADCE8EF762D11C4F15FF8B97F9BED"
		  "C97E1A8AE554C0B9E66BA1D48BCFAD0D56F91B65E25B03A5E48"
		  "CF4DB0DBC467BBBC1E037743991AC72B4B259D65013A2708EB8"
		  "25DFDC65093A4F4F5E98427DE7C60138467D8900ADFCE1F2740"
		  "8638DC66957D27CB774BBAFF0BEDEE0F6553BCBAC9724A20E8E"
		  "D0CE6B2F764E4F0D80E4521788A5CC88C3D36F19925AC2F205C"
		  "F6EB7E6D6888EEFD0BE49BC8BFD8A80B9E05DC7513BEBA7BE52"
		  "D39BF9BD2B2DCD03FDC8F8E7F90A83DAB2AD85F51D040BA0F1C"
		  "0A9DC38E929A50DB7F99A26836612E601C6E4346E5E895EF4CC"
		  "FCD6A5CC3777FC11FA64A9AC27D2CDA67E29CB18A7E99E2CA07"
		  "30C6"));
  test_hash (&nettle_shake256, /* 7 octets */
	     SHEX("119713CC83EEEF"),
	     SHEX("C02561C12CCDD167CA959D9756CC70946F7FED8BA705E3EDC43"
		  "3D3C45D9299D0AEFE9E8E25D602C4DB0D14ECAEFDFDFED2DE13"
		  "4AC5D0C4DFC02ABEFFFDD7667A434936151D529A93CB266100B"
		  "94AD0449597B1590398A1A63C42649385B4CFAA828C89037E0F"
		  "97BEDA8450A68520143889A92C258645664EB57CBA01C3B1134"
		  "318E11A1848D912D0D2CBAEBD59A71B52495671C548522F4013"
		  "94FE48DB49956C03044CAFBD40371EF657F1AA86E4969459BBB"
		  "D21DEA563A51482A47147EF85487A5CCDBC532BC2AEBC3265E7"
		  "1BD8D56F4521CD93354A73D175FCCF5AABEF2768DE2D8E851D4"
		  "FC21FF032829411DB2668300853BA5F8BB1FFDCE3FF59197BD1"
		  "833D2ABFB8C3027B36B21969F7E36C3B6F3FABA9454ADDECDAF"
		  "C213D347A44B0D364997A9E608BCFF3D2004139CD05005B9BD8"
		  "FEC09E5976EAB977B5DE8E39373C10D9EE2D2EC196B33689783"
		  "EF3ABC3DBCECA9BF33E8DEE9A1AFDEAA71A2FE42A3B0C60781C"
		  "9A2D5714278CC7CB349A3B6E7251B96E4E2FAC606B459C28AE8"
		  "1B551168D28ACD01D4A08416B6714D8F82485A1C6D5348F7DC4"
		  "981AA38E7F39EDABC981022C4CD3345383D8C25C929FBF66026"
		  "F91664998BC34F9689375D06FF95D208D4ACC7791633B222ACF"
		  "D3EA26F8B5C4858997A1FF77050F280AC7B7E5325ABE9E55FC2"
		  "7EA8ED1D9D32194B0C5D8807F94AE6F737276065CA08C323E9E"
		  "D421"));
  test_hash (&nettle_shake256, /* 8 octets */
	     SHEX("4A4F202484512526"),
	     SHEX("8A804FC3A4FEE6CEF6808C756A17BCDF6B235808493FC78E797"
		  "01E59A9E9A67D60F0F256AA69DC0258A2510F9936AEE2BCBD0F"
		  "679696F439F37BF9AFB170D44A58DBCF71EFF14CEC1E624D3D8"
		  "F1DDDDD3E33421B1E305C794C2C88FCADF9D7C51F52E4352BF9"
		  "C9C895AA457F5449E82FB36E2A64D1AA69771EF3D675F09030A"
		  "95653837AB2237DAA7FCEC36651355B2507713CC6A2186E950D"
		  "E82DC699122644193F8F231C607BA7F3BDD0A214B4EC0155D30"
		  "517F681FDC2A89D3143040BE0E0B6DC7E5185C723464CCAA2FE"
		  "321AF3B4174283355FE3D1CED518E00B6063DDD607B166BA388"
		  "978294244EA8EC773A528003A04914B76E9BE3337D806CA20C5"
		  "84B2BB66AFCD144417E3D93243F185D26DBA90EA39259C7F23B"
		  "8A6427AA8A7622F277605F4A463F78A0C189C8DE2C553AE2077"
		  "3D7CB4F7E26A13F4204B3752D9CEDDF29849798479A4BD0D7CE"
		  "4D22CC51F88127435BD161F026B1811A723E786DB1DC01F921F"
		  "E076C3532FA969EF1F8993E0A4FB6C17597D8DB38DD7AF259E3"
		  "22751CC0A1CCA2EE4940F4EA56CE9179941CF4696256CD04AB8"
		  "53266D12E7E679377D54E2C2F24832974C573192DD2FDD4DA5E"
		  "FD72114109248B03F0AE03123252FFFF977BDE87AF8D8022C4C"
		  "51DA68EFB87ABEEDA7A72EB4D0D1A2EB65EA4CEBC7CCABF3787"
		  "B9BE98E14E4A273635F69E3E4BA557A1A42D1BF69EBD359B895"
		  "320A"));
  test_hash (&nettle_shake256, /* 9 octets */
	     SHEX("1F66AB4185ED9B6375"),
	     SHEX("7B5562ABA86E6329693CE1CAAB2506AF93AD28BE4DE045E66A7"
		  "B277C4DC3C2C9E698AD5F6D8826230A38FCE87142DD83B28F05"
		  "5FA7A9212F5017374A7EAC5BB7824C1540597DE34BE9EC894EA"
		  "BCF751BFD5DE029A1CADA7E599D433BAC0499AAD9782579D282"
		  "23A5D16264EFBF55EB1370748E7DAC4B56255468FA9694A93C4"
		  "B3759F691D9A4A08B53A15F61FAA1268E38AAEB43B3FCDBF0BA"
		  "003E2037EB52C8E9931DB9C60FFC1D1FCEE7663D1017AB6DCC1"
		  "B92472A88A3D568AF5AFF930AEADBAE751753CBF2F987338DEB"
		  "1AA822FE28788866073319662438B6CAF6BD8D79DE14FDB6D30"
		  "F7983F5711177765D7D5E1EFB610FAFC9DE733C9DB3FD2F1C35"
		  "421D29CB96F87A0DD1E83AD19ACDC7C2BF7A6F047994A96E6D9"
		  "1BEC34AAAF18AAB7B32A0B379A60F12AFC697770CC6039031C1"
		  "3B5DD66E0D368609D3018ACCFB02FA4973BDA707CDA096D712F"
		  "AF59B43BE3EF5EB7C227530C987DC4991955796B2F588359112"
		  "B9E7566CA785CC5706AC333D12E2DA37A97A13B69540084D1A5"
		  "908ED3BAF73A0E1C55A6BD036F1342F8EC671593B9B555DEFBA"
		  "BB3364914E4396D6E2B2ADFA951A6A263AADA6D879B4FF330F8"
		  "48D2598CB418E530BC63BB11E7A247C5A2C21D5C2A34D237219"
		  "55567E4452DF9F5E2108C1FEA8FBF672422BB3B85C7C5664B91"
		  "8D3F3644339367CEFBE609A9B00C250111801C3F365BA752709"
		  "B76E"));
  test_hash (&nettle_shake256, /* 10 octets */
	     SHEX("EED7422227613B6F53C9"),
	     SHEX("C92FDFEA927EEEE73162C436C6A3FB97291C33787BF7C2CD80D"
		  "ECF6004C043BC4509A2CFAC1FBEEB0BCBC7C2974B168AF3291D"
		  "EE1AF2D0826935DA79C5BF7CF3A211F347C2A44630DCC20E964"
		  "E3BE24A20766A2EC50310F4B1990A3987FD8D144D3A6F091C6F"
		  "8ECB7B2069905A7A1BCE8A49A9779E9132BAE521FC44689F786"
		  "1B7C735815448984D6809C2138ED64A45C4320E09BF459A985A"
		  "41D024DC4CE0D95681DC9379FC4DEE3C024436E7A949589A74B"
		  "434F6B5D5742501D1E4761E0DF89F4F1943614C5682086D31F6"
		  "EB2B65A005FCA29B48777748132A15BCE74228E5CDCB95C05F5"
		  "1730E8E27E9DE71D582A95AD3E78E64B978D872B91A88C6175E"
		  "C3F92D5AAC14BCEFD0F6D8D5E2E19DE467C530FDE86AAF6FA5A"
		  "E0254EA01D70860278C24BE7F7C52669AEEEC52C606FE664D7A"
		  "8E857EB64948D663DC11E9DE33A410DCB3EB61FDF6E9622B66B"
		  "9B9F4A39C4B578A8D346906D14AA713D2B266DB5B85AED05186"
		  "CCB2809B38BBB3C9F99C2861793FC70D972B51F2199674E734D"
		  "46FE0E8EB675777AC9E4B578A7DE4DCFB5AC1E7A2EB0067237B"
		  "985050D66885F85CE3410467B5CA374A9F420D8CE5DA4A4934F"
		  "E78065541C53D071C88F0FF132EEADEFBDAC5E98251C597930C"
		  "DB32D79656EB44F95C78BBDAFAA59B7E36DA08A58AF3DAEE995"
		  "36EFB56319643FF422CFB53D152EA302A6458F58B5EB9F2C81C"
		  "31C4"));
  test_hash (&nettle_shake256, /* 11 octets */
	     SHEX("EAEED5CDFFD89DECE455F1"),
	     SHEX("F772DECEFF020D1BE280E7F93DF5056577E78D104CB1E2AFD42"
		  "2B021C6206D94AF56EB9404F8294403173E4A6932A1576BCC47"
		  "580B6CE2CC80B20FFB274FAC017FA05B3D804A6CD8E874CF258"
		  "E5D46CCFE06D76DCC18C9874CD3B7543973EBE3367AD38E51F0"
		  "A46B92D47A68018A819F2D90724EA9F5FC51F444003757B028F"
		  "BF96F5413296AA9BB8532773EB8CB7CFC803578FAF52082D4BB"
		  "8AF606301FA927A94FB62A73E2E9FE367E782351185516C181A"
		  "A674E19301A38D1375E1658A181DBF927973C02207BAC3A0B65"
		  "C0C6904E8195A3C6550D13CFD46C1885987263E82CE56485759"
		  "BFFB2647F420625F2B483C2750FB0B4707A2014FF57B1A42375"
		  "B37B444B0811CED4CB93D5C45551D5E1C8DF20BCCD9437B7E43"
		  "9156DD9AD4D905393AFAC1347AB2979811CD1BEAC7C454C4B71"
		  "66B85BFE52FF6C4905C55D83C15C3A9A7760BDA9FC4B90EC4AA"
		  "BF9CB3660A883BE7137733A1477971D479049268ADD74E0C821"
		  "0B1A9FABA84FB940F62B1193CCAAC3F020D10CE14DD8C058C46"
		  "BC35846C07A10DAE9C03C3EDC323CBCC83987DF1FB2ABD141C7"
		  "C0694624AA68DF9C306D2E179FB4D603F42C99EA369C90C1764"
		  "95104FA7DE772EA71A9FB014B467CA220F9C004F287F8F0786B"
		  "3D47C9DAF3FEEE80B2CE4A92AEB0FF8E8C09448DAD76F5FB01D"
		  "6997EBB139D7FA5DF8C0BF12BBE0AF2B6E354C4089F32B8C529"
		  "4634"));
  test_hash (&nettle_shake256, /* 12 octets */
	     SHEX("5BE43C90F22902E4FE8ED2D3"),
	     SHEX("00F3525289B33FB3C179398EC76F2E31B824A532B2AA6B2DA6F"
		  "FAD5287E07F9F9A2FDD2F2032AC202119CA0A01B7C9C6DE7248"
		  "77FFC874D5E578066A4BE43757BC2D9B09AA7B483749E0F8858"
		  "2CF9CC3D47B8A0A98D70AC793326FC5EB6533F00EEC34BFAB35"
		  "18FF4E98B79582621C2BC7BB7AC8AFAA8D60DF3F072FCAAF514"
		  "FA7FD9EFE2B1FCD3CB96A7F70A87436922584B93EF74BB03254"
		  "16EEFEC14EFD150432366C1C1E5F7949C2573CDE16A24C42918"
		  "C919ABD988F8951207149F873F78D2793FD642F3B3E93041B88"
		  "55B8CCF62A2F6A2B8ECE4DC704CF693069BC789D470FC4938AD"
		  "23D5B5DD20D13E31345B2169F750E69C4913E3DFA1F1E54A301"
		  "9A8333B6648659A6A2E93882FC4F8AE0234FD149123D831F148"
		  "213F578E8961DD07CEDF94907ABADF7EFD84500E501BF2588E3"
		  "B155E7D24815DA9584595EFD2FC6F1768F651C1EB10E0029553"
		  "4F89A8DDE9D37C05F7E4EF9EA236B615F82BFD07E7F6FEB1C62"
		  "DFCDD4F598BDFB3CBB0FC3259ED96D52AE156B20F0F6EBB0891"
		  "27CF98F045BDCB12DB84C2EE47800CE36AB10F982231FE746D3"
		  "2FC5002700E264AAA6432E7BC6BF4F4AF41E82E256C3DD6DDED"
		  "B24B3607015CDDF78BC1AC56F835D03D8297BDEE57F87EFFBD0"
		  "66C3DE372F285EB2E7D7359684BD2072B090BB78C3EE71A45A0"
		  "CB7C22EB4081CEB3CDB89CBAF3D124AD86A9A830457E273C8AB"
		  "8B33"));
  test_hash (&nettle_shake256, /* 13 octets */
	     SHEX("A746273228122F381C3B46E4F1"),
	     SHEX("1CF1698F0702653368463A9EEF102E8588FD5E711F4BA16E379"
		  "8D7C7730ACDCE6C65EA3849CDCA9D08437BEE09F78D52397C92"
		  "D0F00ACC9D5E9450213F337B9A75697E8FBBDDCA67CCBB7F8B2"
		  "CAB1209397A4EDA1BA9AB1B44366E4F7A0D2DB47D5480247194"
		  "D486D63D72FCFECEAC8CFBDB7F8784D4CC9214B3893FC7EEBC9"
		  "7A33B9DDEC9605484C96BB77CAE4D212F16229DD507622A01C8"
		  "61043320D9C06373E1E0D9649D95B42A990B0B447ADFBD307DA"
		  "D1394C30D12E010AD5F6C8ACC729904CCDFCA2162C32C5EE565"
		  "4316E10A97A192EC23BAAF594B711277A301FE6EEB4B54903D1"
		  "BB736D4788F65DBEFECB4761685C683DB56C05D0F26F179010C"
		  "B323C2FC8B9A44DD4A172BE2228C6B0A0779B0637E6B5DE0477"
		  "76597A17FE94D86AAED911A1DDD27F8F61710CCA8C5F38504A5"
		  "0F01304B559419F06B5F043DFE984B4DEA2DB6090C1A836F26C"
		  "0728048C0D89401722B9576577F11700BBC5A6BBF1C23C687BB"
		  "478112DA0DDA52C1ED622D0227382F76EDAE51DB2FF14C098BA"
		  "E8D25A2A53D98508DA98C99AECDDB7EAD8DA98AE41BF21A59B3"
		  "FE4B3DD68F0FB15242EF3056FBF77462D4FF67E0B344BE02CCF"
		  "03FB98F6DD5F6D306A4C2B2451EC8AA2933172689BD11A7E391"
		  "1B236905D6D8CE1C030E754A0B493CFBAA39123B0DC4B7085F9"
		  "FE5988B0447B706CF226EDD34B644BCB591A002A08CBA030F6B"
		  "1927"));
  test_hash (&nettle_shake256, /* 14 octets */
	     SHEX("3C5871CD619C69A63B540EB5A625"),
	     SHEX("413131ADD0DBA2B0A48443D3399896AD65BF5F4426D234AE20C"
		  "9D2DB3D9DFAB81401D7722842A58312F837E7DE13069802F58C"
		  "7CE9A3AD1703E9C7170D1AE751023147464CF8694515AB5E268"
		  "36B935ED493B9D66D9229BB0B66F1C5AC721D8AEC1F883DEC33"
		  "D0380BA79BE198A6AA1F2CF555818D54CD28C0682EB44FA0F18"
		  "10DF50A8F005557C9BA52F02D70EF769EE7724B852A94E1710C"
		  "6758307FFEE309C8F884DD0597012679F7C97CF59F8A41B06A8"
		  "45CCDEFD5CBDA268885A2781FEE48FE4ECF4D0559868607F352"
		  "4AA25D1A5CBE5C33FE0E8E8E47B6705203D49CAE7F1749DA49C"
		  "BC469B8870C1CD7D029F6398CF7AA0D3114F56F78885093A59C"
		  "0C188ADAA40976827AE1000BD6AB4C7A2154D80AC6786CC23AD"
		  "EF333A2C538152584045A87E942B8033C565506BD319AD99C85"
		  "A4655F6E47979F55B5F9F9822F9A338CFBAA50D580E27953BA9"
		  "D13BDB3FDB480226EDD2A8F8F56C75015714A51EA63C73FFFAC"
		  "63DC3FC6FB8D509BC40B87B9332FA37DAAA441D05984BA56781"
		  "AE25D324E508B300E6B07629EE5BD2DE33D3BBE6FDDB699E35C"
		  "324FDB521CDEFCE30D01570AE82803B0B54A7C1AF4B45121461"
		  "A848477C11F535B700633D88A68AD2ABD9AFD09A4DDC1BE5511"
		  "B574CA6E525EAC7A6CD7C26EED3E5656142AB8B24D97A76FAB9"
		  "4EDD8AF21CA7170A98660A5CBDA15925E67C5C579363DC4F697"
		  "0C95"));
  test_hash (&nettle_shake256, /* 15 octets */
	     SHEX("FA22874BCC068879E8EF11A69F0722"),
	     SHEX("2FEAF391D45B2B8632EA10806E35D465F2F767DCCEA9304502A"
		  "4CF9CFD85F7CD46BF0E5CDF2C31A43C6AADEB0C4683B3FE54E0"
		  "42C014457D0542A8AC09529C1931FD1CEEB0EEF6C5A1D28F7FD"
		  "1E7998537D78B9DCEAD530B0AB257E273432D3D7205E451CEFC"
		  "6906451DDDC6B3B0DA1D1F2DF3F043396858E7619B8A420D0CB"
		  "2CDFBBE43E19BD2BC670E2F5DA51A1B8141F7FA0FA79538FC90"
		  "9D31847C46653C8184114505C51C49C844EC18AE5C6EF782040"
		  "0D0859B3820FED6B023D1A3455C2BD6E1A7D25169B58BB71A82"
		  "107893F27E994BA04C79D209ED97E359D02B991E402CF0D14EB"
		  "61D4C1ED931B4526D63E9E9517FAA5BC83A23521620FB5B9E9A"
		  "C1CDF45536AF54E567F5D9BC31196D23C58C0C70939497A3E11"
		  "FA544A37991B5C06B8D7F57D3057E83C3EAF1758EBD0F569E5A"
		  "8979D340740A9FA1ADE5A9D956F171D9A0D0EFA871E98AE8D5F"
		  "3D5733FDF0388FD8D9564E6C267C6438F0F78D7A224DBADF7E6"
		  "E645EC6952B96A9D6FB441CDD640B576B2C92BC7C5FB11CE767"
		  "B0105DE7907DC00DD4F8F25DC85B6A84083500F3127708ECC12"
		  "50AA15D350437664D0D6ED61E83DBEB951260C1746C0F06423C"
		  "4B84FE75A752E7F034B2584920288EE697926B9BC87FD086299"
		  "2C3D9934876A978744043D8949273A3583A7C5DDBC25DDC0C7B"
		  "EFBDA1F306C9EF7A2F21E15543EBD5EF87C39814C91E1B60B07"
		  "9700"));
  test_hash (&nettle_shake256, /* 16 octets */
	     SHEX("52A608AB21CCDD8A4457A57EDE782176"),
	     SHEX("57119C4507F975AD0E9EA4F1166E5F9B590BF2671AAEB41D130"
		  "D2C570BAFC579B0B9EC485CC736A0A848BBC886CBAA79FFCD06"
		  "7CE64B3B410741AB011C54422568089405BF1E8DDD9E3BCEFE1"
		  "A713DDA18CC2B73FDE0EB323FA7518DE2808C875A6C55111BE3"
		  "E0CD20663B794048F5FF44638EF871FBA0F4C2ED41A96D36216"
		  "06740935E9EA1ABEFE15A1A3BD55C8AE6B2C021CC772B34DA21"
		  "9115C8F6072A09F2B718E26ECD2538E5F12068F577ED7D03A2B"
		  "BCC7CA7DB81D2CBAEF2AC8F33B1355798576CD3545B9DC792FD"
		  "BB9C8D1C123EE0407C6328E09103FA6CE1B4DC9FFB0BE7236DA"
		  "B3ABD29E704D0C352C524FAC14E12FB61929D98DED973D7E878"
		  "5A8ACF52AF56C01CE62AD93660C93B683FC22088D7302F72D0D"
		  "AE54553B0C3E6DA7C498BEB3AA46E7459779A1B0E1FB19556A7"
		  "1635B404C0BBBF3F34536F2776FA12206513FBB1F2A11CE9683"
		  "460D22778867ABA7335210D817B720B3D8A8C48243D128EA2A4"
		  "BA8996D160351194C0AD3988ED0AC5ED61C1F576A33C914C2BE"
		  "BEE0EEBE55878E2B43A51E510251068E3C0F7C7292189573EB6"
		  "AF979CDAEBA8B8E359E6B632BABAFE3528773CDD4A1861B7AB2"
		  "532113F2B259D45598A76D54C739DC2F4AF2700F3B5CF22431E"
		  "D9F73D53CAF41D134F5CC67ECF8F99549C091CA6697FF20E08B"
		  "F6AE9B6BE74BC77F26DB50D25F48E67A94DD705521F02D3CBD5"
		  "FD56"));
  test_hash (&nettle_shake256, /* 17 octets */
	     SHEX("82E192E4043DDCD12ECF52969D0F807EED"),
	     SHEX("87F2A509E9D42B0F595662CC93094F26F06A86D79D98398B7F8"
		  "7EE2BB5FB25384F9FE3F7A72AB5DE21CB3B7836F73618307242"
		  "D53937C67CC5F36A7552F917DB12A4364736B08671314EBCD3F"
		  "34C670E0BB066D3E8792ACE223608F5A152ECBF9FCB4242FEDC"
		  "1E27143835F0C06320174E7EEEEB60F6B5A8DC9AAE97ECA494D"
		  "79B3DDFDDBE91D3294AB4C3B3CBD1341EA4917733FB13A15BA3"
		  "4B46F0D5981C9DC457A123C4152776E15B37870C24EDC0BD9CB"
		  "73CFA5D290739C17289C2DE3A71ABA0DFACE3B76DF71CDB6769"
		  "7321772357587E3C2319C15D867A76F5C7BF5CA2F523265D411"
		  "8EAD7F96BB30349F5C4373D5E4FAB4076CEB6AB8C11CCDA6B72"
		  "72121A11F00F67D7D5AE31E86EDAB458BCDE042C58E3D3368DC"
		  "485BDABD40A3F6B7294C038B8ED3026E70ECC9EC7D6060504F9"
		  "31AABA816D5AA5EC9FA7CEFEC123E71C5135178F252885A7153"
		  "A4714E0D9C95AF171A6AAB936F0B464AB07E32AAF29CD4D9315"
		  "62F700CA1419BCA2ED1243F8EE1B442CDB1F5252225954136E2"
		  "E80A7FBAF9095108647AFF7F31D4C4CAA050EB1A90AB1D1AC0F"
		  "67B9F411006A4B76D5CB4B48D2154DA87303ED37C9F911064E4"
		  "D2F0531474E98F3F96B1A0A008DDD0AF4B86435F8B26BFC7693"
		  "C338C6AF5823FF7A9BEFA57D72E2F1BB2982DA356A8B3D08D4C"
		  "19355FD6EFB35D68B3A41144B49713A470BF61D0CA05E520447"
		  "68CD"));
  test_hash (&nettle_shake256, /* 18 octets */
	     SHEX("75683DCB556140C522543BB6E9098B21A21E"),
	     SHEX("20561970BF01E4108BC498DA4B8F900A0FA378770DAAE9703E1"
		  "66898874A1821AB0432D7CB1D0A7006E5461DD3049387C41163"
		  "27D1D3E0755CDE00A3C40BB5FCFBDE5AD0A7E46650AE7CB37DC"
		  "2F0E4F506DBA47437EEEC9F823ACD89F3EAEF767AEB39D38F88"
		  "00DA54790F362A3C183CB9BC420047EE86B3EC22E2B2F4C029E"
		  "F64A5270C21255D39363894AEE6478788436C758E4101F2096C"
		  "D7F13FE76E54CC5CD85C9183E6E1A1A3A218379997AEF9DA643"
		  "D8012F402FBBF88E6D0AE991EC4645F3118FEF0F66D7C8EE745"
		  "28D622E1E9EF1A69E39CA7EA0E045BB568980D4659ED20F0CE6"
		  "8503620A94ED7D5C9337DC20805D2DA7D0F14B5AB3B93637F77"
		  "1C9E63D0B57AAEE353AFF672596C4660CB39381A9010600CE93"
		  "197CC6A06219A3778B89B0F6AE003B5F9BEF76CCF6D551B852E"
		  "076A678A74E75E408E1D4D6B1960488FF21B6F6B7C308205F93"
		  "BD9DA1C6DA1DE9707307410BD7AE22FF24E24BBFDAE849FFE26"
		  "66B1BDBCA2E08CF9D0F3F7698C4F983D4B92EE28B5B4B8BD98C"
		  "317AEFCB41E56DD5534F59DA6C84B3D21587D2EE740F54A0C7F"
		  "AD62A63B98AF747237B68F78A39CBA4596F81A223367D345576"
		  "88B324B5335CFAA67E78A8095C92D70802A2AA76129DEE69B91"
		  "D175BDC0A485A58C91CB8EF326B2C8D1D82325C4ABE4A0F764F"
		  "C01A9F0A743E7F107CE384DC223D7AD7481AC8AB7BB273237CD"
		  "735E"));
  test_hash (&nettle_shake256, /* 19 octets */
	     SHEX("06E4EFE45035E61FAAF4287B4D8D1F12CA97E5"),
	     SHEX("F29097E10DE56C3A4A24746EE4E7A6507050D18E40F1CB8C701"
		  "B5E589FB645A4E400A3FD2C9C4D48E57BB9E146ABB1EF98D0BB"
		  "C6375C266754C7F1CF62682F0C6A6D6B11E2E0AEA41533DCDC8"
		  "51319A34FB197D56F4A35D3B82C3D314ECA59E03774E84391E8"
		  "846D236D0CC51E7A5591205C6C1738B746E616B2A8267709103"
		  "F51D0ACF3156BA42CB226BE3E94F2293E9E0FBC19BDF5717633"
		  "E17B5944A02505CC53B01325D1D77965295EA6B6B50E1294176"
		  "7F7288B9F4E6CCF3F567A1EA3D17379BD1DDC357039FB356D94"
		  "57956A8E149E30AA5092A4F85DBEE2083A62610F04FD28A5A18"
		  "80183366211287E9CDBE8B261E93557A38F6EC2D13FCDDB5DD9"
		  "05599B5469257F3F2E83125DAE5F30E8E7514D55241121D9685"
		  "08D1FB55C8F339530A437BDBBE04C42B6B229666763400F5493"
		  "E31CAAB061A67ACCF41954C2AA0A7FA0087A4B828A1AFBC62BA"
		  "09E91A3AA44FA4A74652882B7FF38C9C1692BF83FBAAA17F32D"
		  "EE6D25518FA0721463334FADFC6D050E0CB195E477150CA1198"
		  "EBE4D19ECAE1B1321003823BF04D8F9D4866BA3013C9A47FF0B"
		  "53FA6C70F57D220D86E8BFE71A61354F099245004487B65E7C4"
		  "5E090630E6F73FDC56B2EFE6B3469BD31D23384CD4A830A716D"
		  "50C7A91DD09A45A79F47D73BB32D14BBE85CFB56BBAAC47D973"
		  "A6F3F9FC23EDD774A31B533FCF7E78A2A75872C5954788097E0"
		  "4845"));
  test_hash (&nettle_shake256, /* 20 octets */
	     SHEX("E26193989D06568FE688E75540AEA06747D9F851"),
	     SHEX("BF4EEE53BAECA39E2884780A1F007B5069E7A90F1ABBD12EF77"
		  "C77A4038CA56609D52B9BEE113536D8FEC7D264F2951072FADB"
		  "F2D3A0B21690ED5478A540C02D903B3675787929AC44A2BDC65"
		  "97B2EF2956B3640B3771EDB189E98FBE8FB4CDDC8C641ADC707"
		  "D47EA2DBE17616AA0ACB4AF8D7197626577A5AB5A71AF722332"
		  "7FE96C4472B3F21FBA242416A266BD863BE3598D57DD910A265"
		  "335D17A7B51F59DEBF9F4049ABEA37CAD33B8F2032A4FD5FC6F"
		  "124044FBF2A9E47421FD6C1488DF87B942CF01F9FC7F13F7824"
		  "C7C09AAF6D739F4E3876B383546EC682DEAA6F633E630101646"
		  "BD1456CC22A7BAC50190548EE56C5FB0115809BE6B1D7CF5488"
		  "A26660E0A80DCA89EF655A963E8013A61CE27DA232430183D41"
		  "C9C9AF96C14E846ADDF3684782E207916820F57E04C60E99934"
		  "0F9630E8505C98F467E2A122FEFD1031D67789F890BE96D7B62"
		  "9828CDA153F9CC19BF8B6D0B0716F931F17014D9AC4B6D983DD"
		  "41EDFFABCF33693F7558DAB690F669C53C21A4D8DE7BC32ACA6"
		  "CC45225A2C980AEF2307A7097995D97EE060005CB965512CC85"
		  "CC81A75D3F2F0975183D9CEE293BC69A06C896634962369E01F"
		  "475098C62D574A0D6A57A5FDAB004DDD8C3D96AEFACB39B3E4E"
		  "1523447BC8FE397B488D8A5DA6E4978AED897AD0A2FB88C5ECF"
		  "86CCA7A67438B988C04F3C14A754A74ED3D03D478CE9EFD752C"
		  "0531"));
  test_hash (&nettle_shake256, /* 21 octets */
	     SHEX("D8DC8FDEFBDCE9D44E4CBAFE78447BAE3B5436102A"),
	     SHEX("A697F36886208CFBD55C3CC1C407C3F978A31574E4B9A790716"
		  "E601C6EE0ABC421AA5E5488B57E293CAD8AF40065C161407D0D"
		  "CDFCBD172FFE700B60D7EFA289C4755733FA535365DB90082EE"
		  "E0B812442E367BCD4451F152FCD54687DE4AE20F7FC60FCB3DC"
		  "D1C419EDF74251EC84471D35E4EA43E684995101ACAACDC0D8E"
		  "CF5F39840850557C70BE1477A2F168CBE97FEA85EA188594A4B"
		  "DB583EF9DD2C5B682774E7E7E0CC85C1723CF91603793C711E2"
		  "DA65869BE3A24D481C032A8674C1E0CD91A267302CC084B53C8"
		  "4AF3D0A7D350BB62A2F3B9027E1F253C24D481A34AB8D8B164A"
		  "78B0A60C95C13454BC4AE87C5E4E101B7D04C6C3F00E6BC4B4B"
		  "B922F39C87A4AD843704E125F5CFCFD8A1E8E894230CDD7DBD7"
		  "BD40A9CABABEB5BAE9CC49DB656AC90EBEE4AFBE3897B0F7A64"
		  "059BB331A4767983403DF36C919759AD6BFA379920594D0F775"
		  "BA2390A522C3FA7B0348781155EDD29BE53891B41F44E16BC5C"
		  "290C1CF31EC33E66E588ADDD70E79FB26EEE21DFDF40A3C54D8"
		  "E1CD79F2848B4238F7B803891EC01623AD92E53D3EA22197DF2"
		  "9BB6554AEA8CF2D29EC4B673FAA0A0C64675F3BDCE07288EED0"
		  "3D980C36E4D10689DD5E525F34DF78B1E76CD87F8FAC22DA1DB"
		  "1FD9533F58577D136FE161A688F3FB9CFB42D909EFB98085FEF"
		  "BD8D26498A847C9493E02E5364290CBFE45570BA39BCE1415DD"
		  "5BCC"));
  test_hash (&nettle_shake256, /* 22 octets */
	     SHEX("57085FD7E14216AB102D8317B0CB338A786D5FC32D8F"),
	     SHEX("DA7660020D2EFF07955C3724CF85B1B9533278FFD2EC05522B8"
		  "A876DB92C7F9998F0F029B2156D630DF390B3F8F5A31A8E74B9"
		  "43D6B36E1C97BC4F059408DE495462A8E039F590D5FCBC3965A"
		  "663F33BE332F9E56BF37D9108554DC9FCDEA2E771A5E22EDE61"
		  "AD5476153969D1ECF5ABB108BBAAF6323D7E570B2747EC04C35"
		  "E7DE5EA2AC0306042CAECF38B0060984460D000DDD3DCABCDCA"
		  "CC83414E56EE2C44F6D6DCF4875EEDCC4EED761B62E1A338697"
		  "0571EE127D84563034F1BA93D2A84310956C532F0EC4414BB01"
		  "AD19BEDA20C714125176EC202A9F8F9917359794239DDF0376F"
		  "AB51A5160F0FC5CD16E7B7DFB0943CD524064104BD1CCE97A34"
		  "2035FA47138C27DD9FB00F6528409E10BEC016503B17687C66F"
		  "2A00072B6245EC061598DA570F36B9FE8F4C35E3E8FCB3CA1A8"
		  "D913FEAB1890C8AEDE727E63C6085B5C196E82BBF872906C814"
		  "40EC8B0A65B2271285616C7111C702964E3232AC34E7CD4AD5F"
		  "5568C9E30B813941C02466B817CB1523D6717E58B4ECCBCD74A"
		  "923E806D82AF352395F22526E06A71585B71569E87C3D1AAFE8"
		  "C06098204042CB2E41E92117E4264ECD229A7F2BF2A9188A532"
		  "64195766707430B432FC332E160D786A9415A0260DFD91482B6"
		  "83AEC0E62B0D1A480650041EB7CDCFFDC9601AB353D9197CA15"
		  "604351FC226BD574C3B66B1C6227B7CF6577DF01520214A961D"
		  "0403"));
  test_hash (&nettle_shake256, /* 23 octets */
	     SHEX("A05404DF5DBB57697E2C16FA29DEFAC8AB3560D6126FA0"),
	     SHEX("F887A018985C64AB3D5609BE2121E73F7AB70D4A90BF47660E5"
		  "35B72DFC0A1DF20EC0C372862BBDD1C74DB76A0ED12BBC03228"
		  "3AA5FA615AF320E1C15315EA171EB224184691D13D38EF39DEB"
		  "FA2629DF136FEE74C49D61282DB8DBFD0CAC30B135330DD4F25"
		  "644B73FE8276C66325BA588C7178D39EAD3C8DB5BFB6A5F30F5"
		  "90181C84C6B88C421A4ADE52B6DFB74D1F1D82EB206FBF8678D"
		  "95442DB8292189EA349D3335C8A4B57E88C4F91B12CD0A3FB10"
		  "830783869ACC88BA943ACD63ACDDFA8634FED8339626233614E"
		  "FCE39D60DD2D8B1F6BC87799C4B7EBBB7D407C5621E56CB2E8D"
		  "3952E07D5560544F810B2BBC56054DCFA0731A9B67067CD1C8C"
		  "4951FB2AD6466098DA17C9547BC5831646AC73BBD06E596AEDC"
		  "C57804B01EB817EDAF9D46E3728F003D286B95664317361EF0B"
		  "5ADC00DCB1F6368295656DF2A41C950367BDD25BA8DB2B6B7AE"
		  "EB1161FD089D02E35BB9188E7A91415DBA172F906264905BE8E"
		  "E1E9DA5AF53B3ACFFB44437157DC275D26323402C07C77070D4"
		  "79CC89AFBA181029FB97F9F0C9DF1B4BE0F0B000A218E380CB8"
		  "4A40ABDD79E267578EE379C3B674A25A30E902400B0BFA6A69C"
		  "B7AB96661781E7CF05A6F225F301065B59C033C5D873A5BE330"
		  "7B6E59708599595FAB5D2A91C601FC75E97DAF27B9453BFE950"
		  "29ED5B7F2B95C35377E5153DE2FE9AFC847A82888E9469E8DDC"
		  "F4E7"));
  test_hash (&nettle_shake256, /* 24 octets */
	     SHEX("AECBB02759F7433D6FCB06963C74061CD83B5B3FFA6F13C6"),
	     SHEX("2DBE1B6F378F1C56AACB5F9A71A12EB3516942F2D45CD4628E2"
		  "D6DBAAA4CA282D640806C927C93B3DC6D1C96D528FC2FBDC91D"
		  "7773BDBC26B2E5429063889F0842640D4EB0A69F2C705A11FD4"
		  "0484F16A0248077897C8039C575C1A1C7BCCA09952EBB491457"
		  "6C9A9D35B2043C7A5ADE1EC97B6D46CD647815941164BCD8D11"
		  "7E5C4892B6A5A5155B066BCA61A9293E705EDD00DED6402F12D"
		  "0DAFCCF608391F24657283102C30E71777B7B9F39F069B90B44"
		  "A6E84D87660D1BCF9587A25EB59F814CFFA8CBCE7D7EAC35B27"
		  "E8707F65F5B78FB16BE7C17812857B8CB6A447BAFC1CDF793F8"
		  "3B6D29824B3C355ECD204631EF1E176B53A6B82FDAF3D224851"
		  "FC7BC0F6B83A6D05D0432FB34567A48BE56368272D2012C0F29"
		  "6324BA45B7438F13759A4A1AF293C244C5B326D6C68D28A5A3F"
		  "CEF1AE25CEEC38D866D747D62CB9B5F8CB38B047151192FF44F"
		  "FE1B35B5D08E95344E09BD7194CAC4D66722C9E365F3C0B7B48"
		  "FE0A2293E478B4E6C8E6B7C93F41637BB0FD15FDC24E6443595"
		  "FAAC6C342E8DEF79F01DFD96E624A8954B0AE93591375B8FA6D"
		  "7EDA21578A51AE0EB69D97DE7AFC1F4FA797FA6D11D5C2663A8"
		  "B09665463770000249DB5A23233BF83D36C7A15D3E8673F52DC"
		  "C06270B246086E5AD0648236C68EFBF04A311A9A090C72A6A79"
		  "116EBBA4BACFE7F46B5630F27C99F257A3C563B864933B8FC36"
		  "72A3"));
  test_hash (&nettle_shake256, /* 25 octets */
	     SHEX("AAFDC9243D3D4A096558A360CC27C8D862F0BE73DB5E88AA55"),
	     SHEX("59425FE5B259EAD7E04232543C65EB5ABD0C4DB6F9E03C18492"
		  "8E877805B9E7EC156D8C1BD0EE27E4FA931832B41781DAD3C7D"
		  "23BEA681C74251B1EC51A90077356D3C5CA2A2779BDE6F45BE6"
		  "ADC20FE6DD0BC489B3D6E43DEC4FF6E33A24C62421A20B8740B"
		  "A1E13BD6447E2D6CCEDACB692A4468C898F815DF88B8A3E0240"
		  "BCD64D2277D8456642CB129F8627EBA5380A9329932977021DF"
		  "D641498B40F8D75DB5A4F9F62ED2EFE831EBA92EA600BC95152"
		  "B3E2FDA2873F54800C15C2256768099BA5ACAD759734533E8F4"
		  "B5C729B42FCACCFEA0D6C7B5DEAD8A5A1D0D115836D63476EB3"
		  "C90325F3FFC48C964F69B2957E121D9501405ECF5CDF5C78270"
		  "19BF4A92417B1E44C444D8B714EE85154BAD6462317515CF1B6"
		  "80AADCCAD3F3B69D70875020FA770A0F7A969E4B1672DCA70CC"
		  "DB8D7122A7E8AE77CA7E0819D6CEE706D71F0732DD1AEF5106C"
		  "65AD8C66FD350C13F2D02237D2B4B9AC3DDA52E3771FE5473C4"
		  "7C77BF84DD90EF38104D8571AE25F6B7AE6B292708F0B967A99"
		  "8A6D8F66429C1C81CE5C03843877AE4C56F14FE4F5A24C8F930"
		  "72CE79E3A3BE409D902881B73D9BDD3AA0C8EDA88F0511E98EB"
		  "93B271FF5F244B956B1DA8C7A8F377F39C803355F1D70D55775"
		  "C7FBA96CC98C02999A47EEFD94E52811FAF30C099078C7C1C50"
		  "772057FE65E954FBF28CC414CBD5C2EB12AE86A0912D3812A04"
		  "ADFD"));
  test_hash (&nettle_shake256, /* 26 octets */
	     SHEX("7BC84867F6F9E9FDC3E1046CAE3A52C77ED485860EE260E30B15"),
	     SHEX("48612B3A4374024BF509409FBC6B12E049F2F5065A0CC31C49E"
		  "EDD57FFD19F1C1AF6263ADE73B4C29AA77E5C05BC0AAD40C904"
		  "106A19A8FD55DB489AC77904569ACD0D62F18BAC28A287299EF"
		  "6CACD0B342760ABAE5A35A2E67F85E854DF55C5BFE831408C14"
		  "EA2B0A3C0FFDF29D8B9FEA747EDB856DF6591B3CC76D3F4BC77"
		  "70E38FCF603F4455C9B5019B7D8A2C939340BEC5247E597CFCE"
		  "386A458BDD4BA5007D03BC82BF310B4A8CD328A62E2FA1E8FAE"
		  "543736C10633B3C70B21B98C769ADF07567C3F6D89A006BE4B2"
		  "AD21643BEC00DE1EC9F7CA7CA733ECDBCE3C58B45119E5F593D"
		  "CDC5C95BD07082CF5D9B9A1AA11530E302E1FC8D6D2FC3B0FA8"
		  "34AE3B2E98B428853C8A1B9266AEB7B8436543BFFAF69467D7E"
		  "ECE47F2DBC790FCF2EFE2791213482B641388D58B5045774308"
		  "78EF15485E437A4A1F2FEBCD994C088283D92C90975C976331D"
		  "051EA2C2EA887A20DF5C90E4FC77A9370306152FD2CA6C6410F"
		  "BD529BF9CA22B4C8FFD4444EEF136EF0B61276E42B0D0424F26"
		  "82F291C7A704ACA96B355AEDA4D382C16B3AD879BC6B6881972"
		  "94203661F21E3E881BBF4265C6B71431B5DC29F614E090D723B"
		  "A086B23051B00B7E81AE8D2315C41A90E1205CE8A1C5A744F82"
		  "B7582A099D61FC5FA6B2E9A28920E503B32A84F4831201DAF2B"
		  "AEA8CB98E034F7AB63655E670952EEA74E3DDBED01EA9DCA953"
		  "3184"));
  test_hash (&nettle_shake256, /* 27 octets */
	     SHEX("FAC523575A99EC48279A7A459E98FF901918A475034327EFB55843"),
	     SHEX("775025E1E1E0A1C865E2B57B7B43ED67FF002DF27DE0A62AC17"
		  "76F894E1DBDB2921029AFE631201298CD5D51B9A2A6E49717E6"
		  "8A180EB27B9954A807667E4D24024DC3A60A60A819EDF287026"
		  "0AEA535698F9B5D83B820013547E570847EF9F4563931138791"
		  "B44AD298217000F039A9933FF02B99AB66A571D2D5E16507DCA"
		  "9EEF54B1910E26EF361887512D646CBAD74F99DEB082F7AD971"
		  "E2E9EE37AF77B80B2A763DE07D91229E8B5A0E9BF220B0C15DC"
		  "939253BDF45CAF4A8172AFDA5D096B47793461654341D08599B"
		  "FAEE496369D5D6E37562F9AAC399A413FF892F9DC07E624DDBA"
		  "B1C72FA0F2D1AA98EC442BB1BBF7609074E61ED16D25367869A"
		  "5F54E33B52DEE5020B9ACE4DFE5115ED049A6CE62227661BDDE"
		  "DDBF77AC1D45114CB734D9CABC634908FF540C2AC162CBAAB65"
		  "5CB35138CBDF81CF5318E92516EC9C4CF05E14547CADD180937"
		  "55458621C0903FE6ABC29AD174187F0D5D6B314CAE5A4F95860"
		  "12D396BBC69D81A66927352A9F4B926E1CAE962A73401587FFB"
		  "C6D9FAFA35ECC1B678D0D050F790BD8CF4B6EA3C3B663CD77FC"
		  "6004C257328256F26205EA41EF7A423CC757E81BD7BC408988D"
		  "AB45E8D632B669DCAD4585DCF6348C8C5D2DC651A4AC1806F3A"
		  "55DBE5FAE59D50E2B9F25F89200DD88BAAC59C14EA0798194B9"
		  "0641642B425317BDE993AD1467200432B1EBF23440D0F789EA5"
		  "A83D"));
  test_hash (&nettle_shake256, /* 28 octets */
	     SHEX("0F8B2D8FCFD9D68CFFC17CCFB117709B53D26462A3F346FB7C79B85E"),
	     SHEX("36DA0700FD13A62A128D5C87F69863A9EE8DFC65E5E1F97C495"
		  "A675ED8253CADE4F79D45D9E34C8FCD3D5AF760344FA76B271A"
		  "916A539D467F9FEA26DF0CB98DCA0409602910D7F298D6E9360"
		  "0EAE6447CD2EDE4A76526014FB58ABD4A48ABB38DA5ACCA349F"
		  "140CBB391A6705E4E153C87A80C93E912C912B9C97CB89B101E"
		  "6BCD0B739CF89D18B522A1FFB858B837C643240DCC8F8553508"
		  "596EDBA0282847EA724B5D25CE6704CD3DF3EF40E5410D0C674"
		  "6F97F390DCD77F69E6C41CB009FFCC9EF3C6F9B2DAA646DD07F"
		  "CE398A9603BC223AA22AF0676931D4FDD5D2C06C6F2F4D31D48"
		  "54F72E44EA149ED172BF152CDD08D33AF58878B8E268CE44E82"
		  "46360AD41076FC4E9BD9F460CF5AD24EA05F4E32EE2AF4B1F56"
		  "B8813599F23DDDA7FBD9195A40C5683988D1A84FBFC951D3613"
		  "AB6FF84D372AA4A0EB2B68FFF9B37AD1ABF88725A5479E77E55"
		  "F320BD8BF69A2E174F715B10D8FF4E8CA72F2E29AB501F9A74C"
		  "D4DDFFA99522D1C48C619B1669ACF555F5E5221D34C7B36C56C"
		  "53C3EA3AC302C197D74735A1FB295A39AE38D87F451F86C327B"
		  "67C16A1D1BE50C453F3CA5C9BBC5F78D7C46F5FBE1B44C36615"
		  "9AF8797FBD29C6B025894B66266332078860D083A58873B8EDD"
		  "E27300B4177F4B1A70EF2557610F19B2FD8DF4692FA548A018A"
		  "014F55B90D77ED357987A033ED8E83E07A5D5D3648612854EB1"
		  "0430"));
  test_hash (&nettle_shake256, /* 29 octets */
	     SHEX("A963C3E895FF5A0BE4824400518D81412F875FA50521E26E85EAC90C04"),
	     SHEX("4D2453D01B5B3FBF0FBC4E6F7D2F28914EFDE730877568F3DA9"
		  "CA256C3252560DC9F9AC94AC203AB2A3829A3202E6F4D385D21"
		  "5B843D64B52B3CC28B11E3876248B4CC692D4F7DDD46C36FB72"
		  "92794875D8353D945438BFF037737E3E9133FB6559CF65E7129"
		  "B98644141A1E5870B22751076F4AF9A14F2B0D198FF4772B91F"
		  "4F5329B5BF9326FE39D5BEA9B6D4CD19BEEE8A6B8FE1CF3EA3E"
		  "256C7E1A6A95C5B9336AC8CD2AF6D35C776B5563563CC59906A"
		  "BF1ACFA707AE71E6E52D6A77CBA9E960BCE2D3C11E7A6AD2378"
		  "74F6316D982207CD5CDEAAB3C5EAE28A3E5364626891555721F"
		  "81068F60301B1DB9CC8D8FE00D9949EF2998273BCEDF012A982"
		  "4B5942831AFD4922908624F4C2D3D218FD440652F59F536F8B3"
		  "3EC88CC3E9DB8A4513EB8880B95F29876C4747A8F83D089879B"
		  "29350DCB6CAB4497A8EB14AFA848432310C213D7439968D4942"
		  "CDA32B5E6DF26D29F42BCD98FE7C67020804380D0D9E4A931EF"
		  "9AA117F872C7C0AFF2E5F11451AF753C383144E60E661D6DB11"
		  "FA78E542FD2C9DF493DA608CDEAD669E0AE4ACD76744CFA6808"
		  "8420FA24A43B6E2C1706A10DB468D8A7B6E18CE68C4EAF1CB1B"
		  "9D794AF9457292C2999DC18337531ADCA5BE1225BE1825B3142"
		  "7EFA14BE2B500C0174333CBD3553E05AD1E4E27858BCA6F2BF2"
		  "A966A8EAA6012F4ECF1F3387F3E3409D3986F2E2BF7DD239832"
		  "C9FB"));
  test_hash (&nettle_shake256, /* 30 octets */
	     SHEX("03A18688B10CC0EDF83ADF0A84808A9718383C4070C6C4F295098699AC2C"),
	     SHEX("2D46D78B2BD91664CF875974D2CFDB0AE27E00FAEDEC3370C5C"
		  "B83129EBE05EC438A0A02232601C33AD623162B8BB160F5D8F5"
		  "C892BBA4606D1937A1E4F90782205CF7EDEF07C0EDACEDD95F4"
		  "86157A0C64E0303D30029CEF36894100E79693F3E794596CE99"
		  "ADBAF2E3D4B4ADA5D43B9735DE4B24520EA9C8041CACEA8AC65"
		  "9699DF823C7BE9AE82D52BCB294A43A751A71F6E0B939BF9F08"
		  "31443026A292058B62817FC9EB139DD0637B7EFA73344858078"
		  "AED6FE93A6F22E1E14B92E87CC3CB5C4FD7170413C4949E5D06"
		  "DDD44FE90AC4EBEAE413A7BAAB1FD2B99BAFF442E155E94EB49"
		  "D75EFDDD9C0BBB6C1EC6AC6496145C755788DFEB766CA0256DF"
		  "9C658B7BC660808F875E9E38B3E297543D2C9032B454161F9CE"
		  "E80B0A7F9F49039728D2F12142EA22C4708307B6593CFD0EF5F"
		  "38A1E29FE43058A735BA02D4EDC010C02F4B625AAF832030949"
		  "E81264B213AA8A16BFB10D35C5A369B86CBBF4EEE8DF699B302"
		  "B052315AB495B9B287E9883B05C6273B4C32FCA1D186D2DF481"
		  "F54355DA3AF3E7EAB49C1406857CB9F590E7B1B8EA5B7A0C5DB"
		  "B7FD778A4FB570BE2CB735F21CA599AF8903596E80F8379B517"
		  "D6A0F7F7D5125A7D4375BD90175EFCF5316F13C8B09219BC840"
		  "52B7FD2BC1D55656FDB9CF1918997717C6E2AA0FDB46530C271"
		  "14263E017FE5D91153FDDA221DD63F14C25E3439E05F5DDDB3A"
		  "C966"));
  test_hash (&nettle_shake256, /* 31 octets */
	     SHEX("84FB51B517DF6C5ACCB5D022F8F28DA09B10232D42320FFC32DBECC3835B29"),
	     SHEX("B3DC434BAD278ECE68C6DFBAC1416BF47FB37645AC6E6B7E4DF"
		  "D79E4605EE32B9795ED18683FCB56F910E223704FF1200F015E"
		  "AFB2EE06181E9EAB1BA17BC5D84BC22A2D5C134991C906A71D8"
		  "B20F6ECD6BBC27FE715EDABDCF1C4E1A374B15AB4D76EA6AC58"
		  "0904BC66010CD8352CAF365DA80094F461070CFFA34A86DF705"
		  "B87CC277D80196C86C602326E8E3AACE1BE7F0136C0988FAA11"
		  "A2FF91AE941799EC4DE96E9F167E4088C822BBACC46DFA327DF"
		  "7210C9B31A9F7306AE753152A86F9E0ECB03FDEB7415C9ADBA6"
		  "D61FCA7F4D3C776EE6FBE6901860292FCC6CB89FB45A68F6165"
		  "E36885DD40671EE372283591BC90C2B4A542282B13BEE71EBBA"
		  "4E12797DF59FE47649A27AF116DA9F41E0F0B6B962F7260DFA2"
		  "F569A97BF47405A4EC4A6463680E3903CC7C3CA2F39E9366FCE"
		  "CA8031DA89E447B37F3B80769FDC0449291FAF1BB8F9CECED3C"
		  "175062DAE783F51637581E913104C7042BC328E1F2571CAA557"
		  "2E75EE3F5A0F559B50191F3ECBC1FFC039BD3DBA90F7007AADE"
		  "D690FF35D8D84FD0A9D427B2171072F1ED51A8EA9AA8A066FEF"
		  "6B88915265D80CA283EAB1056B6BCA093D60BFE288E3B9029E5"
		  "AA395C9F3E3913D4B22BADA270A282FF83C963E97F1F7543AA9"
		  "2B5F419973671EE03A79D1392C40EE57265FDAF75977C9F0E98"
		  "C2CEDDD7F7C52B4122A4E517280C9547DF99FFB30692ED39929"
		  "FA16"));
  test_hash (&nettle_shake256, /* 32 octets */
	     SHEX("9F2FCC7C90DE090D6B87CD7E9718C1EA6CB21118FC2D5DE9F97E5DB6AC1E9C10"),
	     SHEX("BF7A4FFF6C7B53893BFBA9505594DFFF317A09278C3A59221B8"
		  "DFBDBAFD08EE237FCC272ECB1C7672B41B8DEC7D54B988B4CCA"
		  "65CF297E1D54DA65A176350AC489947359E11E34AB3E4D6744E"
		  "AD1775DF90BD5BE0712BD54EF46DC279587527976461D6B8F9D"
		  "0C0FE14B10372550ECF74B3CB30C463F25090ABF7DB84D32F51"
		  "E8AEE7298C49607E4D3DD79B9348E33F2E58A7D0D76394C0B1C"
		  "79642CF2C3077CA0764F291FF48A8CD723C953276A829CC1E7B"
		  "798BECA33E8B0FC0BD44224CAB89BA334E52D6943A1ECAC72CA"
		  "3BC825BD61DC4CAEE1A12E053EF31115456DA434747AA8D6DEF"
		  "E56E35FE67D5E761881D681AD8EFD3AD5C69430FF75794FAEA5"
		  "553FD91A4ECF7730EE8205AAAA0DB99AEDE2C52591696514AED"
		  "17F7B8942026AE16D67588A943E5D13BF47F15F1A458F06ED83"
		  "A48641941DAB491594DD02C681950B84718CAEF0E6187F23C89"
		  "FE46A40D5C22AE297A05E8B613B264D204334BE5922A6DEDFF5"
		  "B978767233AC58925C672F4F4B0B77326A5B283CB1DF217BDDF"
		  "DFBF12E4FEBA42E23C18675E0FBF0269E2D53A024B4286FA906"
		  "85C2D8E67E3A60BE0D7072C3AD8BE9F4A389FBBAFE191CF06EF"
		  "F95605A89C3C668BAAB7657A90F23B6B791421D214767736CB6"
		  "17BF1481CE103768E5A8A3384978EAA8F56A6F2E729D6307973"
		  "ED19B270247867E5861172467821A22872E52657BA2FFDDF605"
		  "2025"));
  test_hash (&nettle_shake256, /* 33 octets */
	     SHEX("DE8F1B3FAA4B7040ED4563C3B8E598253178E87E4D0DF75E4FF2F2DEDD5A0BE046"),
	     SHEX("2EA40CA6662A79D253A43DB2845535098F8F4D791E45EF610B4"
		  "7F90D0F28309EC047BF1C565F799253E32A3C538EC6AD842406"
		  "D9BF127CC56AEBC442B4B8FBB5DA5C759E1BE0C201246B96032"
		  "4A087B3A3093EF07CDF45D915C2A4B0FF70D64C1611C17C629B"
		  "E9852CC408AFE212FC303C196F422A085A72684C10460E294FB"
		  "60CE9F2D5E2A599E4F1E3FAB86740F61EE2DCDAF7E1C8D8BC29"
		  "8F40B993CBD0A8EDBBE0EB8715B6A58D2CECD21FC637B02E05B"
		  "AD75E3B8EF0C2BDB157288F763F0B190468016C186A94503257"
		  "2E71952325D30D2DCDE2A915169C51D0C448644D785AC658B0F"
		  "0132B3CC96884C41DDEAD9B57425FDC013D067697C0FDE6A2DD"
		  "31AFE31CCCF9D85B46325D6CEDD80B70EC717278D70C4344757"
		  "088906A577C2EB030F8AD2B18A1A196DE26648915560D228F70"
		  "2956B27046ED63EA6BB863F797CDE111A02B19A6F49BDC7EAB2"
		  "0E040938B9977D534BDD01D19B15781235C249A24EC832770A7"
		  "6520F5C0E5F5ACC6E6DF757FDE79DE7AD8C43028515B82614D0"
		  "DA808400C574E5A982D0CF9461BD25EB6521064152DC7BF3295"
		  "595B7A16410012FB01AEF4ED6ECF22C209FCE011888E29B8C3F"
		  "84CB29F42AE0EC1C4C6D4C02619F90BAAAF7219BE0F2129723E"
		  "AD133ECA7E0D50478DE346F90089BAED5D1D04563F64E3A3D5E"
		  "228988D71DA3DF85D906C9AFC972CA6314482BA62C557CE7CE8"
		  "986E"));
  test_hash (&nettle_shake256, /* 34 octets */
	     SHEX("62F154EC394D0BC757D045C798C8B87A00E0655D0481A7D2D9FB58D93AEDC676B5A0"),
	     SHEX("489A2DA0386AD9F87188CEF4BC2B22186F74B1561D79D8ADE98"
		  "EC2190003B99795DAFEFE1851276ECF233EC9ACC0142DBAA3FF"
		  "1D94692430F67C2B1D809A88A683B73E00BBD678D3728EA98DC"
		  "1A751FD5839B8EFAD1F4CB45009411B8F9B9442AC797B55E182"
		  "7B0B6FD2A4B1E9F078FD43D9553399B9A52E7567CDCE4A27647"
		  "820D8483F16360104A7AC8F3594CD786F51A9ED172443BCAC49"
		  "E1F4B32E4E915A3CBD7AC18FE691E4B9BF35F31D1E8FA401EA3"
		  "7458EBBB1CB6C4E99C6B031C6C6981D95422EC1A9AF5E74C021"
		  "8F403AB62D54E28AA116FF96CE0D6E8076420846DCCF95C9AAC"
		  "FAB3A4864F93B5D3869A7DADA5BBF8197411966166B2F4F52E3"
		  "150FE802C96C47AFD6A9F64F7554FE3D52B79BDACD6C95401C9"
		  "A78F11594347E77F727ACD5E0131CF898C4F36DBB067EE05E43"
		  "8B6FAA70D8B7720EB94728440147B159577CB5CFCCFE9A9EC32"
		  "1081D429FF88C4ABE02C7BEA2B10B61E521889E5CCC29470DC1"
		  "6132489FB43092B6C9EFC9517CCDC0B0824A95501D74B445606"
		  "62141114F0EB6C7A0B469CCABD358546CF1DF37DEA67E3946AC"
		  "D70A504A787ACF86B07298938882D0B67EE67D69A8CABD0B835"
		  "152633BCA5A76C2E6DB958FDA265756669D1105A67889F83C01"
		  "918398BD296ED801A0B5F0FC8AEFDD5205793ADB85A8D6EA15C"
		  "FBB6A82153EE69C10C2C92CB38C90CE0E73E32489522994096F"
		  "08D4"));
  test_hash (&nettle_shake256, /* 35 octets */
	     SHEX("B2DCFE9FF19E2B23CE7DA2A4207D3E5EC7C6112A8A22AEC9675A886378E14E5BFBAD4E"),
	     SHEX("578B2D3B14B19C349DCE30CAD612BC2FD7CE8D6F034FBF4D328"
		  "E7A570EADEF3F9624B65251F992132D25E5363BD653122D5257"
		  "002F806DCE93D0B15919FF9EF2C303C748D9782588C90E3B236"
		  "D51B69B1EAA1301A703B8B6EAFB6C200745D5B08A07A69B6204"
		  "BE3DB36CF6127E5F8399E5901318F9F0C18CEF9201CBD31324D"
		  "87CE630A64518AA7042AB3E2C64F39231AC3DE7BBE859CD3A4D"
		  "90109B2955FBE75A38F45C4B841C7E1E32F22EB69DB97F6AA4D"
		  "45D12EC5180AFF7863C17FE1CB7DFF337DA46F7F5B0DBEF813B"
		  "40DD67F985C978B9ECCC5685E059F55E503B5BE924769A11FE1"
		  "873EA406AC79242FFD6D45334D7C51FD4D1D5EAA28E1388E3BC"
		  "EB4D67320D9723D060EFF815766587FC11C13675DCC53F30EA1"
		  "660DC8C20332A3FF05E2E70AC821E0CD250132A59DCC4F40ABB"
		  "C8783690FAB8BD7693BF106E8F8C7A6D59495FB7301390819C8"
		  "7C86BD94442185DE8B4CC784A690D742A439F73AE7E8252A3D5"
		  "C0E4C0410FE919CD7A53F85F522C138A3D654D9B947963BCF29"
		  "3BEC3D253DBAB8FD39A48A340F8AFD75C97519BFF4506550BE8"
		  "F1C1C7C345C89EDAA8A56F128AF2DA8E86A19A6494BC33DAD41"
		  "2E6DDDE813E3338D16C4C34D40C2E6E1CA61FAECB934D4ED0BB"
		  "0A19D3AB88EA32F7E277BF8521AA733D7240394733A3A1FD762"
		  "3463857E1633F0FACE8BB83437DCC308E4F6791D8790A54FF8E"
		  "06F8"));
  test_hash (&nettle_shake256, /* 36 octets */
	     SHEX("47F5697AC8C31409C0868827347A613A3562041C633CF1F1F86865A576E02835ED2C2492"),
	     SHEX("ABED83F7273F3C7A73A700565F16AECBB0E03D278594A29FAC8"
		  "3B38689AE49A72411CB6FCA5D636EBA8C560CC038B47DA87442"
		  "9BE7C50A9215B5098F94110EF4E38F88D8DB1C483D3EA4728F4"
		  "3B25D94E0C89BB50C66CFA25A62FBF536EBE3D8E7E71529EC53"
		  "33A2823DE42C50F02414C99CFC61B1E7B4C767828748216BF6B"
		  "14658C114C52B7A53C52A3F8BB35048343B19C5B19FA11FCF5D"
		  "78769F1712E6F8FB5B31C017D3E23091B1E8184752ED0669AB3"
		  "8E890D3D7880A71B0B22149038E81AF04C4CC7CCBA848702790"
		  "8CF2F50EF3D6563141DBB9B898D06B7253B4057E9A69D026A27"
		  "F5DF48BBF2FEE2B603240D44B9453FF485B2212828061CEB54C"
		  "DDC17C8DE430D2E86045C790B23C84499C6BB04151BEA457C8A"
		  "18DCF0094F969365AE1F25032AE0524C2A09B69F71BD3EF46E8"
		  "871AAD3B58C92BC4B694BD5D47A8A80CA8DAEBA6584E5D5B4D6"
		  "B5671C823E6F78D8B67A6745236BF675E28A17C2F8CACB4B5D1"
		  "67991D27813C5500951A5E78086EB87C18D0E26863260F69199"
		  "89F5EB180EA48DE6A3750B9E2C1F780230E0070E216D373D3E0"
		  "ED3DAA0F668A8810DA759CFF1494E3BD17B369EA71DE73CF0E7"
		  "EB3912515E454544BE78178DB41F9FBC4BE73D1EAC8C015496B"
		  "08839344D963F070E5AEFDBF29BD056B77C2B98826AE397ECE0"
		  "701DCA8AA45245F26D718D9F46C90F34AD8EF70972ACC600BF5"
		  "EEA9"));
  test_hash (&nettle_shake256, /* 37 octets */
	     SHEX("512A6D292E67ECB2FE486BFE92660953A75484FF4C4F2ECA2B0AF0EDCDD4339C6B2EE4E542"),
	     SHEX("A9D48CDDE90093FF80DC7846BAECF4C7D6F876703AA4404BE05"
		  "0345BF7D2F9FF3F93CE132E38EBEA01AB691BAC9F8943CB33AA"
		  "E7887E9783E0A11EBA2D139A615DF8A62EE4AB9E57BCE1F7FA1"
		  "9C3B3F6AA88B547F0BD5AC0E2237B514145173AE8DEEB6F606C"
		  "AF1545C9F442D84D36080BA514759E3C0D1D3688A3D8B7C28DD"
		  "8404AA5E09AAC0AF3C14792D78434762169CAA935E2AED3264A"
		  "1C0A005C359B3B58BB5F9B9A6E5ECE5186F9DC26C952A298B76"
		  "C49138D20798BFCB843D06CF20AA1D540111BCD50C6D39DFC23"
		  "EF7A8AA8E93526690C8608623A109B1554C840F15E670DC4C33"
		  "E3BD81784287C3FE98FFCDE6AEF5DA5C0038E31D111C333F48A"
		  "D6634ECB47C45213F23AC0C5787CB9665D3B31EC2A0FF9C597A"
		  "4372DF2CE34D443A86B8091729049C4EE58E4ED17EC95033607"
		  "39C966B1B3A005237C668A6258E67CF9B5B3611A91AB5779D62"
		  "82ADCB74958B7D45C06819E78E904B4C557BEC431704FFC563F"
		  "40321E64A7CC456211B5C981C6C987469FCE101A81076DDABE2"
		  "2408E949E86457658E07F61F5BE6BFC68B968FFFE044BFD278B"
		  "D91180A05A40FE65F26D907D205411B880D3EB4945652FC0C11"
		  "2525E06C4795C01F4B67BE8868594E617BDA761465D7BEEBB2C"
		  "564AB66FBF976D38521F83B8DF85A203197F0B6AEA473D3CF39"
		  "46E4DC87739A418101AEF50D1A4355B1E8E5E1D638191DD815A"
		  "5445"));
  test_hash (&nettle_shake256, /* 38 octets */
	     SHEX("973CF2B4DCF0BFA872B41194CB05BB4E16760A1840D8343301802576197EC19E2A1493D8F4FB"),
	     SHEX("41CD43AF243BE7F0580A9B406AA597E72958D24E45F52A2E1A5"
		  "9BC535C207EFCBC9D8B65A567CA05DA450458C2FFD58CA8CA3B"
		  "01617AB6DF19A4C384294AA174D728CB93D6D82BC135EC92F13"
		  "870DCD556DB4421C1C6563803EEE5EBDBCF14B3AD79EDB4E062"
		  "970BAFB3E801D5DE4ACAF052A6E07B0235B0CFA4CDDE1E54C3A"
		  "0CEC7E8401A4D38E12193E7CBD66213B7C7B0B8046220FFAFE8"
		  "68E31CE93FE8BB7EAD861D96CC5A3EDD4820BC2C62DD8A4D449"
		  "FE9B938F1A55262E7CC25BDE92315E329D8F612DF71649EE0AA"
		  "A3983FB4984B6E1127188EA9F3D40590D854F5BE570543ADDD8"
		  "E330BBA204919D7911E56702EFBA881793087ECF5CEA975A88D"
		  "B706FB08AA8A10BA5F72D2302DB1A01BF15FCC3B05C96937005"
		  "0CAC51F54990D57D9A37790B5E26D08B56DE0AAFA61EB238687"
		  "D4396B2899FF6D250F92B1DD2ACBEDBCC1C0A194BA00DA478D0"
		  "BB659D901009FF593F6A4F66092F67238909056E1A0EE1C8D51"
		  "CD39BA3794CA92786AAB18D03C13CFC41F08373D1275D75FC58"
		  "A9314133E62C14E05467EC5829967856817E7AF85DC0A83AF3E"
		  "87097210E71D7325748022E25194512E54B9B18697C3B70B96C"
		  "A8D572861D25178CBA90E503B84930BF714AAFCE91CECC995B2"
		  "623C63F15A1F0F50849F57F5037BEEC1056D4163D728C59BDE6"
		  "F526F331DF17B5A34177D7B05103735461CD325EEC332FE02A8"
		  "5B15"));
  test_hash (&nettle_shake256, /* 39 octets */
	     SHEX("80BEEBCD2E3F8A9451D4499961C9731AE667CDC24EA020CE3B9AA4BBC0A7F79E30A934467DA4B0"),
	     SHEX("A0B529DD351C0F7C539BFAC5A3302F5E88906A29E665DAF3057"
		  "1447C3DCE7BCDEA802E364B65B3037903C5EDBDA2412E084CBC"
		  "D4F51A4024CA6E154DA70C4781DCFC7836F06E5E4606D16D2E1"
		  "BED0D60FB36B84F582EAFB152191CF804D121F0F69324156B73"
		  "3DBC79E1355B9B4D1726E393C22C82BF1EE85372CF04A594018"
		  "C63A02EDFBB2892221875F20DC73330EDCEE0E62CB57D2BA43F"
		  "A1E8AF7ADF157A79861FB48ED49C97104C6D89D290061229D51"
		  "BDFABA62CB77C498A8D71F627372244452549D4F7469ACE1E29"
		  "EC39E60E8563D3FFBACA5804D6660C7C1559879FF1804CF864A"
		  "E8D1C8EF258B41B21B48400ADF439A76C0BDD62FBBC2607D85F"
		  "4295AA0875601FCC1C3BE531B526ED081188DA38B0E40F7FF4D"
		  "7A41C4CBDBF8DB3F46C00D2DA9D7F8FA3459F60C7419AC6F691"
		  "473136ECED7C342D3EB5A39F611EEDBB0DDB3EB12AD67F1BE69"
		  "6795B951A7308DF5FDB83BE7BF1B650C1C9222C22677A4B37EF"
		  "E0CCE03D7237699FCA345A31238067F83547C5442946FAB4E12"
		  "9BDDD409F8A0176CE87A341BBC353CF8A3EB72E5C7AA696B12B"
		  "16E12C13FCD597D864771C43777E8433899F157A0DD21117839"
		  "D2BA21E85FF2E09FA0ED73C9AF26173B9C3B8AC5B1FC4BF7C87"
		  "F06B5AF7626CA15EEF79D529454ACD23A7643F91811365146FB"
		  "18CD0DDC3D46C72C7D1E97E4414225A76B55A19B226D5A77FB1"
		  "B5D4"));
  test_hash (&nettle_shake256, /* 40 octets */
	     SHEX("7ABAA12EC2A7347674E444140AE0FB659D08E1C66DECD8D6EAE925FA451D65F3C0308E29446B8ED3"),
	     SHEX("5EB982A2754FE46D19D8CBF86B4D594D14B421C8AB3AE447624"
		  "47E9F789271F5A1175CA9B6C604FE8046A9A8C11A9FF9354BDC"
		  "91BDF051AB6FA1F6A44D0E42977897F5342958ADBCC1E612A49"
		  "B46EE66420E34F2233E8AD857359EA92E3FBE3F84191127833D"
		  "6BDFF88019BA5B6982F446326877B3D3662767C6E91B0A6F9D2"
		  "888BEA239B25F1B6A248A6CDAC1775976958FA9624EDFA7DE30"
		  "50CB73602D24D13763C2D67A016308BF4C53D9C7B4A95AB5425"
		  "49944B74A2EB30F688B1FC9138B57FB76AB320AC7BD48727CF4"
		  "CE4BD34FC3B0C2EC0E95BDDD0EF0F5F70FD34C6C89098297861"
		  "3D2132FCCF37541042415FDDE4AF368A207D59251F50E14D7A5"
		  "310BBBD378781DF611B425495FC69A2A6644FC6B29ACCFA9918"
		  "EBE05436EE9E191946AF9CBF15011BB53CCED508C4CEC0BC4E1"
		  "56D0884347AA46A902F216ED6577E56129B441EFE731CF6C0EE"
		  "AF378F5A29586F5A62D10BAB2778F106FC6C3EFBF2452C75045"
		  "D578C38222E14247DB42FA60FA6D909E0AEC09EDFFF90CC41B3"
		  "2A9725714E41FFA9F350FF3C1008511534337DECE84A0F3D5DF"
		  "085993BD3D5C755B95E6ED4B36FCEE70B68AF427C3C80386983"
		  "920307878C1A01C5D782AF85C89C8BCBB0EDB2274BF7172DBF1"
		  "6AE9BA1CDD7A97D6B2BC2D06F20BD0431618115097621A36C41"
		  "C9788DC68B95955EA5E097177D66B0897C7BFDAED8487A79E14"
		  "DCDA"));
  test_hash (&nettle_shake256, /* 41 octets */
	     SHEX("C88DEE9927679B8AF422ABCBACF283B904FF31E1CAC58C7819809F65D5807D46723B20F67BA610C2B7"),
	     SHEX("78C7D0D0FDB5FFBCFE55AC06F0E6E58C805FBF76FC2D13911B6"
		  "B63888088CBED5F7E2071F7BBC12063DE085394A691DEE09DD1"
		  "DAE2EEE0DD9491C23B0949C870102EAEE2F39E20A62D9AE95B0"
		  "426C0FA05545C0F128FB53E0B8BE090C1C82F0B66DF4D7CD0F6"
		  "3C9A85E17844988523296849AF7105F845E0F26200A26091212"
		  "8FE7F2265C9E41D7FE587BECD50B8B729EC445C7CE9135C9591"
		  "4C6C9686F43A56CF652A6DD275CA6A883651F78E8FCB8168E8D"
		  "D8FD780735F087203EE1A30B5A13D5F6B7CEA34CCCA3BD336B2"
		  "08DAC5878E85BF147B0B4E70D87F73DE0487BCAFB6ED9C3002E"
		  "2DF56F215B78C7020A1BC9F433161BA7FC94B4CE3679EE955D8"
		  "0363420768C9C7E3C6696E94ECCB0A748FC1F1F402EBEC06807"
		  "8862F1E2DFC8CD6FC23FE1051E51DFE3D65B3779CDE8824BFBA"
		  "8AB63FEF474D648FE4A8F9C083EE0E6CF8FF6768D1F54F31370"
		  "5F4A3D7568192D91E9F0408E8EBB101CBE97AD9FF76A1F347FF"
		  "152F45D9B20AEB7DE7CC2047E47635032953FC5CC5CE45D66AC"
		  "D2E11C77FAD0DA8DCF15FF123365DC9F35DE92838173EBCBDA9"
		  "DA15FA35D10B64E7BB71539708D8806145A1892FBD4176B2E99"
		  "31514BEC17FDA5CDD82C8EF9F6EB4E2900D039115D5B1E940E8"
		  "796FA8FD74127FA53990639F130732EB48CF50620A9DDCFB01F"
		  "D69490A8B5105CFC7A3EA274006F51221C2356212581FB16963"
		  "E7EA"));
  test_hash (&nettle_shake256, /* 42 octets */
	     SHEX("01E43FE350FCEC450EC9B102053E6B5D56E09896E0DDD9074FE138E6038210270C834CE6EADC2BB86BF6"),
	     SHEX("832BEFAECA90FEF41577649B6348182753EBEE32EA688306DBE"
		  "EF8665D6FFC6C5A26EC69EC8C90A0FEA854893C0C4885A2DCA4"
		  "AE77243F735048FF31B80D1A4C84F055DEB1C11F76BF45DAD95"
		  "16899555AAFEA86A129C4EED1D4CFC7604045D22120C8E06F67"
		  "793644533202A69B683B2ECB19AED3DB158397C922DC3A90B64"
		  "A8825541BA5DD9AFD7F82DBC603F156B9F619A83C7132A0D8AD"
		  "71205C83AA4A52B6223507493E0F9FAB9530762A190C9FC5968"
		  "0C961B08CF36DEB67A9A472FF93698FDB84DCF3CFF84C6E9D48"
		  "804ABEB6AA47AD62DC9463C131BFFDB127F9F63F8FF88D9EADE"
		  "CC6B0E00C657CE54E7BF800B9A35173DFC6669BB54A069D9EE6"
		  "2BAFE9B3C791DD51A21994CEE03FA37FAA5D56518A6B65572C1"
		  "B3A17EF52F642C7B2CEE065282FE742E5F9C9BA5003BF1676AB"
		  "07439CCCFFF7D7B76DF84E3D99FAA269F0475CB5EA7DB25D3BE"
		  "BEE36B96634B58596AC788C5F31B8DABA35211D4708EB1F778C"
		  "A38ADB4C15A831D8769AA16110DDFC406B05BCF25B456113196"
		  "03C17A40A24CB2C38079E5684B473DBDFB98877FFE06298C333"
		  "37326CC2AD1BAEEF678BDBFFD89BB1E35827CE296E20B0084EE"
		  "AA0AB1FC647EFCC0231DF9B9B5ADED6DAB34C75E6CBE206DEE3"
		  "E794AA42500FBD837B6F6AA015BB4C923F8F1F769B049888933"
		  "773CD24B688B0F787A2B74AD15275AD6C4C336CDFB1BD929ACE"
		  "58CE"));
  test_hash (&nettle_shake256, /* 43 octets */
	     SHEX("337023370A48B62EE43546F17C4EF2BF8D7ECD1D49F90BAB604B839C2E6E5BD21540D29BA27AB8E309A4B7"),
	     SHEX("0CB8AB1E6149F41D29171DBB444B0F3C07FEFDE56D9CBD1150F"
		  "C43E29009E0F5EDC50F89EA7CC0EA6C9E23236C6AAC72BB3321"
		  "A51AFCB32F9AB5D1051ED244E2A8D417431863649699077079E"
		  "1687B36F75C995D6A78B87A9E949A56A2CFBCC15CEFCAA6C2CB"
		  "51F054072A05C12FBA86B10D303B3EFEAC18F8515BFCDDB1172"
		  "D56B8C92C1AFD4AD5DE2476CF073B7FB4B7DDB8FB2C4BA8AB0C"
		  "53474857F1FF47CD3B1060612F7C564988D2881189248CD13D5"
		  "EA347F10E1F21E299294651342EBDBA1657862306844E274D7B"
		  "2DC9DC6D92F395AF2F5BD01B13FC7C7AFF630E854DB63EC517F"
		  "D3251D12C10B5F6C17DCB8990FB39CF1AD3B65F2F019B24591B"
		  "813D56D661A904405DD814AAE4334A8EEF7CFA455C994C3E31B"
		  "9507FC0E2DE00A1A0AF4DBCCBCDBF393391C5AEC9E3B4D3C8E5"
		  "F879D8E75F63078E686A1BDD5E6358C5807A669F89F90D4EBB3"
		  "1B7396C0152D5DBF665D28BC59327BBFEB466F1E7570462ED89"
		  "8A31E386DE5538DB505FBD1B9020C984A49BC39968A69D5B3E2"
		  "B9A9B61D6776D5349D4334C3042C0317A2A34234E8CB64B89D2"
		  "9E3A8B6E8799822F57C3C6C739B3C263CB4B283EF0F859305C8"
		  "4D11A9DD0DEE36CC8976E6EA1BCD04451B0E0EFE343634DFEC7"
		  "868AD52EBC551D5CFC0277B15B854614FAD548337F20467D175"
		  "EF3DBB135FC37367B916CA35E714B998C3553D5F494903A8B9A"
		  "4B7E"));
  test_hash (&nettle_shake256, /* 44 octets */
	     SHEX("6892540F964C8C74BD2DB02C0AD884510CB38AFD4438AF31FC912756F3EFEC6B32B58EBC38FC2A6B913596A8"),
	     SHEX("9FA49C101D9901053C789B01A505853C8370FDD00F7FEC13ABF"
		  "FC91C7B06E1211DC293FC0D52911CC12AA9CED634B398284F38"
		  "70E82B7F5AF9C3298D53BCFC3A3DE488985CD1047318C2319B9"
		  "E2AC85752B3B0BA2C151FCD08C99E1176CA599557D3A63F4F1E"
		  "BF2FA31CCF9AFB729713387A7CEFFB6F61FF8162B1CD943D915"
		  "4ECB362E6ECC6E2669F746B1422904A3DCD93313E4031238B6E"
		  "B117166B374FA8AE8440A358C1EA7B11B88FDDBDE5737083664"
		  "4B72C0219747A0FE793A1D77F42E66D2D695BC1D9567CF1A1B7"
		  "43C33EE309CB4956BFC426B06FE20FFCE98A72FCDAD31EC5DED"
		  "9EA45D494C89AE577BCB79967CB532B97CCC1708EB2AE8E830C"
		  "AC0953C34ED28BE988E89992C9F0C8FC58FD5E7346DB2F24BBB"
		  "57606302B0520D7AAE54F3492DF6040916AF04B30F7F2480B22"
		  "B93AC47FC5DE2CCDEB27A47C25045695AE9F3E54FB27052CB58"
		  "141918105E0621156FF4BAD6A35DF5BCAA0FBA8D67A18130F11"
		  "7A09FF90B8760A7027614BE569FB67659BF0DCD1F3B7D13222A"
		  "338F9E5736D5DAD6032189E149D22D56861B72A561A9DA575D7"
		  "20F56C365C5C8BD0455C18B7B73DFA4652C1D570A338A5B1D2A"
		  "2D00A4387613D11BAA57160A513F4B64D91739E032ED72BB2DC"
		  "FAFE6BA6136FB381857071250CF63051059F9BA3719305D33EF"
		  "9DC8D33FD6D427402EE348324C78920278D6E5B2678C1C4FD40"
		  "8760"));
  test_hash (&nettle_shake256, /* 45 octets */
	     SHEX("F5961DFD2B1FFFFDA4FFBF30560C165BFEDAB8CE0BE525845DEB8DC61004B7DB38467205F5DCFB34A2ACFE96C0"),
	     SHEX("7818EC8E1A13963C319BD2199294E1459323556D3E1CA05EA11"
		  "081FD706655C1CAD1A9270495E550DD34A71D6D6B2554C2CCE9"
		  "776B30BF0CB151ED0A87CD09735CEBBB03A188EB8A1D62EC0FB"
		  "614BB8D1D67418F91DF9E7FEF2EA9971CD46A1E6AD52D2433DF"
		  "FE98DD59E0C9F32A4A493ACCE9E6B2D946A56DA432A43E45A79"
		  "14959AF23F1D08F5CC55B3C06669308021C004F0AFB3DF5BC10"
		  "A7F19A7ED2DCA14297281D49952D067B094A4068493E7DF9940"
		  "3063AD007CE8CE76E2A7ED02F7023543E43E9CFECE94601379A"
		  "048A73663A9B06B04704D59FDB6DEBA49799D3E8052C1AB3430"
		  "2A2A2392B53A5FB9547628ED4F8F157DB8D795CDEA73F992122"
		  "97512A04269B0D5F418E2795BFE76E939F8BC9F2137141DA25C"
		  "F095F2B754F6D0CFD84B3901A903445B7A71612539F4F736DFC"
		  "1D4DA1B9A8CFA87F15E34D4A813808CCFE2C9A9A710A1B97521"
		  "66996EDE550E14B55DAD529C8A99BB9FE2688CF2CF294245707"
		  "15C49EAF94C0EA0BFF227ED445435E3626F31CD5E887CF1490A"
		  "9A2B7951AD42EBA5B24B02EE726F95E1E6862DC30636983C88D"
		  "C915361F209DD5603672C9B4D4ADA1703F56955015B9128FF7C"
		  "0547DFEDF772E63AD7EC847B946A66B6E4D3DC8A8EC3B50745F"
		  "F7841318BD115BA649B662C99B973F4E7C26CB1604D407AE95C"
		  "E967406E70396558CA27ABD91FA7190A5D0F4AC89A070F45B85"
		  "8B86"));
  test_hash (&nettle_shake256, /* 46 octets */
	     SHEX("CA061A2EB6CEED8881CE2057172D869D73A1951E63D57261384B80CEB5451E77B06CF0F5A0EA15CA907EE1C27EBA"),
	     SHEX("5F6539A11001926EF3E0DFFB0142588D4E48ED4A212753128C4"
		  "C944CC6503E5FD4111583D67FBFD4B2ED2D82447E985DD03CB4"
		  "DA9CCD8ECF25069E84B4741A4C5756D6295E55075601098D996"
		  "A375DFC3156D5F56E28FD8F3BD4793601603BC999BF93659D63"
		  "F3B95AD0AF1F5E1749DF5197C95AEB05C683DCF37E9F362DB1D"
		  "D64578E8ED9C1CCF0A3AD93E69B82FAC004ABB2489B0734C4CA"
		  "C5B1A9316CC30F44AC42914AE696698862938DD0DBB3D8FA6A5"
		  "3D1F70D25A175CF810AD901D29DC3BE40E4F19A5308CCB795F4"
		  "4F2EC364946D99F59D2DC1DC21C5DD162B7C8CD5BBE8F4CA8F6"
		  "06A0EF5B4619EA93B278DCDC2E943455F178ED62FA747193215"
		  "D76C8F76077F9209A9311F1F848009483C82A82BE1A9FA17EE6"
		  "9D1DE59AA305DA992F709F2C9A1EFB33DBC8108A7D22AD38934"
		  "D2995A3A5D58DCDB8E435D18099D9B3A5F9CAB18B940F9E1B2D"
		  "B9BC9768B6D29C2098273BE3EB77D07DC6E48868CEED85484A4"
		  "6D94A3807CED72C292FD699ACFB6B1DA030EABFF5456FCDFAAE"
		  "70E851BA49142CA528D91D00CE148758B54F0293E6C53C7540B"
		  "55288322B015B6067DD16C0CB711C63970CC16BAFA980A8C3D8"
		  "DA5FF2236D8DAB951A2FAFE29FC72257AD99409F418E405858F"
		  "BD1A9EAB6930977C034B3D020A86C25586F8524D6577557D732"
		  "68D225CB8E38892D3E7F0095F68A98C1B7355B5E331D69BD487"
		  "FE4F"));
  test_hash (&nettle_shake256, /* 47 octets */
	     SHEX("1743A77251D69242750C4F1140532CD3C33F9B5CCDF7514E8584D4A5F9FBD730BCF84D0D4726364B9BF95AB251D9BB"),
	     SHEX("B3120C516E7A7039B7F8D2A886B4774FA9E14B80A7F7B1F33D1"
		  "B4FAAAD74D549E2195E5B6345B19AFCDEBA3ACD5ADE7200500B"
		  "CB9C94D6848357D6FA88CF9125E59D0CBCE876D681A68B6AE4E"
		  "25DD5994D496B7C64F0B9124BEAC30348EA6B80803FF17C0846"
		  "E19F9ACEE413E07E6FF849801C131DAF7A36F2EDE51CF9D21CC"
		  "0ED6300EC5D6B64E6FB9F2A9B909DEB4069738D60D6F9F6CC50"
		  "D749AA7EF264E2BBD6DC680823C45AA8D3A349255823EFBE4CA"
		  "C62C0D38A4388B1FED31EFDC5BD3D6227106FE03B4A40F21F3F"
		  "0411405A4A80E92F3CC045C7967CB7AF2F33879DCF9DA5E7ADF"
		  "813091EB357EC9C0DD097B868FE2D715C124AD0A44E2B6DADF5"
		  "B360FAF6A727448D5D7B76AB2C716543A09E2F75D570103A8E9"
		  "589AE58D01904ACFA0DF54E6FFF01ED725266F7E52B992FB341"
		  "44568173065E406A1E98D8DDE1889D0B7274011CA00357BCD07"
		  "D4CF8323F103D682B272A98395A60A95237482F00DDD5220433"
		  "2947D4708D9B8697F9CE529ADF3599C440E2B435F4F6D039B35"
		  "1FBB198BF360D3A184616B86573E7E32113B1A5E1EB52A1D15B"
		  "EBB3C49407AD6282EE2551859F9CF87D0D8F0AB362A439EC535"
		  "40B4B024EB49B525ED29626DB292D1333F579B1896A6357F9D0"
		  "B51BD283718D2A7ABD8F3786A7DF2B512070A2C9A1B53457F27"
		  "C6B859A2C6956BFF50AA7906D546C67786B68EC1DF3E93CC60C"
		  "6968"));
  test_hash (&nettle_shake256, /* 48 octets */
	     SHEX("D8FABA1F5194C4DB5F176FABFFF856924EF627A37CD08CF55608BBA8F1E324D7C7F157298EABC4DCE7D89CE5162499F9"),
	     SHEX("F59933F9E9FC474342B7BCB8B4DD6BA7B02AF5EDD2C1824B227"
		  "BBD2A56287267804534E943EA69B571990EAD3AD5DAD06CA9C0"
		  "BF97FEA1F3878824782E244D77407E3E83D5F527A1F5BFD8C74"
		  "7FC08C5170E4F6CB114A9CC34E96AC2758782344451BFDCB090"
		  "2EC52A9FC82087195CD1019A8E5B03E496A42762FDCCBA6EC57"
		  "3D5C0740391697D95934F983231ED9642A0C6A2F92198B49670"
		  "AA7F76C1715F3C80648E606B97CF708596AB9D8DDC7D039DEA1"
		  "F4DE4D22E6ED297790AD73D9EBC2B1BC55315BB60E90EC7A2FA"
		  "9A7922FA5A4E824BE742EDF41693F595571CA33EEAFF4EFDF58"
		  "FBC0B9B1CB610AB2E658CAF31FE6E0AE74064DB62CE339C1F43"
		  "9A55FDA0F594635573C511D040CB59F2BEF552F38D87F31AF2E"
		  "98B59E9F0E67BC57D59F51204663511865BFFE1CFC42B00AC0D"
		  "69487AABCC648BDD8201362A43AE19A9570DEF75BCFFA6D0009"
		  "62E931AD32E36A9118D74C777F9A6D853496E9638332C3E6D7B"
		  "0A5F030B2B4198C9B31A82CE117412C144914E784D9B0DBCB8A"
		  "3320FF22C4F4F4810D5885C7DF3D0FEF30B4F2272230278C600"
		  "834133C4E11AE65BCAE2069FDC1C863CDD701DB750DAE4CDF37"
		  "A2314E39DBA691ED7D4A0087B8057EE27B3F6AF14AC0A742826"
		  "FEADA8C5B5D52557952DA2FFE45E7C328E04C3CD61A5D8F5E11"
		  "B3ACB6667A08498AE0582FE0E06105101CCEC61C63FF3CDE8BD"
		  "1798"));
  test_hash (&nettle_shake256, /* 49 octets */
	     SHEX("BE9684BE70340860373C9C482BA517E899FC81BAAA12E5C6D7727975D1D41BA8BEF788CDB5CF4606C9C1C7F61AED59F97D"),
	     SHEX("5DB567B895164625131B8049728A0A66BCDD2B27D37828AF413"
		  "53DF5AFA10DF9FCCCCC110DDA3071A35319EF50821CF1796B5C"
		  "319D4FAE433E8A2599325C511C3C03E6EBB7C10E3BE35451756"
		  "75F7AB719E906640DFECCDADF77C345DD798AC7E2B7EE83C98D"
		  "D194CA19BC1131200494A4B67428C23061CDC4000762C12565F"
		  "A1C731A574530D282C2502356A75F03685A4FD08D1E938BCC2D"
		  "6E7EEE748DD1391950A44AFD63B73ED549895F0CB38F7D29269"
		  "7994520DE110F782B0CF47BFB07CFFBCCA6D476D468B227B340"
		  "6F44CBF3C6EF6920B293FAC69927DCB2D153350C7C8BA2D84D4"
		  "11E3CA70EDD9321CC0387A6DFB84221915094EEE0A272F267DC"
		  "111F18E46C6D82CD6B98933261B5A880FEDFD23AB2F7B6CE349"
		  "789EDE97CC034D0E0408E3E3B75D951D7DF039700629CE147BF"
		  "B2E2772EA80C8681D6DB87667A63E3FF358E74AB45712F0BC1C"
		  "747551DD96F72A6290C5E9679A45F37934E7E22378A2BB4A031"
		  "52DFD450B98D76C233E107C57142B72389350072027BCDF9BD1"
		  "82A304E8B255F28459D96687083765E46F9B4BB14DBC374BB40"
		  "1EA0DA7557ACB2684A96F2791E1B8DB0919365820CD315B730E"
		  "F8BB83124577E55A139DE4C9236182B885ADFF4CA2298F157FC"
		  "460972D6BD67D538CBA7E32F9051AFCC00FCDF797FC8231360B"
		  "A6B7ECD5C4D647338F23E1F578469E1558F49F87FEFE28CD761"
		  "BDE6"));
  test_hash (&nettle_shake256, /* 50 octets */
	     SHEX("7E15D2B9EA74CA60F66C8DFAB377D9198B7B16DEB6A1BA0EA3C7EE2042F89D3786E779CF053C77785AA9E692F821F14A7F51"),
	     SHEX("B3D7178BC88BD577C5D95A67B110545C8ACF8F6FB56B5DA15F5"
		  "738785E6C950347905B928DCA2E70CD2A5E574811DDBC894EF7"
		  "FA8EC3446C7897B2BB2D68A61763A892632A3C5C6A3764AA0F5"
		  "76717ADA5CCDA30BACCDAF71740316910CC08E8272B67521292"
		  "FE1749026FADDEA743358421EAB011087D10D4208FAE8D5D607"
		  "EE0D89976BCD79A12F4ED8B1C5D0189B0E2ED5B0B396C85B277"
		  "63BDBF52D56B92E015CF080DF4F2EC80398DFBD733519F3915D"
		  "3F102D0066D425EF9FE1AA3AFC8584C8F4D594ABA86AD52E71F"
		  "A77893266C4C606A63891C41A679964A5E3AEF1C277D6D35D41"
		  "0ECA6156F8CFED13DADF8022AC9159F913C4838AE1AF73EC268"
		  "F9D3BF085639D69F2CC51C39EA594948FCD238D093A799F266F"
		  "BD90B07F4619E342E38AFD178864760AB109CB4949E518305B8"
		  "2ADCD68FE6E922D1ABAB2D132D10E0AABDAF78F0CFB74DEEF78"
		  "CBCB422BBDE5083E57F016F4C5679E9D5010D4091B3FE615458"
		  "86A65AFA49716EF8CB5A6C0EBBA4A2386A6591BCEDB3563BDA7"
		  "F0E792CF28D257BD066D44E3AEAE8478D093FA9482FC26C77D7"
		  "3DD3150D53D9F1A030063122916FDA66BAC51C69CEAE533F7A1"
		  "569A3B9536CD35067EB190D2812EB2CD089D3E1DB98F2DFE5A6"
		  "839F5B04B6EDCA6732837D23F985E1CC9444AC04DF0B2BBDBB4"
		  "4E24D39DF2F447D72FFB90B7E9C68A46329ED630CDAE9470770"
		  "4428"));
  test_hash (&nettle_shake256, /* 51 octets */
	     SHEX("9A219BE43713BD578015E9FDA66C0F2D83CAC563B776AB9F38F3E4F7EF229CB443304FBA401EFB2BDBD7ECE939102298651C86"),
	     SHEX("341AA5C875EB43F10E8DE28505858865870E62A9A4FF45BF977"
		  "B947D0BE17B2DCB74D5880FB5316C5C0B5E1C7DA24D8C8C72E5"
		  "CDA4A9405912D12EC5316611EB40AE84A903703F67D3817BB73"
		  "AC7C2C449D4D6D737CBE06C144A20F99DFFC12A8F104DF9E448"
		  "A96DE38DC6D7F83324FB3A2626BC91405EE5917F3765D9F2288"
		  "7777226BE829584D6E0F6544B5E1C25F939FC9136BA2E0C0D6F"
		  "6EE5F4EA97DE937352D9428222667D974F4840FEE0770657DAE"
		  "58BBADB5F47EB9C88232FB7BCC6F1FEDA76600FA2659F7E8311"
		  "29B3E3856ACA6D78E0D7C582852503A67876877DC6599C10D42"
		  "DBB81E6AFD2919A18E4EC01EC7A032ACF2A8320C5AE348D28CA"
		  "B36C85CD256DB5C30BA1F8891005E68F91D7B0CB0AC4476937D"
		  "1748F2ACE0A1C512AD07514E31CB0CFD743E5C29633C2646AD1"
		  "882176BA15E7CC8427FA20B2295100621A6E01F7FF4F27A6AC0"
		  "766F9C25934E64546F16F61B5CEC865028BD0D7DCAFDB3D6912"
		  "C5786372E1E4ADC6CC8DD3E0A0F299B65BEAC5FAE69E8219787"
		  "A8EFFEFCB50E40C8BC1C3D033742D98FF30D64D72D089AEEE28"
		  "489791F6099CE354385DB83256150A1F9B6CC7444665BB95614"
		  "AE6E6F511CEE0C49288B3EB31F9C7D04B3D126A1535531505F0"
		  "C2710CF6CE7A6DB31F043C7C535F89467CE1F6E85A10AAF5486"
		  "A3A6953CD351D865F818FE959E6E74F2DE42C1FD6E219DEB606"
		  "6C86"));
  test_hash (&nettle_shake256, /* 52 octets */
	     SHEX("C8F2B693BD0D75EF99CAEBDC22ADF4088A95A3542F637203E283BBC3268780E787D68D28CC3897452F6A22AA8573CCEBF245972A"),
	     SHEX("BA270D4DAD10865DF673F0DBB9D72EF788689673116CDC469A8"
		  "CB9C89E6D8D511DBFFF6FA38E29257C9A5820781094300C088C"
		  "DDA2A17AFBF164D13E360EC04FA44AE98CBFB972A68D35E6FEE"
		  "E3AE61CC217487D7461F7315449E201D707F8CD0E9FBC64D920"
		  "12B22175E677A304D903E556A5D1FEFAF8177A8EC3E50635449"
		  "6CD693531ADB22BE09BA71B4710549BCDD6D5EE1ED3975258C7"
		  "7760725D1C4416EF3F3B45DD8210433BB44882667A2154D4179"
		  "2A37F2B7C5C85DFF707BC2A89AFD652D83904EC084D1E03496A"
		  "7F402FEBA9F77254772904580CFFB1E7A0A8D489B6CAD6D97C9"
		  "A5824545FDA36FCFA2E0A6F7E11710A8BEC33767303FE884319"
		  "5E9F17B1B75A4625C67E39FDADECADA6A7E7755DA7742548B79"
		  "7662B6A24194E07CE2E0302FEFF4671E6CD972324173F14F5A6"
		  "0D443F6162CF2D6A3E2E0A7F2A6BBADACEC5F4B43D4AAD80900"
		  "3722982FC7AF821AE4143D123AEA6B7D8550541DA9D704D81D9"
		  "E12820EC03E8443866EAA80A534A5983581F1DAFC7F124915D4"
		  "2F9A24887207C2232B5EF9D8DED3A3ADCB4D493FA2FDD6061F3"
		  "9F28CA3B489676CCE7E0600DFAE247A62E96BE8A63ABB977A4F"
		  "35F8361B71C8578BDD63F35D17CEA1463AE0709353F4667367F"
		  "0FA0B6B6B6EEBFA049BE6133350F71E9CC1B157ED18C9D90A1A"
		  "4D134E553165549C18004279ABA0C4EAD5F342CC05039DAE1C9"
		  "CFAF"));
  test_hash (&nettle_shake256, /* 53 octets */
	     SHEX("EC0F99711016C6A2A07AD80D16427506CE6F441059FD269442BAAA28C6CA037B22EEAC49D5D894C0BF66219F2C08E9D0E8AB21DE52"),
	     SHEX("C2911768C4579E3876B6ABF0A4A30646D8271FCBF3DC682C48C"
		  "10E41A878609301F10073F3EA87BFD934854147D129DEBA9124"
		  "FC69B6E29C271062CC8B3A5785367FDEC382D365D4D60E7E63E"
		  "946154F948B55E7E31A0CAB5D25BEA7CDCC82D8B6E200C28C7B"
		  "4946B4CA30B46970EBB415FC18D9BB60C4F81A308BECFB2E40F"
		  "796B8D6F914FA7F134664B8EEF9FFCA684B296D26BBA051BF45"
		  "F3EDB68ACC5D0C79C08724BC7EACF702633C99C47DBF4E08AF4"
		  "3A48A51A25620C1F16BBFE4CE1A52A60B4DD603524DE38ACC2B"
		  "E65240F51E36C6533D85A7B0FB06FDAE420A7E84C4644B943E5"
		  "CC4AC48505CF042ECE8FBB4AB90B16B7AF3ADE3E24F871DA2D8"
		  "6813A100C82E62F949C3570FB117407AB71A660BB4284A114B1"
		  "F6817621EEFE246CC0169CA7C09FE6684A9709BB5E7C5C09A35"
		  "B4D5E109E133D23CEFFF9E438BE621577A98D9B49B0F30740E7"
		  "E3038C8BCF07E88ACB767F9A43B60DA387B617D6FFF8A8D8784"
		  "513D649EF3A142ABAC0BCD8159F4FB65DB24616225FB4FC636D"
		  "6C1D94DB757E647934BA77C946BB7D010AE5C78E05157466590"
		  "CBDA903630809649BEF7F0D27000A9156B2D05DB89AC90A91F4"
		  "D1F295D296D7783E6872FACCB70CBCCD244BF620D5499382421"
		  "C8CB3829E756E6B415AF3C26F952539D14BBBE56C034B58124A"
		  "D973A726676B7F278E7D4111701D69681412BDF08731C4CB14E"
		  "ABFB"));
  test_hash (&nettle_shake256, /* 54 octets */
	     SHEX("0DC45181337CA32A8222FE7A3BF42FC9F89744259CFF653504D6051FE84B1A7FFD20CB47D4696CE212A686BB9BE9A8AB1C697B6D6A33"),
	     SHEX("C2AFC53CC5DBE46697F64D5B6B37BF1DB7DDD06D54EE9DA7873"
		  "C77D91DE47377DBC5D525BA8C14D5F24A2BDD473DE53FB1BCBA"
		  "E0ADF93B4525C0A77D1952A21BA5E575AB9AA889A19D85A02A1"
		  "AE4D420613BBEB7BD2A7032137F196E0566C284CD11822EF938"
		  "C191763BEB392EAE3FD6FAD77EA7252EE72798E5B4318961EF6"
		  "7E595BFE0BE036C478C88B80C50C3F7BD047066F4CBE031A867"
		  "644529AFBBE607C6FA77602ACEF9A635DE3B1FBD6C967B61330"
		  "40D1A6649F1FF5598CE0E76AF8ACE89406FC02F2818EF8C29FA"
		  "FF1F46A551B534BCE2C30E6FCA6F62DF3BDDDE56FFD8550466F"
		  "48C0D14BEBB386F5BADCA240D848EFB66AC2D339A54AA1ACCB5"
		  "C753B081F0B6F782388E7B82C77304F30E03B5D3BFCE0F1B515"
		  "8AEDECAB4749C17305DCF231A04EA2436F423F5A818C461E90D"
		  "65EDA69DDC5D977B19F26E4C9DB874F2602A3F5BE5AB8C5C70C"
		  "DBC57E5BF757037768E1962D0AC697645B598C46D639C7A0DD3"
		  "1B7CCFB88E4745BF2776505308C28FDDB084F67618B84D0451A"
		  "B2AA45437202474ABAA0780935EE78D47EBB4E07C6466D3F8E8"
		  "3C1B27EFFC1064FE01880D2A7D571A955DCD4F55D631DFBB3CB"
		  "550E3541254C0D4041479FBB331EC591F8AFE5B644F6DF9F430"
		  "0375805BED126EB96893BCAB7FC5AC5CCAD3596D8C011258F3E"
		  "D269A6B0A6C4736D467BEE9D495D414B475D9354174B36E9756"
		  "55C8"));
  test_hash (&nettle_shake256, /* 55 octets */
	     SHEX("DE286BA4206E8B005714F80FB1CDFAEBDE91D29F84603E4A3EBC04686F99A46C9E880B96C574825582E8812A26E5A857FFC6579F63742F"),
	     SHEX("B9E75F5D4B74FFBD244CD9566DF861526B5DE9584D3280BA5A6"
		  "84EAC9D44C91C0DCEC5827DA4EC0FA7F60D29286D348F576F4E"
		  "2FA03AC5BE8F27E9F9912BF500D0CD549F5E5799697C61F0403"
		  "C264C4B2D986DDBFFE72FD8CA6439FC01D1F7138A92947364D5"
		  "86D67C300F27EC2E3E75F9088A5B787490494EBE0C42D00401A"
		  "2ABA6374CB3E33BBD737ECFBF80EE24D4985F6D11FE24FEDBEF"
		  "B387D4EDBF4AE1F409E67F10719F47397968E406A51B0DABB4E"
		  "9391FEF03F9F7BC47193215205386914FADE37017EDFB0DFE8C"
		  "51B4C35C0AEB6F8A8E6D4D69FB538ECDBF65CF92325A7F280E8"
		  "069CF9FCFCDE610BD024CBA87827E92CAF22482413C694F9F1A"
		  "DE65B67F8E1D32E4B5D51F038C2FA9D89FDB2BC1848E3E7B54C"
		  "4CD4D021F4C8BAFF61875899B79EF4A1468B0443691981FABC9"
		  "5D076070BBD5890EFAEE194A6484E584364BAC001FCB37C2285"
		  "8F74E1A01EA8EEBD49D9A55A19F5967B898C5B71D5F2CE93F4E"
		  "52CE4B4D00D7C42E58D71C174E9326C5AE1E40BFBD8BD37DBCE"
		  "9369062124CB47D683DB245A4F289F052F89812440BE2ED28B4"
		  "0049A9DBC75B3FB1003BE8A1B996BE44270A83DC20A38D62C59"
		  "71D09D06FFDC3F89379A99923F2BE2FB6651407EE37F4BC8073"
		  "E396887166AD4F0A4E156C72463A524EDBF462F62AAB251E321"
		  "59CB3D79A2CB3A8FCBC196721642682646A83EE60425DD7207F"
		  "7360"));
  test_hash (&nettle_shake256, /* 56 octets */
	     SHEX("EEBCC18057252CBF3F9C070F1A73213356D5D4BC19AC2A411EC8CDEEE7A571E2E20EAF61FD0C33A0FFEB297DDB77A97F0A415347DB66BCAF"),
	     SHEX("EB24ECB6C9CCCDA1E8AB1CA26683B63A43CA864E23AA7681A49"
		  "27AFFBD8ADEE82704B24B321D6C909A1CBFCB453AC084557FE8"
		  "4AB25E044885ED8051A0EBBB457D9821E9C132230FF27C4F279"
		  "51432DA415D90D59CF7148569CBA02AF32A8F4B774D5E4667AE"
		  "594C2CC4776AEBA2CA7D5F678C522A03B426AB2325127E56A40"
		  "57837CFEE2607207DEA5F913CD64BED09E7FA723148ACA13F52"
		  "2B584AF9A36E74E86415F7C064C7575176488895F0EE6A6795B"
		  "51729AC947BF09668A9ADEEC199592B77227D62205857B75E01"
		  "E62E3D2757F02C4892A9E9C2A9187F7F3DDB16B7ADC28C33F58"
		  "89F87F949EB0A61E5FE431A3C11191DA6D127BFC899CDA0787E"
		  "B269D0296F31F3580D9F630E0693F292D2B9954444E56A82D6C"
		  "81319CD99D5DDB2F042E9242244455D5D9F59CE9DADF0BE7804"
		  "94FDCABC72AEE397F13CBD656D246B6240A1C21BAD6E0126CAE"
		  "A1C93096A41CF648BC0ED7A051E03DCB51E2C48AE32C8818863"
		  "4021A4B64ADC4B2CA9A5118D4B6698E87C1E3D4C2666996F875"
		  "4242214F3B07806C8EAC8CB63AFCE0A03B64D63D46AD53F0345"
		  "DC3A9FE550D43E9E569F8C2BABB2C025D843F7D568DEE6BC762"
		  "D3B4DB80B8A3119DCB19A17CF5FE1639CA313FC74EACEAB5F9B"
		  "AC2E546C36DD7C292B959A2C9CA98CD143C5822BC94179DEF33"
		  "D146A16091688CE192BF03A7E334BD033DB66A3D544A8FE17D2"
		  "F5CB"));
  test_hash (&nettle_shake256, /* 57 octets */
	     SHEX("416B5CDC9FE951BD361BD7ABFC120A5054758EBA88FDD68FD84E39D3B09AC25497D36B43CBE7B85A6A3CEBDA8DB4E5549C3EE51BB6FCB6AC1E"),
	     SHEX("75102A95654D885655B910E6778F8A72B263B0854442230F8D8"
		  "923AFCC92C5BC98EE5F2E8D53EE1EB7F7A85A7562FB96535AF6"
		  "C78F0491EB11B4BE2CC71CCAD0442DBA422B84FB16ED529278C"
		  "57A543A3B8910D17232B2B7FC4B81BEA56A6E99F64A50DD73AC"
		  "D26E506EDD60A788548DBA27AA66BF6D41C8CA0CC603D8CA515"
		  "F3BC948AFEE2E7EC3F8CEBC56A01682E66A082846E8EDD0CED8"
		  "5A6D5ADA77A9567424C73F049DBB2CDC180E6DD85AD73D624BF"
		  "E10BA0CE3609771E50B51004BD62029259300E333D18697A3DD"
		  "0408D37C5E275CC8A4C6C5AD85040365B3A427F21BFF9F1660D"
		  "AD79D439C5DAD6855E75840577B6AA822F614B42CF57561A70E"
		  "E8216F9CCDD8B10F36455A2677CF93756A3E30E2E177029C4E0"
		  "A3860CC01619549AAB73B52E7E5340B42EBAE37478CD899B71F"
		  "9FBAF7DDE36C57450CB5793E371C903EDE8143F9CA3C8EA2231"
		  "BF5F3191A49681EE8F45374D81A774E9F8C6EE5465B567BCCE1"
		  "C77E8289C72411A1C8F5402906046F42CAFE88B54A5E8C7FBC5"
		  "311B1A7C43A1094047935A27B2D89E9470EA28569211F1E6D99"
		  "E9FCD1AEA301A6271F90F51DDAB660E90A50D10003BF425858C"
		  "FB0D67ED630F74D6FF46E57FFB7236D55703A55531F89D92EEA"
		  "C17C6A050F56B46BD0466A0A918EC972E0947A991B024AF21D9"
		  "F92B6147155A1EF08EBF620FBB65C309863A25F3BD1D9ECD818"
		  "E354"));
  test_hash (&nettle_shake256, /* 58 octets */
	     SHEX("5C5FAF66F32E0F8311C32E8DA8284A4ED60891A5A7E50FB2956B3CBAA79FC66CA376460E100415401FC2B8518C64502F187EA14BFC9503759705"),
	     SHEX("603CFB5E53D83CCDECB22C5C75E67F5DFA0DB7D874E6D280DE0"
		  "14555B1300192D9B769482878950ACDB484E87C26F366D2DB04"
		  "9FFE8C92799954FE31DDE56A061E2F80A5DA15B65A0C9382C77"
		  "90EBEE8E676373405BC1BCFBE974797CBAA998A0315CD9051BF"
		  "669E00611880AEDD88DC690192D8D485726B47879861A85349C"
		  "2017D9892692DE8DF315EC3EB4E6C508B85002FC7E6D7812FC7"
		  "E025FCA92F14EE57EC9F507EC05FB8D143DB2EF6B83B8785289"
		  "C2FABA51E962B77754C0081CC1253DE4CF8C36D2150B263FA01"
		  "E19E5157EAC5521E29414407FBCB604A442FDCE7DE9C99D46D7"
		  "1F78D1B9931033AF38940D70651884B766762605EAE11457A60"
		  "F44BF5CEED9C0E9A9C600DA70B87B57E4B5C280E3AA568BC6E3"
		  "988B5481006298BCF9239583F39C5EF439E551B6166DAF31970"
		  "DDF7BB6B7E068DAE50C15D7A1B3D4F3587A91BDEAA59DC66622"
		  "0C2DFD238BEF11F99DD47B87DBA8C60598B531013468EA5F15E"
		  "DB0E2F43D4F4EFA673E88AADF4572A50C25B3B7953C4FF0CA8F"
		  "32D374FDC3C130CCF67E95ABDD315FE7390D5160F6B307F3A20"
		  "497C76F69CCF18151E5995452F01ADF1D55BF3A678D0ACFF44C"
		  "50852F940DA91AA3FA8A5BC8CC72DF7AF11AF0E19C167F6031D"
		  "C780DF8583F290059F5ABEC6917F7EF54339538424234454004"
		  "BE4398CD0CA6CD1E71D0372F2229591F7647142C13C5B089780"
		  "6F30"));
  test_hash (&nettle_shake256, /* 59 octets */
	     SHEX("7167E1E02BE1A7CA69D788666F823AE4EEF39271F3C26A5CF7CEE05BCA83161066DC2E217B330DF821103799DF6D74810EED363ADC4AB99F36046A"),
	     SHEX("724E26AD36068E3825E1F997CFED8AD10194B96E76EF425BD34"
		  "7254BF19BB862DD449E0182E9B8E20C8CB5440E34CB7391935B"
		  "6F5D49F3319A984E313A9F4ED7107C43D737F016BD20D7F454A"
		  "670199566CAB98C4D05F0D31B2811A48EBA4DF019936B0C6E98"
		  "3C3557DCFFD42238885A7CBA1376CDAD15BC2FEF1E75013CD3A"
		  "CA86FDE4B616F2BFE310131AA5EE826F5B0451D4DFC0F0EBDEA"
		  "CB36A8B6AB96D7D81B2A21F7E09C5E908FCACFA763DED4361E2"
		  "D9EC86559DF71A827662EF692D503C72987CC9F5A9FD69F9AAC"
		  "4231BF27E8C64ADE3F9B51FF7DF4775B966A6C2FA6F913FD191"
		  "F9B79DDDFC077363E1E62810F9D82C82A0CA9FD45B9267B1799"
		  "8C2FB7E812195CAFB0A08831CA47857E00E329072037B3A96DC"
		  "9780FDB52379E7180ED0E84866D42779D89F56F94A8D86C0600"
		  "302A09255F838EFABBD08FA7B342EBCE887CBDE92743A36C14F"
		  "61386BC9FA5662F7A2E39E0CD165E0E0F635478EB28217C1E97"
		  "EE7DF96D6E6143918C7CFE29236E9911D10C756871E749E2CA7"
		  "21DC4F1D0DAFB1715591F3708971409A54712588AF7CEA3FD2D"
		  "0D06C036CAE4F76B885F50B3FE11A39F304C9702BF5F24DD5A2"
		  "006E9FE6AD23AEC9598E34B4B043B092CAED032C3FA42306064"
		  "805E73FE03560ACE3BD87D974C8FB95DA0C68E0EC4B245C849B"
		  "BD06B94A319209514707BF5447113EE3B14AFCA06A6BF308DBE"
		  "03A8"));
  test_hash (&nettle_shake256, /* 60 octets */
	     SHEX("2FDA311DBBA27321C5329510FAE6948F03210B76D43E7448D1689A063877B6D14C4F6D0EAA96C150051371F7DD8A4119F7DA5C483CC3E6723C01FB7D"),
	     SHEX("D0F92C3953C2696FC132FA48BA36EB7576D34C5BACFA4E9D9F6"
		  "DEA8F7B0B6B746F6F7914EE8B9C25EBCA91D601E781B29C99D0"
		  "DA92A1C8670918927A45B2AF16E9BF00CE9A877E361E797F951"
		  "1B9A116AB72209D7E8DBF0E299991B173E4C654021517A94F9F"
		  "4CBFE6FC4BB3C3EB5171435219EC0D80BE57E3900BCE9327D10"
		  "DFDCE70BCD32C129FD5D5BCC54C4BC972F67562796466F9BCC7"
		  "399DB8444C2EE896ECDA88E0F9ED729985992F0BD707ED3ECE9"
		  "7064480AA1D10D4BB176F65DB3327A0F34D3CC32140A95594DA"
		  "7707F565849D1258435285C1B9828723EF42C475D30040AAFB3"
		  "237A759141EDE73070E8995D2FF72C727B1F8A215AF3E511385"
		  "2FEBC029413C2D2F21849CCFF5269B8D188C147D4FE3843D195"
		  "0FC09409BA0E5FD2C5567F11F0DD098810924E56463CE5B6C74"
		  "37163C6201A9252DC484800303FACF0D6B004CC856BC68AEDEC"
		  "5496021B37A667B02F5F7E7234484B531F07ED78B6EB747CF59"
		  "5367EC3EF33DF41D25424A858F50A63D5D503EFEB895334466F"
		  "C3B01DA6246A5801B9D06071D765A65E64F2214F93B6F072115"
		  "343D711697EF2E1463E021CF3ED9CFCBE5A81E54DAB729D52F2"
		  "F309BECF754E3D420CF0CA060C7BE4589D748B84028517AF792"
		  "3767833858A35B26B0AB5C5E3B14E68BBF50AC28A3129F6AE08"
		  "6BF783612749DAF4CFE87E75A4C89DEF988064E32D616F1CCC1"
		  "7D46"));
  test_hash (&nettle_shake256, /* 61 octets */
	     SHEX("95D1474A5AAB5D2422ACA6E481187833A6212BD2D0F91451A67DD786DFC91DFED51B35F47E1DEB8A8AB4B9CB67B70179CC26F553AE7B569969CE151B8D"),
	     SHEX("F31DE8CAD77A334C3480D93E3B30844DF1EC344A8D44CDD277B"
		  "0099F28001239EB0DA5B566FDC383F0E1577F67EDD2BC0F11DE"
		  "67F3E7A4D97C56C9F3BB0FCB4D3E4D08F7F3D5C30FBD2D964CD"
		  "45F36826F614668193554A33BF0170AC3E64C10E4E3570EEF84"
		  "DF387944A6436A814DCC53645D01968C4EBD1BD0BDD76354580"
		  "805BFADAC96470FD354BA2630F7FF8F7AB29282ABA946B1A9E0"
		  "63C316A6D337A7CEC2FB4B562B899F626418BB61EB4F9E9BD7B"
		  "294C7ECA75AB522D642CE495640E4BB1E2B14A10775704DCE5A"
		  "DC7D7E3C091520B48DD18A291841CECE5EF56E1969915FB4979"
		  "60D011A8F46A5B08EB39232283EF332F7AB0D8FDBD900BC2001"
		  "83186BE31A6EFF7F5F4CD27C12BBEEDB0715A0E6E1F6E35753F"
		  "E7F3996C75AE34D2E8E76E43F49EBDD505DEE66536E5F2332DA"
		  "F4EE462B7B524F3BED1293BC45DF9511FC03D2734DA036EB3F8"
		  "C62BB0E623031939E83745744F7DBE93E57ED65EAC1016E4147"
		  "15B54AE83A15FAC6E76057D77C3811491F390C0B4EA0BCE2923"
		  "72A8633E26C10C11E301E57983109903DBBF4D08434A6287B8D"
		  "F665BDBFB03957CEA7AE4EEF506AB1F7AF1358EB48E7FFFC960"
		  "6C266E6CD903C75772E2E88E954585DE90111A250EE62FC12D7"
		  "5C5C58CB7C03C9C06C2AAC9EDBFEB02A1207AE5F9A92D32DCE6"
		  "528A13725EDBE757F309449A40E93286388D8C09F9BFBF49E5F"
		  "B826"));
  test_hash (&nettle_shake256, /* 62 octets */
	     SHEX("C71BD7941F41DF044A2927A8FF55B4B467C33D089F0988AA253D294ADDBDB32530C0D4208B10D9959823F0C0F0734684006DF79F7099870F6BF53211A88D"),
	     SHEX("A3D9ECECA03A1016A8546AD75C7B35102D5ABB0021FB5F2B1F5"
		  "3B591624EF0F6C199F64DBCF325E37F4A361368CF60F3BDA9F1"
		  "1BCF5FF2EB1A4C0B0546B5E898204EA5E4117F519F72F2F5DE1"
		  "017B2BAE0723D91A430B2B3C1987FDD24FF0D0F1CDE2483A22F"
		  "C37F296CE16998C12994603CFAB4B496D3333B25ED48ADB1EC9"
		  "26A44CD5DB14C2072588F15EA752A31A8A3AA559A35EBC900FC"
		  "E948111AF7522AFBDF7C56065B196CDD00FDBAADE3A80D2DB10"
		  "271BDF6418F0F817FE991EC055CCA474B61E1AF6BE6FF6396AB"
		  "04716809433BC8AF75369049E605C1C0028A0D370E7CD0C1EB5"
		  "71FB379B757E8BD21AA61C2E2F2B0D0DBD2D73ACB2DD0887923"
		  "B840079BC74F6C69682118B2B3DFD3FE94D3A76EB8BD68CFA94"
		  "034DF0B5419104675B9F5E56F8C6E0EFF12D9B20E27535A09E9"
		  "FA103803B62C66D7AE09F8E39A5F505F3BF93D27EEE1B16546F"
		  "97AF0616BD4923425A1F0FEA1655334A528C5DA469850A02574"
		  "96C820A351D826EEDAB469A3871182B8435615E8CB1C8C81D34"
		  "F8904DF7F81D48FFDE990B69F0608E6B05AC130709D3DFA9E8D"
		  "9F343622991BC686E80B4F0877C03D9B1E0A190D4C33C11246A"
		  "20CFB297E127FC359AFD037B1A90C040D9E10F77A3F6D9FC45A"
		  "2877711559C8B09348B203FC49A1770730D1206DD03B9E64C39"
		  "0A308BC27896309BE2EBCED1A7ADE8D7187E8F6F9A76523FC82"
		  "0E30"));
  test_hash (&nettle_shake256, /* 63 octets */
	     SHEX("F57C64006D9EA761892E145C99DF1B24640883DA79D9ED5262859DCDA8C3C32E05B03D984F1AB4A230242AB6B78D368DC5AAA1E6D3498D53371E84B0C1D4BA"),
	     SHEX("46F35DCE4FF3A34E944AC51B64AB60D314E30C2CA9C5B4675D7"
		  "59E1E852835977B6E247C025A8FF0248C622C492A04B87C5A2C"
		  "906B2C1CC8F9CF2E03DBBE046644ED26F37B2C4EB2D16B558D8"
		  "2E2A3400EA2BEE9B4D0A3641B7065FCBC34ABD67EABDF5AB7B3"
		  "F278C47D24EE127515A2C94BA955A01A098BEFB59261082F233"
		  "83210FE82208574B95763FCA6132AFB3E644461B1D78F3C8E04"
		  "D860FB0952F9A33A7E56C1AEC1CD3CA913CA62FD35FBCFF62B0"
		  "032484ED07AB9C510C589F62965D676382EC7E55E359F5BFA71"
		  "24A54B83C245725AD796987EF4E9186C74E8B1B53CF02753334"
		  "1278B0A6B621D7FC68B556F0B3E1A95312F4FD3471EF1BF3C3C"
		  "5ECE8269B29A51470ADE59A3F15F2600B858EAF8B6B109339CF"
		  "1B15B0EDF36E0004992BED6E1B187BA60ED3E92A542A83753E1"
		  "4E0CA85A0DF441E5B03A9709EB1AE714615C7C5F2F54C373BC5"
		  "ABE342124BCA909BD4F6696950E0483BC798059A94CD8D852D6"
		  "C5E596A6CE12FF053874F459E68062C5650ECC934EEDE1E7206"
		  "BDE104CF33C95FE103D279108B60B4078DB9522F2CEB28B77DE"
		  "F8E4E59C93552B8C09C0D74E8AECB3B6C50BE0C123E2EB39470"
		  "908BB7288E451C51F6F7154143DC78D5F5C4ED402217A3B6146"
		  "6CE80CCA2257D77DBB99B6069FDC6CDCA949BCF279B3E4DE579"
		  "68DD0EF83C19C5E1A07B19B85FC60E593600470C3F4603178BA"
		  "08CA"));
  test_hash (&nettle_shake256, /* 64 octets */
	     SHEX("E926AE8B0AF6E53176DBFFCC2A6B88C6BD765F939D3D178A9BDE9EF3AA131C61E31C1E42CDFAF4B4DCDE579A37E150EFBEF5555B4C1CB40439D835A724E2FAE7"),
	     SHEX("77B7496ED08C3933BD75983C0C0494BDD8262493A4B55DDCCC6"
		  "4167E67EAC0F6E6307ACC15C33F3963744E26CA6C504D393B3E"
		  "E8165E4D49EB3B6E649207653048F8B822FF884DC74937443B1"
		  "C4A888C7A768C63D5B5D29E7446873923B9D7A56FA5D9E90760"
		  "AB86D5718E3464821B79EB46D169141FF16120BFB650C76D4B3"
		  "E5B3F6CE61FEBDBE09AED7F4C91066D903AF6E56531E8FF7154"
		  "9508B6E420CAC6BEDFE0CBEAE6BC228476BC8C00EAE43D40C82"
		  "CBDF6B460C376D7C11648EB2815B6506ABD4339B25D58D45CDD"
		  "0A0B9E35A88E251FDC34D4810D659D179F59EBD03717FD31A63"
		  "94CE12CD5569066E138885CB2BDEBBA06367557CE849EB869F3"
		  "CAC38800D51C22B666AE2701E580796394DFA02F4910BF5F86A"
		  "AB53951233364EA20CDA35AFBAB445BE7F686643856F825394B"
		  "E7B4B6D2C918D0151F46FB9AEE8A7BA2D706E48CB0BC429B064"
		  "262C1A0EB3524FF14632F5184575C15F6F4A3446E93CB4E86B6"
		  "A931BA268409CE30B4595FD2059A27183B3BA8D0ACE8E482866"
		  "D5C7D5B03DB8DBD24B99D59EB6EEFFD209E124535D154B98F99"
		  "91D84FE1AA763C5133D41ECC233930957DCEB7896AF70F735A2"
		  "F5C1E79480AFD50943BC5014BCF0A7354AA7F713163B55A1E41"
		  "BDD05FBBA9C1DB2C69043ED9EEA4FA45C990CCB4A8DC41AFAB1"
		  "8164018E54C47AC5BD6980FD796ACF0DDB42C7042A4877E8BE3"
		  "DE29"));
  test_hash (&nettle_shake256, /* 65 octets */
	     SHEX("16E8B3D8F988E9BB04DE9C96F2627811C973CE4A5296B4772CA3EEFEB80A652BDF21F50DF79F32DB23F9F73D393B2D57D9A0297F7A2F2E79CFDA39FA393DF1AC00"),
	     SHEX("8229BFC635A769D86656396B7723FB46BBA9868712F27C37792"
		  "5CA6B358B8391E0AD8C30DA71FC8F0716BB95ACB0DA00C61F3A"
		  "7BC8DF1315715E925F1EBFBFB5D72CB3E098F0C014A80E88A44"
		  "231DDA194DFA8E35E359F60EA5D7A5E0D1E64A8FDE294F4ED2E"
		  "3E98294DAB838B2C6B3FAFCB2995AC17AF67669A24FB7131823"
		  "3A82DC8B934C8693B3DF784A5BB34CE9CB3FDE79AFCBFA66C07"
		  "D1202C954E849DADC0C1D5BA79BFA74919C1C6FFFDBB834287C"
		  "852522A65AE3D32E337C2BF16C3B5E22D4577F5B05F1B872185"
		  "32041B9692B0EC561881479C924BA92E154B57A1AFBA6214F68"
		  "FD067D109A92A9B0E127A0A6B78E85808268567CBD1A2653201"
		  "233B4E80BE103EB951748A1D1F8A205856A650C58DF15E8E1C7"
		  "6644E52614AB4DABF51A2320A98D259F8295412291464E1D918"
		  "C9BB8F5282301BE5F91DCC507F140D8642B7A6FD37327CF38F5"
		  "10679845853CFA039FF4CBF749D48032D6650BC7AC2BECAEFC5"
		  "672CA7C61A8F6A1BD69D321D2AC1E6095B3AF7111F509BE0062"
		  "707617C62DA3336C3086C39B2DCB9DA7F23BE732EF32F906243"
		  "250CE4D38668ACC8CEEBEE87C43F077DF2DF1EA4F6345477490"
		  "FE37725ECCB8D4F47B14A85B0D36FEEADD4A020DA3DDA4A4889"
		  "5DDAA45B5EF8499E937D06BFE6DF88897A828DC19D9AD93C622"
		  "E0263E74F818F921C4200856C9EC9D36CC99A6B0BD59FCCCE72"
		  "CE3D"));
  test_hash (&nettle_shake256, /* 66 octets */
	     SHEX("FC424EEB27C18A11C01F39C555D8B78A805B88DBA1DC2A42ED5E2C0EC737FF68B2456D80EB85E11714FA3F8EABFB906D3C17964CB4F5E76B29C1765DB03D91BE37FC"),
	     SHEX("66126E27DA8C1600B68D0ED65E9F47C4165FAA43DC4EB1B99FF"
		  "EDDC33E61E20B01B160C84740B0F9FE29FDA1FB5EFF2819D98C"
		  "047CDD0CF8A0D396864E54A34657BD0C0355C75C77E5C3D9AD2"
		  "03E71FC2785A83D254B953277B262EE0A5BB7D0C24ED57FAED4"
		  "FDB96D5FD7820E6EFEEB5A9E9DF48C619C4872CF3B2516DBB28"
		  "073273E2693544E271D6F0F64BE8DC236ECD021C00039FD362A"
		  "843DC3681B166CBC2407495E18903E469403807FE623F3648F7"
		  "99F18FBD60FFF7705D07464E801E0AED4F2F0642B9A2C5CDD0C"
		  "902B59B1DA19A09375C1C13175B618091B8882A0E7205EE63A9"
		  "219ECBCFA943A10D2D9A50C8C0B5D43B003F67EF0D52ADBF9F6"
		  "59BB62FA6E00678BB8D4449648872A99EECDBB3DC381B5199FD"
		  "500912AFA93C63A6B23D00D0A416468FDAB93AEDD9115265BE3"
		  "A4440DD4029FF7F88D9755623E77F9430B934DAE529BE9A6B30"
		  "7B1B292AB5918EB24B14598554B4CC6269419C701494B7CBA5B"
		  "3D69F6CDCD5181FD03E0748D08E1E0AA5C4EC62C47877C10858"
		  "73C016EF24E7E45DA71D3DB9DB23B153CCEDA9A9AB5CCD8C546"
		  "6CEF29810098E976E4867075601F83A2D2CDA1A476A1E990CE0"
		  "4C4567FFB99AAC428922D9D8B25AF68C36463D3AA4F689CD778"
		  "F79E743E0BB5F935E6D45F978DCB2AED12DFCDCA469556556E1"
		  "9F25D4C959C98785FB471D4BD1675D3B84742766D5BA4BFF2A3"
		  "F912"));
  test_hash (&nettle_shake256, /* 67 octets */
	     SHEX("ABE3472B54E72734BDBA7D9158736464251C4F21B33FBBC92D7FAC9A35C4E3322FF01D2380CBAA4EF8FB07D21A2128B7B9F5B6D9F34E13F39C7FFC2E72E47888599BA5"),
	     SHEX("EFAEE039C0412FCE8F55F6E7772889EC18D0604FB18040DC1E5"
		  "783596CD820B421A0DCAA528C8A62B17A22164430672DA6D818"
		  "E2E555AA8E79665A6F8F5721A4E17FE8FEDA551ACC9116F1F50"
		  "E95839FFF2427DC1D988B0269838102547D4D46C11D2548BE3F"
		  "885111D53F3061A74972C56579C681C20BD5D47A4C2A9523BCE"
		  "154068FDF3813F5D45A8D446758C212614E3A6E80EBCFEF81E4"
		  "4034E0F2D02FCD0AE5E6B10DC24EA09B94DBC47873768C0DC6C"
		  "F2991B9477448540924CB57D3582D7B8E453E946C57129285B5"
		  "48FCC831B3E311CFFFA3161941689E3CD649C3F47D96037804D"
		  "0C6A4FA8C09B11A7D5A35F6C4AB89B64C735153422A3E529E19"
		  "B9AD7F7CC346F904912E1A6C098CCED3BE9755137A26907CFD7"
		  "F7AEB1A573A971C4A5760CA5399CBE642F0106497AA1D364DDC"
		  "DABF375C547BDDB6011F26B564D0CCF4E055DC0869BC280391E"
		  "5C0203005D92246E377DC560D16F3A9588068473B14FE7E39F9"
		  "C25108EA279D98DF21902E60DD4EB03266E873D3B0C24DD3306"
		  "6991386C4311E58197F24AF80FA150068407821C2327E900340"
		  "550E7826B2F510AC65D4B21E9336610945A0E5A0EC8D132D694"
		  "316F2B8A21CC24520C6204EF036116B2FE66A79CBB202F65E1D"
		  "1782AE10CC71BE51DD44718E2FE9D229C705B94B9BA6B27A392"
		  "5E55DA290875588C8EDB8A1866FAC9EFB7FB292F69A89ED772C"
		  "C68D"));
  test_hash (&nettle_shake256, /* 68 octets */
	     SHEX("36F9F0A65F2CA498D739B944D6EFF3DA5EBBA57E7D9C41598A2B0E4380F3CF4B479EC2348D015FFE6256273511154AFCF3B4B4BF09D6C4744FDD0F62D75079D440706B05"),
	     SHEX("F64CB396701785B64767D4F17A768EE198297A74E5D1617FF74"
		  "A3323C096579C0279E351C2407BB889CD4AE718B09ABA711FE3"
		  "AC14AE7D7DE0072E8BB0B6A1F92A1C93DDB46DE891CFB1F3681"
		  "4E585F672BA871601E817BCD6327F3A7FA7DE35AF6A0948DCF3"
		  "DEDBC8A7154CE8529033F67FC656F95FD47B60E49681895EBE0"
		  "824DF50F8EA0480B50EB3264243A88BEF29F5F4FBA012B86A5D"
		  "FC98054AF4D15248CDADCB160193DE7068CE71EB76E32ACDD0D"
		  "D94F6B27B5158D9E63EB25831219768454C8A951E9678F41EF6"
		  "F654CAACB0F2AB5DD614BF0BBD932DE06FC31D7259309B23DF8"
		  "B5864322EB2D8F39A07E5A3F324B64876612BE4A1C47B06F7C5"
		  "BB814169D94B78CE22AEB7249C96A29C52BDB5550C7B292C96E"
		  "A9ED6700F5D3032D2A81CD68ED3A6DE8FCAF9379ED0D95C94BC"
		  "B0082915FF7DB5EA1BC25645EF3B546A728AFF4CA20C6F8288F"
		  "F0FF4B7DBF838E1A3EC5463AB88CC827D03A748FBB924797A98"
		  "D309BAC09E73215BF8BE798C151D322D6110C280D85B45785D8"
		  "54DA39A80F64897918C169BD7E1FC52D003999C084BF69B696E"
		  "6D747E859DD2D6EC6FA1339A395858477BC49241B302FC74E01"
		  "88A2A8138507331923C357AB4EED3F64FFA908CDAD9116039A4"
		  "69229F9A62070799026097EC7F5A71A7FB01B5473E8035D383B"
		  "9F236F2FAA0E06DBB135A939EF9CB31AF1E312F47C6C9BE1F50"
		  "DA36"));
  test_hash (&nettle_shake256, /* 69 octets */
	     SHEX("ABC87763CAE1CA98BD8C5B82CABA54AC83286F87E9610128AE4DE68AC95DF5E329C360717BD349F26B872528492CA7C94C2C1E1EF56B74DBB65C2AC351981FDB31D06C77A4"),
	     SHEX("CD85DEB9DA581AF66C0B3125B697371F16EE34606A577AD6D8A"
		  "C7E4E4A948B36C844EC9EA85EB168D7A5A1EB4B72A685AEB80A"
		  "75075B420C9B53B963D960F7BC88DCED631C734114C28B2E20F"
		  "0657D9256AB01B1AFEDDA9B9F85FD30D0DE3B98DB38AB90DE60"
		  "045836CFB12C41E5BDAE57B937B637C11ED47F7DE242EB9F72D"
		  "3253C5D883B22333F181FFAD60A541DA08601791FC0F8D9F44A"
		  "94B551B1A879B76FC2B7931A2F4301D121A4E6861F5C84FFCB0"
		  "139FC37268B5F8A4C3FD490809CF44F68BC1E665B369D4D74DD"
		  "0BDEA71ED0514C37A47124CE146927274D95067C80036ED9F1A"
		  "5B0A10AE71E837A09DBE4DC358DF4687392D99B2ACE8BEADA96"
		  "56677518B1607C8E13B7F45100BFCEFA1D4E38B9BB29EB23D17"
		  "B9CC66F70635A6C531BE9CB890AE833CD7CE35498CC9F81C576"
		  "493913BAD87532B711F3F88873E9ED48A83B6B2C503E096A33F"
		  "824FEB4CCC702E82CEF00FBD938FF9BEF6E3F80E149EB343381"
		  "6A1D3FE7F005734192CCB5A8B0D7D43327B535547A9CC7F5FA2"
		  "86F9EAC26E9E6A1CFB86DB35831F75A9902A3E78F6BF9B47288"
		  "36C81B3C614923C3EA88D6C5F55449A83EBA0F5FF7B4F070842"
		  "06D4590BF817C1FEB43595462037AFB6969A91EEB963BD244AD"
		  "B1B654FC98A0B0BE99029B3D5BDD69D2158939D677B054CE55D"
		  "ECF0F332851E0A74EAF2BF3EB672D4CB1F467D0E5391F98501F"
		  "EC2E"));
  test_hash (&nettle_shake256, /* 70 octets */
	     SHEX("94F7CA8E1A54234C6D53CC734BB3D3150C8BA8C5F880EAB8D25FED13793A9701EBE320509286FD8E422E931D99C98DA4DF7E70AE447BAB8CFFD92382D8A77760A259FC4FBD72"),
	     SHEX("A90053A8F738A68C18CB87FBAA8370BD14270412D0014C5DE80"
		  "08FBB74E619182A0442C121F7E2449A5B019DC0CE597BF1D7D3"
		  "E64D1184946D45208A864723A73CC1E242F7187811C522F880A"
		  "CF53056D835BD150FCB7DB61363D22074349FF54A3E5FF25712"
		  "72910A7333959D8CC29F69B540E5ECCA50DF7799236338E2925"
		  "5656930E22679A3A585E32B88C27452BE75BDE73789B45623E6"
		  "95FB7408DC51891D50A3BF84C5088C5F6FBB1B740DEB7FE38BC"
		  "A05F1A3BBB516E9AFED0A41F8FCE94411CB84D8BFD51EF91718"
		  "4A66C56B31A190AE02B86F62CDA2FD7471EC030E94EDB56C144"
		  "025A1BA9B792A515DBBF402D33BB6451825E52A87B7B82C44CE"
		  "E32669F96F3D8DAAC5923127B51AB5C5ED43E5A9344B32C5A37"
		  "05345EE98CCC5259C9D3342CC10D660274DD628F1C2C031FE9E"
		  "D282500D5C3982C7976620FB46190C57856E519C6FC1760A306"
		  "C3497F2E001C7113248F53EA1A4BF9DB370285EE441E4B43459"
		  "B6F8690CF10BC1785138F8855DF859BCF1AA581DB4A6EFB576D"
		  "938FC273E7D126CAAB7FBCBAD62DBED1FE2C33F24640AFA899D"
		  "EF2825AC2C0FE928DF223B1043117C061F1C7EEC723C5CBFA83"
		  "14E1B18EA6CB63C02B9D6FA3B27929B4D42F1D785813FEFE124"
		  "9F65B725D4DE59AE071A4F6A40AAA26935F4DEFDFA3760C98CB"
		  "E805A50DEBB3011E006015FBE8400CFB1B6B3D2162014D675DF"
		  "4246"));
  test_hash (&nettle_shake256, /* 71 octets */
	     SHEX("13BD2811F6ED2B6F04FF3895ACEED7BEF8DCD45EB121791BC194A0F806206BFFC3B9281C2B308B1A729CE008119DD3066E9378ACDCC50A98A82E20738800B6CDDBE5FE9694AD6D"),
	     SHEX("27BF218A01125514BB1B78E582AAD16720267373BB27FF96A59"
		  "4B7F10CC1D0A393FA753F50437C89301542D27C12C03F53FF77"
		  "1CD0DF4B38F40B607B67CF419020D34C18F5BD3BD424D39F47D"
		  "118C84A53A635E17365F84FA0340BE3212056C155FD227D3B52"
		  "F9BE75538FC23287B5DEEC0E5737C8484ABBA0BE6CC43D956F1"
		  "7A41CF81DCE5E78326633CF09326E0004B172763089A593DFBB"
		  "C1A0960A16F207F464F9EA2AFFE732208E970E00AA0BF122863"
		  "4E98031155B61F6EE509648D4B0BF58FC9CDDD0B470B95A0AED"
		  "8A3A96DEB2F7FCF65CE08A826300D4E48A681019C5A8EED66E7"
		  "FD5FF098308138B4E11886528AA9ED32617392B0F6E133A9968"
		  "3DADDAC328838008BEFE9DC680720BF4231E88848660C8EF4A2"
		  "973046D8E70D8EE8D8497AED8A748B3185A77B238B5650C9909"
		  "5D8702209C0C31FE1770042FE3BDC2582F5FB841EB5A234CB4C"
		  "9B637FB4C5DD9F90225DB4C62DA420F3D1895FB2EB05E3A3D67"
		  "67F73FDBC8B94D48A555CE4CDE5045BF5325B782E0C242FB3CD"
		  "27D963A9EF014DDBA4B0608F2627FCABB2EDF570A49AD5377B1"
		  "BC7193D9CCCC23F5D35C2DB59D05DF8D4A8E5146298B6EBE1DE"
		  "25BA1089DC746EFEF6BCE22B43FB99EEABDDAA76566452F544E"
		  "AA85CE4559BA683A122CE76927242050FB236BDA091480AB1B9"
		  "ADCA148B639F986A3E936BBA565B54F727A0D45F369DF25B9AE"
		  "8528"));
  test_hash (&nettle_shake256, /* 72 octets */
	     SHEX("1EED9CBA179A009EC2EC5508773DD305477CA117E6D569E66B5F64C6BC64801CE25A8424CE4A26D575B8A6FB10EAD3FD1992EDDDEEC2EBE7150DC98F63ADC3237EF57B91397AA8A7"),
	     SHEX("2FCE215362A523D7B76FB836EB458667099C58BC51D7EE6F6DE"
		  "B9894B6A337576B0DAF9A806488C3D238762B7A2E7B12AACD1D"
		  "696517EED142AA5DC5A0747F08E5E39A2486D75F497CBD6C489"
		  "9D6F5BF2AAAC9189A6ABC6E787B0D9C5283C840E518C6AB0393"
		  "CF60337369251DF935A02DE0463DBFA4DCDC1DE9034FC9C21FF"
		  "B9DBD7C48384DFF31012BD5F5A9CDFDB7139C3C35F560913D64"
		  "3DDB728910413F1267BF1BF1586C5EE69DAD26B53154934762D"
		  "4C80C34D511BDD8A391F83D224F9F8F83B61DAA046C542BC78F"
		  "153A3AED27E1268058F6952258D274F11C8D9634A5BDD4E69C6"
		  "0DCF1F89DF2238EADFE783CBA652A554186D9E8307C7773722D"
		  "1049C3815526F583612BBD8F6C587D058B80FBD38113957E602"
		  "12777A9F67B612312583FEFB67533C065286CF371001440B1A7"
		  "A43DE2405E5E92D633F12A63EE2EBC408164545DF22803FF239"
		  "DEE66EAA20780D103D940880E8BF68B9D815EFCF16F59E0D858"
		  "AF33FE4D287A2B8E61C6EBF26E16E354B3EF63774EE01A14691"
		  "B149D81F010A655F308E966C990E1BEFCB6E4253FF43BF473AF"
		  "A78AA88C36EBBE735B6B2F92BA9D07A738E000C9B42BF8A3A4D"
		  "8FB6C38AA36B3329F909676183FBB520196D0DEDC80DC6BA0DB"
		  "4D72AC6544551CC38B23AFE000A91F10A89956D2120494543E3"
		  "FF5D7D1B2791207860A0B921DEBBAE769E55713A1D611F6B117"
		  "5BD0"));
  test_hash (&nettle_shake256, /* 73 octets */
	     SHEX("BA5B67B5EC3A3FFAE2C19DD8176A2EF75C0CD903725D45C9CB7009A900C0B0CA7A2967A95AE68269A6DBF8466C7B6844A1D608AC661F7EFF00538E323DB5F2C644B78B2D48DE1A08AA"),
	     SHEX("992E831E9BF5FD994A73F17699CA0ACAEF65E8212E5C4A49556"
		  "9654B5DB523BB04431B7910A343FF254F4432485B4A6A88701A"
		  "FA889B11C45C53B4A1C4A1872FC685D644DEB5DE2340511F29D"
		  "062F6CF0F39803BC333044CDE83103155A49F77FF8946A77FB0"
		  "92D1CF1715F02297ED4F6B0A43254863641C4C244DB87FE486A"
		  "3EB78DD3B57CCE06722CCBBF53ECCC862A8256625AB03563FE4"
		  "730F74C8F78037BDE37D538DA822AE9141B8098A8B57D74FCEA"
		  "A072746DBED8C4B23F91B3DE8B9B733D06CCFA2FC18747590C9"
		  "AC01A9FA675D8CB51505098F8993C494592C2561F8F2FEE3B26"
		  "7D8FB77A1AB853432850F90B8C52D08E23A19226A61E0589842"
		  "ECEB48059F8F1B749D563E2C652B70C426B91B8B0E19EF2B319"
		  "F2D7DFC25F0D712A76CA9332F92BB38CF89DED9802BB7AD5654"
		  "B2357325CB1CF6F1C9FB364EF41EE8B0A8BAF5FF9E8878E56BA"
		  "4BEEAE384BDF029E4911DF8E84F5B570704D53B67D6BB1AEF37"
		  "976B38F405D9AEA67C6A6D77370E37BD78034645A9B64056723"
		  "66DC061BF384EB0E9E73466FF5D018C9BA52DD262DC97970CB4"
		  "B8AB467847C23DA0FB101F5A7B9EBA51ACE6C0119ED03C7A14A"
		  "FCA676BED44922EDCBACBC79B6DB231C60DCD4ABBBFA0C13C0A"
		  "B3D680AECA78EB9BF0F71CE6886AFF6C309594A9DF2D01692C5"
		  "6265A6E9256C366C4C53B6C0FC38B6EF18ED8C571D94EE27E85"
		  "0D4C"));
  test_hash (&nettle_shake256, /* 74 octets */
	     SHEX("0EFA26AC5673167DCACAB860932ED612F65FF49B80FA9AE65465E5542CB62075DF1C5AE54FBA4DB807BE25B070033EFA223BDD5B1D3C94C6E1909C02B620D4B1B3A6C9FED24D70749604"),
	     SHEX("47CB72FC64AEB3F76085626534EA346A2B4797B5D61839BD7A1"
		  "5DF05D5E3552C2751C697BC2C532E2B3A8934B9ACD898567145"
		  "0DB65D6F44D9B6E27972FBE050E6A76537EED3E9FB15849ED61"
		  "C3B2F6E30A8523DF8E95BFAE3A93BB306343212F1C288113594"
		  "42759C85C3E6829A179C03F86A442DBABF49940297DE3F29CA9"
		  "F6EE25EEF49B9C7B37DC4499A9D16D3B32DA2AB391AC1E6BA6F"
		  "3967532FA18E50B1E52D6D3EB2447EE20CE459ED970D71E33D8"
		  "0A20E933F09E494A753D15590A29B15733FBC93217B47B36851"
		  "00AA31D5029DB16FD7E293C51C55E54E15457F99009590EF8C7"
		  "B7BE110453B73B4652B2F9C048BD7F2EA2590D33ECC5508BD62"
		  "BE3A0918C174D99C5939FEE9C48592CFC220F59F28F57777DDD"
		  "43AAACBD23A61A8473D4CCF01389E982877282E8694F03683EB"
		  "B44406C798104B9F3223B0DF50EC964665492EEE586CBDED57A"
		  "75F17762A2903604F8120FD1D981AFED7D6AEA59FE70DDCE738"
		  "A41F53F611B927EC3DD3AB7E89A0B0B3943EAA2A74FA2C02098"
		  "4DCEB577F6B3E7AE98CBB81F3CD9780E52D44BDEFFAF466FDAC"
		  "7A4429BC89295204B0FBB9C8A22A5F202E8536542C244CBF300"
		  "0C0CC1CBB0C4E18A29CC92697B0E58AB1AE77AE10B1D3EC92EB"
		  "697364CA69279C4EEEF39D27C06E63DAC76028DF7A5A974F8C9"
		  "7AF5C7AA31B5BB5F1F4DB2E63B3D4C9F6849129084C77ADE975"
		  "62B6"));
  test_hash (&nettle_shake256, /* 75 octets */
	     SHEX("BBFD933D1FD7BF594AC7F435277DC17D8D5A5B8E4D13D96D2F64E771ABBD51A5A8AEA741BECCBDDB177BCEA05243EBD003CFDEAE877CCA4DA94605B67691919D8B033F77D384CA01593C1B"),
	     SHEX("458FF16E716EE4C47F3B60B3E473D91157767621E9E3A200EA0"
		  "BA18B511B00C9BA311C7D542289D03F07ADFA6110675DCB43D5"
		  "D8EC69C0EEF6AAF4664F917E01FF8DC5F203E89540EB96640FF"
		  "807ED60D761C4D3D92F55604EFA648C5CABA130731FC2BB5351"
		  "46EF6EA4FF3A7CF388647EC25C3879671874BA259C6ABB31B4A"
		  "8090BEA5AA7D6B05F5647CAEADA7BA35B45FEEEE744BC0E76D8"
		  "EA6B84CC4103D358192BACB126800CB5A6E114709E7DA7766B7"
		  "61B6FD0A83203B82B74E87D426E71AC1AFAE590F4F06008537E"
		  "79E2025007CF2F5ECD0BC1A0E14C5CB8384B9DAEC501CD985C9"
		  "858212CE2B9AB2422F6F7EE0E081FDA987AA90A6D51DDC1173C"
		  "CBEE0592E970CD3DDA52431C57FA323758AC0A4E47D049EF3C5"
		  "1211A4F9312614312B6DC119336D7BEBBF325C2C4D24727E290"
		  "9058543BE4065C58F4E9E3F3F0ED88B8C392CBF610680123210"
		  "D5214650EF8DE411B7F6EBBCF2DD95F7FEACD074A29516E42EF"
		  "BE37C239E9AAAE6E0E1E1C61B418BFF50F0D7D56347579A75AC"
		  "A891D51CB5299987734EE208F0D85512C846AED6DB2A2F453B1"
		  "10FE2475852C92FF1BB3439D9D6F3EB76B49E239F72883D067B"
		  "F9D1BFFBB1B668D677F940940B9D042C06EDFA9256B6AFAA37F"
		  "918A99309E4E40CD3D374DB5A37BB43DA60682FFD1D01CD1EBF"
		  "9E90971FE16176DFFDA5867150FB83F271E4402ED52C1DF78F2"
		  "BF52"));
  test_hash (&nettle_shake256, /* 76 octets */
	     SHEX("90078999FD3C35B8AFBF4066CBDE335891365F0FC75C1286CDD88FA51FAB94F9B8DEF7C9AC582A5DBCD95817AFB7D1B48F63704E19C2BAA4DF347F48D4A6D603013C23F1E9611D595EBAC37C"),
	     SHEX("021F06C7138F663522BCD444E597C2B78F4C8369C88747A1550"
		  "80005E09A68C6A73C4F8D1C7C088D3CDA2B48F310D5371F336B"
		  "C261697542028D48FF92523DD78C100CD858FC5FD1F49181424"
		  "47E11281881D9C22680800DAC76D690444458D53E35D263B2E0"
		  "81F470C9E8577565EF7F8C530F78AEF2039A0B253A28E29E6C3"
		  "0FD26FFF9677E65121B5175822D0942D0DBBA9F764514FB9363"
		  "69F74324E8F950BFCC192A30921F04552245EE79CBFA319063E"
		  "4DEC133E10EC6B02002FA61EBC288B8404DE81D9A513FA7FE6D"
		  "B2C61BC4ADA6DFDDB4B49B5CAEE1D7CCF5BA6F40A39AF92EA26"
		  "222850D4F4FA9079F79365806440B89663C705A247629C8E28E"
		  "77FD17D39ACB37BDA2DEF7847C12D7F180A67BC788795D6AE91"
		  "87E6AEDF164E282C1E76322A8F38856C6D95460977FE8F6A760"
		  "F49111400467E7E0EAA5AD7E9C5F9A17B462CC8B45F308CBA67"
		  "71CAB434F407CA6CCA371A6C7313CAC055A13D5C79195DBD949"
		  "A4FC9C176C26E6D5ECF343B199E478A25DDA55FEC4FD416E270"
		  "8AF001BB8AC3773E7824BBAC1C0656953984B1109F1F95DB0AA"
		  "DE0CB53BE77E88CA83A86563E1E87011E718FB3616BA3E54A2B"
		  "F9DFF761A42EE1E809DD1A431C7A832442461DA90E7AD5FCD58"
		  "1827BFB2C8FDB0A046F09D30773FBC1481DA0FBB92A6AFD073B"
		  "A96811ACDEEEDAADAFCE7C70E4CE75421C85E9C1AFB270D120E"
		  "186A"));
  test_hash (&nettle_shake256, /* 77 octets */
	     SHEX("64105ECA863515C20E7CFBAA0A0B8809046164F374D691CDBD6508AAABC1819F9AC84B52BAFC1B0FE7CDDBC554B608C01C8904C669D8DB316A0953A4C68ECE324EC5A49FFDB59A1BD6A292AA0E"),
	     SHEX("DBC0350CCAAEE7F6E18576E435CAFC7CC65EBC81B27DA2F18A8"
		  "88ADEE19418BF6F4D1B3088E5824BF663028A690354F4A953AE"
		  "73CDCE6B095A835CD45ED5752D72E699ACF031529D73348218E"
		  "AB5DD6F98D675E33880F6E79D0FB3A78E843F26E018F543872A"
		  "313560F8024A6756013DB3AB13FB23661D33EF9520EA60A0467"
		  "5454B5F3069230447000ED2A879A1B342F560A8AD2F37AFAA80"
		  "668E90001D315BA266D03152E714434FB0F52E287A235EF5DC4"
		  "252AABC8722B6920A069C98F69C64BFC31B1E13D01DD0052440"
		  "1D4F8494111137EE283EFE82C2215FEA54304C3297E6A1A88E4"
		  "6F000084AD090551A994308165AA2D0D96585DD4C826554CE80"
		  "D3C00344140D4387322650F857350812C6C4E660D2E3EE5DEC7"
		  "3D27A6455D6170569FB0F56313A561BC6FD1BB6FC11BC6A830F"
		  "32847050EAC3451E153C00BDAB83D8CEF319894DB18DD80F111"
		  "2E560E2353DE9C2DB6BFD428327AABA644C218FABF3D75CC42B"
		  "3773DB2113D037588AF31F1B1F21D076F285F1F5CAFE5312E7C"
		  "2FCA07AF2E6FE3651AEFA91BEA2740AFD1C2AC77EF03BBCCEA3"
		  "6940AA9A7D733289F9575E9E14617AAE7402DD7847114C75EB4"
		  "432FFC2D3D0BD56117F9286609DC91E9CA63E1E064F09F2653B"
		  "A905CB12C8AB6D7772203B0AFE6332C4E9F73BA4B652FF16E97"
		  "59FB70AD5F548540C7ECF6C6DD7D17C5D2A9B45C548B7EC8819"
		  "E8A5"));
  test_hash (&nettle_shake256, /* 78 octets */
	     SHEX("D4654BE288B9F3B711C2D02015978A8CC57471D5680A092AA534F7372C71CEAAB725A383C4FCF4D8DEAA57FCA3CE056F312961ECCF9B86F14981BA5BED6AB5B4498E1F6C82C6CAE6FC14845B3C8A"),
	     SHEX("7AC057C0DDA89D3A001AB997F2E405B4C1D1906191C3A50399A"
		  "9A49B2E01CB2C3D20DB6618A7AEC42F8143C580505145E60D6E"
		  "1D4559FE3F46F5AA1373502EB0661A608C31839BC4D7FA44AD6"
		  "586FBB53FBAD598C05B9443D6CAD708C8664079FB276947C4EF"
		  "022C1BC7267B41C1764B249C8E7B347BF006DF1498E31CC9EF2"
		  "558F96F7F7AE323D6A633E1BF23312C1E9E2146676730954648"
		  "727B61A5DB507F1959250D49A52F9804668AB3A56ECB6C49EA7"
		  "819D093E85A0A1335CCD4F54360466F7C37DF0F65CE52F7B396"
		  "B49D39A3AAAA0FE33F053EA711351BFEC78A1E05F55954D7056"
		  "B7380430B45275E2CF57AC13F7FE332B760D8BD793CE4F88613"
		  "0C3E4581A5995F865DF2C68CB2FC01B558997A54AB8A684F5BD"
		  "67855972CB3BD54A9620F71CFD3C9F0FF260B80CB14CFE4FA9D"
		  "61583CFABB12BE42C4C64C85D1F26D3B0645AC60065F9E85C70"
		  "883BE7F06B9376737F83313DEBECAAC3F1146B050F8B360A614"
		  "B6C72EC91A3E44B196713F5754F1249F6DCEEBAEA8049FF32F3"
		  "08513F9C0C2353C9814C0E67CAD64A1F332624490A39DF8F9BC"
		  "FA61C315CF2563031429567334038F1D086BE0B9A4BADCD9C4A"
		  "0840348BD476C00A7CDFD8862E80EB9A833BD2BB56D88CB55D3"
		  "D064326B8FA084F92F3DC2D8CDFBA3E3A0E3EB9E44B1EFE0563"
		  "BC7A75F9A736A87A89A15A6812110FE92ABF5B8F77EA88C2207"
		  "517F"));
  test_hash (&nettle_shake256, /* 79 octets */
	     SHEX("12D9394888305AC96E65F2BF0E1B18C29C90FE9D714DD59F651F52B88B3008C588435548066EA2FC4C101118C91F32556224A540DE6EFDDBCA296EF1FB00341F5B01FECFC146BDB251B3BDAD556CD2"),
	     SHEX("F0C453711D0CE1B3A120BFAF25570E3E949207F575782FFBEB8"
		  "A6600D7CAA9938D28EC6872D1FC914BF8BB02331C40728D3BE5"
		  "31197694CE644DDD148D0EF54C077D56B6506A4A941B087A61D"
		  "3D57B8AE53824DA8D8CF069D1CF5DF46E309108D74F318D38FE"
		  "1968F1550AD7921D976ED5ABC60953E4FD073A3AE7FA378A105"
		  "E0FC3BCCB3CE3469A536EFA3F9E5C8590F19EC7E8708AF92A9C"
		  "A35CAEC1743F2284DA30E613AFB371507EC61C5574771BD7436"
		  "583415D8E4406EF39150E50CBE233A781E8867639DD491B1F53"
		  "217FDE1FC5B93671C1B6931E07966DE705D162F14EF3C95E67C"
		  "40AB1C0841380F9A9165C5222477FEB97E9A3C88104D7CEE9B3"
		  "EC6876A0ECF0198D258B9E308083B826F32FEE17E2A59FC2004"
		  "6CA815FE0955CA88A81FB56A7C90BC922136AB0B3F3E034B626"
		  "34FC830837F2DFD86798A11B335D5FBA27398E5023CDE6F1340"
		  "241509E92C6A4B298B3BD330ECB2012B8F95DEDF67B68D5309F"
		  "A1AC9DB856561D6E81666A49AADE4DE719E1A4B519AECE0BD49"
		  "41A36492E0B79CB7EFEFADF4EDC6A344C4B0640A17F0E7F70E9"
		  "0E77DDE40A82457DCD58658316B4B9378BF058B05AE39D0C750"
		  "692B92BB2D16B5F43200DEF81CCC61B2A26CCE991BF939534D6"
		  "1592328BFBA68AA8C9F51F3A04466759D7D1B966789FCBA85EE"
		  "120C66A8F3C308E6A1533220E4A38C21DDAA4B2FC51FB49F59C"
		  "84B3"));
  test_hash (&nettle_shake256, /* 80 octets */
	     SHEX("871A0D7A5F36C3DA1DFCE57ACD8AB8487C274FAD336BC137EBD6FF4658B547C1DCFAB65F037AA58F35EF16AFF4ABE77BA61F65826F7BE681B5B6D5A1EA8085E2AE9CD5CF0991878A311B549A6D6AF230"),
	     SHEX("23F75EDBCD5D5F329F01E45502B66995CFB9A5F0748E9E1BCB3"
		  "44B639873BBA79D21EF36B8CC1BD8A2357D8DFBF3C32BA6C0C2"
		  "E8936FB968389D8D1388C20B9F572BF9CEF93891C643D672C86"
		  "55BD67ECC3D0EC9452C6D1A0789AD3C734B926188E4A1B3D684"
		  "F14C0A5D9E72CD2A10C5CFA8BEDC0BA5F3AC6B5150A256F2B94"
		  "5586981F8092E0827294FB846986D3885F38BC6F945B900EBB5"
		  "46094E29FACDEAB1BDC1D7FE1C6C83E562D09E08319E2BCA1F3"
		  "EBD4A82B9ECBD8E6B38906D861E2E0704EE4F55DED4E7A74F18"
		  "8210EE439183E80DFA8795A1F75F0EFD49A4FCEFEB97F187360"
		  "69BCA02273FF542ADBB6EEC13128F1D74C87D6C6B45726F62EA"
		  "57ED49321BE6FA516B1AA2349B3C85A5F971C4E15715BADEAFD"
		  "2E8A217D1188129CF49A54FD85803EBB361D73F0D8D0A7D557D"
		  "0E17C8D83C27DDCE473908A1CF9ECF0F8085E77FDA3DC89C826"
		  "09A647E25FFA8D2FC86194B2A793C7B32B4C893115EAB9A7040"
		  "85047F9225A29236C6188F017AD4972661C796A3258ED494935"
		  "33BEF9A59A27DCBCF614AEAAF20BEF156AD261BFA23FBEE2D84"
		  "D76AF0E721739AF02DF710D020CB80E895BF20685D3AE61DECA"
		  "82D34206351870AABBCE23CDA3567DC1AFFEF9319BB1A212D52"
		  "BAAD92D069A332BD020420FD34E6FB6F97102C9AF3C877ABD9A"
		  "790BDAB7E7940B1CC01702D2E4CEEA1A3A2F20E3B4027BDDC6E"
		  "2FE0"));
  test_hash (&nettle_shake256, /* 81 octets */
	     SHEX("E90B4FFEF4D457BC7711FF4AA72231CA25AF6B2E206F8BF859D8758B89A7CD36105DB2538D06DA83BAD5F663BA11A5F6F61F236FD5F8D53C5E89F183A3CEC615B50C7C681E773D109FF7491B5CC22296C5"),
	     SHEX("0AF273F2384078BF8BECF3731049A763753EDB89CA1AE2BA03B"
		  "82EF5DDE8C5E23E692982C524035D406DBAD8FB897AF5D7DB96"
		  "1652A0D6F5C18D71EF1F1A541D0A29085AAA1D26D2C4366DA7E"
		  "41781C34FA40A3FC7E9132C3F0CFFFF267E2BC771DD29266E2C"
		  "649A94F3F15FBC71A561968E0A755D4D0AC7540B9E674539AA4"
		  "D15D2FE81AE969EE492CE89104F994C6201EAB2F6A726D9AB88"
		  "479E324789BFD003EABC2944C43BC061C1E24B7DDFE1C980F65"
		  "DD598C33942482475F4D717DEAE78BA3D8A7A887120D9A814D5"
		  "81BE9D593422AB033A1776C3C7A6DA30D4D83E8E120AC733303"
		  "C401B68D1CA76B2A20AF4FF0FE49780E25B43850D495BA91AA7"
		  "27DBE6E28B677C8C7DDABFEBCFAAF67B8ECD8BE1B459BDD68B4"
		  "29E7B25B6B0A02DD2BF0ACEFC5CBCFE5CD07C814E3266027EAE"
		  "612C39077C810E766A872A79D68BC8FE8EDFA04CE2C80325C2F"
		  "EB03B5C838425860B24A6BB2D91BFA5F67102A3F6ACD3DD6C93"
		  "37BDE3E94699F3B3431D8293F793886238228BDDF4B38F229FE"
		  "244ABE471B16A1BCC73BB7BAE6A93788DE0D349CB75729D422A"
		  "FB32CAECCEFFC42B7BD0694B154286F4CDA4517534D0C2FA7B9"
		  "8E72AEF0E2B8D8E0BB6A5FA82EB802C27511256C1352B20D2E5"
		  "9D0D3D63FF5EC33670FA27CA69D2F586826D3A1EA098CD9CCA9"
		  "231E74B91E999EFDA12F63FA8076A44894B40840FDBC4D25DFE"
		  "1CF7"));
  test_hash (&nettle_shake256, /* 82 octets */
	     SHEX("E728DE62D75856500C4C77A428612CD804F30C3F10D36FB219C5CA0AA30726AB190E5F3F279E0733D77E7267C17BE27D21650A9A4D1E32F649627638DBADA9702C7CA303269ED14014B2F3CF8B894EAC8554"),
	     SHEX("7FC1F7FC27223766C8F4310087A3C54A5A8AA061EB20D367194"
		  "243BB62C7D90099684BBF368CB6BB71802C62C9F3258992C0C9"
		  "79557622A5FB15CA0E50A26891E73790EA3DF4859829B4F683C"
		  "7F5C2DB0FE08ECF76B9D4E897AB80530D9F1D1242B42170C198"
		  "C7852566DFADAC5740D61A52C4882F46E29AFE2C7335C5A1157"
		  "E5B93610D0A8E4529D8473330547EFE22DD9757503C7A67ADA4"
		  "E9B2CEAB8AC5B383F2D6170D6D6A17C03EA9F9B9399FD6C1CBA"
		  "624E7911DEE9892B31D463B3A44946FBF246D773629EA484CD2"
		  "0B03D8AC428BC8AE87CC8182D96F8D220A7312DDBF191F45849"
		  "3769691E7585CCD740C30FE7D63101023867C9A3F02D999BC0A"
		  "D653B71DC9A611BE56A44D4DF2DBDE402A0B7B6D7644B5754D5"
		  "0DC81F59FB17F7C38E62427A844DB406846BC74DAE737E24FF8"
		  "06505C5C4351A54AA920CAD01DDD8A5960AE143F476608A6DB6"
		  "168FD5D72453EA010B1139417277E6C5116D1D32008A40CB300"
		  "6FB65EE9D4F5F9AAB1648B3990F5B9DD338836460075B8BC504"
		  "B4DD2FC04EC57DBC63D32CA39C07EC7C2348A459FDE1533B30F"
		  "496F84502367BE809D0AB0C6E86474EB2D7A3A4CE29CC8D0ECF"
		  "B87B4662FDD789A99034A64C5B40F763D098888FB32346EB644"
		  "C28FD6BD6BB62E0D5AADAFFEF0BF0BDB801A809F76D79B84606"
		  "183067B5056528CFFF3ABA4F0EC0DC926FFCFC2377035C8B5F1"
		  "0D57"));
  test_hash (&nettle_shake256, /* 83 octets */
	     SHEX("6348F229E7B1DF3B770C77544E5166E081850FA1C6C88169DB74C76E42EB983FACB276AD6A0D1FA7B50D3E3B6FCD799EC97470920A7ABED47D288FF883E24CA21C7F8016B93BB9B9E078BDB9703D2B781B616E"),
	     SHEX("1BDC443B8F1E6A47C575A1158C00EC9E69048BE22AC067CDC12"
		  "33171A69E773B6319B0BFE0281F0CABB4DA1F75C583320A96A6"
		  "2FE8DDF10D0C3F7871023562177E827FE0B58A02D38C5A4903B"
		  "7CD8B0CA80012238372DC5DDA701A4287D071FA2B29771F7054"
		  "8CE43ACB0F2E5A49313FC76EBAFE8B75E121F0294E417323EC9"
		  "BCA9FAFFA81FED2AA775AA2D2C641D7BE624EA861BD5C24CCFD"
		  "AC1FFDFAC4627C3985A41EE5EB57E229E1A1B9312C6FDA42916"
		  "05D26BA4401EF106DFB5BEFA7DEB27500F11A09617E8CFFBD39"
		  "AE95D9EEE736EBA41AE469988947A863FF6BFEEA212EEA29297"
		  "025F96453DC3AD935E75F12C5A7F0E6C12213BD7BE910D70EB9"
		  "78624843CA772959B5918E29AA377A7B4413946A97E546A9A05"
		  "BF107C881C9499B9E907A667CCD1B3C64D674E3A5796EC33F6D"
		  "6E4A2AE7D114D744BEE7729773A627C063133AEEEC4757F999E"
		  "A7F01298A783C32934D29FEE45E2048322AA3FBAF20F9D70C1D"
		  "8A0183A630E3E73CB2EB1C934DB4DC9101C949C46C1954D95A1"
		  "7B7D362B87FC51BE9FE98FB76B19B7FE9D1D76104D3D49FA7F1"
		  "CAFC252F2E4FF32C3CA43A23947EEDD4B88EA640E58DE625CB0"
		  "A9E11212EC1FBB24BCCB39B06AC90973BBDD279578666E4D169"
		  "290B0321C5B2197B6A5759FCCFD8CFA820ABA055F81F1030840"
		  "F3BEF889E8ECFF87D0EE1C53DB99F67827710DD0CF59346F2CD"
		  "53DB"));
  test_hash (&nettle_shake256, /* 84 octets */
	     SHEX("4B127FDE5DE733A1680C2790363627E63AC8A3F1B4707D982CAEA258655D9BF18F89AFE54127482BA01E08845594B671306A025C9A5C5B6F93B0A39522DC877437BE5C2436CBF300CE7AB6747934FCFC30AEAAF6"),
	     SHEX("78FA82E4172CD8CF41CAD3D431E39EA0C6A88544402B5CAD9FF"
		  "3D8CD6DC8890260D989AEE087DAB49C092A94975EE3D5B8706D"
		  "0F1F522E5AC350306B01B4B19A81377667B20C201241315C3C0"
		  "5A9F7484EBB70E79588EEAD5D9BEBE5ACC3E226B807B46192D6"
		  "4A2CB886E4B811817CF7F219934C57D5DC2D00E3EE234B579B6"
		  "B6E2D5346D3876CDB3752624E65434E88D55E1284297CAE6249"
		  "95B1B37671B89C57E876FB361ED8AC6345693D82BDEBE00AC37"
		  "DE6617239205AEF566C1619F406F4CB1C9777AF2C07F693E35E"
		  "4289ACBD91C59F067C502446F21CA1602D10CB1D78D18DCB043"
		  "C4B4E06972178BAB4C90062342FF3646CEC8120A5CD14E19715"
		  "F66EC6F619DA0EDBF1C9D9BADA80F0BB210F09476085CDF3832"
		  "06FCDEB987623CD69D591F301D8DB94018EA3F90B8F067C3E14"
		  "7771F148A60B440CA6BE7A1943F544375D50D45DB2BAF339441"
		  "90F19446F7A1F04F7E45D59B9548E01EAF6E4D6D525B37A6576"
		  "9D280DB6DB391F27A9D84E4D97C7CE0AFCA3BCCA7A97636FF37"
		  "56C6CBA855DD5C625574277ECA6A2195027705827C0E4FF945A"
		  "A57E25EFBB65F5128D312145C5A8DA0A4D46D805C3F6ACFFB15"
		  "1B0484EF811770BDD87935CDDE0444276CC8B64E84B877A31D8"
		  "17E95963F3621AFE6FCE57D94D50771F91CD0AB1BC61DD97BCF"
		  "03EBCBC2C2EC687B705A2BCC5D0DEB20CBE6459644D14633C65"
		  "B256"));
  test_hash (&nettle_shake256, /* 85 octets */
	     SHEX("08461F006CFF4CC64B752C957287E5A0FAABC05C9BFF89D23FD902D324C79903B48FCB8F8F4B01F3E4DDB483593D25F000386698F5ADE7FAADE9615FDC50D32785EA51D49894E45BAA3DC707E224688C6408B68B11"),
	     SHEX("414EADC6832F91A04B7700E668AA487E63DB05151E29EF1718C"
		  "7D9F4C6EDD1C88B05910F23D8A854B94C7156C421126272EDB7"
		  "3A6E158350EC7A39980DA19316346D1F330C3FB987D7B4C46B6"
		  "469B7A59EB6B1649BBB7BC0E074C00A152517624F22A4A58800"
		  "447E7505D55F687F08FB939D1AA6DF8E73BAC6CE88467B30976"
		  "150636146526031BAC9F6667AA71516F9C5F70E502D7D18868F"
		  "6DD359D8D026A040AF3CAB533EA5A95DD7CB2006552796C5E1E"
		  "B2C3B46DBD9F2481F1B428CFDD1287299E7C7129B377E3954AE"
		  "0EE96F955B7F0354610B84652705A0594C045C3CDBD1B564BEC"
		  "43FDB5928B905023B1BA41B448E39E649C0B65CABE745C1DABE"
		  "9352D05F165BBED160DC2C52794C5D0F07D1E18F47CB94F2F97"
		  "190944BE08C4BAE53755B30D062B3D697575AD56FD4C75ED40C"
		  "F7C239941F7B052500EE5CA7CEDAA9190988F47A21216E907A6"
		  "3C795D2292CE926D541F331CBFFA9F16516D54999DFC9911310"
		  "CF564D8F1F00B92BD9FAC0AAC99A951EAE3175AD20558E8B934"
		  "E05FF58351056466A61EC5CF4F11A01E3FE0CD6EA280E278388"
		  "99BCAF7FBD8CD4308098E2EE71A4E9EC258D03118F0BC4619C6"
		  "6E52F35F1AAD2378D82BDA8ADB750313705D41263CF2FDEBA1A"
		  "D5E025FC76DC620CDC3B684ECDE0283F9A7D69A8463B58F4AC7"
		  "F5B1E2F3F580CA7F4188F32706D722BE4543E8E0942B3169603"
		  "49B3"));
  test_hash (&nettle_shake256, /* 86 octets */
	     SHEX("68C8F8849B120E6E0C9969A5866AF591A829B92F33CD9A4A3196957A148C49138E1E2F5C7619A6D5EDEBE995ACD81EC8BB9C7B9CFCA678D081EA9E25A75D39DB04E18D475920CE828B94E72241F24DB72546B352A0E4"),
	     SHEX("0CF9AA0F0478443DF90BEDFC2457009F3E58B06EFB22D1F282D"
		  "A4A2320623A250D584F27E2D45F708DA667A9BCC26E35C19C55"
		  "B16A993AE9D98948FF2366D8A0AA06915EE5712D137D46D48C5"
		  "36DB9A25BDF88F99A941683E342B5541AE481B1BA6A889B7ACB"
		  "E9A6593C53349C1B7D6FC8FC66627A2FAF7811C0F0A49D90424"
		  "6B95E9B668E8BB25D521F0905841B7B2AC64B2E359FB48FF29D"
		  "7A314B1D2E7E01B0D210986664C249DC711346449FC77BAEE4D"
		  "F54344CC18A816AD9C980BD0D9D01C4AD4EF0C702CFD8780510"
		  "3910E81CD3F6CF4D13D1398E755E5470E23ACCA6B510CA59FFA"
		  "0B523A9D7FF7A5D85FBAEB3E5A9C11D947AC5FDAC04B0B77E4A"
		  "E7943F696849DF0FF931E2300CB099F0DEF86D835A8AF4B53FC"
		  "6C3D38BA33158A1F95690E8C5560C060B0E48783AF1E2001BB0"
		  "4CB4399CD27280715AA0EDA7AE754B8A13F84916B003F87DCEE"
		  "BAB5938FC4342167EFAA2A889C48CF92F6892BD9BA41B61CD1B"
		  "3BF94D612C2ED603687DE8644613605FE9F066E4FCD8FF75DA8"
		  "FBC9C63C37D1828C173BC4AC668AACD20AB1E3F449A7BC4F4EF"
		  "0A6AB8D5B0A70CC2D706B6C6821771DEAD7CB29F2332F4292DF"
		  "7E397BDEE393D57B06A1AAD3DA51CB8F1D11E43D2FDBE72F69F"
		  "951D0FD9217F4462CBE5C701FFF1025515829CEBBAED488FC26"
		  "798BEE9AD6CE88F33029061A588E177BC1CBB24EB393481F3B6"
		  "1616"));
  test_hash (&nettle_shake256, /* 87 octets */
	     SHEX("B8D56472954E31FB54E28FCA743F84D8DC34891CB564C64B08F7B71636DEBD64CA1EDBDBA7FC5C3E40049CE982BBA8C7E0703034E331384695E9DE76B5104F2FBC4535ECBEEBC33BC27F29F18F6F27E8023B0FBB6F563C"),
	     SHEX("2B8F5423A417D9B827D98AC1D21EB7C1CDF4348F28FF08B6F06"
		  "389E4CC2311C5721509A5888DF03F7F4B94D42CB66F3B885EA8"
		  "B130934A1025220769D1C352F72D378F3A63CEA1E762ACEA574"
		  "66C3AF188772872C16D889F06BF0AA321170FC1AA842B413BFF"
		  "9FA32FC1E060F868A370434C99FE586E8DF4C6DF1D167912F35"
		  "E7BB983D668225DFE5A00241A1050702A5CEF8CB5EC957B779D"
		  "6CA1C9F6858ACEB4E927E104A1DC59C5E90B31F4A6E27623B46"
		  "D40D472D6D4B6EA6CE8BA4721A5A765DE104C4ED808B4EBD73D"
		  "60A56363727CE0273710E63FB8A2D565C49823823F7B001E44C"
		  "8571B885AAC4E14F8651CC95C11EDB3599BD85989C931F866B7"
		  "C63F80EFF135DB2FE1B17D5DFF147959606FACA0FE22E5BFC88"
		  "D43F6057DB91EB0AD0C5491B66B2EC9B20EA01E215F40922F41"
		  "ED4ED64616960272617F32D62F28897BA03C5A8657B1DD4ADBF"
		  "0F8E53492998922A47BB2D7897502F8687C465C8A7615089062"
		  "108D70164677E11E1F63917A4F230E8C577346F880B7B8ECD06"
		  "40D10622131731CF447662AFEBBB230C902B28D8445C5F3EA03"
		  "1AE7AF04F5B22D3AE7CE58C6CB4EC4158DCA5266AF0C50F609B"
		  "12C5F4527F836F8804777BD6C80F5D7D9B0F317CB5A663D3B7F"
		  "3225361EA81415CD5AF56BF24629D511A80ECD169AAF9558AF4"
		  "87D46A569495A70C76CFE83462733BF58B2A7C78A7EA1FD0561"
		  "1DDF"));
  test_hash (&nettle_shake256, /* 88 octets */
	     SHEX("0D58AC665FA84342E60CEFEE31B1A4EACDB092F122DFC68309077AED1F3E528F578859EE9E4CEFB4A728E946324927B675CD4F4AC84F64DB3DACFE850C1DD18744C74CECCD9FE4DC214085108F404EAB6D8F452B5442A47D"),
	     SHEX("C52001EE31F26ADB120EAD7699EE35AC178D71FB5B073129955"
		  "2B57DF2A4F521A021CC56D54F2B43D18D7726FA977FEC221F68"
		  "19B5B7F933BEE2EC643B67E16BCC26A7DC1BB1DA3B40F2FE026"
		  "45CF5BDF3140E89ABACE926E710ABF0F07205291A336187AAEC"
		  "AD9371DA35867EE8D8845E09FB47269ED94D04B47A3B6C460BF"
		  "419F13AD2F65D63824805ED264AFFCA9E7EC9774C4E36695806"
		  "86C0A02ECCD98277365940F4CBEA5DD5B0BC84F981C16FA6CF2"
		  "D6F1A292EC956C7D0714C6895E96A7C884173E662CE5DB1018F"
		  "B512CE773ED4752CFABF9045269922D11718C572A94ACCA1EDC"
		  "04CE16D533B064B416684D921D46A8F2859E7D09660F620DAFC"
		  "C232029A20886A552D29CCC30509A3941847244609911CA6C59"
		  "6E2FD915FA8BE37D7EB41D403043683AE9C74DC7D6F4770086C"
		  "F8C6E101E989FD6A5C60E34183C4EF08EA869FE2AF0FE5EF701"
		  "A6A64EE0D0F813A14E574D504E219292C103469F9A11DE75083"
		  "2C2BDE0261475856D5C9FB4727CA49EA81908A1F7FAA20248C7"
		  "39179CC09D93E4901ED63F14A57B20EF9190176A27BC07F912C"
		  "C62DD1328B032B893642118EF3426B1A8B95A0E36FCB8A77570"
		  "25606E9E485B56E84D0E691234E072AD64FB8E836118634074F"
		  "D8362405DBEC4D8AA12E9E84068EE3B29B7F573CE1B52649CA0"
		  "DC1F758C1D4F5E2B72A861D6326F01084B6B49E478FB4CD8EE9"
		  "2750"));
  test_hash (&nettle_shake256, /* 89 octets */
	     SHEX("1755E2D2E5D1C1B0156456B539753FF416651D44698E87002DCF61DCFA2B4E72F264D9AD591DF1FDEE7B41B2EB00283C5AEBB3411323B672EAA145C5125185104F20F335804B02325B6DEA65603F349F4D5D8B782DD3469CCD"),
	     SHEX("386C3B1A524F2EC39C33FB31CDD3981FDD57F1E33D04FAC5482"
		  "8C54F42EF763856B72202799D4CDF4AECF073B8B9EAC39F0F69"
		  "54545BF60DD55CE2FAB4558DECD7E1A5A40B87A293D35394DA6"
		  "4A4F606956213326FF4D8849A3F19781AFA9DBC0AD0E0BED551"
		  "164275D4FC11638AF3ACF95C86994C27AB72F0E5EE366CA8EF3"
		  "9C000661BDDF23551AA368F347B4526474496CE14C30503C251"
		  "3501404987446A193DAE20C4B384D0D9926814BAAA863320946"
		  "B9759C7BF1BC890F88826DA59E028F641CFDF7223DB20B34B63"
		  "89B048CBB85C4A0E4E84DC5C5241D2DAEAA82E9092ED266971F"
		  "D185D3B661971160C05C4B42BA8E1E8E31FE588C0BAA81A2BF2"
		  "017B1E0B99BC360FAF13315CE88CE653C20BBCFAAA8ED013EB8"
		  "9FA5DA39FC49D99E3C050E622122B7E393BD933B72C5392B165"
		  "E60538BB6E29DE30A5947053D4A23038AAEFB13DB8BA4FB3B1A"
		  "65B474D94E4CF985B2402BF22CA7FB723DCDD83BA875DF3C603"
		  "50886CAB6A1BD02FAB32F1DCA0CCD879B351196E1F8BCC3531A"
		  "C653604932B69E7365B8B61E557585B7A836379D0229844F6B5"
		  "D2C4F31A6BA2CF4A2585B7153C6005D10526DBC144189B0747C"
		  "BE58EDF2F4FFCA80FF99A4BD7C8CCE3BCAC36AA5D0D7DC2B231"
		  "A5B888198AD71042CDA93C5AEF246CF3FBE7A663FE60BC04CEC"
		  "70480CC2D83C847371FF5347A93D54059926B6B8F52CE6AFD2E"
		  "5630"));
  test_hash (&nettle_shake256, /* 90 octets */
	     SHEX("B180DE1A611111EE7584BA2C4B020598CD574AC77E404E853D15A101C6F5A2E5C801D7D85DC95286A1804C870BB9F00FD4DCB03AA8328275158819DCAD7253F3E3D237AEAA7979268A5DB1C6CE08A9EC7C2579783C8AFC1F91A7"),
	     SHEX("6BB18C45BE71508441EE815061ED782F4BB5ACF5393BC9BB825"
		  "FC79CAD025C6D970D615F4285185696CE226969D5E06E1643F8"
		  "BA93F5617AFB096ABF46DE2EE7D32BBAA335628C41F1FF30841"
		  "8556C51E6DA30639BA5DA36A14976DFDC013551F632133723AA"
		  "9555B8B11BF0594457A6B288B01185927A48412019E58621806"
		  "FAA968CA26366B884C428607D2D6FE9E8924E6D8D769A7D2DE9"
		  "0195AB6DEE4896ABAC3119FD15EE61251FCAB1D3858527E2E6E"
		  "1B8EC05BF8659EA345E519CADEDD779E40D2508BC6EAC6FB531"
		  "DE15CF7F3478A4288DF556D4B6C9DDC5F1B6071FEDF3C4F9A96"
		  "0B18F8301064B4F83620F2402614C600EA80FB2B24BB9B8B777"
		  "BBA3010AC1FBE61AE18BFFF9502F66111B8CA1E0DF671C311A6"
		  "F11697DD8C245C4B946B8DB03D6C07EF14C45CFFC328E6C1294"
		  "8D4137D6803469E996C1D84C3C1DBB8CFCE592217155DC8F9E1"
		  "E558CB58693620F57F5F81578F060B23A1A75ABE8010FC53852"
		  "4DC1B5543D84F5F28CDE4AA509B8A034522F2F61F8E3A683EAD"
		  "34ECE04EE84663475D65D6DB75CD6FBE011D6F1994D76BE3566"
		  "04D93311CD0410D3B022FC328CC850D7BCC1BB96AE45E8B4C9E"
		  "53904FD8BB8B0FA1D8AAB9BBEBBA0724DBA6A2C71F28AD705FB"
		  "E6BFD958FE493AC47D70A1C4B3BD738C519558DF9F66130C688"
		  "31A7D748B7B382023810865CD9989735E25186690FA187E8E24"
		  "B77B"));
  test_hash (&nettle_shake256, /* 91 octets */
	     SHEX("CF3583CBDFD4CBC17063B1E7D90B02F0E6E2EE05F99D77E24E560392535E47E05077157F96813544A17046914F9EFB64762A23CF7A49FE52A0A4C01C630CFE8727B81FB99A89FF7CC11DCA5173057E0417B8FE7A9EFBA6D95C555F"),
	     SHEX("51F3DE30531335B02C0A8CDA5285FBEF9AF618E433142E9E971"
		  "3388602B0486A99707BE420AEAB53DCFACA68CC6CC6A3133075"
		  "A518D2EB1C500F0AC2E0B8A91D3303594BB38FFC45B80AFB8E2"
		  "5C4304A85280E6124891E739FC9DF8E34ABD9262DCB4330C2C0"
		  "F22E78447B3FA67677ACC1F3D8E836EEA94AEA4D616132650C9"
		  "288804C3CCA5F1C0139624E198643D52C42960052FBBCF89AF0"
		  "0D58600A9597BD12A39A33AECEEB161EE527398BD915ABF6427"
		  "46E4C965A77E916E700CA905CB70222A3A8B8C5DE3ACF75A9A0"
		  "065681497F27AEE6B1F4402EA711CE6B24592BC637F83BAF912"
		  "D3E10E0A7E03CE9314FB264E5FFDC3564307362837C28AFBE2C"
		  "94BD89AD3CD4A48C1A2EC38090C38E6CB4174149CF0D5841049"
		  "C1D7DDA1E51DF8F3216CAA119A9E3229CAA062039EB165F4D5D"
		  "E68FCB76ED32F1E191166BE83CB6384FEC4528588D8A3188A9D"
		  "A7BC1F5DE68C42D9E2934CC1010DC535EC87804810C009982BE"
		  "6646B3C50DCAB7EAF4DC7786503F29EB3F1558A7341E49348F7"
		  "3479A2D3EE6B8AC076BF5AD17375D81F9E4DBB365B2489F17C4"
		  "0B9A4CC10F6DE034B4B81A3D4F81475BABB9581C4BFEE62C562"
		  "6FC436983570D5C6144FDD5A55E1898919A8BE4CE3780FB2548"
		  "32CCA90F988004D12039A947502216651775160D987DF82F432"
		  "75FC7B8A308C63DB7C28959AD9D8E6D27A5A551E685B6639744"
		  "65D9"));
  test_hash (&nettle_shake256, /* 92 octets */
	     SHEX("072FC02340EF99115BAD72F92C01E4C093B9599F6CFC45CB380EE686CB5EB019E806AB9BD55E634AB10AA62A9510CC0672CD3EDDB589C7DF2B67FCD3329F61B1A4441ECA87A33C8F55DA4FBBAD5CF2B2527B8E983BB31A2FADEC7523"),
	     SHEX("41143CE195B170FC23ED53D7AE8CA61F397CDBB72F261750A7B"
		  "D26CFF56212AC7BB1B18D002493F418185ACC703DF4417F44B9"
		  "3F4A7684D3D68F8F042A73C3841B6B5FA0079163D4881C39BCE"
		  "CAA548A50237E3F8DF20F986552978ADA9C4EB2E57062DB8D91"
		  "564BA4EB96503B932B5FBE3D88EFABF452A03EECE5B3E78492D"
		  "C555ECC95714A1C5C1B201CB42E10852C9A5FE101ECD53FC6AC"
		  "D7E1D0D6EFECF81BA76582E6E9CF80E70E52C0384EE05978A31"
		  "234DDD1C2D501DEC0BF1F26706F1C6AF4A6DCFE1E7245D7BFBB"
		  "96DA619EF27B4CD67527BD6B5117E4E55297B9474A5BCD251F0"
		  "F17B10F196CFA95ED3CE61458ACC55CC2BA83904796AADD71E9"
		  "78D4615E6588954F7D48B1BC705B497E073DFE2ABD5224792EC"
		  "A6C09D611C311C544BB332B2C9D144D24950F3A27857B9F9AD1"
		  "DB4DFB9AEF65BCE2C224EE9FE9785593CC9CC74D076BB5DC575"
		  "B4EA65FA59A6DFCE634EF83072CBBABC5AAA53AEF5CB4D2C19A"
		  "613D0054EDA3954BE7E649C2C38CFA9F7746941E2B77DEB63E6"
		  "2A9FFEC59D15329CD7D02FCE7006406F3119CF579E1F6F0A1F4"
		  "FC74C7031894D240B5BCC9A7306754AF9B43DF80005C7B62F88"
		  "5574AE9C44B90D5CFB93990CAB41FC3E9962CD0F8047CA1AEB1"
		  "399FAAFC6D6FCCF66B2F02CFB9532899CCE1AA8E822EE9498A6"
		  "7C3972407DFE98969F9DF9CE328A4B9A1DAE1651BAFB32AE29B"
		  "F666"));
  test_hash (&nettle_shake256, /* 93 octets */
	     SHEX("76EECF956A52649F877528146DE33DF249CD800E21830F65E90F0F25CA9D6540FDE40603230ECA6760F1139C7F268DEBA2060631EEA92B1FFF05F93FD5572FBE29579ECD48BC3A8D6C2EB4A6B26E38D6C5FBF2C08044AEEA470A8F2F26"),
	     SHEX("1B0D344DE5A414960E67D684D301901D8E89C25EBDF7494DB9A"
		  "1FFED44E5D688207F4533CADD09FC8BD428FDC3592959E90956"
		  "13BD808797BFEF5130FBFCC1FC72B58D93BC4A291284115D5EB"
		  "47911FBC0A4AA5875AA2B6EE1E1BCB296507183AA5399AE8192"
		  "EC6AE579D9AD172F72F5A1957BA10B0FA4E9666FEE9611B0487"
		  "30275A0F4D6D48F98D860939936C9B41D1AF5D729B11C930853"
		  "46D8769C141B81FED9F17FD85C0197380482483C5F45B3D8BEA"
		  "2C2E90EEF2EB489B6986A891BC0F29EE7E2943355E4223C2414"
		  "27C49D7A897C956323ED10B074132449FB6371A9BFFDAB8D113"
		  "282016AF1F7E8E26807A0B0B4809BC035BAE86F476F7FFE002B"
		  "BE7E30B4C06B00E712F1D54E954D59AF083DA123034E73B9854"
		  "B45E9BC2EFBBB7282743DC5942BF372D94432379EBEA4A65699"
		  "7088004A5C2AEFD6A4CE6FE94C2A06C8A0446495DF224269E31"
		  "0F1DC184DD37ECE4EE46038369C031FF90AD3787331AB6BB1CB"
		  "AAB7A0194FBA947FC6485B371A684E7E1CCF4E9F4D272158A36"
		  "B559451E48095B3C09328DBB52C7659C7E163504587962D87A5"
		  "E60BB3C4868CEBC204238FA08B97AF71DE9ABE7F409AD0D2945"
		  "5E1CE59433685EFEAAECCBE01462B1760FE25BACE44CFA6E1B5"
		  "C28DCA00790D96D2B6FAE377CE4BAC7CE64A97AF68EE913C331"
		  "07E4A62EFD4CC752DADB23877B54795A43A7AF8593E085480F9"
		  "B43C"));
  test_hash (&nettle_shake256, /* 94 octets */
	     SHEX("7ADC0B6693E61C269F278E6944A5A2D8300981E40022F839AC644387BFAC9086650085C2CDC585FEA47B9D2E52D65A2B29A7DC370401EF5D60DD0D21F9E2B90FAE919319B14B8C5565B0423CEFB827D5F1203302A9D01523498A4DB10374"),
	     SHEX("69271E30637CE53639A9A80582F8E9D91E0A19AF1C25A4E316A"
		  "4ACBFC4534D7B6DB55BBDEA37F2526E5CA0407D0A39D06E2AFB"
		  "6538C125B3390A860E033378440419FF91634BCA42EEC9AD80E"
		  "55E65D43147358146038C9BAEBAB808C1A9B2D796BF22BAAA54"
		  "D92298212E7101E56D1A0A229CBCF9E08791C6BD464EB997164"
		  "502445C2993B1D2A6AC4242D1F4108C2AE4DBF48FC403FB8F35"
		  "3C5CA3EDC09C5D4993048962D1DDB474928BFEE27DF6AF8BEAE"
		  "B264EDD09DB4950AC5E12561751BD3ABC41C9A81F6C5C339AA9"
		  "F7266CB4A28EE26F29571A7F4C9B350974BEB0AAAD642D9934E"
		  "9EFF77ED136BF825A845041B9C01F0559BB1987FD6FC951322E"
		  "4A4B4EE50C7A7CC43324391603D689BD19ADA54982E03A2BD12"
		  "610A09BD076C0C3F6D6E0CA348110D8DC5DAA5C842198AC5EC9"
		  "2A79099ADEA1FDA0DF0B35E8871373B312D61554391DBC9E1C5"
		  "F9007C9D3799C24DC4A0894E175706DF7D04EB2D69D90BAB571"
		  "17E04B681FB24945F3BA3DD7F1912171556464500D4FBB90840"
		  "08A79EAC50D8BF59163AFBA8E0F1E5F82E15255398F62890C95"
		  "21A0571813210FCCB582818A6D1747A31822DB55C4B6BF15A19"
		  "076B7D6349873C88E86BD87D08E0058A212780C86915CA559E5"
		  "81EC3DC7D5BFAB9C5403661F09484F908DFB2D98B65F8543FB3"
		  "F102C2C5D0B89B15E565E9B04E07F7141E35102BDB056801100"
		  "B160"));
  test_hash (&nettle_shake256, /* 95 octets */
	     SHEX("E1FFFA9826CCE8B86BCCEFB8794E48C46CDF372013F782ECED1E378269B7BE2B7BF51374092261AE120E822BE685F2E7A83664BCFBE38FE8633F24E633FFE1988E1BC5ACF59A587079A57A910BDA60060E85B5F5B6F776F0529639D9CCE4BD"),
	     SHEX("2730675F5F4DB76D342EAE88CF856B370982277E24A1F95E071"
		  "EB0C28D6655D9517CBA67DDE32BA6A4A322D37AD4A4EEF90A60"
		  "E2FE3A568417BE75432F57964BB1DD7A5A165C24F38F4085CC4"
		  "551C35BD0E2663198DF04EE86803D75EE5ECBB14D7BA6166934"
		  "32B0FFC83F27E82016D7E3436C2384D1A7AB87E4CE758A5A83A"
		  "4FD63831D6F88E4E80D1CD257ED4B418FE1BB498004D1CE8CDA"
		  "CEDE429A53F06EB77D0A6AB47BEEAEE12FEBEB07D434E26A719"
		  "57E55F4F3284BA0F8157DE3F1C8E0E4874DB4E65E5243982BFD"
		  "57A9E70D18BE390834A9C3791E6AB6341739A963A946E1DA81A"
		  "4D9B3722BAC231191EAA013E424C2B965967AE986D50426D983"
		  "0E92499E0808FD6EA92A8A1054F93A0F84C11BF417DE01A7B0B"
		  "A5890172BCAABB3D2CC23853822960E666558B2B4695F38A22B"
		  "576757C0A1A84437E7B627989A4053B14954ECA09DF2221D6C6"
		  "E88654EEA2A741DF3BBD6EF2A8C463D797222E72DE8E44C8C6F"
		  "2FEB44E3D2310ECBB139AAF3FE2C4B2BC9D7142C0291BF9FC97"
		  "71178A484417202DC721876800E010E53F1972E3630B5DCE021"
		  "6351E687552AF2FABA699A4E3E0C5385D80DC367FD0D07258DD"
		  "59A6BA5D2E0FFAE60A5E0C48243794913CACDFC14D20001CD9A"
		  "097A9CC57D313006E0B8945ADE9791812B63E98A291CB009E89"
		  "9DFCC7BDF9F58F3866F6C33AE77718081A517720C34DAEEFA52"
		  "7641"));
  test_hash (&nettle_shake256, /* 96 octets */
	     SHEX("69F9ABBA65592EE01DB4DCE52DBAB90B08FC04193602792EE4DAA263033D59081587B09BBE49D0B49C9825D22840B2FF5D9C5155F975F8F2C2E7A90C75D2E4A8040FE39F63BBAFB403D9E28CC3B86E04E394A9C9E8065BD3C85FA9F0C7891600"),
	     SHEX("DF6FD8F4E768031B0CE25C199F02EC29053EA8200B7EB306E80"
		  "2C8DF23CB213CFA06C37630A3B14570968948C5269A67706225"
		  "27C87D432098F0CBCCB733A5D28B035CADA562BD0FCC032D2F4"
		  "5DB8D2C948FB341AC2B0C5C699C62BAB551553099E8A21518AF"
		  "F800C8ED42B44DE7C30F29E691C1435CE72CB67060307D1B1C3"
		  "8192FE198AD3D20E9E12A8C9529F8A134A0CCAC8DE552AF3FC0"
		  "5D48FE2E3ED1DE5ADFA0B7C324320E262232A14D3A7163A0498"
		  "0CFBF087BEC24E356DFD8AE4DE9B0620BFFD1FF904E58B39085"
		  "22AC13D1D541F662D95008B1F8C52DA67829B7CD208BC9B8A3B"
		  "6927E575BFBF7E29F1CAD832D9D01ED41E5512ACE419B92F71F"
		  "A0CF5F79E33016D5E19EEFF561C94827584623118EA367D768B"
		  "C24927C92FC19999FEFF28E3A09D2F266B28433CDCD51589571"
		  "1FC081F8B1D143323B680E8B7469EB0FDD467143C292C8D822C"
		  "C16111E9AB2A0C9EAD40B2867303686FF4B9BB4DEC4C8B552E3"
		  "796CD91443AF8C4542897769EADEAEA39A366E91D65E92F06CD"
		  "6554A1F6CA905996F5F62CFD559D1B300D912E6FF9166853488"
		  "0E6A8C414FAD3C107A180BE3E0BBF4F33CDA343032D01C98C63"
		  "20F6DE582DAB7B15FDD6E75AC3AC7FD079B191233D872AB351A"
		  "E1A56F0A4CCA8D0215CA1FD5F9C45EE171F4BC72D18E78ED6D9"
		  "D651E40AA77522F6E3451995FBC3207FC0F30D1B39EE836F937"
		  "1A36"));
  test_hash (&nettle_shake256, /* 97 octets */
	     SHEX("38A10A352CA5AEDFA8E19C64787D8E9C3A75DBF3B8674BFAB29B5DBFC15A63D10FAE66CD1A6E6D2452D557967EAAD89A4C98449787B0B3164CA5B717A93F24EB0B506CEB70CBBCB8D72B2A72993F909AAD92F044E0B5A2C9AC9CB16A0CA2F81F49"),
	     SHEX("3C891240ED9F62F45658B5C1D3E4D77975E45CFB10C45513927"
		  "EA9D911B3E41414DC0EEA38C8A6868A9E0AF20F96BAC833C1DA"
		  "F71D0BFEABF41B8C26C011495F8DEC94D72ACB5C5C9ABB1C372"
		  "F8B779BE741C860E722A0F85CAAC3D21C6C9EBE61C6489FF558"
		  "1ADF5650B6416E0E88A889CA60CC641052D601D491057BEF36B"
		  "4DC3B5B76BAF0DE4E7248A40D9BE6D55A7E8C26663828D08495"
		  "DB33F94ACC7E7E49A7A61F352816D8C4C0E23F36540418AE6DC"
		  "009FAB33C945C6E48ED60BC466478788002220DA9A5560425A6"
		  "162AA03AFD5700CEBFAE439D20A85F2CB826CC8814940F2CF25"
		  "E34754D8DF03DE448FD1D1AB38BBC2B1577BB1BC39DA7B91B7E"
		  "63F78C5F43A875C427BB110E0D5FF07E004AC9D810D333300A2"
		  "778A06369B7CD684AC86AEBC97D7B9C09442C35E66925BB0E4C"
		  "04C5B3D11267F0F3812A5BEE8E18653D98B6B586D5735D4D3C9"
		  "2E3B76DB5BE509AB32060E954D97FC8B6A428509EA98E4F8210"
		  "F42DB3229E07E1EEDA684B47911556A8C34FB0DCC0998A781FB"
		  "CA574FA891C24B35251C9D0D8429763468CBF0214B2D1EC94AB"
		  "3FAE82E57C748122531C615BDBD4CDA56ABF319D6EAFF738BDA"
		  "7683BA4CDD78737941DCAC4FC45E379BD6512EAB6D14C4F3D57"
		  "48A3CF45713792D314F81F1D1F5ACED67A0A9AA026433F32053"
		  "0632E5F72122B63ACF01F2BCCDA9BD4FC579E6DDC1371DFCBF7"
		  "347B"));
  test_hash (&nettle_shake256, /* 98 octets */
	     SHEX("6D8C6E449BC13634F115749C248C17CD148B72157A2C37BF8969EA83B4D6BA8C0EE2711C28EE11495F43049596520CE436004B026B6C1F7292B9C436B055CBB72D530D860D1276A1502A5140E3C3F54A93663E4D20EDEC32D284E25564F624955B52"),
	     SHEX("201561B1D68F1AFB93098777B7D95E56B790194F45057B3B50F"
		  "08EA89436E54BAC9A5E945BD1CBB51ACFBB0D61E539D1B3E896"
		  "A1BC1426C1FA408B5D16D706457C35E3D67377AB2817127A852"
		  "DA463FE06D2EFDD25016521F19B8D5E54014A54E297014230C8"
		  "1C917A983FE4429B47F6A460FCC4AA5A4E1E322CF74A9DAFD30"
		  "D313D2C0F9EBD1A8468A300F12E6CDC0E971BB7BFF172593C7D"
		  "B286F467899B9A23ED3E222BD6124CB9C4FE2AFAC157CF81FF0"
		  "98F021B95F1069D7609963621C368AFD3EEAF84A17F1469AEED"
		  "DAEC2DB5BFFFBCC39A73AA2A62578775100A92AD74DDCE3A0DC"
		  "B27A740A6E7C0F4C3E36E4AE55B04F04CB7D9527176BCD94DF3"
		  "B884FD4CC2EAD608CAA016EDA2E2232D0AF888A3ADD12CDE0D8"
		  "488650EC6047199842056744B77FA806BBBB96A47ED253192D4"
		  "6B47D828EEDB9DC0CB38D576735E870826F829D76F58CA7F3A8"
		  "0FCAEA1CAC1BC38A6339C307F23DC25E3231094211FC867B0F0"
		  "BD9D1584FBFF5A52EC3B5624DC814208E484B5D069B78A577EC"
		  "CC017FEDBBD139C13B50B395AD16CC551430229B56B4D75A145"
		  "88A765728774CC234D1DC46F0E64E7845CF237330C3D2422CDE"
		  "F5CB1D741741DA30F39D99035D7F7C49D6A3149D02DD061F853"
		  "23D6E0054DB7395AB8FC496418B7770355B854FD36FC1DE72CE"
		  "F330976C2FA6FC73469B71B80F5D9B5CAD008C38746A54578C3"
		  "D195"));
  test_hash (&nettle_shake256, /* 99 octets */
	     SHEX("6EFCBCAF451C129DBE00B9CEF0C3749D3EE9D41C7BD500ADE40CDC65DEDBBBADB885A5B14B32A0C0D087825201E303288A733842FA7E599C0C514E078F05C821C7A4498B01C40032E9F1872A1C925FA17CE253E8935E4C3C71282242CB716B2089CCC1"),
	     SHEX("42A5915CC1DBAC56FF74825DB482AFB8752991EDD5B8F5D040F"
		  "E45B30161C75178D5D56F8A6FE9B11A71E19D0EE241A656171E"
		  "BD9501AA4B5F2CF0344FBD23EA5035D9A2C42FCFACB5F13D921"
		  "2805F353932169A0628D015644ED450E8F8717A1901D5A21FD9"
		  "F8360054946A29CA31DEBE683956FCC9F862BB5AB04F07AB11F"
		  "751FC8D7CDB0FA9C616DF691794C6D26CC1A18A45847FEA7601"
		  "7D57D9BD6BFD1E5CEA4C92604ACC860252DD0F53886C48D0187"
		  "054BDFE1939A2E6D5C570A82C300A6553BEE5E107EE319435F0"
		  "AD18B92552ED9B4FFFD472CC482DF76B6DAAE5432D17C247544"
		  "4BB76CD79313CD14620C1D2B3487D91C25A47ADE446E1DEFA7C"
		  "6D2C2ECA3163106F10EDA5779EA6C21F8D778C29CA3601FE5F4"
		  "56B74DD08C7ECDE8FF599B26540624BAB26314453247A945612"
		  "4F680E68CB91B8247E0E5A06CD366E46055F31712DCBA81B590"
		  "FBA34C8E619C8F3EFD39D2B69CCC6C3D184FD058A9BED65148C"
		  "E65680F31715373526C509EFF8DDF378A0127E1482809CA13A8"
		  "34FB3A1B00231F6B69A8523F72EF580150A4805981F9B1A7488"
		  "AC880BC4018105545707B85F24569E4F864C30E66E750AE2065"
		  "BFCEAA2DB440250568C6909590AFAC5C1DF88D3E6F02D2B4D60"
		  "9460A6BF76FF7994EB5F64B00831C849F7851A3E743DAED3866"
		  "8FD1548B333A01568140B1DCAA00E180429EC6BFEF620B06C98"
		  "DDAD"));
  test_hash (&nettle_shake256, /* 100 octets */
	     SHEX("433C5303131624C0021D868A30825475E8D0BD3052A022180398F4CA4423B98214B6BEAAC21C8807A2C33F8C93BD42B092CC1B06CEDF3224D5ED1EC29784444F22E08A55AA58542B524B02CD3D5D5F6907AFE71C5D7462224A3F9D9E53E7E0846DCBB4CE"),
	     SHEX("BDA93EAFDEDDE22B029F4618B2135E9F7AFA583C8BAFC3B62CC"
		  "667C4704CBCDD4AC6620361A0690AB8E857335C150C2D70FAC2"
		  "C2F2D60D6472AE760C082775D54C8EEC450421D536B1A94A5DA"
		  "5D803D951709A7F5BFFE63454306D7B229CCD3A89DBF9436C9C"
		  "91C01A1D0964D86DE868B4FEAE6E3B88592A85E2BDC82CD5C52"
		  "968C85F429B08A2E61E1FAAC0945606EC462EA7B2AF859609EA"
		  "AE3025E43B4489F7B1274922FA72619B9EADE2F6C04D5863B03"
		  "EF76189A81B9BED9A47BCC53501E96C2367067012036D07B95A"
		  "C0604FB0E6A793881CD81D9577BF495FF769B72DC8B0D554CE7"
		  "0FED81FD3AED5426525241924423A4B11391D3AB16C02577ADE"
		  "31301960A579833C3FF5F9DC0EC068C3C92234B9DE3051B5833"
		  "0AD87D0BB6F704B210E1C5A167684025AF9791757CB0DA0517E"
		  "4A271C89AEEEEBE529609529B69E9A53C1631AB89AFA3D0FDC5"
		  "82E923E5F2293903B75A4C321117BBECF2E114BE505DE8CAC60"
		  "8C1DEAC6FA33C658D72454FC5C3784989803A839D565DA08436"
		  "E700F3BC8DF99BD3465CB4C1E8886FB20354E132A629FC964B1"
		  "0714F006978121E9737D70A77DC73FF77830E2DFB3ADBAABC83"
		  "657D8700206AF318714867A1BA3BDCB8354B91F4F1A8B09BF54"
		  "805BA01BC303E5F9887210584913C5130B643F15AB603ADCA3D"
		  "918C3D5F15E97EC27FDA51E4E6CB73402AFAD08678217E46B0E"
		  "2283"));
  test_hash (&nettle_shake256, /* 101 octets */
	     SHEX("A873E0C67CA639026B6683008F7AA6324D4979550E9BCE064CA1E1FB97A30B147A24F3F666C0A72D71348EDE701CF2D17E2253C34D1EC3B647DBCEF2F879F4EB881C4830B791378C901EB725EA5C172316C6D606E0AF7DF4DF7F76E490CD30B2BADF45685F"),
	     SHEX("E4D90A392BAB78DD363C8747C586035358F051EE8909762C3D9"
		  "AB4B2F1B7EB50B4738423C4A5087A5B12A9DEEFE186D0086F4D"
		  "D0F9131ACCE891E43D131627316AE63C4E7D1250D809E78F94D"
		  "276EF45C3C8EF4A37AC5910F1DD5F95989C938C98C55771B6DE"
		  "1AE68F3A9015C1FBF2447F3F72C15847EB55E0296B711881E0C"
		  "8518995BD7F75C3AA9C1193F3F7B47B9C133AF09CF53E1F55C1"
		  "7CD96318C821DB4A6C7E79DD718684CD83D43E3EAAD165FE26F"
		  "A204B797C59F33589E92C43E7E799E5A857ACEE310E66F1C99A"
		  "2406CF4D2690DAEC060CC7E3019CC4A842881657E58818BA193"
		  "E4C8B8113EA7B3BA930ABD58A7965F4A176CD5EA1447B417476"
		  "94FB36775FA90999D1898949F8CB5943013A9CBE7654FC5D7BD"
		  "7C915C1D251A22D6D8F7671D77417459366581C5587065C55AA"
		  "EAEAC19488876ED0D5E5F2F3F0A07006500D03EF8CC151AB6B4"
		  "6D587DFF8930AC8EDC3845BD2596387A56825A0036E1FEFE70E"
		  "2A53A4C19B5E45AF6AD1C463042EEE139B244A7751DAACD4CA7"
		  "C5CA075D2B639FB6AEE355DDD4FEC35F13C74E822E5F2C23A52"
		  "EB2C2E209B62106195BDB80F19EF1636CA813E2FDD96425C971"
		  "B6D0ED6B468A5E8F462D521E41586D7C848FE3103559D65B388"
		  "A0D0F84FF6C4A48318BA3860CE5F2C419696931CD89F41733BA"
		  "B732FCF8551B4ED28A3FD2F1DD14D2BD570048FABD6026C984E"
		  "CBC8"));
  test_hash (&nettle_shake256, /* 102 octets */
	     SHEX("006917B64F9DCDF1D2D87C8A6173B64F6587168E80FAA80F82D84F60301E561E312D9FBCE62F39A6FB476E01E925F26BCC91DE621449BE6504C504830AAE394096C8FC7694651051365D4EE9070101EC9B68086F2EA8F8AB7B811EA8AD934D5C9B62C60A4771"),
	     SHEX("AB362A6667C3143E58DB5D5E18294445643F1FB212FAAFCEA65"
		  "6B4C9C9DA509609E7B99AB0BC1C7F782CC8BDD2C2C49A029910"
		  "9F842442A6433D9515BADCB9E961C0EAA0CD90906C2970171EF"
		  "25193CD49FFBE6609BC0F5702CC3FF1CAA859B50E59ED6407D9"
		  "DA7AD2D44EA36BCF8B3F48AAB75C1EAF0C2A8A41E33135358EA"
		  "ED3C5005CDF6114D35E4EA2B14E2F8EE6B995CA73FD75E281C3"
		  "A145309C620B66718E058E171740649DBF4786E8B83C19A5CD0"
		  "FE67B68E3FE2509662D968F4212224B466D04E4878C70B47409"
		  "E5FB56FF2A5F324A96E799CCFC6613B28DF7787A6D967056968"
		  "7D8345FF1F187DEB2A909C66F5B890CBCDFE18594DD52328F5E"
		  "1D3B15F6398F9C3B880AF4D314B1924CF2FC30F96B45869E40E"
		  "4A147D94A4BD92E3CBF64CCCA69B3476477B79B571FC5F91152"
		  "9E151C3DD2FAEA561E9FDDF7DE4C93E5CE3BF3B90D25642EF62"
		  "740EEAFA53CB4356CD598E9CF40339CD8166FE3CBEFC35B5990"
		  "E642D5E2578D893C0A46505AF49280D544CE6865733C1AB75BB"
		  "F194A4B6AB0447EE0F33FDBE80A41E860D80026A2D0C3FC2274"
		  "C9A1BE2C07C57482B3E3982CE5495BFA0F9B5A10D2BC46F5DC7"
		  "8E78168B552CE4D6FC9EFFC0FAA5CBA0CFB66AE078E6DC8DB24"
		  "59C127F970B379E87A267C3A0A5E1672B5A16C93E8E239C9C3E"
		  "51A96EC777361D0583010D4A0773921DC48B63BA3155DA483C0"
		  "3D5C"));
  test_hash (&nettle_shake256, /* 103 octets */
	     SHEX("F13C972C52CB3CC4A4DF28C97F2DF11CE089B815466BE88863243EB318C2ADB1A417CB1041308598541720197B9B1CB5BA2318BD5574D1DF2174AF14884149BA9B2F446D609DF240CE335599957B8EC80876D9A085AE084907BC5961B20BF5F6CA58D5DAB38ADB"),
	     SHEX("D5D2074E7D45403D43D0FCE78C3AB81E2BD601A0337B163AB2E"
		  "2E11CC4F0E73BC0E41ADEAECB5CE2B0422EE97DDD18869BCFDF"
		  "9B4CFA9CDFF5DDFDE53B05D65B5ED250911B83BE8B9E5865568"
		  "5A8D33C72826A1EBC1F244A433A9440B24DCFC691D9A78F3237"
		  "9BD8BBDF4F8E2F8EDA7CA1CF74FC4E8C3C07349DE823980D9EC"
		  "8229A3C43BEA5FEB0BB8F25610127ED1FA1741F90CF25DFE733"
		  "15D440088C3523D9D5BE12431EB81E23EF5F5E9CB1571F08DD2"
		  "4A7EB421CAA74C374F7CCDCDDEB4ED57297222210FA4930888C"
		  "9AAEF06E36C78C43BF9196848CB07E1D941C719CBFB6AF8A501"
		  "4EDA9DF9B205DC905B28EF6F29E2C29CF457317E01BB34ACD8E"
		  "34C67F468CF41729BEE3532833D147CE4E5618855F28936D3A4"
		  "EB6E0382049614A5DE0B05B317240A6E091CEF8A26C7E13280E"
		  "02D19A6635033633EFB03FCD57AE2678C639B262F016647C61C"
		  "E5DED0151951C7EB1FA1E3BBD1808210A3D22ABA98835A686DF"
		  "3612AE6536D2180DE2DB5103E4BE1B2D74A8482450BF1B9F4FA"
		  "91861505F5739F64D7C8CB8C5A3A2F864161F9B495445A1F668"
		  "EB9A86E0C25D1BD0C50FA6431C4D4BEED9F53B6E918008B3DCE"
		  "F98B484161A8DAC12C642DF9278601CCD80F789B770F05E8359"
		  "495DDEF5D6FE54086827FFA8E7B2BDF03A33397CA2898214DE5"
		  "A36D18AC33EE1CC0EE0FC4416A86815C7B98EA08C63D23CEEA1"
		  "977D"));
  test_hash (&nettle_shake256, /* 104 octets */
	     SHEX("E35780EB9799AD4C77535D4DDB683CF33EF367715327CF4C4A58ED9CBDCDD486F669F80189D549A9364FA82A51A52654EC721BB3AAB95DCEB4A86A6AFA93826DB923517E928F33E3FBA850D45660EF83B9876ACCAFA2A9987A254B137C6E140A21691E1069413848"),
	     SHEX("8A80E00C325A49D17ACBB96FD41E874EAF00A0D141CB59DD136"
		  "3365DB060F20C1015929E88E0BA0589FB35CA348B3F02CB256C"
		  "D5FAFC7B1D1A7445C238167F325A2611F7C5E2B7ED44395938A"
		  "5CE501D6CF78C9B9560D8F84210ECA3AC12ED40954EA0C8EF36"
		  "2DE08CF11FCF47C34A3D56F271453BD62F74578F06B1C3958BE"
		  "6D87331D8128070C244900B6DDE1423B2561504156FBC045755"
		  "E1B591234960E1A44FA54CACB1795AD8EEAAD38FAAAC0A4EB84"
		  "6A18B42A526018EAC0A5041DD1200C4169124590ACF93F2F26D"
		  "B5D173608FB0B6C3C83F5C0B01349C558ED0E5851C9DE615CEB"
		  "684A3C9C663119444008DA4A12E9386C98E15CF23CFEEB2F863"
		  "2EBF15381D17268A5F92FD15D464D9E1F6EF896828420C9367B"
		  "14C2B7E886F42C439801E0F119D1086F2A99B5E402E786B8C5A"
		  "3B7902D2274FF1B5CB706AE0B5DA9CE63EA943B8AD8743C0AF4"
		  "CACFA6EC9A2BFB45EF680385568AACC518755B7A2C4ED40F276"
		  "E25BE1447F1131D064A5C54F3BDB80D4E6E1338108251B8294C"
		  "BE4D3F182F3F462B3983ABB7F25B4D9AC8A3E2F3CCA397148EB"
		  "CC154DE7DBA5A4820C677FADDCB502B6A67AF0DC0602CDE5BA9"
		  "83C7BE0235C8BEDF1916F51433D80AD87B4815C6C397A9906B2"
		  "D075BFC0F72431DA10F04AAE61545A62364F336754248606AB0"
		  "62799988E6B0D71C2DBF907E211C963A3823332D6ABF4450D31"
		  "1080"));
  test_hash (&nettle_shake256, /* 105 octets */
	     SHEX("64EC021C9585E01FFE6D31BB50D44C79B6993D72678163DB474947A053674619D158016ADB243F5C8D50AA92F50AB36E579FF2DABB780A2B529370DAA299207CFBCDD3A9A25006D19C4F1FE33E4B1EAEC315D8C6EE1E730623FD1941875B924EB57D6D0C2EDC4E78D6"),
	     SHEX("B645903F62E2A0394B7D98A526CE6C1725A03F4B35CAB927AD4"
		  "FC2831F460CA6EB0959BDC5D90BB37E1D0E3DDC6CBDBB8F61D9"
		  "21B7CC496D903A92166BAB4A436BBE6E9230DD0F91A6ED79188"
		  "9B67727338EA636F54AFFE59D52CAD1B0D26ABA8A29C92874B5"
		  "D6276369FC45769A3323C9788A7AB268081689614622BED3DFA"
		  "207A0382680219D24B2854EEF3024A34DE3FD4ACC827D52FF2F"
		  "1A4B40F1E957729609F8D3AFEF23A94D9E3EFBFA3E66BC1ED44"
		  "C426D27ED4FDA0B4BBBAC6050DC71D064DE236DD800D0E24188"
		  "EA0B1899AB6D0AC16986034BB1C8A21C7AD7F069E706730450F"
		  "046D5F9B3C066C6CBF98C5151E5B45D6B0F3ECEF0377AEE19D8"
		  "3941CAD4484AD07338E7DA24439EE38875498207CFD01349A3C"
		  "B8BCAB058C70B48827C22278667026ACCE65A8F643BDA30A3A6"
		  "FEBB5EB7B981963CF07F6FDF20AED8769C8EC35425C4366EEEA"
		  "26AB28BF43E7FDC8B4F762A2CAC06D10E1A8C6EE35BDA64C6F7"
		  "1CC02675EA4CD20B112B6E12D45DF6D37D83F04CE55DADB9F54"
		  "C8A68E2C76AC55AAFB1A3231C277AE27BF2742D08FFEDF08DF3"
		  "F098C5C60B59D2A89D2AF150B3FC73E399EF94B5002DC0AD109"
		  "7767F47BA07F442D4DF4A4D72E5AECA7B2A37899F50FA3CDED9"
		  "1474DA106FCC96B28D8418EA340BC983344BB58F244ED6755AC"
		  "342B9E6B00CB19F9AFEEB3135E0374CE803CDA44E316EAA0B6E"
		  "1E93"));
  test_hash (&nettle_shake256, /* 106 octets */
	     SHEX("5954BAB512CF327D66B5D9F296180080402624AD7628506B555EEA8382562324CF452FBA4A2130DE3E165D11831A270D9CB97CE8C2D32A96F50D71600BB4CA268CF98E90D6496B0A6619A5A8C63DB6D8A0634DFC6C7EC8EA9C006B6C456F1B20CD19E781AF20454AC880"),
	     SHEX("B15D1DCC49B25B2ECD567E1D0F46732CAD3F9559EFCFBBD1793"
		  "DA46A5D9BCFAD723E504B12B7C1B65EAFB9647DC35A48448534"
		  "4B9DEC5C378DDE04227F7224CFE6F06A962F380AE104146DBC2"
		  "425FEE1D7AE7305E2D2C4E2F91D9188BC88799B730ABF3647A4"
		  "365562ACDA20CBCC7DFDC5C6DEA8DDDB20443B9253466E4E215"
		  "360F4066D918D938FDD204CA826E4A3358D169FB34410D45F8E"
		  "783E7FCBF97A9CF0168E8EE6402C8B627DE5BC4DFFD8AE2C880"
		  "7F863D10EDD57503A99A46035173A20DC10DF3E6BDA62B518E5"
		  "8E94E5624659890F1174B3B7D798206691D3FFA692494829382"
		  "1616184E160F237C1241F11779CDEB4FDD858B73F3DF45F05F4"
		  "3BC81E0EAD584999DA04F71EA4914F8D8AE17DE086CDF69C31F"
		  "CCB48B528781FBF8DCFC710AC421D28E93EA95DB4A6D38C1D47"
		  "C800FCB3F816E36C86D6B3E5D6BB31C325C7A0DCA4D395867F4"
		  "36F077C6711BB96DCBA9A6DEC6318FB9BAB829AE40BF70C5514"
		  "6B87D9FAD9A839CEF462249BDC49B8022926E3C664D66595231"
		  "23A6EE980B69BEA87437322DA1DD6235A74163968C0994AC0C2"
		  "1156476D83F2B8697A1B510795D936DBFC2CFE67ED401CE708C"
		  "7531F1B4DE75E8506DB082FD9B3B24673938242FFF4F30BE97C"
		  "AD8ECD2F88BB5F497A8F154465921C9469B70FBA1F45D76C9A4"
		  "0228A93D67836CB7CFE2F25A5F56C216084521484A7224A7A97"
		  "6323"));
  test_hash (&nettle_shake256, /* 107 octets */
	     SHEX("03D9F92B2C565709A568724A0AFF90F8F347F43B02338F94A03ED32E6F33666FF5802DA4C81BDCE0D0E86C04AFD4EDC2FC8B4141C2975B6F07639B1994C973D9A9AFCE3D9D365862003498513BFA166D2629E314D97441667B007414E739D7FEBF0FE3C32C17AA188A8683"),
	     SHEX("7AF1CD3572930B440F871CC3388678A1C612D6A590CD2B8692E"
		  "0B96010AB4E122CF856925DC84AA0A87C7FB1661C5C74B08114"
		  "88D3D486014E81B0D29B73450A0670A6F8223DF95B542590B3F"
		  "6506CC18949FB6593F1A552E064B4E1576AD7A7CF202E0B8F90"
		  "EAF90147D44EF16031D390940DA5D74C35A269B4ADF4A663584"
		  "292007DA777B04C15C03F7635BEF56C1CA807D1CB3A8DAE8218"
		  "821C44C7E6275CBF64D82453F764B458ACEB881816234494F71"
		  "7F5240CFADDB3F11C1AF02B269F0F76B9DBC8ED643FB3B7C9F8"
		  "A540A453C1EC85ABF9E8417B1E4DE9DC65181EE32802ABD6170"
		  "A581AA2DD01188752C73D406461892D0D0909C5D3FE0ECBCA3D"
		  "28F65B905A2DFF142E61CB088F9086D3C492E73C0EFF3AFBC37"
		  "216EE98533B236406D80797933E54D3C21AE6B1B81F459FFD7B"
		  "8BD1EB872FD62CC0B95F5384668A1BF91A474BFB97335DBD8C4"
		  "761F3BB936E2032365256B534F98E9F4B3C50B4034478EFA268"
		  "E9ED9EB9864044E90DCEDF4A2ECB9E8617C4E4ABE7CB4990511"
		  "43C30BF73D397836E422A2CCD222A299603A3E4EAEA9BD2C3D6"
		  "4A0ECB3CC7C0BC3D867C79231BBBC8C2ED4212EDC8B25C04C57"
		  "E9A3EE8C103AEE4AD6AF025016C1A177288F4A17FE20A2E210B"
		  "24A7AAB63D3FB4E687ED3BA2BC163F17C9067D07C18204D0AF9"
		  "6A41CDF5BE9C6B89502D7CF1283E82410B11537CC931443076D"
		  "60CD"));
  test_hash (&nettle_shake256, /* 108 octets */
	     SHEX("F31E8B4F9E0621D531D22A380BE5D9ABD56FAEC53CBD39B1FAB230EA67184440E5B1D15457BD25F56204FA917FA48E669016CB48C1FFC1E1E45274B3B47379E00A43843CF8601A5551411EC12503E5AAC43D8676A1B2297EC7A0800DBFEE04292E937F21C005F17411473041"),
	     SHEX("6B1B8AAC47F782208F50AF3F977EAF8F2D69FC02390EE17BF17"
		  "61ECB73D1088D236A5345380B6297AD1FDCD5117D16A3C410AE"
		  "EA092D9892AE95C7B977930655C75CF7834C101F69D2E33A1A1"
		  "7A489E040C5DB3F83A5216ED5778D391630D17405EE0A96FD7B"
		  "834448FE0BC61CEC5014A6AF9B0F85827832E7469869B087249"
		  "7033822D3419D43379AEB6C31406AFD1199ABAECF3AC8E2546B"
		  "9736F8C4E399C8698B674DF5CF6BE81C73E52DCA979D3281200"
		  "B9DF8D99BB6C65A7B6F620E1E4E6538A918C99CDE6CB33150F0"
		  "896BB75421B8265AABC241C02D93E63476C351396E5B86DC66A"
		  "05C74AE2EC87EA1D175D15A12ADF18F9792970DE3B9E30890D8"
		  "896309815D57AE238895AA7E76DD6A46019F005F193C737B452"
		  "D32649A8F39B1ED00AB8961A6EB04F699EA62357B7A38C423BA"
		  "9D421529266459EF292B324A16062538E8C6550F822C2C14081"
		  "6340F907D5D10B563198BB1539B82E5BEDE700F7FAB66EB8DA0"
		  "45B23035DECD3F8DF11DF98A297A49F5B896A3536BB5A3E17A9"
		  "DA1B266EE18FF6D4A6E59AF619306BF30316FDCCB7FBF580091"
		  "6C18832F4CD21659774411F105DED35DE8B0D4B0B54755440E2"
		  "6230E519B66319696E6ABD180FF5314EE184D3B2781F338FDD3"
		  "E3935855F4C8B4021154EAFE67497599D0AE393A516673A970C"
		  "FB9B7058826D7F0FC29F3ED34ED31F45790A63E5AC7918A4C3C"
		  "B494"));
  test_hash (&nettle_shake256, /* 109 octets */
	     SHEX("758EA3FEA738973DB0B8BE7E599BBEF4519373D6E6DCD7195EA885FC991D896762992759C2A09002912FB08E0CB5B76F49162AEB8CF87B172CF3AD190253DF612F77B1F0C532E3B5FC99C2D31F8F65011695A087A35EE4EEE5E334C369D8EE5D29F695815D866DA99DF3F79403"),
	     SHEX("F76B436C25E0BF2D75A0DF0152D04117ED8DC0AE55B13A2CF83"
		  "9E41A90C6070415EB8EF70A90BF205B0BB330435ABB41BB3329"
		  "D015E9E56E86418BD671D0375E3AF8056EB882AE4191F82E062"
		  "339D5505526AB42BF6CD54F0D36251A78EF274CBC797B89672C"
		  "A224CB9B94E36648B362338D4981821B0B388E08CE97253B5E8"
		  "C776E9B3AEA33DFC3815B5929A3800BB83543D42E0F331C264B"
		  "1855AD3C002BD80D035932268DFE4CD6600F2B1F215F1670D7A"
		  "4512E89C3D1509271ABB42E2608D7B4A900E00593E0509F7C88"
		  "2AB0008AA83F5879F30D95A319A4058987804E5014212D415BA"
		  "0E1C2407B9AA7B80945CF460E60609AC885BD7D639C9BCC961D"
		  "6C32A96F041528B64747B5AB9557971EADA1EBEAC47F78B3992"
		  "EEF2BF2806213952489934E7F5410602ED0987D8470AF3728A7"
		  "F5BA326B77835694E32675E463C604C18DB4E4F5E1ABD056689"
		  "66B8D102172E232B4EAEDB3C73E7A406C36DDC641469350D593"
		  "336E9E5C8BA3C246154BD3CAC8E14CD60C9EA724FF059C931B7"
		  "78B00A3C6E1016C1C41B3002F01D937499518A969D02AD08B5A"
		  "B385A60CF5D018EF405DF9CAA63BC49015407B04AEBBE02F206"
		  "DAB67E72388022D99D5BA4F6D57695186873276770452D60263"
		  "5AEBBC988B7718F6EE99F9F05A6028E074BCE86FC134ABA4CB8"
		  "2E1529ACC2E50E53F466B9B38FD3FC7181163D9A3B58F143AAB"
		  "6297"));
  test_hash (&nettle_shake256, /* 110 octets */
	     SHEX("47C6E0C2B74948465921868804F0F7BD50DD323583DC784F998A93CD1CA4C6EF84D41DC81C2C40F34B5BEE6A93867B3BDBA0052C5F59E6F3657918C382E771D33109122CC8BB0E1E53C4E3D13B43CE44970F5E0C079D2AD7D7A3549CD75760C21BB15B447589E86E8D76B1E9CED2"),
	     SHEX("C870CCF74F9A979827E1D0FF8E1DAC93A2E0F10A0BA7A4DE041"
		  "853915DB7D86C65010DB41421201025D3465FF1267EA5DFC8E9"
		  "0CDB4D933FA58F5A5E830C646A74E5BF807495D973929CD2D47"
		  "63A0615902B8DF2D04A9D2553BCA94696DB2A332EC56B5612DE"
		  "10CBD2E117AB226769A0F5E12E380E890489C4CEE3156B4CF1D"
		  "4FA2FE55E20DB06D84DBB0F9833A3364F64481567C74049517D"
		  "B8D75F6607467EFED0D8976ADC7C7F265371A552767BF6DDF87"
		  "F88DD1BD326C46CC7C9A895D1D88D4452A1D72E7ADBFF6C02C5"
		  "114ED5799C4E75944294BFF16DC782D529341BD289DA6136E0A"
		  "4223F27E886D8AD057446E6BD41AB8EFD676B5084084A313E0B"
		  "BA30C1DBC3C461AFD8B5C0EE02AA0D1BFD0A1ACE35369F78D55"
		  "31286FB268F94D4EF50BA6A502C2EB2B744651B71B59974202A"
		  "8088780CDEBCAF18AB8CA8E5AE928309EFDB5D5A639D7C83BEF"
		  "87E5AAB4D403F7A9D22EDD1CBCD409B822F33225F05AE21C2BB"
		  "E9E7A0B116B502DA93D6547E3931763F4FBE1507DC031CBAFDB"
		  "08F4492396D4D32FDF8936D0BD082D45E70B3CDF8B807D974A7"
		  "0BF17FC54A4A034BCD7F5DDA8C1002B92A641DA7E717D8C742B"
		  "7F22FAB4918B1B3929C2CF7533666121184383746FC5FBB06E2"
		  "796EEE83E6387DC610717FE4C7FF316D87367F6236E36524CF5"
		  "25691D656F77732B4989D87FBADCC67886E151F7A61FC9141AF"
		  "81B4"));
  test_hash (&nettle_shake256, /* 111 octets */
	     SHEX("F690A132AB46B28EDFA6479283D6444E371C6459108AFD9C35DBD235E0B6B6FF4C4EA58E7554BD002460433B2164CA51E868F7947D7D7A0D792E4ABF0BE5F450853CC40D85485B2B8857EA31B5EA6E4CCFA2F3A7EF3380066D7D8979FDAC618AAD3D7E886DEA4F005AE4AD05E5065F"),
	     SHEX("7AA373D1AD451F79A4B17BC9D3E2EC4D43457C6251E958E44F8"
		  "D5EF360EB256CE5ED5082B938095DF2EC30DEAC299EEA526894"
		  "6CC240AE0CFC0B8565235A77BED4B58560F111FA5630996AD51"
		  "065A6A957979344C26C0FBC9843251B64C899ABA1DBE5124A21"
		  "A604E706A50478DB8E93F239CD11D53B2F168A946E979AEF98D"
		  "6A3CEAE032713C241B8176F1F32A7F132ECECE3FC57309F6F0D"
		  "C9876B9327C4C03C42A74EC216E0DBE026DD167EE94FE8B0032"
		  "F9BA033ED3E6BEF8DCDDCB288CE5F8D1B4DBC01A5BF51FD7252"
		  "06283C7028D8013A170F0DC4AB107A442C95B0214A71D3BD7A2"
		  "7CC48CABE6BDE4A5AA4698F56258E175BA1C6F7BE6D7EE60CB5"
		  "BC4926CEC0D94A422554FECD2D7666462E125FA3C2FB84E98F2"
		  "095B119AC492EF8EB3666A3BBE51A6E4CE3CFD1375A21D1570B"
		  "E8A1CD22009E8261B3CD6F2E2085443A6AA76C91A349A7C49CF"
		  "5F477FC017AD18439C580DBAFC94E96739732DDADF4829F9DA5"
		  "FCFB6851AFA968E208D4090AD99F61F9657EE958698FDD1DD5E"
		  "F21543D238D05C4C604E26D21DCCC8C878344EA888F12737221"
		  "8A4511B803E3E6EA03CDDB2574E636EB40CA242FB31698290EB"
		  "9C017D326A35D884A88C98B89C33E8656B717DD734223EE76B5"
		  "EB147AF7521BF1F827827B37A6563E601AFDC3796D8DA4D1084"
		  "FA462CD4AF969A800F522375E2C754BF20104F13A95FF2DADA5"
		  "674F"));
  test_hash (&nettle_shake256, /* 112 octets */
	     SHEX("58D6A99BC6458824B256916770A8417040721CCCFD4B79EACD8B65A3767CE5BA7E74104C985AC56B8CC9AEBD16FEBD4CDA5ADB130B0FF2329CC8D611EB14DAC268A2F9E633C99DE33997FEA41C52A7C5E1317D5B5DAED35EBA7D5A60E45D1FA7EAABC35F5C2B0A0F2379231953322C4E"),
	     SHEX("A1D00913CC206BE2D1C38C22B6C732A05368C8AAB30865BE534"
		  "3BE3C73AA6B7E8AFF69A544257DB1E6B6677E6CCDB6EB358A0E"
		  "D096BC9FFF0DBF89B1E5345037864132FA7B820DDDDD5A88B95"
		  "3CE834F1AA89BF56FBB02670DFC9FA29892FFFE61DABA419585"
		  "0CA669F6E4AE00687AE4AF3A15B27A4E489A4AA01AE62E23FA0"
		  "12CFE38BB601B343095B5DA20CC2B48352CB245C5306A2FA9F6"
		  "CFB324B6DEBDA6513221197B24EB1EC2564E2527491B539289E"
		  "AF593912458D835E3F8F46E058CD6B3B85D0833DE691A1E08CD"
		  "6742E7B2957DDCD0E04858FD64987C66B6BAFA7A082F32D5158"
		  "52E931A382792AB4FFE17B39D8918CFF002CB01C0CB1E1FD32B"
		  "B8B02DF964752523E64DF6C74B8442719C3D7DC75CE9DCE5474"
		  "AD6F235280CBD1F0268F19E695549016E948A7141FF4602330A"
		  "096C67699B29DDBBB8F6600770CBD0E398B013932B04526A055"
		  "B51D92F0AE8535A7E91936BF21AE70368E09005B14F3FD9E363"
		  "5F35960D89E134E9B1447971B0D1CACF554845AE74CD93F73FA"
		  "FE94189BC22D1DB9815A65B414DDE694FD053AECCE3BBC3CE73"
		  "62F353A9434A49FABCDFD7E3CBE73ECE064DECA30E4BE01B113"
		  "5FF6303E45E8065EF415BEFEFA948ADF42DCEF4520C2772C8E8"
		  "438B60113919975CA0DCD2798BB026D4B59D3904FBCA4998533"
		  "ADED822B15CCC86457E8507AAADC7E3EE221543472A87B730F4"
		  "E5C2"));
  test_hash (&nettle_shake256, /* 113 octets */
	     SHEX("BEFAB574396D7F8B6705E2D5B58B2C1C820BB24E3F4BAE3E8FBCD36DBF734EE14E5D6AB972AEDD3540235466E825850EE4C512EA9795ABFD33F330D9FD7F79E62BBB63A6EA85DE15BEAEEA6F8D204A28956059E2632D11861DFB0E65BC07AC8A159388D5C3277E227286F65FF5E5B5AEC1"),
	     SHEX("E28D36BD926B9E8C105FE0099CFF1957ED7ACCB9618027421FE"
		  "BF9DDD3129F4E6713376B3B7EE710430EF381CB12B831C80C7B"
		  "1B93765F7693506733A75AC0E8FB5D3C30C6F4DC735C4C35991"
		  "61ED02F4ABB5CDB174E53D93D91E1321EDFD79C0F63C92C7E5A"
		  "253C1B8C4842F053CED0033405FAE5F16E2DC30024A6B1663A8"
		  "6829ABFF72DE6F2A91C1CFE5955CFB584F8C65B9334590C9C46"
		  "8A9B0FAC172792AA152F7DEDEAA4FA22FA7BA44131A1143F790"
		  "53C7E2CB5AAF8ADCCB6AC72300892F792590BF5A1A9745FE593"
		  "4C712CED382A316C487EFE0E6F146C8A562E6B40834FE20C362"
		  "E495FE9E5CC871EDD2D61361492EE60CEF26ED38B4DA0480B4D"
		  "FA66F5E71D44F10159016093841E02C34001800C5CDAAD0F9CD"
		  "2A3144911031139CD6C54443A1D82405921B86DEB03E8F0C1EC"
		  "D58F63B88944182F034BD07B0D021EBB7447F660E66BBC0A821"
		  "C7852D183B806F0855AE194D80073280F3D07C2DC03571D5C6D"
		  "0F0675AFD58ECFB211EA27733E870A6E86AE5108D8C1C3524B9"
		  "9648541B61935746310FC6E99935E7C9A5D144E0735F2CC32B6"
		  "DC42B34679052203A777AECF26D9461946699F834940BD34F8F"
		  "CAC7C22E29B2B0CECC60183C44A5A63078127C10C1C14112226"
		  "520CEBF9D47A7FBD4002AD482CDE289E7DAD3AA891BBA0B6B2B"
		  "2FA378D26D629B4CE428F375BAE3FEA5E5C6B048928EC92099F"
		  "3C3E"));
  test_hash (&nettle_shake256, /* 114 octets */
	     SHEX("8E58144FA9179D686478622CE450C748260C95D1BA43B8F9B59ABECA8D93488DA73463EF40198B4D16FB0B0707201347E0506FF19D01BEA0F42B8AF9E71A1F1BD168781069D4D338FDEF00BF419FBB003031DF671F4A37979564F69282DE9C65407847DD0DA505AB1641C02DEA4F0D834986"),
	     SHEX("E3E693B2585C32926882C97BEA160D46ACE5ABF0FA47A9084FC"
		  "FDB22DE3C65F43717358C2C2CE680D6D132443D478478A30471"
		  "1BE3C5795F3EB0A7BF043CAE06E84443EB599DC259B3CE1F418"
		  "4ABF1E6794556E9F2ABA99E5363DBC7E9C2C802976C87FDD7E0"
		  "E3884B971AB33DA6CAFB65CCABD6CE492A59D359BC17D221638"
		  "1AD838989DE9FE82245CE07CCB1239FA5A3D664D5B8F2E3410C"
		  "023066AA47E030FC73BE5EBA27D83D656106401BEA73E36CC09"
		  "F8A78BC0F4FD7C4B2B4E7D36F4E3CC31867D1EE207A7936FB3D"
		  "5FCF512669AF63B9F01E4188ED39B1142CAF6578F788B53C3D6"
		  "303C25C545DCE3F0B037456E3E7870392F43051FB2E8FDE663C"
		  "80350F94F7A52D511A56CC268335D1ED24C9948C6E0426AEDBE"
		  "1E082C0F075863E3DC0269BE5C83ACE9A98AF1EE45C79BA263A"
		  "A81902E11F880205B7BBCDBA54F766150E744BD1E2C97E64B4A"
		  "50A179B940C01A7AEFEA23E22DC3443D655E31FFAFA67FC5557"
		  "27C8D02C3FB3DE7D6C13438F6323B89B7C975FC5B90D80BE8EF"
		  "D41BBCE90F636CA01FFBE3AB39AA0F43460F33DFA492718A411"
		  "711E9E7D38EAB956655314F35D36120237BEA48EEEDC5492560"
		  "90DDEE8AB4BF2669131CFB590C65BBCE018CEEA3CE349A8CB83"
		  "E3518CE8F353EB07F9007600C206C7F6B6B1DA24F1A263F0397"
		  "73EF5F99FAE6D9B2B3160199F17D49AD69D51BF66915451F43E"
		  "CDF4"));
  test_hash (&nettle_shake256, /* 115 octets */
	     SHEX("B55C10EAE0EC684C16D13463F29291BF26C82E2FA0422A99C71DB4AF14DD9C7F33EDA52FD73D017CC0F2DBE734D831F0D820D06D5F89DACC485739144F8CFD4799223B1AFF9031A105CB6A029BA71E6E5867D85A554991C38DF3C9EF8C1E1E9A7630BE61CAABCA69280C399C1FB7A12D12AEFC"),
	     SHEX("A178F3B574A841C88EF21B6A20A100625334F4FEE5966AB2164"
		  "686151AB9FED082CA4A9115A0B5F099C5DD4BE4EF1082D53321"
		  "2BB862F084225655D8856E29F279418DE6FB723ED15C341CE31"
		  "0B53E695C08689F776E4474FB6DBDB7DA46A9A2DFB7BC374AF8"
		  "6984E1E10AD9D48B744A5FA98A0B161738468E06EE4E836BB71"
		  "8BEFA381CABE18E9973FF07DCE1920B19DD797F84CDDC52DBD4"
		  "9B4A462FD1A12ACD364804A290BEAB30AF0376CA098D44E5D14"
		  "1D1636F6073569F2CE7213F6407EE376680E8D676942211A9EF"
		  "B756EF9DCA40574AAD2084FAE0B617D727D7951C1321616F265"
		  "69D96DE5ADA8EE101BCFD2C43C338F8957DEC234E23F5A1A0F6"
		  "63EBECB98C35102DE1C101B99E8F42BD8EE7C528B1651099545"
		  "F7BC9BC315C8861FD7EBF26B97A519931A63E0B06A73B14CD23"
		  "65383B44A2107FE1AF0998F1574D9FA8818DC51D642240DBD0C"
		  "104E89393B2C381944308D828720564763D3BFEE221BFA99D97"
		  "6701717D4214CCE2FC25763C1B25EF67286BB7E497593B1A077"
		  "912E3AF76993A4E20D4E8ADD7AEF4F715E5F1C8BD68E8913D9D"
		  "8057447F273FC41F70A0290A5EBFDCE46209090032801F2CEF4"
		  "4C8015E865D75759C7B2117C6E8EB6B57398B992B0E073C1FAC"
		  "374C819B7AE16C464AB5019A45E7A6AD57B75380FF42D35339F"
		  "23436A0D58497F7A12E4C3EB34816C4AAB9A2D3655079C12E05"
		  "EA36"));
  test_hash (&nettle_shake256, /* 116 octets */
	     SHEX("2EEEA693F585F4ED6F6F8865BBAE47A6908AECD7C429E4BEC4F0DE1D0CA0183FA201A0CB14A529B7D7AC0E6FF6607A3243EE9FB11BCF3E2304FE75FFCDDD6C5C2E2A4CD45F63C962D010645058D36571404A6D2B4F44755434D76998E83409C3205AA1615DB44057DB991231D2CB42624574F545"),
	     SHEX("11C57A7AF45809863E024835F2A86BCB81763412DBCEB53F311"
		  "53730FDFDC32A8ADBE3A8BD6D4778EA33AFC063D9767BC9E3AF"
		  "EEA3C78FFAF5FA3023093C798F39111A182516218F0BC78C030"
		  "7C75EB212B0EDAD184500261E00823F423D84BC2DAD3FFC6C2A"
		  "B675A24CCFDF0DE5F669D1EAF91973071D50EC7D2C0CC4B6E42"
		  "552199CAC4B65938F7E704248CBFF272C647E953FB9719FA8AA"
		  "9C319FCC45E4FAD0752E7416F9DF0063F1A8BF43E316533C046"
		  "C4F2E7254A02FEEAEBDCD4DF19B01C18B8DFEC1D291FD670BA6"
		  "D434B41EB6068A365973082107ABD6DFB62FD428B9B183E1FCF"
		  "CB654A6C7A55B91E0615658529FA4F6178FA4F5CEF329376169"
		  "AF143B137B9B81B273796CD26CFD8372AC3184600EBCB2DCE06"
		  "AA7F22C055A7B622717DB8E15571ACC61D94D8DA250FB8E6354"
		  "9528CB53050322520925FE7FCD065BDEF0EC79AA38F1F105348"
		  "614A57605C94BCD50DF80E692D1117C13BA3563AFB44E91117C"
		  "9F6E4E7FA35397DFC58FE7F2DAADCDFC756EDAF5BDA38D6E1EB"
		  "819A1080CF354F2AB779F63E2BCC763C8AC73C6D811AE2B5252"
		  "E6FDEC79E2E97D8971D1EF39E2A144E84EE8087C673AADF3C46"
		  "C4E6C1D9832E68BCAFF8C1E4DF1A5454D3A677040413BA32BED"
		  "431B2F7D3F356CB060431A7F5B1B6CFBD5709736A408EF6CEF1"
		  "B455EA74191F09FEEF230A1A0296807E8AB33403A9E3EA1650C"
		  "D2AF"));
  test_hash (&nettle_shake256, /* 117 octets */
	     SHEX("DAB11DC0B047DB0420A585F56C42D93175562852428499F66A0DB811FCDDDAB2F7CDFFED1543E5FB72110B64686BC7B6887A538AD44C050F1E42631BC4EC8A9F2A047163D822A38989EE4AAB01B4C1F161B062D873B1CFA388FD301514F62224157B9BEF423C7783B7AAC8D30D65CD1BBA8D689C2D"),
	     SHEX("8E03C6B48B4E91BF5364CF9311172442253B6ED58B95D147A46"
		  "74BCA639D75820FFE93509AB08FBE7EBA70544A9331C3BA3347"
		  "B0B44E0FE36F9B799963212514AB2E673133EA5D9CF3181FFAC"
		  "7F79475971369F7A0188F1A3687792258617C7521CF2EC97924"
		  "1D8E9345B9205243A1CDA692418F8855BA61B57C75CE8A7B2D0"
		  "19880D2C56C61DC0E3CACA0CD632E8819DCAFF9336498E1D573"
		  "C16269D2D56509BECED579C80C3F09E4165081EA7910BB8D13F"
		  "0373FA802F498AF190EBF8D7565A93A771AFCC8C417EA460E67"
		  "F4AEA898C73ECCE5B17EBCF6EEF8A6E31D160038604499B3996"
		  "5212588826D1F896915500497BF4F949E2CCC4C3439315E7AAC"
		  "C2904A77AC0113A2AEFE64F6D625A7EF547E99FDCF9374300A1"
		  "CC15373494BC1210B8DD5B5CE04B190A824566A9AE4A02DFC70"
		  "FF8DF3191528A91E97DAD97057AAFB0357AE7F509FF7E464675"
		  "124D0D23399AB6DB37B80397BCAAE48B8520860B79EFD72D955"
		  "F53611E8D782A28C729F51D3174850E4FD61A4328709DE8170D"
		  "1BB7B5A811BC3E0FD06F20F03CE1567CCDC5891673CD8E65947"
		  "2015EE6A557D89B104E22E70315B40701B49C76BA0196F68A92"
		  "55EEF2623E531D73D151A7F321409BDD8183CAAFFC18C6DD63C"
		  "1EBA47F07B5B3D404C2203373EF356CD26D89DFD8F5BA2F7006"
		  "46507AF3CF63DB907F9F70DDCB125259F28275CE8F41B63D140"
		  "58C3"));
  test_hash (&nettle_shake256, /* 118 octets */
	     SHEX("42E99A2F80AEE0E001279A2434F731E01D34A44B1A8101726921C0590C30F3120EB83059F325E894A5AC959DCA71CE2214799916424E859D27D789437B9D27240BF8C35ADBAFCECC322B48AA205B293962D858652ABACBD588BCF6CBC388D0993BD622F96ED54614C25B6A9AA527589EAAFFCF17DDF7"),
	     SHEX("845E0DD7FCEA5E19E5DBC6CD03D58C2BAC4B81CE809BA7D0C75"
		  "CB867DA6ECDC9CEBCA177AB9C3273C8EAD7B8C9F571D6D1D14E"
		  "D0CAF6FFAAE4F1B7EBE653443E9BF099DD54CE3A021E0944BB1"
		  "F8B34CB8CF199ED9B23FFC94E2FCB6FBA03D50D0C84C2C943F1"
		  "7C1BD9EEC662420EE42ACD5C5F18223B92AC269B320BC190355"
		  "0C1D55B7AE5804EAEB6F67C57537161EF8460E45E34FAB823AC"
		  "0EEFA232404C7907DB88E4B1D74AF7DA5A2E38F82F96A86A506"
		  "DF1E717E636AF1FA85BCA3259C78FE7CDE8FF56CD2461997FF8"
		  "D3B31BE985B73E4EB0DBF36D39923CBA9982870DDEB52ACC02A"
		  "97CAE8688E230364D5637170C328175208F8D225E2B19289A29"
		  "205A9568F426FDBDA7BB04224808B2F21E4E80E2F18F4E79065"
		  "4E0D6E486340464B6FA48C95F99A4106E95A6A378458B863655"
		  "6C700D79FC45697C09A84D2EC41BCFD560F0A147ECE64006071"
		  "3A86516ECC642396E16A63203CCE11809E7DADB5BDE418348E9"
		  "1E30DC65E66242E29B69B9CEC5548D4BEF3E15A6CD4937C492B"
		  "2F0FF83DD3ACEF224EF66AA27726161A418820196738EBD9569"
		  "B49D51D1583129AFAEE4F9128C55813151895A046E4D5B4E7D1"
		  "695B0D47BC657EF7795107148B165D0484F346C9C49A8DEE187"
		  "E3C1F28A6D57FC3B7DA49042B37250DD02BC9807FE1AF57D5A3"
		  "1664C90D5371542B28965E805224651F96836147E0300C2A938"
		  "639E"));
  test_hash (&nettle_shake256, /* 119 octets */
	     SHEX("3C9B46450C0F2CAE8E3823F8BDB4277F31B744CE2EB17054BDDC6DFF36AF7F49FB8A2320CC3BDF8E0A2EA29AD3A55DE1165D219ADEDDB5175253E2D1489E9B6FDD02E2C3D3A4B54D60E3A47334C37913C5695378A669E9B72DEC32AF5434F93F46176EBF044C4784467C700470D0C0B40C8A088C815816"),
	     SHEX("EF1A40B1819AC20C49EE1F31E36B853619C8E960DA1A5BC8667"
		  "4C5720C0F255E8099CD4572D09FA5B8F035ABEE5FCA1725F988"
		  "954513E2E7FDCA92A55A757145B0E1A223A7CD4E0D18D3EC7C8"
		  "BB33221B24AEDEA4A08F01B2130D58E19570272456697DF883A"
		  "B29F4D4F86DDBFD1D97F1018277F84FFB615451190762B9B725"
		  "56677591E6EBA3FE45F8769072E234602E222C77DFD8CC095B7"
		  "3D1B231F3F09B89C2ECDCB741FED58085CD763E2B016623BF23"
		  "92631984B6E4D2FE558B7F17973A8F58BE66D73AFAE5DE5C5B4"
		  "6A6205E15F8A7F5B3ECCE65985EDBFE4C86D3830E642C99E44E"
		  "F35AA55B834696DC519635240D6A693AC09CB3D567105420449"
		  "664EB7B6BC367C482CB2682B1A34A9A051470EE3DAF8045B0DE"
		  "FAE0EF10581830A1C21F75045D1297C9EF4B874E9004D4BF86F"
		  "D770B820A9AA1FE967388215AAF00DDAC57F9B3B9CF0A67FC81"
		  "E27565F1F3378A620C9A99B50AF301359E98D31EAD034DCB033"
		  "4BF45D3CECAE0A95E4F0F0D60B0725F97AAF0EC1C4EA1D7D062"
		  "D5B13A69B68CE5FB382FF7E25A219C7FE8CD1D78B47660319F6"
		  "EA04418970EB3B5FA6778D84EBD4377C8B3DB2632CB616631DB"
		  "82CF68CC4A03A44768B082F8A5FB606FBEDE8AC366B9D0813BB"
		  "58017098A9B2D5BAF9C5313451F7FC5144F8A89EC34DA3CDDAC"
		  "B079A963BAD7E08A4D68DE34612B116A0587731F4DF81355999"
		  "40B3"));
  test_hash (&nettle_shake256, /* 120 octets */
	     SHEX("D1E654B77CB155F5C77971A64DF9E5D34C26A3CAD6C7F6B300D39DEB1910094691ADAA095BE4BA5D86690A976428635D5526F3E946F7DC3BD4DBC78999E653441187A81F9ADCD5A3C5F254BC8256B0158F54673DCC1232F6E918EBFC6C51CE67EAEB042D9F57EEC4BFE910E169AF78B3DE48D137DF4F2840"),
	     SHEX("B9A11246D631A7C9F33911215F36281030AA4959E598F2E3660"
		  "2AD66E236F4EADC9E87A662FBB5E30555F7F2048571DB711D9B"
		  "28389525DC1BFF502C1766BF7EC7A26688CF30EAA16A23D00BD"
		  "408E11756BF0E9DC3265ABED5B83643260DA0F7B58919078EDC"
		  "C9F62B2D14029657426407CFE5B514E17410F3351C401EE163D"
		  "1897525441D6738ECCB5976C16EBF309ABE15FC5B15B3623385"
		  "16BB8FC50DC69EBFD5E3CFC7B79571C7DCA79D57970AE090454"
		  "7677B30CDD7D09A0973AE65D6E00F3F9205BA0D806CF3DDB5AC"
		  "BFB5AC1DEF3AA8766E4921B42DF7B9E3AD5299747300C9D2640"
		  "6CF195DCA47A673E232454DC80031FDAA0E3DA4B66DA9DF72B0"
		  "40893378AEDD6911D35D3E007828EB749BED4E16657E9D33867"
		  "E0319E6D4BADA0F16C466F3B1B8502BDC2584BCBD7128A976C6"
		  "C8BF2D553D38AB28B982802D1E626C47B5EE08530FF22A9852A"
		  "C113C560DDEF6931F6E6145C3E525F5E6269E7DFCD4C0517C16"
		  "884C681B1BD78D4A415039439BC5947C65CC35BD52E7E2D0FE0"
		  "C88B2C92C903A0DB75D5BAC3D3D6A5A323CAEE0E55C61E8BBC1"
		  "113A7D8AED0B031020CD7C50346B2C9EEFE61C20C14AB947CDB"
		  "7B3D153172D5F32BD267B4D77A7606CFB5080058F56E51CE9F7"
		  "3E7D75C94A46A4E917BF575827D37773F2F9EB5A552420AEAD9"
		  "8DF822A7EED212538E2B27DF5EE2188561775BD862902F3DD19"
		  "A9C6"));
  test_hash (&nettle_shake256, /* 121 octets */
	     SHEX("626F68C18A69A6590159A9C46BE03D5965698F2DAC3DE779B878B3D9C421E0F21B955A16C715C1EC1E22CE3EB645B8B4F263F60660EA3028981EEBD6C8C3A367285B691C8EE56944A7CD1217997E1D9C21620B536BDBD5DE8925FF71DEC6FBC06624AB6B21E329813DE90D1E572DFB89A18120C3F606355D25"),
	     SHEX("EA756978CE1BC77F9C84A151604F37CED854B00328DF4A486E9"
		  "626C7DA6098F1A6380EFD8434FA59B037AB18C3488B404C03A4"
		  "838A9E6C60B37C9348AA2CC3E09F016B1F07FED8D4A0B93BFD9"
		  "121B4200A706887297A91AE12E1B385830536A88AFADDA80D4B"
		  "14FC5CC9C49916CCCBB47F970378AEB5EA6F6F583D131E53AA7"
		  "08196F749644DD153A3177F424921254A7FC92CA5CA241D54C6"
		  "A1E565298D2BAFFCB173D22160241A535D34358FD0B54640CFF"
		  "9E04890BDF23BDA5DC770240FEBB778D1D0B4B3C98CA7AFD251"
		  "2596EBDF02F8E1EE5C74BA26A35E1482784F6284D0D5FA17CBC"
		  "695508026CD49F30D82FE5819378DE13150805555406B8EBB6B"
		  "BB8DEBF55A28B5F1350EF5AF18DAC71C9A02E76054EAF28E2D1"
		  "259EE670492E75E1B3BE45492D6E3BC5096BA2952E7B1E97DBF"
		  "5E18618EE7B197E6FC3B0A31B2F071376F6C55CEA08BBDFA37C"
		  "F676A24DE82F23878566B48269141C17C7304D6252952158078"
		  "B481FA4270C2DD631AFEE7D7518D1F1DF43B9841F10871D3B80"
		  "E4680DBB89CE8C4CDAF4E16A4892A83F927AC7E4EA692C6EF1F"
		  "E6A6B2FED3C8393A5100DDEF4B56FD158E45A95844E2195160C"
		  "3D3E02FBF51A715C0117627C783D503CB241C636DDE3443ADFD"
		  "AD82F5D9226D49921FF60788295A47B0D76AF43977FCAFEA4D6"
		  "1DAD09E2318EA16E32C44B14AAAB6224193C3B9EEF105057A2B"
		  "AE72"));
  test_hash (&nettle_shake256, /* 122 octets */
	     SHEX("651A6FB3C4B80C7C68C6011675E6094EB56ABF5FC3057324EBC6477825061F9F27E7A94633ABD1FA598A746E4A577CAF524C52EC1788471F92B8C37F23795CA19D559D446CAB16CBCDCE90B79FA1026CEE77BF4AB1B503C5B94C2256AD75B3EAC6FD5DCB96ACA4B03A834BFB4E9AF988CECBF2AE597CB9097940"),
	     SHEX("609D7E2F634D0457E311972039E1645E66392CDBD1414D5A8C8"
		  "39F8B22618AC73BE81A745B9E2B83814A527F78D588119F7464"
		  "CFB235E56D9E2F794A5BD510008689ECE3EE818707BE101A3B8"
		  "A5E1211E3116AC0E9746F79A01AB9DCD01783E961C316EF4EE9"
		  "08CD69C7C2B0E3560FB6DD5649896403D07F021539F8ADF22CA"
		  "4E3535A81EF643AC230A0C94913E9584FD8AF77926360D280FE"
		  "CA2283F8097DD843D509C5E34A9FA8B02BBA9015753D3F728AE"
		  "F7053407A90DC48F15EA272DF1F478B967DA4C02C8D4F2903F9"
		  "A98F90CA1589BEE5590195E0F6B24913641C2FFB42DA043B454"
		  "8D82F8F0F580356E9946022B06DDB0BDE4947D4A25767C65D1C"
		  "A02148954D1F33F07ED0886552C23286696067FACDF37B5D468"
		  "B1A5F29126A2805BCBA687F61D920C53B8346579EE0D312691D"
		  "FDA9E53B3A0EBDA1CE213C78DE4CDAA3C5D9C6D5A4C66A5B532"
		  "37A7E5D80E8C1DB49656E28B2DF50C02E9BC580A62EE5E6F72D"
		  "9681435BAA70C0B51AFD2CCC109D0D45634072BD404964FB44D"
		  "2107A68E1A9CD60E63B847EA9C75EF14D19101E391D8F80D05F"
		  "6BF0966C949DA6D9C8E2C865C0468C825E29BFB9CA2070518AD"
		  "838A6B1530723F6261277EC7D1291104049503198B6C3B95D1D"
		  "0EFCE786D5982F6D0120D9AB2D1C0DD9EA89644103442A76E68"
		  "04A57B54FCA199E6D714576D631422BDA28C2A7F8838CEFF879"
		  "E0D3"));
  test_hash (&nettle_shake256, /* 123 octets */
	     SHEX("8AAF072FCE8A2D96BC10B3C91C809EE93072FB205CA7F10ABD82ECD82CF040B1BC49EA13D1857815C0E99781DE3ADBB5443CE1C897E55188CEAF221AA9681638DE05AE1B322938F46BCE51543B57ECDB4C266272259D1798DE13BE90E10EFEC2D07484D9B21A3870E2AA9E06C21AA2D0C9CF420080A80A91DEE16F"),
	     SHEX("F5D0630D6BD21787A4101E2287550ADED2C2BAEB0CA91D24CB6"
		  "1EACCF83B0D15700C6A95E643D8EA2383929D23DE5A1819EB75"
		  "71F38CE173F378AB7275776A771380443B0CC1CFE1B9EFBAA85"
		  "79FE6E002FE75B80FDC73AD20C727B003B6281A8BEFB7F7CB25"
		  "45F42586AB879C9FF523D6ACF70F24994342EBADBDA27D3168A"
		  "755C2DFC3C99360FE377C8A19F1465F6157CFF76A1D7C8A6FB9"
		  "91058D582D5F2E19ED583136320C4DA569D9EBA4692E0BB19E6"
		  "20C600260BFB0C95F7A8FA4560B1B715727CB04A8D37410DABE"
		  "2ADC5CAB24465BBFB41AEC1E946BE1C50ECA676F47EA99D217D"
		  "68F6B8F3536E879FDE16A355E71857143F7D07C9A207F4D61B8"
		  "9B82354F0A0273AECA1361C3F949EFC2A45BEC3933E8A6E7A86"
		  "E90E0EFE5D11F6A20ED9811A1CCA507E33CF47FEA26D7E67FA8"
		  "8D374245B1A24D8F5C584DAE932BE1EE25ECC5EE61D31111A3D"
		  "1EEBF220665EC401E28BB85B93D615B64EAD0FEF0E6AF725CA5"
		  "EA8454B571A784EF95A020251FE236952D130692D81702B16F8"
		  "DCF03CD82A809877B08774635A4859E1A516E9C4135FB415100"
		  "359CA6DA69426D4139E90F00EFCADD155F85A4AB1CE154F8CBC"
		  "84EFDFA7603FF48D5D1EE94A4D27DC9A450E35C60D9CF43594C"
		  "9182B69FE9DF540F498C5C92DB68065F2ECDC52EE7D22736561"
		  "FA617970375F6EA3D01FF39E2E22C979AB58DF80FFDB75B4258"
		  "F49A"));
  test_hash (&nettle_shake256, /* 124 octets */
	     SHEX("53F918FD00B1701BD504F8CDEA803ACCA21AC18C564AB90C2A17DA592C7D69688F6580575395551E8CD33E0FEF08CA6ED4588D4D140B3E44C032355DF1C531564D7F4835753344345A6781E11CD5E095B73DF5F82C8AE3AD00877936896671E947CC52E2B29DCD463D90A0C9929128DA222B5A211450BBC0E02448E2"),
	     SHEX("F3B73032FBC450B21DB7C6C263ABA819ACCB297B724CD95C552"
		  "80FB7F4E6A5100181FDD6E010964C62FC51C0FB540794CD3BAC"
		  "D695A082B20016E1BE1F8FD89E77AADA0747AFAD6ACB2F963BB"
		  "476F1B515CC0604CE5A7B18506BFEB875BF483189D268255146"
		  "FAB7B7820A26DFFE97C737A7D5BEE677524C4B64539CE760CB4"
		  "AB6D03FBA155D05130A50D5C1EA81873C8A45F1FB83C2EBC11A"
		  "2A1BDDD45AABBCC572D9E93611C4A1429B802EAF44EE19A4172"
		  "43C069C99825959093A9E0E5BFF186C757A7270E33D02B81768"
		  "D94014F17EFFAA3B2D92FEF370E552C8A8EF2B3557C2F930B7C"
		  "79E9F6C0573BFFD8585000954D9264611E3D6BB305B08973D05"
		  "83C7C2C5B2B0B5AA1B0ED46877FB02646472672121A60424194"
		  "81D4A0A5EE001E1E067F6A7EE1EA4B880D00F7815FA9658AD9D"
		  "D643139A37B3FEBB46585A2A74BE0EC2B8BCEBFD8F63672F2C1"
		  "F213511E611C677CD92B2D6C4E43792CE3709AE1CACA3103F7D"
		  "267940A0CA5868AFE6296458811BA67205A5BD4B27F96573F8B"
		  "88F8DF2733B83C420CB2015DBECC6E0146657CDAE3C7A2D2DE5"
		  "EAF8BC9EBD0A7FAB0FA37B24206B6118CC56087B0F573FBEE9B"
		  "D0C4B70512EA4784D85D8A89E75E08F86D204A4A03D4C247CBF"
		  "9AF7C20FD532A65D0C89311246CB572C77A35CBC9C65BE40FD7"
		  "EA2D41A3F5A4AEF0BE01C4DF6AEF0190171932B03EB960172A3"
		  "8101"));
  test_hash (&nettle_shake256, /* 125 octets */
	     SHEX("A64599B8A61B5CCEC9E67AED69447459C8DA3D1EC6C7C7C82A7428B9B584FA67E90F68E2C00FBBED4613666E5168DA4A16F395F7A3C3832B3B134BFC9CBAA95D2A0FE252F44AC6681EB6D40AB91C1D0282FED6701C57463D3C5F2BB8C6A7301FB4576AA3B5F15510DB8956FF77478C26A7C09BEA7B398CFC83503F538E"),
	     SHEX("FBBA34F9B8B88C9851E1B6A8613641A26E5E6944788449E7F78"
		  "C66F32AAB088229599E67B048058A2BB6713C84033A2177B563"
		  "407BF832D136FBA8A506B3E0C3AC8632F8C6DD058E1D242A247"
		  "54D64D8A0E8DB424000561EA5298EDF2E5620237F4297CD4B2E"
		  "48C87AF4ABB28167662B032F617D89C4CBBF064EF09AC2A1227"
		  "85D55C411970A610ADE46475CC0E371ACE344B3BEE172D73E9F"
		  "A4CFFD1806070E7CF48B722DCEC0756EB1EA059E8481DEF23BD"
		  "F94A0050CE2848AF21B49B59320D5F4EDFBF2C1A653306BE31F"
		  "3634E5871156C44D15F8C39C7F50FCEBB93C61DF9258F81941A"
		  "EE98C0AAE76D92560FC4987C0CA8CBF07CDD9B96E1F24E83F59"
		  "77A2C561870D6FD4752796900CC41AA4F2B256988403065A9C5"
		  "06FA78C50E0B5B9592A4116FDE380EE9E6C7FC67FFCB5123FCE"
		  "3B27A6D71EB875E88E0A8F20ACB50B3F2D46963913FCE830FB5"
		  "E01BAF0F2FEDFA8E9717576B96F788FF8F8734C33C3CD9595EC"
		  "2F6270BE920AA2A4FD56F83442141C1C1834820C1A41A51DA30"
		  "8F5CB8457E6C835C4F075706051A7CDB3901413FB9B42DD9E0A"
		  "2B8664BE7490CB4043ACAD67CC806886EA5C41BC8D81E00EA1C"
		  "C7AFB5BC34E150EC26E4D6D8D5F0C5A362A28C98ED6B3915A09"
		  "35D7A237BB24AA18D41DB5271BE154F4F51DC25AF823A978B8D"
		  "41F46AE142470ADC4D008B260928F7382BB08C320F4802705AF"
		  "6EED"));
  test_hash (&nettle_shake256, /* 126 octets */
	     SHEX("0E3AB0E054739B00CDB6A87BD12CAE024B54CB5E550E6C425360C2E87E59401F5EC24EF0314855F0F56C47695D56A7FB1417693AF2A1ED5291F2FEE95F75EED54A1B1C2E81226FBFF6F63ADE584911C71967A8EB70933BC3F5D15BC91B5C2644D9516D3C3A8C154EE48E118BD1442C043C7A0DBA5AC5B1D5360AAE5B9065"),
	     SHEX("DD4FC79F3B1D5201767CE9A0F89D3AD0A3949BD75668744B955"
		  "23562A5107CF45ADFB39B5999D79CD3CBDB1F622B2B250C8664"
		  "7A45419A243ADF4E7FBCFE227FFA2995BA39FDBC5BAE183B786"
		  "704D8631EFF1F88CCB8563F112440DC48D87B97BF8E8CD8B894"
		  "CEED6B43C857627AD1EFA81045CE505FEE7F370CFA478BCB4A4"
		  "174D133D9A7057702E7122229F53CE03D281771CFE58EBD825C"
		  "4285B4D7AAC7CB00A835FA094683F503E92B690A4A35D0E4F5F"
		  "01DEDE83FC829E4F3C5F783737A68DFC9EAA3CCE682E3D52786"
		  "75D17A2469FB0A814BF71FC212D7162ED80707AAA514CD5B682"
		  "B95B948A8A9D79EE3CE1B9AD44D3E6E6EB455CF198927072D8F"
		  "28950AAE0B5FDB628ED5280695E7A83665C94A23EBD16353679"
		  "7EE6733166AF631F21F6C1FF398D8D363D6F7ED518FEAC9EEA1"
		  "F98E9E3976706F2EBB3430AF64CB788A2461B736D91876FD481"
		  "F5F0135B458B57CC00959BCC1D145B2B776D7B5B1BD0ED154DF"
		  "B602F78C8D567F30ACF34341C9356BE887DC85E9415602E19DB"
		  "053E965601CE5DB866C1A9EDE8C12A81850F1A46CDC15411256"
		  "0A1E38C40A014E1109195FF23054954F463AA825FFE1A377D3C"
		  "B1DB279D2E5BC7FEA045CF3B2681170CF42D4C0518BC9E51C3F"
		  "854052263D31C50867426552E6CAEFE37267A7725A4460FE2A7"
		  "F2D951DDCF55A66AB59DE1A66A7FD15169BFEE7339428308C04"
		  "77B0"));
  test_hash (&nettle_shake256, /* 127 octets */
	     SHEX("A62FC595B4096E6336E53FCDFC8D1CC175D71DAC9D750A6133D23199EAAC288207944CEA6B16D27631915B4619F743DA2E30A0C00BBDB1BBB35AB852EF3B9AEC6B0A8DCC6E9E1ABAA3AD62AC0A6C5DE765DE2C3711B769E3FDE44A74016FFF82AC46FA8F1797D3B2A726B696E3DEA5530439ACEE3A45C2A51BC32DD055650B"),
	     SHEX("FE149A9560E60DD76DAC07CD65722403AC64637033653BD6005"
		  "3F3958D0DF1EF1DAEB343FF6488587F49945A19479820A98DF8"
		  "484CEC8E3608A8C7D15594F61FAF8547FB0BA643ABA3FE942E1"
		  "91D61E2A84A548A4B76E0144B71E061D01E5B46629DC585EDA2"
		  "1696B6F716ECD7D98E54FE49692B5FF7E74ED15183C4A4728A6"
		  "A4FDC85FA56D49615E0141C65614DE51776C73A46D19E2EDA71"
		  "226B5716FFD83C06813B631CDEF6BAC2B5F480CB59FB54DBD0E"
		  "0FFF0539FF70FC1F5872D8B78B3E03335675925E2A6FB7FDC93"
		  "78D3202F396F05DFCD0A2193FFA3DA954EB44486C054D7D8BC2"
		  "2194612047F2F8C5FD5423504D6A06F0FCD4E611880C74BC84D"
		  "7D5C3B66E017DCC79EF5EE41F9322F7DD863AA5428439D8E5ED"
		  "3222F2346E305759B46826E039FFA8A2B0C7147C2AA44060262"
		  "6FB79B0699ED15ED9CA54125603E9A9525DB238A518708E7870"
		  "9D5CEE0F3A196B4CAB52198ED5A58C98775EECBDE3EC721214D"
		  "B7A981C81D419F013F1DB230746BA5EBF30CE37882376DF8785"
		  "ADCA0E04E9E18D0EC75778369F7516F5215AE92D67C4C475109"
		  "0246A8F3A719BF83E3B310D61FADDB911B2080266D2B43F31FC"
		  "9E3A5D9747EEBFC8816F63E398C9F506C00821D0F1097349FEC"
		  "C2DC05FC420FE673F5398FFE828CB99B3EEEBE817EAB6208803"
		  "339D8D905C845FC934487F51FFCDB4E83D8E337DACBCDC57F86"
		  "7EBD"));
  test_hash (&nettle_shake256, /* 128 octets */
	     SHEX("2B6DB7CED8665EBE9DEB080295218426BDAA7C6DA9ADD2088932CDFFBAA1C14129BCCDD70F369EFB149285858D2B1D155D14DE2FDB680A8B027284055182A0CAE275234CC9C92863C1B4AB66F304CF0621CD54565F5BFF461D3B461BD40DF28198E3732501B4860EADD503D26D6E69338F4E0456E9E9BAF3D827AE685FB1D817"),
	     SHEX("E38785ED93686FA741FBB6E5BE64933963C3C872F7A4E8C8D54"
		  "0EC3F82284605625D32A24BCE8B40264EB51B164DD86F318ACF"
		  "D9867F3BF23998275042AAF23BDA01F602622448957B81E5147"
		  "5F15CDB31A9297EE390F681E460EC01A024F183110C728BB09A"
		  "12DDE89E6F5DE2B4F8C17D981E3E9C531E6D4C19448A4A6BE28"
		  "853AFA2FBA216F7C8E2CE4E4DE31F6B0E129CB5DA118B4A5956"
		  "9A439C1095EB50C51CD83105B1A12B3F7086947EA7381969A78"
		  "308F8FDDAC8B2D87F4540A8E7CAC8932CA76201F86561ADD09D"
		  "833361851CEB9759AD1EEED4E00ED19C642BC6D0AED034276B6"
		  "6D818E8791A7C1F42F8604E8B026D4635E3BDB27CA0FB28E751"
		  "7BF662BB99AE3A3C698AD0AA2F02408E76A5F93ABFC933B1691"
		  "A89EE3EBCA2885EA633D0FEA4DBCD03B05B68E0F4D267144FDC"
		  "0898DE46011832ADC872F4A7F0D8933CDD482CA54FFA81774BA"
		  "083D3D9FE07DE94F8E03FF66D9F1BFE7504E8A497591BA8F527"
		  "58F92E0A4CA6A93979CD1F55EE9DBA17BAC6C69E83DDED4CE2D"
		  "BFFB0B48C0C6AED657405DE18E7891B3C854127459E89FE5442"
		  "CA6D5C223B046147BFDEE435CF4EFAEFD705373DC87E20B7D31"
		  "C7D37907D30B8B32054BB9ECA80D0AFABB5EC5053D94D517F15"
		  "8EA958C7458CF71FCA85B607A352A9A3B8F4F161E3B8853F6BB"
		  "61864BAD12DD9B57ECDA507A3CAA63A1DEC726B518CB4FDDEED"
		  "6A34"));
  test_hash (&nettle_shake256, /* 129 octets */
	     SHEX("10DB509B2CDCABA6C062AE33BE48116A29EB18E390E1BBADA5CA0A2718AFBCD23431440106594893043CC7F2625281BF7DE2655880966A23705F0C5155C2F5CCA9F2C2142E96D0A2E763B70686CD421B5DB812DACED0C6D65035FDE558E94F26B3E6DDE5BD13980CC80292B723013BD033284584BFF27657871B0CF07A849F4AE2"),
	     SHEX("66FB94B62645F96FD55EA261E08A93AABE783F160AAFFE07253"
		  "A1C3D45DC65F6AFDE9DF121A894A1A13CED4E5D492675BDA359"
		  "26A1B147D5317D3CC09D4D34EBC1BB223663B349D3F66F05DDE"
		  "7A66616E7F89B0BC6859A3EB8402B881890B21979BE6F60CFDB"
		  "9F6C2FDE570D79CA9D24099820E433DD995E4C2271EB269937C"
		  "BB96839A03BA7AF726ADF23B7FA2F86348C0489E09A559D0FEB"
		  "FFCBF33D77CED28A439F09E2EFAA7D6334A783075B588134C69"
		  "433AEF0F8665C05E97EC2AFE49336EE2CEC0724842A64123C7F"
		  "43689D1EE88873F89AAAE9730B1A29768BE0E49A87B52C1F5AD"
		  "08D700EF9B57A20F5DCC7C238A5E56CEA3A0EE1F9BDE550F279"
		  "2741F607E81744855A5F109FE91887BC580199966DFE28A30FD"
		  "1FBEA9AC4259E7BDF7135E4A9D83234F9D7ABE3BFB1F264E23A"
		  "67F456096E27F540FBBD5DF0E8DDB5A6A455ECE063D4D528C25"
		  "82BE8F111E1F7C7F204B82D40F60C2AF6099DBDFD64DF85899B"
		  "A2A02A26870E3FECA6C07E99AD43F93D21DC275E1AFD1E136D2"
		  "F49142B4D208EC865F91DDBA5822148C6884B7CB283DE5AACC4"
		  "E8BB66BE3B08804246C808569B2DF0AEB08BD4C255AE1829AA6"
		  "2AE9495A89D238DD93E2BDF5D14159E48510FC82B572402302C"
		  "63956CD215341A1D367135623C644094CD845B08ABC7A8CBD4F"
		  "3B66F48375DF7155BC5A781E69272EC1B3AE4E3CFA1D8D39BF0"
		  "B4B1"));
  test_hash (&nettle_shake256, /* 130 octets */
	     SHEX("9334DE60C997BDA6086101A6314F64E4458F5FF9450C509DF006E8C547983C651CA97879175AABA0C539E82D05C1E02C480975CBB30118121061B1EBAC4F8D9A3781E2DB6B18042E01ECF9017A64A0E57447EC7FCBE6A7F82585F7403EE2223D52D37B4BF426428613D6B4257980972A0ACAB508A7620C1CB28EB4E9D30FC41361EC"),
	     SHEX("295E0FBF3C5B510D6B244381BEDF20F8BC3D915CFA4BF9FC42D"
		  "BF6932F44DCA91744D812CB9F639A9C3D376C49CE3C7DA90D26"
		  "7D794805A61D20D00C79F42C37BE018C67857B6A02E46B6D6A1"
		  "47506210651E31B280043610584A1440A07BDC8540AC46E637F"
		  "7D947792A65ADCE3B790636FE1794E90598CA6340982EC58F3B"
		  "24AE1C703FBB91412F1AE7C23A3CAF78C4BBBF32C90C365726E"
		  "D4832016C5A19E94685A7CCEE376184A10BEED6DA7E26D1A426"
		  "0EFFB7E9F7A0A0C71D7F164CC967C7526A0F1EDF1FF54AEB114"
		  "109F7B34361DB5AAEF9EAD31A4D4896CB86556D2619DE0ACE07"
		  "C133B14006119BD4BC8CF4F8EC09CD2ED91B4E524012261EC95"
		  "37B3BC4C67100102293836E22AB5524C979F1C1AF51FBAA2ABF"
		  "3F0EFE6818BDAF37CEC4D6CDDC98A994BFF1AC458637D528BA5"
		  "F701DE0BF4E4E9E691149A6CD4677BBD9821511DDFA6561C419"
		  "B473D2BA019CF1B83B7AAA3BBEE141E2FA38F032635AA559CA6"
		  "29068DD4639CCF579118496A903BA7DDE88FF54806AE89BA397"
		  "4B9DC0AC520241BEE278E3EBA1B3DF1C026A8E5F0D5F28C35C5"
		  "B04145B570D2F06D25F45BEBF4DE2785D75B0BC9F1E6F9AEBE2"
		  "AE510D422180CEF13F6B0C8A0CAF76609A158FB6F3B17E7282E"
		  "168DCBA19E76DB7841E384B54F3AB29D78C86861560CDD934CD"
		  "B63742933253D7BB82B453CF829F33B99B8E56B906C86131BD6"
		  "4F64"));
  test_hash (&nettle_shake256, /* 131 octets */
	     SHEX("E88AB086891693AA535CEB20E64C7AB97C7DD3548F3786339897A5F0C39031549CA870166E477743CCFBE016B4428D89738E426F5FFE81626137F17AECFF61B72DBEE2DC20961880CFE281DFAB5EE38B1921881450E16032DE5E4D55AD8D4FCA609721B0692BAC79BE5A06E177FE8C80C0C83519FB3347DE9F43D5561CB8107B9B5EDC"),
	     SHEX("9E8438CFFC23A52D75838690E7074D2C713B1188375D1A77F8F"
		  "C8C24A4B89CA5979F58692D90DB14DFBFCAA5EF0AC0C2D16F3E"
		  "84E6F771ADC26FA9081F3B10E02A3EE1A3DE40DB5DE17F7628B"
		  "A3E20E5563BAD66BC32BBFB1CEBA7598AA4480C86A0C4ED1036"
		  "08CCEB103AA13182F48D9FB363913000BA6559C855652B4EC29"
		  "D06BC99148569B66C77F58A757B75C3BD5EF2FF832FAE020EC1"
		  "C9E19C1117A107F0529EA66F0607A499DB0C32966A95D882699"
		  "B96C4949912380150F951991AE7768E5A26267A8A43CA76AE9C"
		  "5EDF31D7C52F2B09C9EB15E1CF47F85B20BD55CA47FEC048B8D"
		  "F899A08DBE2E9C2F72BFDF3BB803AF10B5A1BBCD7B4E1915E09"
		  "9C22307154012C67A39FA77720D7006B86A1ED4F095B6644AF9"
		  "72CED451453B3DE57B1FAD8B4280AC62E1ED09E219873B92BFB"
		  "50646343C46CD3C42E84E9C7F05F1AA5AFD583F685A20F996B7"
		  "C0683002691CD28D454F71334C2387DFC43D32E633900699C8F"
		  "C5E3A3F6494CA0BBA96EF10B7F367DD31623DDC14E9907F9FFC"
		  "2C65DA666D06987DB1DF8ADCE1C74BDA7BC6C6D5263E3CD777A"
		  "78B9020DB37D331A16C38D7419EC5036FD95B896373461E6B83"
		  "84B2ACF96E46C7C9AB6D43C6EEEBF62B7BE5F95F8AB6B2AA5E0"
		  "5D66BC4DF8A2FA1AD9C3CD9362CD183C10C8CCE216FE48AAB2B"
		  "1C4F6FFA39CA9F8D7B14AC5BBD8C71F55776F6C0EA8583ACCD1"
		  "956A"));
  test_hash (&nettle_shake256, /* 132 octets */
	     SHEX("FD19E01A83EB6EC810B94582CB8FBFA2FCB992B53684FB748D2264F020D3B960CB1D6B8C348C2B54A9FCEA72330C2AAA9A24ECDB00C436ABC702361A82BB8828B85369B8C72ECE0082FE06557163899C2A0EFA466C33C04343A839417057399A63A3929BE1EE4805D6CE3E5D0D0967FE9004696A5663F4CAC9179006A2CEB75542D75D68"),
	     SHEX("07024716B8E95D104783A37F82E66D4F5B0BAE8897F1EF6C45D"
		  "ABF3A59253261300371A4FD20601503F6D187CED1067AECB11B"
		  "4A4B5A31FDBDF4A2B1BA9C0EEB0C2BE7730178FE435905C1B80"
		  "AFD9CD27BCC631C57A10F3BF3F285BB80D9C4F1C70ED3099694"
		  "1D1ED225386B7F4777C18A01EFB5F1086BE223EFDD082E01755"
		  "ED517AD6A3E423E924147761C95B74F6BD338F1DC0E53715AA3"
		  "1BAC84F7E5BEDDCAF56574626B23DC247D0AEE8ED77760F53E7"
		  "FFE467564AA3159AA300AD159159259DC3252D95942215C81ED"
		  "5FAFE0CB065758F72366E50DF7FE6F2B41E36371408503C5E23"
		  "A4CA0C93460C0ED713872F35CC71D1B71A34A80210069B12CCA"
		  "79CBBB04836E6D9A15E78A638B0879F74034339BB4FB39EED0C"
		  "8D6684A67DD89B201B46010FA73CE5FBAEE87FD89D4B5C40F04"
		  "D9CEC8510BDC3D07539C82031FE9734826CF36BB41B30426D32"
		  "A10225209C24B631A071FEB2A9B834810E81B6F87A6128F6781"
		  "5E4AF6A6582EA36D9BE5DCBB1D7FD0BDA2D583D1F6BC2AED7A5"
		  "34C33A8D183D5EDF42F3080BD07CE9D78647DF008D417D0DED5"
		  "021D7B6996D6B541DE75DD175E820728037B531521851C9743E"
		  "D03F8DAF292846B2EC019238655256EE960E0981DB7B65A3692"
		  "B50F62033FD79A29F4A36C27BBE7A3B7BD1E813F3A2EDA1CEBB"
		  "29979D91720CCD125D0C43D708892BC38D0D297047637447A19"
		  "CB09"));
  test_hash (&nettle_shake256, /* 133 octets */
	     SHEX("59AE20B6F7E0B3C7A989AFB28324A40FCA25D8651CF1F46AE383EF6D8441587AA1C04C3E3BF88E8131CE6145CFB8973D961E8432B202FA5AF3E09D625FAAD825BC19DA9B5C6C20D02ABDA2FCC58B5BD3FE507BF201263F30543819510C12BC23E2DDB4F711D087A86EDB1B355313363A2DE996B891025E147036087401CCF3CA7815BF3C49"),
	     SHEX("585F27AF67FEE2446767B05CE40627BB06D403430D94A62719D"
		  "AE2864CD44E7A8EEA340B98DAD0520D8C16C1BB1AB562C0FDCF"
		  "E6F1DE4F8A0601EB9A73FFDBAB82444E77487213D17BA31E2BC"
		  "39FEC719979080A501C8E15932AD588000959A18447B98735C7"
		  "3A2A8F4A949A8FF187520E6A9901647F47DB442BFB2772E8E8E"
		  "22FD759F88722A98D24D01308B15A0BC715ABE9568A34201B86"
		  "75204BF68AFEAE8CB063D6CC9EA68AF9E4CD692F9718FB5EAB2"
		  "70A2B74AB623B521B4B085700CACE113168F56A994A639517F3"
		  "6E3165029BF100A6F19162A8696B858A0B0716751430DB7BF07"
		  "480427C2CDA01070F5E78E3DED7471874DDB05D2944BFB04487"
		  "629A98E06CC4C984AE8C5F7695248D01CEE11D75A9C7F545CF8"
		  "6BA745B8AD5564F33B0A8B72E78954A7FB253C07EC8FB3D3BDE"
		  "29DDD91DBBD94D332FE9B7438937018E8243C2D2C30657959EA"
		  "4D23A319854FB2CB994CED4286836E83324FEA54C58DB55A969"
		  "165A1755D965C416E1D67468ABC701925F4A80BBD9AA9DA82D1"
		  "48A26158712C2B9830804A4CD1CA14A2B246DFE8E8A12EAC9C1"
		  "07E8CBBF34EF9517674350973E75799A251E72A352D2700291C"
		  "C3F1AAD40ADC9050BEF9881170EACD3207AC0C573E238B65500"
		  "B997DCEB347B94FCD5069CF43BF15AEEF15E4C135DD21B4046B"
		  "201F411513AEA9629B9887D21CDC43768FC1C1B12EA4CA4595A"
		  "D463"));
  test_hash (&nettle_shake256, /* 134 octets */
	     SHEX("77EE804B9F3295AB2362798B72B0A1B2D3291DCEB8139896355830F34B3B328561531F8079B79A6E9980705150866402FDC176C05897E359A6CB1A7AB067383EB497182A7E5AEF7038E4C96D133B2782917417E391535B5E1B51F47D8ED7E4D4025FE98DC87B9C1622614BFF3D1029E68E372DE719803857CA52067CDDAAD958951CB2068CC6"),
	     SHEX("E57DD7DF74504F52D00B8079631F542A532610D77DDB4BFF891"
		  "5BDFBC9B8F25352AA3D9A2D00CA2E14C77FF661B3AA44E30BCD"
		  "3D4B110A8CEB65E150DFD9109EE948FFF11CD01F6A98C0100D2"
		  "323F52E323E32E9FE7BBA6D4411EEE3CCF70122DA5BFEC8B92E"
		  "72119D4B1CB1A1626DC0CB79FB4E4C938A4A7CAA7962B04A112"
		  "9B4A0A9C457FF38E8D3379044E4B65B82336709B95401CD0CA6"
		  "4C10F5D22A226678A1EA4E2C54509085ABDC41AEEAD761C7946"
		  "F33140AB775D61F7B001F141D00F5209081B11FC2ACE5D9143F"
		  "BCB710176BCC8AEE25D0182461AC1A446F3A94788B3513B88DD"
		  "DF5DB0BEEFCB331CD2553416A3606F44B5CF36318C72501FA20"
		  "07F76BA506EC510508B27D692CFDDEBF075B39E531BA243F3E2"
		  "71B5F70EA6B58AFEAA0B8264202BA2C430C33D1409693AC5251"
		  "9F44B001ABFA9C5595C0726DD1ED36C8B7CD2F29FAA912E2613"
		  "D1C51E2D0C6F8B27FBCDD6137307ABDFC54C0E6C6FCE0630386"
		  "2A8EFECEE12EB87CBF8423BFEF01C6FB87225DC467CA7A0B024"
		  "3CA9D531811A5E5090351313FA6F73D413755551F6E7187057F"
		  "DF4CB58A1B3B3A63F28A17339A835A73BF21EC013EF4B2BF89C"
		  "B3BF1369A87239A546A6816D8B06EC5B1D5A179E67A86665EEF"
		  "CF439D667F5F9890DA8B9E1E7E1AD08F757CB0E3EC6D8CB8FD4"
		  "A140846E3768076D342FDF0BE4CB135D68FAF5C93FAFBA48137"
		  "86D1"));
  test_hash (&nettle_shake256, /* 135 octets */
	     SHEX("B771D5CEF5D1A41A93D15643D7181D2A2EF0A8E84D91812F20ED21F147BEF732BF3A60EF4067C3734B85BC8CD471780F10DC9E8291B58339A677B960218F71E793F2797AEA349406512829065D37BB55EA796FA4F56FD8896B49B2CD19B43215AD967C712B24E5032D065232E02C127409D2ED4146B9D75D763D52DB98D949D3B0FED6A8052FBB"),
	     SHEX("6C60955DCB8A663B6DC7F5EF7E069CA8FE3DA99A66DF6596925"
		  "D557FED91F47091407D6FDE32023B57E2EE4C6AC97B077624FA"
		  "C25F6E13F4191696B40A4DF75F61CD5521D982C6D09D8342C17"
		  "A366EC6346E3528B26CFF915BE9442B9EBCC30FF2F6ADD0E82B"
		  "A904C73700CC99ACFF480CAF0487CEE54CBA3753B6A5DD6F0DF"
		  "E6571F0115E8737B071031023B6BB0D79864C3F33162E78269C"
		  "EE23FCE47B91B4FDF91F98464A1D21E799D17F76C1BB807DEE6"
		  "67B0B273054BE298299BD12B7A80FB354CE3E6D1ACF98443879"
		  "A554ECA6B96DF061D04A117C98AEEC1CDE1AFA9CEF62DD686DA"
		  "91BB2B1F12379BBDC9FA32A6B6998B77E8EB0B50507862AFA77"
		  "99D018E272091F51CADD81ADB587EF67BA67618C45D1F3D559D"
		  "BD299ABC26EC712DA8FA34BA33BFF400D1F0F8B6345CF57269B"
		  "858578C0072A91A63EF85F9D378900CD1A55D2BD4630DB829EB"
		  "484D89CE7A414ACA173C52534AD5F9355E80E395E79156D751A"
		  "930F7F8B5D9F4D5A2C9A753723083C5E8EC6CB24D8EF93C8FEF"
		  "2D1BE4ECA222C6E6C2ACFD684893CEA65CBF5B096B3D8660071"
		  "36126A33EF496BF2310F293BFA4C93AB826821E2B93259C464E"
		  "0AEB06D6DF8FFA30B1C1E7E384C7E427A2BA3D99FF8A666380C"
		  "5C1B678F742C57B0C3B08849FD65300DF13499DD894EFC33116"
		  "E7D0774064331FDD407487417D13BBA4285299AF650D3065D95"
		  "1131"));
  test_hash (&nettle_shake256, /* 136 octets */
	     SHEX("B32D95B0B9AAD2A8816DE6D06D1F86008505BD8C14124F6E9A163B5A2ADE55F835D0EC3880EF50700D3B25E42CC0AF050CCD1BE5E555B23087E04D7BF9813622780C7313A1954F8740B6EE2D3F71F768DD417F520482BD3A08D4F222B4EE9DBD015447B33507DD50F3AB4247C5DE9A8ABD62A8DECEA01E3B87C8B927F5B08BEB37674C6F8E380C04"),
	     SHEX("CC2EAA04EEF8479CDAE8566EB8FFA1100A407995BF999AE97ED"
		  "E526681DC3490616F28442D20DA92124CE081588B81491AEDF6"
		  "5CAAF0D27E82A4B0E1D1CAB23833328F1B8DA430C8A08766A86"
		  "370FA848A79B5998DB3CFFD057B96E1E2EE0EF229ECA133C155"
		  "48F9839902043730E44BC52C39FADC1DDEEAD95F9939F220CA3"
		  "00661540DF7EDD9AF378A5D4A19B2B93E6C78F49C353343A0B5"
		  "F119132B5312D004831D01769A316D2F51BF64CCB20A21C2CF7"
		  "AC8FB6F6E90706126BDAE0611DD13962E8B53D6EAE26C7B0D25"
		  "51DAF6248E9D65817382B04D23392D108E4D3443DE5ADC7273C"
		  "721A8F8320ECFE8177AC067CA8A50169A6E73000EBCDC1E4EE6"
		  "339FC867C3D7AEAB84146398D7BADE121D1989FA457335564E9"
		  "75770A3A00259CA08706108261AA2D34DE00F8CAC7D45D35E5A"
		  "A63EA69E1D1A2F7DAB3900D51E0BC65348A25554007039A52C3"
		  "C309980D17CAD20F1156310A39CD393760CFE58F6F8ADE42131"
		  "288280A35E1DB8708183B91CFAF5827E96B0F774C45093B417A"
		  "FF9DD6417E59964A01BD2A612FFCFBA18A0F193DB297B9A6CC1"
		  "D270D97AAE8F8A3A6B26695AB66431C202E139D63DD3A247786"
		  "76CEFE3E21B02EC4E8F5CFD66587A12B44078FCD39EEE44BBEF"
		  "4A949A63C0DFD58CF2FB2CD5F002E2B0219266CFC031817486D"
		  "E70B4285A8A70F3D38A61D3155D99AAF4C25390D73645AB3E8D"
		  "80F0"));
  test_hash (&nettle_shake256, /* 137 octets */
	     SHEX("04410E31082A47584B406F051398A6ABE74E4DA59BB6F85E6B49E8A1F7F2CA00DFBA5462C2CD2BFDE8B64FB21D70C083F11318B56A52D03B81CAC5EEC29EB31BD0078B6156786DA3D6D8C33098C5C47BB67AC64DB14165AF65B44544D806DDE5F487D5373C7F9792C299E9686B7E5821E7C8E2458315B996B5677D926DAC57B3F22DA873C601016A0D"),
	     SHEX("1279A95CB87B1CDF2F8A8A47CE55F0DA3766E35EC58CF921E07"
		  "B43B176CFD6CE8508A208C9D6F5412515E0235C95397A47D2A4"
		  "B13A357DB4882F69B9C910C985A5F82187BFCC46BD48CDDDA7F"
		  "65C3E95481A37202EFF9C116BB3F784BD46574FBD49E19B45E5"
		  "E2D18F57BE7DAC826A447ECA6E2A6BB44B0061930DF56864C37"
		  "8E020A183DEEE8445648EC2F95EE5F09CFB196E3D8090535664"
		  "46FCA6BC36896215BCE115B0AEE55737A4421316D2058F24C36"
		  "D46279B458E901D3A8062300024068D99D8C1B8BFB6F3E5883B"
		  "FEF3C1AED55989151C2CAD1EB940CC82398DEA1E5A92351F0D5"
		  "AA7D47E16A949F396194EAE2EBEA1FB731BEC12D2734C2F1E74"
		  "64CA2FE27F036BFB28A32A9657C75EAEE79F86F2CE5EFF1AADB"
		  "68DA0B32A4BF88A37F1D66D13DF4ECE65059BD4ABF91A3EBF98"
		  "2A1F5E9A6DA639623D0C8E5FC5C0C8071965221C4B79CDE7D44"
		  "FC258F20CABE3C38862851952741FC9E9E87C06AB0CF8B8FED6"
		  "C18666C5C70EA25973FED36D90429C54B157174A2583E142E26"
		  "F2ED492A9FA74F1985FE52A421C2F97F94B73EC7D881F0D0B0F"
		  "930461FB896B1806C704307CEF682834CB583B6E996BD31A6F1"
		  "D8586D416FD8C91EBA59935FEB12A1E77D0F3E05F80842B14F3"
		  "4F27ADD947EE3CA2E54BBE018FB8CA27FFD1C24227572CC277F"
		  "723535CBA0133E359217D5522676485181ABBEFDBC31C8164F1"
		  "7847"));
  test_hash (&nettle_shake256, /* 138 octets */
	     SHEX("8B81E9BADDE026F14D95C019977024C9E13DB7A5CD21F9E9FC491D716164BBACDC7060D882615D411438AEA056C340CDF977788F6E17D118DE55026855F93270472D1FD18B9E7E812BAE107E0DFDE7063301B71F6CFE4E225CAB3B232905A56E994F08EE2891BA922D49C3DAFEB75F7C69750CB67D822C96176C46BD8A29F1701373FB09A1A6E3C7158F"),
	     SHEX("7A2DFE06B3AFA54B49414379BB9C7A80EFBC8D0630B8CA170F6"
		  "B22CF65BFF168F2BA6E32806E6E7D9D366D53A319BC2C8B8AAC"
		  "7CF8158128617229D76AE594AD71C0E0541B86078A62D71E299"
		  "2E555EE6FBEA906CCFE58FDC7101964EE8C3C05D0C2C7D8A01B"
		  "7E1629BCF62233C0487E36E1C25474E129B72F1F9BAFD6E0F7C"
		  "9FDE8DD44DDC1DD9668840294C5B39C408AA1BD1395D1F4A236"
		  "8D9D1E5168093A63732545FE594B32EE91F89B297E3A33F53BE"
		  "9A1A00642C17DA7061C10D1525F34180D04C78BABF6B1C866A8"
		  "AC1D19FC17C24F13CA4032D650DDA748E2A09D3D97A2125D52E"
		  "670C1DA4AFE56E60DD789608F50E914317EBDAF125BCCA849CE"
		  "37F38EEFE8998BB2B1B10CC1B28AEAD1FC48FBF1C95B2A0F063"
		  "7609CD569050BB5F3711024CDE92BBEFF1C73C9B6C8DBA71B2A"
		  "A238AA26B4007978EFB346D3FE0031C391A70F362C4C5FE1DA2"
		  "D3C73E4BC52A400E5998660D8C127C46CC7FD589DB2C952D2F0"
		  "77E4001B2B6B46EE5E56A44578B4B8DC1AFA33C3994CE7CE49B"
		  "4D8181195076B938BF605F61A74F9C505C64C2A75601E7EC3A9"
		  "4B22157308BBD481553ABFD8ED5575B828CF247F6D321FF25FB"
		  "AA245F9B30B39171E1CFF354B9F20D55196EF66D32C4693FDF2"
		  "49F528C86EB82779B5FEA03F8A95F31DC0579B1CBF178B23B27"
		  "6E5618941760B117AF65B83374C919D6D423B575C5C459B5AAD"
		  "6A2D"));
  test_hash (&nettle_shake256, /* 139 octets */
	     SHEX("FA6EED24DA6666A22208146B19A532C2EC9BA94F09F1DEF1E7FC13C399A48E41ACC2A589D099276296348F396253B57CB0E40291BD282773656B6E0D8BEA1CDA084A3738816A840485FCF3FB307F777FA5FEAC48695C2AF4769720258C77943FB4556C362D9CBA8BF103AEB9034BAA8EA8BFB9C4F8E6742CE0D52C49EA8E974F339612E830E9E7A9C29065"),
	     SHEX("FBA46834A6995933AADDF85B59F60EB3B77B5AD38BC21A86207"
		  "9D8D2227CB9B3CE3ECE89D90C70B990F7507D5C3A565974446B"
		  "54D43288ADCCE2BFBA8914EB498DC6001BA169417AF6DA3ACF6"
		  "85DCBE3CFA0C59963F0F0AB94D39F05A69D5DFD320EF409DE76"
		  "FF85E01636F57DB9B2DF436D94EC2845F0C0D1959DB6308941E"
		  "40C946F4C3B3277A065847C9E3CCDCF113DEC5DCBEF5AAA1C5B"
		  "91C19E05EEB66F6E698178B7B02FE00B89DCC98526FA0E7B2C6"
		  "247E1C175A21EA34F5260B0C0A625D7AEE1680B20B4667B44F7"
		  "2F1353BC71DF6CCA4346EB54306DE3C380438A9B8E6CC47E6D1"
		  "8D84FBECBF9A6132F63C191635999105967C2E171B74615F456"
		  "D2E111E7FEE5DF00217B49BB49270BEFE5BDFCE1486C13C6A94"
		  "E31A20C1D377C410324C08A17C2D04710CA267E85A2CC1E17AD"
		  "A367C0F65AC3A70AA5FE14378371132026169657D1BC1B84C19"
		  "08AB46689494C3E8A34835F4DC43D19E9328ED0A16341E5E9AB"
		  "B80BDDEF1BF5FEB5DDEA560253532A6607E2E128480DEE2083A"
		  "F54C67E971CC6B56383A564B0BBD2D802C6FAA7BC62629595B1"
		  "7D04C3AE20F73B35A9AE4D356D401345E0A0D5174BC457256DB"
		  "BD844F9A4BFCE0DD48F35C5F8A915E61A23899C40FA63A51F0D"
		  "1160E16260BF4DA1902AF01C67E8997B2AAFE1989D013846CF9"
		  "85164C03418930E61FD06F9D30F06897460DFA1987D4B5D73B6"
		  "8CAF"));
  test_hash (&nettle_shake256, /* 140 octets */
	     SHEX("9BB4AF1B4F09C071CE3CAFA92E4EB73CE8A6F5D82A85733440368DEE4EB1CBC7B55AC150773B6FE47DBE036C45582ED67E23F4C74585DAB509DF1B83610564545642B2B1EC463E18048FC23477C6B2AA035594ECD33791AF6AF4CBC2A1166ABA8D628C57E707F0B0E8707CAF91CD44BDB915E0296E0190D56D33D8DDE10B5B60377838973C1D943C22ED335E"),
	     SHEX("4B60950081987C82C2F1A1BB9EBAA9A72E125557E296BCE9895"
		  "561C6A959371EB1AC9A13914F4419B4A0A1048B3D422B53261B"
		  "0AC26EB9852E9933251E154996DEC6219A7063D87228A8D72F1"
		  "3FE94A0E75485C8F23DB2341EE7439FAF87A1B359CEAFE42903"
		  "19F4705B49CBA0ADC24DB4FE8800AE05664C544D53EE119CF4F"
		  "1572DAA355E48DB605695BDAE5BF22CFFB60C019934F2AA9529"
		  "18B388EDD9C53DC8FAC2186CA0F77AC6543AC379B1ACEE9913C"
		  "D75CA8E0F9D89616383C9541603430AA548E9A9DD8F90F64263"
		  "FE42A9B8F66991CB2B8B45AAF683F51D194A01098A10D4BA110"
		  "7C56173825AFD79EE0A334CE739915D4F046C814412DF526ADF"
		  "F4F23811BF1C6A254372E8E46F0008F6332637980417043BAB8"
		  "FC21E3715B713AB4BD1DDB3D50FB9D5D6AEBF94FFDA51D3BD4E"
		  "25C82B295BADC7F8F11CA787DCA045DE2B52A79226ABB945F69"
		  "732A1D0DA3A3228D44D8F43828D8958E621851F4824590C0719"
		  "982CDBB3602465C28A97CD4CFA33D6EBA65A3F6428ACD37D669"
		  "67B8C996969D866A02FF6EFC1B6240AB557321260EB4C9D233F"
		  "F4E1C293ECCA825DAC943DD1B9A624AC37ACD1A818D2F74AB99"
		  "B2883F0837F77D4D00A52DE0924EF55C7536916677D2AB4FAA8"
		  "F19E8DCE97C05690570FF0117C32A3F500C2FE817AD651DF2E3"
		  "C28F5E867083F607E8D0509745588ADD19F1BB7482087A50821"
		  "7DAA"));
  test_hash (&nettle_shake256, /* 141 octets */
	     SHEX("2167F02118CC62043E9091A647CADBED95611A521FE0D64E8518F16C808AB297725598AE296880A773607A798F7C3CFCE80D251EBEC6885015F9ABF7EAABAE46798F82CB5926DE5C23F44A3F9F9534B3C6F405B5364C2F8A8BDC5CA49C749BED8CE4BA48897062AE8424CA6DDE5F55C0E42A95D1E292CA54FB46A84FBC9CD87F2D0C9E7448DE3043AE22FDD229"),
	     SHEX("5CBBC93FED206474A11100E1973984D5BC5D06AD92996FC8928"
		  "C1E6BFDCFAD5E32665F4743EAE8FD28289DD226A0136ABBD5ED"
		  "BA5016E975381D4C3CC95065FA4CDAC2336B39342BE8070AB2B"
		  "59DBABDC77B8742E4B3C86F1EE9ED0C2B5030FE1ADBE28F8271"
		  "11FB137C180B9994FE150B1F5144081AD59131A6017CEB8FB07"
		  "15DD2DE6B3BBECD2FDA4C3E9E8A652B5DAB4D0B39CAF6841383"
		  "8CB1960241AF59A46C8B5FCA40552693CEDA564862DF42C5827"
		  "EB371311FEF9231FE1B23ED07F72DB6410172597EAFB5D077AE"
		  "D168197F20C3B01CC3C52A94546B722B4AB56211B48192F908E"
		  "7E7A7B7020A8F6370A0B73E046E3211DF3352DEB1D59E4940E4"
		  "5129A6667F160D2ED6EB703EF962683540E3A9C7C94A61875CA"
		  "C09C4E22ECCED9E2CA6DC519698042F67AA518EAF3D338C5ACC"
		  "D5C2DE5C3DF6B4A2273A0FB8880CF3F81D046AD8B24E73A8F3B"
		  "D7CA65CD21AC2D950A7E36D0A25362788F1AA26B71CA434AD14"
		  "EF3B002BAF6327D4A5293D91CB838E4212949675DF185B70D8B"
		  "42C0C01F289478F0A9FF736E6838EEC5E54413CA6F7F22F51AA"
		  "DA7F3E9C69B9009599BC437C0D806BAEDE6DC1B2FB9BBBA43FC"
		  "3BFB5CFB408A9D8EC1C1DD1528CC2AAF144FBF9079F86577A4A"
		  "042A085975BB1CB5DDBCD728DF6BA69CB23A8BDA0368924466A"
		  "89DA791D4328DC55800FE8639987EFC172BB29836A901B3F4F3"
		  "FE9F"));
  test_hash (&nettle_shake256, /* 142 octets */
	     SHEX("94B7FA0BC1C44E949B1D7617D31B4720CBE7CA57C6FA4F4094D4761567E389ECC64F6968E4064DF70DF836A47D0C713336B5028B35930D29EB7A7F9A5AF9AD5CF441745BAEC9BB014CEEFF5A41BA5C1CE085FEB980BAB9CF79F2158E03EF7E63E29C38D7816A84D4F71E0F548B7FC316085AE38A060FF9B8DEC36F91AD9EBC0A5B6C338CBB8F6659D342A24368CF"),
	     SHEX("CA9CF74ADED61A4BC20BA078E8053702BC1A444CEF69AB058B9"
		  "C0F586286C138CF02AA36CB69E0CBCC52F455623D27B9597C93"
		  "68EA5DAEC4278E75AD5E3FBCD480DDF8EF39B3D10808D202470"
		  "D18118A215838A0F6C547D324FAAB761F7C8CEF462BE150DD49"
		  "A26DF4E453F997C31FCD27280E846B15F640FF8C39BE270DF9E"
		  "44CE57CA82F6F6762777089276A3E67D8EAB0E4A0C2E1946FB0"
		  "8B3FDD3FAE438CC2E9EC8BC1754275FCF4060E31428B9F08E95"
		  "3F9FB4F80560F2040E4B6CC7BFD6C278FF01ABA41F43549DC0A"
		  "A0407E5C8346543B1F1C24D1E05475327A48821C1C1058850C1"
		  "80776EAD66996DD5A02865702E0E4705E8E2E7D4D25F058528A"
		  "C980EB831D6B70E7CC06EDA314D80E1433B2A63EEC405BEE2AC"
		  "B18CAC238ABDF254DDBBE72640C8C609F70D04C77D2BE9190F7"
		  "8C8E448035A9206841C55CF9152AE5F7CACD798AC696299551A"
		  "5D45ED113C94CD862197EE118CAD47B8C08BF4FA1C6270A61DE"
		  "6D538D608E126C0FFBEBC7F44D418BB87E557E550CDDA20C2C4"
		  "7F2CF635411189D30DEDDBB9075FA6F0521D2C4FF0E4D641006"
		  "CF5EEA1231A87D020664366EBA5C5610B63EE5B5A50C2C8CA6D"
		  "7EB8CDCBDDC34AED08ED7E0EE437E235449D4C1CF095514E581"
		  "E3C8A15EAFDAEF19FDEB31DE6619BEE6B17B31A96140CDEBF75"
		  "34E0C277EE33C54319B140AF0135D6F91F7D2166026C67C71A2"
		  "1287"));
  test_hash (&nettle_shake256, /* 143 octets */
	     SHEX("EA40E83CB18B3A242C1ECC6CCD0B7853A439DAB2C569CFC6DC38A19F5C90ACBF76AEF9EA3742FF3B54EF7D36EB7CE4FF1C9AB3BC119CFF6BE93C03E208783335C0AB8137BE5B10CDC66FF3F89A1BDDC6A1EED74F504CBE7290690BB295A872B9E3FE2CEE9E6C67C41DB8EFD7D863CF10F840FE618E7936DA3DCA5CA6DF933F24F6954BA0801A1294CD8D7E66DFAFEC"),
	     SHEX("9FB76F906CEF4A8C7A9CDDE3254E6D4581DCF745B26995FCBC1"
		  "23E716F079E0A0B9F7C8CCD9DC5AF8894B2D9783395F7BC05D9"
		  "5E20419C15D74EEAFB23B6F1BA427B4B20BAE38C2F6812B74A7"
		  "A671384235DF3618A7531123412113D57B9C2924EE4AFDB9CC0"
		  "55070485A2960E8E5CF1971171F604781865624B075C3C01FA2"
		  "7F9B602C29E257821ED92B35F7941637C2BC6F0048A579E3845"
		  "03182CA8498AE7493D398476727A2AD321785D2273F9F548F6A"
		  "CA80FA7F1ACC1144EAE7C0AE58C3C4BCA20EEDC93C3742954F2"
		  "E2963282084853AF5C8F30A1A4333374B4A66188DE73F773FF6"
		  "A576A0A0DF3C690CC59521378CADB123BD673A04FAFB26DB9D6"
		  "53D60A87F2834683D97F5D6E23FF2ACE2AD0A5FCAC71CB00320"
		  "39EE01A2C3E37B3E1C14ACE44A5A6E43B1E1A0FBC8AFB0C811C"
		  "911F2853C85F10D06253D29362D08523834273961E37C4BB0EA"
		  "D4F79E33D77EBD0D843709B121F8B509C24396549D2187DF52C"
		  "996EEA9805C4A3C4F7729003D7BD7FFAFD6384779D6B6163AD7"
		  "C632A408922507188EA22D685E05C8851EA9C6E625D9461EF06"
		  "AF864E93F9B4544F00244264DC57F4A91920FEC0120357B5430"
		  "4ECD0A45DD8A6879EE01F66F222C6CBEBE8617FC06B28805F48"
		  "0CBFE90D42A38D1981A4176985775CC92E3AA632CF4D5A49B20"
		  "1EAE275628A3CA7BBD4992A39ED5737CE0B4C32ED4C4D84D9E1"
		  "5A46"));
  test_hash (&nettle_shake256, /* 144 octets */
	     SHEX("157D5B7E4507F66D9A267476D33831E7BB768D4D04CC3438DA12F9010263EA5FCAFBDE2579DB2F6B58F911D593D5F79FB05FE3596E3FA80FF2F761D1B0E57080055C118C53E53CDB63055261D7C9B2B39BD90ACC32520CBBDBDA2C4FD8856DBCEE173132A2679198DAF83007A9B5C51511AE49766C792A29520388444EBEFE28256FB33D4260439CBA73A9479EE00C63"),
	     SHEX("D512EF86B19C0FAB408DF09A8CEFF07E8259CDCCBF0C060CFEF"
		  "D8FCD3951BED94BCC30005DAD8002227AC4524A8E6270961266"
		  "6F54866074F2272E6CCA23A88D5A05DD5B44DCF411E137F4464"
		  "5187BF03D8CA19DB835974CE314ED8F3F1CB3407CC48650A61A"
		  "8352FA34C4C307B8DC438634C292E91871A5DC55775BA740B8E"
		  "1085C14BC779A0260288B48C80AB4B2CEFF645FC47751D6DFD3"
		  "039246A5E7F8B1D66FAE0E2B71172C24D599CD5836FAF10B567"
		  "CBB834804C16D111B56D8961EC67CE6DDC989B0AC0FB3975581"
		  "D3BD70043725331685B959E22AF6509D4C9CBB8D4E10C9876DD"
		  "F6460D62AB09A97A220038952F7025993F3F3C5EAAEDC9D9AB2"
		  "AE948F56F956CCFA1DC670C47CE1946EBBC7F1F537CBDDD3938"
		  "686E65E58F1573806EE42DB6E22E1D5070820787198F27B8424"
		  "09B337BF0A5647C63DD0355838F52971616FD7318164CF33164"
		  "1FAB6E3B2282547288A76BBA5B973E138E4182783FADB159CDB"
		  "1141DC950B70EF4609055076E7EFB532C9A944C419BAEE914EB"
		  "EEEF7E1890B292F27ECB8B7DA25B0C7DEBC52FA75F50247BB67"
		  "1087A0DFC15FF3DFF6A3E57791F2471889F356D44A3A10228DB"
		  "2C3CE778ECBAC53F127DE3F4621E6D83F8714C0D66763460A01"
		  "1958101123E9CE31F5562B73EBD4F08947475E62C6D815BD02A"
		  "3BCB12B8D058DEF7A9DE8B4A70A00D039381D61DC6C0317A0DE"
		  "62AA"));
  test_hash (&nettle_shake256, /* 145 octets */
	     SHEX("836B34B515476F613FE447A4E0C3F3B8F20910AC89A3977055C960D2D5D2B72BD8ACC715A9035321B86703A411DDE0466D58A59769672AA60AD587B8481DE4BBA552A1645779789501EC53D540B904821F32B0BD1855B04E4848F9F8CFE9EBD8911BE95781A759D7AD9724A7102DBE576776B7C632BC39B9B5E19057E226552A5994C1DBB3B5C7871A11F5537011044C53"),
	     SHEX("FBE450F412E4DEA6DF16609A1C5513DDC550F7D7FBEFC4F5F9F"
		  "193F3AC88351CCFB1C25955818DFFE5DF50AD08B3DFAFACFD7B"
		  "FD484A628F6C9BB4E32F271F4846BFB90BE00B80572C12457AD"
		  "35541C7FE0FCAF156AB66548A7D90458713F8E10E725C575908"
		  "6D33685D43F3ED682B1F53453707FE17F0D389D0A8497E25856"
		  "6E93062B31E3538ED95691CE730B1DF2F498F010A2C2DE04DC6"
		  "98410A3D4669329E685827DEAE4BAF2C15DBBE975C9DAA13430"
		  "C440543AD94624501AA63AC4D012EDAE314DF68D7D4954D4119"
		  "C5B9505D566B8AB9B54007F4B8D92D9FEEC562B9E0E37FD8643"
		  "907D8E7B6FDE785195650580373D3FC69619FDD039606064E84"
		  "C6C6E44118E4921A2DBD185E7DE90508FFE7380ED1C9917E0E3"
		  "8DE26D8941D6AB0D0B4FA213DB4B075A362B314BC7C3CF21A83"
		  "41C611551057F9E07492C8D923D54336F3F0AA44422C423E2DB"
		  "0D598216889753280B08103417F431658F1EBE267237802E1EB"
		  "FCA862027A29AEBADF8ED9A603983E0BDBBE84B3D9A5F862ABB"
		  "F5E18E5552C0D3E7577B96916123B4504AAF2107F643067B88A"
		  "DE293F5B3FD3F6955CCD9AD7E134C9F5EE60F6DA78032AD1C85"
		  "D710BC21939008111602147E0BF1D79F7F5302686729D44E44A"
		  "7A6521B8270CC57CC8A6F16770BA66EDE8FA823FA088BB7DA7E"
		  "EE980C6C947D7F510DEB93BF0B5F4380390408823AA8EB980CE"
		  "0F04"));
  test_hash (&nettle_shake256, /* 146 octets */
	     SHEX("CC7784A4912A7AB5AD3620AAB29BA87077CD3CB83636ADC9F3DC94F51EDF521B2161EF108F21A0A298557981C0E53CE6CED45BDF782C1EF200D29BAB81DD6460586964EDAB7CEBDBBEC75FD7925060F7DA2B853B2B089588FA0F8C16EC6498B14C55DCEE335CB3A91D698E4D393AB8E8EAC0825F8ADEBEEE196DF41205C011674E53426CAA453F8DE1CBB57932B0B741D4C6"),
	     SHEX("6658EB2FEB9A739FF1724FE1526089D0AE807945741C2C6CB63"
		  "0E8F5B9C88F3455017BD81A5ED8F441AFD50414ED0B9C267E52"
		  "7786041C4E3F8DACCF1DB60A0ADE7199F689C8F0C04CC846318"
		  "F0B8C34B122BB98386F48B0E69317938DF76C9C6D502BFAE1FD"
		  "A37DDBA108AF9E2D07CFEB50BB79D828FD5A67C9C3771FFAA07"
		  "C7A47FA681A1358611BA76B079F4FE14DAB9A0D3A4E5D2AC41F"
		  "4F1374319FF9B81D970511464F5DA9AE9D6312FA8BAF22FCECA"
		  "2B4BEC1A8D887F47811F1E7ACC1884DEE4740AB2F6E2B4F8C11"
		  "C81C7E40BE19F90F27FD1722AB95BAA214B4096B4B11A1A9B55"
		  "3C4ECD789AED9F22FCE58AE1963A62E450D754CB048C68EEF0E"
		  "7D2C2A146300A902C56ED8D149BB5C36566CB8A3DFA8582562A"
		  "48927DDEC45BE750F639881CBC33D3B48C128D91108F2A55724"
		  "BDC5162B652CC03D5F591E96F4BB40A571E1B22F504A6BB6D8D"
		  "897A73884057E1D2EB15DE0D3585BC14CDDFAFCE3AD5D6A861D"
		  "7CF82B5F1AF784637A99B79F91528C4976720F34A3A12BA0A4B"
		  "C84975CEE3167D4E72292E9295629503D642A29196B934E71C6"
		  "3425AD701820E79DDC9220DA7C7A2AAAFFB852793E822B69157"
		  "34FEC8B93344BBBC66C5192B7F1AB6BCC63713B11CC23D54144"
		  "1C04428ED11444BB2F7EA981603ED4103F86E6623014FF97CD3"
		  "2DA3C6E1CCC22EA8BBF3B6853764A733E61397302A96434ABF2"
		  "905F"));
  test_hash (&nettle_shake256, /* 147 octets */
	     SHEX("7639B461FFF270B2455AC1D1AFCE782944AEA5E9087EB4A39EB96BB5C3BAAF0E868C8526D3404F9405E79E77BFAC5FFB89BF1957B523E17D341D7323C302EA7083872DD5E8705694ACDDA36D5A1B895AAA16ECA6104C82688532C8BFE1790B5DC9F4EC5FE95BAED37E1D287BE710431F1E5E8EE105BC42ED37D74B1E55984BF1C09FE6A1FA13EF3B96FAEAED6A2A1950A12153"),
	     SHEX("CF7D80DED88E700DED518B2F0049405AEA695691924B4EF0620"
		  "5DD2849E7A0090647C3A23407CD4EBD5B582C32B6554C506E41"
		  "6B9D13903A3AD69234826E36E3C5862BAEF9BD93EF345125678"
		  "E32EBD5BAE0961935232D5BE54AC721FD75FE1165341A41FD88"
		  "0D239DE12065471AEE6CF970CB6A0F02ADDEA861FEFFF91E6C2"
		  "E482EA9BC255FB00CCE1FE8CE3DA479EB28768F356C2EF83F40"
		  "0A0F0CE859606F3862A3579E609DE48F1CED3C86A9EA302462D"
		  "E9A27B0F251798F9D0ACB22E6C71011F0DD45B11E8E33543A32"
		  "75B24B6464980A5B6FFAB2E9C8055CDA16BE6C60FB616A8F3E8"
		  "B759552A2DF2B05B846BEAE18A01DCAB1E92545B0B3CC2DC251"
		  "34379021DD7DC294BD97A775C174A04F2F67591841E080308C5"
		  "87E208D1B497E229220C38469944EF41F61A64C83E06A4390BC"
		  "F318E53E4A6E9214F943D1EE957B71E8580E0A183C6B05924CF"
		  "32809772CDBA700EA369448AB56F399385D4B8D851CC8D6E800"
		  "3F1035C6E739C91E03F1FCD9662AC99E372B45FC08610265E53"
		  "0F53FF726F599C33820AB468E6BC2C551204260271FE46C0A29"
		  "75A18BCCFD87A53B2A7EEE2095A180B0A5BC7D64D57FF8EFA62"
		  "8349E00C84B271DDFE13EB009387B1055F3B0E7B556CE6A738B"
		  "9DC880B22E79666148A6C4D97B2CA581864FF4400749750BAB0"
		  "3AF9232815BD21A1A6460483505C34F282D59863B9F491A39D0"
		  "28D5"));
  test_hash (&nettle_shake256, /* 148 octets */
	     SHEX("EB6513FC61B30CFBA58D4D7E80F94D14589090CF1D80B1DF2E68088DC6104959BA0D583D585E9578AB0AEC0CF36C48435EB52ED9AB4BBCE7A5ABE679C97AE2DBE35E8CC1D45B06DDA3CF418665C57CBEE4BBB47FA4CAF78F4EE656FEC237FE4EEBBAFA206E1EF2BD0EE4AE71BD0E9B2F54F91DAADF1FEBFD7032381D636B733DCB3BF76FB14E23AFF1F68ED3DBCF75C9B99C6F26"),
	     SHEX("FDB4CBEF885524710F96F1B489099A8E5F9F5EF2BCCCC23B6A6"
		  "19BC81F9DBFF63B9AC609BD5DA0FA7AFAC0E67BB96A54A1A64B"
		  "4B943E0A492A5BFFF399CC9A1FDDB5787434F5034A30CA767CD"
		  "8A31B69276435C99C64F3C567EBAF3C69FBCEC540CB2FB671CA"
		  "3835ECD228C7C7649410107D399EDD661BC8947D43D5BC441AB"
		  "2EF39AFAC6788BCCDAC5462D5698F60616114A8346707606D15"
		  "01C834447031CFEA921A85AC882194750FB8E4D22573A27F290"
		  "1B75C6887D085A6B47E2BED5AF785C9B60DA38BC9F61750F16B"
		  "2E06CAA1CC77CAD2B95C5D583D0E1BCD7E10B0D8FA8DA7E0975"
		  "07D11873395166E3591BAB851EF4CD7E1F05CD8401AE4967378"
		  "94EC50DA4EF475424485BE599A4E8387B0309629D5B634BD70C"
		  "4C95BE723932EFC24CA8FF978CD31B08666F2F6DF9E94A6F8AF"
		  "CDDE6C6A01AECB63950A4B4BCFB4ECBC1FEAF3B54454CBC7292"
		  "A65282C9B0D7C6A053FFC11F6888DA2BBCD6496CA74CFAF0F49"
		  "A6F5F56E14E51955B73994015313404D46BCF0CA0854407707C"
		  "176F36C3AE5986F67984A33F252B86B2758CDB7D2FBD63606B2"
		  "E43F7BF0919CBF97D2E38F01ACA8B3983421CA447A78BA415AD"
		  "D0580FE56E391817783F11533C8D70E3B8FFC4A3CCDEC56297E"
		  "D51F80215BE98C26B343736196F5C68A65B308835C869328006"
		  "1865435383272A77397DD6FC3249B2CE17AB3417AC9DDA62EE7"
		  "4C43"));
  test_hash (&nettle_shake256, /* 149 octets */
	     SHEX("1594D74BF5DDE444265D4C04DAD9721FF3E34CBF622DAF341FE16B96431F6C4DF1F760D34F296EB97D98D560AD5286FEC4DCE1724F20B54FD7DF51D4BF137ADD656C80546FB1BF516D62EE82BAA992910EF4CC18B70F3F8698276FCFB44E0EC546C2C39CFD8EE91034FF9303058B4252462F86C823EB15BF481E6B79CC3A02218595B3658E8B37382BD5048EAED5FD02C37944E73B"),
	     SHEX("748DE9C6E6A8886E9C34D67A5B2625315D8B0057766CE2853F8"
		  "DD047A3B58A410B2327E56234572DB7EA886C90F6DF31B795E9"
		  "3AF95A4C52632D7E03D255668255F54C0DC1264EE989E25DD0E"
		  "A77A8855BC74EA1066F201C999C0BE63D1817DF33DB624F540E"
		  "BF463C4B48569136E891C1082A37E6ADD6922C08FE512D92563"
		  "9CC37536A324F3850932B474F313647E65FBF28CC1804D2EF15"
		  "492CE09E84413A465C69FCA84014F15F2FF3FF5BFA6A9C38CB6"
		  "9E75D66145BCBDC462E3A6FCDBD7835C44B9D95E1F6D72C1A7A"
		  "F0E9E9730559499553715B0A7EDEB6E5C8C5A3536CABA0B67F6"
		  "19AB317AA8F60A005860599BD1A1ED20FFD191CF7AD26B63264"
		  "CE022E1CA0EEC821D6A4BFC1E97482CC33A14BE20D3F6EF96A4"
		  "7DA800CE1A8FD6939069DF50C10145BEE28799F8A1C43D37CAA"
		  "377AEAE5A2A19E6DA2F173124CED0AA5652AA34632BE11CFEB8"
		  "FC3C9A0B4C45A7933B671432AA666FAFFE679562D727F786960"
		  "609A91D0972C35B4B9AD27DADA6A24E00589CB1EB9E4706A476"
		  "C5BF749091DF4370CF99FF6ADBE1B029505BBAD815AD50EBDCF"
		  "22208937FBA80A453A99CA49004706A3CC3A95CFAB7F0C2FD2E"
		  "03617F379FDFB8A91442754DE62805797ADB4CDAFDFB7A69F2B"
		  "0CE9DC530A9EC8C741E892758512C4AAC208564935791713B49"
		  "B95A9DF215FDD74F906F8EAA7B13A0EAA443D78A482364ABB88"
		  "3813"));
  test_hash (&nettle_shake256, /* 150 octets */
	     SHEX("4CFA1278903026F66FEDD41374558BE1B585D03C5C55DAC94361DF286D4BD39C7CB8037ED3B267B07C346626449D0CC5B0DD2CF221F7E4C3449A4BE99985D2D5E67BFF2923357DDEAB5ABCB4619F3A3A57B2CF928A022EB27676C6CF805689004FCA4D41EA6C2D0A4789C7605F7BB838DD883B3AD3E6027E775BCF262881428099C7FFF95B14C095EA130E0B9938A5E22FC52650F591"),
	     SHEX("1060697962EA79B9519FE3AAEF385C997B2E8028D68DEB08AB3"
		  "B76F71CE161EADE98C89D223EF0384232AB3AF3A6CB1D43EF5D"
		  "20C1B3BB819423C026BEF0BF54F4800CBF8E9E2D40F9A4C6CBE"
		  "7487600BF451783A378558BA27268A5074AF0630F5F2CB890BD"
		  "A971A4AB65CDC37C0215DB1C09BA96240BE066F95F158CF863C"
		  "C18346866522529503E615E4CFEF990F49583EDE896BEB63B14"
		  "89B6A62BBAD9330D5C3DEB673D42151ED98A3854C1E80E0868E"
		  "27D8CBCDB90DBF3505610FE170537919EC7C191579148E06D7D"
		  "89BE3F5C88C430E38FD36D820DB5A941680E08B5D1859EA327C"
		  "82EF50ADB1D8E6E5184652064CAE2C617337ED36CBD1ED42CDF"
		  "3ED0C411CBE1A9E92E0F4ECD3CF84CB6C07093D51CD4D5E7C75"
		  "43D385B9F420248749675CABE9FBB87F56C865325B826C4D32E"
		  "28B1E4BF1889F2D00BA87EF01720D73EEFBC38828CA76027C65"
		  "6E3758DDD1CE730E2AAAC0354EA71E79D53C0D3526ED8FC024B"
		  "B8CD675AB290FEED6CEDA91A29A71F5C128473EC94EE59A29A3"
		  "81E5046EFEFA8BE3AF7FF0858DD54065B41FDE35F618235D522"
		  "1AABE2B4BE62C1E7FDE8BEAD94F7BDDBAFB8A471199B3BC099E"
		  "77D8190029CC7508D576DB31F362B09E5C1131B0572D86D4A7D"
		  "4B533737EB4D7E504DD0852E079F2A1B2B3A526175948227EA2"
		  "511ECD4385F616B3BB086C54F191046BD1782EEE96257960104"
		  "93B3"));
  test_hash (&nettle_shake256, /* 151 octets */
	     SHEX("D3E65CB92CFA79662F6AF493D696A07CCF32AAADCCEFF06E73E8D9F6F909209E66715D6E978788C49EFB9087B170ECF3AA86D2D4D1A065AE0EFC8924F365D676B3CB9E2BEC918FD96D0B43DEE83727C9A93BF56CA2B2E59ADBA85696546A815067FC7A78039629D4948D157E7B0D826D1BF8E81237BAB7321312FDAA4D521744F988DB6FDF04549D0FDCA393D639C729AF716E9C8BBA48"),
	     SHEX("F31639CA0F9455E11389937E9E58792E3D73B9CB950ADBE8066"
		  "07B1AD53D80F2A199F577E278137035136AD2AFE7F8FAFFCF0E"
		  "60F0B2A71E0BD9B938B28FCBA94A912F249FB9AE1779F87BB1A"
		  "A714D1FCC57D4BFB899A233B99586F81F4885DE6693DCE78D9B"
		  "6631957F1689E843C1463045CDA6B921173054E14BE9DBD0688"
		  "90DF88932F68ECEFAB3AD3EECAC1B4C398292446B17E826AA8E"
		  "35F6067C908AFCE46FD92899726677E2C0589F525AE021A22A6"
		  "01839BEAFED00F6A8FF6223FC5A92EBCF6207341B63FA2E66E9"
		  "DEE2FEF1217D9D39915924E522BB52FD29501247B1A7B9DBC06"
		  "678C5F29AF7CEA3D8722374446992A4BFFB0E3E623E881EFA7F"
		  "957DB1946496C2EAAE98F0929EF3DF633E67730D06483770197"
		  "B9DD17C7C9838AA5003C5BDB4BC54E7389D453E769765879485"
		  "ABB3D565749C8CD5E50DA9C4517F31ABBA38F1D7A0FAE0D2816"
		  "CA0BA1DFFE569DF4215D215676DE0FCF55A75FA5F7E899F045A"
		  "4B6051DD9355032DAC56FC56C4988A4C082FB85B74A4C8A407C"
		  "E1A06C344B2E833281CA64C2F23FD8A9F996C563E9AF8742137"
		  "617C0E46C5168E0B43877446E07563DEB3D7654D9EA3C8AE3DE"
		  "C58C5753F34FDED605A49236336F0B355E95DEF3BEA7A367507"
		  "F19F4E0A6DD03ED3090C285F761C48A3D3C5795CF277A87FCA6"
		  "88495EFFDF5D10FBE07840C3F202D85044FA3BC41CDEC4D0AAF"
		  "59BB"));
  test_hash (&nettle_shake256, /* 152 octets */
	     SHEX("842CC583504539622D7F71E7E31863A2B885C56A0BA62DB4C2A3F2FD12E79660DC7205CA29A0DC0A87DB4DC62EE47A41DB36B9DDB3293B9AC4BAAE7DF5C6E7201E17F717AB56E12CAD476BE49608AD2D50309E7D48D2D8DE4FA58AC3CFEAFEEE48C0A9EEC88498E3EFC51F54D300D828DDDCCB9D0B06DD021A29CF5CB5B2506915BEB8A11998B8B886E0F9B7A80E97D91A7D01270F9A7717"),
	     SHEX("F04B251F8F1A9FE87E9F8B0BD8A641566174CD2157ADFBA255A"
		  "4DDB7A001ECD9B479B7877C38E487DFF1D830D01DAB9684F143"
		  "8C4812BEF8DA03ECFE57E7EB68A68E4FEFD3E0A70D607EC7710"
		  "2538D5535C293339532953138995CB4F23AF709F5C103E662E7"
		  "EDC447995FE5BC716C7BC482CD0CD13F17AB48D33C8BFAA67AB"
		  "6AD30482B6DB8BA5B1290656BE81E1B543793D47C1D0EE7BC95"
		  "3C812E187535A0659B2CAE6BE47284419744C6AA66B9E53F76B"
		  "9F90D4AEF8DFCB04A98C82FB434C0946E5B98F2C3BF25C128CE"
		  "F77EFB9F6412EB11F36E484FAF2043A40CF44C6C3D28761F9CB"
		  "7149C3DB4620D09B220720767AE41BB097C865720185C19A474"
		  "E4C3D706A636E42050A23766C2FF3A21C4E9DD7AF62F47ECA67"
		  "1F473E2B487A65ABBFCA74CED063E33F75955E9958240BC3907"
		  "BB90EC6F49F384DF5E936AE14FCB308F8ACC1F7A406DA680E4E"
		  "FDC2D2BB74B922C7B49E4972E388596B4F05FBA68ADF9A22FE1"
		  "6737394DFDFFE2F0D32F1283EA81996B8309C2D10D4E32D4B7B"
		  "C3D88484BAB364D0B7E35FF5647053CB34180E037A66349A254"
		  "AE2AA8F81CC7FB5514E76D9A01F96394E3A0044039C452ECF0F"
		  "DF5B8A57A2A5600AB31AFFD1E15D4642F1A37DECA0EC60EF753"
		  "797AD668B7D9F2BC02DE248C719311E5696E11688DDDDF71F2F"
		  "1994331CBCFBB9FF22438825620033734C27F9FE72766BD9C7F"
		  "B021"));
  test_hash (&nettle_shake256, /* 153 octets */
	     SHEX("6C4B0A0719573E57248661E98FEBE326571F9A1CA813D3638531AE28B4860F23C3A3A8AC1C250034A660E2D71E16D3ACC4BF9CE215C6F15B1C0FC7E77D3D27157E66DA9CEEC9258F8F2BF9E02B4AC93793DD6E29E307EDE3695A0DF63CBDC0FC66FB770813EB149CA2A916911BEE4902C47C7802E69E405FE3C04CEB5522792A5503FA829F707272226621F7C488A7698C0D69AA561BE9F378"),
	     SHEX("8728A95B55D97271B701B1BF2C2987AB3045D39FE0978C9467C"
		  "41D44C006486CF3CA11B8C0D22C619A8E4652AEBE67FE9FD7D3"
		  "5357633C8E11B865BD4C55571D764E6F92E29B5C93606273522"
		  "A958EEE0CB2315E89A1B5C45E763B145B1166A4717784E993E4"
		  "A7A699FAC6385957AA914BFC856E8CCC62DD39159F0FA5D2F4C"
		  "74A4C9E1F3A1BF04F8D73BA5A3133243973562AFE874C403310"
		  "86E7A9705AC76CE8482B7B029278CA61CF74B5AED280A6E7959"
		  "06C4C3F525C5CB382E55AF0AC5C9EAFDC520B0AC74F14E0203F"
		  "1209FE6641D1D971C76370204FC14BCD6A8223E30CA81B7D9E9"
		  "4F35DC53634646A21C28E873BAF741705E4862AB6945653A312"
		  "125F534A75B6F2D1CD4409D8F842DBDFF9FEA090F6677937DAF"
		  "354F6130ED77C32711AA0C7CD448D92075909ED7CFED67BBD5C"
		  "CEDFA34495624EE4373D66FC2F3B25B78AD6A1A9E9FA9A714EA"
		  "24240161C8CD88F24FE72CC3160D5F68F2387347B580E945F40"
		  "1B0B025A21E97FEEC0D4DCC412D2597504489B3CC20E3C883C0"
		  "D88F26591AB840799B5E9B253FE02656CA6AE1A1EF1BE5F1961"
		  "F29CDBE35A40BDAF2760C7BBF051ECB62322DD178753B3922AF"
		  "4CEF0AFF4955C32F32AC28D3A44B7F1E8CF93FE6547E8C3DD35"
		  "159E92A9F3C6E4CF2A75C97CACC0D732B984A8B7A6AF9B81898"
		  "96FEC2DBBEA60C3B94904F154BDC8FC291E027ED03EEECFED06"
		  "7969"));
  test_hash (&nettle_shake256, /* 154 octets */
	     SHEX("51B7DBB7CE2FFEB427A91CCFE5218FD40F9E0B7E24756D4C47CD55606008BDC27D16400933906FD9F30EFFDD4880022D081155342AF3FB6CD53672AB7FB5B3A3BCBE47BE1FD3A2278CAE8A5FD61C1433F7D350675DD21803746CADCA574130F01200024C6340AB0CC2CF74F2234669F34E9009EF2EB94823D62B31407F4BA46F1A1EEC41641E84D77727B59E746B8A671BEF936F05BE820759FA"),
	     SHEX("B0EE2D0215F195934D2937D523D38F1AF153FC142C04E5D6848"
		  "34596F3B909241938A56209D414BD1AE0CB43EB2B46C314C1A0"
		  "A0B155D432947C318642FE5C5B4FD6D0E4C21A9DC594DB20530"
		  "7B41EFFD14141454EE55DAD5667CC7459D0283CA6652CD5092E"
		  "F8BC35C908ADD07FD5DAED66AFC6740BAEC7FCD64B838368A8F"
		  "F0DCFE25B0D46FEE0B842F9E9F59EEE18D73C2549CCE009C388"
		  "B5A8EBBD54B35676B6140DEED07EFD4624D5A7F816189DFA779"
		  "929C5F6B1B9489E27492AD0F1BD085EE890794529B75E10BA00"
		  "6C1B66F3F662C30DF8B1295E1FFC301DC7060010862CAA14D7F"
		  "A09817F0D91E928ECD89FD8E76E69B3EF4986A15DFFCEDC9AC2"
		  "4E5E28783667731F07998F8A36DA512BB5BA0CE50640A1A5328"
		  "FE037F5872A85288158CD5C66B6C03F1925C12258AC4B5E675A"
		  "CD73716AEAC389F9C235B02FE363A98CB10D9A0B44FEED935AB"
		  "067C2EEB499CA83FEDE967F576681A13C999FC7C1C0FDE2754D"
		  "FBA713453545C6AE722F4FACA5F5F2647378BA9308DC27D91BD"
		  "1538624F459D0769F303AE818CC52647AC15E2AF9BEC6476A23"
		  "2C1AE81BA4B355C9DBCD37323D125228A53086E0AFDC98C95F5"
		  "6E12A48843170E9CAE703F5001B52DA42C0AC9D285C8A13AD72"
		  "0104244F19C30B38CCD620EF8453401CA0FEEBC8B58025BBC6E"
		  "6C2C788899871D3A5A363D51D8991B5CB5085F8AD525A71E985"
		  "B3DE"));
  test_hash (&nettle_shake256, /* 155 octets */
	     SHEX("83599D93F5561E821BD01A472386BC2FF4EFBD4AED60D5821E84AAE74D8071029810F5E286F8F17651CD27DA07B1EB4382F754CD1C95268783AD09220F5502840370D494BEB17124220F6AFCE91EC8A0F55231F9652433E5CE3489B727716CF4AEBA7DCDA20CD29AA9A859201253F948DD94395ABA9E3852BD1D60DDA7AE5DC045B283DA006E1CBAD83CC13292A315DB5553305C628DD091146597"),
	     SHEX("19E39F7444A4A0E1C003631423A621C055D0E07F0E11EF5BE18"
		  "36B71690CB0DE1565684C355EE9F2E6E65983B8101B5100F391"
		  "57DDB8C8D92C736604693ABC56491C58CD66BD66498143554D0"
		  "CE34D601072BF9D0E18EA1B018A7ED9A65EDBD0B81212EC13AB"
		  "745728ED3A328EA157B9C4B3195F8CB8BAD94A977015B94FE80"
		  "50F37A0BC14E58EEC7654B336A9E30BAAD27C441CF8AFE09313"
		  "2234CE47CE3FB3722990F6C5916468B50A13C611B194D148A15"
		  "1B9AC97A89539D1EFD2F9C6F72F56A8D22A557B23EC7D932A6E"
		  "41B3F2C177CA9F2ECC2A1EAD4061DDFC04AB1198DB0E26C4B00"
		  "301E35B18754DF79332A46190ECF89C0DA6834731CD19E2C8DA"
		  "3A09ED16813D2A41F7A0A4ABE02C93F7307FEC1537ABA546865"
		  "A3130D38ECC496E548A8EBD2858EC00A57E15E307C1A35D1355"
		  "B8B97E9DECE5A63AD7CBC97B30F2C9DE69E4975DBACC31E04F1"
		  "5A8C02E5D23ED01506AEFEE60A449DE20C5A50F1208097BDDED"
		  "F6D2CB34036DA708BA53DFF7BD6A1E41057E05BB57B3F756197"
		  "9EB69C40348DEF728BFA8382F5BC8DB63FBC08E6506AD80BCE0"
		  "C71FC94036EEAB1C0C89A2C8AF4002BD15617834A901FFA42AA"
		  "249F7EA700E6FF4B211F7B052ED5BDE9B242C9874ECF39B318A"
		  "3713F1DC3066425ABE76C4AD47F43ECFEDC86A45E1E93544989"
		  "CE7ADEA7E4CB3CF91047C4F9EAD7C98AE963F34E99874B4002E"
		  "3FF1"));
  test_hash (&nettle_shake256, /* 156 octets */
	     SHEX("2BE9BF526C9D5A75D565DD11EF63B979D068659C7F026C08BEA4AF161D85A462D80E45040E91F4165C074C43AC661380311A8CBED59CC8E4C4518E80CD2C78AB1CABF66BFF83EAB3A80148550307310950D034A6286C93A1ECE8929E6385C5E3BB6EA8A7C0FB6D6332E320E71CC4EB462A2A62E2BFE08F0CCAD93E61BEDB5DD0B786A728AB666F07E0576D189C92BF9FB20DCA49AC2D3956D47385E2"),
	     SHEX("18AF27FE1B4AE8A470778480F2586A370C93640BCDDC33453E6"
		  "F3EA1346B5A4ED3EA3E2EA43B143A22FC66501DD52DF724C0E8"
		  "F222380B4BA8BC23F2967D0A56EC241A0574E2783A461098F26"
		  "3533389D503B5D8550C8F60FD6FE953C5631635B6990434F8C6"
		  "8F8E9AC3701B3600C2D80F5B12D14218481904D1F15A776CFE1"
		  "FA745103B995F9CA8E8E2CA062C5A3F1FDC848BFB0DE4A47EEA"
		  "DA11C51C743CD15C32BF921D07F242B83086660605B29FC36AE"
		  "394C215251B49E9777A93AD91179CFB0551E113DC372F992674"
		  "ADBCC2D90AEBC572B4C82B454CEB33910B60CF81B1BA558C2E0"
		  "1EC1D837FEE34B86124E15D7BEB7155E5456DC94D2427F7D916"
		  "B9C34D24312EB2610A48002415182037AFE89B29DDF5DD7E70B"
		  "28BEC4A1F3FEBD72CCFEE8360E838FD787D0AA2AF3A502D6454"
		  "33AEB1CFF47E8E6769834CF0A2A85BAFE9A89B43302BBC64C88"
		  "AB6FE4712F3181A056921F6AC281C872E3EB21590C50B915F38"
		  "172849CC8415BAD8161DFE0A39C2532A34551441D78825FE790"
		  "4B3FEBD8F645BE5AA6F7DCC6EEF96431086781C17615141D415"
		  "9D6DA40A3A5C22631BDDB7F00040F0D39107BB366D827478515"
		  "48F3ED3DBD028E3261561C8D7FCC077D3905FF42BF3CAA9E351"
		  "207137F7E8271EE970117E56D30B35294E8B91588DD9D990027"
		  "4DBA4108BE1595833284047739DC4E90F9E20BA5EFB2B736DE6"
		  "548C"));
  test_hash (&nettle_shake256, /* 157 octets */
	     SHEX("CA76D3A12595A817682617006848675547D3E8F50C2210F9AF906C0E7CE50B4460186FE70457A9E879E79FD4D1A688C70A347361C847BA0DD6AA52936EAF8E58A1BE2F5C1C704E20146D366AEB3853BED9DE9BEFE9569AC8AAEA37A9FB7139A1A1A7D5C748605A8DEFB297869EBEDD71D615A5DA23496D11E11ABBB126B206FA0A7797EE7DE117986012D0362DCEF775C2FE145ADA6BDA1CCB326BF644"),
	     SHEX("AE82057AC5EA69CC1B5EBAC395B0EFF26850C394AF8178B55AD"
		  "808879AD634569BFD8D0C5F34F9A3B26821F63082C5D2C322F3"
		  "47FA6976AA88ACFA9ED8393E126F7F47DFB523C0560DA535793"
		  "7A74488B51FA67583C7A7E1230C015FB5F233E25B744F2FB683"
		  "0278B123F7784C4F6070C8F905F96486B2D5FE137BE8D57150F"
		  "47CC7DCC0011FD99F35DF4D9E15F0F937762C87153AC17645ED"
		  "67581B38E2B34899A7D726A9ECC1CF2E5198ED5AFF13F2544AA"
		  "A1A069CE8A30CB66E1B984002D20F769C0F94C95F99F914D324"
		  "02741D5C4D631B9D6E561D32D1ADBD1F1DD692BB89F48C6BC4F"
		  "88B669FE4BD2922CE6A3184605053322952E113C98B330385D6"
		  "C4B6527303563DCDF099BC2102F915DDF3BE0989EC8BCEB029C"
		  "E61974AC41A26D6DC7F53405A4F3A9130C5CCEAF71F828B3F7F"
		  "58117EB476A46263423F68ECC3427D9A94C292EF7C4C1D67B32"
		  "3FE15F69894DF362E54B441BA5C94D3E0B56B772AFB5FCBF8CE"
		  "62647DFFD78BA7F48DEDD2E457AE9FEE60580084A05F318907D"
		  "FB90FAAEB9F4D9C160259A6F8523AF77C8CD8AF344F41DD511E"
		  "46C699A8F9A5A85EACA57780005AF313705820D8051D780506D"
		  "02A09B04470398D0AD323ADBAA8B3DBF61AF68D9FD9DF29A76E"
		  "0A37678C08C9D102BA960DB0B5D20B6BCDAD7750D3F8CDE6E99"
		  "4267E190D7F3DA06C36CF7539B6B574034B43F55EF8DD370F72"
		  "7B97"));
  test_hash (&nettle_shake256, /* 158 octets */
	     SHEX("F76B85DC67421025D64E93096D1D712B7BAF7FB001716F02D33B2160C2C882C310EF13A576B1C2D30EF8F78EF8D2F465007109AAD93F74CB9E7D7BEF7C9590E8AF3B267C89C15DB238138C45833C98CC4A471A7802723EF4C744A853CF80A0C2568DD4ED58A2C9644806F42104CEE53628E5BDF7B63B0B338E931E31B87C24B146C6D040605567CEEF5960DF9E022CB469D4C787F4CBA3C544A1AC91F95F"),
	     SHEX("39B12FE62B75D707C8E18FE1C8CD23D3C1E15B7F46FFCB24C70"
		  "A1D86604FE6BBC36282595346A3374B0B2305F60CEBD77DE112"
		  "5373F5230535AB0E4BA011C6154F65216CE58A65856C7EF6091"
		  "2877E44ED2527C3038CA202B5445E97BF817BB6419725ABE41E"
		  "A1529B6DDC64920DD8396A7FE3B26054C09E310465BF65AA7B5"
		  "6B1877D949DDA06057B7373023EB09E4CA1790D66A773BF34F1"
		  "B9D4011322E08BBB7139B1750278C2DCE88626AB16813803E32"
		  "0515640DDD9D6A7C5B0DFF8FDB445CD65D6BB454EDF83D6BF4F"
		  "3387CC98E381C65AD3811C8DC3B6A42C203FD2AD6BC464783B0"
		  "73C1A9D37C1D29F85061E9C426DD195A98DF3EB0C2584219628"
		  "57EF89D643F8BB18DB27F339BA178E85144040BBEFEEB116088"
		  "2C3EF2A1C4950873AC4E5C853E48DD7788250DAB58A06BEEE19"
		  "03C2E68FF0F8D7E8F55AEB58CB0D65F5E74FB5E301BAE225831"
		  "4D5B82E3C0D7C7782AA1B34E80BEF2EB258E0BD322451097B0D"
		  "F4532A717E2F0FC166817E2A9E24A318E6A4DB618B6A8382A1B"
		  "D1A600918CA08EBBC224AE9A7C1CCC8978759F512D7A97E4D1F"
		  "05173B33F670BF3C40E2807128F752FD2995EC3A60891575427"
		  "5EBD215821A4F04AD21DD823341B15A484D0C12BB3D829BF99E"
		  "182FE774D2EFC726EB37638CBBB521FF14E10D73C7FFBBD8B79"
		  "EB86470DA14236C087EC3BDD336E8244E780454DA1A8E70524A"
		  "19DA"));
  test_hash (&nettle_shake256, /* 159 octets */
	     SHEX("25B8C9C032EA6BCD733FFC8718FBB2A503A4EA8F71DEA1176189F694304F0FF68E862A8197B839957549EF243A5279FC2646BD4C009B6D1EDEBF24738197ABB4C992F6B1DC9BA891F570879ACCD5A6B18691A93C7D0A8D38F95B639C1DAEB48C4C2F15CCF5B9D508F8333C32DE78781B41850F261B855C4BEBCC125A380C54D501C5D3BD07E6B52102116088E53D76583B0161E2A58D0778F091206AABD5A1"),
	     SHEX("939D443F3AAF7809342B2EC600A9D1A47E0C4195C9E1D95CE22"
		  "F95CB98AA970F4FDD29A7EC9DE471A1C342D4830D20DFC5ABFA"
		  "98F8BFD4CDF752E00BFF151D7CA44D5BBE7AC18A9ECC1157354"
		  "B2055AB08B1AE7B304623D643F6D471F23A82C2E1C67F11D527"
		  "985208C5494138815C3CF6227A6A6FACC9E2A1E3D58B32E2964"
		  "15E5889E58E1DC8657B0324BD1EAC60FF8DB83674B75683348D"
		  "69978F04697A1D1D3E6D29D02094C277CF034D8203DC4D705DF"
		  "8EAB7046C4811C18F2EC906B735BFD1C2AA3306FA0C9AA47E16"
		  "0B6EAB89E0B390F37E53B92A1A160560E9A30EA3F005D100F05"
		  "21AFF20A2DE0ADE53BCBEFC517647C81AB8684B2754FD62E5ED"
		  "A0EBBF56AEED4F4992A0FD4D26AEC29BD186B2E254DA07E1C5A"
		  "D03F8194AB350510998A3BC11F08EF33C61069D1CB7D271F7D7"
		  "C799057E6C2ACC1D3FD8DD666E1B2C2E15CA2F99E55163BB01B"
		  "CA7F9D53F6389704874B3E5DBE6E74D91F110CF8A4D5EACC7A5"
		  "19F664C962B11A48E022CF65D4DB4E4E506951513D83669C786"
		  "FDF84A79AEFDAD75414A6CC82EA84EB3A173E887D39230631B6"
		  "CC2AD26C55C938C6C82181F57820FE88DEA96AEA2B20E330C29"
		  "C09A645134472B1B1303B1F46DA40901A63C079689DB2C84E46"
		  "FC548270FE88A4A2FB5EA27C58A4374F72849B9C5C7C18AE207"
		  "5546868A05EEDC3D867F9601C5C8EE585F14A1E3A84E2FE9A1A"
		  "99A0"));
  test_hash (&nettle_shake256, /* 160 octets */
	     SHEX("21CFDC2A7CCB7F331B3D2EEFFF37E48AD9FA9C788C3F3C200E0173D99963E1CBCA93623B264E920394AE48BB4C3A5BB96FFBC8F0E53F30E22956ADABC2765F57FB761E147ECBF8567533DB6E50C8A1F894310A94EDF806DD8CA6A0E141C0FA7C9FAE6C6AE65F18C93A8529E6E5B553BF55F25BE2E80A9882BD37F145FECBEB3D447A3C4E46C21524CC55CDD62F521AB92A8BA72B897996C49BB273198B7B1C9E"),
	     SHEX("FE0E80E36E66E0610581835D69A73908F1D951A81DE93FD2BC5"
		  "F10736DEF8A887EEFEDE8BB6074862EEA4BB0F131E48D2C3FDC"
		  "8B8B8B93AEF4C06BF97214F4D0C9000940F81664E04D88316F3"
		  "732B39E7F9685BE377B90FD7343947E7A79FEE7BDF5757C386D"
		  "E6D02035FD4082735CE2FE898F18BDF00F3DF5BD160D792D3A1"
		  "56584EEE92B273F9A52ED221831942EB0F148DFBBDBC2196006"
		  "4D0E9E1CFE4E08C7927FDF1F0956288419B0372D3DFF7D1A252"
		  "8062D31D97D17F7F8CB09ACE01187710E9DC2CD842CC8C09A0D"
		  "1FE34424283FCE75E64446608CAE796C1F6385F989FC93B31F1"
		  "EDC36BD2C3EF3B71D14F25048B2E30A5F279180FF3FD0B8308C"
		  "A93F4D1E72F7F45C7313AB6315D1F0B198F8D5B47742E94EDBD"
		  "33643397661391A82832744EF99CAB293F26C8E9B0F9D6C9C82"
		  "BEEEE0A4B4E6ED3EF2E7395FEF4222F3DF7B032EA28430478C0"
		  "38E45D36379DFD182877F827D86081647DB55B2A05067913DE6"
		  "94F84EE85BCCB1F60902F787CE27407F14C445EB5159BBF643E"
		  "F6195DF2F23B0BC41ED595665D47B91223740709DBC82A86E35"
		  "F615897BCDE5F597FE05047A6CEFD2529394CB85C4945F4188A"
		  "078693C56A6124E040070771A4E509036DF1C9B24E2185908A4"
		  "3E4FF81ADC7D10D5578B00A6DB9FB56CEA6D51590B1CDA9BCEE"
		  "86CBD8263557AAAFB7F81418FB9DCEB0DD67EE6EE7BC761920E"
		  "F784"));
  test_hash (&nettle_shake256, /* 161 octets */
	     SHEX("4E452BA42127DCC956EF4F8F35DD68CB225FB73B5BC7E1EC5A898BBA2931563E74FAFF3B67314F241EC49F4A7061E3BD0213AE826BAB380F1F14FAAB8B0EFDDD5FD1BB49373853A08F30553D5A55CCBBB8153DE4704F29CA2BDEEF0419468E05DD51557CCC80C0A96190BBCC4D77ECFF21C66BDF486459D427F986410F883A80A5BCC32C20F0478BB9A97A126FC5F95451E40F292A4614930D054C851ACD019CCF"),
	     SHEX("083A1EBA0EE14492AF390762E0673F2B41C1179C8616870FA5A"
		  "A69C40236A6CFF19F322A4FB8AEBDD659C5FFB5F1A953E65FC8"
		  "8C42C12E37FBFFD6184035F0F31AD80F7E257FB2E898C9E5DAA"
		  "5A20603A0EA41FAE59E7684646A826C4501F0A7A8C622CF5A27"
		  "D37E998ED954F0DFDA36A86CE59C8A8DF3DA65DA575DF054192"
		  "69438F0DC0F41FB34064E48BC4EFEEC93BBA805B5DC97F7AEDF"
		  "F9E08E3D4B7E2AAF58747255F09A4B53658D7AE61D97EF42396"
		  "161CBC2080DC7BB7392BFB348AEF7F4B2925CDF9F641F3BFFA3"
		  "F140C0142D0E61CBE3D1DF06B016579A748B56C89D5BA88466D"
		  "B79E891CFA7B70532A4D6649C4176FC21531F62F7BAF5D9EF29"
		  "76227ED6EB614DC44CA99F9580C82418FF886E169506F8BBED5"
		  "68FAEA7CE2551346EFCF70FD5C32B6A30572BB72C87C62B8992"
		  "EDB3DB0B4C3EF8CA466B5144658BC4148909BD8D9E1AB5CE3A1"
		  "801D15F93946217CE9D8C0ED536B3C2E75C17B7FEA2E99820C9"
		  "01E80EFEB0FF7FB8226CAD685CD768A41B8CDE2C5CC8BC44B67"
		  "3CE5C44F63840C99E557F3B45C9EC92981CFDB3E20EE040AF13"
		  "A92DF41E8A0877D0559F6FF5D9DBCA2F53A4E2A8B47218CFAD9"
		  "CA6AE3892F21A0D9C0FE91A4F027BC25985947AF0C906537E82"
		  "7B3C79BBA1F37EE38F2897CBE9ABA9438CCE9D972EC262D358C"
		  "33E2E6F2F814F0E28B4709C2ED119B50594BE98F594CCD2B7D5"
		  "6F03"));
  test_hash (&nettle_shake256, /* 162 octets */
	     SHEX("FA85671DF7DADF99A6FFEE97A3AB9991671F5629195049880497487867A6C446B60087FAC9A0F2FCC8E3B24E97E42345B93B5F7D3691829D3F8CCD4BB36411B85FC2328EB0C51CB3151F70860AD3246CE0623A8DC8B3C49F958F8690F8E3860E71EB2B1479A5CEA0B3F8BEFD87ACAF5362435EAECCB52F38617BC6C5C2C6E269EAD1FBD69E941D4AD2012DA2C5B21BCFBF98E4A77AB2AF1F3FDA3233F046D38F1DC8"),
	     SHEX("AF0FF42EE50B300F202648FBE2DC08423944429568354570F78"
		  "7B96531A0045C76189E453369381CB7B0A537B9EC05AA22679E"
		  "96C7BD7CA55793DDC73B6C3FAA3A0D2B03D7561D321FCB377A1"
		  "BA87C098D9FCD1C03D7B7BBA6262687D71B1B608BF148694673"
		  "9A69DBE87B95F0E7010103AF17A6B0C50DB308DE00442641B51"
		  "C5852E91E621133EB561C24655C9D59C22B6AEA470E4D235350"
		  "46FB88105E7DDFC08339E99D8A750B641E81EAAB46A18CAE3EF"
		  "AA493790A9E942EEB6990EB0E14BEB6FCDDAAA0BDC5D67BEE8F"
		  "DDF1931201F18D0B0EA4227D1E89BCEE6C293EEAC60D3A724FA"
		  "6FDDA3AF62D5C4C5200E14E2D40F3E031E71A8DE27CE3D3956C"
		  "ABE9C6A5BCD67C0FE25DBBDEB5720B823777815EC277B594C91"
		  "4F07C989824CF0B1060ED0D4B3ECE3FB65A5AE726412E6EB6FC"
		  "F3047C6326922A0BD1BA7604830DC4D4981FF12B0364435544D"
		  "4836505719EEF8BCD246A326CA1A756193FA0A94EBE98A632CE"
		  "5A0D88E115A4CCD1B030EA7B1651F43F22D2D2A530E2821F238"
		  "0857E916920FAB974AEB69FDD7C3AF313B46F2DBD14D5B4BEA5"
		  "14041E015EC93A6A633BD6477DE78B51E592B3D87A708B831A3"
		  "7B70AC70730C606A7C880CE3FBC11B727E25849F2BE6586D32C"
		  "F7F31E20A739CCB25B62C03C1DB5AFA7B1C00B003DD9C5BFA49"
		  "606830CD5A9B9204E1A51EFA571688496D5A49F4733DAAA295A"
		  "EE26"));
  test_hash (&nettle_shake256, /* 163 octets */
	     SHEX("E90847AE6797FBC0B6B36D6E588C0A743D725788CA50B6D792352EA8294F5BA654A15366B8E1B288D84F5178240827975A763BC45C7B0430E8A559DF4488505E009C63DA994F1403F407958203CEBB6E37D89C94A5EACF6039A327F6C4DBBC7A2A307D976AA39E41AF6537243FC218DFA6AB4DD817B6A397DF5CA69107A9198799ED248641B63B42CB4C29BFDD7975AC96EDFC274AC562D0474C60347A078CE4C25E88"),
	     SHEX("95D216D4FB11C900CB83674FCD99D4A0B0909B307A2020BB00D"
		  "6D6A5590B6F82268D377E255BDDD9179249C3769AB3EA09D935"
		  "7447E99F9C8E924FA8C39B378790CB629C46C914EC79BDB4AEF"
		  "7C4D0ED5092B620A3CB3BF6D19DCB0ED4ECFDAFC9C049E7A8ED"
		  "ACD6C91A3C15D7587C58669E0AC51E151B613503DB2AFCA0AE8"
		  "1782CA58BEA22467B9FE7CF5F095B376BADFD9EBC0BDAEDBCC0"
		  "832D80AECDD6919AB2D31646D9426CC46DA1E5398DD475E94FD"
		  "FBA98129C8997AB4EDE6A204C17CAF9B1691160FA4362909B1C"
		  "1BE81420092A8F5005A3FCD26FE0FCDEAD650665E4ED9CD22DB"
		  "75382D898075BD2E68D240E13BC340C6E4A59AE944A626349A3"
		  "E27F9C82F4DEE5B593CBBEBA0A2BA46A5C0FEBC5290675B2FB6"
		  "36EE868AFEBE35F7BD08F1137ACB12C6D8376CC8E737827F199"
		  "C899B3D8AAA937349CC04B3C64691931DC13F10EC5C57F4D5A1"
		  "3A38E26614302C23E30F28AE09669C4A01F7FAC9FF0B2E10683"
		  "43862364E799C63D74C7BFBC5F7CFFE79C24A58BC2CE1041045"
		  "FB01653157C9945B626B3CD315D9D25CB0B694074AE8F29D372"
		  "067C308F6BF0C44CECBE47E5528E0EF9E373DB63E554FDA5212"
		  "E35898975D708652AD71C14A51C6E5E8A04CD9B7365ABD43F61"
		  "FC7B3EFBE60715574C0AD0BCB0C1D53DBF51172E669E35B19F3"
		  "33A3CAEDBC4E08617C47E51AB3863D61A13A8A2B2D5FB54190A"
		  "0B80"));
  test_hash (&nettle_shake256, /* 164 octets */
	     SHEX("F6D5C2B6C93954FC627602C00C4CA9A7D3ED12B27173F0B2C9B0E4A5939398A665E67E69D0B12FB7E4CEB253E8083D1CEB724AC07F009F094E42F2D6F2129489E846EAFF0700A8D4453EF453A3EDDC18F408C77A83275617FABC4EA3A2833AA73406C0E966276079D38E8E38539A70E194CC5513AAA457C699383FD1900B1E72BDFB835D1FD321B37BA80549B078A49EA08152869A918CA57F5B54ED71E4FD3AC5C06729"),
	     SHEX("C2C5A2155AB4FBD3A3997BD00ED2EA8F049205A6E0668DA09E0"
		  "DAAC49484B0168E6767AD78868347272E6D8C97EEF06AD0A3CB"
		  "9FF91678DEC7385124AAC34817A6DD435A0A88DB8E19A4FC75E"
		  "8F9EB34376E8C1CC7951623171DB22DB1CC5DF9D3D607E5FB17"
		  "E450C15A20A748C340A56142289E0B1EC83CE82D7706DDBA124"
		  "759E895CB7B91522BC46E011B4EB98927D860D09BA48EAF80D1"
		  "FFDF9CCA026B5360AC8C2F04E9B0C30582611B8E8085B8E986D"
		  "28C330BCFE80DCF11246D3544F4AE7C655E46654DC5151242CF"
		  "A7B620C429757B1D527071FEE890C73EABF1BD1630DC5BB4621"
		  "ED4C5C6FF7F41F4BCED4D31B215DD2BE6A5952BD06DADC42A35"
		  "5F628616DB922DB19CEC424BFCB893F948F1364AE7BE1F79413"
		  "A879FA372EFBD59F62FC3879FBA03622CF947D0FD2C02460D1D"
		  "735E41ECE1A2AB970FA9784DC67F0F7C2A3A0A5460F1B31029F"
		  "B65D1A9A6809B4F2AE29F3A68CC2C780D0000F0790998C0FEBF"
		  "AA59A1A56CED99F752684AF6DFCE832901B998156C73E007C77"
		  "BF401CFDAB4E6BC582B6CD477FCCBB1549C870FD6E62633B90E"
		  "D103EDB47D749D04C39D2DCAA3DD4793957EFD12D78B73F58D4"
		  "3087F456833432124E312D499779CDAE6332EE8D884F5719D07"
		  "786A7C85A9642FEC61B66834F98C25E0A802B3F84DDE862D912"
		  "7923E3A20E5A20E184984F8661058D53FFAA70C5ABBBCAB51CA"
		  "8E0F"));
  test_hash (&nettle_shake256, /* 165 octets */
	     SHEX("CF8562B1BED89892D67DDAAF3DEEB28246456E972326DBCDB5CF3FB289ACA01E68DA5D59896E3A6165358B071B304D6AB3D018944BE5049D5E0E2BB819ACF67A6006111089E6767132D72DD85BEDDCBB2D64496DB0CC92955AB4C6234F1EEA24F2D51483F2E209E4589BF9519FAC51B4D061E801125E605F8093BB6997BC163D551596FE4AB7CFAE8FB9A90F6980480CE0C229FD1675409BD788354DAF316240CFE0AF93EB"),
	     SHEX("B3CC34C482E9402EA5A209CCAF5D017BC118F61E52BEDEE634F"
		  "D56C7616E6ADAED42D02F072AB4EE1D8BCCB8038B00FF3072DC"
		  "D971616FD6A2B0012581638F835E5441842631D12FE303D6E53"
		  "3D22EAB7CFCB702E4BDA509C832BBB5C24EC62A468CA885AB7E"
		  "4E3B951C5380A30CB6F132408D31FADDEC570800B8EB58AAED9"
		  "EBD8A0D6618D439B46F83116715568A5957D2DB94E0C2CDCF98"
		  "21D86A71A96C622BA5C0F7C214375686E71F4AE74FCF5A75A6D"
		  "B5F8EB274E6D93B52A2A5110C0B77A1E21E599ED0ABB78A93A5"
		  "469747A6D1FC4E3E3BAECF8EF33580F00D55CDF9124E2BEEDB8"
		  "814C8F5D5171CF6596DE35AFCE94CEC25FE5D6877F40DAC4B2D"
		  "2FDAA0E98FC08DB03FDAFC3A4A2A98D0E865C99B766F55349B6"
		  "88506D7B701BB01BB06F80D6BE98DD374BD84211C61112DD0B9"
		  "A25EFA21F9804D724154881906A7BF815F206F4AA1175348212"
		  "C54538BEE673B56582501A659061A157444DCE84C226936E99E"
		  "51252418498DFEB02F2DC665B8CC4622C0326978FA7A7A8D171"
		  "D445CEE519A32D81721F99DF529614BF09E4273B24A6809CFAB"
		  "9A1E81E5280C84ED3CBA6D4B232080BDD034F1895F4010BAEF4"
		  "62074898949A1CD5892AEF5DCB2B09D46201509847179505980"
		  "A2846E58DDB22EA9FA0FCBE3AD9DFFAAA42F4FEF4A9DD839149"
		  "5BA470A2EB80D5992DA78DCC8E893C1BA2C909CF8208A18C31E"
		  "BACC"));
  test_hash (&nettle_shake256, /* 166 octets */
	     SHEX("2ACE31ABB0A2E3267944D2F75E1559985DB7354C6E605F18DC8470423FCA30B7331D9B33C4A4326783D1CAAE1B4F07060EFF978E4746BF0C7E30CD61040BD5EC2746B29863EB7F103EBDA614C4291A805B6A4C8214230564A0557BC7102E0BD3ED23719252F7435D64D210EE2AAFC585BE903FA41E1968C50FD5D5367926DF7A05E3A42CF07E656FF92DE73B036CF8B19898C0CB34557C0C12C2D8B84E91181AF467BC75A9D1"),
	     SHEX("1A2EB5A7E3266CDADC696D082DBAD5D9F73777F5B95DAD3E9C6"
		  "A936271F12762FF3E8D1D959ABBAC6933BB19D2A0125AD59477"
		  "611F9F912F935D662ADE6E1E40412BDE3E00A99C06A4906B908"
		  "55AF6095DD7E6114EA48DDC1A76F3C8FFFFDE5AC246F5C0F854"
		  "831EE7B48DF6A32CECEC0E4243A792CE3FCE40BD5AB22838DEC"
		  "6155C817D272F3AD602F58EF870D0126EA18AFD1F1527931C32"
		  "1A8D6DA1BF60665D077AD15EF615584A249ECF7D3728C0E9D3F"
		  "C09BF8D3FBB9D260FA4FF13BBD4DBA6835590D67EABFAFB92CB"
		  "3B0435F26E7EC265F7E2077446E1580FD1747E871D7AD1E992C"
		  "1A909E068CA3F8B8A927053CF31EF7E071D5C7B4044F2CF61DD"
		  "D48E7B7115AE0CC2861DB7E26302DF02CAD460358D06ACB7E2D"
		  "07615C0F3786BA02000893FEBD088584ACA9FBE9F14F85C8894"
		  "3671D24E9CD0B8FA46601C2E40D3177EEB500E8AA2931473DC3"
		  "BC46F71C749C281E02B67D8F0B3D012DE4942B1B3C60A2595E8"
		  "45CA454028489417842D9A6C4706424C5C0A073BFD080E5F820"
		  "CCCEE99CC4FD372B0BBC447C5F3F54D87755C0A59C986D98E31"
		  "6F8F1899821CFAE312B9416A4622786B2910910B86D19692C8A"
		  "7ED50D63E252A3A5655D6E49AF3082BDBBD07690DC6D4EA42F8"
		  "A7BC57DB816929B9FF8375C4596F57D9359F9FCDF8A3FAB33AE"
		  "27624265296A093DBCBDEA292450D345B8116B86FC54A0B4B55"
		  "F12F"));
  test_hash (&nettle_shake256, /* 167 octets */
	     SHEX("0D8D09AED19F1013969CE5E7EB92F83A209AE76BE31C754844EA9116CEB39A22EBB6003017BBCF26555FA6624185187DB8F0CB3564B8B1C06BF685D47F3286EDA20B83358F599D2044BBF0583FAB8D78F854FE0A596183230C5EF8E54426750EAF2CC4E29D3BDD037E734D863C2BD9789B4C243096138F7672C232314EFFDFC6513427E2DA76916B5248933BE312EB5DDE4CF70804FB258AC5FB82D58D08177AC6F4756017FFF5"),
	     SHEX("1E67E9FBB0568660EBA48D1C701D75653D97D9A94E770C355E2"
		  "F3F6DD9B7C3CDC771FE82EC87BEFE4AAB21BA7C689FACF51593"
		  "CB278F7A8A4D81B9CE798CC2FBA50145EEB0DE079F016362FDE"
		  "491620CB0A3C26CB5D5FB09980E783774C8FD2C9D8900EBB103"
		  "D73F0BAB02D0A8E2827D2471DB29F3D6EE5D9D620D4D13D8925"
		  "F1933B1C66869BF74EF967204238E095927961ED5925D1A271B"
		  "9277AB9DD7A30DE1FECD4C6BD9615618BF79850EE7E247121C9"
		  "28AC9BE6F81D2B41378E984F4BA494180D7E45CB7ABB6DE308D"
		  "B53804BCF43DACC10AB3A98157285FBAC9B183AA49EC18E98E9"
		  "B40D12B022F40B213641CB2E18E3303E4C9078DA836466DB8E2"
		  "E85962E5520E9FF66CF99F96E7699212816124313E5BA6347F9"
		  "9DA7109940081419B4EED3F4ED2763D655A00EC6D183CA87907"
		  "4F237BBC49D40D598FF2BD8F47D0D69F2E41397042B4398D4FE"
		  "449CF553F6096BA25C1A2185A13BCE5545FD0DEFE4D11168BD8"
		  "1B85CA5B0502233C43B6E863447578EB6F074EE2EEFBBE6C52D"
		  "0E8B1046F304DE2CAF6484AC24783DD77774F0FAFEBAC731E42"
		  "9DBACB0D95BE662ED9C71FDD6C0E6595CA063C748ECFF992064"
		  "41F8E96C96063EF61D741B4D622B6EF6A6D547B738AB27B50FA"
		  "D2EC5C479EA8069B9D67654C2F12D8232AB8A97C83AF190B8F0"
		  "9E2B8F62B7DAA334DC8F26A3EA62A999433FBA70A27B9CF7FCB"
		  "61A2"));
  test_hash (&nettle_shake256, /* 168 octets */
	     SHEX("C3236B73DEB7662BF3F3DAA58F137B358BA610560EF7455785A9BEFDB035A066E90704F929BD9689CEF0CE3BDA5ACF4480BCEB8D09D10B098AD8500D9B6071DFC3A14AF6C77511D81E3AA8844986C3BEA6F469F9E02194C92868CD5F51646256798FF0424954C1434BDFED9FACB390B07D342E992936E0F88BFD0E884A0DDB679D0547CCDEC6384285A45429D115AC7D235A717242021D1DC35641F5F0A48E8445DBA58E6CB2C8EA"),
	     SHEX("7204F8652F37D125BF692544B8BA0C8859BEAE66E4B04ADA568"
		  "5C6B4C3C1B8A3825B2AD6BCB2F5443B4C28ADF57388FCFF481C"
		  "A629934CABF872354E4A33942B738CCD4E1941621524E895188"
		  "D2ACC9EFC69DE7A170BC974C430B9830E25DF96097B3785A2F6"
		  "B86F39E59574E1D9C2B91EED2231D4D1AEFBD5DCEE3E5FAA424"
		  "72949D0D19DA8980BFC6276E41DDD600DDF78E82393EEE5A9AE"
		  "0C87578071D667B886EABBE6757A52AF9826F65E7CA80529825"
		  "23F61184B5892465B3F82BAE68FE2FEC1577EB935309EDE8041"
		  "63DEE46EFBF5C93E7A9DC3BA79CD9A866966BA1FC0A721CA4AD"
		  "17364C3A4033E805616FAA6075672DD6FAE31A67D891646B74D"
		  "D8AA916E078BA7736A9B00F73DF90B09732A42E38FE3EC9F8FC"
		  "907A5868CA170C669ABAF99571D14771B91F46C68F57DC6215B"
		  "94DC0BB03F890BDE02DD41CE5D0F8F48E4FDAEAD1F0A05DA9A4"
		  "56DA80B82F44EFA53DB9899F42BA31CECD9D7CE6A5DE33B70DD"
		  "6427D3A9B31C83ADEE1EE073E06EC4238FEE4EA0029876FE6EC"
		  "A5DE76E4D32B65564EFCAA326AF3519EDA46EB3E443A85E78A8"
		  "E26D21B158328A56AF40B07AD19734E341A45A5F43D1EC2C9EF"
		  "FBC7C5DD92ACEF61607F3CFDA8BC72D33C045A6B883972CF4FD"
		  "1282A0ADE3978D1803EE78BC6F6FD297EC9E3052460483DBF79"
		  "E6C35AFDBC10FA87D769AE6A4E2849AD112EDDEC1135CDEC2EC"
		  "FB6C"));
  test_hash (&nettle_shake256, /* 169 octets */
	     SHEX("B39FEB8283EADC63E8184B51DF5AE3FD41AAC8A963BB0BE1CD08AA5867D8D910C669221E73243360646F6553D1CA05A84E8DC0DE05B6419EC349CA994480193D01C92525F3FB3DCEFB08AFC6D26947BDBBFD85193F53B50609C6140905C53A6686B58E53A319A57B962331EDE98149AF3DE3118A819DA4D76706A0424B4E1D2910B0ED26AF61D150EBCB46595D4266A0BD7F651BA47D0C7F179CA28545007D92E8419D48FDFBD744CE"),
	     SHEX("A3D5CFCDCC03334632027FA1B1ADB97C740F9C5D33F0B6D8468"
		  "B4AA9B37A5EAE697D6DE7717B6F7C3E872DCC4A62682DDD76D8"
		  "7657096D143DEDF97F2B9BA36734B9FF5816E247243B1F32F5C"
		  "A122AB49647FDA690B83AE0FEE45625B4A2606C8E366CD5031A"
		  "0C938407CC9421414CE4631477C3D4494570916BB41C60FC05A"
		  "C125A3E81346759DCA0CEBDD763B61493997B774A582475D226"
		  "1F6F8500D2C51C70DE11123756EB6B958EE5F20FB4A49429358"
		  "D743E4B62D76904D23FD5DAC9ECBFF14854BF27DDA819B3F524"
		  "21329B0576CC399EAC734D73FD9BB8729168E35E2A3490D7BFA"
		  "A39E53E54340457AEC4C5C8F4C0FEE9339053646E4D74EF1A14"
		  "06EBA208822B8E66DE410CFCD49A464D9FF545604C26CAA4FE8"
		  "4B018C69BE18112B5C3D7325481078C712979DC88842E2842DF"
		  "19F39025D28FDF45A5DD6E8FD2D12BF27D227E79210F6266B86"
		  "CDC7BC6F81DB57AF57AA252C5BB95A235746B9C869BA7F8C90E"
		  "0AD3F5DED409947173D071DE216ED1B137846752653CC6CFA3C"
		  "52A32AD6CA0BCE29A5B12475C049488F0A79ADF5ADC4510E646"
		  "8E714BBA37C0005A9CFA3DDB24363290A04BAFBBA92BBB2C2E1"
		  "6CDD9D40707C56F49C05C5276C89BEAC3B239DE284867611CA3"
		  "DD8250E8C7FDF522E0E84EA47A3A06554DFFF6B7DA13349A5F3"
		  "6CA447499C45DA4C491DFC113892672B036507FF3A1E7E07B43"
		  "FB95"));
  test_hash (&nettle_shake256, /* 170 octets */
	     SHEX("A983D54F503803E8C7999F4EDBBE82E9084F422143A932DDDDC47A17B0B7564A7F37A99D0786E99476428D29E29D3C197A72BFAB1342C12A0FC4787FD7017D7A6174049EA43B5779169EF7472BDBBD941DCB82FC73AAC45A8A94C9F2BD3477F61FD3B796F02A1B8264A214C6FEA74B7051B226C722099EC7883A462B83B6AFDD4009248B8A237F605FE5A08FE7D8B45321421EBBA67BD70A0B00DDBF94BAAB7F359D5D1EEA105F28DCFB"),
	     SHEX("23397CF5C6041FCA9F1503B12A47F1389C4C35301F1747D574B"
		  "48DEB270CC2FD79DA5718D0D5ACB78A7F30782617AA2E03B9F4"
		  "6736620E512A6AA4E55E2B94537A74357E5C1E6216BDA724677"
		  "BFC4CCAC3E3DCF4AE21D7E56D55ED2EFE6FB50A2222E98139DF"
		  "C739FC759BE57963D8427D0B5D2093CF5F42347A6D9832E1FEB"
		  "B4576935FA18E93DDB20E8CEF2F2EBA33C66F1E5741171BB64C"
		  "49F128A7B0A9F095A35F5A20A996A3F9D7A68AB3B0D0F84E8F4"
		  "FD43078A80266377D0D7020DE67E18DE20656BD59B86726C99B"
		  "1DC8FA25B83A1FC8B7C256DDAFCC67540C1287CC377AC10D392"
		  "88CE00839AF31A1C078B403A863171CC669DD72ABF4D48CC72A"
		  "F222FE939A2D75CD7F195D3BD8CF8069F6557F1FB37BC8F9B67"
		  "7865E0D23BDD29C9B240CD2993874FBDD5FD3B8CC57B6226C40"
		  "890057994A2E13129F4613D3ED31EA5804006F5753047F1128F"
		  "9B8C0A7ABA719A1449B3A2D02ACFA0877E81306179F9AB717AC"
		  "84819029006D73D48EE55CA13A3D39ED293AD0AFC8EAC8F8D41"
		  "971242877788EC1286ED35F06330683FE33042A6255835CD54A"
		  "A07C0166350B851F98F2AD2A06FDA56B180F11B9616BB51ED98"
		  "A2FED5BC2F9EEF7046A38754B91543EBE97B8375A81B7FE9CE5"
		  "8DE4DCAFCDF8183541E44CC96F855CAD024C24EA87313DC05EA"
		  "A1D933D6E238E9B3A0B9628239344E8E81B5341D615B63D881F"
		  "FE45"));
  test_hash (&nettle_shake256, /* 171 octets */
	     SHEX("E4D1C1897A0A866CE564635B74222F9696BF2C7F640DD78D7E2ACA66E1B61C642BB03EA7536AAE597811E9BF4A7B453EDE31F97B46A5F0EF51A071A2B3918DF16B152519AE3776F9F1EDAB4C2A377C3292E96408359D3613844D5EB393000283D5AD3401A318B12FD1474B8612F2BB50FB6A8B9E023A54D7DDE28C43D6D8854C8D9D1155935C199811DBFC87E9E0072E90EB88681CC7529714F8FB8A2C9D88567ADFB974EE205A9BF7B848"),
	     SHEX("3753866FC2D045059FA925412D3FF0FD1254724DEC38E00F109"
		  "6AF36D0A7603903089900F95CF9BCFECEC4CEA14B7D80CB324C"
		  "56E60569AAD9E0F45D3A726F87E6F18559701EE6AE0E7A8622B"
		  "45DCC6991A1FF15DE6B78BBCB96F976089F26A38EF190EE0C60"
		  "9DB54F443F5F1014762F336F62EB5F7CB7B102E99FABD87F36A"
		  "ED3359EB0DBF739DF4EEC4AAC458546C00D16D412841142ACD2"
		  "E08CAADBDDE86843ADD14E3D28426295762E564E1F4854E7C61"
		  "7F48EB6F2FAC79FFA8CDDB09FA09A8DD73EC389285036A52440"
		  "41DBD60AB71F8FB59EB73629BB4A827A7584B20C12B96FB1F0B"
		  "950BD3C87117DA69C7EDA0C1645B6C8C81DA8A6397DD821A983"
		  "DE84390B1CA902F135160DB58F9F7B7AC9D010CE8D6BF713627"
		  "0FAE0D3994F80612BB12916993AA73F22FAE642784D75E1CB5A"
		  "CDBCC20A3AB148D6F3F171402391121BB404F8F576EEFC9ACD1"
		  "903BF9AD2CD4EB1208F143DC84643E52570D406BA31F8DC9F32"
		  "83B2691F826C4ADBAF431EB2F1DE1E860F9B4C0153C9F002C80"
		  "9822875F915F7131EF63262ABBD8BF4A5D9D1699D893435C731"
		  "DE48756C8D03C94D11D1D484DDDBC5C35660B3318468FEFD827"
		  "78CF907FF52D652BCDF68C20349F64006C640E7A544DEC1FE17"
		  "09056FFD9EA7D41590F84F653AD4F7474D008765BFAEFFBA015"
		  "DE99D523A56A6C989BA11AFAE097F4D5FA63F04AE0917212E4E"
		  "AF48"));
  test_hash (&nettle_shake256, /* 172 octets */
	     SHEX("B10C59723E3DCADD6D75DF87D0A1580E73133A9B7D00CB95EC19F5547027323BE75158B11F80B6E142C6A78531886D9047B08E551E75E6261E79785366D7024BD7CD9CF322D9BE7D57FB661069F2481C7BB759CD71B4B36CA2BC2DF6D3A328FAEBDB995A9794A8D72155ED551A1F87C80BF6059B43FC764900B18A1C2441F7487743CF84E565F61F8DD2ECE6B6CCC9444049197AAAF53E926FBEE3BFCA8BE588EC77F29D211BE89DE18B15F6"),
	     SHEX("2381AA1E335043C1F7360D64C703FCF6190747A616C550F787C"
		  "33A573B2F9E83DE427AD8EE50610A2FBB07AC92E4F912B3E5C1"
		  "0001044FA2CC3FF1C2182C672BC03183AF502D6530443D890AF"
		  "4571611DF65A2FC50879D08E9D664D78D5F81D74957D38726CA"
		  "EE5451AB88DF8853CC4A034065C532346A4F0D548D65745AB64"
		  "9683CFE185C649CEEA12D79E0904E2C059549F895DCB1ED12D1"
		  "CDF00D530D4653B56EF9FCF8D3BCC70FA91397B87F34FF839D3"
		  "C0DEB053D46FDFF9262219004961AF9D03F50E1016E737605F3"
		  "5EDED16ED51F4AA8FFE0A253AA21011DC002C5C0E0B1AC706E5"
		  "3DBFDDCB1B02C0148C3AFA9D69B6CF7A72B1F793EDAC9A99ACC"
		  "99F988EBEE210EE108093CE8EED146FBD98436299B159760EDF"
		  "3D07BFEC6A9C1D844E98DD632B1E773D93B9604CB635457E33C"
		  "063991D2A6D4DD8AA72883FDD742E34ECA12975AFDBFA2EB997"
		  "609F91CB4BD53CA40BA4793B9849ABD509A2B6C01094400526C"
		  "493FF33E0F8F1908BBF783E0DFD92105830947338537C9F7B6B"
		  "2D193CB2923F95CAF113C9248C601EB39147661F7C639FB0528"
		  "1E0728B916C59D98F36DAEC1F61FAD84E7292B6A5821009FCC3"
		  "0EE7267969243B36134675F3C8F1F18E9C341AFEB5377426BB0"
		  "4517D1498FA7A271029936FFBBB4248A786EAB072387B991222"
		  "017025BB84EB4DC084781F21251CD8D8CCEA15EA2C37C89632B"
		  "B67E"));
  test_hash (&nettle_shake256, /* 173 octets */
	     SHEX("DB11F609BABA7B0CA634926B1DD539C8CBADA24967D7ADD4D9876F77C2D80C0F4DCEFBD7121548373582705CCA2495BD2A43716FE64ED26D059CFB566B3364BD49EE0717BDD9810DD14D8FAD80DBBDC4CAFB37CC60FB0FE2A80FB4541B8CA9D59DCE457738A9D3D8F641AF8C3FD6DA162DC16FC01AAC527A4A0255B4D231C0BE50F44F0DB0B713AF03D968FE7F0F61ED0824C55C4B5265548FEBD6AAD5C5EEDF63EFE793489C39B8FD29D104CE"),
	     SHEX("325A6883E6918057ACC4D7E6863D245ED59957957AF4E8E59EC"
		  "6ECA35EB380641BD39550268A68EBAD703A51F9B47DD05CA256"
		  "81BDC853EF0897CBD4A0DA6BE9E911C26E801DA8F5360140FEA"
		  "2EE9276747E1AD0879FD741E52A7DC8F0EE3CB199826F9C1D39"
		  "7E150641E8EC348300BA4F7EA7746ED0E94520577FDD253E04C"
		  "47223521661A308D1C996A90DF8A9AD18418193D590E75F0F17"
		  "FE0443B5B130C1AABE9F60E538E6193A19690368A2C17516D7F"
		  "EBC3DF95DED8445ECC260BA46156C88B5218E582FEE9EBC4F28"
		  "CC4171936F7C5BEF008D7AD76A70BE3CD3F0701791702A23716"
		  "0798223EEFBFEED7BC108E9C793CC42F93381BB9D5F97103499"
		  "BDA63610303A55088FEF10FEE330C4F35D0A4D0F36C1CA06CD0"
		  "8496DB0D06553BD612848212539DE0D698A530DC287F2FBADDE"
		  "DBAAEF0195E050E3968850FE8E1C72F8E3F11F24BCAC475593A"
		  "A28AB2CC69C3FE6F3ED03A38D8B278FD678697A3F6DFED6CD52"
		  "AFA4E9947CA56D1CDF10ECEF90BA6EF87D9C2765F2777C14323"
		  "25C464C685557129808CE3B91310C301547B673D498A4105833"
		  "4562074A3BD3C64FC8B02BEE8EA642B12783ED71E990CA90B95"
		  "B7020D5458FFAC57BD93F882F3EADD4707C98591CAA667BB0DA"
		  "EE9880C3549FE9DDF7C9314387A85D308516C64FF98CCE50E25"
		  "3CDC710927DDA784DEEC547C4442A0818045696F4D4BA4E0D72"
		  "9D13"));
  test_hash (&nettle_shake256, /* 174 octets */
	     SHEX("BEBD4F1A84FC8B15E4452A54BD02D69E304B7F32616AADD90537937106AE4E28DE9D8AAB02D19BC3E2FDE1D651559E296453E4DBA94370A14DBBB2D1D4E2022302EE90E208321EFCD8528AD89E46DC839EA9DF618EA8394A6BFF308E7726BAE0C19BCD4BE52DA6258E2EF4E96AA21244429F49EF5CB486D7FF35CAC1BACB7E95711944BCCB2AB34700D42D1EB38B5D536B947348A458EDE3DC6BD6EC547B1B0CAE5B257BE36A7124E1060C170FFA"),
	     SHEX("0E479E6C1C78B597DE67FF30F0BE1D251DC86FF0AB0F4D5C6F6"
		  "4BDEFDCD6A77DAF29B772FCAB3EED9344D32BC3ED616C10F69C"
		  "F4AB3B34709F3941F3951FCA1A173CA2CD707FA41B8E4586343"
		  "77596865BEF5CC8A3CF52C0ECDA7CCC0DFD8AA8CE0CD7CC1917"
		  "A7BB9A3BC9084AFE456E5DC02569B3FA4D9B9DA03B8E4B223F7"
		  "CF033C1A8BBD6002B3A457C0DE35A222A30A0E86F3EF9C9F255"
		  "D449CE4EF5AFB51577391574F8271E07EA98820E0308EE56B1E"
		  "EA91B35058030AB6AFDE356CC83D526BDAE2E55B1AD4E4D8016"
		  "FEF14247BC7D95C34F06DAB7D9FFD1A9FEF761AE01F8DEE1D46"
		  "75172C1F0CA15361CBA994EF062596BB767C52A275792F48B19"
		  "1B1078968C1793A9D274166DB592267E1822585FCB1A9734D4B"
		  "503F5E1CB0754A7D9F47359CC91E0646C4A2E754A29584363ED"
		  "4C8C806797FD102D6220DE1814E665409104D32BE7A346D3B12"
		  "6FCD51F55E5625A6BCAAB74F0E69BA1F75A4D5C4625E2F5AE12"
		  "4CEC4EA6A702858A3B03CBCB0B014D3D841CD7A87D02C622F96"
		  "AAF3AAD96A62939602A2AA1F9D88DC555038560217BAAB657EE"
		  "87367E05345E1612532F9B6D83043DB59837E1F75D82B8149B9"
		  "3F1105BE13BF51E20F18CECD5DF87067A0716D3F860DE8C998A"
		  "487C8A197149E7C8AF4B414217B8579C314F698FD6A3B612B2D"
		  "D67FF6E3471856FD9F66799FCEE14627293EA94D1C32D915CB8"
		  "1C5C"));
  test_hash (&nettle_shake256, /* 175 octets */
	     SHEX("5ACA56A03A13784BDC3289D9364F79E2A85C12276B49B92DB0ADAA4F206D5028F213F678C3510E111F9DC4C1C1F8B6ACB17A6413AA227607C515C62A733817BA5E762CC6748E7E0D6872C984D723C9BB3B117EB8963185300A80BFA65CDE495D70A46C44858605FCCBED086C2B45CEF963D33294DBE9706B13AF22F1B7C4CD5A001CFEC251FBA18E722C6E1C4B1166918B4F6F48A98B64B3C07FC86A6B17A6D0480AB79D4E6415B520F1C484D675B1"),
	     SHEX("0A5995025C3DD9437884196FF09C0B92B5B0B51B59D0C3C9041"
		  "AB172A1685DDCB3B0324186F11E2D7DCB6851F888ADE6805249"
		  "7BD16C4F1D98DC468D833E4AFDD2A9E3DADCFD188C9B1E35BA9"
		  "FB9549F88C8C7C74352A7B420C1019D2E7CDF02601E5609635E"
		  "AD96CC57FE29A5AD98B2FFF301B3640BA441CBBED877477C0DC"
		  "1D6143454456C0EFEA8C4038281E97C02774A7ED709A446941A"
		  "2819F3372E18BC1A6C119EB24CD16A623CBF91816CF2921D2FC"
		  "65DEAC91F3E0EF24F97CA518D167F295454AF8208AB25540228"
		  "A9AB1BFD0ED0C8D09AE579B7116389B9BAA6EE388612F323D1E"
		  "1627D1667F4E524C562B0993A807BAA0E81D02241049A4C0BC8"
		  "EC95701E9CB045A91C4340D8C3067686347867D4FD94D48C566"
		  "672DA8C89468D71B488F7D5E1409B3F1AA17B019C57D5A94C01"
		  "53A887EA366253DAD36E9ECBB20DE67E5F09F4516DAF9376170"
		  "0BB4E094B88ED56CF935F9D0E5454D2168FE5F5CE0682838386"
		  "AD55B4C664526B48138EC270BE30135ED84A63FE04CFB99DBCE"
		  "6B70CBDB32110F954491044970D3AAFB62B5123F2B43F3B8275"
		  "CD2D1034068864A94A1B63532F31D2E65B9C197E13199F86638"
		  "808D17DC9E3D914AC6E078CDE60A1EF9DB9F94E56C29F385418"
		  "AAB5F5D5CC324F0B1E2B9F9CC9259971FC57A2BD18686045E04"
		  "A3A74C3606B8EDE2E2E49E2B3F01CEB4B4516E695388AE331CE"
		  "C186"));
  test_hash (&nettle_shake256, /* 176 octets */
	     SHEX("A5AAD0E4646A32C85CFCAC73F02FC5300F1982FABB2F2179E28303E447854094CDFC854310E5C0F60993CEFF54D84D6B46323D930ADB07C17599B35B505F09E784BCA5985E0172257797FB53649E2E9723EFD16865C31B5C3D5113B58BB0BFC8920FABDDA086D7537E66D709D050BD14D0C960873F156FAD5B3D3840CDFCDC9BE6AF519DB262A27F40896AB25CC39F96984D650611C0D5A3080D5B3A1BF186ABD42956588B3B58CD948970D298776060"),
	     SHEX("9FF0E63D6E1BD2C9B671298EF08FB86024EECF3A41662D465D8"
		  "011E44C55E4312A2F4680A92984471B885FD730A3C3AAA6329C"
		  "CC06E9684350543475D8CFE2F18079DAFAB03B9AA1368AAFD26"
		  "5C5ED3D0D160EF317BF5CA211642F7C86D4C59504C8E6D8EF5D"
		  "52C7770659C91AE1121EABCE4318D58F2644DF56D18909EC977"
		  "D75D27D25D291EC706A39A3EB13E49691F6C3188C300B827E3A"
		  "D1F6F8088318DA476D07666AE4192C08D4AB797E53445239CEA"
		  "485704F0266C49ACD7ED67D24DA7333AF799B40B301AAD88837"
		  "03F03869628F8F1513629FF0888656782C23D43D435962A022B"
		  "D19FD7E4AF8BE40CF34ABDB1D0DBAC698079312CDFD5B01D166"
		  "E6933DE574914A62472C5B2954740A5A26217E9A67329CF4720"
		  "5701D8BAF5E660E231E16BB87EFDFCCA4E982F44CAD53B2893A"
		  "83DFE251ACE8C49B225C3E929221D4FBE452CBBEE2F6E59F786"
		  "8F876E20AF55F8B1DCB4AD5CCE9F0B8966D51128D08698AD8FB"
		  "267D3164023985F5F5E48A237F4B638437E074FA4CF83030DA5"
		  "846E64E03A23ED2EE12EE42646D1075B8F91B861D6DAA8BBDB0"
		  "7FBD56AC72FE8675031C1CD0622AFFDFC9227F4DCBB642146F8"
		  "7D2930B859181CF5AD7D6AFCC87BB905AD387AC8B8A7DA70CE6"
		  "26CBE3E13E115572780D38DCE4F3AA6A95641842109B9750AD3"
		  "FB5F711E1F1363119BEA84B4589D0B7287BB4CD094564DB6325"
		  "1CFD"));
  test_hash (&nettle_shake256, /* 177 octets */
	     SHEX("06CBBE67E94A978203EAD6C057A1A5B098478B4B4CBEF5A97E93C8E42F5572713575FC2A884531D7622F8F879387A859A80F10EF02708CD8F7413AB385AFC357678B9578C0EBF641EF076A1A30F1F75379E9DCB2A885BDD295905EE80C0168A62A9597D10CF12DD2D8CEE46645C7E5A141F6E0E23AA482ABE5661C16E69EF1E28371E2E236C359BA4E92C25626A7B7FF13F6EA4AE906E1CFE163E91719B1F750A96CBDE5FBC953D9E576CD216AFC90323A"),
	     SHEX("9EC7D516BB276EE7BE68912E86FCD71EE08EA4BCB5A44B9520E"
		  "84F7433811160E5A742BF8E444329F4FBE22D72F002F82459DD"
		  "538D7C26401399B8882463CBCBF2457A7080F8ECC02A05F357B"
		  "AF6BF8DE31984ABBFF66AD082CB995A180D7455BDFBAFAA83B7"
		  "4CD95488CD8F5CFCE16EBB2D9F08E54BA341345648D0FC55700"
		  "2487FE6F0D0418858698CAA9FD4171CCF3880FC0A9E751BFB56"
		  "6BA907E13BB78A19E7D0CC543F9B7303C74BD957149BC505530"
		  "B246A8D41638023B83B84BEEC79A91CD21D3982B7E57582780C"
		  "92BE8CA813218B287C280A42E73DEB3A84F1027010F79EA2CE9"
		  "D4EE57DE696DD1D4A13F099E4E1CF4BC7CCEAE2C7A454775628"
		  "D09FDCDC55C6B38F545566682D3546A6F6AA8B57D44EDC36B0C"
		  "4B9C1B92CC2465B6BC7091EB78CA0030DB7D7E31805AB459040"
		  "494DF1120307A2DE8315C3CE802491EBB0FF3F83FCF2F99D9F5"
		  "69D3E3F32F0CAF2D7D6057FFB6183D274919B9D4B4CBEB125C9"
		  "002A816A83941ABB5120AD9AF40A76398D31B07E464482FAAC7"
		  "67BFEC63CC221DB2A54860BED4D5E9405821CB176B47838249C"
		  "689FF1D9990CB3CEC4EF1A0D9280F35E8F23246537313B77F26"
		  "D517221F0A21E7E6D4D28D88632B44D7F1C381F8E7442A71C0B"
		  "0473F7BC702326364489A943B6A0CD0A8B868F21D7F26A1AA3B"
		  "8C7A4C7CEB574FC2B266CD8067D83A53AD469DFFCA9088ECE43"
		  "9207"));
  test_hash (&nettle_shake256, /* 178 octets */
	     SHEX("F1C528CF7739874707D4D8AD5B98F7C77169DE0B57188DF233B2DC8A5B31EDA5DB4291DD9F68E6BAD37B8D7F6C9C0044B3BF74BBC3D7D1798E138709B0D75E7C593D3CCCDC1B20C7174B4E692ADD820ACE262D45CCFAE2077E878796347168060A162ECCA8C38C1A88350BD63BB539134F700FD4ADDD5959E255337DAA06BC86358FABCBEFDFB5BC889783D843C08AADC6C4F6C36F65F156E851C9A0F917E4A367B5AD93D874812A1DE6A7B93CD53AD97232"),
	     SHEX("045E2BEF203B8E72121FC29E68F01176B8AD2E0F24352FE555C"
		  "9F0B70FFB38AAFDDAE1B2FA04C3BC2DABAF4BF3BACDF7658A62"
		  "3446FD6840536572EFF9393F5AB66C080A68D2341AF34AA2A13"
		  "B6EB57D8DC6CAACCABAEA593970B4D91A3B861AEE0B6E53F326"
		  "3DA68DDF75CDE76E5BD94AFB4AC78ABED89788BA89804D6F199"
		  "7684CFFED40B3761A782E3EC1F1A1FF12F8151E91A935A088D2"
		  "AA2311C43FD731CEF31503C775781EF5724508B910976DD89EC"
		  "BFE79B17F1813B01B82B6DCCEAADD6615CFB8D2EEA27EC7377F"
		  "8911A39E9A15E622F3A91F88333811CDA86007E57EE652797BF"
		  "9177C898CD8951B8C123B8188BDB2E60D32493F4E94C34B9FF3"
		  "A00893795AE0028061050E9C03A53AC787787A332DD4C75FECC"
		  "1B7AC1E6DA30E69A46DC94C87368C2150EEB3717582D5E5851B"
		  "B5695CCB416E4D8462F0448E0D711B42F1B6FA0DBAD40E96C6B"
		  "69E67BB1907B460DC319B8F01591867C73655F8A28F594EA245"
		  "8E163EF0D562B36580AABC8E9FCCE61D09CD83BC4DEC826C8F7"
		  "DBD76028588BC905777AB6B5A5DBE81665971B23C94DBAE8B51"
		  "3AFD7DF0B6C83469ACD22CBEF596359BA0203C6F320983A5527"
		  "4812B49E7663A6F48B420748AB10062EBCA4DCE4440D1EF9B72"
		  "BF4E121B76075D209EB82DE30C074CFFF86CC28E464A6959691"
		  "C66DB995C279BF50CE2ABECB94CBC850A33315E4ECF650FFC27"
		  "1DAD"));
  test_hash (&nettle_shake256, /* 179 octets */
	     SHEX("9D9F3A7ECD51B41F6572FD0D0881E30390DFB780991DAE7DB3B47619134718E6F987810E542619DFAA7B505C76B7350C6432D8BF1CFEBDF1069B90A35F0D04CBDF130B0DFC7875F4A4E62CDB8E525AADD7CE842520A482AC18F09442D78305FE85A74E39E760A4837482ED2F437DD13B2EC1042AFCF9DECDC3E877E50FF4106AD10A525230D11920324A81094DA31DEAB6476AA42F20C84843CFC1C58545EE80352BDD3740DD6A16792AE2D86F11641BB717C2"),
	     SHEX("7FD29D970989B8F0DAD5286CD8CE7F5E793B80EF8F62506F75A"
		  "119FB3ACB39D726CDBE6F4914F8A5930D30A0AC1E36E285AED4"
		  "90CD30FE63A2713AB0DD473A7A764A19A70BBD9AD5BCB2B2A0A"
		  "D63A84FAD80466D1C9F513D4336E3D6C7F93645C3FA30AC5A54"
		  "B46205322265D3EDBE4C8C58B5D8AEA1856D2FA81C81E12F27E"
		  "ADE6958F014ED3FA3C844A65321EAF484A2E7543D362627F2CB"
		  "C4A83415FA3CA616B9CD7EA8CD1724F426C0161AE41CE8175F0"
		  "598FE4AAE91A3F12D6C8659B063FF5C64FED548C206329F0C5D"
		  "A4AEFBAD704D16A1B67A38D807E64F03A3A9DAE4B464C78462B"
		  "842B0DAE0036EB466AC7CBDA234F2DE626D14764F6F49A8AFEB"
		  "062402CC9DBA983268333225A6678D2FD9E506ECE66135E6AFE"
		  "7A9DC3A16295F71440ED04E8E391EDC134E91C1699F0D31F81F"
		  "CA6105F485566C13A93A2DB218DBE8D64F4B2E242462A67FE3A"
		  "98A785BF276E510ADA824E88C5ADBD9889EF2857D5347544E43"
		  "1603E717EC7A7D17C98A7C326C0A4A83955C7EE31379B241C2F"
		  "2ABE038B2357D3A9D940D875F2A5634CB47255DC92FF11AB7DA"
		  "C4D675E13134C24FF7B058C2F99C985840287CB3009B6CC3D75"
		  "AA5E79C29C68DFFD9D95E37592B6036A4CEEC5E34EEC8DE7EBC"
		  "A4B80F0E103CF07D46E73734C831C0CC2B6E31C14F304D88D77"
		  "D433699FFEA902058F8F4ED0FC7AFBC4874CD6786E6D82A9689"
		  "3A68"));
  test_hash (&nettle_shake256, /* 180 octets */
	     SHEX("5179888724819FBAD3AFA927D3577796660E6A81C52D98E9303261D5A4A83232F6F758934D50AA83FF9E20A5926DFEBAAC49529D006EB923C5AE5048ED544EC471ED7191EDF46363383824F915769B3E688094C682B02151E5EE01E510B431C8865AFF8B6B6F2F59CB6D129DA79E97C6D2B8FA6C6DA3F603199D2D1BCAB547682A81CD6CF65F6551121391D78BCC23B5BD0E922EC6D8BF97C952E84DD28AEF909ABA31EDB903B28FBFC33B7703CD996215A11238"),
	     SHEX("EFA2939B59DC8F3527C78E384B1CF12C7E184687A0C3A1FD9FF"
		  "B9797D72F13DF8E199B2916A6BA82FD7A914703687EE707C6F8"
		  "7845442DECA59F08E1D5DEF48BCC43BB1A64744D4DD020C8666"
		  "27AE212E5FF4EF9F63C14D2D1CFBB404E03573F9D110001ABD1"
		  "56754EF41A4C48E664E31B5922A27BBA26D2B3AC79F57720A41"
		  "90AE55A5634AF6E43CDFB87450EE8CC690749A45367250A44B0"
		  "7E54FC1CA8EC1CACCC979751544568F0117AEBBE6F5415A3524"
		  "4A9255FB023E7C30B013D6C47DE992141929F25B3A43DA91CF8"
		  "853EC88BA42DA3B17CF27351785BB16849E6BA680745F0294A8"
		  "5601D5A08CC924BB962CD8B67EE0667AF1E118E2E5144169DA0"
		  "FBF03C974ECB202FFC473315185723B57521737CDFBCE99DDB8"
		  "1F88D81C71C5F02AC5DB801437D46DAF31BB9308074A21DA98F"
		  "6899F6280E449C1B55CEB3E04522D038307AABE964B720AB331"
		  "A172A53CCEF174A82F046F42820D11947D1B316EDF390C86EAB"
		  "E22408E0008F396A0E7F3D51B9988CD2FFBC49869FE0EC03A33"
		  "8A1A8E0A663246A37DEA111DE5249196114996102223D628AB7"
		  "0FB7F53B842B60F375DD6779D9426AEBAEE547662B6962E3610"
		  "09C83CDCEF6D8BC6311CC1760E784CB3CF6241257EBE14FD955"
		  "0EC8316D95508390909BCD23EF9F78CDCB485E4C403671E5F20"
		  "E7EAF4E0C06344D54E157791FC1B581AAC0C4EF947ACCB642E7"
		  "8CFB"));
  test_hash (&nettle_shake256, /* 181 octets */
	     SHEX("576EF3520D30B7A4899B8C0D5E359E45C5189ADD100E43BE429A02FB3DE5FF4F8FD0E79D9663ACCA72CD29C94582B19292A557C5B1315297D168FBB54E9E2ECD13809C2B5FCE998EDC6570545E1499DBE7FB74D47CD7F35823B212B05BF3F5A79CAA34224FDD670D335FCB106F5D92C3946F44D3AFCBAE2E41AC554D8E6759F332B76BE89A0324AA12C5482D1EA3EE89DED4936F3E3C080436F539FA137E74C6D3389BDF5A45074C47BC7B20B0948407A66D855E2F"),
	     SHEX("97922E827163B8869D9A7654AE4532EA26F9895625152012F14"
		  "7BE1E818DF802689D96B469C8B0E9010D0423DA90AB6EED2C83"
		  "5A01CA4C386C630A22D885B511F12799EACEA96D157A4432A32"
		  "0F3EC1CBBB6B8EFEDE3E92D99B1E79389B93AF7ACB27D7DC97C"
		  "F88485DE6C85400A2DD88ACA540B3C921180002847B34A45446"
		  "5D014F4742815EC64F5FBFED6229294BA5EA89211E35327CDBA"
		  "C7CE85565EB8F23C4154AFEC0DEE188E9D854ED56BCA174E14F"
		  "EFCD0ED039B8DDAC3A3242CEB72D9116482F8B750A126505397"
		  "ED261D7CDFE88F8E2F4E4A80628D66A7676E28D4E68C3E97A47"
		  "C3844FB06692E5F4664B8A583A1836ADA9AEC095D011F12B3AD"
		  "6688B0039B1B74D016F1BD477C932DDC1944B5F301F4D5690E2"
		  "12E45D26908BA09C52D6CA22098B7A080B0056F0CB891411D0B"
		  "041CEA27030A2F9066AAF58CA5357344DD9AEA0B0D80932E98E"
		  "205EE315312D19413240B2EC4B854C21BC2DC16040718D25095"
		  "15BD45B6557083677C1882D48F687EA0EA86B05CC3F5E330DD4"
		  "BCC17E5B4F2CC4F2C64773EDF3045FA48C53081929ECF31E84A"
		  "905586D3AFB3120031FF75ABD8CDAD7CEE66386A7A718A90B98"
		  "EE970864A167FA48BB7A3F78ADCC2105E0FEB27414B1B6C59C7"
		  "BE5F7509E55D561C32523FC8C1C12628EE642D9A69F7D300B49"
		  "A5B6551B487655B21FDC690F865196A35F138BB8E69571280FE"
		  "F496"));
  test_hash (&nettle_shake256, /* 182 octets */
	     SHEX("0DF2152FA4F4357C8741529DD77E783925D3D76E95BAFA2B542A2C33F3D1D117D159CF473F82310356FEE4C90A9E505E70F8F24859656368BA09381FA245EB6C3D763F3093F0C89B972E66B53D59406D9F01AEA07F8B3B615CAC4EE4D05F542E7D0DAB45D67CCCCD3A606CCBEB31EA1FA7005BA07176E60DAB7D78F6810EF086F42F08E595F0EC217372B98970CC6321576D92CE38F7C397A403BADA1548D205C343AC09DECA86325373C3B76D9F32028FEA8EB32515"),
	     SHEX("289C52C39044BCE5CE8D320E2AC43DDF8DBB4A5E6698D536E30"
		  "4424CDDC251F7B181F58AD4948360972835CFA265E9B658F6F7"
		  "D4B7BD92D68E75F0DA84F47C969C7F12CF5170117B7EEED8B77"
		  "E3AEF06E52ADBF8F0EE6AD01DE3EC5E726BAE631A68F6FF364B"
		  "AA1754D928386496010F0EE59390E041CC0A6EB8F3864087BF7"
		  "BA5A57A116808BD496632081400B89CCB2A8A8EC795BF07C609"
		  "51B46815E6BA72DB555B9CC03CAAC4BB8D60BAD5EC6C5E48D66"
		  "F4C5B2D1669115E0F0BC80106EF06D48BA29E2899C0EF4D5C5F"
		  "4E29CB29C8F578A824CD462A4018E889B67DA31A90BC58178D9"
		  "04C43296E4D3F4977F0CC1009FE51F37B2F8979C97DACC92C8D"
		  "0E0FC45A004928F96260FE2A6F27DA15307AF55EFFBAC4814C9"
		  "F4446253064897FA1E1747FE65E4C406A3879F8E2EBE5CFDE37"
		  "65F2D7A2065A6E0D38B6B6A5F9FC3EC3CB1B35D33386EA75EDC"
		  "E250D5917A17A6CB9D1517684472FCDF27D61BD03F4BB43C3A6"
		  "359286E73B67D1A618A75EFC9AE2212BB507DD1BCB12D0B11FE"
		  "2030BAC84DEC40D04324C8354C3F3C6ED79D7257B8A189C565A"
		  "41F72AE3585A9BD91C9FD43184814BD64F7303607CD02B6613B"
		  "E736BD13B3AA8C668F91F398A8F5AA6BF8EF4D9058637FAA3F0"
		  "F0710ED60C2FA0F6DCE8EF0675EFA719EB03FEA27ABDE60C5B3"
		  "81EF4026FCEF49AC245483DA8B9B50DADCC310CD15C61E7A0AE"
		  "AA6F"));
  test_hash (&nettle_shake256, /* 183 octets */
	     SHEX("3E15350D87D6EBB5C8AD99D42515CFE17980933C7A8F6B8BBBF0A63728CEFAAD2052623C0BD5931839112A48633FB3C2004E0749C87A41B26A8B48945539D1FF41A4B269462FD199BFECD45374756F55A9116E92093AC99451AEFB2AF9FD32D6D7F5FBC7F7A540D5097C096EBC3B3A721541DE073A1CC02F7FB0FB1B9327FB0B1218CA49C9487AB5396622A13AE546C97ABDEF6B56380DDA7012A8384091B6656D0AB272D363CEA78163FF765CDD13AB1738B940D16CAE"),
	     SHEX("03E492E2E4C353A74245745981AEE10CB10ECD72015FC37621D"
		  "2E07411AAB55DAAEAE78EC376CB20F268FE40392876A29B4163"
		  "C3F1732FB58FBE26379F88C4388513C9E65DFB7A4B76DBA139B"
		  "28673EA6691096EC526DE0084FBC516360B07ADC853A690264E"
		  "C8C6E7F200D71B593FAAE6723C0A08B7CAC263D7F7DC5376B51"
		  "A2C7C640359721FA93A950A7A1F07F366B0FCD5C79471EE237D"
		  "11936B763840173710FB8CFE41DA287F61D13FBB174B451F30D"
		  "A1A1FB0F02C2B34689A725B1F86A96247E6F48897E524E84D91"
		  "BCC9F08DEAF189EFA84AF2A32DA7480ADA982E579A832FC3CB1"
		  "FED55E8D0D4AA710EA5B0B779CF0667ECBE9F5827FD0ADF6FA6"
		  "D98DEFDC0A3F3E84D5AE28D676CE703E1F060E4BAE4C0D931F8"
		  "4A76C2777D71B02F0DFD3DD00604C6EBB32642A9EBE9D34601F"
		  "B78AE60145E32B6AD2A362BFF609CB2EFCDC66781B5AE2976F0"
		  "249FDA97DCC06A8F7CEBDB32A2849B61142A7E55CD15FF51A82"
		  "1D233B3C35D9F69D47ABB6D2B55169AEA211A67475FD48D39B6"
		  "863EF6E4ADAE89530B0A84F6D6AA9B0475289C084AE2E607878"
		  "5289E03C6440DBEE352B9D39D11C9300730E4D224FE008696B8"
		  "F27F6A9261C30C577B48401B2E90ED5830BD38C13416D19F9B6"
		  "EC96F235F7A0571961F18C0F3A977740E9F5CC9CE62F695F9A9"
		  "F797A33A2FB247A62BC635449E4FFEB2F24298BDB152E777226"
		  "0F9E"));
  test_hash (&nettle_shake256, /* 184 octets */
	     SHEX("C38D6B0B757CB552BE40940ECE0009EF3B0B59307C1451686F1A22702922800D58BCE7A636C1727EE547C01B214779E898FC0E560F8AE7F61BEF4D75EAA696B921FD6B735D171535E9EDD267C192B99880C87997711002009095D8A7A437E258104A41A505E5EF71E5613DDD2008195F0C574E6BA3FE40099CFA116E5F1A2FA8A6DA04BADCB4E2D5D0DE31FDC4800891C45781A0AAC7C907B56D631FCA5CE8B2CDE620D11D1777ED9FA603541DE794DDC5758FCD5FAD78C0"),
	     SHEX("6C937D60095B762B69A69116DDE17251B6FC76117D2173E0702"
		  "408BFD8A215A057F25D720FAB9410BF39EC033D68CC87572C7F"
		  "C8B3C920C720C0A34FF468711F32753E8E6813D9D8ABC30AC3A"
		  "5EA3F6509A22E01A389399FA09052465316BE70501B70CEBC21"
		  "B5CE57FD935F7A265FB101232E95289E4234B43AC0BB45BFB8A"
		  "29489E7C3410F4A64028CA7BDEB97B327193A30FCC54F1A9835"
		  "E5BF497E0BBDE269E3F01BAD497AC3E09425B7E4D5D5A345B9C"
		  "DC256683C3D73722B6B33F525DFDF21106574ED3A1B38990C0A"
		  "18AEAA51621C6D603D58AAE2B1D89F57E802E66F8A7122BF4C4"
		  "C1DF38FF9B147BBD4D084E119298C9C4AF1BC6390AAC20BD1DB"
		  "5E68EAA7361601A40119A51EF26E2328F6CD52874E6F24B3460"
		  "A7D531DBC5DE94B0D62C35C6F56D4E8984B20BE06A248E59754"
		  "3E29C314D67DB0B112E158321E6BC9494E14C9E33705E678AF9"
		  "A6DC44BB567146E6D03AE590CF76C76398A4D5BFB9F0F12CB70"
		  "7EC79BB52B5488DBCF1A19623C9777F30F153C3E71E5F9A4841"
		  "E1F53E88421947903C9DBB58CDF75CAD297755B704E6919F25E"
		  "BF43520CEF6D71FD41AAD277AAB2BBC86B997B6A35C4C1A8B8C"
		  "04E6C7741E88CE198A650841997287E53CF24A2FD939F1AEF79"
		  "AC90DC787789AE0B344B658BF493F6DA6EF59112EF76FD39A8E"
		  "0CF7973E002FC7AC455CC4FD98C0030C7C76178D1EDD1AB12E3"
		  "493E"));
  test_hash (&nettle_shake256, /* 185 octets */
	     SHEX("8D2DE3F0B37A6385C90739805B170057F091CD0C7A0BC951540F26A5A75B3E694631BB64C7635EED316F51318E9D8DE13C70A2ABA04A14836855F35E480528B776D0A1E8A23B547C8B8D6A0D09B241D3BE9377160CCA4E6793D00A515DC2992CB7FC741DACA171431DA99CCE6F7789F129E2AC5CF65B40D703035CD2185BB936C82002DAF8CBC27A7A9E554B06196630446A6F0A14BA155ED26D95BD627B7205C072D02B60DB0FD7E49EA058C2E0BA202DAFF0DE91E845CF79"),
	     SHEX("176C854006AC910006248F1CEFFCF0BCA14015B7939B0065E59"
		  "A838F213B9ED46C68A377AC3110DD08AE358D4C3C926D262A49"
		  "06C196822D2D2031990D1F02B472B635EBD7B48AD7469671EA8"
		  "5683328A949C4B01E9AFDD0E780EC10F2A6EECD0524B77E6C98"
		  "93416EB73C53286CD52DCE11550E96B70154F0A06521195B7BF"
		  "6B260AD67D85FD2D3BA79D96B3C84D2EAF96350342FB33CBF5E"
		  "4DC12D7F7B6C5AA0F82AE2864A38026B6392DEB539637D323C5"
		  "A246EBB7A87EC7048DFCEC1DCA98D3239FF560598AC31D0E722"
		  "9EE0A36BD7500B991BADC5F7275BD650DE740DAE8421E1A7230"
		  "13C9E52AF448E21919EC70FCF5D4C5FD888B96676E65B6C52D2"
		  "353FE5D3D1423A73410234A2FF632CB09E922656FC233BE050F"
		  "BAB499D3B0864439F79B56DF983CC67FC460B027AD14B8E112F"
		  "898771545061DDF52652DD14AA034137764DEE5F896E3FCCA26"
		  "F70B63B37CD8D8DF7106876F4EAECF211875D4A3392E09B8E42"
		  "19326BA939DF7500183C3F039D1586D00BD36FBB97835FC4C73"
		  "845DAB9639A0231C27E3C3815B4478780CD646F1157BBDA4725"
		  "77FAA829F8B13E2C8DA1F7685719F8B979E895996DD05D1BC5F"
		  "0ED8F8F302E38E44A0F5174FCEB3253EE9CEA73315A1DD655E9"
		  "A6BB59A3B67F5F15F41A8E483D8C98733916638D5E8EFBEA8C5"
		  "BBA2000D2DDF330EAF3DB1E9A6072E5BBFDD5FD201F8A13800E"
		  "7719"));
  test_hash (&nettle_shake256, /* 186 octets */
	     SHEX("C464BBDAD275C50DCD983B65AD1019B9FF85A1E71C807F3204BB2C921DC31FBCD8C5FC45868AE9EF85B6C9B83BBA2A5A822201ED68586EC5EC27FB2857A5D1A2D09D09115F22DCC39FE61F5E1BA0FF6E8B4ACB4C6DA748BE7F3F0839739394FF7FA8E39F7F7E84A33C3866875C01BCB1263C9405D91908E9E0B50E7459FABB63D8C6BBB73D8E3483C099B55BC30FF092FF68B6ADEDFD477D63570C9F5515847F36E24BA0B705557130CEC57EBAD1D0B31A378E91894EE26E3A04"),
	     SHEX("F59CA6ABC514DF7EE5AF9B918A5355CAE65ADDCA95C80866EDE"
		  "16594F5147DED549BB9187E4F51535D3EEFC20F59703E1EF74B"
		  "86ADC45F8DE3265207B5A3B4223C3B70E9A9B24F08C2B1F15F8"
		  "58763381C89F2AB14AEC657E7E4CCADB9EAE9348B26450E9CAB"
		  "F9B54A56893796B2D4C04AA2B3B741D6F85E9A8CB6FBB0DD91F"
		  "FB91E9B179D5692F9C689C4315ACE33C2E7934D49183A5305B8"
		  "F50051242D73A225C557D787BAC8894D5987C8B4C91502C295F"
		  "DEBB492285561286262DA7A78A87F068E07B4CBF82344E6FBA3"
		  "C8B70A7CAE9023BB004C18A446DF8432C895A531DCAAB8D910C"
		  "5623651A362C4E2034A314F00154A421DA604CBAAB56B40F9F3"
		  "242E8F61CC9A9AB4C91304588AF9805180C8ADBFCD727C20D38"
		  "F2DFDC24CC1A84FFFF83CDE8F94359AC5D1848D3385B419EAFA"
		  "F09EF0317F99409F732EBAF75A7FCC74A4F0B7D038626078C8A"
		  "775B1EE75591C6CF99EE2754EEEEA7DC9FB382F25F55D030293"
		  "D53ED2F9D9DDAECB41172387FB19731639B37E3A26AF58B033E"
		  "850880FFCA4EBACB945F7D445983042FD3EBF4F70DC2BF0E95A"
		  "7CA9DA3C8F094BC937568A03486EBC127ABF44D150AC6DC0B70"
		  "80A41884102172D6E6B6D819AB088CA55287D6912018B5D7F4C"
		  "0C2A6973E532B40F9D0402C1525B2E7E11A948A5C833EFD7788"
		  "B3CDA4501DD7884AEACD53B503313500C3D4ADAFFFFC69A2EAA"
		  "EEDC"));
  test_hash (&nettle_shake256, /* 187 octets */
	     SHEX("8B8D68BB8A75732FE272815A68A1C9C5AA31B41DEDC8493E76525D1D013D33CEBD9E21A5BB95DB2616976A8C07FCF411F5F6BC6F7E0B57ACA78CC2790A6F9B898858AC9C79B165FF24E66677531E39F572BE5D81EB3264524181115F32780257BFB9AEEC6AF12AF28E587CAC068A1A2953B59AD680F4C245B2E3EC36F59940D37E1D3DB38E13EDB29B5C0F404F6FF87F80FC8BE7A225FF22FBB9C8B6B1D7330C57840D24BC75B06B80D30DAD6806544D510AF6C4785E823AC3E0B8"),
	     SHEX("A8C3DD2CB2C0B8CD01C2550A1C20308B3A4280AAF80447484DC"
		  "DFC69B5C2C2FBA310272E606D1CFB61D078E2D860F1A056A3E0"
		  "86AF9D7485611D64D1CE5F850B91920E778010A23390033DC15"
		  "E372AF077611C9B5A945981CD27801C9B701D892143DA0F882C"
		  "5D8F7F06B72B7107FBD22A4D890D1B00857FDE7E9B188088CCC"
		  "9A67BC12B0A496F3FAC72789E7E52ED7CBF36BC7B846F0B4D3A"
		  "221A8297A4BE76A1658A1547B32F534FD13C466BF1803C13005"
		  "93F8C53CB5530632DABC03ACB369D7E87CDD61EA7DB6D5980C6"
		  "8F00AD5D28F50B5B499A373DAE132811C5DAFBBEDCA231E7461"
		  "0271BCD0E733A59D5ED6A6EE77AB4EE0D64CB00AC7EC586AD30"
		  "F1F3BF5066268519209A9EE408B0DBB1E6E7CAFB63BCA7742A5"
		  "A7C78D7933B30CDCE0F8B502E3666FBA0C2CAAB6CB0826641A7"
		  "4C02E5F9DF0616C001273487487C2EB693E8818A281CB58D10E"
		  "0F911FA8EDA5DEF3CEA2136197E79A17F58E56C4262F3BDDD87"
		  "D9BD40AA21762C43F7091659693D2BF5D8BC03632EAD0C680B6"
		  "EBE53EADFED63727143FE47684188ADA6391EFEDB6A59770DA4"
		  "06B00905C124680B3B23D54C0255967C3BA8722DDF064F10F32"
		  "53972C0A4FF5B1BA17F8B9247E2D81FA1E4F7E3BD064E71A3F4"
		  "F18858D040027B45CBFBAE7CC6F04C125B867CCF3513C40F008"
		  "C2C961020B597F864BFF2347B8ECDCBF4AE460B1915A5A5EB98"
		  "AD31"));
  test_hash (&nettle_shake256, /* 188 octets */
	     SHEX("6B018710446F368E7421F1BC0CCF562D9C1843846BC8D98D1C9BF7D9D6FCB48BFC3BF83B36D44C4FA93430AF75CD190BDE36A7F92F867F58A803900DF8018150384D85D82132F123006AC2AEBA58E02A037FE6AFBD65ECA7C44977DD3DC74F48B6E7A1BFD5CC4DCF24E4D52E92BD4455848E4928B0EAC8B7476FE3CC03E862AA4DFF4470DBFED6DE48E410F25096487ECFC32A27277F3F5023B2725ADE461B1355889554A8836C9CF53BD767F5737D55184EEA1AB3F53EDD0976C485"),
	     SHEX("066F28311FEED21D06D61EEF566DEEC1F88E8D99DA9F6AE33E5"
		  "0321FBF7C1C458DB1E0C85C2685E806C583336F5620F44FF355"
		  "96A725D37B1A7F149B30EEE6EDF2C46EE8F7FF1B51D5ABBC09E"
		  "AD9AEF74188FBA0EFEB82DF86C304BF505FB02FF05B1797A7EB"
		  "3549EB9E74BF685B15FE615BC7FAB569A2E8EAC5136F97E3939"
		  "7E091F97D7C11D8E6B6390961C0A1A08D2A75D00CB679C87087"
		  "9C24A4AD0FB4EEF187E60297668634F06B7F2D98B85CDCEEC84"
		  "2DB321CC7B7823D8F332FF5AB7C44D9F862B61D3423ACA66392"
		  "9EEA2B47C9C78F890D50A4FD04079D4F77A9B303FBCA8A852D5"
		  "CBE0B09DF7DCD2178C8450370F60CE76AFECCFD3B4B346C7963"
		  "1439D879109357FAB4ADE37832D42D70B6390176FD864251FC6"
		  "5B695DB788F535B76077E3F69B1B2BB2BD26BBC0D1273109F71"
		  "199A58053A8AB6D33C9A75B8C7D4423A330D03C5BF6A2B2CE43"
		  "F27536AAC58D9D7CA8397BAE7C6C96D7279C19ACE6F7487412C"
		  "7370992FAFCE97342E145795D7CD22F2135D627F5BD528FA351"
		  "1B3EFCAA28ADA12B8F229BB594E802615A213D21CFDA091B736"
		  "DAA0B18D9E77E9AD98441F23926F6C06DECA62BB91F6430352E"
		  "857AA1488B82820F3624BE30E7ABD2785E8E9D6A3CBA0A926AE"
		  "425253E1417AD70F80C1CFE49C730E1B287EF47FE65255AAEF1"
		  "408C66F622CD1B6484FEC1675F1A74AFCC349EF4746B4468DAB"
		  "F577"));
  test_hash (&nettle_shake256, /* 189 octets */
	     SHEX("C9534A24714BD4BE37C88A3DA1082EDA7CABD154C309D7BD670DCCD95AA535594463058A29F79031D6ECAA9F675D1211E9359BE82669A79C855EA8D89DD38C2C761DDD0EC0CE9E97597432E9A1BEAE062CDD71EDFDFD464119BE9E69D18A7A7FD7CE0E2106F0C8B0ABF4715E2CA48EF9F454DC203C96656653B727083513F8EFB86E49C513BB758B3B052FE21F1C05BB33C37129D6CC81F1AEF6ADC45B0E8827A830FE545CF57D0955802C117D23CCB55EA28F95C0D8C2F9C5A242B33F"),
	     SHEX("DC2A3560C3644C4ED659240CC9B275B7163F5C88B28839FA74F"
		  "CA4A055B265E8E8DE186EC35097511D12F125B9EDAC4665EFE2"
		  "E41DEFFFD4FECA65386D9A5B06DDE6B1AD1F70772ECC871C445"
		  "29D413C603D7D8BF2CFDDC72C11D3B9684CDF4696EBEE1A88F9"
		  "39A997802AEE46BBE7ABE0915C4DF68C43EB06A1412C690EDEF"
		  "8FD21F94D16EC47D42168C30B8A3C2A87E348352200040CEC76"
		  "97835DD66A8E0596DB943ACD2BA8424A7D031C5AD0322C913AA"
		  "9B11C4C1E09ACCF8B941FDA2CEFFAC3F38BD43FEB3E67D002E8"
		  "D45D9C091D287A5F6C4B70A58129867C46FF1E98D402202C27E"
		  "088DA5063157E0BE4A7C82638EB623947A59D15B1FBC965A734"
		  "49711FEEEF521B61C4F94B7AFB015185BC7C696200F10C2657E"
		  "233F643C0B20A72EE57E7E17EC806B0B4A128F9F0F6BAF129CC"
		  "D82B19FCC72F62FA10456CBE37317BDE49BA0EE9FD83559E322"
		  "04387CBB5D26064127448D2F2702CB37C3CA5B49989C7B90385"
		  "4907E1A748FD9E589A01EEDA6FB374178CE57C6027334399A78"
		  "96C91FBA3F8C2EC917F2FDE3942D60FD37BEFDA64909A34F5D8"
		  "BDEEEC6DEDAD93894E9C6A894637B8193895973C5DD8424326B"
		  "592723C06583F683281AA90C3FB3A754AC8AEF614DC30DC608A"
		  "D26EAAB42B4899137F4D42E4142F93706F88C8C98BB6B15E099"
		  "2834785A27FFCC9FC2037A3A3B4B3C262ADE59E3A6718C79A8C"
		  "0C82"));
  test_hash (&nettle_shake256, /* 190 octets */
	     SHEX("07906C87297B867ABF4576E9F3CC7F82F22B154AFCBF293B9319F1B0584DA6A40C27B32E0B1B7F412C4F1B82480E70A9235B12EC27090A5A33175A2BB28D8ADC475CEFE33F7803F8CE27967217381F02E67A3B4F84A71F1C5228E0C2AD971373F6F672624FCEA8D1A9F85170FAD30FA0BBD25035C3B41A6175D467998BD1215F6F3866F53847F9CF68EF3E2FBB54BC994DE2302B829C5EEA68EC441FCBAFD7D16AE4FE9FFF98BF00E5BC2AD54DD91FF9FDA4DD77B6C754A91955D1FBAAD0"),
	     SHEX("5D5AE597772925F37C5D78176E15392BD53FB65B2925BF589DD"
		  "F97D92EE412F2847ACE1B5A4A22515C910A0A7479AD4CFF5948"
		  "C4A736EF021F062850383EA778389BF5F33C5357CE1D11D11DE"
		  "27BDD076720692737E5887B0FC6B8FA46F2675650072848E26D"
		  "94D596B921280649389A9C1621037AE9D91D460BDA8FEBDA762"
		  "7C181F8EBB77F43DE66F9D64EF1D7CB66622E13FACEEBC7AB34"
		  "6C0C71A1FAA0DF2659D980AE4ACF04317379B81DBF8E9206F66"
		  "7EBCD2A8D736766D2FC64EA442A4BA0E331E7D3EC0AB1618BD9"
		  "0A9B97A4AB6556EB09DA3C551AC3F2714BE4CB094AC454F0DEB"
		  "D1964ECF15118F677464C60108A87E8895046269DB21168E35E"
		  "C4DFF11487E93B49A8FAEE8EE6C3FF05394D1D0C9030123CBFB"
		  "9576C9015C30F9DA04331C995101F4589D27CE664D2D6928C80"
		  "3EE6ED655B661F286F12499427B0B3D4C19A870CC6065528D0D"
		  "482FAE121766275ACD12731B127C454D17BBD32F779BDFD59D2"
		  "221CC923764A023D48A109F259C2C4D9213E35411A7E4F03C18"
		  "78E576F9AF4B84A983F64ED1255EE7CA9A4696762701950C97F"
		  "374F207ACF4023C0E42651C650C330EBA2A7773AF833A9A7B1C"
		  "9360BEC7FD788DB63AF83B699E1BAC014A86AE976F15809E4DF"
		  "C8EB8225DE57C97CCCD0683ED0BEB67B752A6BD5EF128D9BC6A"
		  "D343B04AEAE8C76D7A24E3025D436ECD9F03432991A19D1B164"
		  "2AF5"));
  test_hash (&nettle_shake256, /* 191 octets */
	     SHEX("588E94B9054ABC2189DF69B8BA34341B77CDD528E7860E5DEFCAA79B0C9A452AD4B82AA306BE84536EB7CEDCBE058D7B84A6AEF826B028B8A0271B69AC3605A9635EA9F5EA0AA700F3EB7835BC54611B922964300C953EFE7491E3677C2CEBE0822E956CD16433B02C68C4A23252C3F9E151A416B4963257B783E038F6B4D5C9F110F871652C7A649A7BCEDCBCCC6F2D0725BB903CC196BA76C76AA9F10A190B1D1168993BAA9FFC96A1655216773458BEC72B0E39C9F2C121378FEAB4E76A"),
	     SHEX("B34E968BDD16E68C71ED69202A38C554F8362FA382F1DECCF89"
		  "536B0D2FFA7CF8725CEC1D2D7AF57665496B643E8206566F8CB"
		  "E4CBC42DB65481750319BBF37D9DEF13D050B2DAC53E163AC36"
		  "5E81AAE03BDB40B67D0786EDEFC2B339FEE7176DEDC90C92B16"
		  "A726E61A3E4AB5B29741869A4E8FF254D335C62053EE9FCB5BA"
		  "D309B11B916EDE9BA5D1594B0ABBCF73A7A231B4104037D10C3"
		  "2433F99DC35A356088F90298BB9BD150FC8252017BECF06111D"
		  "AEEAC705BCCF25E8098995A8F7EFCBB10FD2C443223325FAC91"
		  "72BE8B5FEF5FBCCC8D9361DEB61B361FBEA3D5CF13F4731E8AD"
		  "4F567BBEF2655A79213193131020123E5FADC2782E8224208EB"
		  "98C736FA9AF25EA3D00CAFDA04EB8F0544B58B29DA5D438A969"
		  "14CA1D47F2A729128F60E56F5634A73133C1A0435A40FB7FF6F"
		  "88105BA307B4B6479D0DCBB80A5F1A71954EE049E99B944E41E"
		  "E89DE4B78A78786660A1516A3BF64779DF724BD82045245413C"
		  "D82890340D29D07879C000934FBE4B4FEDA2EB4CD795C528E83"
		  "BA11A92C9476E3273DD5D4553798489B8FFE76104EF154FECE0"
		  "5E4BE9775256F278DEB61552266745FC21DFD699DB5A34F4DAE"
		  "4FB3E23146DB6767561F15EA776B67240C92C695EC83EE10978"
		  "FCDAB7A57C692861F15E5E6AE0E3DC2C58489445DEF81F884A5"
		  "0DF8E78CE67C23967B27B419F36B905DC2D9EF1D27BFC94B456"
		  "836A"));
  test_hash (&nettle_shake256, /* 192 octets */
	     SHEX("08959A7E4BAAE874928813364071194E2939772F20DB7C3157078987C557C2A6D5ABE68D520EEF3DC491692E1E21BCD880ADEBF63BB4213B50897FA005256ED41B5690F78F52855C8D9168A4B666FCE2DA2B456D7A7E7C17AB5F2FB1EE90B79E698712E963715983FD07641AE4B4E9DC73203FAC1AE11FA1F8C7941FCC82EAB247ADDB56E2638447E9D609E610B60CE086656AAEBF1DA3C8A231D7D94E2FD0AFE46B391FF14A72EAEB3F44AD4DF85866DEF43D4781A0B3578BC996C87970B132"),
	     SHEX("2D48CAA033F0273CEE712401A35D143C7E91216139682A77CA7"
		  "75F08BA77620DD02EA885667355740363D95DA3CEFD3E9F8D05"
		  "75A41213C1E7F2A6C4B7212DD617C9B0A41D48674B6A19C63C1"
		  "AB3F2CE7F1A0FA694CA8401A0121077F281CD5D6A5424AC1D3A"
		  "11AA97F88783722E68ABDD3684907F63EE045D7F3E10545F33B"
		  "84C30200DE3B135B4BE5033B854084E5912F17B14F2AC16D027"
		  "FD02FCFF1AE096C983F0F4FCE7D6B641D2280DFB90D8A066D8C"
		  "F53E2A5A1620DC50008CA375743C416C379FDC5AED3DE23244D"
		  "E23AADC33BFD9609702DA4776881A1F588EE1FD98602FB831C8"
		  "D857CB73DA50C99ACB34DC79D0F32FABE25FEFB54DBD494014A"
		  "24EC55B4D4ECEF9D4B7A98A8BDE06E324E6B2BDBE7BEEE6A45E"
		  "4FE8948D331561090B737CE14D302C9CB1D7EE3E78D0B7CDA6F"
		  "D12CF038F8B92E35F7BD6E1664A5F4F75EE569F67C41A434330"
		  "DFC06F8C6C7156208A11BD6708B22F3115E5D4D99F54451D967"
		  "BB7935FC937AB2EE21C3DA93B1A3DF94DDD600D6AE8F43A35F8"
		  "5794E3BDABD82D4ECC923B0BC91EA85C3A2599F492495E8AAE9"
		  "70E1605FB642B6B93E3947B45D78ED6A563FA9D10CDFF934C61"
		  "D78B46AEB754CFFECF00AED06839365D1A4B7D9CC5852A4F755"
		  "F01723C8E3961C92F320A43EC23D8292C4F44336E78424A4E65"
		  "0D74BC47EC05EA8D692560206B9DC2310C893598370E5DF5D93"
		  "ACB6"));
  test_hash (&nettle_shake256, /* 193 octets */
	     SHEX("CB2A234F45E2ECD5863895A451D389A369AAB99CFEF0D5C9FFCA1E6E63F763B5C14FB9B478313C8E8C0EFEB3AC9500CF5FD93791B789E67EAC12FD038E2547CC8E0FC9DB591F33A1E4907C64A922DDA23EC9827310B306098554A4A78F050262DB5B545B159E1FF1DCA6EB734B872343B842C57EAFCFDA8405EEDBB48EF32E99696D135979235C3A05364E371C2D76F1902F1D83146DF9495C0A6C57D7BF9EE77E80F9787AEE27BE1FE126CDC9EF893A4A7DCBBC367E40FE4E1EE90B42EA25AF01"),
	     SHEX("6F7549D58E91D97F0C1243C519EB6AD2CD72AE27E3C5DA2C6B9"
		  "2407823D1275210ED93B95692880015BBC14E72892A36D4D492"
		  "4A48DBDD2E774ED36B739F587D200D9A789FF56542061AB4DE0"
		  "06B08749C31BAC057DDAFA581C6F756BC3FF23BBC43C1048BD2"
		  "848204588C89E6C3199EC38DC9F4BB10BCEC5A5D236AB28113E"
		  "4E43D529DE8424F118CB14A5FB02BA721BA4650CFFA35663820"
		  "148E00A2A8C0B8F4C20DBBE20C146E4B5294A17C99C2DF7EA62"
		  "F9107ACD5FEEB01ECDC064F2911A466BC1024B1AFE10F3DD784"
		  "3BDEB8682339A95F7A0E615A216C89AE7E8A688926884F82BC2"
		  "A487CB2695C658DB670EC286E63AFB5B244950497BE33081CCA"
		  "0FD14A4958AE4EC94B642E3A9BFECA9481FEEDCEE34B6C13F30"
		  "1CFAF882105C14F20679B5E7E4263C29397F62B094490BDE56C"
		  "BB1CFFEC550819E0ADD12DA85935AF2A9C08537ED3034D39933"
		  "B80471D98B1562BA12C9B98A462093E79D247BFE4EE59CDF9F6"
		  "533DAE9C2ABB66437B90D845B16C5318D0D12EC5CE30AA63B8A"
		  "754CA93F04A3E3013A197A34DD8AF463A7AAE48D51C7EEEDC45"
		  "B5BD4F9E5BE06D47C7991956E7BD828DEE21A4C4A69C3BCBEB9"
		  "14FB93832BE8A986A077FC3A990301532CEDB59CC03A0EFBE5F"
		  "F88413F0DB07481D134B77200E05341C21C43A7FC0EF20F254A"
		  "4438F9F49C28136EB7FF5D64E8812D4137CB3410B2482D3289A"
		  "CC07"));
  test_hash (&nettle_shake256, /* 194 octets */
	     SHEX("D16BEADF02AB1D4DC6F88B8C4554C51E866DF830B89C06E786A5F8757E8909310AF51C840EFE8D20B35331F4355D80F73295974653DDD620CDDE4730FB6C8D0D2DCB2B45D92D4FBDB567C0A3E86BD1A8A795AF26FBF29FC6C65941CDDB090FF7CD230AC5268AB4606FCCBA9EDED0A2B5D014EE0C34F0B2881AC036E24E151BE89EEB6CD9A7A790AFCCFF234D7CB11B99EBF58CD0C589F20BDAC4F9F0E28F75E3E04E5B3DEBCE607A496D848D67FA7B49132C71B878FD5557E082A18ECA1FBDA94D4B"),
	     SHEX("2D534DC9501C6B18F9662128E227E3C7D8BC7F6773945B8CA65"
		  "3BF93F9B07BFBAD84F195CCFEAF35FFE97E1109212DDAF5701B"
		  "F2F81A72B51A66FC1590534823172927BC3EA342A96DF9B0CF5"
		  "952690804C37814BAD3A41849D71AACE71304183A4477D31365"
		  "4C731CB0A9AF39CAC8A8E4B6883AF7C4F94720A5919C2371C1A"
		  "E472724F7062E47754066976B583CDFF98CB41796664C9B0C76"
		  "3C4530FB62DC22AE7B3C6B4DB593942F6144DA79AAC3F3CD53F"
		  "90B7B634CAB679ED27341C83A3F9DE8763A0D083C4EA4EB3B8C"
		  "189967E6970178CE26A3F2D43D1A62E4D26D61636257DB145A8"
		  "9EEEFB8C38AA9D69F0630B99B4FFAF9395A6AED9C63E78A7F8C"
		  "EAF4884E4663E7C9E39F737703F5D1C73B2E846FA5AF03528C7"
		  "F1D1378C9AC7CE02599278455FA6E1B7BEDF453ECD680A89A8C"
		  "77FB727A688AC573A5ECB545218F5DDAB304DFB786219451467"
		  "4A52F2F0EB2151BD936974FB5B677F621FF9E92A10C76E3B6A3"
		  "4E4279BF2E395743EC8FEEE4F4BB1D951AF744F9BB923A13C1D"
		  "6F159F96B90D0371B135A8F64351DC8C9DFC0AF38098D3A8583"
		  "DDA8978103B25495BFAB2B144E4D82774C06C4CCEBE085A5B90"
		  "2DC391B60035F4EA929DE8A56DB1EFECFB928BE97C23E6DACC7"
		  "9EF2C05D5C08F2CB3ECA5CF7626F13A95F29239AC8F457DCE85"
		  "815BA83F099B1DCE79A32C6E04280B8CEFE8EC213E2375553D2"
		  "214D"));
  test_hash (&nettle_shake256, /* 195 octets */
	     SHEX("8F65F6BC59A85705016E2BAE7FE57980DE3127E5AB275F573D334F73F8603106EC3553016608EF2DD6E69B24BE0B7113BF6A760BA6E9CE1C48F9E186012CF96A1D4849D75DF5BB8315387FD78E9E153E76F8BA7EC6C8849810F59FB4BB9B004318210B37F1299526866F44059E017E22E96CBE418699D014C6EA01C9F0038B10299884DBEC3199BB05ADC94E955A1533219C1115FED0E5F21228B071F40DD57C4240D98D37B73E412FE0FA4703120D7C0C67972ED233E5DEB300A22605472FA3A3BA86"),
	     SHEX("4D0E55472FAEDA7E9F092EC4A561E835E261935BA5E6D115E9D"
		  "4C4021725E7831C5B7F102F2D1494B019FFE9BF5525822C4978"
		  "83F0E5194B0B0989DBDAC834CBFCFDD31294DAE5752ADB56C23"
		  "D45668F370963219D3A9BE504D8E93721242DB7D4D1CCD22F84"
		  "687E3945E2037BA24670649913712FE6B5D99BB6C8492C3B853"
		  "E481CA9C32B3E601D31C60D6B5F43A240B0711FDFCFC0AAC6A6"
		  "E5F881B2AE1123F7568663D8415DB5ECB3000AAB268256888DD"
		  "47C0FBB5C95AAA9450947244EE4CBCFABD76326D2D079C9F1BC"
		  "4F3D43AE1D01028FC1705F74CD16132C79D3A43539D2A842DF2"
		  "B69243CD37491CDAB57AC7E4E00350F8C08737B6CCB8A7C48DD"
		  "50365A85C07A525C15AF65EC3B2AC1C73AA6B3433483D5EC77E"
		  "D832B73A30C672E9DBF0778DBC20BA5A3513EC93B4165A60E68"
		  "717AAABED5041D9EE445611EBDDC8597FDA3BA05245228D7C56"
		  "7BE1CE5D2BA75A256361E8259F43F3402E09B96A14014389B8C"
		  "E19461A6512A4DD7F866E73ED26B1C454E014F901DD313D5E72"
		  "14CE7A62C476DD28E4152EEEE89F309161C8F494F0BEE613B72"
		  "696F496BB41659A3804683729B40682DE5D0F17AFC0FA9C27B2"
		  "C96B55A05C536D283957783BB93C8F3E021A30D6051C9839DC4"
		  "8A70C127611449823B06F0DD460CA1DD990AEBCF3FCA9732D01"
		  "2E9650907BFC44729DF2827AE4055FCCA246ED3D75DC3B69DDB"
		  "3666"));
  test_hash (&nettle_shake256, /* 196 octets */
	     SHEX("84891E52E0D451813210C3FD635B39A03A6B7A7317B221A7ABC270DFA946C42669AACBBBDF801E1584F330E28C729847EA14152BD637B3D0F2B38B4BD5BF9C791C58806281103A3EABBAEDE5E711E539E6A8B2CF297CF351C078B4FA8F7F35CF61BEBF8814BF248A01D41E86C5715EA40C63F7375379A7EB1D78F27622FB468AB784AAABA4E534A6DFD1DF6FA15511341E725ED2E87F98737CCB7B6A6DFAE416477472B046BF1811187D151BFA9F7B2BF9ACDB23A3BE507CDF14CFDF517D2CB5FB9E4AB6"),
	     SHEX("C7EAAAE0CA10AA1CB0225920A8F2288EEC0E5889ABEB096A983"
		  "10782D918AF4CB8C0A41384D8E0C408950EEA95DE8CA6EB87F5"
		  "E86780FB91293C5D14A798DAD9B77EA8A5B50E17C19EF37ACC5"
		  "166A74A1B37712290CFCF98AE6349DBD4842D8ABEC6735ECD39"
		  "2D59591824963D403B2804135BD46D17E26D5C498FD7111BF27"
		  "82F758802A20AE6A169AA65CB0D6640396E982E30126C37F68E"
		  "19CAE5FF9AE3743E38F902D306D8FEC8042DD914003F960625E"
		  "8F2104C369CEC0CB8D3847C28D050802EEB53C5AF08D70EC029"
		  "79019700E182B7127661DF869807C16F309EB08F376C5C30B01"
		  "89F1B4B32423BE060402C1EC9B1E7AF2432FAFC09E4EFF7DC6F"
		  "55A52108C5599BBD5A07F433B3EB4D439D43EA79407FA711132"
		  "CA9A5E2F528E18821F38B894F624EE42400832F8C85AAAD0FEA"
		  "54261E2B55647D9FF5F31AC212E7C38CC96689C72A4F543A256"
		  "7D8EA79913727DAC3172F69CB16694E5D4EED7C16494A297C87"
		  "6C73256F6B30F88B6E180E8E62DB68458F2CA416C4A0317F70A"
		  "3E2465BC543FB07AE518B6571664FE9C2E49E2AAD57AD933869"
		  "C04A1CAB60903F9AB1C14E4FB3B6502F387B30693AE4A85A172"
		  "FE6813C42026DE57291B3D854A191F553893B3D4A79C84B08C3"
		  "7E80C1ABDF03FAB58966410C2B8B950A275F68852F006B2B38A"
		  "49E09165AAE5BA74B934A6B7133E431F01099F930B5FAD2790F"
		  "126E"));
  test_hash (&nettle_shake256, /* 197 octets */
	     SHEX("FDD7A9433A3B4AFABD7A3A5E3457E56DEBF78E84B7A0B0CA0E8C6D53BD0C2DAE31B2700C6128334F43981BE3B213B1D7A118D59C7E6B6493A86F866A1635C12859CFB9AD17460A77B4522A5C1883C3D6ACC86E6162667EC414E9A104AA892053A2B1D72165A855BACD8FAF8034A5DD9B716F47A0818C09BB6BAF22AA503C06B4CA261F557761989D2AFBD88B6A678AD128AF68672107D0F1FC73C5CA740459297B3292B281E93BCEB761BDE7221C3A55708E5EC84472CDDCAA84ECF23723CC0991355C6280"),
	     SHEX("CF55585AD2CF64217E5CA23319EBA00C4EC74B7B9A453D1D7C1"
		  "44E8B68FA8EAA16B853344ABC905EEC90B35B681FD8A4A52460"
		  "999BEE62489C467151EA5C0D7C8EF2B9959FD44D171D530AD4A"
		  "576A0C76B147127921817B167FB43050D7F2F552F8E77ED5161"
		  "AB00907599AB80FC546C859DBDC04F2CA3E38A7593DCD22EC89"
		  "E73C4924152C7D1B5281699154A3325196E97E73DA29251F60B"
		  "C68706885462002C126517E00D02DA6A23B0CA9410EB916C19F"
		  "EC76DE9A907447A8DC1F8304EEBB6E38D0694821EE03865A5BE"
		  "DE036E01AD74AB397AFBEF6C62C5DC211A47BCDCDDECB48B17B"
		  "F5336FF831DB7EF9CE7643B285504D495161D4E864B6FEE67D0"
		  "EFAD2888C04A8992F019E52A6632F72539FD65271E81537AFBE"
		  "9E7795C658AA0C3B2C64EB34BDD0E5F19EA217B6EC9520FCDE4"
		  "46C2207C9AEE94A6AAFF4F1F39A91927DAFDC37BCA8173034A5"
		  "4FD535BF03212995975240F3002446DA1D111C0F97C92E395D9"
		  "6E120B958F643CB4EC52C6D2C148E6A0A497323565500AB1174"
		  "21C4A0CF0A055DC8B4D287F66FCF7EDAF931B614589EF6C6F46"
		  "84A80A9DBFBAAAD85059B9399CE2617B55E04C2112B32A210D7"
		  "530A3C58678B1D48EF24E2FB6B6F90D4FC605805B903085198B"
		  "2C50432D9A99454B68AEF0C9D0D81931123D7D14C0DCFF84A54"
		  "F3D4DCDC9E232BED26E2EF3334220F7B39EFADAB2CB9E247EE4"
		  "E880"));
  test_hash (&nettle_shake256, /* 198 octets */
	     SHEX("70A40BFBEF92277A1AAD72F6B79D0177197C4EBD432668CFEC05D099ACCB651062B5DFF156C0B27336687A94B26679CFDD9DAF7AD204338DD9C4D14114033A5C225BD11F217B5F4732DA167EE3F939262D4043FC9CBA92303B7B5E96AEA12ADDA64859DF4B86E9EE0B58E39091E6B188B408AC94E1294A8911245EE361E60E601EFF58D1D37639F3753BEC80EBB4EFDE25817436076623FC65415FE51D1B0280366D12C554D86743F3C3B6572E400361A60726131441BA493A83FBE9AFDA90F7AF1AE717238D"),
	     SHEX("85225E389DF45458881A78277858D3B12EC5527BEE0D03F41EF"
		  "BA24671F1C245EF33BC8CECF3CFB7EA82263EF8F5F4E2C0D033"
		  "D1DF6AD35F1ECAEB2E40F22ED9F4CD0D1B9FB1FF5AFC3ED6054"
		  "9F1836E325AD6641E4444913A9AEACF36067A7E6EAFAE4F5D17"
		  "38F4B42F5185AF334FEEE2838FAEA6203032F3C5C3D22863C02"
		  "783D1562E12B0C73DF3098CEC208BC855BF84A9C231021AD5F1"
		  "FBBE51A21E2BB66BC555612500822F43DB5594D0D60532B726E"
		  "3A430C26D9E5BCCD57F83CD1CE3A5A31DDCC21DFD8A76D03A10"
		  "6FC77EB55EDF19A46023B5D70835C865952C463475769067FAB"
		  "2BA2A11FBF7F29D6400D10A34B1DB497B5B1D8D8A30A5C4DFC4"
		  "0EBDD1BDC992380074ACCAC904DC24F22547A99E7F5ACC53BB8"
		  "3BF94A522E595D9AB54C4E7EE052ABF95FC5FF52967CA2FBB92"
		  "7ACF7127349BA7FD26AC820C380107B0E341866A9F57852DF4C"
		  "E28C2989F04A40FF14033B3D36AE7D24A8E377AA40BABFB8502"
		  "B0B9B12DF9B42C2A9982AA29697A2ED522E344814E59D91132F"
		  "39AA537A521BB4B781CBCB5E1503354FE6A25072A52C54D248E"
		  "13F8253FCAF93E849597C23F6952D0F7F38D1A4E7D08FAFAEF3"
		  "ACBFC4371779C6795D1CAC6BF86C34FB227AAEF0AB51C480C45"
		  "10C9F1F463CE311CF931E8219D71BCD3FB55D3A8E9A6C86710D"
		  "18F9FEF6A00D1FB878069AD4A4E32653B0EE15866268D5772BD"
		  "4990"));
  test_hash (&nettle_shake256, /* 199 octets */
	     SHEX("74356E449F4BF8644F77B14F4D67CB6BD9C1F5AE357621D5B8147E562B65C66585CAF2E491B48529A01A34D226D436959153815380D5689E30B35357CDAC6E08D3F2B0E88E200600D62BD9F5EAF488DF86A4470EA227006182E44809009868C4C280C43D7D64A5268FA719074960087B3A6ABC837882F882C837834535929389A12B2C78187E2EA07EF8B8EEF27DC85002C3AE35F1A50BEE6A1C48BA7E175F3316670B27983472AA6A61EED0A683A39EE323080620EA44A9F74411AE5CE99030528F9AB49C79F2"),
	     SHEX("561A3E8D8D57E0E32D7E9EA8AB6F58AEEA509CCE8740FE54E8D"
		  "B9AABBB14AD782BA33C7F9A5007796CA2BA9089F525BCF80BCE"
		  "DF4C09E24E8D14F1766AA97A82B807FB1ADF65430BBF5F87AF0"
		  "A7FF26578C97AB50148B6B15711311FEE962A8B9C3D1ABAF624"
		  "41CEB100DF3A8E588DFE3F6A356B348E312CA017C94143B3DF1"
		  "7F255DFAD5A7341B163C0562D24897FE9E4B99262FE5BF6920B"
		  "79ABBAF856CE38AFC756A77F2BFD5C3A039C08480BAC81D641F"
		  "8B955B22B66B455BC0637DD49A1218C4E8CA73BDB68B8849D21"
		  "48C021C75BF8470D36167AC32FC88128C4E0568724BB2D631E7"
		  "FEDFABB91F0343D267BF3A57BBCA57C21D56E339CB13DFA5E56"
		  "F3A90F0241DBC0D0D73F40BA282ABD5E031364ABAF427713463"
		  "EC5072B7AECCA62E074097BD0CB3169DDC06243F08B62527921"
		  "F5FDC9332E99B2E6903D65BF809DCA706C5C0F93AF85B9BFEEA"
		  "40504A3DBB390C6A417203979687A7C86CFE5AD3F0737DA1B90"
		  "7F8F941EDF44F8367B65EA8BB92B5285A3ED8E1F7EB0FC00536"
		  "C15005E4F6D9F2969AD96721BFE7A6A4824A37159EF08B0FC60"
		  "C68518FD4CE54E4033C41ADE6CB218CD7DAEE76622B711A1609"
		  "168BC115AD4F6219EAFBC492E35BDC446E866433C7446506692"
		  "94E8D1827A2500F3E30893C14820DC24E36B3B451E9459BFDCA"
		  "E62168706228BCD9C3EA22C4F1D0286367C7939DF31E28E7A2B"
		  "28CF"));
  test_hash (&nettle_shake256, /* 200 octets */
	     SHEX("8C3798E51BC68482D7337D3ABB75DC9FFE860714A9AD73551E120059860DDE24AB87327222B64CF774415A70F724CDF270DE3FE47DDA07B61C9EF2A3551F45A5584860248FABDE676E1CD75F6355AA3EAEABE3B51DC813D9FB2EAA4F0F1D9F834D7CAD9C7C695AE84B329385BC0BEF895B9F1EDF44A03D4B410CC23A79A6B62E4F346A5E8DD851C2857995DDBF5B2D717AEB847310E1F6A46AC3D26A7F9B44985AF656D2B7C9406E8A9E8F47DCB4EF6B83CAACF9AEFB6118BFCFF7E44BEF6937EBDDC89186839B77"),
	     SHEX("3340B37AEDD2F0C66F2483ABDC66C97B45055275231F1C7A925"
		  "687B946C9135BB75AAB2D11E6463073A1A8CED0EA83A42736AE"
		  "85D1B99D1D2EE9AAFB6FB9F4E7ABACB39BC9F6D59883A9D1B0D"
		  "F86C259394A05842684E8A4573F1A3CEB46ECE59DF77259A0B0"
		  "25B831C2CC0DAE259551C86DB7DE0C434FCBC35328810201903"
		  "B92802CD5DACF9A6F4ADCB05EC5E0629A3FA99BDF2A5CBDA2AF"
		  "81CBD1438821A319408D67E5995342BF9BB4DA85BDCF43C5A2E"
		  "AD9CB4FA26542EAC044DC70857FF1B1CF95026B6464B96A46F3"
		  "A138FCB26B525E6A09F71ED05E57ADF4A3FAB611CD8069E016F"
		  "ED7B9F2F14C38102612D66803EF0854B2C7678E870AE7BE6CE4"
		  "3EDF447E476C0AD407F0D1AEAE1265521F43AD6CCF1F2E0C82E"
		  "5395ABFC41E08AAB436A806083E246CAC452513BF8E9C9E47DE"
		  "645055AE184D988D0D3CC1E5DFF6AA98B3EED26E788EA342C6B"
		  "1914269113248B15D0B79A6BD71EEA6C0BF9D89698A2B943AE2"
		  "4F843B00B34620DEE9DE5AE1FA1E00677D8FD9716CF43EA1EBD"
		  "BBFC2984FB1B27D444AF3B2D01A86412A32090A45385A912F39"
		  "4373F0610AF7D3C06D8564FAFAAD3B88C70196A1DEDF247AB51"
		  "5E5FCC8490A889F974E12D8DD5FDDDF8DE55500CDAF60FF2B70"
		  "E7D26B481389C2232FDEE43AE99B221893FB7CDB688C0648657"
		  "1A09B9A9B5D2DAC509BA166D0C5F5D8EAC363324234C657678A"
		  "0CD7"));
  test_hash (&nettle_shake256, /* 201 octets */
	     SHEX("FA56BF730C4F8395875189C10C4FB251605757A8FECC31F9737E3C2503B02608E6731E85D7A38393C67DE516B85304824BFB135E33BF22B3A23B913BF6ACD2B7AB85198B8187B2BCD454D5E3318CACB32FD6261C31AE7F6C54EF6A7A2A4C9F3ECB81CE3555D4F0AD466DD4C108A90399D70041997C3B25345A9653F3C9A6711AB1B91D6A9D2216442DA2C973CBD685EE7643BFD77327A2F7AE9CB283620A08716DFB462E5C1D65432CA9D56A90E811443CD1ECB8F0DE179C9CB48BA4F6FEC360C66F252F6E64EDC96B"),
	     SHEX("FC9C58940908E1F78AA90E888879C9EF903CD450DB94E5BEE0D"
		  "8ACF6A4E453A32A610C6209D9F5A44ECABD634D45D04F5D946B"
		  "CD388E2943D00AD8A6C4682D985F45D796ABEA85F419D3990BB"
		  "DC5A21294970FD42B6D06363E18EF7C00250E96AA96FF730DE2"
		  "D92E69BD19C2EC40BA78F94B7D120CECE0B0A44890CC17529A0"
		  "54EDC97C25BF7AB6BC3C955466F656FE450BB25C34997DA9425"
		  "8A4473128DDBBF6EFBA41C1843F69AC2B86A5F13EA652EC19CA"
		  "A10FD0D605B4F31CE170F9FF1D6C7DC064EB4106C6F4F54C88A"
		  "463EA9B97206E567B370DCE4DCA5B998831362DE38D8DCBC0FE"
		  "04E87BA9EC6CA571E6A3BDC246D316A55FAB5CE8AB0B71E0E8E"
		  "DD9790B26E3482A95C732A304C1DD48A3955ED995DDB86AA7B4"
		  "E6406D08E189596BC8C5A5F61A2AD336E39D94F6188DA5A1E4F"
		  "2CC9593E92B0D89E12CB4E6FDF89C5C24ED83D66F5AC6BB0A6A"
		  "3C88B151C35123DE2F3D2D18D8D302FB10299F68F65C9EA2E55"
		  "21AED6CB9FF5E357DDC1B9D4D5BD7B13DC9B4FF63941BEC300C"
		  "C787FA6DD0BC617F5389EB6058D05BFCAA16F08393F4F1DC5FD"
		  "4226542F49F771E5E93D87E06C4E663ACFB2CC365F3F7901635"
		  "6656F92EE52DC6DDE14482E2CC8A9EDA71189E3C5A431EFE0A8"
		  "768D014F05D45685E472BF22E359B8A6C8429D8F048314BFEF4"
		  "9AF2637108E774D5E41D6B38FDA700FA2145B0235E1DBA03BC8"
		  "DBF3"));
  test_hash (&nettle_shake256, /* 202 octets */
	     SHEX("B6134F9C3E91DD8000740D009DD806240811D51AB1546A974BCB18D344642BAA5CD5903AF84D58EC5BA17301D5EC0F10CCD0509CBB3FD3FFF9172D193AF0F782252FD1338C7244D40E0E42362275B22D01C4C3389F19DD69BDF958EBE28E31A4FFE2B5F18A87831CFB7095F58A87C9FA21DB72BA269379B2DC2384B3DA953C7925761FED324620ACEA435E52B424A7723F6A2357374157A34CD8252351C25A1B232826CEFE1BD3E70FFC15A31E7C0598219D7F00436294D11891B82497BC78AA5363892A2495DF8C1EEF"),
	     SHEX("F723334037CE684CDEBEEE3F14B0D109B075EDE85C489EDF798"
		  "1B3BF88B946E2851C9453B0E891349080405BC3B16A99879FD6"
		  "3AD7314FB7D26C050061D3A2F971E2A79C4B23338FC92E21466"
		  "C27BEABB1FE045A0A10F33516BD77E7D87C49003F1CC173B61D"
		  "4ABAE5151D5A723C2E6571B0B7039FCFDA72C149744512D5FA2"
		  "0DB8C736A70FC009211FDBE0D7245C5C26DD878CE52DA760AA9"
		  "0D45A5501EFA74677B85A736B2860876B23424D1CC47660040E"
		  "919DDEE1F67314FD902A1D5C1DC69B70BAEA890B5691322C81B"
		  "210DDCAA3CAABCC770E68B1BBA5B08C3D7676771608924A853F"
		  "D770B2AEDD8D42B1E7CA6CC0499E22646732EDAEFB94761405B"
		  "76C3A58D5CDD1439ED1BB06CEDEC79DCDF38E373ED17798A51B"
		  "80665C026DC2BBA3EEAB43CB9024FF6F04B371C07423CE2D0B2"
		  "E2A800B213D9FE193403FE868180569EBD002591959B8687E09"
		  "F81C9B2B8DE7672D173F8EBD0794ACD309A2629ED0AA7E14B58"
		  "635115A58CEAD5BE08D3105E2E65429D1F7823BD8DAB99084E1"
		  "D63BBC637EE37AD4627E57E15DE35CA52FB8407760BD2AED167"
		  "B6EA65B0C72B9B53C58A16D030A227CD92F19F14BA5AD6BB00A"
		  "5775E4A091D7844AD2DE44CB9D866154127CA1B1A2422483A14"
		  "6C729AEB85E67D597EC6F700C9DC615F2BDFA6A1C9AB038B81A"
		  "D0307DAEEFF021EBE7ABBA47BA002C27C2A07611027FC69C9B6"
		  "3992"));
  test_hash (&nettle_shake256, /* 203 octets */
	     SHEX("C941CDB9C28AB0A791F2E5C8E8BB52850626AA89205BEC3A7E22682313D198B1FA33FC7295381354858758AE6C8EC6FAC3245C6E454D16FA2F51C4166FAB51DF272858F2D603770C40987F64442D487AF49CD5C3991CE858EA2A60DAB6A65A34414965933973AC2457089E359160B7CDEDC42F29E10A91921785F6B7224EE0B349393CDCFF6151B50B377D609559923D0984CDA6000829B916AB6896693EF6A2199B3C22F7DC5500A15B8258420E314C222BC000BC4E5413E6DD82C993F8330F5C6D1BE4BC79F08A1A0A46"),
	     SHEX("37A534CC68DE78F87B85212464858780E143A5E0DC7D46419EB"
		  "17E45E1872186DF8051353A6AF703C167B9233612ED68E95352"
		  "44DD99A613FB43F277C833588F7A2ADF2EAD15ADA2D6ED7CCA6"
		  "AC786741B7453EED605C7F8EA3A8E2AA8D5688D72EDF964C085"
		  "42422DE589D402DA18E10E7ECBBD132A10FE382A7EA0A94C9E5"
		  "C7F31C7BDF018F44BB6693EC64B266E1AEF1A3A941E336E479E"
		  "FBE2BC897C93576F5F1BC9C6D4F6E948A14464287D1E82DFDD1"
		  "E0CF0A68E9FF696B5D5392F4C46845DF19AD5BBF2B158659E85"
		  "6BBF6732A9AC0ED722BB5EB2C25D35FF83B892DD30BB01FE20F"
		  "49A28C52B6F28550FE56D176790AE2D96648C66E389569BC2D4"
		  "7D0A5775E0F6F1B7AB8F999415B3EEE8AAE8A3B663F13E0EDE5"
		  "C81CD001307FFB5AEA2801CAFCD1B09D31182220F52D39855E1"
		  "DF732F1CF2671538AE37726B3B47760830B391EDBBAE70B52F1"
		  "F2116762C685FB8523742D90A427C991F80DB823E3496994566"
		  "10720744BD2E4FB9C40D1371DEBDC7D01B2F5BE45C0231D297D"
		  "E09D66616509BBDB854D84A2D7C140E8489884794D3679FF8E2"
		  "4F7F04CD72058EC305EC21823AB5A47AE9AF8D10EFDBE2642A9"
		  "7DADD44F174622597E4BD88698FA75F24314E82ACA60B42F66F"
		  "84602C831FB9FA73E95955151F931841F1B9DE7C3F95DD78689"
		  "3B45DDF66045AAE6533AAA59060B415FE392BC585EB4C39115C"
		  "C163"));
  test_hash (&nettle_shake256, /* 204 octets */
	     SHEX("4499EFFFAC4BCEA52747EFD1E4F20B73E48758BE915C88A1FFE5299B0B005837A46B2F20A9CB3C6E64A9E3C564A27C0F1C6AD1960373036EC5BFE1A8FC6A435C2185ED0F114C50E8B3E4C7ED96B06A036819C9463E864A58D6286F785E32A804443A56AF0B4DF6ABC57ED5C2B185DDEE8489EA080DEEEE66AA33C2E6DAB36251C402682B6824821F998C32163164298E1FAFD31BABBCFFB594C91888C6219079D907FDB438ED89529D6D96212FD55ABE20399DBEFD342248507436931CDEAD496EB6E4A80358ACC78647D043"),
	     SHEX("60C681E527396D6C9DC1E5FD71842F760F9873621A9BF2B9B98"
		  "C5DF450DC7194379991D7A94397A206913AE1821259D308A5DC"
		  "4D53E552DBF2287B8694F0B6184B737D5E32E1258CD7FF58E5A"
		  "A9B5F37571413395B833CCC5620E37B0A64112968D4D7ACFFA3"
		  "A66E044C36E23363F1A09EEE48EB6713FBACEA29ED63AF0DB94"
		  "ED2BB22CC01B89D05391C1BC8844BE8BF105448925AD2373D46"
		  "D3417F920046E8E4AFA0C11A8057B4A42FA4CAFEC592F8C2F8F"
		  "1E2D603612FA38DFC3D3FD768EF7317802342E76E70D30745FB"
		  "AFC0BD007EFF874CB32761D700C88E27DBEDB57CBFEEA688E0F"
		  "06801847C1046CBF47E2C12EC0A4016D5E8238916DA887DEAB8"
		  "25F965F936C79A7316465B48A2B8BD5889BD04580FF7DC91C50"
		  "DE8EB60AF74E74BB4F896FF73CD3D1C55948C19DC0ABC841D7B"
		  "C3384AE4D60BD855F24BD2E402FAF5E47D66DDD2F0533F9A73F"
		  "D483E3965E967C979382F643CDA535C4AE3AECBBDAA51CD7A89"
		  "C8B2D9EBC0B7F9CC1C1EA355F9FECCB62345BE39A7D1073CBA7"
		  "C5C46F2163DBC1E9A6190EF620D6053B88586E73B18E816206A"
		  "72F297EF946C8B302A6E3FC92A3E7A9AA87E26C1466251CE1BB"
		  "D204BBA7D08FEB28F99CF27BDB3E8479C9C57FAB8F00797938D"
		  "471D53A1666156B66853C067E64695978D2E9AD5250FFF6CD14"
		  "91F7BAC09B77ABF42249FBBD63DEB5919A1FC96A4E26DB9D789"
		  "320A"));
  test_hash (&nettle_shake256, /* 205 octets */
	     SHEX("EECBB8FDFA4DA62170FD06727F697D81F83F601FF61E478105D3CB7502F2C89BF3E8F56EDD469D049807A38882A7EEFBC85FC9A950952E9FA84B8AFEBD3CE782D4DA598002827B1EB98882EA1F0A8F7AA9CE013A6E9BC462FB66C8D4A18DA21401E1B93356EB12F3725B6DB1684F2300A98B9A119E5D27FF704AFFB618E12708E77E6E5F34139A5A41131FD1D6336C272A8FC37080F041C71341BEE6AB550CB4A20A6DDB6A8E0299F2B14BC730C54B8B1C1C487B494BDCCFD3A53535AB2F231590BF2C4062FD2AD58F906A2D0D"),
	     SHEX("D8EF59E48FE97B07649929B9194BCC57EAD8E0EF010FD65031C"
		  "18B4CFCC0933152054AE17841D06C36BC375CB1F98D83E2F9C3"
		  "DFD54B88312AE71D7A00590A759E4B8D524124CF480BCEC6816"
		  "FCEECABC9E13356F99ADA2319C3EB0CC9E3BB504EDDCF7D3839"
		  "511BA0B412A72C3A8C49F7964A4F2EED7B814722A487E3EA6E3"
		  "47CDC2585090130B4DC535C3FA98E085F107FA43222B167EBDF"
		  "2BBD65ACE691D8C23D64DE52F966983A3D4479CEAC8AAB17794"
		  "7079656A454B7BA4548BCD5762974264C7BEB17DCB4FDAE572B"
		  "D67067F304990611E7C7F4D50BBD2B29D19C02D51918D5BE6E8"
		  "56BFD8DCECEAB2EA9F82F5BE53E57D4B00AC371436EFF3898E9"
		  "30193DD1A21C5B54506C6C91EA665D0D9992BE999F1DC434BA5"
		  "87DA4EF05BB296087314C4408CE0C5851660D0B5D1746232E03"
		  "1934284F85B7AA38420FA6EB469171FAABE88F85CE64A7FEF84"
		  "20ADA1F9BAAA4BC93D779C5A268A3F1F8A57CD2FD3B19CDD4E6"
		  "48E622552C1F16455F32631A3D91EC2F7A882DFD0EB13C4B85B"
		  "0A23E2477B53E0FE5E174F058268F6A24BA3CFA38D090AE6088"
		  "339957863733E1DA245432AB0394EAC05ED82428C4418DDEA00"
		  "C45E57A1719C7D2D04A781F8D0187C7C33117A3FF5473BD5C12"
		  "11476FB4EB64AEDE5A9E2DBFEE00671E7A9CCFEF7624E1F688B"
		  "77CBC755B95B6EC391D8E9DD0B65CFF4E1FC22F566F0B598BF7"
		  "ED6E"));
  test_hash (&nettle_shake256, /* 206 octets */
	     SHEX("E64F3E4ACE5C8418D65FEC2BC5D2A303DD458034736E3B0DF719098BE7A206DEAF52D6BA82316CAF330EF852375188CDE2B39CC94AA449578A7E2A8E3F5A9D68E816B8D16889FBC0EBF0939D04F63033AE9AE2BDAB73B88C26D6BD25EE460EE1EF58FB0AFA92CC539F8C76D3D097E7A6A63EBB9B5887EDF3CF076028C5BBD5B9DB3211371AD3FE121D4E9BF44229F4E1ECF5A0F9F0EBA4D5CEB72878AB22C3F0EB5A625323AC66F7061F4A81FAC834471E0C59553F108475FE290D43E6A055AE3EE46FB67422F814A68C4BE3E8C9"),
	     SHEX("984570A811CB6B53213280F8C06C69FF1D17E767B163F8AFD76"
		  "674AC018A1CAFDA4D94B69BCF58F3AE8A53B107976ABEA6C616"
		  "D85AEB98BC10F29177A3590BFE943F53D2512323E5E79D64F9F"
		  "1319D47D6BA84C7EA37392BD15224D94BB16D99E0E1E7062FBC"
		  "3EDD7C81418D816E875FBC9D7434249D89A6002BB777E367F79"
		  "219BC7FACA03E1B9488307DEA98F304258FBCC53C58A483A6FC"
		  "C78D9D9B28D99DB3709E32C1EE91FE305F72B041933C8CAE6F6"
		  "8C6229658D591CE9B1100D6974706BC21A45224778D12DFBA63"
		  "CBE9A1A3FE9680A34771D9BAE39B4DC9C191C0A4E092F8EE92A"
		  "F2A02C95242CBB24580E04774AB41B07FCB6F61ECC6ACB2E48A"
		  "77E544AD9F25EC0ED7F440C26B8C9D25C7CF4494AF0802C4B48"
		  "2DA80251F7D1572DAE337725B9A976F70BB70B5037F333D53BC"
		  "3C252AD9AA13EAA7C9DE1AE4B24F6C91D17A489AC9D392038B8"
		  "0EF7679E75CA6994821F153C62D6CD7EF58927636EDF95EF6F0"
		  "5520025C4BBBA23685A862F244AB3802AE1D842184A5F5F344D"
		  "1193E1239990C331488C1A8B98F1707C389C05AFA919F5C17EA"
		  "C158064350ADFB9532EB0AC9111E256251B11D082DB28206AF2"
		  "AC514707D956A35E26495EDE4E98C2578427B48BC99B9AEB29B"
		  "0FCBE44A0E51CE9BB37864DEDC30BBB5FF9A055423D8BBD453A"
		  "65AB54E65CDF773CA695303FFD4767193D65F26706FED1CF35E"
		  "740C"));
  test_hash (&nettle_shake256, /* 207 octets */
	     SHEX("D2CB2D733033F9E91395312808383CC4F0CA974E87EC68400D52E96B3FA6984AC58D9AD0938DDE5A973008D818C49607D9DE2284E7618F1B8AED8372FBD52ED54557AF4220FAC09DFA8443011699B97D743F8F2B1AEF3537EBB45DCC9E13DFB438428EE190A4EFDB3CAEB7F3933117BF63ABDC7E57BEB4171C7E1AD260AB0587806C4D137B6316B50ABC9CCE0DFF3ACADA47BBB86BE777E617BBE578FF4519844DB360E0A96C6701290E76BB95D26F0F804C8A4F2717EAC4E7DE9F2CFF3BBC55A17E776C0D02856032A6CD10AD2838"),
	     SHEX("2C2E6AFFFDC10D54F74AFAAD553CCA0DDA3AE5A5EDE3BEB611E"
		  "4C1441AD4E51B67EAA2306ABBB39115457FDF15FCE9607EBCF0"
		  "537E4F9AA2703D222A2B5FE602F87F6B54CC7A9427807A10E56"
		  "BE5B29BC3DD91FED0322F291D87F563BAF5B705534AE8BAC35D"
		  "4AF68A23963C040F4A1AED9B3E51124303DFEBBAEDBAB35D7CC"
		  "BD38D482EE45E261BCEFC53D0A25DFCA5DFB60CB2672D34CD41"
		  "54873F92B33FAFB8068A95458BC1166B36C3977643BA159B655"
		  "654D1755FD5E396AB9690718266EF6AE670452FBE332BE84750"
		  "BF7A26778E555FDB59AE0F48D6EC3FF4A1FEABF2FF7F1CE4DBF"
		  "3269DCD91E3CA9962D88DB9C527874DF15F7A6DB9682B1F0D90"
		  "4AA875D29A8885AE63E90461C69B688046FC771321FD6A61A6F"
		  "1A93FC95FAC72BF86ED76EEBDFF0CD6960D72FC460753ABA032"
		  "4F2D7CD7372F413D4A49E761CADE4F5911D18F7C712F61CE1A5"
		  "C9F07ED2B042D5C86B4DB97E1B7ABA57E2F2DB98CCB4041A16E"
		  "E21E08ACE213412C6D61CD3C42BB5D8DABD7B5A945FE512F2E6"
		  "4C3B1817D9BC5840D6E16F7E5E2AC36585BF8107ADDD0C269B8"
		  "510DFE11222DE1BE13CCC2A5BA8536887A11A0A733EB81D9372"
		  "9D487481655F531BF4AAE624918401097EC0951F7FC535CBD85"
		  "691E0E0EEEAC4A5848A61AE8D6D6121D3FBCBA345AB755C4E84"
		  "41ED7853FF46CAA48090728A7AC39496C3002BAA99DC90F049E"
		  "FE4C"));
  test_hash (&nettle_shake256, /* 208 octets */
	     SHEX("F2998955613DD414CC111DF5CE30A995BB792E260B0E37A5B1D942FE90171A4AC2F66D4928D7AD377F4D0554CBF4C523D21F6E5F379D6F4B028CDCB9B1758D3B39663242FF3CB6EDE6A36A6F05DB3BC41E0D861B384B6DEC58BB096D0A422FD542DF175E1BE1571FB52AE66F2D86A2F6824A8CFAACBAC4A7492AD0433EEB15454AF8F312B3B2A577750E3EFBD370E8A8CAC1582581971FBA3BA4BD0D76E718DACF8433D33A59D287F8CC92234E7A271041B526E389EFB0E40B6A18B3AAF658E82ED1C78631FD23B4C3EB27C3FAEC8685"),
	     SHEX("E36806AC60D67682D34D80AD11A1415901FCCC216D7548DE3F2"
		  "428B1963023A9394BC0E1766E279AE3509AE85D773D11DC8F11"
		  "AE65C5C0D6401575CB33835081492C2B657F34514ED6B0DFCC2"
		  "639FB758B5FDDF638576A64BD34759D629FF133112CC5200267"
		  "D8C9997E1538D602CA356E821F06EF84C3125D8466673312C69"
		  "03CE166FB97FCB3EAC9D30D61E13A1DEABB922DA6E5675DC897"
		  "6CD4426EDF7A7A797DED8936E0A3E721DE5C4D956E8287C4B05"
		  "D1210BBD36CE0E36B7F2A31945FCA6BAC1CAAB3C46D4C8245E3"
		  "E589DBDDBD00A3B99D8DFE33783235026C1A990F1C383CF71C7"
		  "55446D8F5DF4A114B9004541065BDB81AB81B98ED47A1D05595"
		  "AD5E168159EC620B1DA439C6CCB6C4D57A2ADB33A0E113FD0D8"
		  "ACBFF5C759BFFA29B36E8416EBA1B8F45285EF3F496F6946EE1"
		  "DDE798CF8F5BB78900E59785597101DAC9EC5E5F5CE0F0D6B17"
		  "3B6418A75BE30A84140FAC4899C4EB8DDCF87A0F78F22E23780"
		  "6E0871CDA9976DA0E7E42C4AD73FA49D3F9C257ED1CC6C38980"
		  "6BB2C32B1F7744DDCA53BA515237F1588F5FA7C636360B6B8D9"
		  "8DAFA7175DB320A1A3FD1B3A092D9E8A4466971B87E5B051898"
		  "29AD8D3B407F3C5E4265026C8B801C391BE0F9C9DBAB2F56362"
		  "F820384D859EAE1ED9B291F5F03F14F12D0FF8ADDAC128988A5"
		  "9707FACBE23814873246C99DB793099E6DC79712C612D41302F"
		  "8B59"));
  test_hash (&nettle_shake256, /* 209 octets */
	     SHEX("447797E2899B72A356BA55BF4DF3ACCA6CDB1041EB477BD1834A9F9ACBC340A294D729F2F97DF3A610BE0FF15EDB9C6D5DB41644B9874360140FC64F52AA03F0286C8A640670067A84E017926A70438DB1BB361DEFEE7317021425F8821DEF26D1EFD77FC853B818545D055ADC9284796E583C76E6FE74C9AC2587AA46AA8F8804F2FEB5836CC4B3ABABAB8429A5783E17D5999F32242EB59EF30CD7ADABC16D72DBDB097623047C98989F88D14EAF02A7212BE16EC2D07981AAA99949DDF89ECD90333A77BC4E1988A82ABF7C7CAF3291"),
	     SHEX("939F4A4B5E37B675450782B0E8554BD6A2821EC805CE07FD4F5"
		  "A3BC3816A2305F2353FFAF15883E760A3DEA064DF1583CFF5ED"
		  "83A97A62DF9D174AA80958E7649460FB4880DFA21DD7C00F373"
		  "AA9F010EBD1DE7E1AA73F51F84DF36BD2DCE6B37D7A1112C6C6"
		  "9EFB734CCE2AB12517FAE38F0F3592A4693251CB4F41AF7E820"
		  "F1E6F432CB9F88B82FCF579B2A5FD0E9A5B3C28542ABDAA9C65"
		  "1EF1A05A385051682B3238262475AEEB53894B8274EF37B223F"
		  "66A32C0D3DF45F3F191414FC0E8878E3D58852B4CDBD826D236"
		  "A7010CA49C43095786CDE671070EDE3F5BB74F34DFFACEEE281"
		  "042EAF3D2B39E53C25A61F3B0B2D79E4305B9A17C47787747D3"
		  "CCA46E88BE79464BDFD16BB44EC7D862503077D3E4E3C55BA1A"
		  "1FCDA217D53F5C22A35F883FBD0E786AF38276B341733F82BEE"
		  "BC4118D486C32A3E7A460E24951BFBE760556E36981805B40B8"
		  "37B514FAC3E8213937ECB018449319AD3FE1C0EC19B502E8E03"
		  "99398F3F8C74601F61A9BA3021F153ED45A99BC150AA886BF09"
		  "0B0134DA25B2740EE5156CFE28151EDA24A2C758303F505E465"
		  "F9D6A409651D4705A80B3FF660B2D66AD061960A3C6B81FA8D8"
		  "31659830C5243AAE26334917660FFFF3B977FBDA3730313753F"
		  "ACF7E319CDFEFC064F2072AF7195A70EA74DB24E5FBD36D6FA7"
		  "8C0D0D1085D05E81A62D3449F16007D3CCA29E4403DCE5DE01D"
		  "579B"));
  test_hash (&nettle_shake256, /* 210 octets */
	     SHEX("9F2C18ADE9B380C784E170FB763E9AA205F64303067EB1BCEA93DF5DAC4BF5A2E00B78195F808DF24FC76E26CB7BE31DC35F0844CDED1567BBA29858CFFC97FB29010331B01D6A3FB3159CC1B973D255DA9843E34A0A4061CABDB9ED37F241BFABB3C20D32743F4026B59A4CCC385A2301F83C0B0A190B0F2D01ACB8F0D41111E10F2F4E149379275599A52DC089B35FDD5234B0CFB7B6D8AEBD563CA1FA653C5C021DFD6F5920E6F18BFAFDBECBF0AB00281333ED50B9A999549C1C8F8C63D7626C48322E9791D5FF72294049BDE91E73F8"),
	     SHEX("3C78C3BD49493560231A71A867BBB666536B22562F150224599"
		  "1E3FF28E6F058264B382BBD61F9A16D4DDA0B3A72F84606F97E"
		  "7CE53F4A57C132A920FE738F818E987095C1292D9873A772E12"
		  "F74BF42EE1281F408DC2ED7BCEEB2FA6FD856B7A01A95535058"
		  "2C111B2CFBB65E26B745939E1DD4781A0265EC5AD4BDAFDE1EA"
		  "40AD504EFB0E9F34E15E1526B1873F251D33C280F9CAD71BCA1"
		  "7C4925AF9C3DBD4C6798BD393625017C5D3B03666FBC3AEE276"
		  "AE669AF2E9E10AE2D608247F3DCE876A8EA3BEE6EE001C3D5F8"
		  "47FC92A2A9CD0FDDEAD45C1F92414B23722005AAF8A7A0159D9"
		  "1B77F4A0E44DEBB9A0967977B04072633B4588DF0D699C8F990"
		  "D60FF5A2E09690A624DBE88B998424620B5634ACDC02C75DED6"
		  "DBA9431AA7A0FEBC01CC5DAA3094CDC818B2A6CA4D808901134"
		  "08565A714558B6BDC3E1735BD1FC9FE84210D7D78E0C50E8961"
		  "C39725F68F083094277B7B7ED3300A1FBF42F72AC9A79FF927E"
		  "4F76ABF0BA233A9E82E535C32D7054078A6A4A63F1EA454C6AA"
		  "33C5A2299DF7BECB5F911C25BD74532EDC882FF4312D2AE0ADD"
		  "4D0F67958A520011851988F9D3846C0100B4C119EA818062A3C"
		  "69401344CF94F49AB995C286F441D8241F60A73863628AD8AE1"
		  "CC780EA99CEFF0DC18A3D85E16CA00756627F6A1B7546359451"
		  "20890313EE136438E01A41593B6EFD26926F48309122CF1DD80"
		  "6C8C"));
  test_hash (&nettle_shake256, /* 211 octets */
	     SHEX("AE159F3FA33619002AE6BCCE8CBBDD7D28E5ED9D61534595C4C9F43C402A9BB31F3B301CBFD4A43CE4C24CD5C9849CC6259ECA90E2A79E01FFBAC07BA0E147FA42676A1D668570E0396387B5BCD599E8E66AAED1B8A191C5A47547F61373021FA6DEADCB55363D233C24440F2C73DBB519F7C9FA5A8962EFD5F6252C0407F190DFEFAD707F3C7007D69FF36B8489A5B6B7C557E79DD4F50C06511F599F56C896B35C917B63BA35C6FF8092BAF7D1658E77FC95D8A6A43EEB4C01F33F03877F92774BE89C1114DD531C011E53A34DC248A2F0E6"),
	     SHEX("1A73A838FBE2AEAC592433B7CAABFD17068510B165896F00217"
		  "A1F9E2093591A77330E656FC53D1223BC9446C11D329673B418"
		  "6A85DC6091140C070843760EFFA56918F414A9C6F2D198E7D0C"
		  "19F44AD19D1E89E25365CEF40890987D255A616524AD68574C3"
		  "C284825A48C3BCB0B48D41EE284B53BE97127DECF90402E8FE1"
		  "317263D179814177B79DF92B9C71EFE9C3C2F3CBF8329BD9711"
		  "E6393457F076C8BCF16A70E854DAF7BCE58C31F55B4D0E9681B"
		  "39FDC19D0703A795C234016C5879857F25A64EAF0700C681E59"
		  "0B6D29605311F1BDBEA47762BBFB012BF103D34C2CAB53D53E1"
		  "962BA6850B779B8673BA280B0FB79CC387E2559105DDBA2884F"
		  "E493B85EE0F0747436013CED0ACE7280854E9BED8A42FC98DCF"
		  "1AE5F845B677F4B802D01AEF9529FFFB6ACFBA047429D4DE318"
		  "1FA959A3B74AEB5934E3F25498CCC850BFAED46738736DB5E5B"
		  "DA73AC03162923D310A1F91658D15CFC5268FD5BCD9AE6F9D56"
		  "52558F74C1DD85696233AB9EFEF5CEEE75F0A3489F46867B0BB"
		  "1344CE24E5BE88ABB706CB49A07FEE3DD5911DCC0486B58FF2B"
		  "024F844C3773A1C03CE35F02256157A6A2E6616569FB751ECC3"
		  "E4CF01EF99BB0EA61E4AEFEBCC65450EF2999505427B23C7CB9"
		  "278E27AB58C5FA2D5B0A432A6172CB1C23DAC420C92631495DC"
		  "30DBE0AF3B6944F1E101D1566F0A1F4BA06B82C36A7CD068864"
		  "FD3B"));
  test_hash (&nettle_shake256, /* 212 octets */
	     SHEX("3B8E97C5FFC2D6A40FA7DE7FCEFC90F3B12C940E7AB415321E29EE692DFAC799B009C99DCDDB708FCE5A178C5C35EE2B8617143EDC4C40B4D313661F49ABDD93CEA79D117518805496FE6ACF292C4C2A1F76B403A97D7C399DAF85B46AD84E16246C67D6836757BDE336C290D5D401E6C1386AB32797AF6BB251E9B2D8FE754C47482B72E0B394EAB76916126FD68EA7D65EB93D59F5B4C5AC40F7C3B37E7F3694F29424C24AF8C8F0EF59CD9DBF1D28E0E10F799A6F78CAD1D45B9DB3D7DEE4A7059ABE99182714983B9C9D44D7F5643596D4F3"),
	     SHEX("674343644B198DA1CB0709F842239CA1559AB643E48F8EEDB37"
		  "3671473CA24A99A74E20AC6368B661BA1824AF92985B3E7F957"
		  "F9C935C715531703A6B8B5113A364F52C7DF7C5488997A3FF71"
		  "873FDACB1A6F4AC3B631A3341BC36CB742F8100B28029ED6C44"
		  "FCB41E93B6F171A064B0F13B2F83B1790330DEF22A387C52CEF"
		  "7F16D1E398123CA0231B76494D708CC694A28E281B34ED163B7"
		  "931312DAA56752080C4966F8638134680CB96C4F815AC678106"
		  "FB243ED3B9C9A5599D3851BF3D89CB7BA463C71A868F74D07F0"
		  "C6F9C6F940F53D88E18892EE6CFE5FE9C17B92569E5C1222FEE"
		  "87792987663757459C35630CB355F445630826807909BB5CBC4"
		  "F5C8E0BC669BE6C705C12BCB2B39552D4AF42204D237FAC5BEC"
		  "56EF9AE6F60BA836AD865983DF2F25A1F8C3F5DF93F87EB7647"
		  "2DBEBDB20FDF894C0167287EF1A6355040C4D4A6949F4886974"
		  "98529217E4532A445CDED47687CFAA14DB98260DE9321A16FE8"
		  "4E4D82D2B1B0C2EE639B2ABA48D85E4D06A369CDD9B03A0A846"
		  "EBF83ECD5ED257876BD5E3E7838108FB3B86B6558F8BB454F1E"
		  "76FABE18F3B66CA1448C1B02DAB1BCD485E709CA800D5271952"
		  "602E9B1BD585F61936D8DD678C900407892756A08730B891C15"
		  "4B33D83BCD1B2F441F1DFA79AF235316BE9A3E20ECEB4EB23AB"
		  "76D9E9564C367BC27424DAE397E35BA16A8CD4BA15CFB7D9EEE"
		  "2B82"));
  test_hash (&nettle_shake256, /* 213 octets */
	     SHEX("3434EC31B10FAFDBFEEC0DD6BD94E80F7BA9DCA19EF075F7EB017512AF66D6A4BCF7D16BA0819A1892A6372F9B35BCC7CA8155EE19E8428BC22D214856ED5FA9374C3C09BDE169602CC219679F65A1566FC7316F4CC3B631A18FB4449FA6AFA16A3DB2BC4212EFF539C67CF184680826535589C7111D73BFFCE431B4C40492E763D9279560AAA38EB2DC14A212D723F994A1FE656FF4DD14551CE4E7C621B2AA5604A10001B2878A897A28A08095C325E10A26D2FB1A75BFD64C250309BB55A44F23BBAC0D5516A1C687D3B41EF2FBBF9CC56D4739"),
	     SHEX("791BE2551F426CD350322EB1AC0E6902996546D446C229E03D7"
		  "4E25CDFCAD06CD99BDD3658026EDBE8708AD18DB91305770245"
		  "0BAA457EE767B7F6085B0F1D35A86D8C5C62A35E5A11B81D409"
		  "C12C8AD5777E245ACC369ABB5EA9D8D9517AE7FB4033EFBBBCB"
		  "B70FD6CC1E55187AA8A27C75E5116EBA46DE2C8378997504A82"
		  "390E6B085A45C5FAE52FC314E876DA7A7F90226CEDD3C4E9858"
		  "87F162513BDDAC323E85737F3954DC58D12E986BB1840D0EB04"
		  "6102C01381DB2915981B23DECE4987661B41B56D30C35A13DA2"
		  "1C6D2AB0BB7951F04804986DC2127F7F33C4937D27427C7B706"
		  "2EF7EDE944EFD0015681C2FA0C115B67652A7EF1AAD03C0FC45"
		  "42BADE7CDA11F0AFC57D8E2C6DF0E5BB5440C2B3ED2807C2798"
		  "CADC6F371A2D6F2D647936B82936BA4B19A16F599689814AC4D"
		  "E20CC7CD05A3452D9286B345037ABD23850F22A56BBA707B1D9"
		  "CF299ED3EA21943882669735F69D25F240F87F3B34CD794B0C6"
		  "353681006AAC5AD75CF46BC6464579BF5B715AC741CD1CC06CD"
		  "CA6E8D579F2FF839E263C94588CCD268B1D375E5DF5BFC8B7A8"
		  "9FD72CFBF9BF5400EFEF53186B50497D942963EEDCD6A742678"
		  "4F38F8808C50F867D22C10FA4D7DEDB4F5354FE1EB1484C9F64"
		  "5397582838F1C78076891C5E1E44959B5C74E28D7BCC3F6ACEE"
		  "4E5393A5CE24E97CDBE1B22AE853C567068ACEE03CDB3157A3E"
		  "F657"));
  test_hash (&nettle_shake256, /* 214 octets */
	     SHEX("7C7953D81C8D208FD1C97681D48F49DD003456DE60475B84070EF4847C333B74575B1FC8D2A186964485A3B8634FEAA3595AAA1A2F4595A7D6B6153563DEE31BBAC443C8A33EED6D5D956A980A68366C2527B550EE950250DFB691EACBD5D56AE14B970668BE174C89DF2FEA43AE52F13142639C884FD62A3683C0C3792F0F24AB1318BCB27E21F4737FAB62C77EA38BC8FD1CF41F7DAB64C13FEBE7152BF5BB7AB5A78F5346D43CC741CB6F72B7B8980F268B68BF62ABDFB1577A52438FE14B591498CC95F071228460C7C5D5CEB4A7BDE588E7F21C"),
	     SHEX("7C0247B191C924335D843B81614C0F1A6A9AB1ADB6C28692239"
		  "7D8C1B26220A35C13C7151E35FF9B3653A086DA339EA1F05881"
		  "F414E7A16C61B1453287F924C48EA933995000A9C9B0EBD5E8B"
		  "EF2C92820CCF9BB6CB0F465AAC124B5E794FDBCC60584B05CC3"
		  "BAD06D41CF9DD5D3D3B7FE1CEF634DE558B6642E591BC2F69AB"
		  "EAE61EF5DA27BEDF291D2A4106B3DA946FB57328E827C8D4EE1"
		  "C3665AF79F96CC290F924348C30205D75B01BB05AA9DFC317CF"
		  "DDEEB7E3764589C003E2AA1DDF60A3C98ADDE4517E40D08CF80"
		  "646205EC9FC6E9A3E1B64B2EAD561957C3E2710DFF720BD7E58"
		  "41481D9E1A048D1C76A3E1A08BF010428C8FD2A6F2DCA4FDE7C"
		  "8CF59C826E6DF246CDE77D5823DD1E0C0759AF8356E89ED4C03"
		  "276A37EBBA73DC196E7915B57BAADE13A7A276AAA4456E23E44"
		  "54D44260D9EADD748D4E7E1857B6D2D9C301AF6118270859ACC"
		  "43603C79A2F6D698871FD5384DAF43B16FEE4ECE1146ED54E1C"
		  "A28EAAF2D94306383609BFB26D6634DF00198E0D19500215E17"
		  "8D77329F7504AF80DC6B727EA7791305FB38FDDB1CEF7F626A4"
		  "9AC6E34C3A6ED630F9310F890392704A2CDE00CC850FF7CE9F0"
		  "E1F0CB4B1D0C161CA3B812309CB3C9E9D071024D61E5265A71E"
		  "7ADC58A3D47A567F5B19F9E13FDCDCD34F3DEB12DA3151FE817"
		  "E5844F5FCE12805B064895671F0C3767AF5A3D203D095BE306D"
		  "4A26"));
  test_hash (&nettle_shake256, /* 215 octets */
	     SHEX("7A6A4F4FDC59A1D223381AE5AF498D74B7252ECF59E389E49130C7EAEE626E7BD9897EFFD92017F4CCDE66B0440462CDEDFD352D8153E6A4C8D7A0812F701CC737B5178C2556F07111200EB627DBC299CAA792DFA58F35935299FA3A3519E9B03166DFFA159103FFA35E8577F7C0A86C6B46FE13DB8E2CDD9DCFBA85BDDDCCE0A7A8E155F81F712D8E9FE646153D3D22C811BD39F830433B2213DD46301941B59293FD0A33E2B63ADBD95239BC01315C46FDB678875B3C81E053A40F581CFBEC24A1404B1671A1B88A6D06120229518FB13A74CA0AC5AE"),
	     SHEX("73F5EDF85CBE959CDFBEABBE2A96A1812134EC9D3CCB72D6039"
		  "D4FCDEF0B57BB72A782B76963C97AE0C2713CA657B9424851DC"
		  "903CC0488A169538A0DB3EC58E67C4EEA1C18ABDB2036C79381"
		  "E76F5C9F6CF1F830E0DAD77D2EB90F120EFBA42B20B15B8C8CE"
		  "8532C5B668464AD4986A281E75E4FB2CBC59ABF886A8540622C"
		  "2458FEFD190908A09FCD7F16B6E5BCC7E950917794FED485C82"
		  "CEEF7FE0FE4665E0922406AE1F8D34E5F77BFC0AF8B4C881BF9"
		  "F76C78A233898D120981BE96F67D135723407ED252432928C7F"
		  "2580FB80000CB51E3EE8DB02D4EF4FCDBABB704DB0CDB9E31FB"
		  "E76930D285702D91A74DB806DACCC01A15C0129A7F7D4712668"
		  "ED633462A6A036A9EC2C4A7E4AACD4D62DF9DA1FE959DB9ACD0"
		  "FB6F01A75F95CD9768F408AAB1760EDAD70E936EEC344E9506A"
		  "586B9D42DB99EF828E58D281128FB0EAC1185BE704CC9D16E35"
		  "9EE3BC1361A776DE4803132822F99F7E477764C9408B0457451"
		  "50AB8A7481D266995C91EACA7A9497F4CA1848828FB5ED127B7"
		  "A4A5FA0F23614165A8E4DE1E619A40D4E55AC28075AB2FF1BD4"
		  "2BEE19F2F9D63630CAC85EECA329A0A1877E848FB2A64176411"
		  "A7FCC2DD63D6CA19452780BBE4EAB208B0FA7B25F21F17BE1CF"
		  "7C5415A04FCAB687B7D4BA2F657E39D5EFC17F36A1A0E3DE461"
		  "0788ECC43CBE70BA2DDC33ABF4C335940DD53891FB6DCE20DD0"
		  "39AC"));
  test_hash (&nettle_shake256, /* 216 octets */
	     SHEX("D9FAA14CEBE9B7DE551B6C0765409A33938562013B5E8E0E1E0A6418DF7399D0A6A771FB81C3CA9BD3BB8E2951B0BC792525A294EBD1083688806FE5E7F1E17FD4E3A41D00C89E8FCF4A363CAEDB1ACB558E3D562F1302B3D83BB886ED27B76033798131DAB05B4217381EAAA7BA15EC820BB5C13B516DD640EAEC5A27D05FDFCA0F35B3A5312146806B4C0275BCD0AAA3B2017F346975DB566F9B4D137F4EE10644C2A2DA66DEECA5342E236495C3C6280528BFD32E90AF4CD9BB908F34012B52B4BC56D48CC8A6B59BAB014988EABD12E1A0A1C2E170E7"),
	     SHEX("9F303DECCEA9369400C533CE2E162818957E18F65C05A675404"
		  "CF8637FDB57F3251FDC2844CA194ED97CD021416E764E072598"
		  "106E9D97D420087087DCF5F8ACF1047A7B4518E683E5AC6DF1E"
		  "F1BA7EC7A418F12E386DFB7A4A5EDD9898CAE54D645AB319398"
		  "2F4363DA670F1FBC7AAA1194A03BB76391A78DAE7E4FFF733A6"
		  "F19393E9C4CC5EB3F5503D8B1CEFEEB7038B0565F29D6CD4460"
		  "090FC63E3C4DBC7D5D3D107F8E905B21CC9B9EF3C535AA0D8EB"
		  "CF787631744F80A05CACA3E50968A05F85E9CB0D775DAFCC714"
		  "FD6D93C8B110E58F9AD1F8077CC80427515FC5C0CF081DE5FD4"
		  "CFE6EA6DD2F33A2DF212918A3F86BBB8ABA45D808BBC2CAD8DD"
		  "09E7B9AF1C33B0E58B988268DB39118B2B2090EABADEC0B84FD"
		  "836DB6944E8E27B5F4F3B2876077D52B4648885EF55DFFF266C"
		  "7264C46EF5A3CB63588D1A86C82D19449E0C93D968619CA4D33"
		  "BCADCB21E6787A5E41F0D4E95E2B47E422FCFE5BD7EBC65A302"
		  "3479330E989B3EDC8320EDC868FB122DF078470B0373732E187"
		  "110D3964B6E7757E84F177106C2BFAC5BCC43A13DC40BA8AD5C"
		  "1E772C58DF0F32FF0F10348686CF0662F2F30D41BA48B121E1A"
		  "146C70E6131E13080CD8B63AB79869AF74E64B9899855C81D20"
		  "F753F22C67F1E4DE2450740944CDF4FDEDED59927F92A9568DB"
		  "95E5B8515C07BF4E9ADF952D4AD498F3DC19A2205640A8378A0"
		  "FD5F"));
  test_hash (&nettle_shake256, /* 217 octets */
	     SHEX("2D8427433D0C61F2D96CFE80CF1E932265A191365C3B61AAA3D6DCC039F6BA2AD52A6A8CC30FC10F705E6B7705105977FA496C1C708A277A124304F1FC40911E7441D1B5E77B951AAD7B01FD5DB1B377D165B05BBF898042E39660CAF8B279FE5229D1A8DB86C0999ED65E53D01CCBC4B43173CCF992B3A14586F6BA42F5FE30AFA8AE40C5DF29966F9346DA5F8B35F16A1DE3AB6DE0F477D8D8660918060E88B9B9E9CA6A4207033B87A812DBF5544D39E4882010F82B6CE005F8E8FF6FE3C3806BC2B73C2B83AFB704345629304F9F86358712E9FAE3CA3E"),
	     SHEX("1CC30597732904C49C0F6EAF777F82B1C33B4CF00FA720C7DB8"
		  "473139266E7304D81999F322AE6803736D242E7408DBB8955D8"
		  "F1C0B14C6F7678CACDF2F3CC2A17208033E8C70BC5141A072F4"
		  "1E759AD8EEDFF745DF0162C2EA877A7D673BC63D80C0F1DB765"
		  "F9F5CDEB2EFF214411F95AD7C3C41F6AFEC798E001D31AE2243"
		  "2A54BADC751F7F713B4A3693267BE3CDD3A1DDE1A5E638143A2"
		  "FAE59A2A359E197CF4E5E1D933DAD9F99773F164173E3E81898"
		  "7F19D122AAB725AF24497CA116EC201B2482F0539EC454292DF"
		  "4D018E286247F8C857B53816E9623D580D69FAEF60009FE72F2"
		  "5B4609E8DAD73A06C006280EDB0137A25D5462368B18B8F3752"
		  "5C2C74FFEB4A7942683B86ACA6FB1D773E9C92311C4E197C546"
		  "6CE90CF01C661EC2D928EE0D35A2FCF0EF1A87FA7D298747E44"
		  "DBAA4C90C655348B8C2771F378FBAF0702EF9CB4B42CD5B2601"
		  "6D8F5FAC1B79A49F8022A180F04F8988B5895822404D346A036"
		  "22A5DF8AD2E3F6841AB14644F6074A72DCDA253D7C9423DF9FC"
		  "15D98F49E3B2A1DF2011CEAA61BB7DCEB3860B9B66DF3926A1E"
		  "79E73EF6C4429AA9F344ADCE7D9A7451AD97D0C9653B916B029"
		  "C4971E6756FBA775F191751416C3B0587C2FC5043F38D2B6205"
		  "804C47DA3E85A99B47F1AAF42D9C709F21BEA94B8CA7340FBB8"
		  "B262DE6C47B79DE5222AFEE7B1C994ED526AD4D5641136F5227"
		  "A5BB"));
  test_hash (&nettle_shake256, /* 218 octets */
	     SHEX("5E19D97887FCAAC0387E22C6F803C34A3DACD2604172433F7A8A7A526CA4A2A1271ECFC5D5D7BE5AC0D85D921095350DFC65997D443C21C8094E0A3FEFD2961BCB94AED03291AE310CCDA75D8ACE4BC7D89E7D3E5D1650BDA5D668B8B50BFC8E608E184F4D3A9A2BADC4FF5F07E0C0BC8A9F2E0B2A26FD6D8C550008FAAAB75FD71AF2A424BEC9A7CD9D83FAD4C8E9319115656A8717D3B523A68FF8004258B9990ED362308461804BA3E3A7E92D8F2FFAE5C2FBA55BA5A3C27C0A2F71BD711D2FE1799C2ADB31B200035481E9EE5C4ADF2AB9C0FA50B23975CF"),
	     SHEX("6D96D224006DFBE8EEC28892FD2BF888FF543501F2F8AE5B6BE"
		  "B67713173C88671E4B3804D695730FDD0CB04FFD873B0E40110"
		  "3DFAE8F014FFB0EA382333E33985D1037403029F0C642323183"
		  "AC8668907DE791BDB0BF7112FD2C8BA5D3B933273B51082810C"
		  "7515C78173DAB93C7AFC48B0A76D5488B76CA0E5DA2226B669A"
		  "2EFA56F0A2041CC4A60212B2B4F4B02749B211856839868CCC0"
		  "A935DB1F7305EF356024FC78795690B69CBEA01EC0B5AA8B3C4"
		  "AC2D1815176B57062DC4742DEFA68081F21384FC875339F4098"
		  "5E85DEBF3AE9EF80CFCA746375492614FE5E76A96E506403FE2"
		  "CF5BCB43AE6E4E697E86237E825BDB16087EB32D3DBCCF2AF2B"
		  "624ACB4A960E0781E7C686B0B3A8A0E4BEB1F904E0D84B00038"
		  "788A02CE66BAA59EF185E2678598013C60208E5CEB50B701BDD"
		  "2694A24837F786931027AC0D64A229400B5783BF29AB39DF9AE"
		  "CED0CC8B9D6FEA52DE3E9220575B4F3AF6E2DF155A20D87FF3F"
		  "5ECDC0437CF7A8A648A0BE9C81F0EC3457BBA5D5948149C3FF2"
		  "411053549AC24DFFC65B54BB89B227499E35226FD9E1A85EBF5"
		  "DD534852921BEB63BF850EDB495F9DC1876C6A481E9B774E6D4"
		  "368974BD5A7BA7A16E90D078BD651F128CA7AC160EEC05CB8F3"
		  "AE3CEE4BF998579B84418D7A70DA95E65E4DA9933EF8E399359"
		  "D6D1993E5D9753CFD1B10737F6930FF0CBA3510D3BD9CF78F5F"
		  "79DD"));
  test_hash (&nettle_shake256, /* 219 octets */
	     SHEX("C8E976AB4638909387CE3B8D4E510C3230E5690E02C45093B1D297910ABC481E56EEA0F296F98379DFC9080AF69E73B2399D1C143BEE80AE1328162CE1BA7F6A8374679B20AACD380EB4E61382C99998704D62701AFA914F9A2705CDB065885F50D086C3EB5753700C387118BB142F3E6DA1E988DFB31AC75D7368931E45D1391A274B22F83CEB072F9BCABC0B216685BFD789F5023971024B1878A205442522F9EA7D8797A4102A3DF41703768251FD5E017C85D1200A464118AA35654E7CA39F3C375B8EF8CBE7534DBC64BC20BEFB417CF60EC92F63D9EE7397"),
	     SHEX("E0335CBFA877DA0DDD79072BD7F46EA6313144E633BA2C207CB"
		  "D6948237CC32A39DB06F951A1A4B9B0B91011158D38938ADC1E"
		  "42408A2D8A95C3EAC579C922943155624F6318E12BE7EB6A836"
		  "79508E0910FB6E85A84998F0FBE06B087EFC6EC91EEE277B48D"
		  "289DD6E12C820D0909E206750C82995DB86031E71B0D005CE89"
		  "8F1CF7C782F10B8CED5370CF27B74C6822910B653BBDD328AE5"
		  "A4921478A4072B8C7C89B446E057F6DC4692186CAC39CF08E3C"
		  "3F0C9457FF44DA73B88CE131A1E2B4C6A235DD464E4777B069D"
		  "39A9EE2A5877E07E8669B0D610513102B00F8E2A7EC210C403F"
		  "2E398EFF22E193A66376D0D08AB3013AEA4D508EE88EDD361FD"
		  "A5AA4E17BA899632CB7222EDDD5B320AF686C80F2777D94FA2A"
		  "3428A47AE7340054D2DEC1C42569A5EEE01759846FD10AF9D2A"
		  "D2AE9F6FADFF825D184EAF7E8AB9D7C7C64AE07CD5A95F2B24C"
		  "C3857BB239F5C0824EA7CA0C84CCCD9601580EA7A4C89370086"
		  "17468B9122B466624D51AF29AE1B66D14C332F2535BD7C3603D"
		  "E07A0B25C6A260E9E70EBD2DA622ACBCA660D425DE2E44B7E62"
		  "42A83077BD24025ABD5F0F2F21743F80D9A6F1552C3957B2200"
		  "88D861F3E25823548BDCB448D19B3CFA14FC8F11CEAC5BB2623"
		  "64F100655D5099D564041F113503AEC817BB03CCB9A6E021967"
		  "06FBB836C28421AFE0AB276772150FD0677ACE7655EA6214F61"
		  "D2A6"));
  test_hash (&nettle_shake256, /* 220 octets */
	     SHEX("7145FA124B7429A1FC2231237A949BA7201BCC1822D3272DE005B682398196C25F7E5CC2F289FBF44415F699CB7FE6757791B1443410234AE061EDF623359E2B4E32C19BF88450432DD01CAA5EB16A1DC378F391CA5E3C4E5F356728BDDD4975DB7C890DA8BBC84CC73FF244394D0D48954978765E4A00B593F70F2CA082673A261ED88DBCEF1127728D8CD89BC2C597E9102CED6010F65FA75A14EBE467FA57CE3BD4948B6867D74A9DF5C0EC6F530CBF2EE61CE6F06BC8F2864DFF5583776B31DF8C7FFCB61428A56BF7BD37188B4A5123BBF338393AF46EDA85E6"),
	     SHEX("74B8E6A2245E18432DB073B046FA0D6F63C85A05B9FB3B3098C"
		  "8F2AAAD938CEC8A8232AC2CC4CD7EA85A29BBB4A2D03311CFA7"
		  "F1ED8107DA574E659358408C60B1CDAE6A4BA6F3091574DFF99"
		  "8C97CA31E410B4132013AFAE543076BACA1989E7B5DA17E654E"
		  "664C856A00D7DE43EB3A2C5B8E347EA3B18A94D534C0448671F"
		  "E2C360034550F145F911ACB7C245A99FE37CA167EB8AE0FF5E9"
		  "D0B65F255D24DE4239BA87C2E8B757192020B3A28D53861D29C"
		  "845D4E4A21604ED2D5DC8E4C107CD47501847B50CE2452A3168"
		  "5012C5949E0CB630088178AAA5C9B0EAA6394E00096CDAFDC78"
		  "1833F37C64EE2955DB09F1C3B3767A0F0928863567513D5B02A"
		  "3D24E66A79EA590D5DBEF5E8B5D9953ED9F72610C0D4F5EAD67"
		  "B3421E90710138E09BB15959DFE9AE408AFA9ACFFCA19DD083B"
		  "DF508D8FBA9F75A18F6229A9850DE36653A0A39987781EAE3F7"
		  "40E2CE387C004D47811E3F8B5F8DA9ECAFE90C5AF4ACE449FF7"
		  "AF57B271B0C3059AE42180A1746EB3AABA944DBA5BF3946907A"
		  "18B1D94DF4819872A24DEEDDA3AF7A3787AB6B3D5F128D0C3CC"
		  "AB6AD0C34371BB76DAA321E36F6FF935FD58B590858F894B54B"
		  "F9997DF8B73939919AF342D6593F31F76C3251AB3812E27F00F"
		  "3AAE0C0535A8CBF9C6A9D20E26E619D6899ECA8A46526D7205A"
		  "3F2F3971C60E432CE438F40C2CBEF0F2AD7253B18AD53C3BD8F"
		  "108B"));
  test_hash (&nettle_shake256, /* 221 octets */
	     SHEX("7FDFADCC9D29BAD23AE038C6C65CDA1AEF757221B8872ED3D75FF8DF7DA0627D266E224E812C39F7983E4558BFD0A1F2BEF3FEB56BA09120EF762917B9C093867948547AEE98600D10D87B20106878A8D22C64378BF634F7F75900C03986B077B0BF8B740A82447B61B99FEE5376C5EB6680EC9E3088F0BDD0C56883413D60C1357D3C811950E5890E7600103C916341B80C743C6A852B7B4FB60C3BA21F3BC15B8382437A68454779CF3CD7F9F90CCC8EF28D0B706535B1E4108EB5627BB45D719CB046839AEE311CA1ABDC8319E050D67972CB35A6B1601B25DBF487"),
	     SHEX("10B27D45E1E91C3568B990F306D4D0338380C4A24C543DB7905"
		  "AD8C20B9C469A79E50F7FE705C03D2D1CFF9E2C35FCE1E6E203"
		  "5D3F464B6D2861EED223C32E960BB223F21193CD4734F36422C"
		  "1DE6F0F81791D6A5E9DB7BA8C2CA1F2B7FEF2F94C5392869591"
		  "3471B5E1B02B75FB9B0DA1FEB46520F3E8558E03427BF371180"
		  "EBACC91F3989F849B4EAB233C529006EE02050D85B548344C09"
		  "CC7A6C6C40FCE5C55348E1BEB801482BA70D6CC5A24B3B0C9A3"
		  "258B0934C3A41FBF07EF50460E2BDA507711954C83FFB91CF2B"
		  "2827D5FCC6DAE70DF5882689C573F159A9B45694458D0FB8A50"
		  "00B109BEA868160FCA781F50DB7B5EE56AE55BAE1FBEABDC155"
		  "EC46C0EB16314851FE0446AC2BF171D99FBCFBE2ADFBAE4A638"
		  "7B42FE41C5EE8E2AD1819D7F80BB1B4EC2DAFCD1D06C2C9750A"
		  "600EE6627F62089B2EE59EA461E0DCF47188D1A8DC80B029447"
		  "9C7EB03953C425D6EB3CAC6FFC34E484A35127B605CABFA5971"
		  "2141C4FA0840FB520A1D0A8DB4B298D24DE2A7246F28E036B51"
		  "1DB91794E35B75DAEBCB3B8833794B764F936200E7784348AE5"
		  "0B95363C027F3AF85762F3B8E6F31B7C2D8BD18814A723C0DE5"
		  "17F7B3937AF574A3FB5F52DE8A10A2055E8EB4281D383797DF4"
		  "B8D5DD5266C5D438A10DE4BAD00E4C0B252110CF368707AA2FE"
		  "785F518CC2F3DA5221634B52118348F231AAE0712DE096D70A4"
		  "3929"));
  test_hash (&nettle_shake256, /* 222 octets */
	     SHEX("988638219FD3095421F826F56E4F09E356296B628C3CE6930C9F2E758FD1A80C8273F2F61E4DAAE65C4F110D3E7CA0965AC7D24E34C0DC4BA2D6FF0BF5BBE93B3585F354D7543CB542A1AA54674D375077F2D360A8F4D42F3DB131C3B7AB7306267BA107659864A90C8C909460A73621D1F5D9D3FD95BEB19B23DB1CB6C0D0FBA91D36891529B8BD8263CAA1BAB56A4AFFAED44962DF096D8D5B1EB845EF31188B3E10F1AF811A13F156BEB7A288AAE593EBD1471B624AA1A7C6ADF01E2200B3D72D88A3AED3100C88231E41EFC376906F0B580DC895F080FDA5741DB1CB"),
	     SHEX("69304677315720D586605521E24D32745DFAB082D2BC23A3165"
		  "C419831C19468A088D972AC672D7BAAB907005A0AE3910D4F06"
		  "F0BC5ADF6AAF9E1D8CD7CB85DD166D0FB9703CDC7D45DAFAC48"
		  "05612137D4720FB3FCE90D69864403E6C456FBD87677312ED7C"
		  "AD9A28F7A98753265294F4820FA50868CFEA157EBA50DC5763E"
		  "223238FB2032DBC6DC00C3710B6976F16034A9D851CD1BADDEC"
		  "6071CADF10C8F9FA123FC0EB81ECAF4C16B8D52A6C1FFE330D1"
		  "14D71CEADE28BB0422855AA1E7E838A93523315BD0F8384361A"
		  "58529A2264D66D65806659C49B7F80D95474A7965A0733D3DA4"
		  "FE317C5AE6180746BD8DC70589F97B15E0FEF29D34D6BE4416A"
		  "CB3F9EBA2F3E34F8E636A6B111DFD4ACEB2C9DE051B942EF021"
		  "C78E0671F771058C4B59D34991DD2EFC6943C6E0156CC0E3FEA"
		  "D10968F77BF4DA82B9FD3AB73B43E2890BC8AEB02D7178FAEB2"
		  "FE1A0A194B1581F6322D817AFF2DE7DAA96F4A725EBBD8C9FC9"
		  "C815D0A8E9FAC425F278F121084AD98AF9BB119235FFFDF1751"
		  "AB6A805FA80F43A5682C2FD5BF9241FCA866DE6B5EFD36E6923"
		  "13F51FA8DD64B52DC2FC33C56687417C942680676143A459068"
		  "401690DF9213E18604058E74011B111F1410531FBED32567AF7"
		  "A0319B5A4A717F580D16D9E9341E78F717E27258A84A2CDBBB9"
		  "CA118FD0F81568639F40176A1754C743B4B5E377DBB0175D160"
		  "B414"));
  test_hash (&nettle_shake256, /* 223 octets */
	     SHEX("5AAB62756D307A669D146ABA988D9074C5A159B3DE85151A819B117CA1FF6597F6156E80FDD28C9C3176835164D37DA7DA11D94E09ADD770B68A6E081CD22CA0C004BFE7CD283BF43A588DA91F509B27A6584C474A4A2F3EE0F1F56447379240A5AB1FB77FDCA49B305F07BA86B62756FB9EFB4FC225C86845F026EA542076B91A0BC2CDD136E122C659BE259D98E5841DF4C2F60330D4D8CDEE7BF1A0A244524EECC68FF2AEF5BF0069C9E87A11C6E519DE1A4062A10C83837388F7EF58598A3846F49D499682B683C4A062B421594FAFBC1383C943BA83BDEF515EFCF10D"),
	     SHEX("B32E9585694532B1018B05B9DFDB23B8924025786F5733769B4"
		  "913B73B0FC35513D4C0FA1A1A5019DB5FC807ABADCD4589B4C5"
		  "658A854FC919105A5F08799F7CBA3B8DFE72DE84027B341CB19"
		  "46D6A4A907D6FFBC15B6A00805ABE25047476B99D53A6AABE9C"
		  "758B3BF4476DFEFEF2E366617A1407853B0D0278DF42FB7003B"
		  "B1610DA2AD52422A0ECCC74818EBB4DF264D0ADB9623B9EE0C5"
		  "371E9F0E73D7FD31A2B41F9DA8C9955310166FCAE6BE9AA75D3"
		  "F7C532F1502BEACFCD81A034B5D4A87647A5B9E2C49B7BF055A"
		  "0F4BC3F555FEF5147BB7B9BD75772878FDDCFFD58FF2B09A61D"
		  "6067C0F3498EFBB5FCB736813F798E213102E47A3DD61A6BF99"
		  "DB2834DB63BEFDA6E290EA953A8B96FA37650DF510082294891"
		  "C86892215850A34AF1EE0659694431DA636C9BDC854AB98FC79"
		  "6C28BD6272C9C5DF1D199CE9F55BB575786EBCD6851DAC97924"
		  "AAFB102491BF4B80351AFD26EF027C745F98607C19ACF96B66E"
		  "CFACD4C512BCC9A8E609DE5FFCDE0DF0E77760D8D44F1230133"
		  "6247F53CBED8894831765D12221C902127B65DD549E6C32124E"
		  "0CA555586FCA8043097B9AE3D9BCF899FA06EE5D0A17EC0A2E6"
		  "57E37CF5D4A55CF4C99E9F10691602412D27F9B8DB5C4C2644F"
		  "C822E11833B32168AC89F2D591748D12B527FD5EAB2291F2015"
		  "79A48AA9C370979DC2265B873AD65266E98FF668579B0F4274C"
		  "73F9"));
  test_hash (&nettle_shake256, /* 224 octets */
	     SHEX("47B8216AA0FBB5D67966F2E82C17C07AA2D6327E96FCD83E3DE7333689F3EE79994A1BF45082C4D725ED8D41205CB5BCDF5C341F77FACB1DA46A5B9B2CBC49EADF786BCD881F371A95FA17DF73F606519AEA0FF79D5A11427B98EE7F13A5C00637E2854134691059839121FEA9ABE2CD1BCBBBF27C74CAF3678E05BFB1C949897EA01F56FFA4DAFBE8644611685C617A3206C7A7036E4AC816799F693DAFE7F19F303CE4EBA09D21E03610201BFC665B72400A547A1E00FA9B7AD8D84F84B34AEF118515E74DEF11B9188BD1E1F97D9A12C30132EC2806339BDADACDA2FD8B78"),
	     SHEX("E6BBDDC5B154ABAB888F4E00C2BFC18A200DB6DED8EBAD70ECE"
		  "03D10D123C52D4B5124CDFDE2D5A803B8BE98C646E29F64F2BD"
		  "D55492531B2741C6040FC51F7DF18106D9AAE62F1930BAB26FD"
		  "5167D516C4E4AFD3035D6AFDDD06DAA59B56BE3F188CBC11529"
		  "909F6CA0618C321349B1B918C1CEE491C8D7996F10B02E0A085"
		  "CE790648C586710D4916A528398FAC1E1FA71971EC85B203B83"
		  "721AA006D7026C2647CEC53296E80EE3FBB6B880BD296D6B9DF"
		  "FDF40C5537551EB0DDE86E85A72FD280888C08D67A9839EA618"
		  "49B47FB010540BAA15530495CC8BF941F82B4B3114F6AE055C8"
		  "52B0978ECAA67108AECEB12E78C339071CA058FA4A534558899"
		  "D71ED36C2164DE0C92612F07CF465F92C0A1987F7462AD303DE"
		  "358DC9C9929AC205F366FE564C0B32B1E4D65EF4E0B45B2AAEA"
		  "6A93C5626ED40C597B54A2074F255FC6C33F7DE535B5041F320"
		  "85274EBDE9D7ACE9B458E7E7CC47193428A8873560C561230B0"
		  "3A72D993100B73FDC24231A39C227B05630B328645349E0025E"
		  "F02002C34E6A7EF07AC5F7E299CFE0ECCCF01E1DFBDAC9BBC95"
		  "B09B2EA505D8E74C85B9BC8E939363F3F062260711C44C132E1"
		  "F3663123EC0B0A253DE7DB1DC2A08481571F64BAC3FCD9EAC73"
		  "7BDB42AA4D392DDF9547126191DDCB2F88A5705BEC9BE7C8D88"
		  "E9521EF7BF9753155F6F3A75ACD4C9C913785CBFF81761DA8DD"
		  "73B8"));
  test_hash (&nettle_shake256, /* 225 octets */
	     SHEX("8CFF1F67FE53C098896D9136389BD8881816CCAB34862BB67A656E3D98896F3CE6FFD4DA73975809FCDF9666760D6E561C55238B205D8049C1CEDEEF374D1735DAA533147BFA960B2CCE4A4F254176BB4D1BD1E89654432B8DBE1A135C42115B394B024856A2A83DC85D6782BE4B444239567CCEC4B184D4548EAE3FF6A192F343292BA2E32A0F267F31CC26719EB85245D415FB897AC2DA433EE91A99424C9D7F1766A44171D1651001C38FC79294ACCC68CEB5665D36218454D3BA169AE058A831338C17743603F81EE173BFC0927464F9BD728DEE94C6AEAB7AAE6EE3A627E8"),
	     SHEX("F40B4A6A7D88482C5BA65D17038D68184E1AAD676D340300301"
		  "7A4348A7E3ABFE89804EF65D96EA5A467CAA938702071918452"
		  "8944AAD358960B66A0E78F2E4C19F244C82CB0D10BD1DF49CFB"
		  "89C24FBCE8AF1DC6B13DC61F40116ED3D1B51B80ACF08A7A9BA"
		  "DC6318DA75EB4CABBF4798923D639C2C3C24778660089384DE3"
		  "3F0EE7F957C78D4BF8D74EE1E3EA5E330C350584C319298166C"
		  "DAFFE0C7ACE6A50D1551B30FA386A0ED5C20DA31DC35FE2DDB7"
		  "FF030AEF1C0F2ECB0C1915554722DDD5A38DF758E321048FC46"
		  "43FDFAF54AE558F9B40EB38DBE516256E8D97701022B68E1F09"
		  "83A14D7DBEFB59DAC5073062DE8698BEB9C94DAE0A49FD51A66"
		  "1FA6AA84FE1088EB4E0F964953625A898B03993ECB248EFD5E6"
		  "6380667C2ED185FC568AE58E15B5A57ABFD84966094EE5C8528"
		  "4F3248C92E1300DB80CD6AE08662412D01503D598A1FCDB0EF8"
		  "CA773EADACE543B867BADE24EFB25518E59E27FC921D8DB6D3A"
		  "C286BD1ABA53D48983DF63D493F95B88610CBCF6DE2130AD63F"
		  "B9860A839C3F98F5532B90225D1ADAD79B478F5C9D836435A9C"
		  "9C0EBE229FA494B5F4253DA92C7382AB388C4F052911E9FF28C"
		  "C43415C8A82065433D8CDAB9152CC13191F1729EA95693B4BBE"
		  "BC3B03F298EF6B3739A0F34A545198FF5D2B4292AA8D6A04271"
		  "463DF95145C99EC769167761690C97361984A665B0D5C2169AD"
		  "DED2"));
  test_hash (&nettle_shake256, /* 226 octets */
	     SHEX("EACD07971CFF9B9939903F8C1D8CBB5D4DB1B548A85D04E037514A583604E787F32992BF2111B97AC5E8A938233552731321522AB5E8583561260B7D13EBEEF785B23A41FD8576A6DA764A8ED6D822D4957A545D5244756C18AA80E1AAD4D1F9C20D259DEE1711E2CC8FD013169FB7CC4CE38B362F8E0936AE9198B7E838DCEA4F7A5B9429BB3F6BBCF2DC92565E3676C1C5E6EB3DD2A0F86AA23EDD3D0891F197447692794B3DFA269611AD97F72B795602B4FDB198F3FD3EB41B415064256E345E8D8C51C555DC8A21904A9B0F1AD0EFFAB7786AAC2DA3B196507E9F33CA356427"),
	     SHEX("92951EAA72195331ACDD63572FE112E0DF4B43CA1B67BA5C918"
		  "4DA5667A8439E3AFD8784461010EB80178B07F278C0BD52582C"
		  "EF5B4D869D77E764342651ACE52F1F5DBB5316A36CBE2D6FA45"
		  "5403A8FF0480A5DF002396B8E444B9005A9A97D301529F5FAEF"
		  "77C0FC04C3B01C7C4907F197927904BC56DA2F850A0767E91C5"
		  "B77D8A41CAFCBDD5CC768E220E844786093CE425BCB3BF6CF3F"
		  "E9CCCBF6A5CB84D46AED8BEB079AC0F12F68836A71D8C9D738F"
		  "9BE5A8311299EEC2AE86AB774D0D8345C6FA7A341223A43DBB8"
		  "5AD5274806B23A3BBE312723BED548F4C64353EE2875A44A8A3"
		  "7153DAFBE98E96EFAA402E7A68BCA09EDA8A744C3BB8FCDAEDD"
		  "7EC54CA4066B69B4828AFE6285B6909BC2AE96CD62F3A523898"
		  "7279F206C2593FD2BF4D5A643484A5155FFF290253650387C7B"
		  "DC0B3FD749EB1A84F7A11790AFCDD3FC72FAE3BBB1C55596584"
		  "B7246E77D0712562AEAFF1C5316A698DB9AA3524DC4AF4AA461"
		  "9C5F9CBBE41AA1FF2A0BAD12F17ECF7C90E6FBC20EBF374D633"
		  "5EF245BF449E3EE5870651587727D35BBED64885A8D375E43CC"
		  "350E398C3D5DE83CE92354C08E16A6AC69AF10EFAEC2F0268CB"
		  "7F998CEB45DB988B193C8D35E89A387DE6C212A2D2D5A9FFEC3"
		  "D4DFF92D66992563C85E4E707730BDB669783357E5ED038ACEA"
		  "7D8214313FC027733D51B2D21F47666DAB6C6F3E50A335E83C7"
		  "2ABD"));
  test_hash (&nettle_shake256, /* 227 octets */
	     SHEX("23AC4E9A42C6EF45C3336CE6DFC2FF7DE8884CD23DC912FEF0F7756C09D335C189F3AD3A23697ABDA851A81881A0C8CCAFC980AB2C702564C2BE15FE4C4B9F10DFB2248D0D0CB2E2887FD4598A1D4ACDA897944A2FFC580FF92719C95CF2AA42DC584674CB5A9BC5765B9D6DDF5789791D15F8DD925AA12BFFAFBCE60827B490BB7DF3DDA6F2A143C8BF96ABC903D83D59A791E2D62814A89B8080A28060568CF24A80AE61179FE84E0FFAD00388178CB6A617D37EFD54CC01970A4A41D1A8D3DDCE46EDBBA4AB7C90AD565398D376F431189CE8C1C33E132FEAE6A8CD17A61C630012"),
	     SHEX("9107BC54F9AE29BD282C37CF4BE15D2EDC4B5A20FCD8C131620"
		  "CEE6C9326AF25896BD25550CC3B46C19BB58D825A7156A4AAC0"
		  "AE72E38601AA6AAEDC4DDC578D145AE8A0AF4E630914B49FCD3"
		  "9C41924A19ED6E52EC150D1BDFBFE2717B9B46D6BE9BD637876"
		  "1F167C1133EE31981F9B681D674E267B3A01B0F455F9A0C86B7"
		  "E6BB3EDEDC2DD86C8EF6F6BF736BA9CFC4A20E602779F0CC626"
		  "E0F3513807C8BC71546DD1E0E5210A6ABDBDB284F90EAEBFAE9"
		  "70CEE9EB8C58A5F0F197FEB3292AA8EE814DF5D86D6B1F43AD7"
		  "87E82AB6D1C01CF44F3B32522609CC51967FA9D2417A8B91461"
		  "B41A5390BDAEE3763E152A2DB2E8B7842A73F1E05A05F7A7D19"
		  "FC88F576B049F87E85835EA4DBED141AC1321B47AEDBFD285B6"
		  "488B7613206E483559A8B0725D3091A04F10736EA1920FE2346"
		  "58AEC2A5FD0F85F50BED9EEA3E5513EE3B45F0CAEE09A1CA3DD"
		  "8E96EBB408EC6A31F07ACEC024097C20038A6DB8B5A10396C6F"
		  "3532D8B356BD664A9C3C153D2248F9D8587D21AD3ADB20D06C0"
		  "20F5AEA40E96C776202A17740F59BAC12195B49E19ABB87F693"
		  "BA0B1B414C4C83DD7474FB212F0F8015D0019D33250BB890079"
		  "278E027F0191B1A92FC40004050C4E88D1CD3469B656A2F6D1A"
		  "B9E49118B18701FA85A5B1497F1FD0D3C999567F99D05CD8F20"
		  "DBC05333F547598615FEE2452861AF0CDE27ACC4D763074DE84"
		  "0004"));
  test_hash (&nettle_shake256, /* 228 octets */
	     SHEX("0172DF732282C9D488669C358E3492260CBE91C95CFBC1E3FEA6C4B0EC129B45F242ACE09F152FC6234E1BEE8AAB8CD56E8B486E1DCBA9C05407C2F95DA8D8F1C0AF78EE2ED82A3A79EC0CB0709396EE62AADB84F8A4EE8A7CCCA3C1EE84E302A09EA802204AFECF04097E67D0F8E8A9D2651126C0A598A37081E42D168B0AE8A71951C524259E4E2054E535B779679BDADE566FE55700858618E626B4A0FAF895BCCE9011504A49E05FD56127EAE3D1F8917AFB548ECADABDA1020111FEC9314C413498A360B08640549A22CB23C731ACE743252A8227A0D2689D4C6001606678DFB921"),
	     SHEX("02D21B8FEE4BB59B4D39D88DF91B675762640994D3A300BD725"
		  "9A3E97DED831112D27339FC4F59C7F17DABEED310EE901D829A"
		  "2193421C4D828DF1248A68F6AB928EE87AFD727DDC2766DE071"
		  "103B144040F9EF22592599A0FA6CEDC2FCD9A774BEF862D2D51"
		  "C8E2538D1DECE789A6756A1F9034C0949C3FF13190A141F3DD4"
		  "F015A89DE643D65E5EC4AB25300739892A81911E3C48D8A5823"
		  "9203C7E828E8867230E46BB8BEFD018E9ACE92061E2BAA09B0F"
		  "B8C0CFA7FD5024CC65826AA96BDDA6E21ECB7D0D77FE651CC25"
		  "EC260C1C41F4D0DFA82963A2EEEF0AC0C6123D9726A02396A06"
		  "12E8BA6CD4988C132E7CAB1B5097174B6BCA798C7B04E5F7175"
		  "A3A5BFAA9D23DFCF5E531557BE9BA6F4B5D806287BCDF1CB503"
		  "4FF94F709544E81D856CD8A1380774C5361710460D52CDFEAE9"
		  "AF4AF3E9DAF7E6B679A6B881A378A1254DE7105396DB2449737"
		  "3430C36D24BD49425E7613FAF651747217EFC4E24A761CB4694"
		  "A2AB3E0B43A8A4CDA5EBC9D7299B27EB600122C7EF54703E48F"
		  "51DB2A5A8F3EB59A2C174DF966AD728DF0BC2E94418775E0186"
		  "D3A527F86547980C8998B32F0365D9E430D33A92CF081A15258"
		  "92FCCF3D8FB4DB1E0A728A856F2E2CC8EC4A08FEFF53DB85C66"
		  "20CC8B57106097A96B668F97C3BDE8CFDC9600419B46EA24839"
		  "D27A574BF63147A97B2B48CEE1C6FDA32F3B945E36FC03EAB48"
		  "5B4B"));
  test_hash (&nettle_shake256, /* 229 octets */
	     SHEX("3875B9240CF3E0A8B59C658540F26A701CF188496E2C2174788B126FD29402D6A75453BA0635284D08835F40051A2A9683DC92AFB9383719191231170379BA6F4ADC816FECBB0F9C446B785BF520796841E58878B73C58D3EBB097CE4761FDEABE15DE2F319DFBAF1742CDEB389559C788131A6793E193856661376C81CE9568DA19AA6925B47FFD77A43C7A0E758C37D69254909FF0FBD415EF8EB937BCD49F91468B49974C07DC819ABD67395DB0E05874FF83DDDAB895344ABD0E7111B2DF9E58D76D85AD98106B36295826BE04D435615595605E4B4BB824B33C4AFEB5E7BB0D19F909"),
	     SHEX("76919B60D28A57717EC190FC9990E32020E6436828D90D8A2F4"
		  "A7AD21F18A4F5F1CA753ADCB0AA010D40334D1E58B8BD165B01"
		  "51E631925055F4800DC01D466D11E88AE945545365B7F6072B3"
		  "6E11CB6F15DA6BA3B53E12473BEED09C5AA275CAD795E03838A"
		  "EBF74A6B1A3371344C1E3104CDE3C768026963DE2ACC8935F51"
		  "62D1D09C966B5FE2943CC723E8007559C0537126400F1A4C3D9"
		  "004DF88126A23F9051630DE9FCAC524BF4A109CCCF5CD7F7972"
		  "B64FCB5A8E5A78A85EE82FE553AB90754CD12EC77E2E3CC09AA"
		  "1F172963DA44D29FC7F95F0C7C1207BDC858FE8B9785EAB55B6"
		  "EE33612863FCFD1668A2FB13715F7910968708F22A9A88D97F8"
		  "4EF4212551D8FCB05992670AAFC9EF5CC34F0843246241C9EE0"
		  "11470131E0E6B56C9EF6A0E38423943BA502EC9D08C39F70797"
		  "BD85F6D9207684022CADC282EED833B43CE450445075013F1B3"
		  "270492AA6E412CCE8E9D9534C0A380B45CAD9328610EF3FACEA"
		  "7FF5199F1E4D0342163B16961B0F98E46E8615AE2CD74D92E2D"
		  "F30A3DF00BBBA2AE43A67F51EC52CDB37E00D979ACE6AF3AFA3"
		  "4A58B8CFC07DDC3C98FBE4A5E18539D03ACFA18152F4CD32CBD"
		  "84AC30A99C325EBD57EF2870F36C711900CC23539BE7C19DC94"
		  "F75457BDFF28B9B5DF876B5576CC37FB30B291DBF44904B36C3"
		  "D96747DCD92A1550CF9323F358D15471FDFD35C2F4C4B104CC9"
		  "1FBA"));
  test_hash (&nettle_shake256, /* 230 octets */
	     SHEX("747CC1A59FEFBA94A9C75BA866C30DC5C1CB0C0F8E9361D98484956DD5D1A40F6184AFBE3DAC9F76028D1CAECCFBF69199C6CE2B4C092A3F4D2A56FE5A33A00757F4D7DEE5DFB0524311A97AE0668A47971B95766E2F6DD48C3F57841F91F04A00AD5EA70F2D479A2620DC5CD78EAAB3A3B011719B7E78D19DDF70D9423798AF77517EBC55392FCD01FC600D8D466B9E7A7A85BF33F9CC5419E9BD874DDFD60981150DDAF8D7FEBAA4374F0872A5628D318000311E2F5655365AD4D407C20E5C04DF17A222E7DEEC79C5AB1116D8572F91CD06E1CCC7CED53736FC867FD49ECEBE6BF8082E8A"),
	     SHEX("72A343DF2C1854005653E1EC9C8F9492AF9EF7FACF46DF729AD"
		  "910FA3A8B830971D22D5E017ECD515C0E972197C87968FD05D5"
		  "9C570634A3864DFEA159229746A967A2847F0F4E3C48DBDB21E"
		  "BFA400E18BF289A2B146758EAEE64AA200287F7F55E57817ECE"
		  "0470C35531D35BE824AD7E825E7A1FCA07CE5325DE6E27D6C6B"
		  "F1602EC1069117E230ED9A0873D6538881992FDBCE93A4E1201"
		  "3A56F4D22F5846B4E04BD15E9AE9590C703EE7AE346B860D1B4"
		  "B181A963C1DCD36AD485AA484CE534138789C2874AB5888B70E"
		  "278E30534ACDF0D9798A44E1701D9447B5D6FD212344603A269"
		  "A9D2E0437304F5153FBEB7BB5A1E437B63014ED6F01D4F6135E"
		  "F189EE290E30C25DC2F63B7D8CF19E9D7959EB2F61703842E0C"
		  "89EA487441B9C17FA6800A63C411F1D3FF4C35F1E89BB820D60"
		  "B25CB1AD71E6583390E753934759A5467680405B1C9B6E391D6"
		  "DDC4510FF24496559FE2604C038A5DF6638FD7F7880E0847B88"
		  "0C1628F47BEF49689D0C31C4867BAF6FCA08FEA898056DCB171"
		  "6C95341ACA367BCD9BB372A93D2FCE947B5CAB707E744F15FBB"
		  "85230FE7F86A9EF7C36069CA79D781D1CC76193D423229E51CE"
		  "0612A2D61680D35CFD5AA482739442F94D205D72080BEC7A259"
		  "B23B8D9B4F76F2A75235CC3BBD6738FD870905FA21BFE1315B0"
		  "C678D7DB5F4DF36F5D43D04C11F3BCE3FB71B3B7AE410AD51CD"
		  "E033"));
  test_hash (&nettle_shake256, /* 231 octets */
	     SHEX("57AF971FCCAEC97435DC2EC9EF0429BCEDC6B647729EA168858A6E49AC1071E706F4A5A645CA14E8C7746D65511620682C906C8B86EC901F3DDED4167B3F00B06CBFAC6AEE3728051B3E5FF10B4F9ED8BD0B8DA94303C833755B3CA3AEDDF0B54BC8D6632138B5D25BAB03D17B3458A9D782108006F5BB7DE75B5C0BA854B423D8BB801E701E99DC4FEAAD59BC1C7112453B04D33EA3635639FB802C73C2B71D58A56BBD671B18FE34ED2E3DCA38827D63FDB1D4FB3285405004B2B3E26081A8FF08CD6D2B08F8E7B7E90A2AB1ED7A41B1D0128522C2F8BFF56A7FE67969422CE839A9D4608F03"),
	     SHEX("AD4DD99AA71DBDEABF22FC7C82BB839F910A22E4CEBF8317899"
		  "D0AF87279EBB321C442DFB9083C4676C16506C4006F18E8BF84"
		  "EA9024B1DEEE4ABE6F90DBE0EFFCEBF4CB7B94D8A8B107C5BC3"
		  "198277BEF87C9760AE2BEA86AC50C39A666AB09F276F2FA4C1E"
		  "B5B8C37ADC3B4C9DE4BDC28074FF4B997518B44FE49B0D4E4F8"
		  "3CFE9E12EF7389625766A54BEA82DB3FB57276F47C418BDD5F2"
		  "0C8146C792DCB8E42C7ABF09331F599318265776CBB19C65BAC"
		  "6DC4A683956C052AE5CEA02F1E4EA690F9E5D3BF3A06A726F0E"
		  "FB6DAB5350753FFA7ADC477E9AE454C5E18F9D64EDEF0157D1F"
		  "01FA78BF281468F14D79C39405D2B5B604C0EE116731C52C243"
		  "2488C45A3E56D3F999855631381B5EF3C99A0352C16B92812A4"
		  "889C1B8A72C84273A90ABF29C7EAD82F2E7C6DE67E7369DC02F"
		  "B0F1D62BB8D245030A6D26E930476171547F33B9C9DBCD542F0"
		  "4C97B0A278D24E2BFEEA0209FA83D634AFE65EC9E7A5550774E"
		  "DEDFFA36250DC5A400AE5FD6696F20E5FC6AFB15F54C7240A24"
		  "CF11BF16730B8E676F948304D3EABF315E95B723FFBDE6D931E"
		  "CF3BE6C69D8CC82F31D1F499BBAD8ADA9654630E7185BF6C1D8"
		  "82DD700F7F1B1ABBF384F58994D55C9E831ACA847B8F654B070"
		  "F424267685103BB339A4CD43DF94894E912B64ACB15AAC65FD4"
		  "96424FDF02BE7A96E161CB6A0034997DDC918F6FB604D6DD55B"
		  "4D92"));
  test_hash (&nettle_shake256, /* 232 octets */
	     SHEX("04E16DEDC1227902BAAF332D3D08923601BDD64F573FAA1BB7201918CFE16B1E10151DAE875DA0C0D63C59C3DD050C4C6A874011B018421AFC4623AB0381831B2DA2A8BA42C96E4F70864AC44E106F94311051E74C77C1291BF5DB9539E69567BF6A11CF6932BBBAD33F8946BF5814C066D851633D1A513510039B349939BFD42B858C21827C8FF05F1D09B1B0765DC78A135B5CA4DFBA0801BCADDFA175623C8B647EACFB4444B85A44F73890607D06D507A4F8393658788669F6EF4DEB58D08C50CA0756D5E2F49D1A7AD73E0F0B3D3B5F090ACF622B1878C59133E4A848E05153592EA81C6FBF"),
	     SHEX("7623AEA16ED0AC8AF23A7911BE015D44F93F3329AD1C0023BB9"
		  "A6E486477D9CF5307B358527BF479FDAA473FE60CDCE59C33C4"
		  "D1DB930057438CACB4006D2E6C327B3FB033BE384FF07EB40A4"
		  "4AE4D39EA9794C29C9F3A91F679EF4690C41422B193929654D0"
		  "0D44D06FCE9B35E5FEEA5C20CC02FD104A74764F3BBFF273086"
		  "023AF4AFB3A29F9F9836C17C5A883BFA0FC4D8FCC9808FEBD82"
		  "DF88F7DFC3F715C8B07DB1F2B61403CB6F4F7B18F898FEF9741"
		  "E016CA0CB3A4B8F54F57DB2F2BB63289AA38A7A9FEBDE32089E"
		  "4B479D9CF184F7D1C418B648CFE7E6079F7DC1CC35F22E31E62"
		  "0704736B44FE1E3A677CAFF61FD956883E6F6CB278284A05BF2"
		  "B4CD9B832DE2058FAB356052C55B03D245BAAD53B8641444101"
		  "0767ABEA201E26F270649EA35B0479A66ADB9439C57A77E84B7"
		  "54A7E328E4AED0618742CD49AEF0AF116462F3A5B24343E141F"
		  "C51F6B9FF376A30989B17BF4CA9DC2ACEBD4F22D8EAB082E62F"
		  "FF99CFC55CA5E7C4B86613356F1CF05A9A3E8FD79123FB19627"
		  "A75810D3AF44213DFDA590245FB22B48E8EC5F58EE1D51BEEE4"
		  "D6BE0BDF7E8D18AEB37995D602E1EA1FC25D0D47D64D8A88B27"
		  "3FA8CC649C017A63F34C140A1D801F2558C8639FBEAACD1D961"
		  "893E111450B85ECA907CB3B66027587FE9B39D635D01C4C3695"
		  "89B9D8907D54A9B73FDE7FFCED9FD443B18892F29FDEFD7B3E7"
		  "17B5"));
  test_hash (&nettle_shake256, /* 233 octets */
	     SHEX("7C815C384EEE0F288ECE27CCED52A01603127B079C007378BC5D1E6C5E9E6D1C735723ACBBD5801AC49854B2B569D4472D33F40BBB8882956245C366DC3582D71696A97A4E19557E41E54DEE482A14229005F93AFD2C4A7D8614D10A97A9DFA07F7CD946FA45263063DDD29DB8F9E34DB60DAA32684F0072EA2A9426ECEBFA5239FB67F29C18CBAA2AF6ED4BF4283936823AC1790164FEC5457A9CBA7C767CA59392D94CAB7448F50EB34E9A93A80027471CE59736F099C886DEA1AB4CBA4D89F5FC7AE2F21CCD27F611ECA4626B2D08DC22382E92C1EFB2F6AFDC8FDC3D2172604F5035C46B8197D3"),
	     SHEX("A2CD7AE2955E2C2CFDCC374AD99E46F29FCE072E2C085E0F022"
		  "81A4A85A822A2424546410466299CCA501BA5A86C12085320D2"
		  "D0116196CD31E67E175BFAB909E279846C8FA2CDA475B0C9D60"
		  "4942D6332F3ED46E81D3DCA880527DEBABE55BCAE22C98EC428"
		  "22BD1874561E7F8E8CC8092DC461577F2AE548580FAAB59AFDF"
		  "EBF359847DEE5271C2E972C12D13424A50A9B3DC3DE5FA6F5B1"
		  "90F3DC060481118A595A8859ECDAC8C57FB16FD4DB9802C7BF3"
		  "44A68A0A91CA8629DD0A8FBB26FBA0E21C4D953626F66C0B33D"
		  "30C2569D73BBEF7ED708212B2F66DDB9AEAC459A11C9862CB56"
		  "74885619D240BABE380BBF23564F6D946D03AA544E348943A48"
		  "8C0FD223B36DB5B0EFBBE5C4E1253C3FF6C058AA95A80925523"
		  "749BD228D89BF99F5CC681237FDCC8C872F652DAE6306A20D94"
		  "22818A8DE2FA68C5A334EFE828A20A48CBC49EB6D7ED779B00D"
		  "F50751446D0C850FF1F096FC50A03A8C407B8362CE03690D2D5"
		  "699C70313B4F2D39E18ABE998EE73DE38308DCBACA27B8F6CBB"
		  "3305B35C98B620A811C94AAE44B5DFA2F8664FB678FB1D73DDD"
		  "F9FC27CF693DB9D5CF3B46C185B72C33E3BD0F6FFC01108C8A6"
		  "5772ACFC13DEFAF21596F2C9B9512F369588F6F8767FF61436A"
		  "C67ECA3DE25D6C5A9746AFB3B3464A49F3A583740C432ED806B"
		  "FD16EAB27D59A12975D7D09F7ECCD67C2527333FA95509F7209"
		  "DFCC"));
  test_hash (&nettle_shake256, /* 234 octets */
	     SHEX("E29D505158DBDD937D9E3D2145658EE6F5992A2FC790F4F608D9CDB44A091D5B94B88E81FAC4FDF5C49442F13B911C55886469629551189EAFF62488F1A479B7DB11A1560E198DDCCCCF50159093425FF7F1CB8D1D1246D0978764087D6BAC257026B090EFAE8CEC5F22B6F21C59ACE1AC7386F5B8837CA6A12B6FBF5534DD0560EF05CA78104D3B943DDB220FEAEC89AA5E692A00F822A2AB9A2FE60350D75E7BE16FF2526DC643872502D01F42F188ABED0A6E9A6F5FD0D1CE7D5755C9FFA66B0AF0B20BD806F08E06156690D81AC811778CA3DAC2C249B96002017FCE93E507E3B953ACF99964B847"),
	     SHEX("A74B8704CCA1F4234742A5C8AD6B138E58D56281C98D95F76BF"
		  "C0A0AE309D3DC227543F9AEE711C85021B89F259D002188F4E7"
		  "53DDDC4340DCA27B84445AFA007A1E502A1903A857583796BB1"
		  "24DDFD64B5C91B69CD9B230F27DFA5D22A09B07FCA1D860DCF3"
		  "16C22E57DA3589FB0F07ECF671DCB150913CE8606DA9CE39A58"
		  "57B8382853056A5930F25E3647015B77B2DF4954314F1F8B1C7"
		  "74DE0BCD1842CAD3BFCAF5DAE8AB9D77290DDD2112AB060F5B0"
		  "17630C318A1A3DF5C15E6F8AA348EE3F3C171C4E8B98CB4A295"
		  "6FA240EB289561E7CA30EEAA9CF79683DB7A2B0F28B3B1BF30B"
		  "D5A62C96ECC8BD74D1F2D131295597C668838A8D3B20CE1A4EE"
		  "4D1103DE547AC21F6442DF2CA10181319C76AE5827FD60EABA7"
		  "C788CB369591E429B5DBBCD95049BFEE2A5829FFA0417772DB1"
		  "BBFAC2EAF112B9712A02C1ECE830E17B22696AA6688E08BC705"
		  "EA7F7010684D7006BC0A017BBAAB1D6119B1B589B326F1936C8"
		  "2C470245298F2167D884398ED408814323D369BD1126031DFEB"
		  "56A160F90359054400DB3B3D267D42FFDA5BB710643721871DD"
		  "4DE52446004AF00BA55706C93BEAF01F8D77664AF27AAD65C6D"
		  "61FAE24E65FEB4310AFE6CA5322EE4DD2A55CA5FD7E651AC8D3"
		  "118C0FDDAE4B2E42862AACCD8CD2134340DDD760BC7918A461B"
		  "16ECABB601861B58EDEE6E78AC3FF75055997ABA8C0C57878AB"
		  "A5A7"));
  test_hash (&nettle_shake256, /* 235 octets */
	     SHEX("D85588696F576E65ECA0155F395F0CFACD83F36A99111ED5768DF2D116D2121E32357BA4F54EDE927F189F297D3A97FAD4E9A0F5B41D8D89DD7FE20156799C2B7B6BF9C957BA0D6763F5C3BC5129747BBB53652B49290CFF1C87E2CDF2C4B95D8AAEE09BC8FBFA6883E62D237885810491BFC101F1D8C636E3D0EDE838AD05C207A3DF4FAD76452979EB99F29AFAECEDD1C63B8D36CF378454A1BB67A741C77AC6B6B3F95F4F02B64DABC15438613EA49750DF42EE90101F115AA9ABB9FF64324DDE9DABBB01054E1BD6B4BCDC7930A44C2300D87CA78C06924D0323AD7887E46C90E8C4D100ACD9EED21E"),
	     SHEX("4E12A03995768C80408869A5C45B174783EA2FB5E4C153A2C01"
		  "4A3FB8AB968D74C9B835D9C4C5FC304B5E68CF9AA72C2ADFC2D"
		  "E961CC9A90E1D515A3234BB79DA1D5FA330C03DEFF46D304F0F"
		  "0378324BE4FEC285758B2DE3788E71EDCD05754366C71B783A2"
		  "6D105A48BBF51C4F4E769BDD19CA477D80FA55695A04B06A097"
		  "0B98BE48EF1E861F17B9540E63F54B6156FC2996B2D2CA69326"
		  "E0CB7A4A5761B39FE089017322BF23B7B5DDBA1490541843BF7"
		  "03508F5512117370253248F23F30DCCF27061B8F81C1090CC8E"
		  "D866263FB50AD6F3A027EFBDAAD7760EAFCEC0928BA40173A50"
		  "C61596B75AA4D72986A8BACBA61DADDA139374D17843735136C"
		  "8DC0E14C2EBCBC9EB110A682CB77A72E790C68E11848431441F"
		  "F70CBBC513A9D389D349A63AAD6A948A42DD798F1A576DF57ED"
		  "0613FFF5F569CE5D24061A386B2A5B4B1FDE6E3359CE4125241"
		  "628B9A882CCFD34F266BD8B7806392D52DC8CCED80C52ECB632"
		  "1CF3B3C911D206809FAF8FDA3F918F93FF507089CA4BD647AD4"
		  "B5498F0FFCC7563FE5DB7BCD613B84EDF4A95A4F8D29E3ABE0C"
		  "39C8A76B6FB5DB51DB09A21A48D3E2C0786B6A13351715F8DCA"
		  "25EF288C46631B25D55A7473E5059D6A16661F948D8A7C1B4D7"
		  "A21B531F915FC5A18305B6FDAADBC1F201C9A09DE221ADA6DE8"
		  "BAEDFA4614E578156137FBF43896CB61CE38EF386FB68D55311"
		  "C63C"));
  test_hash (&nettle_shake256, /* 236 octets */
	     SHEX("3A12F8508B40C32C74492B66323375DCFE49184C78F73179F3314B79E63376B8AC683F5A51F1534BD729B02B04D002F55CBD8E8FC9B5EC1EA6BBE6A0D0E7431518E6BA45D124035F9D3DCE0A8BB7BF1430A9F657E0B4EA9F20EB20C786A58181A1E20A96F1628F8728A13BDF7A4B4B32FC8AA7054CC4881AE7FA19AFA65C6C3EE1B3ADE3192AF42054A8A911B8EC1826865D46D93F1E7C5E2B7813C92A506E53886F3D4701BB93D2A681AD109C845904BB861AF8AF0646B6E399B38B614051D34F6842563A0F37EC00CB3D865FC5D746C4987DE2A65071100883A2A9C7A2BFE1E2DD603D9EA24DC7C5FD06BE"),
	     SHEX("8BB1E121FC89EA0F3D79EA823D876EAA74B6CCFAB369E7E4783"
		  "CBCD3701875F1BE59B0F8B48C2587E7001B6E7ED1821E264906"
		  "9E84201583B2562E5945AF5CF734E2F0C6090DC8281969CC95C"
		  "861ECAEECB1CDA78E8FE119B60B05C54484BAC951B7784ED025"
		  "FB878E29215916C5886B85B24BB2521FEEBEC3B8766BA14FE47"
		  "EE0E81DAD68B238905377AFDBD55E41364A12181D46D2E272BB"
		  "F1FBF41A83C29ED88361435396614A23730D84684D3F5A622A2"
		  "33FF1AD5583FA707DB008E3EED5A9716219EBF3571FB1CE9842"
		  "D4A37B519767B7392F008193DE676F0CD192CD6D0CF651EE5F8"
		  "331B0982F150A7DCC264C1465177AB98CC590BCA81E981AD2E9"
		  "8E47D2D855C0A74895970A96994700A5D55CBBAEA2472C4160D"
		  "A0EF3AEDE987A99065948ADC558E7CB9A952495F70CEDE70CE0"
		  "E3DECCB798EB58941F38C4B13C28B2E5CA4FE3A7CF3C973EB33"
		  "91B705B9E9E7689E264F878396F438C393EB49E70D6A2BB0A6A"
		  "92F80FDC2C2F70AFBB2A1DF1EC4634BC0CFDC9D1239182DFCFF"
		  "D30096E11FC4A32BB44CF510A44D35EC3CBFC354E088E0B5393"
		  "2D0DE4DF83A963D902971392B74D95C475D26184559F6F9F5F9"
		  "1502A07011B9070F8E9BE5F5D1E599634E3BCED756A49AC7CB2"
		  "73396F7ACDFFB3BE1E441E8D8ED0588D0931353736DC1D9E889"
		  "650B503A4FDBF5DF7B5708D4218404306B0E11AA08067B7DBE6"
		  "3E8C"));
  test_hash (&nettle_shake256, /* 237 octets */
	     SHEX("1861EDCE46FA5AD17E1FF1DEAE084DEC580F97D0A67885DFE834B9DFAC1AE076742CE9E267512CA51F6DF5A455AF0C5FD6ABF94ACEA103A3370C354485A7846FB84F3AC7C2904B5B2FBF227002CE512133BB7E1C4E50057BFD1E44DB33C7CDB969A99E284B184F50A14B068A1FC5009D9B298DBE92239572A7627AAC02ABE8F3E3B473417F36D4D2505D16B7577F4526C9D94A270A2DFE450D06DA8F6FA956879A0A55CFE99E742EA555EA477BA3E9B44CCD508C375423611AF92E55345DC215779B2D5119EBA49C71D49B9FE3F1569FA24E5CA3E332D042422A8B8158D3EC66A80012976F31FFDF305F0C9C5E"),
	     SHEX("1887576149594F7FCD27145B5F535219BBC4313E62E7ABB6393"
		  "D6015E8E45B48A211300F8303FB9BF370FE0A117D268624BB74"
		  "1E0BC343487015872E14F84B1EC1E2FAEB828F521ECE9E7E5EF"
		  "B281DFEF05D86BD8EB0B51CF4C46D94EE0AFC72DA29C3D77EAE"
		  "E07E24035A68792E8EAB1E0B96282E1D95B1B9D36B09D6A2E0E"
		  "DD37227B0D29F9E490F1D6156DFCE1554E69A49B7660D288CCB"
		  "6FAEF9A40F656A90A825A07C5EE0730926CEE14FF2694A3BD10"
		  "5D889C88A8CA1D271727DD0FCC0DDD1CA2EE955D02CF8CDA7F9"
		  "E3A00B4362C69AC631523345DABD396354AD862458736D1BEFC"
		  "4BC35EF4775F02F017BA2EB9FFF9229292432AB5603BA20342E"
		  "627D380824050FF61C23B5E789DF2E835FD56C65FEBD001EE69"
		  "60C782B627F90A7087EE24D32A85D0D780A8911945F3719C5CE"
		  "EB9C7024DCB112C0FCDD72237040C5E964CA57EE504849E3015"
		  "C200C44CA3A0EEC097BF718F6959C4D3E39455976F98DB5FD84"
		  "F01EACBC1B2B57B6483C9E3D58EF76E18F8FBFA341E6399A3D9"
		  "86834E315A2762D239DBB6D1EFBD60BA4EE974523D2EC7A9F0A"
		  "171484D3F09D59019FCBB6A480ABEBFC06A21DC5AF8FBF1B538"
		  "8278CF84311E56329A6FC3A9426517C3CC581C8C8EADA1C656E"
		  "568BF108F66287FB66A405EB307EDA530C18DB2CDEE78D8DE9D"
		  "CB8E93F195F54086F875935F7736C7C85EAAC106F00D28D6753"
		  "843C"));
  test_hash (&nettle_shake256, /* 238 octets */
	     SHEX("08D0FFDE3A6E4EF65608EA672E4830C12943D7187CCFF08F4941CFC13E545F3B9C7AD5EEBBE2B01642B486CAF855C2C73F58C1E4E3391DA8E2D63D96E15FD84953AE5C231911B00AD6050CD7AAFDAAC9B0F663AE6AAB45519D0F5391A541707D479034E73A6AD805AE3598096AF078F1393301493D663DD71F83869CA27BA508B7E91E81E128C1716DC3ACFE3084B2201E04CF8006617EECF1B640474A5D45CFDE9F4D3EF92D6D055B909892194D8A8218DB6D8203A84261D200D71473D7488F3427416B6896C137D455F231071CACBC86E0415AB88AEC841D96B7B8AF41E05BB461A40645BF176601F1E760DE5F"),
	     SHEX("EA520B7CEA0750C8F1AE4D7E5C80E95140D5A980D68F48807E0"
		  "408A0D17DD084575191EC17A8D5B3D7A458774C453124509AEE"
		  "866183DFCA7636D743B6B5798C4D62951D7DB5640D1F4467D87"
		  "03E08411270F3493241AEF38FB757F00F0EE6A66C732F138294"
		  "076DE25F0811E8E1BAF602E9F38CF92E96ACF630EBECCD62C28"
		  "37DC0F361C014698B3CCF586D372BB0591AAFE68D138E1FBF05"
		  "D205A6124B0AE5F483BBB429EB1BD517C799FBD2B9A154FDB7F"
		  "687DD320106793D95D2ACECFCB4811C627B3287490116C5F5A3"
		  "082B0C2BB3B703FB2BD92B65E1CC07C61E1AD67C0F5B45C33C9"
		  "81E56CEBD6C12D00D9D33C5DAC18A41C1025AD24798DE28518E"
		  "5FAA6CECA295CD13254367F04CAC72FE0A0FE87214103FBDCD9"
		  "AF3FDE362D6B7C0997B86CDDAB824210D1896DEB7C75A3EB92B"
		  "E415F450CD62CAA1FE4D7A9DA2E8B57F2C31999F81CA8A22936"
		  "684133DDDE9FBEC522C92B928191220E75F8FD20C29A1DA6624"
		  "55FE7B94A2292006052B354DB4F03375ACB65044C2FDAEB977C"
		  "A8E81318D2EA95C3981CE76DD9292678BC5F05351D2B4A4A410"
		  "DEC6790DCFA90AB7D9F8D7FE71B1A1E22BFC3EF06F21C7C99F0"
		  "1E11060ABD9122DD3FFA6F0C2B402D15C6D4E8F8524A5AE6F12"
		  "48B6878170F8B4A7D95C85B04587066E340156D80B20E708F54"
		  "70801F4A5F46A74DEE3332FB0ADF0472CCD592B2462EAE969AC"
		  "5140"));
  test_hash (&nettle_shake256, /* 239 octets */
	     SHEX("D782ABB72A5BE3392757BE02D3E45BE6E2099D6F000D042C8A543F50ED6EBC055A7F133B0DD8E9BC348536EDCAAE2E12EC18E8837DF7A1B3C87EC46D50C241DEE820FD586197552DC20BEEA50F445A07A38F1768A39E2B2FF05DDDEDF751F1DEF612D2E4D810DAA3A0CC904516F9A43AF660315385178A529E51F8AAE141808C8BC5D7B60CAC26BB984AC1890D0436EF780426C547E94A7B08F01ACBFC4A3825EAE04F520A9016F2FB8BF5165ED12736FC71E36A49A73614739EAA3EC834069B1B40F1350C2B3AB885C02C640B9F7686ED5F99527E41CFCD796FE4C256C9173186C226169FF257954EBDA81C0E5F99"),
	     SHEX("AA38A2AA24D83C0055C05A4D8E5FD6E538DDECDDBC6876D8169"
		  "00F4329A5E6D00A25B14CA626F3C9185590A151E438CEC7164B"
		  "D7135BF60AFB78AA0C4148FEBB606C50A0449360952C1BD6C46"
		  "54804102521315E457AFDFA185565D96B6B230A4B36E0DBF215"
		  "B19FB9B0BB3CFD6A8CC2E4ABD6A1A50D978EBB5D351F7A65D39"
		  "855E9AE57D7343E8FF7EB6FE7974791EDF14861F30A428CFA9F"
		  "A08E3EC4C4F1C6C5CF29A06D777459AECC379BF17E89A54B753"
		  "FCBF3846835CEE9FA834CB294AE3699893ADC7DE63D96F1F7CB"
		  "60739E7D672A9D70078F9903F6617EAD83EC5847424EBDB89B9"
		  "A07762887F3A4CE8B7A839F0FFDEE21822DDF92A73E85F112D2"
		  "A6E023C620056E97E3435CCC172E768A195BA50F0238AD3A474"
		  "FDDAD6ECD5B90262170DD026826F69CC2ACE1D7D9AABA9F14C7"
		  "3C6F114ED5E5F75679679B7A2ECC81432BCBA993C4E1CAAD6DB"
		  "7D9908B56E1404EE5080641A4D261D488457A1DA7D5E1F09BFE"
		  "9DB23E3CD4BE23B8FA84C93011958D57588A06D8DF433428AA1"
		  "3CEFD6EC348A52E0076564F0DC849058A601137835F90478B41"
		  "BD58D24ED9B5E4EE5940A0963BB4FD7A2F5817BE1903AB7D960"
		  "F58EF88A310675AC0452AC6D1EE0CD65E2114865C4ADF78B983"
		  "EB7E894F28A716ABBC766A1BAB1858641C8CBA3A18BA55584A1"
		  "FDC2F80A61C36A9A56CD0289257CA0402ED99BC680AADCB2928"
		  "7119"));
  test_hash (&nettle_shake256, /* 240 octets */
	     SHEX("5FCE8109A358570E40983E1184E541833BB9091E280F258CFB144387B05D190E431CB19BAA67273BA0C58ABE91308E1844DCD0B3678BAA42F335F2FA05267A0240B3C718A5942B3B3E3BFA98A55C25A1466E8D7A603722CB2BBF03AFA54CD769A99F310735EE5A05DAE2C22D397BD95635F58C48A67F90E1B73AAFCD3F82117F0166657838691005B18DA6F341D6E90FC1CDB352B30FAE45D348294E501B63252DE14740F2B85AE5299DDEC3172DE8B6D0BA219A20A23BB5E10FF434D39DB3F583305E9F5C039D98569E377B75A70AB837D1DF269B8A4B566F40BB91B577455FD3C356C914FA06B9A7CE24C7317A172D"),
	     SHEX("42DB17A940111F1A93B0D583F6173991252286BC5098E136D27"
		  "196697AE45B878767E17AA8E826DCDE186CD76F2024E6772A0D"
		  "EC4C4A8DB71959D75CFEF0F20CBCCD85F15AB5960C0CCDA5F14"
		  "BA0F02958752A0494D4B50E2CC67C554196BFC061F651440C68"
		  "12CF5319639F93DF1FE51D1511DF6A3A6AA6538F46EA2D00A3A"
		  "0F64B25CC9EE8AD05E321649987C36387E2D31DA77EDC1AFA96"
		  "32D77EE2D4ED54992BD51490A34DD743DCEC567DE957D55505D"
		  "1D8DD43BE3309F25AE61275203BD9743F3027B9F34B34DB7504"
		  "C2D7FD1AA99B936E7284F9E252DF2E3359D868C0C83AB140E14"
		  "C01B4B650D99AC6DC54AB35DD825116EA96C5FE88822B2D7182"
		  "FBE9B3F3CFBEDEE129EF0814C171A207FC9793E1EB72EDCD605"
		  "C3E3B1534106093AA550BC8CFF8DE3C5D7152F18C87E161DA0F"
		  "C90594161F03B0FF9F9EDD41F360DB7347B067D8FE1E901D78A"
		  "77F1EC44E33ED7AADD463C7D676E50B03A92B5417628CF986D5"
		  "F93197333BFEB5D7B6A7C7863707B7EC032E3F5C62CA6A02938"
		  "313140E62479842F4CF68A54A14AE0AB35538025920F3170391"
		  "69636F35F02A8E05C344E15492EA02164C62392B0795D4889F7"
		  "AB68AC969B3B096C21CFE179833BA5FB857751C4713FC928B46"
		  "23DB593C9458AA36BA3B3693E07C3C5C84470F77E5CBDFD9770"
		  "877A0999DCF8FF1AE27F51B65AAEF9754E8A71739AC00BE0F94"
		  "58E0"));
  test_hash (&nettle_shake256, /* 241 octets */
	     SHEX("6172F1971A6E1E4E6170AFBAD95D5FEC99BF69B24B674BC17DD78011615E502DE6F56B86B1A71D3F4348087218AC7B7D09302993BE272E4A591968AEF18A1262D665610D1070EE91CC8DA36E1F841A69A7A682C580E836941D21D909A3AFC1F0B963E1CA5AB193E124A1A53DF1C587470E5881FB54DAE1B0D840F0C8F9D1B04C645BA1041C7D8DBF22030A623AA15638B3D99A2C400FF76F3252079AF88D2B37F35EE66C1AD7801A28D3D388AC450B97D5F0F79E4541755356B3B1A5696B023F39AB7AB5F28DF4202936BC97393B93BC915CB159EA1BD7A0A414CB4B7A1AC3AF68F50D79F0C9C7314E750F7D02FAA58BFA"),
	     SHEX("82F8C16B8487E008284598FED9C5670C86FE35C8DEA51AC59D3"
		  "8722829C94094BB2766AA4CDA3139A15DD2AC0309253EC1A15B"
		  "2969F84899637D3E2A5D55ED1E87561A546A2A7C190008BCF3B"
		  "EC233EBD1DD55242D7FAC3A522C6016E62670F66EDB6290AFE8"
		  "01630E2BCCABB450E4D44A9BD8F63D4B04E9F280A2A0972AB22"
		  "9351906F5F32DE714867E8F04D276488F2AA5E62DB8B6342874"
		  "04A73723ABB7E152107A6DA60F0C7EFAA953DB706398A09835E"
		  "5EDD99393DB5A162FE3E564298395FCDAAE8FA7E7A9081B8E14"
		  "FA0E99E433E0D44326DC2E9E33378EA719EF780D303589F0A63"
		  "841B5D5D74EC6E2BB073B55287A6419BC6AF759D13F8C8875C3"
		  "72C12D392D7F80608412EB9A92A786048BF9A6F03D11C48F48B"
		  "96FBE8BBA329268242A5463C49B112AD71B8EC42A9A7F272F25"
		  "819509E5C7A10D9C9F1A386E034BE2A89AE28917CF014156211"
		  "C795D9D6EFB267D8F2BC48B5AA767CF8269A6E349701E77EF69"
		  "F03FBB56C8E2A0C90905F1FDA649B75D716C67894E932C63B13"
		  "911C80339A1DF4B7AE2035C7572BBB599E66E7CD9CF447F4F1F"
		  "1DEB63F64508EBE19485DB3EB98607F3C87DEA60FAA331310FB"
		  "4BA5730CE4024E07B4B134286472BC7B6C816C3F680618941F9"
		  "6539B81DF82CD15EDD426206F363A9E487C9FADBBB9DB90DA8F"
		  "2D73E940C515A7B4FFA07443D35CAA0C89EE9E511D66C2C8FE2"
		  "F0FF"));
  test_hash (&nettle_shake256, /* 242 octets */
	     SHEX("5668ECD99DFBE215C4118398AC9C9EAF1A1433FAB4CCDD3968064752B625EA944731F75D48A27D047D67547F14DD0FFAA55FA5E29F7AF0D161D85EAFC4F2029B717C918EAB9D304543290BDBA7158B68020C0BA4E079BC95B5BC0FC044A992B94B4CCD3BD66D0EABB5DBBAB904D62E00752C4E3B0091D773BCF4C14B4377DA3EFFF824B1CB2FA01B32D1E46C909E626ED2DAE920F4C7DBEB635BC754FACBD8D49BEBA3F23C1C41CCBFCD0EE0C114E69737F5597C0BF1D859F0C767E18002AE8E39C26261FFDE2920D3D0BAF0E906138696CFE5B7E32B600F45DF3AAA39932F3A7DF95B60FA8712A2271FCAF3911CE7B511B1"),
	     SHEX("706CA15167517D46C47844D044D0FC94CAC5D23B90BDB64CE0A"
		  "2DF052AB37C967E57A476FA23C4D3C160C47371CBADC07A48D6"
		  "A7C0E0B8CE8B337CAC62E74E10B68C6FBB109B03547CE998780"
		  "B605429D6E3061334C702B218C1D6686E3E8C7FCC2C4DEC034B"
		  "9FE68572B45E58DE9661216D6A232D13A0EDDDA45892A4ABAB6"
		  "66CB528694DE50A50D1368BE99D257BC8C245429FE55000D3C9"
		  "CC861FE9228E6EBFB7DDC53148B530E2FFE153B8F40007ED321"
		  "B4464C0BD5DE00C9791073B199AD57F6C242F7CB772FF05A06D"
		  "010D156AD35DD0ED74C9593855576B1C7A4360770AA88BBF56B"
		  "65395B163471E4057D27B59B554EECCD9AB8F24F457E400150C"
		  "5F83B55D703620D68ECD7A01A9B3FEC468B75C0F8BBBB083883"
		  "18D50D38BECDCD30F8644A2EBD11FD8E1D103AF5F9B64D92BEC"
		  "64FDDEFFC73D9932871CCD1DC62515F0EFA438393109C638F5C"
		  "11405FD923D784B217F3417BEC855A7CA290253CDD902802EBF"
		  "400015CC2AF646AEF63684ABB39BB98AA598AB767AC4DF96CDF"
		  "631D0500BC61DC822F2C54D761F0C926E023310AC11F3748E8A"
		  "75564FBDB7039CDD3A42B601339BE64BBDA4B3F70899700FBDA"
		  "55D1337F5E33F5B00DF97E6CA218D212EB771ECFE27B265A8EE"
		  "8AC9D9F34BAF279C49B318EBFBA497CE77233CAC4A9146EF5DF"
		  "7B93672A553A3EC46DAFE8B7F93F233161FB91F5BCA89DB4D5F"
		  "2C42"));
  test_hash (&nettle_shake256, /* 243 octets */
	     SHEX("03D625488354DF30E3F875A68EDFCF340E8366A8E1AB67F9D5C5486A96829DFAC0578289082B2A62117E1CF418B43B90E0ADC881FC6AE8105C888E9ECD21AEA1C9AE1A4038DFD17378FED71D02AE492087D7CDCD98F746855227967CB1AB4714261EE3BEAD3F4DB118329D3EBEF4BC48A875C19BA763966DA0EBEA800E01B2F50B00E9DD4CACA6DCB314D00184EF71EA2391D760C950710DB4A70F9212FFC54861F9DC752CE18867B8AD0C48DF8466EF7231E7AC567F0EB55099E622EBB86CB237520190A61C66AD34F1F4E289CB3282AE3EAAC6152ED24D2C92BAE5A7658252A53C49B7B02DFE54FDB2E90074B6CF310AC661"),
	     SHEX("A90A149C4B7BA649888F90721E9FF91CC3589E2094B07909959"
		  "C9A15FF010C610824F7A49AD3CA2864089E93AD7075C1966033"
		  "AB55E9263871DDD56943D568FE0867221034994B74A3F855544"
		  "E77E313A29EFBA367F309243DB8A3027EB4DA5371439CF39C21"
		  "EB57823E1E6C73C0707356B68828AAC6C8FD05F222F4C58B33C"
		  "EA1AA6344DC33FA88D9A5BECC263B7CBF00045378910814C77C"
		  "2224E56D913D2CBDC48EA6700DE0E0306B499A3888F53527406"
		  "9E7ED493C12D46D23B7344CDC11B3433E39ED9CFAEA6787704F"
		  "FD63BEE0072BA60F526FB52DADDB41EC7E8216AC2A79BBE27E3"
		  "15A2E2824E1DD0BC2943F92393550AA6DC75F1C9755E5D8F811"
		  "CEDBC2139805D5599C1E1F6D4F56AAD38E91FFB529A4820F260"
		  "D752A93D7BE57BC2F75234FA615AC72AFEC7619E0F26071D8A3"
		  "C90F1FB486497D5AF0EE45FD9D0B4AA0E07201B9E5F277E73D4"
		  "8EDFBF75DF8C3F1A635A733591615C392CE37738851D2AE847C"
		  "6EB9AB5D91DCDA7E953041CC6CF02F0E6604E8AE5756DE52864"
		  "5FEAA0E8F1C5F6F0C53ABB72F84D0397A39A7FA3C7E8BC72B23"
		  "7E0D8B81C91ED18E09273CA20B763D34B779BB79D95C3749AAA"
		  "703DE53FD7FCFB361E2D3E13068FA076021C1373CE99DDA0DDF"
		  "931D2BEDD095A3F95E41A0B09C56D01253F29558348D6F84190"
		  "8984E3096A7A0DA6DE77A5DB124AA442EDC394126C2F67B1B4A"
		  "0DEE"));
  test_hash (&nettle_shake256, /* 244 octets */
	     SHEX("2EDC282FFB90B97118DD03AAA03B145F363905E3CBD2D50ECD692B37BF000185C651D3E9726C690D3773EC1E48510E42B17742B0B0377E7DE6B8F55E00A8A4DB4740CEE6DB0830529DD19617501DC1E9359AA3BCF147E0A76B3AB70C4984C13E339E6806BB35E683AF8527093670859F3D8A0FC7D493BCBA6BB12B5F65E71E705CA5D6C948D66ED3D730B26DB395B3447737C26FAD089AA0AD0E306CB28BF0ACF106F89AF3745F0EC72D534968CCA543CD2CA50C94B1456743254E358C1317C07A07BF2B0ECA438A709367FAFC89A57239028FC5FECFD53B8EF958EF10EE0608B7F5CB9923AD97058EC067700CC746C127A61EE3"),
	     SHEX("FCD5FEF0DE750C6BE7615BCB850676E9E83299C522BDC14FBAC"
		  "162F927B0AA4EA6DD76E43B1500C72BE0D569566B57054B9340"
		  "216904D518ECDB25EF39A62891290A1034F262A5E2C0FB37333"
		  "BBD24323730CFD40FC161EDC37EEB7C69769D3AB1F1454560CA"
		  "78FB96CF133689EF8E75F57C2442E2CBAE4F5C648666BFC7FA5"
		  "DA87E7C0DD4816292DFD56B2BCB383B3FF971185B724430F959"
		  "6F3509C767E159873614B1D4EC45FA8586EC2321B791560785E"
		  "5E3CE13AD5E8E1550A3290175ACF2D78CDD17C0EA205ED7C139"
		  "6BF9DC0844168BA3A66B4CB01C22629C80577DE68D72EDB44B4"
		  "19B13A2DB6D67877D9CC0A0201C14DAD69F25C88C3418AC45EA"
		  "0D7ADA58A02BB7DD0C276DFAB29164E1844BAD901ACB6374AB7"
		  "5E21BFE2F3591B2C8F111A4606C8C52D85B4C073FDBBF25FC81"
		  "758214CA172FBFE1C6B14476532F35F981C312739CE435F7AD3"
		  "D0785DCD648E2A5391B64525A8785F3E06446C6BC1E09BFA19C"
		  "7CE52651BFE1A2E355CD7C1C9CED6540D33194A2212B2A8768E"
		  "7347042B560F68B4E22E42ED8F51C38375F9FA45C133A90D4F6"
		  "F17CD5BF19DC9E1035DF78F7E0077411BFD0FDA23E6759388AF"
		  "C45485BF271C8FDB316A32B898C3D6DCC0738B3AA043ACD1848"
		  "FB7C6FE86A844FC764953D392569520E78FA11BCD6B0AF3BE8E"
		  "7BCB0006F56881889CDDE27870FE7C9E703F57B6CBE0D6372F2"
		  "322B"));
  test_hash (&nettle_shake256, /* 245 octets */
	     SHEX("90B28A6AA1FE533915BCB8E81ED6CACDC10962B7FF82474F845EEB86977600CF70B07BA8E3796141EE340E3FCE842A38A50AFBE90301A3BDCC591F2E7D9DE53E495525560B908C892439990A2CA2679C5539FFDF636777AD9C1CDEF809CDA9E8DCDB451ABB9E9C17EFA4379ABD24B182BD981CAFC792640A183B61694301D04C5B3EAAD694A6BD4CC06EF5DA8FA23B4FA2A64559C5A68397930079D250C51BCF00E2B16A6C49171433B0AADFD80231276560B80458DD77089B7A1BBCC9E7E4B9F881EACD6C92C4318348A13F4914EB27115A1CFC5D16D7FD94954C3532EFACA2CAB025103B2D02C6FD71DA3A77F417D7932685888A"),
	     SHEX("D4C1F39098D123B0DD88504FF6AB424E31F5D3726EFBF86500C"
		  "E730BAE87EC671C97F768F9593C8411B57E306B0CB394F33B86"
		  "6493BBFC3F089FA4E875858FBCE814DEBB905FA7249F7DBE737"
		  "F5F6725CF76D2A50B319D96BFCD55E03A6E923B01003E6D5976"
		  "A66AA5250C337C9C4C402B9733C8F626789FB23FF1043494A73"
		  "960297CA9110D23DEF1D72B601F6C42CF83938003B3A386CDDE"
		  "D4A70B24975C958597894FD6E2C412CDC6E32EA7F398242F84D"
		  "4F4BD469DEE9AD10B717EBAA26F2481224F47135219231279C3"
		  "6867751A476AF60AC4AF8F151F301CACF5D7890D62A135B4414"
		  "9CC7405208212B7547EC0617ABF96A7741EB2E08D18AAA17357"
		  "B4F2DDAB92257E893CFCB91BDD73DFF3F8E6C28FDF38DC89094"
		  "C719E69508AF7ABEE4D1403E0F139348632A45708AAB27D4403"
		  "F5D5B85A81AC72EDE9E2448F1FB87571CCA3631BDA9F9D0DC06"
		  "0000B03450FBC7D98AF74F0B0A8A1BBDEED01C4C8300B05E05C"
		  "F7AF7824B246666EB53119FCFAAD3C7AAEEFE0274A55ABC01AB"
		  "F0A38189E46CD3CE56F0A4DB8E9B498EEC68E03DB7FDC917723"
		  "0DFC72F45A41A7064F42F3C527A5E5473B2F68D1C8BA960BD3B"
		  "458BE2BEF2C3623B38A66266EACEA3CC93C09F4CF566327E78E"
		  "C39CE5541E48142FEF380AF5161ED9C63724FA16384F574B1D1"
		  "CBC3F1C852028CB9A8D1F411D6AC119E11E6202E566B49341D4"
		  "0D19"));
  test_hash (&nettle_shake256, /* 246 octets */
	     SHEX("2969447D175490F2AA9BB055014DBEF2E6854C95F8D60950BFE8C0BE8DE254C26B2D31B9E4DE9C68C9ADF49E4EE9B1C2850967F29F5D08738483B417BB96B2A56F0C8ACA632B552059C59AAC3F61F7B45C966B75F1D9931FF4E596406378CEE91AAA726A3A84C33F37E9CDBE626B5745A0B06064A8A8D56E53AAF102D23DD9DF0A3FDF7A638509A6761A33FA42FA8DDBD8E16159C93008B53765019C3F0E9F10B144CE2AC57F5D7297F9C9949E4FF68B70D339F87501CE8550B772F32C6DA8AD2CE2100A895D8B08FA1EEAD7C376B407709703C510B50F87E73E43F8E7348F87C3832A547EF2BBE5799ABEDCF5E1F372EA809233F006"),
	     SHEX("D6FC1E61D2BC27CA5DD5ACD290C2831070E66C38F03D2DC9118"
		  "4AA9EAEAF03A28E009A1AE1C775F4274B3AE6B5491C9B0C838C"
		  "F90D185FE94C49EF46ABABA62D6ED1F550D83466EFC6B97789C"
		  "A36AD1A6A3E6703A5DA9764E47F29C8096E3D317FC7AFF0B79B"
		  "1C278610AFE3F5E7FA7FD52FD36CC9E7000DA809346281EEBA1"
		  "EE9BB06AAD61F3E02CF4A5BB7B211E2437ED21437FAE78378F2"
		  "8B14184B7DE7E80E845837A483C0C6DF8E233687F71F0F9745D"
		  "585411F103356A0C474F9238403E40960F63A3418199325EA6E"
		  "8EADA31C94F78CFF1B83A2EFA4E49045E0CE34A5C6102539932"
		  "8CD7F79A1E4CC792D8AE9F1AA954235B515158BE5F4064B889C"
		  "14A3A83AB59CF26945DBBF70F3A5F3BDBE8674546D6DF384C40"
		  "D56B286D309A114871BC18E144D2334C88C32F141F6DFA70859"
		  "A9FF7D4996BD3FABA0299794E861DB8BE8A324822C4B1DFBFA7"
		  "0F8D9149C3DC01E0E556DC744C98FC2F92C8600C856BD174759"
		  "AF467D6618C4347366CE04D06A0EE4FEB9A27B79BED01BEB7DE"
		  "E41859C1306939D55B0E319DD2E0E001F6174361CB804CD3AC3"
		  "332B8D8A1D8204D5255F81FAA0BDC29AE71693586C987E35C81"
		  "E314A3848B03CF7B37EAEE2AD6ED4736E1F20A098DDAEC612AE"
		  "E17294313DA16453D7901CF80FC729962206AD4820B00E8EF0D"
		  "999857832256E5B784E8A89DFDDD38F6E15DE081A82E19A2A05"
		  "D417"));
  test_hash (&nettle_shake256, /* 247 octets */
	     SHEX("721645633A44A2C78B19024EAECF58575AB23C27190833C26875DC0F0D50B46AEA9C343D82EA7D5B3E50EC700545C615DAEAEA64726A0F05607576DCD396D812B03FB6551C641087856D050B10E6A4D5577B82A98AFB89CEE8594C9DC19E79FEFF0382FCFD127F1B803A4B9946F4AC9A4378E1E6E041B1389A53E3450CD32D9D2941B0CBABDB50DA8EA2513145164C3AB6BCBD251C448D2D4B087AC57A59C2285D564F16DA4ED5E607ED979592146FFB0EF3F3DB308FB342DF5EB5924A48256FC763141A278814C82D6D6348577545870AE3A83C7230AC02A1540FE1798F7EF09E335A865A2AE0949B21E4F748FB8A51F44750E213A8FB"),
	     SHEX("F61115AE2FD5442877D08E39A57DF8D4A5F117D57C2688617A8"
		  "DA64F9482F5A817E0D3DFE41A9CFF331F4D5E3240DA38E43B7B"
		  "87925FE378512CD83172FC3CC2396A8ACDBF5010CA576603581"
		  "BF7DF6D7529423709B30DB81620B9F660B5AA82EA6C98656DA2"
		  "7B4709DA85D364EC015C2EAAF2AADEDA2C040BADFAB16F7C2AE"
		  "39DA6A1691057301B7C3252884CD9EE079E68857D6D3B7F4F85"
		  "556140DB03421F09E98B80A8422500A349B5C0D8FFB25433A41"
		  "21B7682F8517500930E5F2766C687846553C9F76DE38A0EE41F"
		  "76D3A22696B95774BD5EB747F6445D8044147EE38B127892F4B"
		  "0B1932470B454F8F1BB697A2F6659AB13F3E2003C97D292A787"
		  "8E12CE47AA9619FCFE2135D8890710B2733DF40FC75DAD57B99"
		  "042E4F54AEA9332ED52CAB8FFDE5195EDA05368B09E9FD55731"
		  "DA20012A7D255AAA33D74E22917E58BD546EE90947A16044B14"
		  "DFFFAB05916C382D1F038B63D2F20DC09C117C8724A4DB3970B"
		  "7CB143B1522D010881FE6663410DB3400DB33B62A6E70DFD48C"
		  "2FE773054E1494944B60FC5D3BC803027C4B12E6A230FE41CA4"
		  "D072F3628D20782CD7DE4E98A29DF083FEF503E46EEFA787F8B"
		  "0E9B1E7C884B9122977532BD71FB63120C1069156516D265FE3"
		  "559B18965E968B9BBD653484866847E76262D0CCBEE8FB0314A"
		  "8C40B8C35725337CA4A241592E47D5B8981D3212C62AE7E25F2"
		  "7495"));
  test_hash (&nettle_shake256, /* 248 octets */
	     SHEX("6B860D39725A14B498BB714574B4D37CA787404768F64C648B1751B353AC92BAC2C3A28EA909FDF0423336401A02E63EC24325300D823B6864BB701F9D7C7A1F8EC9D0AE3584AA6DD62EA1997CD831B4BABD9A4DA50932D4EFDA745C61E4130890E156AEE6113716DAF95764222A91187DB2EFFEA49D5D0596102D619BD26A616BBFDA8335505FBB0D90B4C180D1A2335B91538E1668F9F9642790B4E55F9CAB0FE2BDD2935D001EE6419ABAB5457880D0DBFF20ED8758F4C20FE759EFB33141CF0E892587FE8187E5FBC57786B7E8B089612C936DFC03D27EFBBE7C8673F1606BD51D5FF386F4A7AB68EDF59F385EB1291F117BFE717399"),
	     SHEX("3BBF30FAF9241C9E3E27EC2C17AB24AA3673799371528CB98A6"
		  "D9B91E553553680505D7CD6590377AB54E5DC7D7C72035300AF"
		  "651AB0EA52ABA9FE1D490EE5C95B146138F61F957C5CEC8D463"
		  "3175C99A00590DDB987D2E032547D99C691B32C12D7392742A5"
		  "AE47D03B5A08A544548BBEC45B9580DE1CECB7B9F06A88614DA"
		  "A116D7200DC8E81EF3C7C5A716044133D2A66FFBFA35E1668FE"
		  "D9C1045DA77DF92564F60087D173F71B75F040BC6CC14923465"
		  "9C1213BEEF18C53CDEFEF2FF466A2F16823D7659E04AD61F6B2"
		  "3DDCA8BF35AF892BFB8A952DF4ED254FFAC52CF71F4C872EF15"
		  "EF11FAE9967FC8F1C218E7CABA5382936F4CBBDF7F41AA2A239"
		  "BCF36FE158B7D771BECF8981DDAAAC31F744D949396623CB3BB"
		  "754189FB5E78DD94171C1BD6AD3953DDA6D4DACD3B57D37B1B7"
		  "3FAB0D2CAA0B5B3F6A2C9F324F4B93C912991E50F8872BC2764"
		  "51969FF379AAEFC3C710DE16128136AE0D1268D106BBCE2AA4D"
		  "2DB0C5EDB171DFF104DF4F7D49FE017C82083DD1E9F7DEE489B"
		  "B38CE091C222C3A0898F17A0DA1D4B2830C7B7C3C372FDF2A4E"
		  "9CD7B468107513D6E3ADCC91F662218A7364A7340966A4D692C"
		  "F394443203DCE8B33503AEE6C1259BBCF36351AF93EF01764BE"
		  "4863759ED0ABCA9E59CA6004F2E5738AF7EFC7BE8F9A1E10655"
		  "1B77AEFBFF9DBE99ED5ABB12207CF38F7FB9BA77D776FA0DF3B"
		  "FAAC"));
  test_hash (&nettle_shake256, /* 249 octets */
	     SHEX("6A01830AF3889A25183244DECB508BD01253D5B508AB490D3124AFBF42626B2E70894E9B562B288D0A2450CFACF14A0DDAE5C04716E5A0082C33981F6037D23D5E045EE1EF2283FB8B6378A914C5D9441627A722C282FF452E25A7EA608D69CEE4393A0725D17963D0342684F255496D8A18C2961145315130549311FC07F0312FB78E6077334F87EAA873BEE8AA95698996EB21375EB2B4EF53C14401207DEB4568398E5DD9A7CF97E8C9663E23334B46912F8344C19EFCF8C2BA6F04325F1A27E062B62A58D0766FC6DB4D2C6A1928604B0175D872D16B7908EBC041761187CC785526C2A3873FEAC3A642BB39F5351550AF9770C328AF7B"),
	     SHEX("32FFBC063291BD2C957DCDC76632DD1BCE0DBFCAC549228D277"
		  "D81B6ABBA904D43330EF8081E164D8FCAF520E12B388C401682"
		  "95A63D365B57A7694D4C50812B99C86E036116A07F0164B7088"
		  "537688D4BAF1E0FCDC84C6DA8D92789466A606F1EE25CE7461F"
		  "A65DF04BDEBC24A3A4BAC3B02B469C8E85F53D9481DBE244DC6"
		  "7B8BCE84CC7175113D4D7B2413601E9140B7802D9FB712938E9"
		  "C5034CEEA753B7C77C08BC1C03D511ED0B4D5007BBAC38EE76B"
		  "675B73AEF8623CA9CCD579E28BAE7D695BE25FBDB673F3401A7"
		  "5D5A57D142034A7BEE554ED30CC7F03CBB363C10A54F818937A"
		  "E468F5845ADA4F9A2A1BF6887105879DA745AAE64D87F26F489"
		  "A622370405ADA3D3688697DB8F04557736C51725ACA02E4B76F"
		  "6A9E88DC6D52D5A63806BBA1BD687FAAC522D7B5B0A8D607328"
		  "3659056E784320822EAF70666180ADFDBA2675B4EAD6F6B2EF8"
		  "26E7598BBD8C1956A15C1864D249368900D82498297C3E77A19"
		  "EBE20A111D13FC03CA88D9332F919EA4AD15BE8E397ABA0A4E2"
		  "BA85F3FA5999EB553780F0D2F35D76A63B0106406FC3B455047"
		  "6572E1464C986870027849ADA0F164B48B839ECD3CA9AFEADEE"
		  "3F63AA6CA06C635990922995DB0BB99A4AC4B7F3AAC52069F31"
		  "30660AE0F074FBAA1D103A086B59AEE4DA3188CCA3227670FD4"
		  "3597ABFC1A2F6CFA67FA24EF0B20B4AADE9CE8E68CAB34111D1"
		  "0464"));
  test_hash (&nettle_shake256, /* 250 octets */
	     SHEX("B3C5E74B69933C2533106C563B4CA20238F2B6E675E8681E34A389894785BDADE59652D4A73D80A5C85BD454FD1E9FFDAD1C3815F5038E9EF432AAC5C3C4FE840CC370CF86580A6011778BBEDAF511A51B56D1A2EB68394AA299E26DA9ADA6A2F39B9FAFF7FBA457689B9C1A577B2A1E505FDF75C7A0A64B1DF81B3A356001BF0DF4E02A1FC59F651C9D585EC6224BB279C6BEBA2966E8882D68376081B987468E7AED1EF90EBD090AE825795CDCA1B4F09A979C8DFC21A48D8A53CDBB26C4DB547FC06EFE2F9850EDD2685A4661CB4911F165D4B63EF25B87D0A96D3DFF6AB0758999AAD214D07BD4F133A6734FDE445FE474711B69A98F7E2B"),
	     SHEX("BA2B1CEC1BBB46EA19AF7A869FDC73CB3791708DEFDF530DC99"
		  "99E9510FCB02AEF571B3C5E7258D86DC80EDB53B71644110B9E"
		  "82C239FF17EF523840E6AF4D597D814D4E5EC2EA98E451C0531"
		  "8402D5BFF6E6E573565AB640E00BF6F8C3646B3F6656047C60E"
		  "D43393AF7F2D8DF8915A0F6F44D69FE193DD5617FE39BDA638E"
		  "4DDDA771B0E771318A53ADECBDBFF07D6CFEAA04A837034C6EA"
		  "0E70C8FC244304804E0E890F31F951007A2F897BF3F00221B63"
		  "EA3DD8D9D1DD3608596EF7817190D56D36EB99266F83B7F1A20"
		  "80F7A5405CDE86017948BF977C6CC78565B3EEC570B36FC0656"
		  "6909951F31CAD6094F296F759E858BF7AA9298B2F4F7F5C3167"
		  "C145EC126C40F6E5DCCD87C82489400FF8B9B79B19E47259661"
		  "8EED3FBC6FE5545BB3276ED710CA128A3C2FA9B831FA6FA3B12"
		  "6380C4BAED23D9186FAC830891F98C81D401A6513E139F6546F"
		  "065E8AF6E1A9595218A025D8292176F8ACC4ACAEE6308C08DB5"
		  "215FC0064D7279FF6165033E0946FB7C10BF5DEC2D26950B2AB"
		  "DF71AE2B5BAB856386A25117214D5830B7D5E250FDF200CC5EE"
		  "A0085648FA2AAB6EEC363DA41AFD82E14EDA957EA3AB7C65100"
		  "F4A8071ACA6F1891B5465B7C46CAA9612CCB0875369CEC4BCF9"
		  "D99BE35E03AE57C83FC80903AFE25ECF23D016CF6989EDA5B90"
		  "0EFCF59C529FD299CF155A1310B239CD364301F520D16C74E27"
		  "0D0D"));
  test_hash (&nettle_shake256, /* 251 octets */
	     SHEX("83AF34279CCB5430FEBEC07A81950D30F4B66F484826AFEE7456F0071A51E1BBC55570B5CC7EC6F9309C17BF5BEFDD7C6BA6E968CF218A2B34BD5CF927AB846E38A40BBD81759E9E33381016A755F699DF35D660007B5EADF292FEEFB735207EBF70B5BD17834F7BFA0E16CB219AD4AF524AB1EA37334AA66435E5D397FC0A065C411EBBCE32C240B90476D307CE802EC82C1C49BC1BEC48C0675EC2A6C6F3ED3E5B741D13437095707C565E10D8A20B8C20468FF9514FCF31B4249CD82DCEE58C0A2AF538B291A87E3390D737191A07484A5D3F3FB8C8F15CE056E5E5F8FEBE5E1FB59D6740980AA06CA8A0C20F5712B4CDE5D032E92AB89F0AE1"),
	     SHEX("481CF76E2ED1A1EB74050E9632A3BCE4913D2D516E251816A2D"
		  "55706314D6483B955B87B0A6EE013F248E2AB4517B800E23C8A"
		  "D85926C36EABAB1F2C36625D0492C7F001760E3715EBA80AA91"
		  "2F5E3E1A7918D1EDE2FA7FDCBBAE23FF75ADB2BD88527BCE6F0"
		  "2FEF73175487781F26C7C7DBFD04143B70D636217ABF7EF6936"
		  "4B8B7C493C64434DD1CCB523E5CBCC81211A64FEFA9F41FFDB3"
		  "8D9D8D5E0A9E51BE38DCCE5225652CAB5A0EF8D8EE7DD175B92"
		  "9ABA19480299176D7122CE1A763345A091AEB8C5DF8AB37CD4D"
		  "96C3230E8EA847C1D28986D3398DEA80F2C48C3B4406524BCD4"
		  "850DC02621B470AC7DF27372D95822795E8A2D81AACCD72F535"
		  "110FA01FDF0CABD7D848960603B80758406FFA78FB6C68DB4FD"
		  "F40C146318C2D296CD1A5BD7C1E11EA967B72792A3A7FBAC908"
		  "9588D3FB555F122F7AFB499AA3B3C43E07BD3D8F9FE980213C1"
		  "74D37DB9FBC8ECD318CC207689766A327614ED6B01BB0A5A2CF"
		  "04F5B54EDD97BE72E5F894328EE86A1C9D2FFB65C9FD0FEA4BA"
		  "865D0EA4F16DEF67CE1D6D42095C1C8488376A4326396FE3448"
		  "484754153B2D4A94C442BDE7CF67E8B9994E958D3CC17D5B0FD"
		  "7A4F203024EC29CEBF40B549F2936CC6C196AB83EC8C4585D28"
		  "E31D3ECF5BB59AB24B2CC6908E51A9BCD5BBCFA92C02E61B08C"
		  "A7E3AAAED73C450909C8B9BDB422CCAB0A86C5A5C3C10A634B1"
		  "7332"));
  test_hash (&nettle_shake256, /* 252 octets */
	     SHEX("A7ED84749CCC56BB1DFBA57119D279D412B8A986886D810F067AF349E8749E9EA746A60B03742636C464FC1EE233ACC52C1983914692B64309EDFDF29F1AB912EC3E8DA074D3F1D231511F5756F0B6EEAD3E89A6A88FE330A10FACE267BFFBFC3E3090C7FD9A850561F363AD75EA881E7244F80FF55802D5EF7A1A4E7B89FCFA80F16DF54D1B056EE637E6964B9E0FFD15B6196BDD7DB270C56B47251485348E49813B4EB9ED122A01B3EA45AD5E1A929DF61D5C0F3E77E1FDC356B63883A60E9CBB9FC3E00C2F32DBD469659883F690C6772E335F617BC33F161D6F6984252EE12E62B6000AC5231E0C9BC65BE223D8DFD94C5004A101AF9FD6C0FB"),
	     SHEX("47929C7D0C4DB7F9EBAEF5EA93BBC7B255C8E249EDF893257C7"
		  "AEB996CC8EEAB1A6CA4520F4D4257144CD5A9A850A372BD00DC"
		  "CA339C4781479E976B7508688DC63783CA9EC7A9FB8196721C8"
		  "C24AFF5BF9385453E9B85A1F7FC75A029E09D45910DC24F1304"
		  "D1E51EC702E79DC731B6120596BF3DF30DC0F756834AC0BB94F"
		  "99243EAC8910015D10DC618252A77D8B900C4141CFF2461C408"
		  "C19B978D1B58E690815C0C07B901137A45F045ABBD85428ACF5"
		  "1410135EB27128EC6725062665D1E6A6B8EBAD3B81D36770B0C"
		  "D14303B9264D25DFEE14B27798E22B46A77ADF456FDAFEB6D59"
		  "7F7E3152AAD40E29AB7E7808A4E46F43E18856518CE6D716062"
		  "029024AFE0EF0A7F3783F755977F8B10EEC658AABC20B2EE0E8"
		  "D3ADE73E2A7E78E4CEC3C09ADA3089247C678238580891ADA46"
		  "0D60DA297A4EB7DDBA67D117E21E7461BCE7CFC7573352696B4"
		  "643D6D9C9EC97EB431F6159AC438479BC6547BA04552748171F"
		  "4AEAAA2EF0DD7ECCB4ABDB2E07AB536F0183516B1DCBF0E12FD"
		  "134F9AD85B854AFDCCE14C1C52D0BF1EE1E52A12FDD63632D9D"
		  "F1B78E7899B03527DA8597124AE3C8797E6832138508AB89069"
		  "BF692A9E5D102CA20807A4BD5033D7BF7C918998D3D52B240B8"
		  "449D12B402EE7F50B211F597F1BD39B69A4DFF9A0E6DBE347AC"
		  "1137B3631DD15F97217A5BFE2C6579BC3C44F8789DED23EFEC5"
		  "1A64"));
  test_hash (&nettle_shake256, /* 253 octets */
	     SHEX("A6FE30DCFCDA1A329E82AB50E32B5F50EB25C873C5D2305860A835AECEE6264AA36A47429922C4B8B3AFD00DA16035830EDB897831C4E7B00F2C23FC0B15FDC30D85FB70C30C431C638E1A25B51CAF1D7E8B050B7F89BFB30F59F0F20FECFF3D639ABC4255B3868FC45DD81E47EB12AB40F2AAC735DF5D1DC1AD997CEFC4D836B854CEE9AC02900036F3867FE0D84AFFF37BDE3308C2206C62C4743375094108877C73B87B2546FE05EA137BEDFC06A2796274099A0D554DA8F7D7223A48CBF31B7DECAA1EBC8B145763E3673168C1B1B715C1CD99ECD3DDB238B06049885ECAD9347C2436DFF32C771F34A38587A44A82C5D3D137A03CAA27E66C8FF6"),
	     SHEX("401A2897FCA682A6949D88F939ECF943140ECA42F90D86D2DC3"
		  "01EFEA2DBD4C03998D686DCEC124334F0C5BA7674F25CDBF36F"
		  "B9480E8DB2111BA4E78221D1855893A526F97535936AAC3D88D"
		  "8908786BF5A9334EEE1A4D9B346A95573993DB7C9C7A75CE692"
		  "DF5118963083B778865396501845E12E4DDDE82035B03A6C436"
		  "97CDB2B1C5C4EB24D0D484ACB105549290B9FCD5C8B4EE31D4E"
		  "B542A2C9637D8E34687B18AB2DCF92A826877F9F1766ECBC43E"
		  "3879E481C4B6AD994586F110818A20B6F93C8CED8C1FBC0B545"
		  "72714EF722E83046A93CE23D118E494969D5BD1A9E9C744C5A5"
		  "6D193D2BC7054A4A0BB22987AA2CA672E69C2B733C860F292C4"
		  "F218557B7B921C4367B611E1D7FE30CF454EEBF09990C5ACD7B"
		  "43A174A92E0B8D8180DFD409261274FF09F55013A7F58242D29"
		  "330426BD4D8EF8365E4DE86469C436FC0FDC3B858EEE3E29C2E"
		  "D6F650266A7B34D2A0E5ED438E60891DF92FE6B950D95BB99F9"
		  "0344FC5EF5B210692059E9F859A476A3AFF4BB478F5DDA80629"
		  "E236AFD3742E35F46046BDF1522590562C8337C39FD38B8F57C"
		  "EDD49198F8242D951330FC4D0BDF5413345BB7315EEB67E4B1D"
		  "561D185FADF23C0979B036D885391643D7514B3FABCD16382CC"
		  "C49E6B7C0707B90AC0F86176DCF9D7F1DCD270374D2AB8E1E9F"
		  "AFB7DEC0F952FBD4456C9E7FCDA80CF0A46CE98AF6E9B588A37"
		  "0741"));
  test_hash (&nettle_shake256, /* 254 octets */
	     SHEX("83167FF53704C3AA19E9FB3303539759C46DD4091A52DDAE9AD86408B69335989E61414BC20AB4D01220E35241EFF5C9522B079FBA597674C8D716FE441E566110B6211531CECCF8FD06BC8E511D00785E57788ED9A1C5C73524F01830D2E1148C92D0EDC97113E3B7B5CD3049627ABDB8B39DD4D6890E0EE91993F92B03354A88F52251C546E64434D9C3D74544F23FB93E5A2D2F1FB15545B4E1367C97335B0291944C8B730AD3D4789273FA44FB98D78A36C3C3764ABEEAC7C569C1E43A352E5B770C3504F87090DEE075A1C4C85C0C39CF421BDCC615F9EFF6CB4FE6468004AECE5F30E1ECC6DB22AD9939BB2B0CCC96521DFBF4AE008B5B46BC006E"),
	     SHEX("8E756B09EE1679DFB957AEBC5B55CCCA4414AEAD58229106040"
		  "A1A58F4768D241A0869A3A7D682DD937FB13352F635EC3D5783"
		  "8469BA6AB93A9DC86EF3598F53E044FEA98301971A2182A1725"
		  "C32BF5529FCE04665C9F2C1275D484543E038E964D855E1D919"
		  "83EA34F7D66E19E27B6F59B23E9690C291432793167A5E8A9B4"
		  "4AEB9ECBAA500E2616AC32AF6B6ACFB6C2B4A485BE4FE5072BD"
		  "1F0E08AE343BF0C5CB241111F894BFD4BE6A26DBC7ED352D717"
		  "44BD9A4D51F149AC4EA5AD323B063D93E56AE12B81D88EB0830"
		  "24D3D164C0FD7B4BCE8A800F69CB7C66594D7AB20818DC79F01"
		  "853FEE81B345210CDF2964C3A25A0934ECE6B13029B5FDC3B78"
		  "0CD85A46E54495328F0BAA63078B3948172443B57D8571A82C7"
		  "17A4D93A6FD9ADA3A1EFC41F3697600D02FCB555EEF1D5162B3"
		  "46D892EABC60251B048979F49E3855FD5A49520A60D099ED2F5"
		  "A5C8764875A4EA242407856B5DC393BA1E71FD04C53181572AD"
		  "90E0184934D7B65E0EA56A8521A284E6A5B98BC704AD92E658A"
		  "8B4C64CF67157AC0673159B9E2A9EA9D94C6FB3E5BD963DFCAA"
		  "D484CF06BE75E3F23A11287B82BC6618B6EDA760D29D1D8400C"
		  "41EC3B42A24EC1E070BCE0EEC7DC6C6F19E2A6BA7FAAB72DD81"
		  "F31814A18CE984C7AAE6D9CAAA800318418F56871C8CFD8778F"
		  "AA7DAD3426849E10B041D6190EB445410F5F5E476068579FEDC"
		  "5869"));
  test_hash (&nettle_shake256, /* 255 octets */
	     SHEX("3A3A819C48EFDE2AD914FBF00E18AB6BC4F14513AB27D0C178A188B61431E7F5623CB66B23346775D386B50E982C493ADBBFC54B9A3CD383382336A1A0B2150A15358F336D03AE18F666C7573D55C4FD181C29E6CCFDE63EA35F0ADF5885CFC0A3D84A2B2E4DD24496DB789E663170CEF74798AA1BBCD4574EA0BBA40489D764B2F83AADC66B148B4A0CD95246C127D5871C4F11418690A5DDF01246A0C80A43C70088B6183639DCFDA4125BD113A8F49EE23ED306FAAC576C3FB0C1E256671D817FC2534A52F5B439F72E424DE376F4C565CCA82307DD9EF76DA5B7C4EB7E085172E328807C02D011FFBF33785378D79DC266F6A5BE6BB0E4A92ECEEBAEB1"),
	     SHEX("8A5199B4A7E133E264A86202720655894D48CFF344A928CF834"
		  "7F48379CEF347DFC5BCFFAB99B27B1F89AA2735E23D30088FFA"
		  "03B9EDB02B9635470AB9F1038985D55F9CA774572DD006470EA"
		  "65145469609F9FA0831BF1FFD842DC24ACADE27BD9816E3B5BF"
		  "2876CB112232A0EB4475F1DFF9F5C713D9FFD4CCB89AE5607FE"
		  "35731DF06317949EEF646E9591CF3BE53ADD6B7DD2B6096E2B3"
		  "FB06E662EC8B2D77422DAAD9463CD155204ACDBD38E319613F3"
		  "9F99B6DFB35CA9365160066DB19835888C2241FF9A731A4ACBB"
		  "5663727AAC34A401247FBAA7499E7D5EE5B69D31025E63D04C3"
		  "5C798BCA1262D5673A9CF0930B5AD89BD485599DC184528DA47"
		  "90F088EBD170B635D9581632D2FF90DB79665CED430089AF13C"
		  "9F21F6D443A818064F17AEC9E9C5457001FA8DC6AFBADBE3138"
		  "F388D89D0E6F22F66671255B210754ED63D81DCE75CE8F189B5"
		  "34E6D6B3539AA51E837C42DF9DF59C71E6171CD4902FE1BDC73"
		  "FB1775B5C754A1ED4EA7F3105FC543EE0418DAD256F3F6118EA"
		  "77114A16C15355B42877A1DB2A7DF0E155AE1D8670ABCEC3450"
		  "F4E2EEC9838F895423EF63D261138BAAF5D9F104CB5A957AEA0"
		  "6C0B9B8C78B0D441796DC0350DDEABB78A33B6F1F9E68EDE3D1"
		  "805C7B7E2CFD54E0FAD62F0D8CA67A775DC4546AF9096F2EDB2"
		  "21DB42843D65327861282DC946A0BA01A11863AB2D1DFD16E39"
		  "73D4"));

  /* Test incremental API */
  msg = SHEX("3A3A819C48EFDE2AD914FBF00E18AB6BC4F14513AB27D0C178A188B61431E7F5623CB66B23346775D386B50E982C493ADBBFC54B9A3CD383382336A1A0B2150A15358F336D03AE18F666C7573D55C4FD181C29E6CCFDE63EA35F0ADF5885CFC0A3D84A2B2E4DD24496DB789E663170CEF74798AA1BBCD4574EA0BBA40489D764B2F83AADC66B148B4A0CD95246C127D5871C4F11418690A5DDF01246A0C80A43C70088B6183639DCFDA4125BD113A8F49EE23ED306FAAC576C3FB0C1E256671D817FC2534A52F5B439F72E424DE376F4C565CCA82307DD9EF76DA5B7C4EB7E085172E328807C02D011FFBF33785378D79DC266F6A5BE6BB0E4A92ECEEBAEB1");
  digest = SHEX("8A5199B4A7E133E264A86202720655894D48CFF344A928CF834"
		"7F48379CEF347DFC5BCFFAB99B27B1F89AA2735E23D30088FFA"
		"03B9EDB02B9635470AB9F1038985D55F9CA774572DD006470EA"
		"65145469609F9FA0831BF1FFD842DC24ACADE27BD9816E3B5BF"
		"2876CB112232A0EB4475F1DFF9F5C713D9FFD4CCB89AE5607FE"
		"35731DF06317949EEF646E9591CF3BE53ADD6B7DD2B6096E2B3"
		"FB06E662EC8B2D77422DAAD9463CD155204ACDBD38E319613F3"
		"9F99B6DFB35CA9365160066DB19835888C2241FF9A731A4ACBB"
		"5663727AAC34A401247FBAA7499E7D5EE5B69D31025E63D04C3"
		"5C798BCA1262D5673A9CF0930B5AD89BD485599DC184528DA47"
		"90F088EBD170B635D9581632D2FF90DB79665CED430089AF13C"
		"9F21F6D443A818064F17AEC9E9C5457001FA8DC6AFBADBE3138"
		"F388D89D0E6F22F66671255B210754ED63D81DCE75CE8F189B5"
		"34E6D6B3539AA51E837C42DF9DF59C71E6171CD4902FE1BDC73"
		"FB1775B5C754A1ED4EA7F3105FC543EE0418DAD256F3F6118EA"
		"77114A16C15355B42877A1DB2A7DF0E155AE1D8670ABCEC3450"
		"F4E2EEC9838F895423EF63D261138BAAF5D9F104CB5A957AEA0"
		"6C0B9B8C78B0D441796DC0350DDEABB78A33B6F1F9E68EDE3D1"
		"805C7B7E2CFD54E0FAD62F0D8CA67A775DC4546AF9096F2EDB2"
		"21DB42843D65327861282DC946A0BA01A11863AB2D1DFD16E39"
		"73D4");

  test_incremental (msg, digest, 16);
  test_incremental (msg, digest, SHA3_256_BLOCK_SIZE - 1);
  test_incremental (msg, digest, SHA3_256_BLOCK_SIZE);
  test_incremental (msg, digest, SHA3_256_BLOCK_SIZE + 1);
}
