/* 
 * USRP - Universal Software Radio Peripheral
 *
 * Copyright (C) 2003 Free Software Foundation, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "usrp_common.h"
#include "usrp_commands.h"
#include "usrp_globals.h"
#include "fpga_rev0.h"
#include "delay.h"
#include "fpga_regs0.h"

static xdata unsigned char combo_shadow[4] = {0, 0, 0, 0};

/*
 * utilities for manipulating FPGA control bus (fcb)
 */

static void
set_fcb (unsigned char value, unsigned char mask)
{
  USRP_FPGA_CTRL_BUS = (USRP_FPGA_CTRL_BUS & ~mask) | (value & mask);
}

static void
clock_out_fcb_byte (unsigned char x)
{
  unsigned char	i;

  // clock out byte, most significant bit first
  
  for (i = 0; i < 8; i++){
    set_fcb ((x & 0x80) ? bmFCB_S_DATA_TO_FPGA : 0, bmFCB_S_DATA_TO_FPGA);	// output data
    set_fcb (bmFCB_S_CLK,			    bmFCB_S_CLK);		// clock high
    set_fcb (0,					    bmFCB_S_CLK);		// clock low
    x <<= 1;
  }
}

/*
 * fpga_write_reg
 *
 * We clock these out the FPGA control bus, most significant bit first.
 * Data is latched on the rising edge of the clock.
 */
void
fpga_write_reg_really (unsigned char regno, xdata unsigned char *regval)
{
  set_fcb (0, bmFCB_S_CLK | bmFCB_S_DATA_TO_FPGA | bmFCB_S_ENABLE | bmFCB_S_LOAD); // all low

  set_fcb (bmFCB_S_ENABLE, bmFCB_S_ENABLE);     // ENABLE high

  clock_out_fcb_byte (regno);	 		// register number
  clock_out_fcb_byte (regval[0]);		// value0 (most significant byte)
  clock_out_fcb_byte (regval[1]);		// value1 
  clock_out_fcb_byte (regval[2]);		// value2 
  clock_out_fcb_byte (regval[3]);		// value3 (least significant byte)

  set_fcb (0,  bmFCB_S_ENABLE);			// ENABLE low
  set_fcb (bmFCB_S_LOAD, bmFCB_S_LOAD);		// LOAD high
  set_fcb (bmFCB_S_CLK, bmFCB_S_CLK);		// clock high
  set_fcb (0, bmFCB_S_CLK);			// clock low
  set_fcb (0, bmFCB_S_LOAD);			// LOAD low
}

/*
 * We currently shadow the combo register.  It contains
 * 4 fields which are not related.  Until Matt splits them
 * up, the firmware handles it with 4 pseudo registers
 * that we map into the correct combo register shadow fields.
 */
void
fpga_write_reg (unsigned char regno, xdata unsigned char *regval)
{
  if (g_fpga_reset)
    return;

  if (regno < 128)
    fpga_write_reg_really (regno, regval);
  
  switch (regno){
  case FR_ADC_CLK_DIV:
    combo_shadow[0] = regval[3];	// regval[3] is the LSB of regval
    break;

  case FR_EXT_CLK_DIV:
    combo_shadow[1] = regval[3];
    break;
    
  case FR_INTERP:
    combo_shadow[2] = regval[3];
    break;
    
  case FR_DECIM:
    combo_shadow[3] = regval[3];
    break;

  default:
    return;
  }

  fpga_write_reg_really (FR_COMBO, combo_shadow);
}

void
fpga_set_reset (unsigned char on)
{
  on &= 0x1;
  g_fpga_reset = on;

  if (on){
    combo_shadow[0] = 0;
    combo_shadow[1] = 0;
    combo_shadow[2] = 0;
    combo_shadow[3] = 0;

    set_fcb (bmFCB_RESET, bmFCB_RESET);		// assert RESET signal
  }
  else {
    set_fcb (0,           bmFCB_RESET);		// deassert RESET signal
  }
}

void
fpga_set_tx_enable (unsigned char on)
{
  on &= 0x1;
  g_tx_enable = on;

  if (on){
    set_fcb (bmFCB_ENABLE_TX, bmFCB_ENABLE_TX);		// assert ENABLE_TX
    g_tx_underrun = 0;
    fpga_clear_flags ();
  }
  else {
    set_fcb (0,               bmFCB_ENABLE_TX);		// deassert ENABLE_TX
  }
}

void
fpga_set_rx_enable (unsigned char on)
{
  on &= 0x1;
  g_rx_enable = on;
  
  if (on){
    set_fcb (bmFCB_ENABLE_RX, bmFCB_ENABLE_RX);		// assert ENABLE_RX
    g_rx_overrun = 0;
    fpga_clear_flags ();
  }
  else {
    set_fcb (0,               bmFCB_ENABLE_RX);		// deassert ENABLE_RX
  }
}

