/* -*- c++ -*- */
/*
 * Copyright 2003 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <usb.h>			/* needed for usb functions */
#include <getopt.h>
#include <assert.h>
#include <math.h>
#include "time_stuff.h"
#include "fusb.h"
#include "usrp_prims.h"
#include "usrp_interfaces.h"


char *prog_name;

static bool test_output (fusb_ephandle *ep, bool forever_p);
static bool test_input  (fusb_ephandle *ep, bool forever_p, FILE *fp);

static void
set_progname (char *path)
{
  char *p = strrchr (path, '/');
  if (p != 0)
    prog_name = p+1;
  else
    prog_name = path;
}

static void
usage ()
{
  fprintf (stderr, "usage: %s [-i] [-f] [-v] [-o output_file]\n", prog_name);
  fprintf (stderr, "  [-i] input\n");
  fprintf (stderr, "  [-f] loop forever\n");
  fprintf (stderr, "  [-v] verbose\n");
  exit (1);
}

static void
die (const char *msg)
{
  fprintf (stderr, "%s: %s\n", prog_name, msg);
  exit (1);
}

int
main (int argc, char **argv)
{
  bool	input_p = false;
  bool 	verbose_p = false;
  bool	forever_p = false;
  int	ch;
  char	*output_filename = 0;
  int	nth = 0;

  set_progname (argv[0]);

  while ((ch = getopt (argc, argv, "ivfo:")) != EOF){
    switch (ch){
    case 'i':
      input_p = true;
      break;
      
    case 'f':
      forever_p = true;
      break;

    case 'v':
      verbose_p = true;
      break;

    case 'o':
      output_filename = optarg;
      break;
      
    default:
      usage ();
    }
  }

  usrp_one_time_init ();
  

  if (!usrp_load_standard_bits (nth, false))
    exit (1);
  
  struct usb_device *udev = usrp_find_device (nth);
  if (udev == 0){
    fprintf (stderr, "%s: failed to find USRP\n", prog_name);
    exit (1);
  }


  if (input_p){
    FILE *fp = 0;

    struct usb_dev_handle *udh = usrp_open_rx_interface (udev);
    if (udh == 0)
      die ("failed: usrp_open_rx_interface");
    
    fusb_devhandle *fdh = fusb_sysconfig::make_devhandle (udh);
    fusb_ephandle  *ep_in =
      fdh->make_ephandle (USRP_RX_ENDPOINT, true, 16384, 16);

    if (output_filename){
      fp = fopen (output_filename, "wb");
      if (fp == 0)
	perror (output_filename);
    }
      
    test_input (ep_in, forever_p, fp);

    if (fp)
      fclose (fp);

    delete ep_in;
    delete fdh;
    usrp_close_interface (udh);
  }

  else {

    struct usb_dev_handle *udh = usrp_open_tx_interface (udev);
    if (udh == 0)
      die ("failed: usrp_open_tx_interface");
    
    fusb_devhandle *fdh = fusb_sysconfig::make_devhandle (udh);
    fusb_ephandle  *ep_out = fdh->make_ephandle (USRP_TX_ENDPOINT, false);

    test_output (ep_out, forever_p);

    delete ep_out;
    delete fdh;
    usrp_close_interface (udh);
  }

  return 0;
}


static bool
test_output (fusb_ephandle *ep, bool forever_p)
{
  static const int BUFSIZE = 16 * 1024;
  static const int N = BUFSIZE/sizeof (short);
  short 	   buf[N];
  int		   nbytes = 0;
  int		   max_bytes = 512 * (1L << 20);

  static const int    PERIOD = 65;		// any value is valid
  static const int    PATLEN = 2 * PERIOD;	
  static const double AMPL = 10000;
  short		      pattern[PATLEN];

  for (int i = 0; i < PERIOD; i++){
    pattern[2*i+0] = (short) (AMPL * cos (2*M_PI * i / PERIOD));
    pattern[2*i+1] = (short) (AMPL * sin (2*M_PI * i / PERIOD));
  }
  

  double	   start_wall_time = get_elapsed_time ();
  double	   start_cpu_time  = get_cpu_usage ();

  int pi = 0;

  ep->start ();

  for (nbytes = 0; forever_p || nbytes < max_bytes; nbytes += BUFSIZE){

    for (int i = 0; i < N; i++){
      buf[i] = pattern[pi];
      pi++;
      if (pi >= PATLEN)
	pi = 0;
    }

    int	ret = ep->write (buf, sizeof (buf));
    if (ret != sizeof (buf)){
      fprintf (stderr, "test_output: error\n");
    }
  }

  ep->wait_for_completion ();

  double stop_wall_time = get_elapsed_time ();
  double stop_cpu_time  = get_cpu_usage ();

  double delta_wall = stop_wall_time - start_wall_time;
  double delta_cpu  = stop_cpu_time  - start_cpu_time;

  printf ("transfered %.3g bytes in %.3g seconds.  %.4g bytes/sec.  cpu time = %.3g\n",
	  (double) max_bytes, delta_wall, max_bytes / delta_wall, delta_cpu);

  return true;
}


static bool
test_input  (fusb_ephandle *ep, bool forever_p, FILE *fp)
{
  int		   fd = -1;
  static const int BUFSIZE = 16 * 1024;
  static const int N = BUFSIZE/sizeof (short);
  short 	   buf[N];
  int		   nbytes = 0;
  int	   	   max_bytes = 512 * (1L << 20);
  // int		   max_bytes = 128 * (1L << 20);

  double	   start_wall_time = get_elapsed_time ();
  double	   start_cpu_time  = get_cpu_usage ();

  if (fp)
    fd = fileno (fp);
  
  ep->start ();

  for (nbytes = 0; forever_p || nbytes < max_bytes; nbytes += BUFSIZE){

    int	ret = ep->read (buf, sizeof (buf));
    if (ret != sizeof (buf)){
      fprintf (stderr, "test_input: error\n");
    }
    if (fd != -1){
      if (write (fd, buf, sizeof (buf)) == -1){
	perror ("write");
	fd = -1;
      }
    }
  }

  double stop_wall_time = get_elapsed_time ();
  double stop_cpu_time  = get_cpu_usage ();

  double delta_wall = stop_wall_time - start_wall_time;
  double delta_cpu  = stop_cpu_time  - start_cpu_time;

  printf ("transfered %.3g bytes in %.3g seconds.  %.4g bytes/sec.  cpu time = %.3g\n",
	  (double) max_bytes, delta_wall, max_bytes / delta_wall, delta_cpu);

  return true;
}

