// Copyright 2009 The Go Authors. All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

package math

// Ldexp is the inverse of Frexp.
// It returns frac × 2**exp.
//
// Special cases are:
//	Ldexp(±0, exp) = ±0
//	Ldexp(±Inf, exp) = ±Inf
//	Ldexp(NaN, exp) = NaN

//extern ldexp
func libc_ldexp(float64, int) float64

func Ldexp(frac float64, exp int) float64 {
	r := libc_ldexp(frac, exp)

	// Work around a bug in the implementation of ldexp on Solaris
	// 9.  If multiplying a negative number by 2 raised to a
	// negative exponent underflows, we want to return negative
	// zero, but the Solaris 9 implementation returns positive
	// zero.  This workaround can be removed when and if we no
	// longer care about Solaris 9.
	if r == 0 && frac < 0 && exp < 0 {
		r = Copysign(0, frac)
	}
	return r
}

func ldexp(frac float64, exp int) float64 {
	// special cases
	switch {
	case frac == 0:
		return frac // correctly return -0
	case IsInf(frac, 0) || IsNaN(frac):
		return frac
	}
	frac, e := normalize(frac)
	exp += e
	x := Float64bits(frac)
	exp += int(x>>shift)&mask - bias
	if exp < -1074 {
		return Copysign(0, frac) // underflow
	}
	if exp > 1023 { // overflow
		if frac < 0 {
			return Inf(-1)
		}
		return Inf(1)
	}
	var m float64 = 1
	if exp < -1022 { // denormal
		exp += 52
		m = 1.0 / (1 << 52) // 2**-52
	}
	x &^= mask << shift
	x |= uint64(exp+bias) << shift
	return m * Float64frombits(x)
}
