/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.netbeans.spi.editor.hints;

import java.util.ArrayList;
import java.util.List;
import javax.swing.text.Position;
import org.openide.filesystems.FileObject;

/**
 * Represents a set of changes made by a hint.  Note that the start/end
 * parameters refer to offsets that should be <i>selected</i>, not to
 * the general offsets of the modified element.
 * <p>
 * In other words, a change generated by a hint can generate things like
 * variable names;  ChangeInfo provides a list of generated code which the
 * user might want to modify/replace with their own text.  The start/end offsets
 * are offsets into the file in question, which determine selection start/end
 * in the editor.
 * <p>
 * This class provides for a list of changes, anticipating &quot;live template&quot;
 * support (where the user confirms each generated element).  Currently only
 * the first change provided is used.
 * 
 * @author Tim Boudreau
 */
public final class ChangeInfo {
    private List<Change> changes = null;
    
    /**
     * Create an instance of ChangeInfo prepopulated with a
     * single change
     */
    public ChangeInfo(FileObject fileObject, Position start, Position end) {
        add (fileObject, start, end);
    }
    
    /**
     * Create an instance of ChangeInfo prepopulated with a
     * single change without associated file object.
     */
    public ChangeInfo(Position start, Position end) {
        add (null, start, end);
    }
    
    /**
     * Create an empty instance of ChangeInfo .
     */
    public ChangeInfo() {
    }

    /**
     * @return number of changes
     */
    public final int size() {
        return changes != null ? changes.size() : 0;
    }

    /**
     * Adds one change.
     * @param fileObject
     * @param start
     * @param end
     */
    public final void add (FileObject fileObject, Position start, Position end) {
        if (changes == null) {
            changes = new ArrayList<Change>(5);
        }
        changes.add (new ChangeImpl (fileObject, start, end));
    }

    /**
     * Getter for the i-th change
     * @param i index of the change
     * @return the change or throws ArrayIndexOutOfBoundsException if there are
     *         no changes in the ChangeInfo
     */
    public final Change get (int i) {
        if (changes == null) {
            throw new ArrayIndexOutOfBoundsException ("No changes");
        }
        return (Change) changes.get(i);
    }
    
    @Override
    public String toString() {
        int size = size();
        if (size == 0) {
            return "Empty ChangeInfo";
        } else {
            StringBuffer sb = new StringBuffer (100);
            sb.append ("ChangeInfo [");
            for (int i=0; i < size; i++) {
                sb.append (get(i));
                if (i != size-1) {
                    sb.append (',');
                }
            }
            sb.append ("]");
            return sb.toString();
        }
    }
    
    /**
     * Interface representing a single caret-positioning or user-modifiable
     * change.
     */
    public static interface Change {
        /**
         * @return start position of the change
         */
        public Position getStart();

        /**
         * @return end position of the change
         */
        public Position getEnd();
        /**
         * @return the associated file or <code>null</code> if there is no
         *         file associated with the chage
         */
        public FileObject getFileObject();
    }
    
    private static final class ChangeImpl implements Change {
        Position start;
        Position end;
        FileObject fileObject;
        
        ChangeImpl(FileObject fileObject, Position start, Position end) {
            this.fileObject = fileObject;
            this.start = start;
            this.end = end;
        }
        
        public Position getStart() {
            return start;
        }
        
        public Position getEnd() {
            return end;
        }
        
        public FileObject getFileObject() {
            return fileObject;
        }
        
        public String toString() {
            return "Change from " + start.getOffset() + " to " + end.getOffset() + " in " + fileObject;
        }
    }
    
}
