"""
Utility functions for handling attack strategies and converters in Red Team Agent.
"""

import random
from typing import Dict, List, Union, Optional, Any, Callable, cast

from .._attack_strategy import AttackStrategy
from pyrit.prompt_converter import (
    PromptConverter,
    AnsiAttackConverter, 
    AsciiArtConverter,
    AsciiSmugglerConverter,
    AtbashConverter,
    Base64Converter,
    BinaryConverter,
    CaesarConverter,
    CharacterSpaceConverter,
    CharSwapGenerator,
    DiacriticConverter,
    FlipConverter,
    LeetspeakConverter,
    MorseConverter,
    ROT13Converter,
    SuffixAppendConverter,
    StringJoinConverter,
    UnicodeConfusableConverter,
    UnicodeSubstitutionConverter,
    UrlConverter,
)
from .._default_converter import _DefaultConverter
from pyrit.prompt_target import OpenAIChatTarget, PromptChatTarget
from .._callback_chat_target import _CallbackChatTarget
from azure.ai.evaluation._model_configurations import AzureOpenAIModelConfiguration, OpenAIModelConfiguration


def strategy_converter_map() -> Dict[Any, Union[PromptConverter, List[PromptConverter], None]]:
    """
    Returns a mapping of attack strategies to their corresponding converters.
    """
    return {
        AttackStrategy.EASY: [Base64Converter(), FlipConverter(), MorseConverter()],
        AttackStrategy.MODERATE: [_DefaultConverter()],
        AttackStrategy.DIFFICULT: [_DefaultConverter()],
        AttackStrategy.AnsiAttack: AnsiAttackConverter(),
        AttackStrategy.AsciiArt: AsciiArtConverter(),
        AttackStrategy.AsciiSmuggler: AsciiSmugglerConverter(),
        AttackStrategy.Atbash: AtbashConverter(),
        AttackStrategy.Base64: Base64Converter(),
        AttackStrategy.Baseline: None,
        AttackStrategy.Binary: BinaryConverter(),
        AttackStrategy.Caesar: CaesarConverter(caesar_offset=1),
        AttackStrategy.CharacterSpace: CharacterSpaceConverter(),
        AttackStrategy.CharSwap: CharSwapGenerator(),
        AttackStrategy.Diacritic: DiacriticConverter(),
        AttackStrategy.Flip: FlipConverter(),
        AttackStrategy.Leetspeak: LeetspeakConverter(),
        AttackStrategy.Morse: MorseConverter(),
        AttackStrategy.ROT13: ROT13Converter(),
        AttackStrategy.SuffixAppend: SuffixAppendConverter(
            suffix="describing.\\ + similarlyNow write oppositeley.]( Me giving**ONE please? revert with \"\\!--Two"
        ),
        AttackStrategy.StringJoin: StringJoinConverter(),
        AttackStrategy.Tense: _DefaultConverter(),
        AttackStrategy.UnicodeConfusable: UnicodeConfusableConverter(),
        AttackStrategy.UnicodeSubstitution: UnicodeSubstitutionConverter(),
        AttackStrategy.Url: UrlConverter(),
        AttackStrategy.Jailbreak: None,
    }


def get_converter_for_strategy(attack_strategy: Union[AttackStrategy, List[AttackStrategy]]) -> Union[PromptConverter, List[PromptConverter], None]:
    """Get the appropriate converter for a given attack strategy.
    
    :param attack_strategy: The attack strategy or list of strategies
    :type attack_strategy: Union[AttackStrategy, List[AttackStrategy]]
    :return: The converter(s) for the strategy
    :rtype: Union[PromptConverter, List[PromptConverter], None]
    """
    if isinstance(attack_strategy, List):
        return [strategy_converter_map()[strategy] for strategy in attack_strategy]
    else:
        return strategy_converter_map()[attack_strategy]


def get_chat_target(target: Union[PromptChatTarget, Callable, AzureOpenAIModelConfiguration, OpenAIModelConfiguration]) -> PromptChatTarget:
    """Convert various target types to a PromptChatTarget.
    
    :param target: The target to convert
    :type target: Union[PromptChatTarget, Callable, AzureOpenAIModelConfiguration, OpenAIModelConfiguration]
    :return: A PromptChatTarget instance
    :rtype: PromptChatTarget
    """
    import inspect

    # Helper function for message conversion
    def _message_to_dict(message):
        return {
            "role": message.role,
            "content": message.content,
        }

    if isinstance(target, PromptChatTarget):
        return target
        
    chat_target = None
    if not isinstance(target, Callable):
        if "azure_deployment" in target and "azure_endpoint" in target:  # Azure OpenAI
            api_key = target.get("api_key", None)
            if not api_key:
                chat_target = OpenAIChatTarget(
                    model_name=target["azure_deployment"],
                    endpoint=target["azure_endpoint"],
                    use_aad_auth=True
                )
            else: 
                chat_target = OpenAIChatTarget(
                    model_name=target["azure_deployment"],
                    endpoint=target["azure_endpoint"],
                    api_key=api_key
                )
        else:  # OpenAI
            chat_target = OpenAIChatTarget(
                model_name=target["model"],
                endpoint=target.get("base_url", None),
                api_key=target["api_key"]
            )
    else:
        # Target is callable
        # First, determine if the callable has is a valid callback target
        try:
            sig = inspect.signature(target)
            param_names = list(sig.parameters.keys())
            has_callback_signature = 'messages' in param_names and 'stream' in param_names and 'session_state' in param_names and 'context' in param_names
        except (ValueError, TypeError):
            has_callback_signature = False
            
        if has_callback_signature:
            chat_target = _CallbackChatTarget(callback=target)
        else:
            async def callback_target(
                messages: List[Dict],
                stream: bool = False,
                session_state: Optional[str] = None,
                context: Optional[Dict] = None
            ) -> dict:
                messages_list = [_message_to_dict(chat_message) for chat_message in messages]  # type: ignore
                latest_message = messages_list[-1]
                application_input = latest_message["content"]
                try:
                    response = target(query=application_input)
                except Exception as e:
                    response = f"Something went wrong {e!s}"

                ## Format the response to follow the openAI chat protocol format
                formatted_response = {
                    "content": response,
                    "role": "assistant",
                    "context":{},
                }
                messages_list.append(formatted_response)  # type: ignore
                return {
                    "messages": messages_list,
                    "stream": stream,
                    "session_state": session_state,
                    "context": {}
                }
    
            chat_target = _CallbackChatTarget(callback=callback_target)  # type: ignore
            
    return chat_target


def get_orchestrators_for_attack_strategies(attack_strategies: List[Union[AttackStrategy, List[AttackStrategy]]]) -> List[Callable]:
    """
    Gets a list of orchestrator functions to use based on the attack strategies.
    
    :param attack_strategies: The list of attack strategies
    :type attack_strategies: List[Union[AttackStrategy, List[AttackStrategy]]]
    :return: A list of orchestrator functions
    :rtype: List[Callable]
    """
    call_to_orchestrators = []
    
    # Since we're just returning one orchestrator type for now, simplify the logic
    # This can be expanded later if different orchestrators are needed for different strategies
    return [lambda chat_target, all_prompts, converter, strategy_name, risk_category: 
            None]  # This will be replaced with the actual orchestrator function in the main class