/* 
 * acpi battary code by Maxim Uvarov <uvarov@hotbox.ru> September 2004
 * License: GPL.
 *
 */

#include <stdio.h>
#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <stdlib.h>
#include "osdbattery.h"
#include <unistd.h>

#define OSD
#ifdef OSD
#include <xosd.h>
#endif

int get_battery_state(battery_state_struct* battery_state, int bnum) {
	char present_voltage[64];
	char charging_state[64];
	char remaining_capacity[64];
	char file[64];
	char b[256];
	FILE *fp;

	sprintf(file, "/proc/acpi/battery/BAT%d/state", bnum );

	fp = fopen(file, "r");
	if (fp == NULL) {
		printf("Cannot open ACPI info! Is ACPI supported by the kernel?\n");
		exit(1);
	}

	rewind(fp);
	while (!feof(fp)) {

		if (fgets(b, 256, fp) == NULL) {
			fclose(fp);
			return 0;
		} 		

		if (sscanf(b, "present voltage: %63s", present_voltage ) != errno) {
			battery_state->present_voltage = atoi(present_voltage);
		}

		if (sscanf(b, "charging state: %63s", charging_state ) != errno) {
			strcpy(battery_state->charging_state, charging_state);
		}

		if (sscanf(b, "remaining capacity: %63s", remaining_capacity ) != errno) {
			battery_state->remaining_capacity = atoi(remaining_capacity);

		}
	}	
	fclose(fp);
	return 0;
}
int get_battery_info(battery_info_struct* battery_info, int bnum) { 	
	char design_voltage[64];
	char last_full_capacity[64];
	char file[64];

	sprintf(file, "/proc/acpi/battery/BAT%d/info", bnum );

	FILE *fp = fopen(file,"r");
	if (fp == NULL) {
		printf("can not open acpi info, acpi supported by kernel?\n");
		exit(1);
	}

	rewind(fp);
	char b[256];

	while (!feof(fp)) {

		if (fgets(b, 256, fp) == NULL) {
			fclose(fp);
			return 0;
		}

		if (sscanf(b, "design voltage: %63s", design_voltage) != errno) {
			battery_info->design_voltage = atoi(design_voltage);							   	}

		if (sscanf(b, "last full capacity: %63s", last_full_capacity) != errno) {
			battery_info->last_full_capacity = atoi(last_full_capacity);	
		}

	}
	return 0;
}

config_struct *  parse_config() { 	
	config_struct* config;
	char b[256];
	char buf[64];
	char* env_home;
	char config_file[256];

	config = malloc(sizeof(config_struct));
	if (config == NULL) {
		printf("malloc failed\n");
	}

	/* getting home directory */
	env_home = malloc(256);
	env_home = getenv("HOME");
	sprintf(config_file, "%s/.osdbattery/osdbattery.cfg",env_home);

	/* always establish defaults, in case user spec file doesn't
	 * set every available config attribute */
	config->vertical_offset = 50;
	config->horizontal_offset = 5;
	config->align = 2;
	config->update_time = 10;
	config->battery_num = 0;
	config->bar_length = 52;
	strncpy(config->font, "-microsoft-verdana-*-r-*-*-12-*-*-*-*-*-koi8-e", 63);
	strncpy(config->colour, "green", 63);
	strncpy(config->shadow_colour, "black", 63);
	config->noloop = 0;

	FILE *fp = fopen(config_file,"r");
	if (fp == NULL) {
		printf("WARNING:can not open config file, using defaults\n");
		return config;
	}

	rewind(fp);
	while (!feof(fp)) {

		if (fgets(b, 256, fp) == NULL) {
			fclose(fp);
			return config;
		}

		if (sscanf(b, "vertical offset = %63s", buf) != errno) {
			config->vertical_offset = atoi(buf);	
		}

		if (sscanf(b, "horizontal offset = %63s", buf) != errno) {
			config->horizontal_offset = atoi(buf);	
		}

		if (sscanf(b, "align = %63s", buf) != errno) {
			config->align = atoi(buf);	
		}

		if (sscanf(b, "update time = %63s", buf) != errno) {
			config->update_time = atoi(buf);	
		}

		if (sscanf(b, "display battery = %63s", buf) != errno) {
			config->battery_num = atoi(buf);	
		}

		if (sscanf(b, "bar length = %63s", buf) != errno) {
			config->bar_length = atoi(buf);	
		}

		if (sscanf(b, "font = %63s", buf) != errno) {
			strcpy(config->font, buf);
		}

		if (sscanf(b, "colour = %63s", buf) != errno) {
			strcpy(config->colour, buf);
		}

		if (sscanf(b, "shadow colour = %63s", buf) != errno) {
			strcpy(config->shadow_colour, buf);
		}

		if (sscanf(b, "no loop = %63s", buf) != errno) {
			config->noloop = atoi(buf);	
		}
	}
	return(config);
}

int main(int argc, char* argv[]) {
	battery_state_struct* battery_state;
	battery_info_struct* battery_info;
	int ret;

	/*some statistics */
	int battery_pecent;
	int battery_voltage;

#ifdef OSD
	/* Drawning OSD */
	xosd* osd = NULL;
	char* string = NULL;
	config_struct* config;

	/* Parse config */
	config = parse_config();
	if (config == NULL) {
		printf("can not parse config file\n");
		exit(1);
	}

	/* create OSD object */
	osd = xosd_create(3);
	if (osd == NULL) {
		printf("Error creating an xosd object\n");
	}

	xosd_set_font(osd, config->font);
	xosd_set_align (osd, config->align);
	xosd_set_vertical_offset (osd, config->vertical_offset);
	xosd_set_horizontal_offset (osd, config->horizontal_offset);
	xosd_set_bar_length(osd, config->bar_length);
	xosd_set_colour(osd, config->colour);
	xosd_set_shadow_colour(osd, config->shadow_colour);
	xosd_set_shadow_offset(osd, 1);

#endif
	/* getting battery state */
	battery_state = malloc(sizeof(battery_state_struct));
	if (battery_state == NULL) {
		printf("malloc failed\n");
	}

	/* getting battery info */
	battery_info = malloc(sizeof(battery_info_struct));
	if (battery_info == NULL) {
		printf ("malloc failed\n");
		exit (-1);
	}
#ifdef OSD
	string = malloc(60); 
#endif

	while(1) {

		/* getting battery state */
		ret = get_battery_state(battery_state, config->battery_num); 

		if (ret == -1) {
			printf ("get_battery_state fail\n");
			exit (-1);
		}

		/* getting battery info */
		ret = get_battery_info(battery_info, config->battery_num); 

		if (ret == -1) {
			printf ("get_battary_info fail\n");
			exit (-1);
		}

		/* Pecent of battery charge */
		battery_pecent = battery_state->remaining_capacity*100/battery_info->last_full_capacity;
		battery_voltage = battery_state->present_voltage -  battery_info->design_voltage; 

#ifdef OSD
		/* Display battery charge */
		sprintf(string, "Battery %s: %d%%", battery_state->charging_state, battery_pecent);
		xosd_display (osd, 0, XOSD_string, string);
		xosd_display (osd, 1, XOSD_percentage, battery_pecent);

		/* Display battery voltage*/
		sprintf(string, "Voltage:  %dmV/%dmV min|%dmV left", battery_state->present_voltage, battery_info->design_voltage, battery_voltage);
		xosd_display(osd, 2, XOSD_string, string);
		sleep(config->update_time);
#endif
		if (config->noloop) break;
	}

#ifdef OSD
	xosd_hide(osd);
	xosd_wait_until_no_display(osd);
	/*xosd_destroy(osd); this segfaults! */
	free(string);
#endif
	/* Clear memory */
	free(battery_info);
	free(config);

	return(0);
}




