/*
 * QtTapioca, the Tapioca Qt4 Client Library
 * Copyright (C) 2006 by INdT
 *  @author Andre Moreira Magalhaes <andre.magalhaes@indt.org.br>
 *  @author Abner Jose de Faria Silva <abner.silva@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA
 */

#ifndef QTTAPIOCA_CONTACT_BASE_H
#define QTTAPIOCA_CONTACT_BASE_H

#include <QtCore/QObject>
#include <QtCore/QString>
#include <qglobal.h>

#include <QtTapioca/ChannelTarget>

#include <QtTelepathy/Client/Connection>
#include <QtTelepathy/Client/ConnectionAvatarsInterface>
#include <QtTelepathy/Client/ConnectionPresenceInterface>
#include <QtTelepathy/Client/ConnectionAliasingInterface>
#include <QtTelepathy/Client/ConnectionCapabilitiesInterface>

namespace QtTapioca {

class Avatar;
class ContactBasePrivate;
class Handle;

class Q_DECL_EXPORT ContactBase: public ChannelTarget {
    Q_OBJECT
    Q_DECLARE_PRIVATE(ContactBase)
    Q_ENUMS(Presence)
    Q_ENUMS(Capability)

public:
    enum Presence {
        Offline = 1,
        Available,
        Away,
        XA,
        Hidden,
        Busy
    };
    enum Capability {
        None  = 0x00,
        Text  = 0x01,
        Audio = 0x02,
        Video = 0x03,
        All   = 0xFF
    };

    ~ContactBase();

    void requestAvatar();
    QString currentAvatarToken();
    Presence presence() const;
    QString presenceMessage() const;
    QString alias() const;
    QList<Capability> capabilities() const;

    bool operator==(const ContactBase &info) const;

Q_SIGNALS:
    void avatarReceived(QtTapioca::ContactBase *contactBase, QtTapioca::Avatar *avatar);
    void avatarUpdated(QtTapioca::ContactBase *contactBase, const QString &newToken);
    void presenceUpdated(QtTapioca::ContactBase *contactBase, QtTapioca::ContactBase::Presence presence, const QString &presenceMessage);
    void aliasChanged(QtTapioca::ContactBase *contactBase, const QString &alias);
    void capabilitiesChanged(QtTapioca::ContactBase *contactBase, QtTapioca::ContactBase::Capability capabilities);

protected:
    ContactBase(org::freedesktop::Telepathy::Connection *telepathyConn,
                org::freedesktop::Telepathy::ConnectionAvatarsInterface  *telepathyIAvatar,
                org::freedesktop::Telepathy::ConnectionPresenceInterface *telepathyIPresence,
                org::freedesktop::Telepathy::ConnectionAliasingInterface *telepathyIAliasing,
                org::freedesktop::Telepathy::ConnectionCapabilitiesInterface *telepathyICapabilities,
                Handle *contact_handle,
                QObject *parent = 0);

    Presence strToPresenceEnum(const QString &presenceStr);
    const QString presenceEnumToStr(Presence num);
    void updateCapabilities(const QString &channelType, quint32 typeSpecificFlags);
    void initCapabilities();

    ContactBase::Presence m_presence;
    QString m_presenceMessage;
    org::freedesktop::Telepathy::ConnectionAvatarsInterface  *telepathyIAvatar;
    org::freedesktop::Telepathy::ConnectionPresenceInterface *telepathyIPresence;
    org::freedesktop::Telepathy::ConnectionAliasingInterface *telepathyIAliasing;
    org::freedesktop::Telepathy::ConnectionCapabilitiesInterface *telepathyICapabilities;

private Q_SLOTS:
    void onPresenceUpdated(org::freedesktop::Telepathy::PresenceStateInTimeMap newPresence);
    void onAvatarUpdated(uint contact, const QString &newAvatarToken);
    void onAliasChanged(org::freedesktop::Telepathy::AliasInfoList aliases);
    void onCapabilitiesChanged(org::freedesktop::Telepathy::CapabilitiesChangedInfoList caps);

private:

    ContactBasePrivate *d;
};

} // namespace

#endif

